#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/LIBC.H>
#include <ARCH/BIOS16.H>
#include <ARCH/BSF.H>
#include <ARCH/BT.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/SYNC.H>
#include <STDARG.H>
#include <VALUES.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/CONSOLE.H>
#include <SPAD/IOCTL.H>
#include <ARCH/IO.H>

#include <VGA.H>
#include <SPAD/TTY.H>
#include "CONS.H"
#include "VESA.H"
#include "VGAREG.H"

#define SWITCH_TIMEOUT		(10 * JIFFIES_PER_SECOND)	/* the largest observed switch timeout was 4.5s */

static void (*DRIVER_SET_TIMING)(struct video_mode *v, struct tty_videomode_params *p) = NULL;
static long (*DRIVER_GET_PIXEL_CLOCK)(struct video_mode *v, unsigned px) = NULL;
static long (*DRIVER_ACCEL_AVAIL)(struct video_mode *v, int ac) = NULL;
static void (*DRIVER_ACCEL_INIT)(struct video_mode *v) = NULL;
static void (*DRIVER_ACCEL_EXIT)(struct video_mode *v) = NULL;
static void *(*DRIVER_ACCEL_DO)(struct video_mode *v, unsigned op, union accel_param *ap) = NULL;
static int (*DRIVER_SET_DISPLAY_START)(struct video_mode *v, unsigned addr, int waitretrace) = NULL;

const struct driver_function driver_functions[] = {
	"SET_VIDEO_TIMING", &DRIVER_SET_TIMING,
	"GET_PIXEL_CLOCK", &DRIVER_GET_PIXEL_CLOCK,
	"ACCEL_AVAIL", &DRIVER_ACCEL_AVAIL,
	"ACCEL_INIT", &DRIVER_ACCEL_INIT,
	"ACCEL_EXIT", &DRIVER_ACCEL_EXIT,
	"ACCEL_DO", &DRIVER_ACCEL_DO,
	"SET_DISPLAY_START", &DRIVER_SET_DISPLAY_START,
	NULL, NULL
};

static int driver_installed = 0;

static int version_major, version_minor;
static int vesa_available = 0;
static unsigned videoram = 0;
static int trident_scroll_bug = 0;
static int trio64_set_mode_bug = 0;

unsigned max_v_refresh = 0;
unsigned max_h_refresh = 0;

struct VbeInfoBlock {
	__u8 VbeSignature[4];
	__u16 VbeVersion;
	__u16 OemStringPtr_Off;
	__u16 OemStringPtr_Seg;
	__u16 Capabilities1;
	__u16 Capabilities2;
	__u16 VideoModePtr_Off;
	__u16 VideoModePtr_Seg;
	__u16 TotalMemory;
	__u16 OemSoftwareRev;
	__u16 OemVendorNamePtr_Off;
	__u16 OemVendorNamePtr_Seg;
	__u16 OemProductNamePtr_Off;
	__u16 OemProductNamePtr_Seg;
	__u16 OemProductRevPtr_Off;
	__u16 OemProductRevPtr_Seg;
	__u8 Reserved[222];
	__u8 OemData[256];
};

struct ModeInfoBlock {
	__u16 ModeAttributes;
	__u8 WinAAttributes;
	__u8 WinBAttributes;
	__u16 WinGranularity;
	__u16 WinSize;
	__u16 WinASegment;
	__u16 WinBSegment;
	__u16 WinFuncPtr_Off;
	__u16 WinFuncPtr_Seg;
	__u16 BytesPerScanLine;

	__u16 XResolution;
	__u16 YResolution;
	__u8 XCharSize;
	__u8 YCharSize;
	__u8 NumberOfPlanes;
	__u8 BitsPerPixel;
	__u8 NumberOfBanks;
	__u8 MemoryModel;
	__u8 BankSize;
	__u8 NumberOfImagePages;
	__u8 Reserved1;

	__u8 RedMaskSize;
	__u8 RedFieldPosition;
	__u8 GreenMaskSize;
	__u8 GreenFieldPosition;
	__u8 BlueMaskSize;
	__u8 BlueFieldPosition;
	__u8 RsvdMaskSize;
	__u8 RsvdFieldPosition;
	__u8 DirectColorModeInfo;

	__u32 PhysBasePtr;
	__u32 Reserved2;
	__u16 Reserved3;

	__u16 LinBytesPerScanLine;
	__u8 BnkNumberOfImagePages;
	__u8 LinNumberOfImagePages;
	__u8 LinRedMaskSize;
	__u8 LinRedFieldPosition;
	__u8 LinGreenMaskSize;
	__u8 LinGreenFieldPosition;
	__u8 LinBlueMaskSize;
	__u8 LinBlueFieldPosition;
	__u8 LinRsvdMaskSize;
	__u8 LinRsvdFieldPosition;
	__u16 MaxPixelClocl_Lo;
	__u16 MaxPixelClock_Hi;
	__u8 Reserved4[190];
};

struct CRTCInfoBlock {
	__u16 HorizontalTotal;
	__u16 HorizontalSyncStart;
	__u16 HorizontalSyncEnd;
	__u16 VerticalTotal;
	__u16 VerticalSyncStart;
	__u16 VerticalSyncEnd;
	__u8 Flags;
	__u8 PixelClock0;
	__u8 PixelClock1;
	__u8 PixelClock2;
	__u8 PixelClock3;
	__u8 RefreshRate0;
	__u8 RefreshRate1;
	__u8 Reserved[40];
};

static struct video_mode *video_modes = NULL;
static unsigned n_video_modes = 0;

__COLD_ATTR__ static struct video_mode *add_video_mode(void)
{
	MALLOC_REQUEST mrq;
	struct video_mode *vm;
	mrq.size = (n_video_modes + 1) * sizeof(struct video_mode);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) return NULL;
	memcpy(mrq.ptr, video_modes, n_video_modes * sizeof(struct video_mode));
	free(video_modes);
	video_modes = mrq.ptr;

	vm = video_modes + n_video_modes++;
	memset(vm, 0, sizeof(struct video_mode));
	return vm;
}

__COLD_ATTR__ static int contains(char *str, int len, char *substr)
{
	int sl = strlen(substr);
	while (len >= sl) {
		if (!_memcasecmp(str, substr, sl)) return 1;
		str++;
		len--;
	}
	return 0;
}

static int BIOS_REAL_INT(int intnum, struct bios_regs *regs, void *ptr, unsigned len, int flags)
{
	int r;
	KERNEL$CONSOLE_LOCK();
	r = BIOS16$REAL_INT(intnum, regs, ptr, len, flags);
	KERNEL$CONSOLE_UNLOCK();
	return r;
}

__COLD_ATTR__ int PROBE_VESA(void)
{
	struct video_mode *vm;
	int i;
	__u8 *video_string_ptr;
	unsigned video_string_len;
	__u16 *video_mode_ptr;
	struct VbeInfoBlock bios_id;
	struct bios_regs regs;
	char mem_str[16];
	memset(&regs, 0, sizeof regs);
	regs.eax = 0x4f00;
	regs.es = BIOS16$REAL_SEGMENT;
	regs.edi = BIOS16$REAL_OFFSET;
	memset(&bios_id, 0, sizeof bios_id);
	memcpy(&bios_id.VbeSignature, "VBE2", 4);
	if (BIOS_REAL_INT(0x10, &regs, &bios_id, sizeof bios_id, BIOS_DATA_PUT_GET)) return -1;
	if ((regs.eax & 0xffff) != 0x004f || memcmp(bios_id.VbeSignature, "VESA", 4)) {
		return -1;
	}
	version_major = (bios_id.VbeVersion >> 12) * 10 + ((bios_id.VbeVersion >> 8) & 0xf);
	version_minor = ((bios_id.VbeVersion >> 4) & 0xf) * 10 + (bios_id.VbeVersion & 0xf);
	video_string_ptr = (__u8 *)(bios_id.OemStringPtr_Off + (bios_id.OemStringPtr_Seg << 4));
	video_string_len = 0;
	if (video_string_ptr >= (__u8 *)(unsigned long)BIOS16$REAL_OFFSET && video_string_ptr < (__u8 *)(unsigned long)BIOS16$REAL_OFFSET + sizeof bios_id) {
		video_string_ptr = video_string_ptr + (unsigned long)&bios_id - BIOS16$REAL_OFFSET;
		while (video_string_ptr + video_string_len < (__u8 *)(&bios_id + 1) && (unsigned char)video_string_ptr[video_string_len] >= ' ') video_string_len++;
	} else {
		video_string_ptr = KERNEL$ZERO_BANK + (unsigned long)video_string_ptr;
		while (video_string_ptr + video_string_len < (__u8 *)KERNEL$ZERO_BANK + 0x110000 && (unsigned char)video_string_ptr[video_string_len] >= ' ') video_string_len++;
	}
	if (video_string_len >= __MAX_STR_LEN) video_string_len = __MAX_STR_LEN - 1;
	video_mode_ptr = (__u16 *)(bios_id.VideoModePtr_Off + (bios_id.VideoModePtr_Seg << 4));
	if ((char *)video_mode_ptr >= (char *)(unsigned long)BIOS16$REAL_OFFSET && (char *)video_mode_ptr < (char *)(unsigned long)BIOS16$REAL_OFFSET + sizeof bios_id) video_mode_ptr = (__u16 *)((char *)video_mode_ptr + (unsigned long)&bios_id - BIOS16$REAL_OFFSET);
	else video_mode_ptr = (__u16 *)(KERNEL$ZERO_BANK + (unsigned long)video_mode_ptr);
	if (bios_id.TotalMemory) {
		if (bios_id.TotalMemory & 15) {
			_snprintf(mem_str, sizeof mem_str, " (%ukB)", bios_id.TotalMemory * 64);
		} else {
			_snprintf(mem_str, sizeof mem_str, " (%uMB)", bios_id.TotalMemory / 16);
		}
		videoram = bios_id.TotalMemory * 65536;
	} else mem_str[0] = 0;
	__critical_printf("VESA %d.%d%s%s%.*s\n", version_major, version_minor, mem_str, video_string_len ? ": " : "", (int)video_string_len, video_string_ptr);

	if (version_major <= 1 && contains((char *)video_string_ptr, video_string_len, "TRIDENT MICROSYSTEMS")) trident_scroll_bug = 1;	/* in Trident BIOS from 1994 */
	if (contains((char *)video_string_ptr, video_string_len, "Trio64")) trio64_set_mode_bug = 1;	/* from Allegro, I actually never tried it, because I don't have Trio64 */

	if (!(vm = add_video_mode())) return -1;
	vm->mode = can_do_color ? 3 : 7;
	vm->x = cols;
	vm->y = rows;
	vm->bpp = 16;
	vm->scanline = vm->x * 2;
	vm->memsize = __PAGE_CLUSTER_SIZE;
	vm->pages = 1;
	vm->images = 1;
	vm->winnum_start = 0;
	vm->winnum_end = 0;
	vm->winshift = 0;
	vm->text = 1;
	vm->vga_compatible = 1;
	vm->phys = (unsigned long)base;
	vm->total_mem = videoram;
	while (*video_mode_ptr != 0xffff) {
		__u16 mode = *video_mode_ptr++;
		struct ModeInfoBlock mode_info;
		unsigned x, y, bpp, scanline;
		int text;
		int vga_compatible;
		unsigned pages;
		int winnum_start, winnum_end, winshift;
		unsigned memsize, memsize_roundup;
		unsigned images;
		__p_addr phys;
		if (mode >= 0x2000) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INVALID VIDEOMODE %04X", mode);
			break;
		}
		memset(&regs, 0, sizeof regs);
		regs.eax = 0x4f01;
		regs.ecx = mode;
		regs.es = BIOS16$REAL_SEGMENT;
		regs.edi = BIOS16$REAL_OFFSET;
		memset(&mode_info, 0, sizeof mode_info);
		if (BIOS_REAL_INT(0x10, &regs, &mode_info, sizeof mode_info, BIOS_DATA_PUT_GET)) continue;
		if ((regs.eax & 0xffff) != 0x004f) continue;
		if (!(mode_info.ModeAttributes & 0x1)) continue;	/* mode not supported by hardware */
		if (!(mode_info.ModeAttributes & 0x8)) continue;	/* monochrome mode */
		if (!(mode_info.ModeAttributes & 0x10)) text = 1;
		else text = 0;
		if (!(mode_info.ModeAttributes & 0x20)) vga_compatible = 1;
		else vga_compatible = 0;
		if (!(mode_info.ModeAttributes & 0x2)) {
			switch (mode) {
				case 0x100:
					x = 640;
					y = 400;
					bpp = 8;
					break;
				case 0x101:
					x = 640;
					y = 480;
					bpp = 8;
					break;
				case 0x103:
					x = 800;
					y = 600;
					bpp = 8;
					break;
				case 0x105:
					x = 1024;
					y = 768;
					bpp = 8;
					break;
				case 0x107:
					x = 1280;
					y = 1024;
					bpp = 8;
					break;
				case 0x108:
					x = 80;
					y = 60;
					bpp = 0;
					break;
				case 0x109:
					x = 132;
					y = 25;
					bpp = 0;
					break;
				case 0x10a:
					x = 132;
					y = 43;
					bpp = 0;
					break;
				case 0x10b:
					x = 132;
					y = 50;
					bpp = 0;
					break;
				case 0x10c:
					x = 132;
					y = 60;
					bpp = 0;
					break;
				case 0x10d:
					x = 320;
					y = 200;
					bpp = 15;
					break;
				case 0x10e:
					x = 320;
					y = 200;
					bpp = 16;
					break;
				case 0x10f:
					x = 320;
					y = 200;
					bpp = 24;
					break;
				case 0x110:
					x = 640;
					y = 480;
					bpp = 15;
					break;
				case 0x111:
					x = 640;
					y = 480;
					bpp = 16;
					break;
				case 0x112:
					x = 640;
					y = 480;
					bpp = 24;
					break;
				case 0x113:
					x = 800;
					y = 600;
					bpp = 15;
					break;
				case 0x114:
					x = 800;
					y = 600;
					bpp = 16;
					break;
				case 0x115:
					x = 800;
					y = 600;
					bpp = 24;
					break;
				case 0x116:
					x = 1024;
					y = 768;
					bpp = 15;
					break;
				case 0x117:
					x = 1024;
					y = 768;
					bpp = 16;
					break;
				case 0x118:
					x = 1024;
					y = 768;
					bpp = 24;
					break;
				case 0x119:
					x = 1280;
					y = 1024;
					bpp = 15;
					break;
				case 0x11a:
					x = 1280;
					y = 1024;
					bpp = 16;
					break;
				case 0x11b:
					x = 1280;
					y = 1024;
					bpp = 24;
					break;
				default:
					continue;
			}
			if (!bpp != text) continue;
			if (!bpp) bpp = 16;
			scanline = mode_info.BytesPerScanLine;
			if (!scanline) scanline = x * ((bpp + 7) >> 3);
			images = 1;
			pages = (images * scanline * y + 65535) >> 16;
		} else {
			x = mode_info.XResolution;
			y = mode_info.YResolution;
			if (!x || !y) continue;
			if (!text) bpp = mode_info.BitsPerPixel;
			else bpp = 16;
			if (!text && mode_info.ModeAttributes & 0x80 && mode_info.PhysBasePtr) pages = 1;
			else if (!(mode_info.ModeAttributes & 0x40)) pages = 0;
			else continue;
			if (!pages || version_major < 3 || !mode_info.LinBytesPerScanLine) scanline = mode_info.BytesPerScanLine;
			else scanline = mode_info.LinBytesPerScanLine;
				/* Bug in Trident VESA BIOS 1.2 */
			if (x == 320 && y == 200 && bpp == 24 && scanline == 1024 && version_major <= 1 && contains((char *)video_string_ptr, video_string_len, "TRIDENT MICROSYSTEMS")) goto get_sc;
			if (!scanline) get_sc: scanline = x * ((bpp + 7) >> 3);
			if (!text && mode_info.NumberOfBanks > 1) continue;
			if (text && !mode_info.MemoryModel) {
			} else if (mode_info.MemoryModel == 4) {
				if (bpp != 8 && bpp != 15 && bpp != 16 && bpp != 24 && bpp != 32) continue;
			} else if (mode_info.MemoryModel == 6) {
				__u8 rms, rfp, gms, gfp, bms, bfp;
				if (!pages || version_major < 3 || (!mode_info.LinRedMaskSize && !mode_info.LinGreenMaskSize && !mode_info.LinBlueMaskSize)) {
					rms = mode_info.RedMaskSize;
					rfp = mode_info.RedFieldPosition;
					gms = mode_info.GreenMaskSize;
					gfp = mode_info.GreenFieldPosition;
					bms = mode_info.BlueMaskSize;
					bfp = mode_info.BlueFieldPosition;
				} else {
					rms = mode_info.LinRedMaskSize;
					rfp = mode_info.LinRedFieldPosition;
					gms = mode_info.LinGreenMaskSize;
					gfp = mode_info.LinGreenFieldPosition;
					bms = mode_info.LinBlueMaskSize;
					bfp = mode_info.LinBlueFieldPosition;
				}
				switch (bpp) {
					case 15:
						if (rms != 5 || rfp != 10 || gms != 5 || gfp != 5 || bms != 5 || bfp != 0) continue;
						break;
					case 16:
						if (!(rms != 5 || rfp != 10 || gms != 5 || gfp != 5 || bms != 5 || bfp != 0)) {
							bpp = 15;
							break;
						}
						if (rms != 5 || rfp != 11 || gms != 6 || gfp != 5 || bms != 5 || bfp != 0) continue;
						break;
					case 24:
					case 32:
						if (rms != 8 || rfp != 16 || gms != 8 || gfp != 8 || bms != 8 || bfp != 0) continue;
						break;
					default:
						continue;
				}
			} else continue;
			if (text) {
				images = 1;
			} else if (version_major >= 3) {
				if (!pages) images = mode_info.BnkNumberOfImagePages + 1;
				else images = mode_info.LinNumberOfImagePages + 1;
			} else {
				images = mode_info.NumberOfImagePages + 1;
			}
		}
		if (videoram && images * y * scanline > videoram) {
			/*__debug_printf("images overflow...\n");*/
			if (y * scanline > videoram) continue;
			/*__debug_printf("fixed...\n");*/
			images = videoram / (y * scanline);
		}
		if (videoram && trident_scroll_bug && images > 1 && (images - 1) * y * scanline * ((bpp + 7) >> 3) >= videoram) {
			/* weird bug: Trident needs to have display start
			   mupltiplied by pixel size --- BUT: if the result is
			   larger that videoram size, it doesn't adjust window
			*/
			images = (videoram - 1) / ((bpp + 7) >> 3) / y / scanline + 1;
		}
		if (!pages) {
			if ((mode_info.WinAAttributes & 6) == 6 && mode_info.WinASegment) phys = mode_info.WinASegment << 4, winnum_start = winnum_end = 0;
			else if ((mode_info.WinBAttributes & 6) == 6 && mode_info.WinBSegment) phys = mode_info.WinBSegment << 4, winnum_start = winnum_end = 1;
			else if (((mode_info.WinAAttributes | mode_info.WinBAttributes) & 6) == 6 && mode_info.WinASegment && mode_info.WinASegment == mode_info.WinBSegment) phys = mode_info.WinASegment << 4, winnum_start = 0, winnum_end = 1;
			else continue;
			if (!text) {
				if (mode_info.WinSize < 64) continue;
				if (!mode_info.WinGranularity || (mode_info.WinGranularity & (mode_info.WinGranularity - 1)) || mode_info.WinGranularity > 64) continue;
				winshift = 6 - __BSR(mode_info.WinGranularity);
				memsize = 65536;
			} else {
				winshift = 0;
				memsize = 32768;
			}
			memsize = (memsize + __PAGE_CLUSTER_SIZE_MINUS_1) & __NOT_PAGE_CLUSTER_SIZE_MINUS_1;
			pages = (images * y * scanline + 65535) >> 16;
		} else {
			winnum_start = winnum_end = 0;
			winshift = 0;
			phys = mode_info.PhysBasePtr;
			memsize = images * y * scanline;
			memsize = (memsize + __PAGE_CLUSTER_SIZE_MINUS_1) & __NOT_PAGE_CLUSTER_SIZE_MINUS_1;
		}
	/* memory below 1M is handled by MTRRs anyway --- do not switch to 4k
	   pages because of PAT mode settings. CPU has a special code to handle
	   different mappings in the lowest big page */
		memsize_roundup = memsize;
		if (phys >= 0x100000) {
	/* round up to fit MTRRs */
			memsize_roundup = 2 << __BSR(memsize - 1);
			if (phys & (memsize_roundup - 1)) memsize_roundup = memsize;
		}
		if (KERNEL$SET_MAPPED_MEMORY(phys, memsize_roundup, phys < 0x100000 ? -1 : PAT_WC)) {
			continue;
		}
		if (text) continue;
		if ((vm = add_video_mode())) {
			/*__debug_printf("mode %dx%dx%d (scanline %d, pg %d, img %d) attr: %04x\n", x, y, bpp, scanline, pages, images, mode_info.ModeAttributes);*/
			vm->mode = mode;
			vm->x = x;
			vm->y = y;
			vm->bpp = bpp;
			vm->scanline = scanline;
			vm->pages = pages;
			vm->images = images;
			vm->winnum_start = winnum_start;
			vm->winnum_end = winnum_end;
			vm->winshift = winshift;
			vm->memsize = memsize;
			vm->text = text;
			vm->vga_compatible = vga_compatible;
			vm->phys = phys;
			vm->total_mem = videoram;
		}
	}
	for (i = 0; i < n_video_modes; i++) if (video_modes[i].x == 320 && video_modes[i].y == 200 && video_modes[i].bpp == 8) goto skip_vga_320_200_8;
	if ((vm = add_video_mode())) {
		vm->mode = 0x13;
		/*__debug_printf("default mode %dx%dx%d\n", 320, 200, 8);*/
		vm->x = 320;
		vm->y = 200;
		vm->bpp = 8;
		vm->scanline = vm->x;
		vm->memsize = (vm->y * vm->scanline + __PAGE_CLUSTER_SIZE_MINUS_1) & __NOT_PAGE_CLUSTER_SIZE_MINUS_1;
		vm->pages = 1;
		vm->images = 1;
		vm->winnum_start = 0;
		vm->winnum_end = 0;
		vm->winshift = 0;
		vm->text = 0;
		vm->vga_compatible = 1;
		vm->phys = 0xa0000;
		vm->total_mem = videoram;
	}
	skip_vga_320_200_8:
	if (n_video_modes > 1) vesa_available = 1;
	return 0;
}

static char locked[MAX_VCS] = { 0 };

static struct tty_videomode_params vc_videomodes[MAX_VCS] = { { 0 } };
static struct tty_videomode_params current_videomode = { 0 };
static unsigned current_page = 0;
static __u32 current_disp_start = 0;

static long VESA_VIDEOMODE_CALL(int cmd, int vc, ...);

int VESA_VIDEOMODE_DCALL(void *ptr, const char *dcall_type, int cmd, va_list list)
{
	int r;
	unsigned i;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TTY);
	if (__likely(cmd >= CONSOLE_CMD_GET) && __likely(cmd < CONSOLE_CMD_GET + vcs)) {
		if (__unlikely(locked[cmd - CONSOLE_CMD_GET])) {
			r = -EBUSY;
			goto ret;
		}
		locked[cmd - CONSOLE_CMD_GET] = 1;
		*va_arg(list, videomode_call_t **) = VESA_VIDEOMODE_CALL;
		r = 0;
	} else if (__likely(cmd == CONSOLE_CMD_INSTALL_DRIVER)) {
		struct driver_function *dp;
		if (__unlikely(driver_installed)) {
			r = -EBUSY;
			goto ret;
		}
		dp = va_arg(list, struct driver_function *);
		while (dp->name) {
		 	for (i = 0; driver_functions[i].name; i++) if (!_strcasecmp(driver_functions[i].name, dp->name)) {
				*(void **)driver_functions[i].ptr = dp->ptr;
				goto found_func;
			}
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, dev_name, "SKIPPING DRIVER FUNCTION %s. YOU SHOULD USE NEWER VERSION OF PCCONS.SYS", dp->name);
			found_func:
			dp++;
		}
		driver_installed = 1;
		r = 0;
	} else if (__likely(cmd == CONSOLE_CMD_RELEASE_DRIVER)) {
		if (__unlikely(!driver_installed))
			KERNEL$SUICIDE("VESA_VIDEOMODE_DCALL: TRYING TO UNINSTALL DRIVER AND NO DRIVER INSTALLED");

		if (DRIVER_ACCEL_EXIT) {
			DRIVER_ACCEL_EXIT(&video_modes[current_videomode.vm]);
		}
		for (i = 0; driver_functions[i].name; i++) *(void **)driver_functions[i].ptr = NULL;
		driver_installed = 0;
		r = 0;
	} else {
		r = -EINVAL;
	}
	ret:
	LOWER_SPLX(spl);
	return r;
}

	/* 0 --- switch allowed
	   1 --- console locked
	   2 --- console locked and release timeout running
	   3 --- timeout run out, switch allowed, can't lock again until,
	   	cleared by vc switch
	*/
static int locked_vc = 0;
static DECL_TIMER(locked_timeout);
static WQ_DECL(locked_timeout_wait, "VESA$LOCKED_TIMEOUT_WAIT");

static unsigned video_memory_valid_to = 0;

static WQ *UNMAP_PAGES(void);
static int ZERO_VIDEORAM(IORQ *rq);
static int VESA_SET_PALETTE(__u8 *palette, int idx, int num);
static void VESA_SET_PAGE(unsigned page);
static void VESA_RESTORE_VIDEOMODE(void);
static void CLEAR_VIDEORAM(void);
static int VESA_SET_DISPLAY_START(unsigned x, int waitretrace);
static void GET_PM_INTERFACE(void);
static int VESA_WAITRETRACE(void);
static long VESA_GETPIXELCLOCK(unsigned vm, unsigned long px);

static long VESA_VIDEOMODE_CALL(int cmd, int vc, ...)
{
	va_list args;
	int i;
	struct tty_videomode_params *m;
	struct video_mode *v;
	IORQ *waitrq;
	WQ *wq;
	if (__unlikely(!vesa_available)) {
		long rr;
		switch (cmd) {
			case VIDEOMODE_GET_FONT_MODE:
			case VIDEOMODE_SET_FONT:
			case VIDEOMODE_GET_DEFAULT_CHARSET:
			pccons_ioctl:
				va_start(args, vc);
				rr = PCCONS_VIDEOMODE(cmd, vc, args);
				va_end(args);
				return rr;
			case VIDEOMODE_SET_MODE:
			case VIDEOMODE_GET_PAGE:
				return (long)__ERR_PTR(-ENXIO);
			default:
				return -ENXIO;
		}
	}
	if (__likely(cmd == VIDEOMODE_DO_ACCEL)) {
		unsigned op;
		union accel_param *ap;
		if (__unlikely(vc != avc)) return (long)__ERR_PTR(-EBUSY);
		v = video_modes + current_videomode.vm;
		if (__unlikely(v->text) || __unlikely(v->mode < 0x100)) return (long)__ERR_PTR(-ENXIO);
		if (__unlikely(!DRIVER_ACCEL_DO)) return (long)__ERR_PTR(-ENXIO);
		va_start(args, vc);
		op = va_arg(args, unsigned);
		ap = va_arg(args, union accel_param *);
		waitrq = va_arg(args, IORQ *);
		va_end(args);
		if (__unlikely(video_memory_valid_to != MAXUINT)) {
			if (__unlikely(ZERO_VIDEORAM(waitrq))) return 1;
		}
		wq = DRIVER_ACCEL_DO(v, op, ap);
		if (__likely(!wq)) return 0;
		if (__IS_ERR(wq)) {
			waitrq->status = __PTR_ERR(wq);
			CALL_AST(waitrq);
		} else {
			WQ_WAIT_F(wq, waitrq);
			LOWER_SPL(SPL_TTY);
		}
		return 1;
	}
	switch (cmd) {
		case VIDEOMODE_RELEASE: {
			locked[vc] = 0;
			return 0;
		}

				/* ^VIDEOMODE=G640X480X8 */
		case VIDEOMODE_GET_MODE: {
			char *str, *x1, *x2;
			long x, y, bpp;
			va_start(args, vc);
			str = va_arg(args, char *);
			va_end(args);
			if (__upcasechr(*str) != 'G') return -EINVAL;
			str++;
			x1 = str;
			while (__upcasechr(*x1) != 'X') if (__unlikely(!*x1++)) return -EINVAL;
			x2 = x1 + 1;
			while (__upcasechr(*x2) != 'X') if (__unlikely(!*x2++)) return -EINVAL;
			if (__get_number(str, x1, 0, &x)) return -EINVAL;
			if (__get_number(x1 + 1, x2, 0, &y)) return -EINVAL;
			if (__get_number(x2 + 1, x2 + strlen(x2), 0, &bpp)) return -EINVAL;
			for (i = 0; i < n_video_modes; i++) {
				if (video_modes[i].x == x && video_modes[i].y == y && video_modes[i].bpp == bpp && !video_modes[i].text) return i;
			}
			return -ENXIO;
		}
		case VIDEOMODE_SET_MODE: {
			struct tty_videomode_params om;
			va_start(args, vc);
			m = va_arg(args, struct tty_videomode_params *);
			va_end(args);
			if (__unlikely(m->vm < 0) || __unlikely(m->vm >= n_video_modes)) return (unsigned long)__ERR_PTR(-ENXIO);
			om = vc_videomodes[vc];
			vc_videomodes[vc] = *m;
			if (__likely(vc == avc) && __unlikely((wq = VESA_SET_VIDEOMODE(vc)) != NULL)) {
				vc_videomodes[vc] = om;
				return (unsigned long)wq;
			}
			if (__likely(vc == avc)) CONS_DO_UPDATE();
			return (unsigned long)NULL;
		}
		case VIDEOMODE_GET_PAGE: {
			__v_off idx;
			PAGE *pg;
			if (__unlikely(vc != avc)) return (long)__ERR_PTR(-EBUSY);
			va_start(args, vc);
			idx = va_arg(args, __v_off);
			va_end(args);
			v = video_modes + current_videomode.vm;
			if (__unlikely(v->text)) return (long)__ERR_PTR(-ENXIO);
			if (__likely(v->pages == 1)) {
				if (__unlikely(idx >= v->memsize)) return (long)__ERR_PTR(-ERANGE);
				pg = KERNEL$PHYS_2_PAGE(v->phys + idx);
			} else {
				unsigned page;
				if (__unlikely(idx >= v->memsize * v->pages)) return (long)__ERR_PTR(-ERANGE);
				page = (unsigned)idx >> 16;
				if (page != current_page) {
					wq = UNMAP_PAGES();
					if (__unlikely(wq != NULL)) {
						LOWER_SPL(SPL_TTY);
						return NULL;
					}
					VESA_SET_PAGE(page);
					current_page = page;
				}
				pg = KERNEL$PHYS_2_PAGE(v->phys + ((unsigned)idx & 0xffff));
			}
			if (__unlikely(idx >= video_memory_valid_to)) {
				return NULL;
			}
			return (long)pg;
		}
		case VIDEOMODE_GET_PARAM: {
			struct tty_videomode *tvm;
			va_start(args, vc);
			m = va_arg(args, struct tty_videomode_params *);
			tvm = va_arg(args, struct tty_videomode *);
			va_end(args);
			if (__unlikely(m->vm < 0) || __unlikely(m->vm >= n_video_modes)) return -ENXIO;
			v = video_modes + m->vm;
			memset(tvm, 0, sizeof(struct tty_videomode));
			tvm->x = v->x;
			tvm->y = v->y;
			tvm->bpp = v->bpp;
			tvm->scanline = v->scanline;
			tvm->memsize = v->memsize * v->pages;
			tvm->pages = v->pages;
			tvm->images = v->images;
			tvm->text = v->text;
			tvm->phys = v->phys;
			return 0;
		}
		case VIDEOMODE_WAITACTIVE: {
			int waitmode;
			va_start(args, vc);
			waitmode = va_arg(args, int);
			waitrq = va_arg(args, IORQ *);
			va_end(args);
			switch (waitmode) {
				case PARAM_WAIT_FOR_INACTIVE:
					if (vc == avc) {
						wsw:
						WQ_WAIT_F(&vc_switch_wait, waitrq);
						return 1;
					}
					return 0;
				case PARAM_WAIT_FOR_ACTIVE:
					if (vc != avc) goto wsw;
					return 0;
				case PARAM_WAIT_FOR_ACTIVE_AND_LOCK:
					if (vc != avc) goto wsw;
					if (__unlikely(locked_vc >= 2))
						return -EBUSY;
					locked_vc = 1;
					return 0;
				case PARAM_WAIT_FOR_RELEASE_REQUEST:
					if (__unlikely(vc != avc)) return -EBUSY;
					if (__unlikely(!locked_vc)) locked_vc = 1;
					if (locked_vc == 2) return 0;
					WQ_WAIT_F(&locked_timeout_wait, waitrq);
					return 1;
				case PARAM_RELEASE:
					if (locked_vc) {
						if (locked_vc == 2) KERNEL$DEL_TIMER(&locked_timeout);
						locked_vc = 3;
						WQ_WAKE_ALL_PL(&locked_timeout_wait);
					}
					return 0;
				default:
					return -EINVAL;
			}
		}
		case VIDEOMODE_WAITACTIVE_WAKE: {
			WQ_WAKE_ALL(&vc_switch_wait);
			WQ_WAKE_ALL(&locked_timeout_wait);
			return 0;
		}
		case VIDEOMODE_SET_PALETTE: {
			__u8 *palette;
			int pidx, num;
			va_start(args, vc);
			palette = va_arg(args, __u8 *);
			pidx = va_arg(args, int);
			num = va_arg(args, int);
			va_end(args);
			if (__unlikely(vc != avc)) return 0;
			return VESA_SET_PALETTE(palette, pidx, num);
		}
		case VIDEOMODE_UNMAP_PAGES: {
			va_start(args, vc);
			waitrq = va_arg(args, IORQ *);
			va_end(args);
			return ZERO_VIDEORAM(waitrq);
		}
		case VIDEOMODE_SET_DISPLAY_START:
		case VIDEOMODE_WAITRETRACE_SET_DISPLAY_START: {
			int disp;
			va_start(args, vc);
			disp = va_arg(args, int);
			va_end(args);
			if (__unlikely(vc != avc)) return 0;
			return VESA_SET_DISPLAY_START(disp, cmd == VIDEOMODE_WAITRETRACE_SET_DISPLAY_START);
		}
		case VIDEOMODE_WAITRETRACE: {
			va_start(args, vc);
			waitrq = va_arg(args, IORQ *);
			va_end(args);
			i = 0;
			if (__unlikely(vc != avc)) goto ret;
			i = VESA_WAITRETRACE();
			ret:
			waitrq->status = i;
			CALL_AST(waitrq);
			return 0;
		}
		case VIDEOMODE_GET_PIXEL_CLOCK: {
			unsigned long px;
			va_start(args, vc);
			m = va_arg(args, struct tty_videomode_params *);
			px = va_arg(args, unsigned long);
			va_end(args);
			return VESA_GETPIXELCLOCK(m->vm, px);
		}
		case VIDEOMODE_AVAIL_ACCEL: {
			int ac;
			va_start(args, vc);
			ac = va_arg(args, int);
			va_end(args);
			v = video_modes + vc_videomodes[vc].vm;
			if (__unlikely(v->text)) return -ENXIO;
			if (__unlikely(v->mode < 0x100)) return 0;
			if (!DRIVER_ACCEL_AVAIL) return 0;
			return DRIVER_ACCEL_AVAIL(v, ac);
		}
		case VIDEOMODE_CAN_MMAP: {
			v = video_modes + vc_videomodes[vc].vm;
			if (__unlikely(v->text)) return -ENXIO;
			return 0;
		}
		case VIDEOMODE_GET_FONT_MODE:
		case VIDEOMODE_SET_FONT:
		case VIDEOMODE_GET_DEFAULT_CHARSET: {
			goto pccons_ioctl;
		}
		default: {
			return -EINVAL;
		}
	}
}

__COLD_ATTR__ static void VC_SWITCH_TIMEOUT(TIMER *t)
{
	LOWER_SPL(SPL_TTY);
	locked_vc = 3;
	WQ_WAKE_ALL_PL(&locked_timeout_wait);
}

static WQ *UNMAP_PAGES(void)
{
	struct video_mode *v;
	unsigned m;
	WQ *wq;
	v = video_modes + current_videomode.vm;
	for (m = 0; m < v->memsize; m += __PAGE_CLUSTER_SIZE) {
		if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(KERNEL$PHYS_2_PAGE(v->phys + m))) != NULL)) return wq;
		LOWER_SPL(SPL_TTY);
	}
	return NULL;
}

static int ZERO_VIDEORAM(IORQ *rq)
{
	struct video_mode *v;
	WQ *wq = UNMAP_PAGES();
	if (__unlikely(wq != NULL)) {
		WQ_WAIT_F(wq, rq);
		LOWER_SPL(SPL_TTY);
		return 1;
	}
	v = video_modes + current_videomode.vm;
	while (video_memory_valid_to < v->memsize * v->pages) {
		char *mem;
		PAGE *pg;
		if (v->pages == 1) {
			pg = KERNEL$PHYS_2_PAGE(v->phys + video_memory_valid_to);
		} else {
			unsigned page = video_memory_valid_to >> 16;
			VESA_SET_PAGE(page);
			current_page = page;
			pg = KERNEL$PHYS_2_PAGE(v->phys + (video_memory_valid_to & 0xffff));
		}
		mem = KERNEL$MAP_PHYSICAL_PAGE(pg);
		memset(mem, 0, __PAGE_CLUSTER_SIZE);
		KERNEL$UNMAP_PHYSICAL_BANK(mem);
		video_memory_valid_to += __PAGE_CLUSTER_SIZE;
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS)) {
			rq->status = -EINTR;
			CALL_AST(rq);
			return 1;
		}
	}
	video_memory_valid_to = MAXUINT;
	return 0;
}

__COLD_ATTR__ static u_jiffies_t GET_SWITCH_TIMEOUT(void)
{
	__u64 num;
	char *e;
	if (__likely(!(e = getenv("@KERNEL$VC_SWITCH_TIMEOUT")))) return SWITCH_TIMEOUT;
	if (__unlikely(__get_64_number(e, e + strlen(e), 0, (__s64 *)&num))) return SWITCH_TIMEOUT;
	if (__unlikely(num > (u_jiffies_t)-1 / JIFFIES_PER_SECOND)) return (u_jiffies_t)-1;
	return (u_jiffies_t)num * JIFFIES_PER_SECOND;
}

__COLD_ATTR__ static int INVALID_TIMING(unsigned x, unsigned y, struct tty_videomode_params *m)
{
	unsigned v;
	unsigned v_refresh, h_refresh;
	if (__unlikely(!m->pixel_clock)) return 0;
	if (__unlikely(x > 10000) || __unlikely(y > 10000)) return -1;
	if (__unlikely(m->h != x)) return -1;
	if (m->flags & VIDEO_PARAMS_DOUBLE_SCAN) y <<= 1;
	if (__unlikely(y <= 300)) return -1;
	v = m->v;
	if (__unlikely(m->flags & VIDEO_PARAMS_INTERLACED)) v <<= 1;
	if (__unlikely(v != y)) return -1;
	if (__unlikely(m->h_sync_start <= m->h)) return -1;
	if (__unlikely(m->h_sync_end <= m->h_sync_start)) return -1;
	if (__unlikely(m->h_total <= m->h_sync_end)) return -1;
	if (__unlikely(m->h * 2 <= m->h_total)) return -1;
	if (__unlikely(m->v_sync_start <= m->v)) return -1;
	if (__unlikely(m->v_sync_end <= m->v_sync_start)) return -1;
	if (__unlikely(m->v_total <= m->v_sync_end)) return -1;
	if (__unlikely(m->v * 2 <= m->v_total)) return -1;
	if (__unlikely(!(m->h_total * m->v_total))) return -1;
	v_refresh = m->pixel_clock / (m->h_total * m->v_total);
	h_refresh = m->pixel_clock / m->h_total;
	if (__unlikely(v_refresh < 45)) return -1;
	if (__unlikely(h_refresh < 25000)) return -1;
	if (max_v_refresh && v_refresh > max_v_refresh) return -1;
	if (max_h_refresh && (__u64)v_refresh * 1000 > max_h_refresh) return -1;
	return 0;
}

__COLD_ATTR__ WQ *VESA_SET_VIDEOMODE(int vc)
{
	int vm;
	struct tty_videomode_params *m;
	struct bios_regs regs;
	unsigned mode;
	WQ *wq;
	if (__unlikely(!vesa_available)) goto ret;
	if (__unlikely((wq = UNMAP_PAGES()) != NULL)) return wq;
	if (DRIVER_ACCEL_DO && !video_modes[current_videomode.vm].text && video_modes[current_videomode.vm].mode >= 0x100) {
		DRIVER_ACCEL_DO(video_modes + current_videomode.vm, ACCEL_SYNC, NULL);
	}
	if (vc != avc) {
		if (locked_vc == 1) {
			locked_vc = 2;
			locked_timeout.fn = VC_SWITCH_TIMEOUT;
			KERNEL$SET_TIMER(GET_SWITCH_TIMEOUT(), &locked_timeout);
			WQ_WAKE_ALL(&locked_timeout_wait);
			return &locked_timeout_wait;
		}
		if (locked_vc == 2) {
			return &locked_timeout_wait;
		}
		if (locked_vc == 3) {
			locked_vc = 0;
		}
		WQ_WAKE_ALL(&locked_timeout_wait);
	}
	m = &vc_videomodes[vc];
	vm = m->vm;
	if (__unlikely(memcmp(m, &current_videomode, sizeof(struct tty_videomode_params)))) {
		int timing_failed = 0;
		if (!video_modes[vm].text) {
			console_gfx = 1;
			if (on_kernel_cons)
				KERNEL$CONSOLE_SET_GRAPHICS(VESA_RESTORE_VIDEOMODE);
		}
		if (DRIVER_ACCEL_EXIT) {
			DRIVER_ACCEL_EXIT(&video_modes[current_videomode.vm]);
		}
		if (__unlikely(INVALID_TIMING(video_modes[vm].x, video_modes[vm].y, m))) timing_failed = 1;
		if (__unlikely(trio64_set_mode_bug) && !video_modes[vm].text && video_modes[current_videomode.vm].text) {
			memset(&regs, 0, sizeof regs);
			regs.eax = 0x0013;
			BIOS_REAL_INT(0x10, &regs, NULL, 0, 0);
		}
		retry:
		memset(&regs, 0, sizeof regs);
		regs.eax = 0x4f02;
		mode = video_modes[vm].mode;
		if (mode < 256) regs.ebx = mode;
			/* | 0x8000 --- this should disable videoram clearing,
			   but Matrox VESA BIOS can't setup non-vesa videomode
			   with this flag */
		else regs.ebx = mode | (video_modes[vm].pages == 1 ? 0x4000 : 0x0000) | 0x8000;
		if (!m->pixel_clock || mode < 256 || version_major < 3 || __unlikely(timing_failed) || DRIVER_SET_TIMING) {
			BIOS_REAL_INT(0x10, &regs, NULL, 0, 0);
			if (m->pixel_clock && DRIVER_SET_TIMING && __likely(!timing_failed)) {
				DRIVER_SET_TIMING(&video_modes[vm], m);
			}
		} else {
			unsigned v_freq_100;
			struct CRTCInfoBlock ci;
			memset(&ci, 0, sizeof ci);
			ci.HorizontalTotal = m->h_total;
			ci.HorizontalSyncStart = m->h_sync_start;
			ci.HorizontalSyncEnd = m->h_sync_end;
			ci.VerticalTotal = m->v_total;
			ci.VerticalSyncStart = m->v_sync_start;
			ci.VerticalSyncEnd = m->v_sync_end;
			ci.Flags = m->flags & 0xf;
			ci.PixelClock0 = m->pixel_clock;
			ci.PixelClock1 = m->pixel_clock >> 8;
			ci.PixelClock2 = m->pixel_clock >> 16;
			ci.PixelClock3 = m->pixel_clock >> 24;
			if (__likely(m->h_total * m->v_total <= MAXUINT / 100)) v_freq_100 = m->h_total * m->v_total / m->pixel_clock;
			else v_freq_100 = m->h_total * m->v_total / (m->pixel_clock / 100);
			ci.RefreshRate0 = v_freq_100;
			ci.RefreshRate1 = v_freq_100 >> 8;
			regs.ebx |= 0x0800;
			regs.es = BIOS16$REAL_SEGMENT;
			regs.edi = BIOS16$REAL_OFFSET;
			BIOS_REAL_INT(0x10, &regs, &ci, sizeof ci, BIOS_DATA_PUT);
			if (__unlikely((regs.eax & 0xffff) != 0x004f)) {
/* if the timing is incorrect, new videomode will be set and error returned on
   Intel chipset */
				timing_failed = 1;
				goto retry;
			}
		}
		if (__unlikely((regs.eax & 0xffff) != 0x004f)) {
			avc = vc;
	/* We don't know what mode are we in now.
	   Retry videomode setting at next console switch */
			current_videomode.pixel_clock = -1;
			return __ERR_PTR(-ENXIO);
		}
		current_videomode = *m;
		if (video_modes[vm].text) {
			console_gfx = 0;
			if (on_kernel_cons)
				KERNEL$CONSOLE_SET_GRAPHICS(NULL);
		}
		if (DRIVER_ACCEL_INIT && !video_modes[vm].text && __likely(video_modes[vm].mode >= 0x100)) {
			DRIVER_ACCEL_INIT(&video_modes[vm]);
		}
		if (locked_vc == 2) {
			KERNEL$DEL_TIMER(&locked_timeout);
		}
		locked_vc = 0;
		WQ_WAKE_ALL(&locked_timeout_wait);
		GET_PM_INTERFACE();
		current_page = 0;
		current_disp_start = 0;
		vga_retrace_errors = 0;
		CLEAR_VIDEORAM();
		VESA_SET_PAGE(0);
		CONS_SET_FONT_VT_SWITCH(-1, vc);
		avc = vc;
	} else if (__likely(vc != avc)) {
		CLEAR_VIDEORAM();
		VESA_SET_PAGE(0);
		VESA_SET_DISPLAY_START(0, 0);
		ret:
		CONS_SET_FONT_VT_SWITCH(avc, vc);
		avc = vc;
	}
	return NULL;
}

__COLD_ATTR__ static long VESA_GETPIXELCLOCK(unsigned vm, unsigned long px)
{
	int r;
	struct bios_regs regs;
	if (__unlikely(px > MAXINT)) return -ERANGE;
	if (DRIVER_GET_PIXEL_CLOCK) {
		return DRIVER_GET_PIXEL_CLOCK(&video_modes[vm], px);
	}
	if (__unlikely(version_major < 3)) return -ENXIO;
	memset(&regs, 0, sizeof regs);
	regs.eax = 0x4f0b;
	regs.ebx = 0x0000;
	regs.ecx = px;
	regs.edx = video_modes[vm].mode | (video_modes[vm].mode >= 256 && video_modes[vm].pages == 1 ? 0x4000 : 0x0000);
	if (__unlikely(r = BIOS_REAL_INT(0x10, &regs, NULL, 0, 0))) return r;
	if (__unlikely((regs.eax & 0xffff) != 0x004f)) return -ENXIO;
	if (__unlikely((long)regs.ecx < 0)) return -EIO;
	return regs.ecx;
}

__COLD_ATTR__ static void VESA_RESTORE_VIDEOMODE(void)
{
	struct bios_regs regs;
	/* If the interrupt triggers second suicide, try to write it */
	console_gfx = 0;
	KERNEL$CONSOLE_SET_GRAPHICS(NULL);
	memset(&regs, 0, sizeof regs);
	regs.eax = 0x4f02;
	regs.ebx = 0x0003;
	BIOS_REAL_INT(0x10, &regs, NULL, 0, 0);
	GET_PM_INTERFACE();
}

static void *PM_SET_PALETTE = NULL;
static void *PM_SET_PAGE = NULL;
static void *PM_SET_DISPLAY_START = NULL;

static int VESA_SET_PALETTE(__u8 *palette, int idx, int num)
{
	struct video_mode *vm = video_modes + current_videomode.vm;
	struct bios_regs regs;
	if (__unlikely(vm->text)) return -ENXIO;
	if (__unlikely(vm->bpp != 8)) return -EINVAL;
	if (__likely(PM_SET_PALETTE != NULL)) {
		__u32 a, b, c, d, D;
		__asm__ volatile ("CALL *PM_SET_PALETTE":"=a"(a),"=b"(b),"=c"(c),"=d"(d),"=D"(D):"a"(0x4f09),"b"(0),"c"(num),"d"(idx),"D"(palette):"cc","memory");
		return 0;
	}
	if (__likely(version_major >= 2)) {
		int r;
		memset(&regs, 0, sizeof regs);
		regs.eax = 0x4f09;
		regs.ebx = 0x0000;
		regs.edx = idx;
		regs.ecx = num;
		regs.es = BIOS16$REAL_SEGMENT;
		regs.edi = BIOS16$REAL_OFFSET;
		if (__unlikely(r = BIOS_REAL_INT(0x10, &regs, palette, num * 4, BIOS_DATA_PUT))) {
			if (vm->vga_compatible) goto pio;
			return r;
		}
		if (__unlikely((regs.eax & 0xffff) != 0x004f)) {
			if (vm->vga_compatible) goto pio;
			return -ENXIO;
		}
		return 0;
	}
	pio:
	io_outb(VGA_DAC_WRITE_IDX, idx);
#if 0
	/* S3 VLB card hates the high data rate of io_outsb and corrupts
	 * palette. Maybe this branch could be enabled on new PCI cards,
	 * but how do we test for them here? */
	{
		int i;
		__u8 *p1, *p2;
		p1 = p2 = palette;
		for (i = 0; i < num; i++) {
			__u8 b = p2[0], g = p2[1], r = p2[2];
			p1[0] = r;
			p1[1] = g;
			p1[2] = b;
			p1 += 3;
			p2 += 4;
		}
		io_outsb(VGA_DAC_REG, palette, num * 3);
	}
#endif
	{
		int i;
		for (i = 0; i < num; i++) {
			io_outb(VGA_DAC_REG, palette[i * 4 + 2]);
			io_outb(VGA_DAC_REG, palette[i * 4 + 1]);
			io_outb(VGA_DAC_REG, palette[i * 4 + 0]);
		}
	}
	return 0;
}

static void VESA_SET_PAGE(unsigned page)
{
	int i;
	struct video_mode *vm = video_modes + current_videomode.vm;
	struct bios_regs regs;
	if (__unlikely(vm->text)) return;
	if (__unlikely(vm->pages == 1) || __unlikely(page >= vm->pages)) return;
	if (__likely(PM_SET_PAGE != NULL)) {
		i = vm->winnum_start;
		do {
			__u32 a, b, d;
			__asm__ volatile ("CALL *PM_SET_PAGE":"=a"(a),"=b"(b),"=d"(d):"a"(0x4f05),"b"(i),"d"(page << vm->winshift):"cx","di","cc");
		} while (__unlikely(++i <= vm->winnum_end));
		return;
	}
	i = vm->winnum_start;
	do {
		memset(&regs, 0, sizeof regs);
		regs.eax = 0x4f05;
		regs.ebx = i;
		regs.edx = page << vm->winshift;
		BIOS_REAL_INT(0x10, &regs, NULL, 0, 0);
	} while (__unlikely(++i <= vm->winnum_end));
}

static int VESA_SET_DISPLAY_START(unsigned x, int waitretrace)
{
	int r;
	struct video_mode *vm = video_modes + current_videomode.vm;
	struct bios_regs regs;
	unsigned bytespp;
	unsigned pix, lin;
#if __DEBUG >= 1
	if (__unlikely((unsigned)waitretrace > 1))
		KERNEL$SUICIDE("VESA_SET_DISPLAY_START: WAITRETRACE %d", waitretrace);
#endif
	if (__unlikely(vm->text)) return -ENXIO;
	if (__unlikely(x > vm->y * vm->scanline * (vm->images - 1))) {
		return -ERANGE;
	}
	if (DRIVER_SET_DISPLAY_START && __likely((r = DRIVER_SET_DISPLAY_START(vm, x, waitretrace)) >= 0)) {
		if (__unlikely(r)) VESA_WAITRETRACE();
		return 0;
	}
	if (__likely(PM_SET_DISPLAY_START != NULL)) {
		__u32 a, b, c, d;
		if (__unlikely(x & 3) && __unlikely(version_major < 3)) return -EINVAL;
		x = (x >> 2) | (x << 30);
		current_disp_start = x;
		__asm__ volatile ("CALL *PM_SET_DISPLAY_START":"=a"(a),"=b"(b),"=c"(c),"=d"(d):"a"(0x4f07),"b"(waitretrace << 7),"c"(x & 0xffff),"d"(x >> 16):"di","cc");
		return 0;
	}
	if (__likely(version_major >= 3)) {
		regs.ecx = x;
		regs.eax = 0x4f07;
		regs.ebx = 0x0002;
		if (__unlikely(r = BIOS_REAL_INT(0x10, &regs, NULL, 0, 0))) return r;
		if (__unlikely((regs.eax & 0xffff) != 0x004f)) return -ENXIO;
		return 0;
	}
	bytespp = (vm->bpp + 7) >> 3;
	/* the advantage of this bug is that it can do byte-level display start
	   setting. The disadvantage is that it limits scrolling in some modes
	   (x * bytespp must be < videoram) */
	if (__unlikely(trident_scroll_bug) && __unlikely(bytespp != 1)) x *= bytespp;
	pix = x % vm->scanline;
	lin = x / vm->scanline;
	if (__unlikely(bytespp == 1)) {
	} else if (bytespp == 2) {
		if (__unlikely(pix & 1)) return -EINVAL;
		pix >>= 1;
	} else if (__likely(bytespp == 4)) {
		if (__unlikely(pix & 3)) return -EINVAL;
		pix >>= 2;
	} else {
		if (__unlikely(pix % bytespp)) return -EINVAL;
		pix /= bytespp;
	}
	regs.ecx = pix;
	regs.edx = lin;
	regs.eax = 0x4f07;
	regs.ebx = 0x0000;
	if (__unlikely(waitretrace)) VESA_WAITRETRACE();
	if (__unlikely(r = BIOS_REAL_INT(0x10, &regs, NULL, 0, 0))) return r;
	if (__unlikely((regs.eax & 0xffff) != 0x004f)) return -ENXIO;
	return 0;
}

static int VESA_WAITRETRACE(void)
{
	__u32 a, b, c, d;
	struct video_mode *vm;
	vm = video_modes + current_videomode.vm;
	if (__unlikely(!vm->vga_compatible)) {
		if (__likely(PM_SET_DISPLAY_START != NULL)) {
			pm_wait:
/* on s3 Virge, this waits for two retraces, not one. Prefer using vga wait */
			__asm__ volatile ("CALL *PM_SET_DISPLAY_START":"=a"(a),"=b"(b),"=c"(c),"=d"(d):"a"(0x4f07),"b"(0x80),"c"(current_disp_start & 0xffff),"d"(current_disp_start >> 16):"di","cc");
			return 0;
		}
		return -ENXIO;
	}
	if (__unlikely(vga_retrace_errors > RETRACE_ERROR_TOLERANCE)) {
		if (__likely(PM_SET_DISPLAY_START != NULL)) goto pm_wait;
	}
	return cons_wait_for_v_retrace();
}

__COLD_ATTR__ static void CLEAR_VIDEORAM(void)
{
	struct video_mode *v = video_modes + current_videomode.vm;
	if (v->text) return;
	video_memory_valid_to = 0;
}

struct pm_info {
	__u16 set_page;
	__u16 set_display_start;
	__u16 set_palette;
	__u16 ports_mem;
};

__COLD_ATTR__ static void GET_PM_INTERFACE(void)
{
	unsigned long addr;
	struct pm_info *info;
	__u16 *p_m;
	struct bios_regs regs;
	PM_SET_PALETTE = PM_SET_PAGE = PM_SET_DISPLAY_START = NULL;
	if (__unlikely(version_major < 2)) return;
	memset(&regs, 0, sizeof regs);
	regs.eax = 0x4f0a;
	regs.ebx = 0;
	if (__unlikely(BIOS_REAL_INT(0x10, &regs, NULL, 0, 0))) return;
	if (__unlikely((regs.eax & 0xffff) != 0x004f)) return;
	addr = (regs.es << 4) + (regs.edi & 0xffff);
	if (__unlikely(addr < 0xa0000) || __unlikely(addr >= 0x100000)) return;
	info = (struct pm_info *)(KERNEL$ZERO_BANK + addr);
	if (__likely(!info->ports_mem)) goto skip_p_m;
	p_m = (__u16 *)((char *)info + info->ports_mem);
	do if (__unlikely((__u8 *)p_m >= KERNEL$ZERO_BANK + 0x0ffffd)) return; while (*p_m++ != 0xffff);
	if (__unlikely(*p_m != 0xffff)) return;
	skip_p_m:
	if (__likely(info->set_palette)) PM_SET_PALETTE = (char *)info + info->set_palette;
	if (__likely(info->set_page)) PM_SET_PAGE = (char *)info + info->set_page;
	if (__likely(info->set_display_start)) PM_SET_DISPLAY_START = (char *)info + info->set_display_start;
	if (__unlikely((__u8 *)PM_SET_PALETTE >= KERNEL$ZERO_BANK + 0x100000)) PM_SET_PALETTE = NULL;
	if (__unlikely((__u8 *)PM_SET_PAGE >= KERNEL$ZERO_BANK + 0x100000)) PM_SET_PAGE = NULL;
	if (__unlikely((__u8 *)PM_SET_DISPLAY_START >= KERNEL$ZERO_BANK + 0x100000)) PM_SET_DISPLAY_START = NULL;
}

__COLD_ATTR__ int VESA_TEST_UNLOAD(void)
{
	int i;
	for (i = 0; i < MAX_VCS; i++) if (locked[i]) return -EBUSY;
	if (driver_installed) return -EBUSY;
	return 0;
}

__COLD_ATTR__ void VESA_DONE(void)
{
	if (console_gfx && on_kernel_cons)
		VESA_RESTORE_VIDEOMODE();
	if (locked_vc == 2) KERNEL$DEL_TIMER(&locked_timeout);
	locked_vc = 0;
	WQ_WAKE_ALL_PL(&locked_timeout_wait);
	free(video_modes);
}
