#include <SPAD/SYSLOG.H>

#include "EXT2.H"

ext2_blk_t INODE_2_BLOCK(EXT2FS *fs, ext2_ino_t ino)
{
	ext2_blk_t root_blk;
	int r;
	r = EXT2_FIND_INODE(fs, EXT2_ROOT_INO, &root_blk, (void *)&KERNEL$LIST_END);
	if (__unlikely(r)) return 0;
	return root_blk;
}

static struct ext2_inode *EXT2_MAP_INODE_SYNC(EXT2FS *fs, ext2_blk_t ino)
{
	char *blk;
	ext2_blk_t root_blk;
	int root_off;
	int r;
	r = EXT2_FIND_INODE(fs, ino, &root_blk, &root_off);
	if (__unlikely(r)) return __ERR_PTR(r);
	blk = EXT2_READ_BUFFER_SYNC(fs, root_blk, 1);
	if (__unlikely(__IS_ERR(blk))) return (struct ext2_inode *)blk;
	blk += root_off;
	return (struct ext2_inode *)blk;
}

static void ext2_update_inode(struct inode * inode, struct ext2_inode *raw_inode)
{
	struct ext2_inode_info *ei = EXT2_I(inode);
	struct super_block *sb = (EXT2FS *)inode->fs;
	/*ext2_ino_t ino = inode->i_ino;*/
	ext2_uid_t uid = inode->i_uid;
	ext2_gid_t gid = inode->i_gid;
	int n;

	raw_inode->i_mode = cpu_to_le16(inode->i_mode);
	if (!(test_opt(sb, NO_UID32))) {
		raw_inode->i_uid_low = cpu_to_le16(uid);
		raw_inode->i_gid_low = cpu_to_le16(gid);
		raw_inode->i_uid_high = cpu_to_le16(uid >> 16);
		raw_inode->i_gid_high = cpu_to_le16(gid >> 16);
	} else {
		raw_inode->i_uid_low = cpu_to_le16(uid > 65535 ? 65534 : uid);
		raw_inode->i_gid_low = cpu_to_le16(gid > 65535 ? 65534 : gid);
		raw_inode->i_uid_high = 0;
		raw_inode->i_gid_high = 0;
	}
	raw_inode->i_links_count = cpu_to_le16(inode->i_nlink);
	raw_inode->i_size = cpu_to_le32(inode->disk_size);
	raw_inode->i_atime = cpu_to_le32(inode->i_atime);
	raw_inode->i_ctime = cpu_to_le32(inode->ctime);
	raw_inode->i_mtime = cpu_to_le32(inode->mtime);

	raw_inode->i_blocks = cpu_to_le32(inode->i_blocks);
	raw_inode->i_flags = cpu_to_le32(ei->i_flags);
	raw_inode->i_faddr = cpu_to_le32(ei->i_faddr);
	raw_inode->i_frag = ei->i_frag_no;
	raw_inode->i_fsize = ei->i_frag_size;
	raw_inode->i_file_acl = cpu_to_le32(ei->i_file_acl);
	if (!LINUX_S_ISREG(inode->i_mode))
		raw_inode->i_dir_acl = cpu_to_le32(ei->i_dir_acl);
	else {
		raw_inode->i_size_high = cpu_to_le32(inode->disk_size >> 31 >> 1);
	}
	
	raw_inode->i_generation = cpu_to_le32(inode->i_generation);
	/*
	if (LINXU_S_ISCHR(inode->i_mode) || LINUX_S_ISBLK(inode->i_mode))
		raw_inode->i_block[0] = cpu_to_le32(kdev_t_to_nr(inode->i_rdev));
	else*/ for (n = 0; n < EXT2_N_BLOCKS; n++)
		raw_inode->i_block[n] = ei->i_data[n];
	return;
}

static ext2_blk_t EXT2_ALLOC_INDIRECT(EXT2FNODE *f, ext2_blk_t hint)
{
#define fs ((EXT2FS *)f->fs)
	char *b;
	ext2_blk_t blk = EXT2_ALLOC_BLOCK(f, hint);
	if (!blk) return 0;
	b = EXT2_ALLOC_BUFFER_SYNC(fs, blk);
	if (__unlikely(__IS_ERR(b))) {
		EXT2_FREE_BLOCKS(f, blk, 1);
		return 0;
	}
	VFS$PUT_BUFFER(b);
	f->i_blocks += fs->sectors_per_block;
	return blk;
#undef fs
}

#define BMAP_CREATE	0
#define BMAP_MAP	1
#define BMAP_TRY	2

static ext2_blk_t BMAP(EXT2FNODE *f, ext2_blk_t block, int *back, int *fwd, int flags)
{
#define fs ((EXT2FS *)f->fs)
	__u32 *run_start, *run_end, *run_c, *blk, *p;
	ext2_blk_t ind, ab, ret;
	if (block < EXT2_NDIR_BLOCKS) {
		run_c = &f->i_data[block];
		if (!__32LE2CPU(*run_c)) {
			if (flags != BMAP_CREATE) return 0;
			ab = EXT2_ALLOC_BLOCK(f, block ? __32LE2CPU(f->i_data[block - 1]) + 1 : INODE_2_BLOCK(fs, f->i_ino));
			if (!ab) return 0;
			f->i_blocks += fs->sectors_per_block;
			*run_c = __32CPU2LE(ab);
		}
		run_start = &f->i_data[0];
		run_end = &f->i_data[EXT2_NDIR_BLOCKS];
		blk = NULL;
		goto found;
	}
	block -= EXT2_NDIR_BLOCKS;
	if (block < EXT2_ADDR_PER_BLOCK(fs)) {
		if (!(ind = __32LE2CPU(f->i_data[EXT2_IND_BLOCK]))) {
			if (flags != BMAP_CREATE) return 0;
			f->i_data[EXT2_IND_BLOCK] = __32CPU2LE(ind = EXT2_ALLOC_INDIRECT(f, __32LE2CPU(f->i_data[EXT2_IND_BLOCK - 1]) + 1));
			if (!ind) return 0;
		}
		indir1:
		blk = EXT2_READ_BUFFER_SYNC(fs, ind, __likely(flags != BMAP_TRY) ? 0 : -1);
		if (__IS_ERR(blk)) return 0;
		run_c = &blk[block];

		if (!__32LE2CPU(*run_c)) {
			ext2_blk_t goal = (run_c > blk ? __32LE2CPU(run_c[-1]) : ind) + 1;
			VFS$PUT_BUFFER(blk);
			if (flags != BMAP_CREATE) return 0;
			ab = EXT2_ALLOC_BLOCK(f, goal);
			if (!ab) return 0;
			f->i_blocks += fs->sectors_per_block;
			blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
			if (__IS_ERR(blk)) return 0;
			run_c = &blk[block];
			*run_c = __32CPU2LE(ab);
		}
		run_start = blk;
		run_end = &blk[EXT2_ADDR_PER_BLOCK(fs)];
		goto found;
	}
	block -= EXT2_ADDR_PER_BLOCK(fs);
	if (block < EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs)) {
		if (!(ind = __32LE2CPU(f->i_data[EXT2_DIND_BLOCK]))) {
			if (flags != BMAP_CREATE) return 0;
			f->i_data[EXT2_DIND_BLOCK] = __32CPU2LE(ind = EXT2_ALLOC_INDIRECT(f, __32LE2CPU(f->i_data[EXT2_DIND_BLOCK - 1]) + 1));
			if (!ind) return 0;
		}
		indir2:
		blk = EXT2_READ_BUFFER_SYNC(fs, ind, __likely(flags != BMAP_TRY)? 0 : -1);
		if (__IS_ERR(blk)) return 0;
		run_c = &blk[block / EXT2_ADDR_PER_BLOCK(fs)];
		if (!__32LE2CPU(*run_c)) {
			VFS$PUT_BUFFER(blk);
			if (flags != BMAP_CREATE) return 0;
			ab = EXT2_ALLOC_INDIRECT(f, ind + 1);
			if (!ab) return 0;
			blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
			if (__IS_ERR(blk)) {
				EXT2_FREE_BLOCKS(f, ab, 1);
				f->i_blocks -= fs->sectors_per_block;
				return 0;
			}
			run_c = &blk[block / EXT2_ADDR_PER_BLOCK(fs)];
			*run_c = __32CPU2LE(ab);
		}
		ind = __32LE2CPU(*run_c);
		VFS$PUT_BUFFER(blk);
		block %= EXT2_ADDR_PER_BLOCK(fs);
		goto indir1;
	}
	block -= EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs);
	if (block < EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs)) {
		if (!(ind = __32LE2CPU(f->i_data[EXT2_TIND_BLOCK]))) {
			if (flags != BMAP_CREATE) return 0;
			f->i_data[EXT2_TIND_BLOCK] = __32CPU2LE(ind = EXT2_ALLOC_INDIRECT(f, __32LE2CPU(f->i_data[EXT2_TIND_BLOCK - 1]) + 1));
			if (!ind) return 0;
		}
		blk = EXT2_READ_BUFFER_SYNC(fs, ind, __likely(flags != BMAP_TRY) ? 0 : -1);
		if (__IS_ERR(blk)) return 0;
		run_c = &blk[block / (EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs))];
		if (!__32LE2CPU(*run_c)) {
			VFS$PUT_BUFFER(blk);
			if (flags != BMAP_CREATE) return 0;
			ab = EXT2_ALLOC_INDIRECT(f, ind + 1);
			if (!ab) {
				EXT2_FREE_BLOCKS(f, ab, 1);
				f->i_blocks -= fs->sectors_per_block;
				return 0;
			}
			blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
			if (__IS_ERR(blk)) return 0;
			run_c = &blk[block / (EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs))];
			*run_c = __32CPU2LE(ab);
		}
		ind = __32LE2CPU(*run_c);
		VFS$PUT_BUFFER(blk);
		block %= EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs);
		goto indir2;
	}
	return 0;

	found:;
	ret = __32LE2CPU(*run_c);
	*back = 0;
	for (p = run_c - 1; p >= run_start && __32LE2CPU(p[0]) && __32LE2CPU(p[0]) + 1 == __32LE2CPU(p[1]); p--) (*back)++;
	*fwd = 0;
	for (p = run_c + 1; p < run_end && __32LE2CPU(p[0]) - 1 == __32LE2CPU(p[-1]); p++) (*fwd)++;
	if (blk) VFS$PUT_BUFFER(blk);
	return ret;
#undef fs
}

static void TRUNCATE_1_INDIRECT(EXT2FNODE *f, ext2_blk_t ind, ext2_blk_t from)
{
#define fs ((EXT2FS *)f->fs)
	int i;
	ext2_blk_t b;
	__u32 *blk;
	if (!ind) return;
	blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
	if (__IS_ERR(blk)) return;
	for (i = from; i < EXT2_ADDR_PER_BLOCK(fs); i++) {
		b = __32LE2CPU(blk[i]);
		blk[i] = __32CPU2LE(0);
		if (b) {
			VFS$PUT_BUFFER(blk);
			EXT2_FREE_BLOCKS(f, b, 1);
			f->i_blocks -= fs->sectors_per_block;
			blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
			if (__IS_ERR(blk)) return;
		}
	}
	VFS$PUT_BUFFER(blk);
#undef fs
}

static void TRUNCATE_2_INDIRECT(EXT2FNODE *f, ext2_blk_t ind, ext2_blk_t from)
{
#define fs ((EXT2FS *)f->fs)
	int i, fr;
	ext2_blk_t b;
	__u32 *blk;
	if (!ind) return;
	blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
	if (__IS_ERR(blk)) return;
	fr = from % EXT2_ADDR_PER_BLOCK(fs);
	for (i = from / EXT2_ADDR_PER_BLOCK(fs); i < EXT2_ADDR_PER_BLOCK(fs); i++) {
		b = __32LE2CPU(blk[i]);
		if (!fr) blk[i] = __32CPU2LE(0);
		if (b) {
			VFS$PUT_BUFFER(blk);
			TRUNCATE_1_INDIRECT(f, b, fr);
			if (!fr) {
				EXT2_FREE_BLOCKS(f, b, 1);
				f->i_blocks -= fs->sectors_per_block;
			}
			blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
			if (__IS_ERR(blk)) return;
		}
		fr = 0;
	}
	VFS$PUT_BUFFER(blk);
#undef fs
}

static void TRUNCATE_3_INDIRECT(EXT2FNODE *f, ext2_blk_t ind, ext2_blk_t from)
{
#define fs ((EXT2FS *)f->fs)
	int i, fr;
	ext2_blk_t b;
	__u32 *blk;
	if (!ind) return;
	blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
	if (__IS_ERR(blk)) return;
	fr = from % (EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs));
	for (i = from / (EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs)); i < EXT2_ADDR_PER_BLOCK(fs); i++) {
		b = __32LE2CPU(blk[i]);
		if (!fr) blk[i] = __32CPU2LE(0);
		if (b) {
			VFS$PUT_BUFFER(blk);
			TRUNCATE_2_INDIRECT(f, b, fr);
			if (!fr) {
				EXT2_FREE_BLOCKS(f, b, 1);
				f->i_blocks -= fs->sectors_per_block;
			}
			blk = EXT2_READ_BUFFER_SYNC(fs, ind, 1);
			if (__IS_ERR(blk)) return;
		}
		fr = 0;
	}
	VFS$PUT_BUFFER(blk);
#undef fs
}

static void TRUNCATE_INODE(EXT2FNODE *f, ext2_blk_t block)
{
#define fs ((EXT2FS *)f->fs)
	int i;
	f->run_length = 0;
	if (__likely(block < EXT2_NDIR_BLOCKS)) {
		for (i = block; i < EXT2_NDIR_BLOCKS; i++) {
			if (f->i_data[i] != __32CPU2LE(0)) {
				EXT2_FREE_BLOCKS(f, __32LE2CPU(f->i_data[i]), 1);
				f->i_blocks -= fs->sectors_per_block;
				f->i_data[i] = __32CPU2LE(0);
			}
		}
		block = 0;
	} else block -= EXT2_NDIR_BLOCKS;
	if (block < EXT2_ADDR_PER_BLOCK(fs)) {
		TRUNCATE_1_INDIRECT(f, __32LE2CPU(f->i_data[EXT2_IND_BLOCK]), block);
		if (__likely(!block)) {
			if (__unlikely(f->i_data[EXT2_IND_BLOCK] != __32CPU2LE(0))) {
				EXT2_FREE_BLOCKS(f, __32LE2CPU(f->i_data[EXT2_IND_BLOCK]), 1);
				f->i_blocks -= fs->sectors_per_block;
				f->i_data[EXT2_IND_BLOCK] = __32CPU2LE(0);
			}
		}
		block = 0;
	} else block -= EXT2_ADDR_PER_BLOCK(fs);
	if (__likely(block < (1 << (2 * EXT2_ADDR_PER_BLOCK_BITS(fs))))) {
		TRUNCATE_2_INDIRECT(f, __32LE2CPU(f->i_data[EXT2_DIND_BLOCK]), block);
		if (__likely(!block)) {
			if (__unlikely(f->i_data[EXT2_DIND_BLOCK] != __32CPU2LE(0))) {
				EXT2_FREE_BLOCKS(f, __32LE2CPU(f->i_data[EXT2_DIND_BLOCK]), 1);
				f->i_blocks -= fs->sectors_per_block;
				f->i_data[EXT2_DIND_BLOCK] = __32CPU2LE(0);
			}
		}
		block = 0;
	} else block -= EXT2_ADDR_PER_BLOCK(fs) * EXT2_ADDR_PER_BLOCK(fs);
	TRUNCATE_3_INDIRECT(f, __32LE2CPU(f->i_data[EXT2_TIND_BLOCK]), block);
	if (__likely(!block)) {
		if (__unlikely(f->i_data[EXT2_TIND_BLOCK] != __32CPU2LE(0))) {
			EXT2_FREE_BLOCKS(f, __32LE2CPU(f->i_data[EXT2_TIND_BLOCK]), 1);
			f->i_blocks -= fs->sectors_per_block;
			f->i_data[EXT2_TIND_BLOCK] = __32CPU2LE(0);
		}
	}
#undef fs
}

static int EXT2_TEST_CONTIG_ALLOC(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
#define fs ((EXT2FS *)f->fs)
	ext2_blk_t blk;
	unsigned sh;
	if (__unlikely((fs->new_prealloc_blk_len | fs->new_prealloc_ino_len) >= 0x80000000u)) return 0;
	blk = EXT2_BLOCKS(fs, f->size);
	sh = __BSR(f->parent->u.d.n_dirty) + 2;
	if (__unlikely(blk > fs->s_blocks_per_group >> sh)) return 0;
	fs->new_prealloc_blk_len += blk;
	fs->new_prealloc_ino_len++;
	return 1;
#undef f
#undef fs
}

int EXT2_CREATE_FNODE(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
#define fs ((EXT2FS *)f->fs)
	ext2_blk_t blk;
	ext2_ino_t ino;
	struct ext2_inode *ei;
	if (__unlikely(!(f->flags & (FNODE_DIRECTORY | FNODE_UNCOMMITTED_PREALLOC_HINT)))) {
		/*if (fs->prealloc_ino_len) __debug_printf("discard ino\n");*/
		EXT2_DISCARD_PREALLOC(fs);
		fs->new_prealloc_blk_len = 0;
		fs->new_prealloc_ino_len = 0;
		VFS$PREPARE_CONTIG_ALLOC((FNODE *)f, EXT2_TEST_CONTIG_ALLOC);
		EXT2_DO_PREALLOC(f);
	}
	if (__unlikely(!fs->prealloc_blk_len) && __unlikely(!(f->flags & FNODE_DIRECTORY)) && __likely(f->size != 0)) {
		fs->new_prealloc_blk_len = EXT2_BLOCKS(fs, f->size);
		fs->new_prealloc_ino_len = 0;
		EXT2_DO_PREALLOC(f);
	}
	f->run_length = 0;
	ino = EXT2_ALLOC_INODE(f, ((EXT2FNODE *)f->parent)->i_ino);
	if (__unlikely(!ino)) return -ENOSPC;
	if (__likely(!(f->flags & FNODE_DIRECTORY))) {
		blk = 0;
	} else {
		blk = EXT2_ALLOC_BLOCK(f, INODE_2_BLOCK(fs, ino));
		if (!blk) {
			EXT2_FREE_INODE(f, ino);
			return -ENOSPC;
		}
	}
	ei = EXT2_MAP_INODE_SYNC(fs, ino);
	if (__unlikely(__IS_ERR(ei))) {
		EXT2_FREE_INODE(f, ino);
		return __PTR_ERR(ei);
	}
	memset(ei, 0, EXT2_INODE_SIZE(fs));
	ei->i_uid = 0;
	ei->i_atime = __32CPU2LE(f->mtime);
	ei->i_ctime = __32CPU2LE(f->ctime);
	ei->i_mtime = __32CPU2LE(f->mtime);
	ei->i_dtime = 0;
	ei->i_gid = 0;
	ei->i_flags = 0;
	if (__likely(!(f->flags & FNODE_DIRECTORY))) {
		ei->i_mode = LINUX_S_IFREG | 0644;
		ei->i_links_count = 1;
	} else {
		ei->i_mode = LINUX_S_IFDIR | 0755;
		ei->i_links_count = 2;
		ei->i_size = fs->s_blocksize;
		ei->i_blocks = fs->sectors_per_block;
		ei->i_block[0] = __32CPU2LE(blk);
	}
	VFS$PUT_BUFFER(ei);
	if (__unlikely(f->flags & FNODE_DIRECTORY)) {
		struct ext2_dir_entry_2 *e = EXT2_ALLOC_BUFFER_SYNC(fs, blk);
		if (__unlikely(__IS_ERR(e))) {
			EXT2_FREE_INODE(f, ino);
			EXT2_FREE_BLOCKS(f, blk, 1);
			return __PTR_ERR(e);
		}
		memset(e, 0, fs->s_blocksize);
		e->inode = __32CPU2LE(ino);
		e->rec_len = __16CPU2LE(EXT2_DIR_REC_LEN(1));
		e->name_len = 1;
		e->file_type = EXT2_HAS_INCOMPAT_FEATURE(fs, EXT2_FEATURE_INCOMPAT_FILETYPE) ? EXT2_FT_DIR : 0;
		e->name[0] = '.';
		e = (struct ext2_dir_entry_2 *)((char *)e + __16LE2CPU(e->rec_len));
		e->inode = __32CPU2LE(((EXT2FNODE *)f->parent)->i_ino);
		e->rec_len = __16CPU2LE(fs->s_blocksize - EXT2_DIR_REC_LEN(1));
		e->name_len = 2;
		e->file_type = EXT2_HAS_INCOMPAT_FEATURE(fs, EXT2_FEATURE_INCOMPAT_FILETYPE) ? EXT2_FT_DIR : 0;
		e->name[0] = '.';
		e->name[1] = '.';
		VFS$PUT_BUFFER(e);
		f->size = fs->s_blocksize;
	}
	f->i_data[0] = __32CPU2LE(blk);
	memset(f->i_data + 1, 0, sizeof f->i_data - sizeof(f->i_data[0]));
	f->i_flags = 0;
	f->i_faddr = 0;
	f->i_frag_no = 0;
	f->i_frag_size = 0;
	f->i_file_acl = 0;
	f->i_dir_acl = 0;
	f->i_dtime = 0;
	f->i_uid = 0;
	f->i_gid = 0;
	f->i_atime = f->mtime;
	f->i_generation = 0;
	f->i_ino = ino;
	if (__likely(!(f->flags & FNODE_DIRECTORY))) {
		f->i_mode = LINUX_S_IFREG | 0644;
		f->i_nlink = 1;
		f->i_blocks = 0;
	} else {
		f->i_mode = LINUX_S_IFDIR | 0755;
		f->i_nlink = 2;
		f->i_blocks = fs->sectors_per_block;
		f->size = fs->s_blocksize;
	}
	ADD_TO_XLIST(&fs->inode_hash[INODE_HASH(f->i_ino)], &f->inode_hash);
	return 0;
#undef f
#undef fs
}

int EXT2_WRITE_FNODE_DATA(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
#define fs ((EXT2FS *)f->fs)
	struct ext2_inode *p;
	_u_off_t size = f->size;
	ext2_blk_t db = (f->disk_size + EXT2_BLOCK_SIZE(fs) - 1) >> EXT2_BLOCK_SIZE_BITS(fs);
	ext2_blk_t mb = (size + EXT2_BLOCK_SIZE(fs) - 1) >> EXT2_BLOCK_SIZE_BITS(fs);
	if (__unlikely(mb < db)) TRUNCATE_INODE(f, mb);
	f->disk_size = size;
	f->run_length = 0;
	if (__unlikely(size > 0x7fffffffU) && (__unlikely(fs->sblk.s_rev_level == cpu_to_le32(EXT2_GOOD_OLD_REV)) || __unlikely(!EXT2_HAS_RO_COMPAT_FEATURE(fs, EXT2_FEATURE_RO_COMPAT_LARGE_FILE)))) {
		ext2_update_dynamic_rev(fs);
		EXT2_SET_RO_COMPAT_FEATURE(fs, EXT2_FEATURE_RO_COMPAT_LARGE_FILE);
		ext2_write_super(fs);
	}
	VFS$WRITEPAGES((FNODE *)f);
	p = EXT2_MAP_INODE_SYNC(fs, f->i_ino);
	if (__unlikely(__IS_ERR(p))) return __PTR_ERR(p);
	ext2_update_inode(f, p);
	VFS$PUT_BUFFER(p);
	return 0;
#undef f
#undef fs
}

int EXT2_SYNC_BMAP(FNODE *f_, __d_off off, int try)
{
#define f ((EXT2FNODE *)f_)
#define ext2fs ((EXT2FS *)f->fs)
	ext2_blk_t b;
	int back, fwd;
	b = BMAP(f, off >> ext2fs->sectors_per_block_bits, &back, &fwd, __likely(!try) ? BMAP_CREATE : BMAP_TRY);
	if (!b) return -EINVAL;
	f->disk_blk = (__d_off)(b - back) << ext2fs->sectors_per_block_bits;
	f->file_blk = (off & -(__d_off)ext2fs->sectors_per_block) - (back << ext2fs->sectors_per_block_bits);
	f->run_length = (back + fwd + 1) << ext2fs->sectors_per_block_bits;
	if (is_bmap_invalid(f)) {
		EXT2_INVALID_BMAP_MSG(f);
		return -EFSERROR;
	}
	return 0;
#undef f
#undef ext2fs
}

int EXT2_ADD_FNODE_TO_DIRECTORY(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
#define fs ((EXT2FS *)f->fs)
	int pw = 0;
	ext2_blk_t i;
	ext2_blk_t b;
	struct ext2_dir_entry_2 *e, *blk, *end, *next;
	if (f->flags & FNODE_DIRECTORY && __unlikely(((EXT2FNODE *)f->parent)->i_nlink >= EXT2_LINK_MAX)) return -ENOSPC;
	for (i = 0; i < (unsigned long)f->parent->size / fs->s_blocksize; i++) {
		b = BMAP((EXT2FNODE *)f->parent, i, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, BMAP_MAP);
		if (__unlikely(!b)) continue;
		blk = EXT2_READ_BUFFER_SYNC(fs, b, 0);
		if (__IS_ERR(blk)) continue;
		e = blk;
		end = (struct ext2_dir_entry_2 *)((char *)e + fs->s_blocksize);
		next_de:
		next = (struct ext2_dir_entry_2 *)((char *)e + __16LE2CPU(e->rec_len));
		if (__unlikely((int)(ext2_blk_t)next & 3) || __unlikely(next > end) || __unlikely((char *)next < &e->name[e->name_len])) {
			VFS$PUT_BUFFER(blk);
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID DIR ENTRY AT BLOCK %lu", (unsigned long)b);
			ext2_error(fs, 1);
			return -EFSERROR;
		}
		if (!__32LE2CPU(e->inode)) {
			if (__16LE2CPU(e->rec_len) >= EXT2_DIR_REC_LEN(f->namelen)) {
				int off = (char *)e - (char *)blk;
				VFS$PUT_BUFFER(blk);
				blk = EXT2_READ_BUFFER_SYNC(fs, b, 1);
				if (__IS_ERR(blk)) return __PTR_ERR(blk);
				e = (struct ext2_dir_entry_2 *)((char *)blk + off);
				put:
				e->inode = __32CPU2LE(f->i_ino);
				e->name_len = f->namelen;
				memcpy(e->name, f->name, f->namelen);
				e->file_type = EXT2_HAS_INCOMPAT_FEATURE(fs, EXT2_FEATURE_INCOMPAT_FILETYPE) ? (f->flags & FNODE_DIRECTORY ? EXT2_FT_DIR : EXT2_FT_REG_FILE) : 0;
				VFS$PUT_BUFFER(e);
				goto inc_parent;
			}
		} else {
			if (__16LE2CPU(e->rec_len) - EXT2_DIR_REC_LEN(e->name_len) >= EXT2_DIR_REC_LEN(f->namelen)) {
				int nrl = __16LE2CPU(e->rec_len) - EXT2_DIR_REC_LEN(e->name_len);
				int nl = e->name_len;
				int off = (char *)e - (char *)blk;
				VFS$PUT_BUFFER(blk);
				blk = EXT2_READ_BUFFER_SYNC(fs, b, 1);
				if (__IS_ERR(blk)) return __PTR_ERR(blk);
				e = (struct ext2_dir_entry_2 *)((char *)blk + off);
				e->rec_len = __16CPU2LE(EXT2_DIR_REC_LEN(nl));
				next = (struct ext2_dir_entry_2 *)((char *)e + __16LE2CPU(e->rec_len));
				next->rec_len = __16CPU2LE(nrl);
				e = next;
				goto put;
			}
		}
		if (next != end) {
			e = next;
			goto next_de;
		}
		VFS$PUT_BUFFER(blk);
	}
	if (f->parent->size >= (__u32)-1 - fs->s_blocksize) return -ENOSPC;
	b = BMAP((EXT2FNODE *)f->parent, i, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, BMAP_CREATE);
	if (!b) return -ENOSPC;
	blk = EXT2_ALLOC_BUFFER_SYNC(fs, b);
	if (__IS_ERR(blk)) return __PTR_ERR(blk);
	f->parent->size += fs->s_blocksize;
	pw = 1;
	e = blk;
	e->rec_len = __16CPU2LE(fs->s_blocksize);
	goto put;

	inc_parent:
	if (__unlikely(f->flags & FNODE_DIRECTORY)) ((EXT2FNODE *)f->parent)->i_nlink++, pw = 1;
	if (__unlikely(pw)) {
		int n;
		EXT2FNODE *pf = (EXT2FNODE *)f->parent;
		struct ext2_inode *p = EXT2_MAP_INODE_SYNC(fs, pf->i_ino);
		if (__unlikely(__IS_ERR(p))) return __PTR_ERR(p);
		p->i_links_count = __16CPU2LE(pf->i_nlink);
		p->i_size = __32CPU2LE(pf->size);
		p->i_blocks = __32CPU2LE(pf->i_blocks);
		for (n = 0; n < EXT2_N_BLOCKS; n++)
			p->i_block[n] = pf->i_data[n];
		VFS$PUT_BUFFER(p);
	}
	if (__unlikely(f->flags & FNODE_DIRECTORY)) {
		blk = EXT2_READ_BUFFER_SYNC(fs, __32LE2CPU(f->i_data[0]), 1);
		if (__unlikely(__IS_ERR(blk))) return __PTR_ERR(e);
		e = (struct ext2_dir_entry_2 *)((char *)blk + EXT2_DIR_REC_LEN(1));
		if (__unlikely(e->name_len != 2) || __unlikely(e->name[0] != '.') || __unlikely(e->name[1] != '.')) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY DOES NOT CONTAIN .. ENTRY");
			VFS$PUT_BUFFER(blk);
			ext2_error(fs, 1);
			return -EFSERROR;
		}
		e->inode = __32CPU2LE(((EXT2FNODE *)f->parent)->i_ino);
		VFS$PUT_BUFFER(blk);
	}
	return 0;
#undef f
#undef fs
}

int EXT2_DELETE_FNODE(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
#define fs ((EXT2FS *)f->fs)
	struct ext2_inode *p;
	p = EXT2_MAP_INODE_SYNC(fs, f->i_ino);
	if (__unlikely(__IS_ERR(p))) return __PTR_ERR(p);
	if (__unlikely(f->flags & FNODE_FILE) && __unlikely(__16LE2CPU(p->i_links_count) > 1)) {
		p->i_links_count = __16CPU2LE(__16LE2CPU(p->i_links_count) - 1);
		VFS$PUT_BUFFER(p);
		return 0;
	}
	VFS$PUT_BUFFER(p);
	TRUNCATE_INODE(f, 0);
	if (__unlikely(f->i_blocks != 0)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "%s: DELETED INODE LEAKED BLOCK COUNT: %"__64_format"X, MODE %X", VFS$FNODE_DESCRIPTION((FNODE *)f), f->i_blocks, f->i_mode);
		ext2_error(fs, 1);
	}
	p = EXT2_MAP_INODE_SYNC(fs, f->i_ino);
	if (__unlikely(__IS_ERR(p))) return __PTR_ERR(p);
	memset(p, 0, EXT2_INODE_SIZE(fs));
	VFS$PUT_BUFFER(p);
	EXT2_FREE_INODE(f, f->i_ino);
	return 0;
#undef f
#undef fs
}

int EXT2_REMOVE_FNODE_FROM_DIRECTORY(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
#define fs ((EXT2FS *)f->fs)
	ext2_blk_t i;
	ext2_blk_t b;
	struct ext2_dir_entry_2 *e, *blk, *end, *next, *last;
	for (i = 0; i < (unsigned long)f->parent->size / fs->s_blocksize; i++) {
		b = BMAP((EXT2FNODE *)f->parent, i, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, BMAP_MAP);
		if (__unlikely(!b)) continue;
		blk = EXT2_READ_BUFFER_SYNC(fs, b, 0);
		if (__IS_ERR(blk)) continue;
		e = blk;
		end = (struct ext2_dir_entry_2 *)((char *)e + fs->s_blocksize);
		last = NULL;
		next_de:
		next = (struct ext2_dir_entry_2 *)((char *)e + __16LE2CPU(e->rec_len));
		if (__unlikely((int)(ext2_blk_t)next & 3) || __unlikely(next > end) || __unlikely((char *)next < &e->name[e->name_len])) {
			VFS$PUT_BUFFER(blk);
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID DIR ENTRY AT BLOCK %lu", (unsigned long)b);
			ext2_error(fs, 1);
			return -EFSERROR;
		}
		if (__32LE2CPU(e->inode) && e->name_len == f->namelen && !_memcasecmp(e->name, f->name, f->namelen)) {
			int off = (char *)e - (char *)blk;
			int loff = last ? (char *)last - (char *)blk : -1;
			VFS$PUT_BUFFER(blk);
			blk = EXT2_READ_BUFFER_SYNC(fs, b, 1);
			if (__IS_ERR(blk)) return __PTR_ERR(blk);
			e = (struct ext2_dir_entry_2 *)((char *)blk + off);
			if (last) {
				last = (struct ext2_dir_entry_2 *)((char *)blk + loff);
				last->rec_len = __16CPU2LE(__16LE2CPU(last->rec_len) + __16LE2CPU(e->rec_len));
			}
			else e->inode = __32CPU2LE(0);
			VFS$PUT_BUFFER(e);
			goto dec_parent;
		}
		if (next != end) {
			last = e;
			e = next;
			goto next_de;
		}
		VFS$PUT_BUFFER(blk);
	}
	return -ENOENT;

	dec_parent:
	if (f->flags & FNODE_DIRECTORY) {
		EXT2FNODE *pf = (EXT2FNODE *)f->parent;
		struct ext2_inode *p;
		pf->i_nlink--;
		p = EXT2_MAP_INODE_SYNC(fs, pf->i_ino);
		if (__unlikely(__IS_ERR(p))) return __PTR_ERR(p);
		p->i_links_count = __16CPU2LE(pf->i_nlink);
		VFS$PUT_BUFFER(p);
	}
	return 0;
#undef f
#undef fs
}

void EXT2_INVALID_BMAP_MSG(EXT2FNODE *f)
{
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, f->fs->filesystem_name, "%s: BMAP OUT OF FILESYSTEM: %"__d_off_format"X, %"__d_off_format"X, %"__d_off_format"X", VFS$FNODE_DESCRIPTION((FNODE *)f), f->disk_blk, f->file_blk, f->run_length);
	f->disk_blk = 0;
	f->file_blk = 0;
	f->run_length = 0;
	ext2_error((EXT2FS *)f->fs, 1);
}

