#include <SPAD/SYSLOG.H>
#include <ENDIAN.H>
#include <SPAD/VFS.H>

#include "STRUCT.H"
#include "FAT.H"

static int FAT_PROCESS_OPTION(FS *fs_, const char *opt, const char *optend, const char *str);
static int FAT_MOUNT(FS *fs_);
static void FAT_UMOUNT(FS *fs_, int nosync);
static void FAT_SET_DIRTY(FS *fs_, int dirty);

static void *FAT_ACCOUNT(FNODE *f, int dir, PAGE *p);
static void FAT_UNACCOUNT(FNODE *f, int dir, PAGE *p);
static void FAT_STAT(FNODE *f_, struct stat *stat);
static WQ *FAT_STATFS(FS *fs_, struct statfs *stat);

static const FSOPS fatfsops = {
	DEFAULT_VFSID,
	"FAT",
	0,
	sizeof(FATFS),
	sizeof(FATFNODE),
	VFS$FNODE_CTOR,
	sizeof(FATPAGEINRQ),
	VFS$PAGEINRQ_CTOR,
	FAT_PROCESS_OPTION,
	FAT_MOUNT,
	NULL,
	FAT_UMOUNT,
	FAT_SET_DIRTY,
	FAT_VALIDATE_FILENAME,
	FAT_LOOKUP,
	sizeof(int),
	FAT_INIT_READDIR_COOKIE,
	FAT_READDIR,
	VFS$DEFAULT_WRITEPAGES,
	VFS$DEFAULT_COMMIT,
	FAT_INIT_FNODE,
	FAT_CREATE_FNODE,
	FAT_WRITE_FNODE_DATA,
	FAT_ADD_FNODE_TO_DIRECTORY,
	FAT_DELETE_FNODE,
	FAT_REMOVE_FNODE_FROM_DIRECTORY,
	VFS$NULL_FNODE_CALL,
	FAT_BMAP,
	FAT_SYNC_BMAP,
	VFS$INIT_LAST_PAGE,
	FAT_ACCOUNT,
	FAT_UNACCOUNT,
	FAT_STAT,
	FAT_STATFS,
};

int main(int argc, const char * const argv[])
{
	return VFS$MOUNT(argc, argv, &fatfsops, "FAT.SYS");
}

int FAT_PROCESS_OPTION(FS *fs_, const char *opt, const char *optend, const char *str)
{
	FATFS *fs = (FATFS *)fs_;
	if (!__strcasexcmp("VFAT", opt, optend) && !str) fs->flags |= FAT_VFAT;
	else return 1;
	return 0;
}

int FAT_MOUNT(FS *fs_)
{
	int media;
	int r;
	int pos;
	int sector_size;
	int sectors_per_cluster;
	fat_sector_t fat_clusters;
	fat_sector_t start_cluster;
	struct fat_boot_sector *boot;
	FATFS *fs = (FATFS *)fs_;
	FATFNODE *f;
	pos = 512;
	test_again:
	if (__IS_ERR(boot = VFS$READ_BUFFER_SYNC((FS *)fs, 0, pos >> BIO_SECTOR_SIZE_BITS, 0, (void *)&KERNEL$LIST_END))) {
		r = __PTR_ERR(boot);
		goto ret0;
	}
	if (__16LE2CPU(((__u16 *)boot)[pos / 2 - 1]) != FAT_BOOT_SIGNATURE) {
		VFS$PUT_BUFFER(boot);
		pos *= 2;
		if (pos <= __SECTORS_PER_PAGE_CLUSTER) goto test_again;
		/*KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD BOOTSECTOR SIGNATURE %02X%02X", ((__u8 *)boot)[510], ((__u8 *)boot)[511]);*/
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FAT BOOTSECTOR SIGNATURE NOT FOUND", fs->filesystem_name);
		r = -EINVAL;
		goto ret0;
	}
	if (!FAT_VALID_MEDIA(boot->media)) {
		/*KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID MEDIA %02X", boot->media);*/
		bad_super:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID BOOTSECTOR", fs->filesystem_name);
		VFS$PUT_BUFFER(boot);
		r = -EFSERROR;
		goto ret0;
	}
	media = boot->media;
	if (!boot->reserved) {
		/*KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "NO RESERVED SECTORS");*/
		goto bad_super;
	}
	fs->fat_start = boot->reserved;

	if (!boot->fats) {
		/*KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "%d FATS", boot->fats);*/
		goto bad_super;
	}
	fs->fats = boot->fats;
	
	sector_size = boot->sector_size[0] + boot->sector_size[1] * 256;
	if (sector_size < BIO_SECTOR_SIZE || sector_size > __PAGE_CLUSTER_SIZE || sector_size & (sector_size - 1)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD SECTOR SIZE: %d", sector_size);
		goto bad_super;
	}
	fs->sector_size = sector_size;
	sector_size >>= BIO_SECTOR_SIZE_BITS;
	fs->sectors_per_sector = sector_size;
	fs->fat_start *= sector_size;

	if (!boot->cluster_size || boot->cluster_size & (boot->cluster_size - 1)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD CLUSTER SIZE: %d", boot->cluster_size);
		goto bad_super;
	}
	fs->sectors_per_cluster = boot->cluster_size * sector_size;
	sectors_per_cluster = boot->cluster_size * sector_size;
	fs->sectors_per_cluster_bits = 0;
	while (sectors_per_cluster > 1) fs->sectors_per_cluster_bits++, sectors_per_cluster >>= 1;

	if (!__16LE2CPU(boot->fat_length)) {
		if (!__32LE2CPU(boot->fat32_length)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ZERO FAT LENGTH");
			goto bad_super;
		}
		/*
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FAT32 NOT SUPPORTED", fs->filesystem_name);
		VFS$PUT_BUFFER(boot);
		r = -EINVAL;
		goto ret0;
		*/
		if (__unlikely(__32LE2CPU(boot->fat32_length) >= 0x80000000U / fs->fats / sector_size)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FAT LENGTH OVERFLOW");
			goto bad_super;
		}
		fs->fat_length = __32LE2CPU(boot->fat32_length) * sector_size;
		fs->fat_bits = 32;
		start_cluster = boot->root_cluster;
		fs->fsinfo_sector = __16LE2CPU(boot->info_sector) * sector_size;
	} else {
		fs->fat_length = __16LE2CPU(boot->fat_length) * sector_size;
		fs->fat_bits = 0;
		start_cluster = 0;
		fs->fsinfo_sector = 0;
	}
	fs->dir_start = fs->fat_start + fs->fats * fs->fat_length;
	fs->dir_sectors = boot->dir_entries[0] + boot->dir_entries[1] * 256;
	fs->dir_sectors *= 32;
	fs->dir_sectors += fs->sector_size - 1;
	fs->dir_sectors /= fs->sector_size;
	fs->dir_sectors *= sector_size;
	fs->data_start = fs->dir_start + fs->dir_sectors;
	fs->total_sectors = boot->sectors[0] + boot->sectors[1] * 256;
	if (!fs->total_sectors) {
		if (!__32LE2CPU(boot->total_sect)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ZERO SECTORS");
			goto bad_super;
		}
		fs->total_sectors = __32LE2CPU(boot->total_sect);
	}
	if (fs->total_sectors >= 0xFFFFFFFF / sector_size) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "TOTAL SECTORS OVERFLOW");
		goto bad_super;
	}
	fs->total_sectors *= sector_size;
	fs->clusters = ((fs->total_sectors - fs->data_start) >> fs->sectors_per_cluster_bits) + 2;
	if (fs->total_sectors <= fs->data_start || fs->clusters < 3) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID TOTAL SECTORS");
		goto bad_super;
	}
	if (!fs->fat_bits) {
		fs->fat_bits = fs->clusters > FAT_12_MAX_CLUSTER ? 16 : 12;
	}
	fat_clusters = fs->fat_length * BIO_SECTOR_SIZE * 8 / fs->fat_bits;
	if (fs->clusters > fat_clusters) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FAT OVERFLOW (FAT CLUSTERS %u < FS CLUSTERS %u)", fat_clusters, fs->clusters);
		goto bad_super;
	}

	VFS$PUT_BUFFER(boot);

	if (fs->fsinfo_sector) {
		struct fat_boot_fsinfo *fsinfo;
		if (__unlikely(__IS_ERR(fsinfo = VFS$READ_BUFFER_SYNC((FS *)fs, fs->fsinfo_sector, sector_size, 0, (void *)&KERNEL$LIST_END)))) {
			no_fsinfo:
			fs->fsinfo_sector = 0;
			goto skip_fsinfo;
		}
		if (__32LE2CPU(fsinfo->signature1) != FAT_FSINFO_SIG1 || __32LE2CPU(fsinfo->signature2) != FAT_FSINFO_SIG2) {
			/* this may happen... observed on Corsair USB FLASH. KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD SIGNATURE IN FSINFO BLOCK %d", fs->fsinfo_sector); */
			VFS$PUT_BUFFER(fsinfo);
			goto no_fsinfo;
		}
		fs->free_clusters = fsinfo->free_clusters;
		VFS$PUT_BUFFER(fsinfo);
	} else {
		fat_sector_t c, f;
		skip_fsinfo:
		f = 0;
		for (c = 2; c < fs->clusters; c++) if (!FAT_READ(fs, c)) f++;
		fs->free_clusters = f;
	}
	fs->overallocated = 0;
	fs->max_free_run = fs->clusters;

	fs->block_mask = 0;
	fs->pageio_mask = 511;
	fs->pageio_bits = 9;
	fs->size = fs->total_sectors;
	fs->max_filesize = 0x80000000U - __PAGE_CLUSTER_SIZE;

	{
		fat_sector_t first = FAT_READ(fs, 0);
		fat_sector_t mask = fs->fat_bits == 32 ? FAT_FIRST_32 : fs->fat_bits == 16 ? FAT_FIRST_16 : FAT_FIRST_12;
		if (first != (mask | media) && first != (FAT_FIRST | media)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD FIRST FAT ENTRY: %08X != %08X", first & ((2 << (fs->fat_bits - 1)) - 1), mask | media);
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD FAT", fs->filesystem_name);
			r = -EINVAL;
			goto ret0;
		}
	}

	f = (FATFNODE *)VFS$GET_ROOT_FNODE((FS *)fs);
	f->flags = FNODE_DIRECTORY;
	VFS_INIT_DIR((FNODE *)f);
	f->ctime = 0;
	f->mtime = 0;
	f->start_cluster = start_cluster;
	f->dirent_sec = 0;
	f->dirent_pos = 0;
	f->n_slots = 0;

	r = 0;

	ret0:
	return r;
}

void FAT_WRITE_FSINFO(FATFS *fs)
{
	BUFFER *b;
	if (fs->fsinfo_sector) {
		struct fat_boot_fsinfo *fsinfo;
		if (__unlikely(__IS_ERR(fsinfo = VFS$READ_BUFFER_SYNC((FS *)fs, fs->fsinfo_sector, fs->sectors_per_sector, 0, &b)))) return;
		fsinfo->free_clusters = __32CPU2LE(fs->free_clusters);
		VFS$MARK_BUFFER_DIRTY(b, fs->fsinfo_sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_sector);
		VFS$PUT_BUFFER(fsinfo);
	}
}

static void FAT_UMOUNT(FS *fs_, int nosync)
{
	FATFS *fs = (FATFS *)fs_;
	if (!nosync && fs->overallocated) KERNEL$SUICIDE("FAT_UMOUNT: %s: ACCOUNTING SKEW (%u CLUSTERS LEAKED)", fs->filesystem_name, fs->overallocated);
}

static void FAT_SET_DIRTY(FS *fs_, int dirty)
{
	/* FAT 32 has some dirty flag in FAT, but I don't have a filesystem
	to test now */
}

void *FAT_READ_BUFFER_SYNC(FATFS *fs, fat_sector_t sec, int dirty)
{
	BUFFER *b;
	void *p;
	int v = (sec & (fs->sectors_per_sector - 1)) << BIO_SECTOR_SIZE_BITS;
	sec &= -fs->sectors_per_sector;
	p = VFS$READ_BUFFER_SYNC((FS *)fs, sec, fs->sectors_per_sector, 0, &b);
	if (__unlikely(__IS_ERR(p))) return p;
	if (dirty) VFS$MARK_BUFFER_DIRTY(b, sec & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_sector);
	return (char *)p + v;
}

static __finline__ unsigned PAGE_CLUSTERS(FATFS *fs)
{
	unsigned n = __SECTORS_PER_PAGE_CLUSTER >> fs->sectors_per_cluster_bits;
	if (__unlikely(!n)) n = 1;
	return n;
}

static void *FAT_ACCOUNT(FNODE *f, int dir, PAGE *p)
{
	FATFS *fs = (FATFS *)f->fs;
	unsigned data;
	if (__likely(dir == ACCOUNT_PAGE)) {
		data = PAGE_CLUSTERS(fs);
	} else if (__likely(dir == ACCOUNT_FILE)) {
		data = 2;
	} else {
		data = 3;
	}
	if (__unlikely(fs->overallocated + data > fs->free_clusters)) return VFS$OVERACCOUNT(fs->root);
	fs->overallocated += data;
	return NULL;
}

static void FAT_UNACCOUNT(FNODE *f, int dir, PAGE *p)
{
	FATFS *fs = (FATFS *)f->fs;
	unsigned data;
	if (__likely(dir == ACCOUNT_PAGE)) {
		data = PAGE_CLUSTERS(fs);
	} else if (__likely(dir == ACCOUNT_FILE)) {
		data = 2;
	} else {
		data = 3;
	}
	fs->overallocated -= data;
}

static void FAT_STAT(FNODE *f_, struct stat *stat)
{
	FATFS *fs = (FATFS *)f_->fs;
	stat->st_blocks = (stat->st_blocks + (fs->sectors_per_cluster_bits - 1)) & ~(__u64)(unsigned)(fs->sectors_per_cluster_bits - 1);
}

static WQ *FAT_STATFS(FS *fs_, struct statfs *stat)
{
#define fs ((FATFS *)fs_)
	stat->f_blocks = (__u64)(fs->clusters - 2) << fs->sectors_per_cluster_bits;
	stat->f_bfree = __likely(fs->free_clusters > fs->overallocated) ? (__u64)(fs->free_clusters - fs->overallocated) << fs->sectors_per_cluster_bits : 0;
	stat->f_bavail = stat->f_bfree;
	stat->f_files = stat->f_blocks >> fs->sectors_per_cluster_bits;
	stat->f_ffree = stat->f_bfree >> fs->sectors_per_cluster_bits;
	stat->f_favail = stat->f_ffree;
	stat->f_namelen = fs->flags & FAT_VFAT ? 255 : 12;
	return NULL;
#undef fs
}

int FAT_STOP_CYCLES(fat_sector_t key, fat_sector_t (*c)[2])
{
	if (__likely((*c)[1] != 0) && __unlikely((*c)[0] == key)) return -1;
	(*c)[1]++;
	if (!(((*c)[1] - 1) & (*c)[1])) (*c)[0] = key;
	return 0;
}

