#ifndef _FAT_FAT_H
#define _FAT_FAT_H

#include <SPAD/VFS.H>

#include "STRUCT.H"

#define FAT_VFAT	0x00000001

#define MAX_FAT_ERRORS		32
#define FAT_ERROR_TIMEOUT	JIFFIES_PER_SECOND

typedef unsigned fat_sector_t;

typedef struct {
	FS_HEAD;
	int sectors_per_sector;
	int sector_size;
	int sectors_per_cluster;	/* 512 bytes per cluster */
	int sectors_per_cluster_bits;	/* -------- "" --------- */
	int fat_start;
	int fat_length;
	int fat_bits;
	int fats;
	int dir_start;
	int dir_sectors;
	int data_start;
	fat_sector_t total_sectors;
	fat_sector_t clusters;

	int fsinfo_sector;
	fat_sector_t free_clusters;
	fat_sector_t overallocated;
	fat_sector_t max_free_run;

	int n_fat_errors;
	u_jiffies_lo_t error_timeout;
	fat_sector_t fat_errors[MAX_FAT_ERRORS];
} FATFS;

typedef struct {
	FNODE_HEAD;
	fat_sector_t dirent_sec;
	unsigned dirent_pos;
	unsigned n_slots;
	fat_sector_t start_cluster;
} FATFNODE;

typedef struct {
	PAGEINRQ_HEAD;
	int cookie;
	fat_sector_t disk_ptr;
	fat_sector_t run_length;
	fat_sector_t log_blk;
	unsigned dir_off;
	fat_sector_t c[2];
	char longname[256];
	int longnamelen;
	int n_slots;
	int shortnamesum;
	int longnameslots;
	unsigned current_fat;
} FATPAGEINRQ;

int FAT_STOP_CYCLES(fat_sector_t key, fat_sector_t (*c)[2]);

void *FAT_READ_BUFFER_SYNC(FATFS *fs, fat_sector_t sec, int dirty);

fat_sector_t FAT_READ(FATFS *fs, fat_sector_t cluster);
void FAT_WRITE(FATFS *fs, fat_sector_t cluster, fat_sector_t value);
void FAT_WRITE_FSINFO(FATFS *fs);

fat_sector_t FAT_ALLOC(FATFS *fs, fat_sector_t hint, fat_sector_t *n_clusters, int first);
void FAT_FREE(FATFS *fs, fat_sector_t cluster);

void fat_date_unix2dos(int unix_date,unsigned short *time, unsigned short *date);
int FAT_CREATE_SLOTS_FROM_NAME(FATFNODE *dir, const char *name, struct fat_dir_entry slots[32]);

int FAT_GROW(FATFNODE *f, fat_sector_t n_clusters);	/* number of clusters to add */
int FAT_SHRINK(FATFNODE *f, fat_sector_t n_clusters);	/* number of clusters to leave */

int FAT_VALIDATE_FILENAME(FS *fs, const char *filename);
void FAT_LOOKUP(PAGEINRQ *p);
void FAT_INIT_READDIR_COOKIE(struct readdir_buffer *rb, FNODE *f);
void FAT_READDIR(PAGEINRQ *pgin, struct readdir_buffer *rb);
void FAT_INIT_FNODE(FNODE *f);
int FAT_CREATE_FNODE(FNODE *f_);
int FAT_WRITE_FNODE_DATA(FNODE *f_);
int FAT_ADD_FNODE_TO_DIRECTORY(FNODE *f_);
int FAT_DELETE_FNODE(FNODE *f_);
int FAT_REMOVE_FNODE_FROM_DIRECTORY(FNODE *f_);
void FAT_BMAP(PAGEINRQ *pgin);
int FAT_SYNC_BMAP(FNODE *f, __d_off off, int try);

static __finline__ fat_sector_t FAT_SIZE_2_CLUSTERS(FATFS *fs, fat_sector_t size)
{
	size += (fs->sectors_per_cluster << BIO_SECTOR_SIZE_BITS) - 1;;
	size >>= fs->sectors_per_cluster_bits + BIO_SECTOR_SIZE_BITS;
	return size;
}

static __finline__ fat_sector_t FAT_CLUSTER_2_SECTOR(FATFS *fs, fat_sector_t start_cluster)
{
	if (__unlikely(start_cluster < 2)) return fs->dir_start;
	return fs->data_start + ((start_cluster - 2) << fs->sectors_per_cluster_bits);
}

static __finline__ fat_sector_t FAT_SECTOR_2_CLUSTER(FATFS *fs, fat_sector_t sector)
{
	if (sector < fs->data_start) return 0;
	return ((sector - fs->data_start) >> fs->sectors_per_cluster_bits) + 2;
}

int FAT_GET_NEXT_SECTOR2(FATFS *fs, fat_sector_t *sector, fat_sector_t *to_read, int nfat);

static __finline__ int FAT_GET_NEXT_SECTOR(FATFS *fs, fat_sector_t *sector, fat_sector_t *to_read, int nfat)
{
	if (__unlikely(*sector < fs->data_start)) {
		(*sector)++;
		if (__unlikely(*sector == fs->dir_start + fs->dir_sectors)) *sector = 0;
		return 0;
	}
	if (__likely(((int)(*sector + 1 - fs->data_start) & (fs->sectors_per_cluster - 1)))) {
		(*sector)++;
		return 0;
	}
	return FAT_GET_NEXT_SECTOR2(fs, sector, to_read, nfat);
}

fat_sector_t FAT_GET_NEXT_SECTOR_SYNC(FATFS *fs, fat_sector_t sector);

static __finline__ int FAT_READAHEAD(FATFS *fs)
{
	return 0;
	/*return fs->fat_bits > 16; this caused performance drop :-( */
}

#endif
