#include <ENDIAN.H>
#include <SPAD/SYSLOG.H>

#include "FAT.H"
#include "STRUCT.H"

static int GET_FAT_ERROR_(FATFS *fs, fat_sector_t sector)
{
	int i;
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - fs->error_timeout > TIMEOUT_JIFFIES(FAT_ERROR_TIMEOUT))) return 0;
	for (i = 0; i < fs->n_fat_errors; i++) if (__likely(fs->fat_errors[i] == sector)) return 1;
	return 0;
}

#define GET_FAT_ERROR(fs, sector) \
	(__unlikely((fs)->n_fat_errors) && __unlikely(GET_FAT_ERROR_(fs, sector)))

static void SET_FAT_ERROR(FATFS *fs, fat_sector_t sector)
{
	int i;
	fs->error_timeout = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(fs->n_fat_errors == MAX_FAT_ERRORS)) fs->n_fat_errors = MAX_FAT_ERRORS - 1;
	for (i = 0; i < fs->n_fat_errors; i++) if (__unlikely(fs->fat_errors[0] == sector)) return;
	memmove(fs->fat_errors + 1, fs->fat_errors, fs->n_fat_errors * sizeof(fat_sector_t));
	fs->fat_errors[0] = sector;
	fs->n_fat_errors++;
}

static void CLEAR_FAT_ERROR_(FATFS *fs, fat_sector_t sector)
{
	int i;
	for (i = 0; i < fs->n_fat_errors; i++) if (fs->fat_errors[i] == sector) {
		memmove(fs->fat_errors + i, fs->fat_errors + i + 1, fs->n_fat_errors - i - 1);
		fs->n_fat_errors--;
		return;
	}
}

#define CLEAR_FAT_ERROR(fs, sector) \
	if (__unlikely(fs->n_fat_errors)) CLEAR_FAT_ERROR_(fs, sector);

static void *FAT_MAP_SECTOR(FATFS *fs, unsigned fsector, fat_sector_t *phys_sector, int wr, int nfat)
{
	BUFFER *b;
	void *d;
	unsigned sector;
	if (__unlikely(fsector >= fs->fat_length)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ACCESS OUT OF FAT, SECTOR %u (MAX %u)", fsector, fs->fat_length);
		*phys_sector = 0;
		return NULL;
	}
	sector = fsector + fs->fat_start + fs->fat_length * nfat;
	d = VFS$GET_BUFFER((FS *)fs, sector, 1, &b, &KERNEL$PROC_KERNEL);
	if (__unlikely(!d)) {
		if (__likely(!wr)) {
			int i;
			for (i = 0; i < fs->fats; i++) {
				unsigned sector = fsector + fs->fat_start + fs->fat_length * i;
				d = VFS$GET_BUFFER((FS *)fs, sector, 1, &b, &KERNEL$PROC_KERNEL);
				if (__likely(d != NULL)) return d;
			}
		}
		*phys_sector = sector;
		return NULL;
	}
	if (__unlikely(wr)) {
		int i;
		for (i = 0; i < fs->sectors_per_sector; i++)
			if (__unlikely(!BUFFER_VALID(b, (sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1 & -fs->sectors_per_sector) + i))) {
				VFS$MARK_BUFFER_DIRTY(b, sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, 1);
				goto skip_d;
			}
		VFS$MARK_BUFFER_DIRTY(b, sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1 & -fs->sectors_per_sector, fs->sectors_per_sector);
		skip_d:;
	}
	return d;
}

static int FAT_ACCESS(FATFS *fs, fat_sector_t cluster, fat_sector_t *val, int wr, fat_sector_t *sector, int nfat)
{
	fat_sector_t v;
	if (fs->fat_bits == 16) {
		__u16 *fat = FAT_MAP_SECTOR(fs, cluster / (BIO_SECTOR_SIZE / 2), sector, wr, nfat);
		if (__unlikely(!fat)) return 1;
		if (__likely(!wr)) {
			v = __16LE2CPU(fat[cluster & (BIO_SECTOR_SIZE / 2 - 1)]);
			if (v >= FAT_16_MAX_CLUSTER) v |= 0xffff0000;
			*val = v;
		} else {
			fat[cluster & (BIO_SECTOR_SIZE / 2 - 1)] = __16CPU2LE(*val);
		}
		VFS$PUT_BUFFER(fat);
		return 0;
	} else if (fs->fat_bits == 32) {
		__u32 *fat = FAT_MAP_SECTOR(fs, cluster / (BIO_SECTOR_SIZE / 4), sector, wr, nfat);
		if (__unlikely(!fat)) return 1;
		if (__likely(!wr)) {
			v = __32LE2CPU(fat[cluster & (BIO_SECTOR_SIZE / 4 - 1)]) & 0x0fffffff;
			if (v >= FAT_32_MAX_CLUSTER) v |= 0xf0000000;
			*val = v;
		} else {
			fat[cluster & (BIO_SECTOR_SIZE / 4 - 1)] = __32CPU2LE(*val & 0x0fffffff);
		}
		VFS$PUT_BUFFER(fat);
		return 0;
	} else {
		int i;
		if (__likely(!wr)) v = 0;
		else v = *val;
		cluster *= 3;
		for (i = 0; i < 3; i++, cluster++) {
			__u8 *b;
			__u8 *fat = FAT_MAP_SECTOR(fs, cluster / (BIO_SECTOR_SIZE * 2), sector, wr, nfat);
			if (__unlikely(!fat)) return 1;
			nextb:
			b = &fat[(cluster >> 1) & (BIO_SECTOR_SIZE - 1)];
			if (__likely(!wr)) {
				if (cluster & 1) v = (v >> 4) | (*b >> 4 << 8);
				else v = (v >> 4) | ((*b & 0x0f) << 8);
			} else {
				if (cluster & 1) *b = (*b & 0x0f) | ((v << 4) & 0xf0);
				else *b = (*b & 0xf0) | (v & 0x0f);
				v >>= 4;
			}
			if (i < 2 && (__likely(((int)(fat_sector_t)b & (BIO_SECTOR_SIZE - 1)) != (BIO_SECTOR_SIZE - 1)) || (!(cluster & 1)))) {
				i++;
				cluster++;
				goto nextb;
			}
			VFS$PUT_BUFFER(fat);
		}
		if (__likely(!wr)) {
			if (v >= FAT_12_MAX_CLUSTER) v |= 0xfffff000;
			*val = v;
		}
		return 0;
	}
}

int FAT_GET_NEXT_SECTOR2(FATFS *fs, fat_sector_t *sector, fat_sector_t *to_read, int nfat)
{
	fat_sector_t new_clust;
	fat_sector_t cluster = ((*sector - fs->data_start) >> fs->sectors_per_cluster_bits) + 2;
	if (__unlikely(FAT_ACCESS(fs, cluster, &new_clust, 0, to_read, nfat))) {
		if (__unlikely(!*to_read)) goto z;
		return 1;
	}
	if (IS_EOF(new_clust)) z: *sector = 0;
	else if (__unlikely(new_clust < 2) || __unlikely(new_clust >= fs->clusters)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD CLUSTER %08X AT POSITION %08X (FS CLUSTERS %08X)", new_clust, cluster, fs->clusters);
		goto z;
	} else *sector = fs->data_start + ((new_clust - 2) << fs->sectors_per_cluster_bits);
	return 0;
}

fat_sector_t FAT_GET_NEXT_SECTOR_SYNC(FATFS *fs, fat_sector_t sector)
{
	void *p;
	fat_sector_t to_read;
	int nfat = 0;
	again:
	if (__unlikely(FAT_GET_NEXT_SECTOR(fs, &sector, &to_read, nfat))) {
		if (GET_FAT_ERROR(fs, to_read & -fs->sectors_per_sector)) goto er;
		p = VFS$READ_BUFFER_SYNC((FS *)fs, to_read & -fs->sectors_per_sector, fs->sectors_per_sector, FAT_READAHEAD(fs), (void *)&KERNEL$LIST_END);
		if (__unlikely(__IS_ERR(p))) {
			SET_FAT_ERROR(fs, to_read & -fs->sectors_per_sector);
			er:
			if (__unlikely(nfat++ >= fs->fats)) return 0;
			goto again;
		}
		VFS$PUT_BUFFER(p);
		CLEAR_FAT_ERROR(fs, to_read & -fs->sectors_per_sector);
		goto again;
	}
	return sector;
}

fat_sector_t FAT_READ(FATFS *fs, fat_sector_t cluster)
{
	void *p;
	fat_sector_t val, rd;
	int nfat = 0;
	again:
	if (__unlikely(FAT_ACCESS(fs, cluster, &val, 0, &rd, nfat))) {
		if (__unlikely(!rd)) return CLUSTER_BAD;
		if (GET_FAT_ERROR(fs, rd & -fs->sectors_per_sector)) goto er;
		p = VFS$READ_BUFFER_SYNC((FS *)fs, rd & -fs->sectors_per_sector, fs->sectors_per_sector, 0, (void *)&KERNEL$LIST_END);
		if (__unlikely(__IS_ERR(p))) {
			SET_FAT_ERROR(fs, rd & -fs->sectors_per_sector);
			er:
			if (__unlikely(++nfat >= fs->fats)) return CLUSTER_BAD;
			goto again;
		}
		VFS$PUT_BUFFER(p);
		CLEAR_FAT_ERROR(fs, rd & -fs->sectors_per_sector);
		goto again;
	}
	return val;
}

void FAT_WRITE(FATFS *fs, fat_sector_t cluster, fat_sector_t value)
{
	void *p;
	int i;
	fat_sector_t rd;
	for (i = 0; i < fs->fats; i++) {
		again:
		if (__unlikely(FAT_ACCESS(fs, cluster, &value, 1, &rd, i))) {
			if (__unlikely(!rd)) continue;
			if (GET_FAT_ERROR(fs, rd & -fs->sectors_per_sector)) continue;
			p = VFS$READ_BUFFER_SYNC((FS *)fs, rd & -fs->sectors_per_sector, fs->sectors_per_sector, 0, (void *)&KERNEL$LIST_END);
			if (__unlikely(__IS_ERR(p))) {
				SET_FAT_ERROR(fs, rd & -fs->sectors_per_sector);
				continue;
			}
			VFS$PUT_BUFFER(p);
			CLEAR_FAT_ERROR(fs, rd & -fs->sectors_per_sector);
			goto again;
		}
	}
}



