#include <SPAD/SYSLOG.H>

#include "FAT.H"

static AST_STUB FAT_BMAP_AST;

static __finline__ void INIT_BMAP(FATFNODE *f, fat_sector_t off, fat_sector_t *log_blk, fat_sector_t *disk_blk, fat_sector_t *run_length)
{
#define fatfs ((FATFS *)f->fs)
	*run_length = fatfs->sectors_per_cluster;
	if (__likely(off >= (fat_sector_t)f->disk_blk + (fat_sector_t)f->run_length) && __likely((fat_sector_t)f->run_length != 0)) {
		*disk_blk = (fat_sector_t)f->disk_blk + (fat_sector_t)f->run_length - fatfs->sectors_per_cluster;
		*log_blk = (fat_sector_t)f->file_blk + (fat_sector_t)f->run_length;
	} else {
		*disk_blk = FAT_CLUSTER_2_SECTOR(fatfs, f->start_cluster);
		*log_blk = fatfs->sectors_per_cluster;
	}
#undef fatfs
}

static __finline__ void SET_BMAP(FATFNODE *f, fat_sector_t log_blk, fat_sector_t disk_blk, fat_sector_t run_length)
{
#define fatfs ((FATFS *)f->fs)
	f->file_blk = log_blk - run_length;
	f->disk_blk = disk_blk + fatfs->sectors_per_cluster - run_length;
	f->run_length = run_length;
	/*__debug_printf("%Lx %Lx %Lx\n", f->file_blk, f->disk_blk, f->run_length);*/
#undef fatfs
}

void FAT_BMAP(PAGEINRQ *pgin_)
{
	FATPAGEINRQ *pgin = (FATPAGEINRQ *)pgin_;
	FATFNODE *f = (FATFNODE *)pgin->fnode;
#define fatfs ((FATFS *)f->fs)
	if (__unlikely(f->start_cluster < 2) || __unlikely(f->start_cluster >= fatfs->clusters)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "%s: BAD STARTING CLUSTER %08X (FS CLUSTERS %08X)", VFS$FNODE_DESCRIPTION((FNODE *)f), f->start_cluster, fatfs->clusters);
		pgin->status = -EFSERROR;
		VFS$BMAP_DONE((PAGEINRQ *)pgin);
		return;
	}
	INIT_BMAP(f, pgin->off, &pgin->log_blk, &pgin->disk_ptr, &pgin->run_length);
	pgin->fn = FAT_BMAP_AST;
	pgin->c[1] = 0;
	pgin->current_fat = 0;
	CALL_AST(pgin);
#undef fatfs
}

static DECL_AST(FAT_BMAP_AST, SPL_FS, FATPAGEINRQ)
{
	fat_sector_t new, to_read;
#define f ((FATFNODE *)RQ->fnode)
#define fatfs ((FATFS *)RQ->fs)
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_FS), RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status != 0)) {
		if (__unlikely(++RQ->current_fat >= fatfs->fats)) {
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		RQ->status = 0;
	}
	if (0) {
		again:
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
	}

	new = RQ->disk_ptr;
	if (__unlikely(FAT_GET_NEXT_SECTOR2(fatfs, &new, &to_read, RQ->current_fat))) {
		if (__likely(RQ->log_blk > RQ->off)) {
			ok:
			SET_BMAP(f, RQ->log_blk, RQ->disk_ptr, RQ->run_length);
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		VFS$READ_BUFFER(to_read, (PAGEINRQ *)RQ, 0);
		RETURN;
	}
	RQ->current_fat = 0;
	if (__unlikely(!new)) {
		if (__likely(RQ->log_blk > RQ->off)) goto ok;
		goto out;
	}
	if (__likely(new == RQ->disk_ptr + fatfs->sectors_per_cluster)) {
		RQ->run_length += fatfs->sectors_per_cluster;
	} else {
		if (RQ->log_blk > RQ->off) goto ok;
		RQ->run_length = fatfs->sectors_per_cluster;
	}
	RQ->disk_ptr = new;
	RQ->log_blk += fatfs->sectors_per_cluster;
	if (__unlikely(FAT_STOP_CYCLES(RQ->disk_ptr, &RQ->c))) goto cycle;
	goto again;

	cycle:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08X (BMAP)", RQ->disk_ptr);
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08X (BMAP %s:%016"__d_off_format"X)", RQ->disk_ptr, RQ->fnode->name, RQ->off);
	fserr:
	RQ->status = -EFSERROR;
	VFS$BMAP_DONE((PAGEINRQ *)RQ);
	RETURN;

	out:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "EOF DETECTED");
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "EOF DETECTED (BMAP %s:%016"__d_off_format"X)", RQ->fnode->name, RQ->off);
	goto fserr;
#undef f
#undef fatfs
}

int FAT_SYNC_BMAP(FNODE *f_, __d_off off, int try)
{
#define f ((FATFNODE *)f_)
#define fatfs ((FATFS *)f->fs)
	fat_sector_t disk_ptr, log_blk, run_length;
	fat_sector_t new;
	fat_sector_t c[2];
	c[1] = 0;
	INIT_BMAP(f, off, &log_blk, &disk_ptr, &run_length);
	disk_ptr = FAT_CLUSTER_2_SECTOR(fatfs, f->start_cluster);
	run_length = fatfs->sectors_per_cluster;
	log_blk = fatfs->sectors_per_cluster;
	again:
	if (__likely(!try)) {
		new = FAT_GET_NEXT_SECTOR_SYNC(fatfs, disk_ptr + fatfs->sectors_per_cluster - 1);
	} else {
		new = disk_ptr;
		if (__unlikely(FAT_GET_NEXT_SECTOR2(fatfs, &new, (void *)&KERNEL$LIST_END, 0))) {
			if (__likely(log_blk > off)) {
				ok:
				SET_BMAP(f, log_blk, disk_ptr, run_length);
				return 0;
			}
			return -ENOENT;
		}
	}
	if (!new) {
		if (__likely(log_blk > off)) goto ok;
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "EOF DETECTED ON SYNC BMAP");
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "EOF DETECTED (SYNC BMAP %s:%016"__d_off_format"X)", f->name, off);
		return -EFSERROR;
	}
	if (__likely(new == disk_ptr + fatfs->sectors_per_cluster)) {
		run_length += fatfs->sectors_per_cluster;
	} else {
		if (log_blk > off) goto ok;
		run_length = fatfs->sectors_per_cluster;
	}
	disk_ptr = new;
	log_blk += fatfs->sectors_per_cluster;
	if (__unlikely(FAT_STOP_CYCLES(disk_ptr, &c))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08X (SYNC BMAP)", disk_ptr);
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08X (SYNC BMAP %s:%016"__d_off_format"X)", disk_ptr, f->name, off);
		return -EFSERROR;
	}
	goto again;
#undef f
#undef fatfs
}

int FAT_GROW(FATFNODE *f, fat_sector_t n_clusters)
{
#define fatfs ((FATFS *)f->fs)
	fat_sector_t ncl, new_chain;
	fat_sector_t ocl = 0;
	fat_sector_t cl = f->start_cluster;
	fat_sector_t c[2];
	c[1] = 0;
	while (cl >= 2) {
		ocl = cl;
		cl = FAT_READ(fatfs, cl);
		if (IS_EOF(cl)) break;
		if (__unlikely(cl < 2) || __unlikely(cl >= fatfs->clusters)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "BOGUS CLUSTER %08X AT POSITION %08X WHEN EXTENDING FILE", cl, ocl);
			return -EFSERROR;
		}
		if (__unlikely(FAT_STOP_CYCLES(cl, &c))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "CYCLE DETECTED ON CLUSTER %08X WHEN EXTENDING FILE", cl);
			return -EFSERROR;
		}
	}
	new_chain:
	ncl = n_clusters;
	new_chain = FAT_ALLOC(fatfs, ocl ? ocl + 1 : FAT_SECTOR_2_CLUSTER(fatfs, ((FATFNODE *)f->parent)->dirent_sec), &ncl, ocl ? 0 : 1);
	if (!new_chain) return -ENOSPC;
	if (ocl) FAT_WRITE(fatfs, ocl, new_chain);
	else f->start_cluster = new_chain;
	ocl = new_chain + ncl - 1;
	if ((n_clusters -= ncl)) goto new_chain;
	return 0;
#undef fatfs
}

int FAT_SHRINK(FATFNODE *f, fat_sector_t leave)
{
#define fatfs ((FATFS *)f->fs)
	fat_sector_t ocl = 0;	/* warning, go away */
	fat_sector_t cl;
	f->run_length = 0;
	cl = f->start_cluster;
	while (leave--) {
		ocl = cl;
		cl = FAT_READ(fatfs, cl);
		if (__unlikely(cl < 2) || __unlikely(cl >= fatfs->clusters)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "BOGUS CLUSTER %08X AT POSITION %08X WHEN TRUNCATING FILE", cl, ocl);
			return -EFSERROR;
		}
	}
	if (__unlikely(!cl)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "FILE HAS NO CLUSTERS");
		return -EFSERROR;
	}
	FAT_FREE(fatfs, cl);
	if (ocl) FAT_WRITE(fatfs, ocl, CLUSTER_EOF);
	else f->start_cluster = 0;
	return 0;
#undef fatfs
}

