#include <TIME.H>

#include "HPFS.H"

__u8 *HPFS_MAP_BITMAP(HPFSFS *fs, int bmp, BUFFER **b, int *pos)
{
	__u8 *desc;
	unsigned sec;
	if (__unlikely(bmp >= (int)fs->n_bitmaps) || __unlikely(bmp < -1)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "AN ATTEMPT TO MAP INVALID BITMAP %d, MAX BITMAPS %d", bmp, fs->n_bitmaps);
		HPFS_ERROR(fs);
		return NULL;
	}
	if (__unlikely(bmp < 0)) sec = fs->dir_band_bitmap;
	else sec = fs->bmp_desc[bmp].sector;
	if (__unlikely(!sec)) return NULL;
	if (__unlikely(__IS_ERR(desc = VFS$READ_BUFFER_SYNC((FS *)fs, sec, 4, (sec & 0x3fff) == 0x3ffc, b)))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ERROR READING BITMAP %d AT %08X: %s", bmp, sec, strerror(-__PTR_ERR(desc)));
		return NULL;
	}
	*pos = sec & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
	return desc;
}

static int SCAN_BITMAP(__u8 *bmp, unsigned start, unsigned n)
{
	unsigned run = 0;
	again:
	if (start & 31) {
		bits:
		if ((bmp[start >> 3] >> (start & 7)) & 1) run++;
		else run = 0;
		start++;
	} else {
		__u32 b = *(__u32 *)&bmp[start >> 3];
		if (__likely(!b)) {
			start += 32;
			run = 0;
		} else if (__likely(b == 0xffffffffU)) {
			start += 32;
			run += 32;
		} else goto bits;
	}
	if (run >= n) return start - run;
	if (__unlikely(start == 0x4000)) return -1;
	goto again;
}

int FIND_MAX_RUN(__u8 *bmp)
{
	unsigned start = 0;
	unsigned run = 0;
	unsigned max_run = 0;
	again:
	if (start & 31) {
		bits:
		if ((bmp[start >> 3] >> (start & 7)) & 1) run++;
		else {
			if (__unlikely(run > max_run)) max_run = run;
			run = 0;
		
		}
		start++;
	} else {
		__u32 b = *(__u32 *)&bmp[start >> 3];
		if (__likely(!b)) {
			start += 32;
			if (__unlikely(run > max_run)) max_run = run;
			run = 0;
		} else if (__likely(b == 0xffffffffU)) {
			start += 32;
			run += 32;
		} else goto bits;
	}
	if (__likely(start < 0x4000)) goto again;
	if (__unlikely(run > max_run)) max_run = run;
	return max_run;
}

static int TEST_BITMAP(__u8 *bmp, unsigned start, unsigned n)
{
	unsigned run = 0;
	again:
	if (start & 31) {
		bits:
		if ((bmp[start >> 3] >> (start & 7)) & 1) run++;
		else return run;
		start++;
	} else {
		__u32 b = ((__u32 *)bmp)[start >> 5];
		if (__likely(!b)) return run;
		else if (__likely(b == 0xffffffffU)) {
			start += 32;
			run += 32;
		} else goto bits;
	}
	if (__unlikely(start == 0x4000)) return run;
	goto again;
}

static int SCAN_BITMAP_FOR_4(__u8 *bmp, unsigned start)
{
	start &= ~3;
	again:
	if (!(start & 4)) {
		if ((bmp[start >> 3] & 0x0f) == 0x0f) return start;
	} else {
		if ((bmp[start >> 3] & 0xf0) == 0xf0) return start;
	}
	start += 4;
	if (__unlikely(start == 0x4000)) return -1;
	goto again;
}

static unsigned HPFS_ALLOC_IN_DIR_BAND(HPFSFS *fs, unsigned goal)
{
	__u8 *bmp;
	int g, a, pos;
	BUFFER *b;
	if (__unlikely(!(bmp = HPFS_MAP_BITMAP(fs, -1, &b, &pos))))
		return 0;
	if (__unlikely(goal >= fs->dir_band_start) && __unlikely(goal < fs->dir_band_start + fs->n_dir_band)) {
		g = goal - fs->dir_band_start;
		g >>= 2;
		if (__unlikely((unsigned)g >= 0x4000)) g = 0;
	} else {
		g = 0;
	}
	scan_dir_again:
	a = SCAN_BITMAP(bmp, g, 1);
	if (__likely(a != -1)) {
		if (__unlikely(!((bmp[a >> 3] >> (a & 7)) & 1)))
			KERNEL$SUICIDE("HPFS_ALLOC: %s: DIRBAND BIT NOT FREE, POS %d", fs->filesystem_name, a);
		bmp[a >> 3] &= ~(1 << (a & 7));
		fs->free_dnodes--;
		VFS$MARK_BUFFER_DIRTY(b, pos, 4);
		VFS$PUT_BUFFER(bmp);
		fs->dir_band_free--;
		return fs->dir_band_start + (a << 2);
	}
	if (__unlikely(g)) {
		g = 0;
		goto scan_dir_again;
	}
	VFS$PUT_BUFFER(bmp);
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIR BAND FREE %u AND NO FREE BLOCKS THERE", fs->dir_band_free);
	fs->dir_band_free = 0;
	return 0;
}

static void HPFS_FREE_IN_DIR_BAND(HPFSFS *fs, unsigned sec, unsigned n)
{
	__u8 *bmp;
	unsigned bl;
	BUFFER *b;
	int pos;
	if (__unlikely(n != 4) || __unlikely(sec & 3)) {
		bad_dirband:
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREE INVALID DATA IN DIRBAND (START %08X, LEN %08X)", sec, n);
		HPFS_ERROR(fs);
		return;
	}
	if (__unlikely((bl = (sec - fs->dir_band_start) >> 2) >= 0x4000)) {
		goto bad_dirband;
	}
	bmp = HPFS_MAP_BITMAP(fs, -1, &b, &pos);
	if (__unlikely(!bmp)) return;
	VFS$MARK_BUFFER_DIRTY(b, pos, 4);
	if (__unlikely((bmp[bl >> 3] >> (bl & 7)) & 1)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREEING ALREADY FREE BLOCK IN DIRBAND AT POSITION %08X", bl);
		HPFS_ERROR(fs);
		goto skip_dir_free;
	}
	bmp[bl >> 3] |= 1 << (bl & 7);
	fs->free_dnodes++;
	fs->dir_band_free++;
	skip_dir_free:
	VFS$PUT_BUFFER(bmp);
}

static unsigned HPFS_ALLOC_EMERGENCY_DNODE(HPFSFS *fs)
{
	BUFFER *b;
	unsigned free_spare, block;
	struct hpfs_spare_block *spare = HPFS_MAP_SPARE(fs, &b);
	if (__unlikely(__IS_ERR(spare)))
		return 0;
	free_spare = __32LE2CPU(spare->n_dnode_spares_free);
	get_next:
	if (__unlikely(!free_spare)) {
		VFS$PUT_BUFFER(spare);
		return 0;
	}
	free_spare--;
	block = __32LE2CPU(spare->spare_dnodes[free_spare]);
	spare->n_dnode_spares_free = __32CPU2LE(free_spare);
	spare->flags1 |= SP_1_SPARE_USED;
	fs->flags |= HPFS_SPARE_USED;
	HPFS_SPARE_MAKE_CHECKSUM(spare);
	VFS$MARK_BUFFER_DIRTY(b, SP_SECTOR & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, 1);
	if (__unlikely(!block) ||
	    __unlikely(block & 3) ||
	    __unlikely(block + 3 >= (unsigned)fs->size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID SPARE DNODE %08X", block);
		goto get_next;
	}
	VFS$PUT_BUFFER(spare);
	return block;
}

static int HPFS_FREE_EMERGENCY_DNODE(HPFSFS *fs, unsigned block)
{
	BUFFER *b;
	unsigned free_spare, total_spare;
	void *dnode;
	struct hpfs_spare_block *spare;
	spare = HPFS_MAP_SPARE(fs, &b);
	if (__unlikely(__IS_ERR(spare)))
		return 1;
	free_spare = __32LE2CPU(spare->n_dnode_spares_free);
	total_spare = __32LE2CPU(spare->n_dnode_spares);
	if (__unlikely(free_spare >= __32LE2CPU(spare->n_dnode_spares))) {
		VFS$PUT_BUFFER(spare);
		return 1;
	}
	spare->spare_dnodes[free_spare] = __32CPU2LE(block);
	free_spare++;
	spare->n_dnode_spares_free = __32CPU2LE(free_spare);
	if (free_spare == total_spare) {
		spare->flags1 &= ~SP_1_SPARE_USED;
		fs->flags &= ~HPFS_SPARE_USED;
	}
	HPFS_SPARE_MAKE_CHECKSUM(spare);
	VFS$MARK_BUFFER_DIRTY(b, SP_SECTOR & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, 1);
	VFS$PUT_BUFFER(spare);
	if (__unlikely(__IS_ERR(dnode = VFS$READ_BUFFER_SYNC((FS *)fs, block, 4, 0, &b))))
		return 0;
	memset(dnode, 0xf6, 2048);
	VFS$MARK_BUFFER_DIRTY(b, block & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, 4);
	VFS$WRITE_BUFFER(b, fs->syncproc);
	VFS$PUT_BUFFER_AND_WAIT(dnode);
	return 0;
}

unsigned HPFS_ALLOC(HPFSFS *fs, unsigned goal, unsigned *n)
{
	__u8 *bmp;
	int g, a, pos, i, bmpi;
	int x;
	BUFFER *b;
	if (__unlikely(!n) && __unlikely(fs->dir_band_free)) {
		unsigned blk;
		blk = HPFS_ALLOC_IN_DIR_BAND(fs, goal);
		if (__likely(blk != 0)) return blk;
	}
	if (n && __unlikely(*n > 0x3ffc)) *n = 0x3ffc;
	alloc_again:
	bmpi = goal >> 14;
	g = goal & 0x3fff;
	i = fs->n_bitmaps + 1;
	next_bitmap:
	if (__unlikely(bmpi >= fs->n_bitmaps)) bmpi = 0;
	x = __likely(n != NULL) ? *n : 4;
	if (__unlikely(x > fs->bmp_desc[bmpi].max_free_run) ||
	    __unlikely(x > fs->bmp_desc[bmpi].n_free)) goto skip_bmp;
	if (__unlikely(!(bmp = HPFS_MAP_BITMAP(fs, bmpi, &b, &pos)))) {
		fs->bmp_desc[bmpi].max_free_run = 0;
		goto skip_bmp;
	}
	if (__unlikely(!n)) {
		a = SCAN_BITMAP_FOR_4(bmp, g);
		if (__likely(a != -1)) {
			static unsigned four = 4;
			n = &four;
			goto allocated;
		}
		if (__unlikely(!g)) fs->bmp_desc[bmpi].max_free_run = FIND_MAX_RUN(bmp);
		goto skip_bmp_d;
	}
	if (__unlikely(i == fs->n_bitmaps + 1)) {
		int nn = TEST_BITMAP(bmp, g, *n);
		if (__likely(nn)) {
			if (__unlikely(nn > *n)) nn = *n;
			else *n = nn;
			a = g;
			goto allocated;
		}
	}
	a = SCAN_BITMAP(bmp, g, *n);
	if (__likely(a != -1)) {
		unsigned x;
		allocated:
		x = *n;
		while (x) {
			if (__unlikely(a & 7) || __unlikely(x < 8)) {
				if ((a & 7) < 4) {
					if ((bmp[a >> 3] & 0x0f) == 0x0f) fs->free_dnodes--;
				} else {
					if ((bmp[a >> 3] & 0xf0) == 0xf0) fs->free_dnodes--;
				}
				if (__unlikely(!((bmp[a >> 3] >> (a & 7)) & 1)))
					KERNEL$SUICIDE("HPFS_ALLOC: %s: BIT NOT FREE, BITMAP %d, POS %d, START %d, LEN %d", fs->filesystem_name, bmpi, a, a + x - *n, *n);
				bmp[a >> 3] &= ~(1 << (a & 7));
				a++;
				x--;
			} else {
				if (__unlikely(bmp[a >> 3] != 0xff))
					KERNEL$SUICIDE("HPFS_ALLOC: %s: BYTE NOT FREE, BITMAP %d, POS %d, START %d, LEN %d", fs->filesystem_name, bmpi, a, a + x - *n, *n);
				fs->free_dnodes -= 2;
				bmp[a >> 3] = 0;
				a += 8;
				x -= 8;
			}
		}
		VFS$MARK_BUFFER_DIRTY(b, pos, 4);
		VFS$PUT_BUFFER(bmp);
		fs->free_space -= *n;
		x = fs->bmp_desc[bmpi].n_free - *n;
		if (__unlikely(x < 0)) {
			x = 0;
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREE BLOCK COUNT UNDERFLOW"), HPFS_ERROR(fs);
		}
		fs->bmp_desc[bmpi].n_free = x;
		return (bmpi << 14) + a - *n;
	}
	if (__unlikely(!g)) fs->bmp_desc[bmpi].max_free_run = FIND_MAX_RUN(bmp);
	skip_bmp_d:
	VFS$PUT_BUFFER(bmp);
	skip_bmp:
	g = 0;
	bmpi++;
	if (__likely(--i)) goto next_bitmap;
	if (__unlikely(fs->prealloc_fnode_length | fs->prealloc_data_length) && (!n || *n < 64)) {
		HPFS_DISCARD_PREALLOC_(fs);
		goto alloc_again;
	}
	if (__unlikely(!n)) {
		unsigned blk;
		blk = HPFS_ALLOC_EMERGENCY_DNODE(fs);
		if (__likely(blk != 0)) return blk;
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "COULD NOT ALLOCATE DNODE BLOCK WITH GOAL %08X", goal);
		HPFS_ERROR(fs);
		return 0;
	}
	if (__unlikely(!(*n >>= 1))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "COULD NOT ALLOCATE BLOCK WITH GOAL %08X", goal);
		HPFS_ERROR(fs);
		return 0;
	}
		/* 4-sector blocks are needed for directories,
		   do not allocate them with precedence */
	if (__unlikely(*n < 8)) *n = 1;
	goto alloc_again;
}

void HPFS_FREE(HPFSFS *fs, unsigned sec, unsigned n)
{
	__u8 *bmp;
	int bmpi;
	BUFFER *b;
	int pos;
	int x;
	int msg;
	if (__unlikely(!n)) return;
	if (__unlikely(n == 1) || __unlikely(n == 4))
		VFS$CLEAN_BUFFERS_SYNC((FS *)fs, sec, n);
	if (__unlikely(sec + n > (unsigned)fs->size) || __unlikely(sec + n < sec)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREE DATA OUT OF FILESYSTEM (START %08X, LEN %08X, FS SIZE %08X)", sec, n, (unsigned)fs->size);
		HPFS_ERROR(fs);
		return;
	}
	if (__likely(sec >= fs->dir_band_start) && __likely(sec < fs->dir_band_start + fs->n_dir_band)) {
		HPFS_FREE_IN_DIR_BAND(fs, sec, n);
		return;
	}
	while (__unlikely(fs->flags & HPFS_SPARE_USED)) {
		if (n < 4) break;
		if (sec & 3) {
			unsigned x;
			HPFS_FREE(fs, sec, x = 4 - (sec & 3));
			sec += x;
			n -= x;
			continue;
		}
		if (__unlikely(HPFS_FREE_EMERGENCY_DNODE(fs, sec)))
			break;
		sec += 4;
		n -= 4;
	}
	while (__unlikely((sec & 0x3fff) + n > 0x4000)) {
		unsigned x;
		HPFS_FREE(fs, sec, x = 0x4000 - (sec & 0x3fff));
		sec += x;
		n -= x;
	}
	if (__unlikely(!n)) return;
	bmpi = sec >> 14;
	if (__unlikely(bmpi >= fs->n_bitmaps))
		KERNEL$SUICIDE("HPFS_FREE: %s: BAD BITMAP TO FREE (START %08X, LEN %08X, FS SIZE %08X, N_BITMAPS %d)", fs->filesystem_name, sec, n, (unsigned)fs->size, fs->n_bitmaps);
	x = fs->bmp_desc[bmpi].max_free_run * 2 + n;
	if (__unlikely(x > 0x4000)) x = 0x4000;
	fs->bmp_desc[bmpi].max_free_run = x;
	bmp = HPFS_MAP_BITMAP(fs, bmpi, &b, &pos);
	if (__unlikely(!bmp)) return;
	VFS$MARK_BUFFER_DIRTY(b, pos, 4);
	msg = 0;
	sec &= 0x3fff;
	free_next:
	if (__unlikely(sec & 7) || __unlikely(n < 8)) {
		no_d:
		if (__unlikely((bmp[sec >> 3] >> (sec & 7)) & 1)) {
			if (!msg) KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREEING ALREADY FREE BLOCK AT %08X (LEN %08X)", (bmpi << 14) + sec, n), msg = 1, HPFS_ERROR(fs);
			else if (msg == 1) KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREEING MORE FREE BLOCKS"), msg = 2;
			goto skip_free;
		}
		bmp[sec >> 3] |= 1 << (sec & 7);
		if ((sec & 7) < 4) {
			if ((bmp[sec >> 3] & 0x0f) == 0x0f) fs->free_dnodes++;
		} else {
			if ((bmp[sec >> 3] & 0xf0) == 0xf0) fs->free_dnodes++;
		}
		x = fs->bmp_desc[bmpi].n_free + 1;
		if (__unlikely(x > 0x4000)) {
			x = 0x4000;
			if (!msg) KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREE BLOCK COUNT OVERFLOW"), msg = 1, HPFS_ERROR(fs);
		}
		fs->bmp_desc[bmpi].n_free = x;
		fs->free_space++;
		skip_free:
		sec++;
		n--;
	} else {
		if (__unlikely(bmp[sec >> 3])) goto no_d;
		fs->free_dnodes += 2;
		bmp[sec >> 3] = 0xff;
		x = fs->bmp_desc[bmpi].n_free + 8;
		if (__unlikely(x > 0x4000)) {
			x = 0x4000;
			if (!msg) KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FREE BLOCK COUNT OVERFLOW"), msg = 1, HPFS_ERROR(fs);
		}
		fs->bmp_desc[bmpi].n_free = x;
		fs->free_space += 8;
		sec += 8;
		n -= 8;
	}
	if (__likely(n)) goto free_next;
	VFS$PUT_BUFFER(bmp);
}

static int HPFS_DIRECTORY_NEED_ZONESKIP(HPFSFNODE *f)
{
#define fs ((HPFSFS *)f->fs)
	unsigned idx;
	unsigned avg_free;
	if (__unlikely(f->ctime + PROPAGATE_HINT_TIME > time(NULL)))
		return 1;
	idx = f->fnode >> 14;
	if (__unlikely(idx >= fs->n_bitmaps)) idx %= fs->n_bitmaps;
	avg_free = fs->free_space / fs->n_bitmaps;
	return __likely(fs->bmp_desc[idx].n_free <= avg_free >> 2);
#undef fs
}

unsigned HPFS_DIRECTORY_HINT(HPFSFNODE *f)
{
#define fs ((HPFSFS *)f->fs)
	unsigned i, idx;
	unsigned avg_free;
	if (__likely(!HPFS_DIRECTORY_NEED_ZONESKIP(f)))
		return f->fnode;
	avg_free = fs->free_space / fs->n_bitmaps;
	idx = f->fnode >> 14;
	for (i = 1; i < fs->n_bitmaps; i *= 2) {
		idx += i;
		if (__unlikely(idx >= fs->n_bitmaps)) idx %= fs->n_bitmaps;
		if (fs->bmp_desc[idx].n_free >= avg_free)
			return idx << 14;
	}
	idx = f->fnode >> 14;
	for (i = 0; i < fs->n_bitmaps; i++) {
		idx++;
		if (__unlikely(idx >= fs->n_bitmaps)) idx %= fs->n_bitmaps;
		if (fs->bmp_desc[idx].n_free >= avg_free)
			return idx << 14;
	}
	/* Shoukdn't really happen unless the disk is corrupted */
	return f->fnode;
#undef fs
}

