#include "HPFS.H"
#include "STRUCT.H"

int HPFS_CHECK_SPARE(HPFSFS *fs, struct hpfs_spare_block *spare);

struct hpfs_spare_block *HPFS_MAP_SPARE(HPFSFS *fs, BUFFER **b)
{
	int r;
	struct hpfs_spare_block *spare;
	if (__unlikely(__IS_ERR(spare = VFS$READ_BUFFER_SYNC((FS *)fs, SP_SECTOR, 1, 0, b)))) return spare;
	if (__unlikely((r = HPFS_CHECK_SPARE(fs, spare)))) {
		VFS$PUT_BUFFER(spare);
		return __ERR_PTR(r);
	}
	if (__unlikely(spare->n_dnode_spares != spare->n_dnode_spares_free))
		fs->flags |= HPFS_SPARE_USED;
	return spare;
}

int HPFS_CHECK_SPARE(HPFSFS *fs, struct hpfs_spare_block *spare)
{
	if (__unlikely(spare->magic != __32CPU2LE(SP_MAGIC))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD HPFS SPARE BLOCK SIGNATURE: %08X", (unsigned)__32LE2CPU(spare->magic));
		return -EFSERROR;
	}
	if (__unlikely(__32LE2CPU(spare->n_dnode_spares) > SP_MAX_DNODES) ||
	    __unlikely(__32LE2CPU(spare->n_dnode_spares_free) > __32LE2CPU(spare->n_dnode_spares))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID NUMBER OF SPARE DNODES IN SPARE BLOCK: TOTAL %08X, USED %08X", (unsigned)__32LE2CPU(spare->n_dnode_spares), (unsigned)__32LE2CPU(spare->n_dnode_spares_free));
		return -EFSERROR;
	}
	return 0;
}

static __u32 HPFS_CHECKSUM(__u8 *data, unsigned len);

void HPFS_SPARE_MAKE_CHECKSUM(struct hpfs_spare_block *spare)
{
	spare->spare_crc = __32CPU2LE(0);
	spare->spare_crc = HPFS_CHECKSUM((__u8 *)spare, 512);
}

static __u32 HPFS_CHECKSUM(__u8 *data, unsigned len)
{
	__u32 checksum = 0;
	for (; len; data++, len--) {
		checksum += *data;
		checksum = (checksum << 7) + (checksum >> 25);
	}
	return __32CPU2LE(checksum);
}

struct dnode *HPFS_MAP_DNODE(HPFSFS *fs, unsigned dno, BUFFER **b, int allow_empty)
{
	int r;
	struct dnode *dnode;
	if (__unlikely(!dno) || __unlikely(dno >= (unsigned)fs->size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DNODE %08X OUT OF FILESYSTEM", dno);
		HPFS_ERROR(fs);
		return __ERR_PTR(-EFSERROR);
	}
	if (__unlikely(dno & 3)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID DNODE POINTER %08X", dno);
		HPFS_ERROR(fs);
		return __ERR_PTR(-EFSERROR);
	}
	if (__unlikely(__IS_ERR(dnode = VFS$READ_BUFFER_SYNC((FS *)fs, dno, 4, dnode_readahead(fs, dno), b)))) return dnode;
	if (__unlikely(r = HPFS_CHECK_DNODE(fs, dnode, dno, allow_empty))) {
		VFS$PUT_BUFFER(dnode);
		return __ERR_PTR(r);
	}
	return dnode;
}

int HPFS_CHECK_DNODE(HPFSFS *fs, struct dnode *dnode, unsigned sec, int allow_empty)
{
	struct hpfs_dirent *de;
	if (__unlikely(dnode->magic != __32CPU2LE(DNODE_MAGIC))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD MAGIC ON DNODE %08X: %08X", sec, __32LE2CPU(dnode->magic));
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__unlikely(__32LE2CPU(dnode->self) != sec)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD SELF POINER ON DNODE %08X: %08X", sec, __32LE2CPU(dnode->self));
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	de = (struct hpfs_dirent *)dnode->dirent;
	if (__unlikely(de->flags1 & DE_1_LAST) && __unlikely(!allow_empty)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "EMPTY DNODE %08X", sec);
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	next_de:
	if (__unlikely((char *)de < (char *)dnode->dirent) || __unlikely((char *)de > (char *)dnode + 2048 - sizeof(struct hpfs_dirent)) || __unlikely((int)(unsigned long)de & 3)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD DIRENT POSITION %08lX IN DNODE %08X", (unsigned long)de - (unsigned long)dnode, sec);
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__unlikely((char *)de + __16LE2CPU(de->length) < (char *)dnode) || __unlikely((char *)de + __16LE2CPU(de->length) > (char *)dnode + 2048) || __unlikely(__16LE2CPU(de->length) & 3)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD DIRENT LENGTH, DIRENT %08lX IN DNODE %08X", (unsigned long)de - (unsigned long)dnode, sec);
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__unlikely((unsigned long)(((struct hpfs_dirent *)NULL)->name + de->namelen + (de->flags1 & DE_1_DOWN ? 4 : 0)) > __16LE2CPU(de->length))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "NAME LENGTH GREATER THAN DIRENT LENGTH, DIRENT %08lX IN DNODE %08X", (unsigned long)de - (unsigned long)dnode, sec);
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__likely(!(de->flags1 & DE_1_LAST))) {
		de = (struct hpfs_dirent *)((char *)de + de->length);
		goto next_de;
	}
	if ((char *)de + __16LE2CPU(de->length) != (char *)dnode + __32LE2CPU(dnode->first_free)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAST DIRENT DOESN'T END AT FIRST_FREE, DNODE %08X", sec);
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	return 0;
}

static int HPFS_CHECK_BTREE(HPFSFS *fs, struct bplus_header *bp, int in, int ex, unsigned sec);

struct fnode *HPFS_MAP_FNODE(HPFSFNODE *f, BUFFER **b)
{
	int r;
	struct fnode *fnode;
	if (__unlikely(!f->fnode) || __unlikely(f->fnode >= (unsigned)f->fs->size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, f->fs->filesystem_name, "FNODE %08X OUT OF FILESYSTEM", f->fnode);
		HPFS_ERROR((HPFSFS *)f->fs);
		return __ERR_PTR(-EFSERROR);
	}
	if (__unlikely(!b)) {
		if (__unlikely(!(fnode = VFS$GET_BUFFER(f->fs, f->fnode, 1, (void *)&KERNEL$LIST_END, &KERNEL$PROC_KERNEL)))) return __ERR_PTR(-ENOENT);
		goto chk;
	}
	if (__unlikely(__IS_ERR(fnode = VFS$READ_BUFFER_SYNC(f->fs, f->fnode, 1, 0, b)))) return fnode;
	chk:
	if (__unlikely(r = HPFS_CHECK_FNODE((HPFSFS *)f->fs, fnode, f->fnode))) {
		VFS$PUT_BUFFER(fnode);
		return __ERR_PTR(r);
	}
	return fnode;
}

int HPFS_CHECK_FNODE(HPFSFS *fs, struct fnode *fnode, unsigned sec)
{
	int r;
	if (__unlikely(fnode->magic != __32CPU2LE(FNODE_MAGIC))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD MAGIC ON FNODE %08X: %08X", sec, __32LE2CPU(fnode->magic));
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__unlikely(fnode->flags2 & FN_2_DIRECTORY)) return 0;
	if (__unlikely(r = HPFS_CHECK_BTREE(fs, &fnode->btree, 8, 12, sec))) return r;
	if (__unlikely(!fnode->btree.n_used_nodes) && __unlikely(fnode->btree.flags1 & BP_1_INTERNAL)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INTERNAL FNODE %08X HAS NO ENTRIES", sec);
		return -EFSERROR;
	}
	return 0;
}

static int HPFS_CHECK_BTREE(HPFSFS *fs, struct bplus_header *bp, int in, int ex, unsigned sec)
{
	if (__unlikely(bp->flags1 & BP_1_INTERNAL)) {
		if (__unlikely(bp->n_free_nodes + bp->n_used_nodes != ex)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD NUMBER OF FREE(%d)/USED(%d) NODES IN INTERNAL BTREE AT %08X", (int)bp->n_free_nodes, (int)bp->n_used_nodes, sec);
			HPFS_ERROR(fs);
			return -EFSERROR;
		}
		if (__unlikely((unsigned long)&((struct bplus_header *)NULL)->u.internal[bp->n_used_nodes] != __16LE2CPU(bp->first_free))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FIRST_FREE(%d) DOES NOT MATCH USED COUNT(%d) IN INTERNAL BTREE AT %08X", (int)__16LE2CPU(bp->first_free), bp->n_used_nodes, sec);
			HPFS_ERROR(fs);
			return -EFSERROR;
		}
		return 0;
	} else {
		if (__unlikely(bp->n_free_nodes + bp->n_used_nodes != in)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD NUMBER OF FREE(%d)/USED(%d) NODES IN EXTERNAL BTREE AT %08X", (int)bp->n_free_nodes, (int)bp->n_used_nodes, sec);
			HPFS_ERROR(fs);
			return -EFSERROR;
		}
		if (__unlikely((unsigned long)&((struct bplus_header *)NULL)->u.external[bp->n_used_nodes] != __16LE2CPU(bp->first_free))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FIRST_FREE(%d) DOES NOT MATCH USED COUNT(%d) IN EXTERNAL BTREE AT %08X", (int)__16LE2CPU(bp->first_free), bp->n_used_nodes, sec);
			HPFS_ERROR(fs);
			return -EFSERROR;
		}
		return 0;
	}
}
struct anode *HPFS_MAP_ANODE(HPFSFS *fs, unsigned ano, BUFFER **b)
{
	int r;
	struct anode *anode;
	if (__unlikely(!ano) || __unlikely(ano >= (unsigned)fs->size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ANODE %08X OUT OF FILESYSTEM", ano);
		HPFS_ERROR(fs);
		return __ERR_PTR(-EFSERROR);
	}
	if (__unlikely(!b)) {
		if (__unlikely(!(anode = VFS$GET_BUFFER((FS *)fs, ano, 1, (void *)&KERNEL$LIST_END, &KERNEL$PROC_KERNEL)))) return __ERR_PTR(-ENOENT);
		goto chk;
	}
	if (__unlikely(__IS_ERR(anode = VFS$READ_BUFFER_SYNC((FS *)fs, ano, 1, 0, b)))) return anode;
	chk:
	if (__unlikely(r = HPFS_CHECK_ANODE(fs, anode, ano))) {
		VFS$PUT_BUFFER(anode);
		return __ERR_PTR(r);
	}
	return anode;
}

int HPFS_CHECK_ANODE(HPFSFS *fs, struct anode *anode, unsigned sec)
{
	int r;
	if (__unlikely(anode->magic != __32CPU2LE(ANODE_MAGIC))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD MAGIC ON ANODE %08X: %08X", sec, __32LE2CPU(anode->magic));
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__unlikely(__32LE2CPU(anode->self) != sec)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD SELF POINER ON ANODE %08X: %08X", sec, __32LE2CPU(anode->self));
		HPFS_ERROR(fs);
		return -EFSERROR;
	}
	if (__unlikely(r = HPFS_CHECK_BTREE(fs, &anode->btree, 40, 60, sec))) return r;
	if (__unlikely(!anode->btree.n_used_nodes)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ANODE %08X HAS NO ENTRIES", sec);
		return -EFSERROR;
	}
	return 0;
}

struct anode *HPFS_ALLOC_ANODE(HPFSFNODE *f, unsigned *ap)
{
	unsigned ano;
	struct anode *anode;
	unsigned one = 1;
	if (__unlikely(!(ano = HPFS_ALLOC((HPFSFS *)f->fs, f->fnode, &one))))
		return __ERR_PTR(-ENOSPC);
	if (__unlikely(__IS_ERR(anode = VFS$ALLOC_BUFFER_SYNC(f->fs, ano, 1, (void *)&KERNEL$LIST_END)))) {
		HPFS_FREE((HPFSFS *)f->fs, ano, 1);
		return anode;
	}
	anode->magic = __32CPU2LE(ANODE_MAGIC);
	anode->self = __32CPU2LE(ano);
	*ap = ano;
	return anode;
}


int HPFS_COMPARE(HPFSFS *fs, unsigned len1, unsigned char *s1, unsigned len2, unsigned char *s2)
{
	unsigned char c1, c2;
	again:
	if (__unlikely(!len1)) {
		if (__unlikely(!len2)) return 0;
		return -1;
	}
	if (__unlikely(!len2)) return 1;
	c1 = *s1;
	c2 = *s2;
	if (__likely(c1 < 0x80)) c1 = __upcasechr(c1);
	else c1 = fs->upcase[c1 - 0x80];
	if (__likely(c2 < 0x80)) c2 = __upcasechr(c2);
	else c2 = fs->upcase[c2 - 0x80];
	if (c1 < c2) return -1;
	if (c1 > c2) return 1;
	len1--;
	s1++;
	len2--;
	s2++;
	goto again;
}

void HPFS_UPCASE(HPFSFS *fs, unsigned char *name, unsigned len)
{
	while (len) {
		if (__unlikely(*name >= 0x80)) *name = fs->upcase[*name - 0x80];
		name++;
		len--;
	}
}

	/* Characters that are allowed in HPFS but not in DOS */
#define no_dos_char(c)	(				\
	__unlikely((c) == '+') ||			\
	__unlikely((c) == ',') ||			\
	__unlikely((c) == ';') ||			\
	__unlikely((c) == '=') ||			\
	__unlikely((c) == '[') ||			\
	__unlikely((c) == ']') 				\
	)

int HPFS_IS_NAME_LONG(unsigned char *name, unsigned len)
{
	unsigned i, j;
	for (i = 0; i < len && name[i] != '.'; i++)
		if (no_dos_char(name[i])) return 1;
	if (i - 1 > 8 - 1) return 1;
	if (__unlikely(i == len)) return 0;
	for (j = i + 1; j < len; j++)
		if (__unlikely(name[j] == '.') || no_dos_char(name[i])) return 1;
	return j - i > 4;
}

