#ifndef _HPFS_STRUCT_H
#define _HPFS_STRUCT_H

#include <SYS/TYPES.H>

/*
 *  linux/fs/hpfs/hpfs.h
 *
 *  HPFS structures by Chris Smith, 1993
 *
 *  a little bit modified by Mikulas Patocka, 1998-1999
 */

/* The paper

     Duncan, Roy
     Design goals and implementation of the new High Performance File System
     Microsoft Systems Journal  Sept 1989  v4 n5 p1(13)

   describes what HPFS looked like when it was new, and it is the source
   of most of the information given here.  The rest is conjecture.

   For definitive information on the Duncan paper, see it, not this file.
   For definitive information on HPFS, ask somebody else -- this is guesswork.
   There are certain to be many mistakes. */

/* Notation */

/*typedef __u32 secno;*/		/* sector number, partition relative */

/* sector 0 */

/* The boot block is very like a FAT boot block, except that the
   29h signature byte is 28h instead, and the ID string is "HPFS". */

#define BB_MAGIC 0xaa55

#define BB_SECTOR	0

struct hpfs_boot_block
{
  __u8 jmp[3];
  __u8 oem_id[8];
  __u8 bytes_per_sector[2];	/* 512 */
  __u8 sectors_per_cluster;
  __u8 n_reserved_sectors[2];
  __u8 n_fats;
  __u8 n_rootdir_entries[2];
  __u8 n_sectors_s[2];
  __u8 media_byte;
  __u16 sectors_per_fat;
  __u16 sectors_per_track;
  __u16 heads_per_cyl;
  __u32 n_hidden_sectors;
  __u32 n_sectors_l;		/* size of partition */
  __u8 drive_number;
  __u8 mbz;
  __u8 sig_28h;		/* 28h */
  __u8 vol_serno[4];
  __u8 vol_label[11];
  __u8 sig_hpfs[8];		/* "HPFS    " */
  __u8 pad[448];
  __u16 magic;			/* aa55 */
};


/* sector 16 */

/* The super block has the pointer to the root directory. */

#define SB_SECTOR	16

#define SB_MAGIC 0xf995e849

struct hpfs_super_block
{
  __u32 magic;			/* f995 e849 */
  __u32 magic1;			/* fa53 e9c5, more magic? */
  __u8 version;				/* version of a filesystem  usually 2 */
  __u8 funcversion;			/* functional version - oldest version
  					   of filesystem that can understand
					   this disk */
  __u16 zero;		/* 0 */
  __u32 root;			/* fnode of root directory */
  __u32 n_sectors;			/* size of filesystem */
  __u32 n_badblocks;			/* number of bad blocks */
  __u32 bitmaps;			/* pointers to free space bit maps */
  __u32 zero1;			/* 0 */
  __u32 badblocks;			/* bad block list */
  __u32 zero3;			/* 0 */
  __u32 last_chkdsk;			/* date last checked, 0 if never */
  __u32 last_optimize;			/* date last optimized, 0 if never */
  __u32 n_dir_band;			/* number of sectors in dir band */
  __u32 dir_band_start;			/* first sector in dir band */
  __u32 dir_band_end;			/* last sector in dir band */
  __u32 dir_band_bitmap;		/* free space map, 1 dnode per bit */
  __u8 volume_name[32];			/* not used */
  __u32 user_id_table;			/* 8 preallocated sectors - user id */
  __u32 zero6[103];			/* 0 */
};


/* sector 17 */

/* The spare block has pointers to spare sectors.  */

#define SP_SECTOR	17

#define SP_MAGIC 0xf9911849

#define SP_1_DIRTY			0x01	/* 0 clean, 1 "improperly stopped" */
#define SP_1_SPARE_USED			0x02	/* spare dirblks used */
#define SP_1_HOTFIX_USED		0x04	/* hotfixes used */
#define SP_1_BAD_SECTOR			0x08	/* bad sector, corrupted disk (???) */
#define SP_1_BAD_BITMAP			0x10	/* bad bitmap */
#define SP_1_FAST_FORMAT		0x20	/* partition was fast formatted */
#define SP_1_OLD_WROTE			0x40	/* old version wrote to partion */
#define SP_1_OLD_WROTE_1		0x80	/* old version wrote to partion (?) */

#define SP_2_INSTALL_DASD_LIMITS	0x01	/* HPFS386 flags */
#define SP_2_RESYNCH_DASD_LIMITS	0x02
#define SP_2_DASD_LIMITS_OPERATIONAL	0x04
#define SP_2_MULTIMEDIA_ACTIVE		0x08
#define SP_2_DCE_ACLS_ACTIVE		0x10
#define SP_2_DAST_LIMITS_DIRTY		0x20

#define SP_MAX_DNODES			101

struct hpfs_spare_block
{
  __u32 magic;			/* f991 1849 */
  __u32 magic1;			/* fa52 29c5, more magic? */

  __u8 flags1;
  __u8 flags2;
  __u8 mm_contlgulty;
  __u8 unused;

  __u32 hotfix_map;			/* info about remapped bad sectors */
  __u32 n_spares_used;		/* number of hotfixes */
  __u32 n_spares;			/* number of spares in hotfix map */
  __u32 n_dnode_spares_free;		/* spare dnodes unused */
  __u32 n_dnode_spares;		/* length of spare_dnodes[] list,
					   follows in this block*/
  __u32 code_page_dir;			/* code page directory block */
  __u32 n_code_pages;		/* number of code pages */
  __u32 super_crc;			/* on HPFS386 and LAN Server this is
  					   checksum of superblock, on normal
					   OS/2 unused */
  __u32 spare_crc;			/* on HPFS386 checksum of spareblock */
  __u32 zero1[15];			/* unused */
  __u32 spare_dnodes[101];	/* emergency free dnode list */
};

/* The bad block list is 4 sectors long.  The first word must be zero,
   the remaining words give n_badblocks bad block numbers.
   I bet you can see it coming... */

#define BAD_MAGIC 0
       
/* The hotfix map is 4 sectors long.  It looks like

       secno from[n_spares];
       secno to[n_spares];

   The to[] list is initialized to point to n_spares preallocated empty
   sectors.  The from[] list contains the sector numbers of bad blocks
   which have been remapped to corresponding sectors in the to[] list.
   n_spares_used gives the length of the from[] list. */


/* Sectors 18 and 19 are preallocated and unused.
   Maybe they're spares for 16 and 17, but simple substitution fails. */


/* The code page info pointed to by the spare block consists of an index
   block and blocks containing uppercasing tables.  I don't know what
   these are for (CHKDSK, maybe?) -- OS/2 does not seem to use them
   itself.  Linux doesn't use them either. */

/* block pointed to by spareblock->code_page_dir */

#define CP_DIR_MAGIC 0x494521f7

struct code_page_directory
{
  __u32 magic;			/* 4945 21f7 */
  __u32 n_code_pages;		/* number of pointers following */
  __u32 zero1[2];
  struct {
    __u16 ix;			/* index */
    __u16 code_page_number;	/* code page number */
    __u32 bounds;			/* matches corresponding word
					   in data block */
    __u32 code_page_data;		/* sector number of a code_page_data
					   containing c.p. array */
    __u16 index;		/* index in c.p. array in that sector*/
    __u16 unknown;		/* some unknown value; usually 0;
    					   2 in Japanese version */
  } array[31];				/* unknown length */
};

/* blocks pointed to by code_page_directory */

#define CP_DATA_MAGIC 0x894521f7

struct code_page_data
{
  __u32 magic;			/* 8945 21f7 */
  __u32 n_used;			/* # elements used in c_p_data[] */
  __u32 bounds[3];			/* looks a bit like
					     (beg1,end1), (beg2,end2)
					   one byte each */
  __u16 offs[3];		/* offsets from start of sector
					   to start of c_p_data[ix] */
  struct {
    __u16 ix;			/* index */
    __u16 code_page_number;	/* code page number */
    __u16 unknown;		/* the same as in cp directory */
    __u8 map[128];		/* upcase table for chars 80..ff */
    __u16 zero2;
  } code_page[3];
  __u8 incognita[78];
};


/* Free space bitmaps are 4 sectors long, which is 16384 bits.
   16384 sectors is 8 meg, and each 8 meg band has a 4-sector bitmap.
   Bit order in the maps is little-endian.  0 means taken, 1 means free.

   Bit map sectors are marked allocated in the bit maps, and so are sectors 
   off the end of the partition.

   Band 0 is sectors 0-3fff, its map is in sectors 18-1b.
   Band 1 is 4000-7fff, its map is in 7ffc-7fff.
   Band 2 is 8000-ffff, its map is in 8000-8003.
   The remaining bands have maps in their first (even) or last (odd) 4 sectors
     -- if the last, partial, band is odd its map is in its last 4 sectors.

   The bitmap locations are given in a table pointed to by the super block.
   No doubt they aren't constrained to be at 18, 7ffc, 8000, ...; that is
   just where they usually are.

   The "directory band" is a bunch of sectors preallocated for dnodes.
   It has a 4-sector free space bitmap of its own.  Each bit in the map
   corresponds to one 4-sector dnode, bit 0 of the map corresponding to
   the first 4 sectors of the directory band.  The entire band is marked
   allocated in the main bitmap.   The super block gives the locations
   of the directory band and its bitmap.  ("band" doesn't mean it is
   8 meg long; it isn't.)  */


/* dnode: directory.  4 sectors long */

/* A directory is a tree of dnodes.  The fnode for a directory
   contains one pointer, to the root dnode of the tree.  The fnode
   never moves, the dnodes do the B-tree thing, splitting and merging
   as files are added and removed.  */

#define DNODE_MAGIC   0x77e40aae

#define DN_1_ROOT_DNODE		0x01

struct dnode {
  __u32 magic;			/* 77e4 0aae */
  __u32 first_free;			/* offset from start of dnode to
					   first free dir entry */
  __u8 flags1;		/* Is it root dnode? */
  __u8 increment_me[3];		/* some kind of activity counter?
					   Neither HPFS.IFS nor CHKDSK cares
					   if you change this word */
  __u32 up;				/* (root dnode) directory's fnode
					   (nonroot) parent dnode */
  __u32 self;			/* pointer to this dnode */
  __u8 dirent[2028];		/* one or more dirents */
};

#define DE_1_FIRST		0x01	/* set on phony ^A^A (".") entry */
#define DE_1_HAS_ACL		0x02
#define DE_1_DOWN		0x04	/* down pointer present (after name) */
#define DE_1_LAST		0x08	/* set on phony \377 entry */
#define DE_1_HAS_EA		0x10	/* entry has EA */
#define DE_1_HAS_XTD_PERM	0x20	/* has extended perm list (???) */
#define DE_1_HAS_EXPLICIT_ACL	0x40
#define DE_1_HAS_NEEDEA		0x80	/* ?? some EA has NEEDEA set
					   I have no idea why this is
					   interesting in a dir entry */

#define DE_2_READ_ONLY		0x01	/* dos attrib */
#define DE_2_HIDDEN		0x02	/* dos attrib */
#define DE_2_SYSTEM		0x04	/* dos attrib */
#define DE_2_DIRECTORY		0x10	/* dos attrib */
#define DE_2_ARCHIVE		0x20	/* dos attrib */
#define DE_2_NOT_8x3		0x40	/* name is not 8.3 */

struct hpfs_dirent {
  __u16 length;		/* offset to next dirent */
  __u8 flags1;
  __u8 flags2;
  __u32 fnode;			/* fnode giving allocation info */
  __u32 write_date;			/* mtime */
  __u32 file_size;			/* file length, bytes */
  __u32 read_date;			/* atime */
  __u32 creation_date;			/* ctime */
  __u32 ea_size;			/* total EA length, bytes */
  __u8 no_of_acls;	/* number of ACL's (0-7) */
  __u8 ix;			/* code page index (of filename), see
					   struct code_page_data */
  __u8 namelen, name[1];	/* file name */
  /* dnode_secno down;	  btree down pointer, if present,
     			  follows name on next word boundary, or maybe it
			  precedes next dirent, which is on a word boundary. */
};

static __finline__ __u32 de_down_pointer(struct hpfs_dirent *de)
{
	return __32LE2CPU(*(__u32 *)((char *)de + __16LE2CPU(de->length) - 4));
}

static __finline__ struct hpfs_dirent *de_next_de(struct hpfs_dirent *de)
{
	return (struct hpfs_dirent *)((char *)de + __16LE2CPU(de->length));
}

static __finline__ int de_length(int namelen, int down)
{
	return ((((unsigned long)&((struct hpfs_dirent *)NULL)->name[namelen]) + 3) & ~3) + (down ? 4 : 0);
}

/* B+ tree: allocation info in fnodes and anodes */

/* dnodes point to fnodes which are responsible for listing the sectors
   assigned to the file.  This is done with trees of (length,address)
   pairs.  (Actually triples, of (length, file-address, disk-address)
   which can represent holes.  Find out if HPFS does that.)
   At any rate, fnodes contain a small tree; if subtrees are needed
   they occupy essentially a full block in anodes.  A leaf-level tree node
   has 3-word entries giving sector runs, a non-leaf node has 2-word
   entries giving subtree pointers.  A flag in the header says which. */

struct bplus_leaf_node
{
  __u32 file_secno;			/* first file sector in extent */
  __u32 length;			/* length, sectors */
  __u32 disk_secno;			/* first corresponding disk sector */
};

struct bplus_internal_node
{
  __u32 file_secno;			/* subtree maps sectors < this  */
  __u32 down;			/* pointer to subtree */
};

#define BP_1_HBFF		0x01	/* high bit of first free entry offset */
#define BP_1_FNODE_PARENT	0x20    /* ? we're pointed to by an fnode,
					   the data btree or some ea or the
					   main ea bootage pointer ea_secno */
					/* also can get set in fnodes, which
					   may be a chkdsk glitch or may mean
					   this bit is irrelevant in fnodes,
					   or this interpretation is all wet */
#define BP_1_BINARY_SEARCH	0x40	/* suggest binary search (unused) */
#define BP_1_INTERNAL		0x80	/* 1 -> (internal) tree of anodes
					   0 -> (leaf) list of extents */

struct bplus_header
{
  __u8 flags1;
  __u8 fill[3];
  __u8 n_free_nodes;		/* free nodes in following array */
  __u8 n_used_nodes;		/* used nodes in following array */
  __u16 first_free;		/* offset from start of header to
					   first free node in array */
  union {
    struct bplus_internal_node internal[0]; /* (internal) 2-word entries giving
					       subtree pointers */
    struct bplus_leaf_node external[0];	    /* (external) 3-word entries giving
					       sector runs */
  } u;
};

/* fnode: root of allocation b+ tree, and EA's */

/* Every file and every directory has one fnode, pointed to by the directory
   entry and pointing to the file's sectors or directory's root dnode.  EA's
   are also stored here, and there are said to be ACL's somewhere here too. */

#define FNODE_MAGIC 0xf7e40aae

#define FN_1_EA_ANODE		0x02	/* 1 -> ea_secno is an anode */
#define FN_2_DIRECTORY		0x01	/* 1 -> directory.  first & only extent
					   points to dnode. */

struct fnode
{
  __u32 magic;			/* f7e4 0aae */
  __u32 zero1[2];			/* read history */
  __u8 len, name[15];		/* true length, truncated name */
  __u32 up;			/* pointer to file's directory fnode */
  __u32 acl_size_l;
  __u32 acl_secno;
  __u16 acl_size_s;
  __u8 acl_anode;
  __u8 zero2;				/* history bit count */
  __u32 ea_size_l;			/* length of disk-resident ea's */
  __u32 ea_secno;			/* first sector of disk-resident ea's*/
  __u16 ea_size_s;		/* length of fnode-resident ea's */

  __u8 flags1;
  __u8 flags2;

  struct bplus_header btree;		/* b+ tree, 8 extents or 12 subtrees */
  union {
    struct bplus_leaf_node external[8];
    struct bplus_internal_node internal[12];
  } u;

  __u32 file_size;			/* file length, bytes */
  __u32 n_needea;			/* number of EA's with NEEDEA set */
  __u8 user_id[16];			/* unused */
  __u16 ea_offs;		/* offset from start of fnode
					   to first fnode-resident ea */
  __u8 dasd_limit_treshhold;
  __u8 dasd_limit_delta;
  __u32 dasd_limit;
  __u32 dasd_usage;
  __u8 ea[316];		/* zero or more EA's, packed together
					   with no alignment padding.
					   (Do not use this name, get here
					   via fnode + ea_offs. I think.) */
};


/* anode: 99.44% pure allocation tree */

#define ANODE_MAGIC 0x37e40aae

struct anode
{
  __u32 magic;			/* 37e4 0aae */
  __u32 self;			/* pointer to this anode */
  __u32 up;				/* parent anode or fnode */

  struct bplus_header btree;		/* b+tree, 40 extents or 60 subtrees */
  union {
    struct bplus_leaf_node external[40];
    struct bplus_internal_node internal[60];
  } u;

  __u32 fill[3];			/* unused */
};


/* extended attributes.

   A file's EA info is stored as a list of (name,value) pairs.  It is
   usually in the fnode, but (if it's large) it is moved to a single
   sector run outside the fnode, or to multiple runs with an anode tree
   that points to them.

   The value of a single EA is stored along with the name, or (if large)
   it is moved to a single sector run, or multiple runs pointed to by an
   anode tree, pointed to by the value field of the (name,value) pair.

   Flags in the EA tell whether the value is immediate, in a single sector
   run, or in multiple runs.  Flags in the fnode tell whether the EA list
   is immediate, in a single run, or in multiple runs. */

#define EA_1_INDIRECT		0x01	/* 1 -> value gives sector number
					   where real value starts */
#define EA_1_ANODE		0x02	/* 1 -> sector is an anode
					   that points to fragmented value */
#define EA_1_NEEDEA		0x80	/* required ea */

struct extended_attribute
{
  __u8 flags1;
  __u8 namelen;		/* length of name, bytes */
  __u8 valuelen_lo;		/* length of value, bytes */
  __u8 valuelen_hi;
  __u8 name[0];
  /*
    unsigned char name[namelen];	ascii attrib name
    unsigned char nul;			terminating '\0', not counted
    unsigned char value[valuelen];	value, arbitrary
      if this.indirect, valuelen is 8 and the value is
	unsigned length;		real length of value, bytes
	secno secno;			sector address where it starts
      if this.anode, the above sector number is the root of an anode tree
	which points to the value.
  */
};

static __finline__ struct extended_attribute *fnode_ea(struct fnode *fnode)
{
	return (struct extended_attribute *)((char *)fnode + __16LE2CPU(fnode->ea_offs) + __16LE2CPU(fnode->acl_size_s));
}

static __finline__ struct extended_attribute *fnode_end_ea(struct fnode *fnode)
{
	return (struct extended_attribute *)((char *)fnode + __16LE2CPU(fnode->ea_offs) + __16LE2CPU(fnode->acl_size_s) + __16LE2CPU(fnode->ea_size_s));
}

static __finline__ unsigned ea_valuelen(struct extended_attribute *ea)
{
	return ea->valuelen_lo + (ea->valuelen_hi << 8);
}

static __finline__ struct extended_attribute *ea_next_ea(struct extended_attribute *ea)
{
	return (struct extended_attribute *)((char *)ea + 5 + ea->namelen + ea_valuelen(ea));
}

static __finline__ unsigned read32(__u8 *p)
{
	return p[0] + (p[1] << 8) + (p[2] << 16) + (p[3] << 24);
}

static __finline__ unsigned ea_sec(struct extended_attribute *ea)
{
	return read32((__u8 *)ea + 9 + ea->namelen);
}

static __finline__ unsigned ea_len(struct extended_attribute *ea)
{
	return read32((__u8 *)ea + 5 + ea->namelen);
}

/*
   Local Variables:
   comment-column: 40
   End:
*/

#endif
