#include <VALUES.H>
#include <UNISTD.H>
#include <STDLIB.H>
#include <SPAD/SYNC.H>
#include <SPAD/SLAB.H>
#include <SPAD/LIBC.H>
#include <SPAD/HASH.H>
#include <SPAD/WQ.H>
#include <SPAD/DEV.H>
#include <SPAD/DL.H>
#include <ARCH/SETUP.H>

#include "VFS.H"

#define RESERVE_FNODES		(MAX_DEPTH * 2)
#define RESERVE_NAMES		(MAX_DEPTH * 2)
#define RESERVE_PAGEINRQ	2
#define RESERVE_SPAGES		1
#define RESERVE_SMALL_HASH	(MAX_DEPTH * 2)
#define RESERVE_BUFFERS		RESERVE_BZ
#define RESERVE_BRQ		(__unlikely(mem <= 16777216) ? 0 : 1)
#define RESERVE_WPAGES		2
#define RESERVE_Z		(__unlikely(mem < 12582912) ? 2 : __unlikely(mem <= 33554432) ? 4 : 8)
#define RESERVE_BZ		(__unlikely(mem < 12582912) ? 2 : __unlikely(mem <= 33554432) ? 4 : 8)

static AST_STUB SYNCER_END;
static void VFS_INIT_ROOT(HANDLE *h, void *fs_);
static void REAP_ALL_FS_DATA(FS *fs);
static int VFS_UMOUNT(void *p, void **release, const char * const argv[]);
/*static int VFS_DCTL(void *p, void **release, const char * const argv[]);*/

char *VFS$FNODE_DESCRIPTION(FNODE *f)
{
#if __DEBUG_INSECURE >= 1
	static char fnode_description[__MAX_STR_LEN * 3];
	if (f->parent) _snprintf(fnode_description, sizeof fnode_description, "%s:%s/%s", f->fs->filesystem_name, f->parent->name, f->name);
	else _snprintf(fnode_description, sizeof fnode_description, "%s:%s", f->fs->filesystem_name, f->name);
	return fnode_description;
#else
	return f->fs->filesystem_name;
#endif
}

char *VFS$HIDE_NAME(char *name)
{
#if __DEBUG_INSECURE >= 1
	return name;
#else
	return "-";
#endif
}

static void small_hash_ctor(struct __slhead *g, void *o)
{
	XLIST_HEAD *h = o;
	int i;
	/* constructor can't construct first pointer of an object */
	for (i = 1; i < SMALL_HASH_SIZE; i++)
		INIT_XLIST(&h[i]);
}

int VFS$MOUNT(int argc, const char * const argv[], const FSOPS *f, char *driver_name)
{
	int r;
	FS *fs;
	int i;
	union {
		MALLOC_REQUEST mrq;
		OPENRQ openrq;
		CHHRQ chhrq;
		IOCTLRQ ioctlrq;
	} u;
	const char * const *arg;
	int state;
	const char *opt, *optend, *val;
	char *dev, *name;
	void *p;
	PAGEZONE *z;
	PAGE *up;
	__u64 mem;
	int syncsecs = -2;
	static const VFSID vfsid = DEFAULT_VFSID;
	static const struct __param_table params[6] = {
		"RO", __PARAM_BOOL, FS_RO, FS_RO,
		"SYNC_TIME", __PARAM_INT, 0, MAXINT / JIFFIES_PER_SECOND,
		"BUFFER_FLUSH_TIME", __PARAM_INT, 0, MAXINT / JIFFIES_PER_SECOND,
		"", __PARAM_STRING, 1, __MAX_STR_LEN - 1,
		"", __PARAM_STRING, 1, __MAX_STR_LEN - 1,
		NULL, 0, 0, 0,
	};
	void *vars[6];

	if (__unlikely(memcmp(&vfsid, &f->vfsid, sizeof(VFSID)))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: VFS.DLL VERSION MISMATCH", f->name);
		r = -EPROTO;
		goto err0;
	}

	mem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_MAPPED);
	u.mrq.size = f->sizeof_FS + (__PAGES_PER_CLUSTER - 1) * sizeof(XLIST_HEAD) + __SECTORS_PER_PAGE_CLUSTER;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: OUT OF MEMORY", f->name);
		r = u.mrq.status;
		goto err0;
	}
	fs = u.mrq.ptr;
	RAISE_SPL(SPL_FS);
	memset(fs, 0, f->sizeof_FS);
	fs->slist = (XLIST_HEAD *)((char *)fs + f->sizeof_FS);
	fs->tmp_dirtmap = (char *)fs + f->sizeof_FS + (__PAGES_PER_CLUSTER - 1) * sizeof(XLIST_HEAD);

	fs->buffer_flush_time = VFS_DEFAULT_BUFFER_FLUSHTIME;

	vars[0] = &fs->flags;
	vars[1] = &syncsecs;
	vars[2] = &fs->buffer_flush_time;
	vars[3] = &dev;
	vars[4] = &name;
	vars[5] = NULL;

	if (f->flags & FS_RO_ONLY) fs->flags |= FS_RO;
	
	dev = name = NULL;

	arg = argv;
	state = 0;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &val)) {
		if (f->process_option) {
			if ((r = f->process_option(fs, opt, optend, val))) {
				if (r > 0) bads: r = -EBADSYN, _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SYNTAX ERROR", f->name);
				goto err1;
			}
		} else goto bads;
	}
	if (!dev || !name) goto bads;

	fs->buffer_flush_time *= JIFFIES_PER_SECOND;

	strcpy(fs->device_name, dev);
	/* it is guaranteed that it is at least 1 and at most __MAX_STR_LEN - 2 characters long */
	if (stpcpy(fs->filesystem_name, name)[-1] != ':') strcat(fs->filesystem_name, ":");

	__upcase(fs->filesystem_name);
	__upcase(fs->device_name);
#ifdef VFS_INO
	fs->st_dev = 0;
	dev = fs->device_name;
	quickcasehash(dev, *dev, fs->st_dev);
#endif

	u.openrq.flags = (fs->flags & FS_RO ? O_RDONLY : O_RDWR) | O_DIRECT;
	u.openrq.path = fs->device_name;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR OPENING DEVICE %s: %s", f->name, fs->device_name, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto err1;
	}
	fs->disk_handle_num = u.openrq.status;
	dev = KERNEL$HANDLE_PATH(fs->disk_handle_num);
	if (dev && strlen(dev) < sizeof(fs->device_name)) strcpy(fs->device_name, dev);
	u.chhrq.h = fs->disk_handle_num;
	u.chhrq.option = "DIRECT";
	u.chhrq.value = "";
	SYNC_IO(&u.chhrq, KERNEL$CHANGE_HANDLE);

	/***** init any static non-allocated items of FS here !! *****/
	INIT_LIST(&fs->synclist);
	WQ_INIT(&fs->syncer_wait, "VFS$SYNCER_WAIT");
	WQ_INIT(&fs->sync_done_wait, "VFS$SYNC_DONE_WAIT");
	WQ_INIT(&fs->sync_buf_done, "VFS$SYNC_BUF_DONE");
	fs->wantfree.status = RQS_PROCESSING;
	INIT_XLIST(&fs->wantfree_pages);
	INIT_XLIST(&fs->wantfree_fnodes);
	INIT_TIMER(&fs->sync_timer);
	SET_TIMER_NEVER(&fs->sync_timer);
	fs->size = -1;
	fs->fsops = f;
	fs->readdir_cookie_size = f->readdir_cookie_size > INTERNAL_READDIR_COOKIE_SIZE ? f->readdir_cookie_size : INTERNAL_READDIR_COOKIE_SIZE;

	VFS$ZINIT(&fs->z, VM_TYPE_CACHED_UNMAPPED, "PAGE");
	VFS$ZINIT(&fs->bz, VM_TYPE_CACHED_UNMAPPED, "BUFFER");
	KERNEL$SLAB_INIT(&fs->fnodes, f->sizeof_FNODE, 0, VM_TYPE_CACHED_MAPPED, f->ctor_FNODE, NULL, "VFS$FNODE");
	KERNEL$SLAB_INIT(&fs->names, MAX_NAMELEN, 0, VM_TYPE_CACHED_MAPPED, NULL, NULL, "VFS$NAME");
	KERNEL$SLAB_INIT(&fs->pageinrq, f->sizeof_PAGEINRQ, 0, VM_TYPE_CACHED_MAPPED, f->ctor_PAGEINRQ, NULL, "VFS$PAGEINRQ");
	KERNEL$SLAB_INIT(&fs->spages, sizeof(SPAGES) + (sizeof(SPAGE_ENTRY) << __PAGES_PER_CLUSTER_BITS), 0, VM_TYPE_CACHED_MAPPED, VFS_SPAGE_CTOR, NULL, "VFS$SPAGES");
	KERNEL$SLAB_INIT(&fs->small_hash, sizeof(XLIST_HEAD) * SMALL_HASH_SIZE, 0, VM_TYPE_CACHED_MAPPED, small_hash_ctor, NULL, "VFS$SMALL_HASH");
	KERNEL$SLAB_INIT(&fs->buffers, sizeof(BUFFER) + ((3 * BUFFER_BITMAP_SIZE / 8 + sizeof(long) - 1) & ~(sizeof(long) - 1)), 0, VM_TYPE_CACHED_MAPPED, BUFFER_CTOR, NULL, "VFS$BUFFER");
	KERNEL$SLAB_INIT(&fs->brq, sizeof(BRQ), 0, VM_TYPE_CACHED_MAPPED, BRQ_CTOR, NULL, "VFS$BRQ");
	KERNEL$SLAB_INIT(&fs->wpages, sizeof(WPAGE), 0, VM_TYPE_CACHED_MAPPED, WPAGE_CTOR, NULL, "VFS$WPAGE");
	
	WQ_INIT(&fs->freemem, "VFS$FREEMEM");
	WQ_INIT(&fs->pageinrq_wait, "VFS$PAGEINRQ_WAIT");
	WQ_INIT(&fs->freebuffer, "VFS$FREEBUFFER");

	INIT_LIST(&fs->clean_buffers);
	INIT_LIST(&fs->dirty_buffers);

	INIT_LIST(&fs->wpages_prep);
	WQ_INIT(&fs->wpage_wait, "VFS$WPAGE_WAIT");

	for (i = 0; i < __PAGES_PER_CLUSTER - 1; i++) INIT_XLIST(&fs->slist[i]);

	fs->syncproc = &KERNEL$PROC_KERNEL;

	u.ioctlrq.h = fs->disk_handle_num;
	u.ioctlrq.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
	u.ioctlrq.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ;
	u.ioctlrq.v.ptr = 0;
	u.ioctlrq.v.len = 0;
	u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&u.ioctlrq, KERNEL$IOCTL);
	if (u.ioctlrq.status >= 0) {
		fs->bio_request_size = u.ioctlrq.status;
		fs->buffer_readahead = u.ioctlrq.status / __PAGE_CLUSTER_SIZE - 1;
		fs->page_readahead = u.ioctlrq.status >> BIO_SECTOR_SIZE_BITS;
	} else {
		fs->bio_request_size = -ENOOP;
		fs->buffer_readahead = 0;
		fs->page_readahead = __SECTORS_PER_PAGE_CLUSTER;
	}
	/*__debug_printf("bio req: %x, buffer read %x, page read %x\n", fs->bio_request_size, fs->buffer_readahead, fs->page_readahead);*/
	if (__unlikely(fs->buffer_readahead < 0)) fs->buffer_readahead = 0;
	if (__unlikely(fs->buffer_readahead >= VFS_BUFFER_READAHEAD / __PAGE_CLUSTER_SIZE)) fs->buffer_readahead = VFS_BUFFER_READAHEAD / __PAGE_CLUSTER_SIZE - 1;
	if (__unlikely(fs->page_readahead < 1)) fs->page_readahead = 1;
	if (__unlikely(fs->page_readahead > VFS_PAGE_READAHEAD >> BIO_SECTOR_SIZE_BITS)) fs->page_readahead = VFS_PAGE_READAHEAD >> BIO_SECTOR_SIZE_BITS;

	if (__unlikely(r = KERNEL$SLAB_RESERVE(&fs->fnodes, RESERVE_FNODES))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->names, RESERVE_NAMES))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->pageinrq, RESERVE_PAGEINRQ))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->spages, RESERVE_SPAGES))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->small_hash, RESERVE_SMALL_HASH))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->buffers, RESERVE_BUFFERS))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->brq, RESERVE_BRQ))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->wpages, RESERVE_WPAGES))
	) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: OUT OF MEMORY", f->name);
		goto err2;
	}

	next_page:
	z = NULL;
	if (fs->z.reserved < RESERVE_Z) z = &fs->z;
	else if (fs->bz.reserved < RESERVE_BZ) z = &fs->bz;
	if (z) {
		up = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED);
		if (up) {
			VFS$ZRESERVE(z, up);
			goto next_page;
		}
		memwait:
		if (__unlikely(r = KERNEL$MEMWAIT_SYNC(__PAGE_CLUSTER_SIZE))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: OUT OF MEMORY", f->name);
			goto err2;
		}
		goto next_page;
	}
	if (!fs->bhash) {
		p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
		if (!p) goto memwait;
		fs->bhash = p;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) INIT_XLIST(&fs->bhash[i]);
	}
	if (!fs->ihash) {
		p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
		if (!p) goto memwait;
		fs->ihash = p;
		for (i = 0; i < IHASH_SIZE; i++) INIT_XLIST(&fs->ihash[i]);
	}
	if (!fs->ehash) {
		p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
		if (!p) goto memwait;
		fs->ehash = p;
		for (i = 0; i < IHASH_SIZE; i++) INIT_XLIST(&fs->ehash[i]);
	}
	if (RESERVE_PAGEINRQ > 1) {
		fs->dummy_pageinrq = __slalloc(&fs->pageinrq);
		if (__unlikely(!fs->dummy_pageinrq))
			KERNEL$SUICIDE("VFS$MOUNT: %s: CAN'T ALLOCATE DUMMY PAGEINRQ", fs->filesystem_name);
	}
	if (RESERVE_BRQ > 0) {
		fs->dummy_brq = __slalloc(&fs->brq);
		if (__unlikely(!fs->dummy_brq))
			KERNEL$SUICIDE("VFS$MOUNT: %s: CAN'T ALLOCATE DUMMY BRQ", fs->filesystem_name);
	}
	if (RESERVE_WPAGES > 1) {
		fs->dummy_wpage = __slalloc(&fs->wpages);
		if (__unlikely(!fs->dummy_wpage))
			KERNEL$SUICIDE("VFS$MOUNT: %s: CAN'T ALLOCATE DUMMY WPAGE", fs->filesystem_name);
	}
	r = f->mount(fs);
	if (r < 0) {
		if (!*KERNEL$ERROR_MSG()) if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR MOUNTING FILESYSTEM: %s", fs->filesystem_name, strerror(-r));
		goto err2;
	}
	if (!fs->max_filesize || (fs->max_filesize & __PAGE_CLUSTER_SIZE_MINUS_1))
		KERNEL$SUICIDE("VFS$MOUNT: %s: MAX_FILESIZE %016"__64_format"X RETURNED BY FSD", fs->filesystem_name, (__u64)fs->max_filesize);
	if (fs->pageio_mask != ((1 << fs->pageio_bits) - 1) || __unlikely(fs->pageio_bits < BIO_SECTOR_SIZE_BITS))
		KERNEL$SUICIDE("VFS$MOUNT: %s: PAGEIO_MASK %d, PAGEIO_BITS %d", fs->filesystem_name, fs->pageio_mask, fs->pageio_bits);
	fs->mounted = 1;

	fs->syncer.fn = SYNCER_END;
	fs->syncer.thread_main = SYNCER;
	fs->syncer.p = fs;
	fs->syncer.error = NULL;
	fs->syncer.cwd = NULL;
	fs->syncer.std_in = -1;
	fs->syncer.std_out = -1;
	fs->syncer.std_err = -1;
	fs->syncer.dlrq = NULL;
	fs->syncer.thread = NULL;
	fs->syncer.spawned = 0;
	CALL_IORQ(&fs->syncer, KERNEL$THREAD);
	WQ_WAIT_SYNC(&fs->syncer_wait);
	if (!fs->syncer.spawned) {
		r = fs->syncer.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR SPAWNING SYNCER THREAD: %s", fs->filesystem_name, strerror(-r));
		goto err3;
	}
	WQ_WAKE_ALL_PL(&fs->syncer_wait);
	fs->mounted = 2;

	LOWER_SPL(SPL_ZERO);
	r = KERNEL$REGISTER_DEVICE(fs->filesystem_name, driver_name, LNTE_VM_ESSENTIAL, fs, VFS_INIT_ROOT, NULL, NULL, NULL, VFS_UMOUNT, &fs->lnte, KERNEL$HANDLE_PATH(fs->disk_handle_num), NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR REGISTERING DEVICE: %s", fs->filesystem_name, strerror(-r));
		goto err4;
	}
	
	RAISE_SPL(SPL_FS);
	if (syncsecs == -2) syncsecs = VFS_DEFAULT_SYNCTIME;
	if (syncsecs > 0) {
		fs->sync_time = syncsecs * JIFFIES_PER_SECOND / 2;
		fs->sync_timer.fn = SYNC_TIMER;
		KERNEL$DEL_TIMER(&fs->sync_timer);
		KERNEL$SET_TIMER(fs->sync_time, &fs->sync_timer);
	}
	LOWER_SPL(SPL_ZERO);

	strcpy(KERNEL$ERROR_MSG(), fs->filesystem_name);
	fs->dlrq = KERNEL$TSR_IMAGE();
	return 0;

	err4:
	err3:
	err2:
	RAISE_SPL(SPL_FS);
	fs->flags |= FS_SHUTTING_DOWN;
	REAP_ALL_FS_DATA(fs);
	err1:
	LOWER_SPL(SPL_ZERO);
	free(fs);
	err0:
	return r;
}

static DECL_AST(SYNCER_END, SPL_FS, THREAD_RQ)
{
	FS *fs = RQ->p;
	if (__unlikely(!RQ->spawned)) {
		WQ_WAKE_ALL_PL(&fs->syncer_wait);
		RETURN;
	}
	if (__unlikely(!fs->terminate_syncer))
		KERNEL$SUICIDE("SYNCER_END: %s: UNEXPECTED SYNCER RETURN", fs->filesystem_name);
	fs->terminate_syncer = 2;
	WQ_WAKE_ALL_PL(&fs->syncer_wait);
	RETURN;
}

static void VFS_INIT_ROOT(HANDLE *h, void *fs_)
{
	FS *fs = fs_;
	h->fnode = fs->root;
	if (__likely(fs->root->flags & FNODE_DIRECTORY))
		h->op = &VFS_DIR_OPERATIONS;
	else
		h->op = &VFS_FILE_OPERATIONS;
	h->flags = 0;
	h->flags2 = CAP_ALL;
}

static void REAP_ALL_FS_DATA(FS *fs)
{
	CLOSERQ closerq;
	int i;
	if (KERNEL$SPL != SPL_X(SPL_FS))
		KERNEL$SUICIDE("REAP_ALL_FS_DATA AT SPL %08X", KERNEL$SPL);
	KERNEL$DEL_TIMER(&fs->sync_timer);
	if (fs->mounted == 2) {
		fs->terminate_syncer = 1;
		while (fs->terminate_syncer == 1) {
			WQ_WAKE_ALL_PL(&fs->syncer_wait);
			KERNEL$SLEEP(1);
		}
	}
	if (fs->mounted) {
		VFS$WRITE_BUFFERS(fs, 1);
		VFS$WAIT_FOR_BUFFERS(fs);
		VFS$SYNC_DISK(fs);
		if (fs->flags & FS_DIRTY) {
			fs->flags &= ~FS_DIRTY;
			fs->fsops->set_dirty(fs, 0);
		}
		fs->fsops->umount(fs, 0);
		VFS$WRITE_BUFFERS(fs, 1);
		VFS$WAIT_FOR_BUFFERS(fs);
		VFS$SYNC_DISK(fs);
	}
	if (fs->root) {
		if (fs->root->flags & (FNODE_FILE | FNODE_DIRECTORY)) VFS_FREE_FNODE_DATA(fs->root);
		VFS_FREE_EMPTY_FNODE(fs->root);
	}
	CLOSE_BUFFERS(fs);
	fsync(fs->disk_handle_num);
	closerq.h = fs->disk_handle_num;
	SYNC_IO(&closerq, KERNEL$CLOSE);
	if (fs->dummy_pageinrq) __slow_slfree(fs->dummy_pageinrq);
	if (fs->dummy_brq) __slow_slfree(fs->dummy_brq);
	if (fs->dummy_wpage) __slow_slfree(fs->dummy_wpage);
	KERNEL$SLAB_DESTROY(&fs->fnodes);
	KERNEL$SLAB_DESTROY(&fs->names);
	KERNEL$SLAB_DESTROY(&fs->pageinrq);
	KERNEL$SLAB_DESTROY(&fs->spages);
	KERNEL$SLAB_DESTROY(&fs->small_hash);
	KERNEL$SLAB_DESTROY(&fs->buffers);
	KERNEL$SLAB_DESTROY(&fs->brq);
	KERNEL$SLAB_DESTROY(&fs->wpages);
	VFS$ZDONE(&fs->z);
	VFS$ZDONE(&fs->bz);
	if (fs->bhash) {
		int i;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->bhash[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: NOT EMPTY BHASH[%d]", fs->filesystem_name, i);
		KERNEL$FREE_KERNEL_PAGE(fs->bhash, VM_TYPE_WIRED_MAPPED);
	}
	if (fs->ihash) {
		int i;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->ihash[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: NOT EMPTY IHASH[%d]", fs->filesystem_name, i);
		KERNEL$FREE_KERNEL_PAGE(fs->ihash, VM_TYPE_WIRED_MAPPED);
	}
	if (fs->ehash) {
		int i;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->ehash[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: NOT EMPTY EHASH[%d]", fs->filesystem_name, i);
		KERNEL$FREE_KERNEL_PAGE(fs->ehash, VM_TYPE_WIRED_MAPPED);
	}
	if (!LIST_EMPTY(&fs->clean_buffers)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: CLEAN BUFFER XLIST NOT EMPTY", fs->filesystem_name);
	if (!LIST_EMPTY(&fs->dirty_buffers)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: DIRTY BUFFER XLIST NOT EMPTY", fs->filesystem_name);
	if (!LIST_EMPTY(&fs->wpages_prep)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WPAGES_PREP LIST NOT EMPTY", fs->filesystem_name);
	if (!LIST_EMPTY(&fs->synclist)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: SYNCLIST NOT EMPTY", fs->filesystem_name);
	if (!XLIST_EMPTY(&fs->wantfree_pages)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WANTFREE_PAGES LIST NOT EMPTY", fs->filesystem_name);
	if (!XLIST_EMPTY(&fs->wantfree_fnodes)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WANTFREE_FNODES LIST NOT EMPTY", fs->filesystem_name);
	if (fs->n_wpages_prep != 0) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WPAGES PREP COUNT LEAKED: %ld", fs->filesystem_name, fs->n_wpages_prep);
	if (fs->n_wpages_io != 0) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WPAGES IO COUNT LEAKED: %ld", fs->filesystem_name, fs->n_wpages_io);
	if (fs->n_buffers != 0) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: BUFFER COUNT LEAKED: %ld", fs->filesystem_name, fs->n_buffers);
	if (fs->n_bighash != 0) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: BIGHASH LEAKED: %ld", fs->filesystem_name, fs->n_bighash);
	for (i = 0; i < __PAGES_PER_CLUSTER - 1; i++) if (!XLIST_EMPTY(&fs->slist[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: SPAGE ENTRY LEAKED AT POSITION %d", fs->filesystem_name, i);
	if (fs->total_dirty) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: TOTAL DIRTY COUNTER LEAKED: %lu", fs->filesystem_name, fs->total_dirty);
	WQ_WAKE_ALL(&fs->freemem);
	WQ_WAKE_ALL(&fs->pageinrq_wait);
	WQ_WAKE_ALL(&fs->freebuffer);
	WQ_WAKE_ALL(&fs->wpage_wait);
}

static void *DFS_WRITE_FNODE(FNODE *f, void *unused)
{
	WRITE_FNODE(f);
	return NULL;
}

static int VFS_UMOUNT(void *p, void **release, const char * const argv[])
{
	FNODE barrier;
	WQ *wq;
	FS *fs = p;
	int r;
	int state = 0;
	int flags = 0;
	const char * const *arg = argv;
	static const struct __param_table params[2] = {
		"BLOCK", __PARAM_BOOL, LN_DELETE_LOGICAL_BLOCK, LN_DELETE_LOGICAL_BLOCK,
		NULL, 0, 0, 0,
	};
	void *vars[3];
	vars[0] = &flags;
	vars[1] = NULL;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: SYNTAX ERROR", fs->filesystem_name);
		return -EBADSYN;
	}
	RAISE_SPL(SPL_FS);
	if (__likely(!(fs->flags & FS_BACKGROUND_INIT))) {
/* this is not required, just minimize time spent in unmounted state */
		VFS$DO_FOR_SUBTREE(fs->root, NULL, DFS_PREORDER, DFS_WRITE_FNODE);
		SYNC(fs, &barrier, &KERNEL$PROC_KERNEL);
		if ((r = WQ_WAIT_SYNC_CANCELABLE(&barrier.wait))) {
			CANCEL_SYNC(fs, &barrier);
			LOWER_SPL(SPL_ZERO);
			return r;
		}
	}
	LOWER_SPL(SPL_ZERO);
	if ((r = KERNEL$UNREGISTER_DEVICE(fs->lnte, flags))) return r;
	RAISE_SPL(SPL_FS);
	fs->flags |= FS_SHUTTING_DOWN;

	RAISE_SPL(SPL_VSPACE);
	if (fs->dummy_pageinrq) __slow_slfree(fs->dummy_pageinrq), fs->dummy_pageinrq = NULL;
	while (!KERNEL$SLAB_EMPTY(&fs->pageinrq)) {
		WQ_WAIT_SYNC(&fs->pageinrq_wait);
	}
	LOWER_SPL(SPL_FS);

	while ((wq = VFS$FREE_FNODE(fs->root))) {
		WQ_WAIT_SYNC(wq);
		LOWER_SPL(SPL_FS);
	}

	SYNC(fs, &barrier, &KERNEL$PROC_KERNEL);
	WQ_WAIT_SYNC(&barrier.wait);
	REAP_ALL_FS_DATA(fs);
	*release = fs->dlrq;
	free(fs);
	return 0;
}

#if 0
static int VFS_DCTL(void *p, void **release, const char * const argv[])
{
	FS *fs = p;
	if (argv[1] && argv[2] && !argv[3] && !_strcasecmp(argv[1], "LISTCACHE")) {
		FNODE *f, *f2;
		const char *p;
		int hash;
		RAISE_SPL(SPL_FS);
		f = fs->root;
		p = argv[2];
		next_dir:
		p += strspn(p, "/");
		if (*p) {
			const char *o = p;
			if (!(f->flags & FNODE_DIRECTORY))
				goto not_found;
			hash = 0;
			quickcasehash(p, *p && *p != '/', hash);
			hash &= f->u.d.hash_mask;
			XLIST_FOR_EACH(f2, &f->u.d.hash[hash], FNODE, hash_entry) if (__likely(!__strcasexcmp(f2->name, o, p))) {
				f = f2;
				goto next_dir;
			}
			not_found:
			LOWER_SPL(SPL_ZERO);
			return -ENOENT;
		}
		__critical_printf("FNODE: %08X, %s\n", f->flags, f->name);
		if (f->flags & FNODE_DIRECTORY) {
			for (hash = 0; hash <= f->u.d.hash_mask; hash++) {
				XLIST_FOR_EACH(f2, &f->u.d.hash[hash], FNODE, hash_entry) {
					__critical_printf("FNODE: %08X, %s\n", f2->flags, f2->name);

				}
			}
		}
		__critical_printf("--- (wpages prep %ld io %ld)\n", fs->n_wpages_prep, fs->n_wpages_io);
		LOWER_SPL(SPL_ZERO);
		return 0;
	}
	return -ENOOP;
}
#endif

DECL_IOCALL(DLL$LOAD, SPL_FS, DLINITRQ)
{
	int r;
	if ((r = BUFFER_INIT_GLOBAL())) {
		RQ->status = r;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER BUFFER VM ENTITY");
		ret1:
		RETURN_AST(RQ);
	};
	if ((r = FNODE_INIT_GLOBAL())) {
		RQ->status = r;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER FNODE VM ENTITY");
		ret2:
		BUFFER_TERM_GLOBAL();
		goto ret1;
	};
	if ((r = PAGE_INIT_GLOBAL())) {
		RQ->status = r;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER PAGE VM ENTITY");
		FNODE_TERM_GLOBAL();
		goto ret2;
	};
	RQ->status = 0;
	RETURN_AST(RQ);
}

DECL_IOCALL(DLL$UNLOAD, SPL_FS, DLINITRQ)
{
	PAGE_TERM_GLOBAL();
	FNODE_TERM_GLOBAL();
	BUFFER_TERM_GLOBAL();
	RQ->status = 0;
	RETURN_AST(RQ);
}
