#include <SPAD/SYSLOG.H>
#include <TIME.H>

#include "VFS.H"

static void BUFFER_WRITE(VMENTITY *e, PROC *p, int trashing);
static int BUFFER_SWAPOUT(VMENTITY *e);
static int BUFFER_CHECKMAP(VMENTITY *e);
static void FREE_BUFFER(BUFFER *buf);

static int buffer_vm_entity;

static const VMENTITY_T buffer_calls = { BUFFER_CHECKMAP, BUFFER_WRITE, BUFFER_SWAPOUT, "BUFFER" };

static AST_STUB BUFFER_READ_AST;
static AST_STUB BUFFER_READ_PARTIAL_AST;
static AST_STUB PREFETCH_DONE;

void BUFFER_CTOR(struct __slhead *fs_, void *buffer_)
{
	BUFFER *buf = buffer_;
	FS *fs = GET_STRUCT(fs_, FS, buffers);
	buf->fs = fs;
	WQ_INIT(&buf->wait, "VFS$BUF_WAIT");
	buf->brq.buf = buf;
	CACHE_CONSTRUCT_VM_ENTITY(&buf->e);
	buf->e.type = buffer_vm_entity;
	BRQ_CTOR(&fs->brq, &buf->brq);
}

void BRQ_CTOR(struct __slhead *fs_, void *brq_)
{
	BRQ *brq = brq_;
	FS *fs = GET_STRUCT(fs_, FS, brq);
	brq->rrq.h = fs->disk_handle_num;
	brq->rdesc.v.vspace = &KERNEL$PHYSICAL;
}


static __finline__ int BUFFER_HASH(unsigned off)
{
	off >>= __SECTORS_PER_PAGE_CLUSTER_BITS;
	return (off + (off >> 13)) & BUFFER_HASH_MASK;
}

static void BUFFER_WRITE(VMENTITY *e, PROC *p, int trashing)
{
	BUFFER *b;
	LOWER_SPL(SPL_FS);
	b = LIST_STRUCT(e, BUFFER, e);
	if (__unlikely((b->flags & (B_BUSY | B_WRITING | B_DIRTY)) == B_DIRTY)) VFS$WRITE_BUFFER(b, p);
}

static int BUFFER_SWAPOUT(VMENTITY *e)
{
	BUFFER *b;
	FS *fs;
	LOWER_SPL(SPL_FS);
	b = LIST_STRUCT(e, BUFFER, e);
	fs = b->fs;
	/*__debug_printf("buffer swapout(%LX, %X, %d, %d, %d)\n", b->blk, b->flags, fs->bz.allocated, fs->buffers.__n_pages, fs->buffers.__n_reserved_pages);*/
	if (__unlikely(!fs->bz.allocated) && __unlikely(!(fs->buffers.__n_pages > fs->buffers.__n_reserved_pages))) return 2;
	if (__unlikely((b->flags & (B_NEW | B_BUSY)) == (B_NEW | B_BUSY))) return 3;
	if (__unlikely(b->flags & (B_BUSY | B_WRITING | B_DIRTY))) {
		b->flags |= B_WANTFREE;
		if (__likely(!(b->flags & (B_BUSY | B_WRITING)))) {
			VFS$WRITE_BUFFER(b, &KERNEL$PROC_KERNEL);
			if (__unlikely(!(b->flags & (B_WRITING | B_DIRTY)))) goto f;
		}
		return 1;
	}
	if (__unlikely((unsigned)time(NULL) - b->time < 3)) return 3;
	f:
	FREE_BUFFER(b);
	return 0;
}

static int BUFFER_CHECKMAP(VMENTITY *e)
{
	LOWER_SPL(SPL_FS);
	return 0;
}

static DECL_IOCALL(BUF_RELEASE, SPL_FS, PAGE_RELEASE_REQUEST)
{
	void *v1, *v2;
	PAGE *p;
	BUFFER *buf;
	if (__unlikely(RQ->pg->release != BUF_RELEASE)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PAGE_RELEASE);
	buf = RQ->pg->fnode;
	if (__unlikely(buf->flags & (B_BUSY | B_WRITING))) {
		WQ_WAIT(&buf->wait, RQ, KERNEL$WAKE_PAGE_RELEASE);
		RETURN;
	}
	if (__unlikely(!(p = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED)))) {
		if (KERNEL$OOM(VM_TYPE_WIRED_UNMAPPED)) {
			RQ->status = -ENOMEM;
			RETURN_AST(RQ);
		}
		WQ_WAIT(&KERNEL$FREEMEM_WAIT, RQ, KERNEL$WAKE_PAGE_RELEASE);
		RETURN;
	}
	p->fnode = RQ->pg->fnode;
	p->release = BUF_RELEASE;
	v1 = KERNEL$MAP_PHYSICAL_BANK(buf->data);
	v2 = KERNEL$MAP_PHYSICAL_PAGE(p);
	memcpy(v2, v1, __PAGE_CLUSTER_SIZE);
	KERNEL$UNMAP_PHYSICAL_BANK(v1);
	KERNEL$UNMAP_PHYSICAL_BANK(v2);
	buf->data = KERNEL$PAGE_2_PHYS(p);
	KERNEL$FREE_USER_PAGE(RQ->pg, VM_TYPE_WIRED_UNMAPPED);
	RQ->status = 0;
	RETURN_AST(RQ);
}

void __finline__ *VFS$GET_BUFFER(FS *fs, __d_off off, unsigned nsec, BUFFER **ptr, void *proc)
{
	int hash;
	BUFFER *buf;
	hash = BUFFER_HASH(off);
	XLIST_FOR_EACH(buf, &fs->bhash[hash], BUFFER, hash) {
		if (__likely(buf->blk == (off & BUFFER_OFF_MASK))) {
			if (nsec & GET_BUFFER_FILEDATA) {
				nsec &= ~GET_BUFFER_FILEDATA;
				if (__unlikely(!(buf->flags & B_FILEDATA_VALID))) {
					return NULL;
				}
			}
			*ptr = buf;
			if (__likely(!(buf->flags & (B_BUSY | B_WRITING | B_PARTIAL)))) {
				ok:
				RAISE_SPL(SPL_CACHE);
				KERNEL$CACHE_TOUCH_VM_ENTITY(&buf->e, proc);
				LOWER_SPL(SPL_FS);
				return KERNEL$MAP_PHYSICAL_BANK(buf->data + (((unsigned long)off * BIO_SECTOR_SIZE) & __PAGE_CLUSTER_SIZE_MINUS_1));
			}
			if (__unlikely(buf->flags & (B_BUSY | B_WRITING))) break;
			{
				unsigned sec, last;
				sec = (unsigned)off & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
				if (__unlikely(sec + nsec > __SECTORS_PER_PAGE_CLUSTER))
					KERNEL$SUICIDE("VFS$GET_BUFFER: %s: REQUEST CROSSES PAGE BOUNDARY: %016"__d_off_format"X, %d", buf->fs->filesystem_name, off, nsec);
				last = sec + nsec - 1;
				sec &= ~buf->fs->block_mask;
				last |= buf->fs->block_mask;
				do {
					if (__unlikely(!BUFFER_VALIDMAP_TEST(buf, sec))) goto brk;
					sec++;
				} while (sec <= last);
				goto ok;
			}
			brk:
			break;
		}
	}
	return NULL;
}

void VFS$PUT_BUFFER(void *a)
{
	KERNEL$UNMAP_PHYSICAL_BANK(a);
}

void VFS$PUT_BUFFER_AND_WAIT(void *a)
{
	PAGE *p;
	BUFFER *b;
	FS *fs;
	__d_off blk;
	p = KERNEL$UNMAP_PHYSICAL_BANK_PAGE(a);
	b = p->fnode;
	fs = b->fs;
	blk = b->blk;
	while (__likely((b->flags & B_WRITING) != 0)) {
		unsigned hash;
		WQ_WAIT_SYNC(&b->wait);
		hash = BUFFER_HASH(blk);
		XLIST_FOR_EACH(b, &fs->bhash[hash], BUFFER, hash) {
			if (__likely(b->blk == blk)) {
				goto cont;
			}
		}
		break;
		cont:;
	}
	KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
}

void VFS$MARK_BUFFER_DIRTY(BUFFER *buf, unsigned boff, unsigned nblocks)
{
	if (__unlikely(boff + nblocks > __SECTORS_PER_PAGE_CLUSTER)) goto out;
	if (__unlikely((buf->flags & B_BUSY) != 0)) goto busy;
	if (__unlikely((boff | nblocks) & buf->fs->block_mask) || __unlikely(!nblocks)) goto badblk;
#if __DEBUG >= 2
	if (buf->flags & B_PARTIAL) {
		unsigned i;
		for (i = boff; i < boff + nblocks; i++)
			if (__unlikely(!BUFFER_VALIDMAP_TEST(buf, i)))
				KERNEL$SUICIDE("VFS$MARK_BUFFER_DIRTY: %s: BUFFER %016" __d_off_format"X IS NOT VALID", buf->fs->filesystem_name, buf->blk + i);
	}
#endif
	if (!(buf->flags & B_DIRTY)) {
		buf->flags |= B_DIRTY;
		buf->first_dirty = boff;
		buf->after_last_dirty = boff + nblocks;
		DEL_FROM_LIST(&buf->list);
		ADD_TO_LIST(&buf->fs->dirty_buffers, &buf->list);
		return;
	} else if (!(buf->flags & B_DIRTY_DISCONTIG)) {
		int i;
		if (boff > buf->after_last_dirty) goto conv_to_discont;
		if (boff + nblocks > buf->after_last_dirty) buf->after_last_dirty = boff + nblocks;
		else if (boff + nblocks < buf->first_dirty) goto conv_to_discont;
		if (boff < buf->first_dirty) buf->first_dirty = boff;
		return;
		conv_to_discont:
		BUFFER_DIRTMAP_CLEAR(buf);
		for (i = buf->first_dirty; i < buf->after_last_dirty; i++) BUFFER_DIRTMAP_SET(buf, i);
		buf->flags |= B_DIRTY | B_DIRTY_DISCONTIG;
		goto disc;
	} else {
		disc:
		if (boff < buf->first_dirty) buf->first_dirty = boff;
		if (boff + nblocks > buf->after_last_dirty) buf->after_last_dirty = boff + nblocks;
		while (nblocks--) {
			BUFFER_DIRTMAP_SET(buf, boff);
			boff++;
		}
		/* we could theroretically test and clear discontig flag here, but the cost is probably too high */
		return;
	}

	busy:
	KERNEL$SUICIDE("VFS$MARK_BUFFER_DIRTY: %s: BUFFER %016" __d_off_format"X IS BUSY", buf->fs->filesystem_name, buf->blk + boff);

	out:
	KERNEL$SUICIDE("VFS$MARK_BUFFER_DIRTY: %s: OUT OF DATA: BUFFER %016" __d_off_format"X, BLOCKS %u", buf->fs->filesystem_name, buf->blk + boff, nblocks);

	badblk:
	KERNEL$SUICIDE("VFS$MARK_BUFFER_DIRTY: %s: RANGE DOES NOT RESPECT ALIGNMENT %u: BUFFER %016" __d_off_format"X, BLOCKS %u", buf->fs->filesystem_name, buf->fs->block_mask, buf->blk + boff, nblocks);
}

static void CLEAN_BUFFER(BUFFER *b, unsigned boff, unsigned nblocks)
{
	unsigned u;
	char *dirtmap = b->fs->tmp_dirtmap;
	memset(dirtmap, 0, __SECTORS_PER_PAGE_CLUSTER);
#if __DEBUG >= 1
	if (__unlikely(b->first_dirty >= __SECTORS_PER_PAGE_CLUSTER) || __unlikely(b->after_last_dirty <= b->first_dirty) || __unlikely(b->after_last_dirty > __SECTORS_PER_PAGE_CLUSTER)) {
		KERNEL$SUICIDE("CLEAN_BUFFER: %s: BUFFER %016"__d_off_format"X, FIRST_DIRTY %d, AFTER_LAST_DIRTY %d", b->fs->filesystem_name, b->blk, b->first_dirty, b->after_last_dirty);
	}
#endif
	if (b->flags & B_DIRTY_DISCONTIG) {
		for (u = b->first_dirty; u < b->after_last_dirty; u++) if (BUFFER_DIRTMAP_TEST(b, u)) dirtmap[u] = 1;
	} else {
		memset(dirtmap + b->first_dirty, 1, b->after_last_dirty - b->first_dirty);
	}
	while (nblocks--) dirtmap[boff++] = 0;
	DEL_FROM_LIST(&b->list);
	ADD_TO_LIST(&b->fs->clean_buffers, &b->list);
	b->flags &= ~(B_DIRTY | B_DIRTY_DISCONTIG);
	for (u = 0; u < __SECTORS_PER_PAGE_CLUSTER; u += b->fs->block_mask + 1) if (dirtmap[u]) {
		VFS$MARK_BUFFER_DIRTY(b, u, b->fs->block_mask + 1);
	}
	/*{
		unsigned u;
		char dirtmap[SECTORS_PER_PAGE];
		if (b->flags & B_DIRTY_DISCONTIG) memcpy(dirtmap, b->dirtmap, SECTORS_PER_PAGE);
		else {
			if (__unlikely(b->first_dirty < 0) || __unlikely(b->first_dirty >= SECTORS_PER_PAGE) || __unlikely(b->after_last_dirty <= 0) || __unlikely(b->after_last_dirty > SECTORS_PER_PAGE)) {
				KERNEL$SUICIDE("CLEAN_BUFFER: %s: BUFFER %016"__d_off_format"X, FIRST_DIRTY %d, AFTER_LAST_DIRTY %d", b->fs->filesystem_name, b->blk, b->first_dirty, b->after_last_dirty);
			}
			memset(dirtmap, 0, SECTORS_PER_PAGE);
			for (u = b->first_dirty; u < b->after_last_dirty; u++) dirtmap[u] = 1;
		}
		for (u = 0; u < SECTORS_PER_PAGE; u++) if (dirtmap[u]) {
			__debug_printf("[%d]", u);
		}
	}*/
}

void VFS$CLEAN_BUFFERS_SYNC(FS *fs, __d_off sector, unsigned n_sectors)
{
	int hash;
	BUFFER *buf;
	unsigned ns;
	if (__unlikely(((unsigned)sector | n_sectors) & fs->block_mask)) goto badblk;
	next_buf:
	ns = __SECTORS_PER_PAGE_CLUSTER - ((unsigned)sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1);
	if (__likely(ns >= n_sectors)) ns = n_sectors;
	again:
	KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
	hash = BUFFER_HASH(sector);
	XLIST_FOR_EACH(buf, &fs->bhash[hash], BUFFER, hash) {
		if (__likely(buf->blk == (sector & BUFFER_OFF_MASK))) {
			if (__unlikely((buf->flags & (B_BUSY | B_WRITING)) != 0)) {
				WQ_WAIT_SYNC(&buf->wait);
				goto again;
			}
			if (__likely(!(buf->flags & B_DIRTY))) break;
			CLEAN_BUFFER(buf, (unsigned)sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, ns);
			break;
		}
	}
	if (__likely(!(n_sectors -= ns))) return;
	sector += ns;
	goto next_buf;

	badblk:
	KERNEL$SUICIDE("VFS$CLEAN_BUFFERS_SYNC: %s: RANGE DOES NOT RESPECT ALIGNMENT %u: BUFFER %016" __d_off_format"X, BLOCKS %u", fs->filesystem_name, fs->block_mask, sector, (unsigned)n_sectors);
}

void VFS$INVALIDATE_FILEBUFFERS(FS *fs, __d_off sector, unsigned n_sectors)
{
	BUFFER *buf;
	int hash;
	unsigned offs = (unsigned)sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
	sector -= offs;
	n_sectors += offs;
	next_buf:
	hash = BUFFER_HASH(sector);
	XLIST_FOR_EACH(buf, &fs->bhash[hash], BUFFER, hash) {
		if (__unlikely(buf->blk == sector)) {
			buf->flags &= ~B_FILEDATA_VALID;
			if (__likely((buf->flags & (B_DIRTY | B_BUSY | B_WRITING | B_FILEDATA_CREATED)) == B_FILEDATA_CREATED)) FREE_BUFFER(buf);
			break;
		}
	}
	if (__likely(n_sectors >= __SECTORS_PER_PAGE_CLUSTER)) {
		n_sectors -= __SECTORS_PER_PAGE_CLUSTER;
		sector += __SECTORS_PER_PAGE_CLUSTER;
		goto next_buf;
	}
}

BUFFER *CREATE_EMPTY_BUFFER(FS *fs, __d_off off, PROC *proc)
{
	PAGE *p;
	BUFFER *buf;
	unsigned hash = BUFFER_HASH(off);
	off &= BUFFER_OFF_MASK;
	XLIST_FOR_EACH_UNLIKELY(buf, &fs->bhash[hash], BUFFER, hash)
		if (__unlikely(buf->blk == off)) return (BUFFER *)((unsigned long)buf + 1);
	KERNEL$NOTIFY_ALLOC(proc, sizeof(BUFFER) + __PAGE_CLUSTER_SIZE);
	buf = __slalloc(&fs->buffers);
	if (__unlikely(!buf)) return NULL;
	buf->blk = off;
	p = PAGEZONE_ALLOC(&fs->bz, KERNEL$ALLOC_IO_PAGE, 0);
	if (__unlikely(!p)) {
		__slow_slfree(buf);
		return NULL;
	}
	p->fnode = buf;
	p->release = BUF_RELEASE;
	buf->data = KERNEL$PAGE_2_PHYS(p);
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_INSERT_VM_ENTITY(&buf->e, proc, 0);
	LOWER_SPL(SPL_FS);
	buf->flags = B_FILEDATA_VALID;
	BUFFER_USERMAP_CLEAR(buf);
	fs->n_buffers++;
	ADD_TO_XLIST(&fs->bhash[hash], &buf->hash);
	ADD_TO_LIST(&fs->clean_buffers, &buf->list);
	buf->time = time(NULL);
	return buf;
}

void VFS$READ_BUFFER(__d_off off, PAGEINRQ *pageinrq, unsigned reada)
{
	BUFFER *buf, *first, *last;
	FS *fs = pageinrq->fs;
	WQ *wq;
	if (__unlikely(off >= fs->size)) {
		pageinrq->status = -ERANGE;
		call_ret:
		CALL_AST(pageinrq);
		return;
	}
	if (__unlikely(fs->write_error) && !fs->ignore_write_errors) {
		pageinrq->status = -EIO;
		goto call_ret;
	}
	again:
	buf = CREATE_EMPTY_BUFFER(fs, off, pageinrq->tag.proc);
	if (__unlikely(!buf)) {
		if (pageinrq->fn == PREFETCH_DONE) {
			/* VFS$PREFETCH_BUFFER may be called while other buffer
			   is held --- we must not free buffers here */
			pageinrq->status = 0;
			goto call_ret;
		}
		if (__likely(!(wq = VFS$FREE_SOME_BUFFERS(fs)))) goto again;
		WQ_WAIT_PAGEINRQ(wq, pageinrq);
		return;
	}
	if (__unlikely((unsigned long)buf & 1)) {
		buf = (BUFFER *)((unsigned long)buf - 1);
		if (__unlikely(!(buf->flags & (B_BUSY | B_WRITING)))) {
			if (__likely(buf->flags & B_PARTIAL)) {
				/*if (__unlikely((buf->flags & B_WRITING) != 0)) goto wait;*/
				goto read_partial;
			}
			ok:
			pageinrq->status = 0;
			goto call_ret;
		}
		/*wait:*/
		WQ_WAIT_PAGEINRQ(&buf->wait, pageinrq);
		return;
	}

	buf->pageinrq = pageinrq;
	buf->brq.rrq.fn = BUFFER_READ_AST;
	buf->brq.rrq.sec = off & BUFFER_OFF_MASK;
	buf->brq.rrq.nsec = __SECTORS_PER_PAGE_CLUSTER;
	buf->brq.rrq.flags = BIO_READ;
	buf->brq.rrq.desc = &buf->brq.rdesc;
	buf->brq.rrq.proc = pageinrq->tag.proc;
	buf->brq.rrq.fault_sec = -1;
	buf->brq.next = NULL;
	buf->flags |= B_BUSY | B_WRQ | B_NEW | ((reada >> __BSR_CONST(READ_BUFFER_FILEDATA / B_FILEDATA_CREATED)) & B_FILEDATA_CREATED);
	reada &= ~READ_BUFFER_FILEDATA;

	first = buf;

	if (__unlikely(off + ((reada + 1) << __SECTORS_PER_PAGE_CLUSTER_BITS) > fs->size)) {
		void *pp;
		reada = 0;
		if (__unlikely(buf->brq.rrq.sec + __SECTORS_PER_PAGE_CLUSTER > fs->size)) {
			pp = KERNEL$MAP_PHYSICAL_BANK(buf->data);
			memset(pp, 0, __PAGE_CLUSTER_SIZE);
			KERNEL$UNMAP_PHYSICAL_BANK(pp);
			buf->brq.rrq.nsec = fs->size - buf->brq.rrq.sec;
			buf->brq.rdesc.v.len = buf->brq.rrq.nsec << BIO_SECTOR_SIZE_BITS;
			goto skip_len;
		}
	}

	read_ahead:
	buf->brq.rdesc.v.len = __PAGE_CLUSTER_SIZE;
	skip_len:
	buf->brq.rdesc.v.ptr = buf->data;
	last = buf;
	if (reada--) {
		off += __SECTORS_PER_PAGE_CLUSTER;
		if (__unlikely(KERNEL$MAY_ALLOC(pageinrq->tag.proc, 0) != NULL)) goto no_rda;
		buf = CREATE_EMPTY_BUFFER(fs, off, pageinrq->tag.proc);
		if (__unlikely(!buf) || __unlikely((unsigned long)buf & 1)) goto no_rda;

		buf->flags |= B_BUSY;
		first->brq.rrq.nsec += __SECTORS_PER_PAGE_CLUSTER;
		last->brq.rdesc.next = &buf->brq.rdesc;
		last->next = buf;
		buf->pageinrq = NULL;
		goto read_ahead;
	}
	no_rda:
	last->brq.rdesc.next = NULL;
	last->next = NULL;

	read_it:
	if (__likely(pageinrq->caller != NULL)) {
		CALL_IORQ_CANCELABLE(&first->brq.rrq, KERNEL$BIO, pageinrq->caller);
		return;
	}
	CALL_IORQ(&first->brq.rrq, KERNEL$BIO);
	return;

	read_partial: {
		int ret_z;
		unsigned i, j, k;
		i = (unsigned)off & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
		if (__unlikely(buf->flags & B_IOERROR)) {
			for (; i < __SECTORS_PER_PAGE_CLUSTER; i++) if (!BUFFER_VALIDMAP_TEST(buf, i)) goto nvx;
			goto ok;
			nvx:
			j = i + 1;
			goto ij_ok;
		}
		if (!BUFFER_VALIDMAP_TEST(buf, i)) {
			for (; i && !BUFFER_VALIDMAP_TEST(buf, i - 1); i--);
		} else {
			for (; i < __SECTORS_PER_PAGE_CLUSTER; i++) if (!BUFFER_VALIDMAP_TEST(buf, i)) goto nv;
			goto ok;
		}
		nv:
		j = i;
		j++;
		for (; j < __SECTORS_PER_PAGE_CLUSTER; j++) if (BUFFER_VALIDMAP_TEST(buf, j)) break;
		ij_ok:
		buf->read_from = i;
		buf->read_to = j;
		ret_z = 0;
		if (__unlikely(buf->blk + j > fs->size)) {
			char *p;
			unsigned endoff = (unsigned long)fs->size & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
			if (endoff < i) endoff = i;
			p = KERNEL$MAP_PHYSICAL_BANK(buf->data);
			memset(p + (endoff << BIO_SECTOR_SIZE_BITS), 0, (j - endoff) << BIO_SECTOR_SIZE_BITS);
			KERNEL$UNMAP_PHYSICAL_BANK(p);
			if (endoff <= i) ret_z = 1;
			else j = endoff;
		}
		buf->pageinrq = pageinrq;
		buf->brq.rrq.fn = BUFFER_READ_PARTIAL_AST;
		buf->brq.rrq.sec = buf->blk + i;
		buf->brq.rrq.nsec = j - i;
		buf->brq.rrq.flags = BIO_READ;
		buf->brq.rrq.desc = &buf->brq.rdesc;
		buf->brq.rrq.proc = pageinrq->tag.proc;
		buf->brq.rrq.fault_sec = -1;
		buf->brq.next = NULL;
		buf->flags |= B_BUSY | B_WRQ;
		for (k = i; k < j; k++) BUFFER_USERMAP_RESET(buf, k);
		buf->brq.rdesc.v.ptr = buf->data + (i << BIO_SECTOR_SIZE_BITS);
		buf->brq.rdesc.v.len = (j - i) << BIO_SECTOR_SIZE_BITS;
		buf->brq.rdesc.next = NULL;
		buf->next = NULL;
		first = buf;
		if (__unlikely(ret_z)) {
			buf->brq.rrq.status = 0;
			CALL_AST(&buf->brq.rrq);
			return;
		}
		goto read_it;
	}
}

static DECL_AST(BUFFER_READ_AST, SPL_FS, BIORQ)
{
	BUFFER *buf = GET_STRUCT(RQ, BUFFER, brq.rrq);
	BUFFER *buf2;
	PAGEINRQ *pageinrq = buf->pageinrq;
	if (pageinrq->caller) IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_FS), pageinrq->caller);
	pageinrq->status = 0;
	if (__likely(buf->flags & B_NEW)) buf->time = time(NULL);
	if (__unlikely(RQ->status < 0)) goto error;
	do {
#if __DEBUG >= 1
		if (__unlikely((buf->flags & (B_BUSY | B_DIRTY | B_DIRTY_DISCONTIG | B_WRITING)) != B_BUSY))
			KERNEL$SUICIDE("BUFFER_READ_AST: %s: BUFFER %016"__d_off_format"X HAS INVALID FLAGS FLAGS %08X", buf->fs->filesystem_name, buf->blk, buf->flags);
#endif
		if (__unlikely(buf->flags & B_WANTFREE)) {
			buf2 = buf->next;
			goto free_it;
		}
		buf->flags &= ~(B_BUSY | B_WRQ | B_NEW);
		WQ_WAKE_ALL(&buf->wait);
		buf = buf->next;
	} while (buf);
	RETURN_AST(pageinrq);

	error:
	do {
		buf2 = buf->next;
#if __DEBUG >= 1
		if (__unlikely((buf->flags & (B_BUSY | B_DIRTY | B_DIRTY_DISCONTIG | B_WRITING)) != B_BUSY))
			KERNEL$SUICIDE("BUFFER_READ_AST: %s: BUFFER %016"__d_off_format"X HAS INVALID FLAGS FLAGS %08X (IO ERROR %ld)", buf->fs->filesystem_name, buf->blk, buf->flags, RQ->status);
#endif
		if (buf == GET_STRUCT(RQ, BUFFER, brq.rrq)) {
			unsigned i;
			if (__unlikely(buf->flags & B_WANTFREE)) {
				goto free_it;
			}
			if (__unlikely(buf->flags & B_IOERROR)) buf->flags &= ~B_NEW;
			buf->flags = (buf->flags & ~(B_BUSY | B_WRQ)) | B_PARTIAL | B_IOERROR;
			WQ_WAKE_ALL(&buf->wait);
			for (i = 0; i < __SECTORS_PER_PAGE_CLUSTER; i++) {
				BUFFER_VALIDMAP_RESET(buf, i);
			}
		} else {
			free_it:
			buf->flags &= ~(B_BUSY | B_WRQ);
			FREE_BUFFER(buf);
		}
		buf = buf2;
	} while (buf);
	RETURN_AST(pageinrq);
}

int BUFFER_READ_PARTIAL(BUFFER *buf, int status, PROC *proc)
{
	unsigned i;
	int flg = buf->flags;
	if (__unlikely((flg & (B_BUSY | B_PARTIAL | B_WRITING)) != (B_BUSY | B_PARTIAL)))
		KERNEL$SUICIDE("BUFFER_READ_PARTIAL: %s: BUFFER %016"__d_off_format"X HAS INVALID FLAGS %08X", buf->fs->filesystem_name, buf->blk, buf->flags);
	if (__unlikely(flg & B_NEW)) buf->time = time(NULL);
	flg &= ~(B_BUSY | B_WRQ | B_NEW);
	buf->flags = flg;
	if (__unlikely(flg & B_WANTFREE)) {
		if (__unlikely(flg & B_DIRTY)) VFS$WRITE_BUFFER(buf, proc);
		if (__likely(!(buf->flags & (B_WRITING | B_DIRTY)))) FREE_BUFFER(buf);
		return 0;
	}
	if (__likely(status >= 0)) {
		for (i = buf->read_from; i < buf->read_to; i++) {
			if (__unlikely(BUFFER_VALIDMAP_TEST_SET(buf, i)))
				KERNEL$SUICIDE("BUFFER_READ_PARTIAL: %s: BUFFER %016"__d_off_format"X IS VALID AT POSITION %02X, FLAGS %08X", buf->fs->filesystem_name, buf->blk, i, buf->flags);
		}
		for (i = 0; i < __SECTORS_PER_PAGE_CLUSTER; i++) if (!BUFFER_VALIDMAP_TEST(buf, i)) goto no_complete;
		buf->flags &= ~B_PARTIAL;
		no_complete:
		WQ_WAKE_ALL(&buf->wait);
		return 0;
	} else {
		WQ_WAKE_ALL(&buf->wait);
		if (buf->flags & B_IOERROR) return status;
		buf->flags |= B_IOERROR;
		return 0;
	}
}

static DECL_AST(BUFFER_READ_PARTIAL_AST, SPL_FS, BIORQ)
{
	BUFFER *buf = GET_STRUCT(RQ, BUFFER, brq.rrq);
	PAGEINRQ *pageinrq = buf->pageinrq;
	if (__likely(pageinrq->caller != NULL)) IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_FS), pageinrq->caller);
	pageinrq->status = BUFFER_READ_PARTIAL(buf, buf->brq.rrq.status, pageinrq->tag.proc);
	RETURN_AST(pageinrq);
}

/* VFS$PREFETCH_BUFFER may be called while other buffer is held */
void VFS$PREFETCH_BUFFER(FS *fs, __d_off off, unsigned reada, PROC *proc)
{
	WQ *wq;
	PAGEINRQ *new_pgin;
	BUFFER *buf;
	unsigned hash = BUFFER_HASH(off);
	off &= BUFFER_OFF_MASK;
	XLIST_FOR_EACH_UNLIKELY(buf, &fs->bhash[hash], BUFFER, hash)
		if (__likely(buf->blk == off)) return;
	GET_PAGEINRQ(new_pgin, fs, proc, 1, 0, label, LOWER_SPL(SPL_FS); return;);
	new_pgin->fn = PREFETCH_DONE;
	VFS$READ_BUFFER(off, new_pgin, reada);
}

static DECL_AST(PREFETCH_DONE, SPL_FS, PAGEINRQ)
{
	FREE_PAGEINRQ(RQ, 0);
	RETURN;
}

static int READ_BUFFER_SYNC(FS *fs, __d_off off, unsigned reada)
{
	PAGEINRQ p;
	p.fs = fs;
	p.fn = KERNEL$UNBLOCK();
	p.caller = KERNEL$PREPARE_BLOCK();
	p.tag.proc = fs->syncproc;
	VFS$READ_BUFFER(off, &p, reada);
	KERNEL$BLOCK((IORQ *)(void *)&p);
	return p.status;
}

void *VFS$READ_BUFFER_SYNC(FS *fs, __d_off off, unsigned nsec, unsigned reada, BUFFER **ptr)
{
	void *p;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("VFS$READ_BUFFER_SYNC: %s: CALLED AT SPL %08X", fs->filesystem_name, KERNEL$SPL);
#endif
	again:
	KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
	p = VFS$GET_BUFFER(fs, off, nsec, ptr, fs->syncproc);
	if (__likely(p != NULL)) return p;
	p = __ERR_PTR(READ_BUFFER_SYNC(fs, off, reada));
	if (__likely(!p)) goto again;
	return p;
}

void *VFS$ALLOC_BUFFER_SYNC(FS *fs, __d_off off, unsigned nsec, BUFFER **ptr)
{
	unsigned s;
	__u8 *pp;
	unsigned sec;
	unsigned i;
	BUFFER *buf;
	WQ *wq;
	if (__unlikely(((unsigned)off | nsec) & fs->block_mask)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNALIGNED ALLOC (%016"__d_off_format"X, %08X)", off, nsec);
		return __ERR_PTR(-EFSERROR);
	}
	sec = (unsigned)off & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
	if (__unlikely(sec + nsec > __SECTORS_PER_PAGE_CLUSTER) || __unlikely(!nsec))
		KERNEL$SUICIDE("VFS$ALLOC_BUFFER_SYNC: %s: REQUEST CROSSES PAGE BOUNDARY: %016"__d_off_format"X, %d", fs->filesystem_name, off, nsec);
	again:
	KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
	buf = CREATE_EMPTY_BUFFER(fs, off, fs->syncproc);
	if (__unlikely(!buf)) {
		if (__likely(!(wq = VFS$FREE_SOME_BUFFERS(fs)))) goto again;
		WQ_WAIT_SYNC(wq);
		goto again;
	}
	if ((unsigned long)buf & 1) {
		buf = (BUFFER *)((unsigned long)buf - 1);
		if (__unlikely(sec + nsec > __SECTORS_PER_PAGE_CLUSTER) || __unlikely(!nsec))
			KERNEL$SUICIDE("VFS$ALLOC_BUFFER_SYNC: %s: REQUEST CROSSES PAGE BOUNDARY: %016"__d_off_format"X, %d", buf->fs->filesystem_name, buf->blk | sec, nsec);
		if (__likely(!(buf->flags & (B_BUSY | B_WRITING | B_PARTIAL)))) {
			ok:
			*ptr = buf;
			VFS$MARK_BUFFER_DIRTY(buf, sec, nsec);
			pp = KERNEL$MAP_PHYSICAL_BANK(buf->data | (sec << BIO_SECTOR_SIZE_BITS));
			memset(pp, 0, nsec << BIO_SECTOR_SIZE_BITS);
			return pp;
		}
		if (__unlikely(buf->flags & (B_BUSY | B_WRITING))) {
			WQ_WAIT_SYNC(&buf->wait);
			goto again;
		}
		set_valid:
		s = sec + nsec - 1;
		do {
			BUFFER_VALIDMAP_SET(buf, s);
		} while (s-- != sec);
		for (s = 0; s < __SECTORS_PER_PAGE_CLUSTER; s++)
			if (!BUFFER_VALIDMAP_TEST(buf, s)) goto ok;
		buf->flags &= ~B_PARTIAL;
		goto ok;
	}
	buf->flags |= B_PARTIAL;
	for (i = 0; i < __SECTORS_PER_PAGE_CLUSTER; i++) {
		BUFFER_VALIDMAP_RESET(buf, i);
	}
	goto set_valid;
}

static void FREE_BUFFER(BUFFER *buf)
{
	FS *fs;
	if (__unlikely(buf->flags & (B_DIRTY | B_BUSY | B_WRITING))) KERNEL$SUICIDE("FREE_BUFFER: %s: BUFFER %016"__d_off_format"X IS NOT FREEABLE, FLAGS %08X", buf->fs->filesystem_name, buf->blk, buf->flags);
	DEL_FROM_LIST(&buf->list);
	DEL_FROM_LIST(&buf->hash);
	WQ_WAKE_ALL(&buf->wait);
	fs = buf->fs;
	PAGEZONE_FREE(&fs->bz, KERNEL$PHYS_2_PAGE(buf->data), 0);
	WQ_WAKE_ALL(&fs->freebuffer);
#if __DEBUG >= 1
	if (__unlikely(!fs->n_buffers))
		KERNEL$SUICIDE("FREE_BUFFER: %s: BUFFER COUNT UNDERFLOW", fs->filesystem_name);
#endif
	fs->n_buffers--;
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_REMOVE_VM_ENTITY(&buf->e);
	LOWER_SPL(SPL_FS);
	__slfree(buf);
}

void CLOSE_BUFFERS(FS *fs)
{
	int i;
	BUFFER *b;
	if (__unlikely(!fs->bhash)) return;
	again:
	for (i = 0; i < BUFFER_HASH_SIZE; i++) {
		XLIST_FOR_EACH_UNLIKELY(b, &fs->bhash[i], BUFFER, hash) {
			if (__unlikely(b->flags & B_WRQ))
				KERNEL$CIO((IORQ *)(void *)&b->brq.rrq);
		}
	}
	for (i = 0; i < BUFFER_HASH_SIZE; i++) {
		while (__unlikely(!XLIST_EMPTY(&fs->bhash[i]))) {
			b = LIST_STRUCT(fs->bhash[i].next, BUFFER, hash);
			if (__unlikely(b->flags & (B_BUSY | B_WRITING))) {
				WQ_WAIT_SYNC(&b->wait);
				goto again;
			}
			b->flags &= ~(B_DIRTY | B_DIRTY_DISCONTIG);
			FREE_BUFFER(b);
		}
	}
}

int VFS$SYNC_DISK(FS *fs)
{
	BIORQ syncrq;
	if (__unlikely(fs->flags & FS_RO)) return 0;
	syncrq.h = fs->disk_handle_num;
	syncrq.flags = BIO_FLUSH;
	syncrq.sec = 0;
	syncrq.nsec = 1;
	syncrq.proc = fs->syncproc;
	syncrq.fault_sec = -1;
	SYNC_IO_CANCELABLE(&syncrq, KERNEL$BIO);
	if (__unlikely(syncrq.status < 0)) fs->write_error = 1;
	KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
	return syncrq.status;
}

static AST_STUB BUFFER_WROTE_DISCONTIG;
static AST_STUB BUFFERS_WROTE;

void VFS$WRITE_BUFFER(BUFFER *b, PROC *proc)
{
	int i, j;
	if (__unlikely((b->flags & (B_BUSY | B_WRITING)) != 0)) KERNEL$SUICIDE("VFS$WRITE_BUFFER: %s: BUFFER %016"__d_off_format"X ALREADY UNDER IO: %X", b->fs->filesystem_name, b->blk, b->flags);
	b->proc = proc;
	if (__unlikely(b->fs->write_error)) goto werr;
	ignore_errors:
	b->fs->write_buf_ops++;
	if (b->flags & B_DIRTY_DISCONTIG) {
		BRQ *rq;
		b->number_of_ios = 1;
		b->next = NULL;
		rq = &b->brq;
		i = b->first_dirty;
#if __DEBUG >= 1
		if (__unlikely(!BUFFER_DIRTMAP_TEST(b, i))) KERNEL$SUICIDE("VFS$WRITE_BUFFER: %s: DIRTMAP IS EMPTY, BLOCK %"__sec_t_format"X, DIRTY (%d,%d)", b->fs->filesystem_name, (__sec_t)b->blk, b->first_dirty, b->after_last_dirty);
#endif
		start:
		for (j = i; j < b->after_last_dirty; j++) {
			if (!BUFFER_DIRTMAP_TEST_RESET(b, j)) break;
		}
		rq->rrq.fn = BUFFER_WROTE_DISCONTIG;
		rq->write_sec = b->blk + i;
		rq->rrq.sec = b->blk + i;
		rq->rrq.nsec = j - i;
		rq->rrq.flags = BIO_WRITE;
		rq->rrq.desc = &rq->rdesc;
		rq->rrq.proc = b->proc;
		rq->rrq.fault_sec = -1;
		rq->rdesc.v.ptr = b->data + i * BIO_SECTOR_SIZE;
		rq->rdesc.v.len = (j - i) * BIO_SECTOR_SIZE;
		rq->rdesc.next = NULL;
		CALL_IORQ(&rq->rrq, KERNEL$BIO);
		if (j < b->after_last_dirty) {
			for (i = j + 1; i < b->after_last_dirty; i++) if (BUFFER_DIRTMAP_TEST(b, i)) {
				BRQ *rqq = rq;
				if (__likely((rq = __slalloc(&b->fs->brq)) != NULL)) {
					rqq->next = rq;
					rq->buf = b;
					b->number_of_ios++;
					goto start;
				} else {
					rqq->next = NULL;
					b->first_dirty = i;
					goto ret_disc;
				}
			}
			KERNEL$SUICIDE("VFS$WRITE_BUFFER: %s: DIRTMAP IS EMPTY AT END, BLOCK %"__sec_t_format"X, DIRTY (%d,%d)", b->fs->filesystem_name, (__sec_t)b->blk, b->first_dirty, b->after_last_dirty);
		}
		rq->next = NULL;
		DEL_FROM_LIST(&b->list);
		ADD_TO_LIST(&b->fs->clean_buffers, &b->list);
		b->flags &= ~(B_DIRTY | B_DIRTY_DISCONTIG);
		ret_disc:
		b->flags |= B_WRITING | B_WRQ;
		KERNEL$ACQUIRE_WRITEBACK_TAG(b->proc);
		return;
	} else if (__likely((b->flags & B_DIRTY) != 0)) {
		BUFFER *first;
		next_try:
		if (!b->first_dirty) {
			BUFFER *bb;
			int hash = BUFFER_HASH(b->blk - __SECTORS_PER_PAGE_CLUSTER);
			XLIST_FOR_EACH(bb, &b->fs->bhash[hash], BUFFER, hash) {
				if (__likely(bb->blk == b->blk - __SECTORS_PER_PAGE_CLUSTER)) {
					if ((bb->flags & (B_BUSY | B_DIRTY | B_DIRTY_DISCONTIG | B_WRITING)) == B_DIRTY) {
						if (bb->after_last_dirty == __SECTORS_PER_PAGE_CLUSTER) {
							bb->proc = b->proc;
							b = bb;
							goto next_try;
						}
					}
					break;
				}
			}
		}
		b->brq.next = NULL;
		b->brq.rrq.fn = BUFFERS_WROTE;
		b->brq.write_sec = b->blk + b->first_dirty;
		b->brq.rrq.sec = b->blk + b->first_dirty;
		b->brq.rrq.nsec = b->after_last_dirty - b->first_dirty;
		b->brq.rrq.flags = BIO_WRITE;
		b->brq.rrq.desc = &b->brq.rdesc;
		b->brq.rrq.proc = b->proc;
		b->brq.rrq.fault_sec = -1;
		first = b;
		b->flags |= B_WRITING | B_WRQ;

		next_write:
		b->flags &= ~(B_DIRTY | B_DIRTY_DISCONTIG);
		DEL_FROM_LIST(&b->list);
		ADD_TO_LIST(&b->fs->clean_buffers, &b->list);
		b->brq.rdesc.v.ptr = b->data + b->first_dirty * BIO_SECTOR_SIZE;
		b->brq.rdesc.v.len = (b->after_last_dirty - b->first_dirty) * BIO_SECTOR_SIZE;
		if (b->after_last_dirty == __SECTORS_PER_PAGE_CLUSTER) {
			BUFFER *bb;
			int hash = BUFFER_HASH(b->blk + __SECTORS_PER_PAGE_CLUSTER);
			XLIST_FOR_EACH(bb, &b->fs->bhash[hash], BUFFER, hash) {
				if (__likely(bb->blk == b->blk + __SECTORS_PER_PAGE_CLUSTER)) {
					if ((bb->flags & (B_BUSY | B_DIRTY | B_DIRTY_DISCONTIG | B_WRITING)) == B_DIRTY) {
						if (!bb->first_dirty) {
							b->next = bb;
							b->brq.rdesc.next = &bb->brq.rdesc;
							first->brq.rrq.nsec += bb->after_last_dirty - bb->first_dirty;
							bb->proc = first->proc;
							b = bb;
							b->flags |= B_WRITING;
							goto next_write;
						}
					}
					break;
				}
			}
		}
		b->next = NULL;
		b->brq.rdesc.next = NULL;
		CALL_IORQ(&first->brq.rrq, KERNEL$BIO);
		KERNEL$ACQUIRE_WRITEBACK_TAG(first->proc);
		return;
	} else {
		KERNEL$SUICIDE("VFS$WRITE_BUFFER: %s: BUFFER IS NOT DIRTY, BLOCK %"__sec_t_format"X, FLAGS %08X", b->fs->filesystem_name, (__sec_t)b->blk, b->flags);
	}

	werr:
	if (b->fs->ignore_write_errors) goto ignore_errors;
	b->flags &= ~(B_DIRTY | B_DIRTY_DISCONTIG);
	DEL_FROM_LIST(&b->list);
	ADD_TO_LIST(&b->fs->clean_buffers, &b->list);
	WQ_WAKE_ALL(&b->wait);
	WQ_WAKE_ALL(&b->fs->freebuffer);
	return;
}

static DECL_AST(BUFFER_WROTE_DISCONTIG, SPL_FS, BIORQ)
{
	BRQ *rq = GET_STRUCT(RQ, BRQ, rrq);
	BUFFER *b = rq->buf;
	FS *fs;
	if (__unlikely(RQ->status < 0)) goto err;
	wake:
	if (rq != &b->brq) __slfree(rq);
	if (--b->number_of_ios) RETURN;
	KERNEL$RELEASE_WRITEBACK_TAG(b->proc);
	b->fs->write_buf_ops--;
	b->flags &= ~(B_WRITING | B_WRQ);
	if (__unlikely(b->flags & B_DIRTY)) {
		VFS$WRITE_BUFFER(b, b->proc);
		if (__likely(b->flags & (B_DIRTY | B_WRITING))) RETURN;
	}
	WQ_WAKE_ALL(&b->wait);
	fs = b->fs;
	WQ_WAKE_ALL(&fs->freebuffer);
	if (FS_BUFFERS_DONE(fs)) {
		WQ_WAKE_ALL(&fs->sync_buf_done);
		if (FS_WPAGES_DONE(fs)) WQ_WAKE_ALL(&fs->syncer_wait);
	}
	if (__unlikely(b->flags & B_WANTFREE)) {
		FREE_BUFFER(b);
	}
	RETURN;
	err:
	/* !!! TODO: try to write each sector separately */
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, b->fs->filesystem_name, "ERROR WRITING BLOCK %016"__d_off_format"X: %s", rq->write_sec, strerror(-RQ->status));
	b->fs->write_error = 1;
	b->fs->write_error_sector = rq->write_sec;
	goto wake;
}

static DECL_AST(BUFFERS_WROTE, SPL_FS, BIORQ)
{
	BUFFER *b = GET_STRUCT(RQ, BUFFER, brq.rrq);
	BUFFER *buf2;
	FS *fs;
	if (__unlikely(RQ->status < 0)) goto err;
	wake:
	KERNEL$RELEASE_WRITEBACK_TAG(b->proc);
	WQ_WAKE_ALL(&b->fs->freebuffer);
	b->fs->write_buf_ops--;
	do {
		b->flags &= ~(B_WRITING | B_WRQ);
		WQ_WAKE_ALL(&b->wait);
		buf2 = b->next;
		fs = b->fs;
		if (__unlikely(b->flags & B_WANTFREE)) {
			if (__unlikely(b->flags & B_DIRTY)) VFS$WRITE_BUFFER(b, b->proc);
			if (__likely(!(b->flags & (B_DIRTY | B_WRITING)))) FREE_BUFFER(b);
		}
	} while ((b = buf2));
	if (FS_BUFFERS_DONE(fs)) {
		WQ_WAKE_ALL(&fs->sync_buf_done);
		if (FS_WPAGES_DONE(fs)) WQ_WAKE_ALL(&fs->syncer_wait);
	}
	RETURN;
	err:
	/* !!! TODO: try to write each sector separately */
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, b->fs->filesystem_name, "ERROR WRITING BLOCK %016"__d_off_format"X: %s", b->brq.write_sec, strerror(-RQ->status));
	b->fs->write_error = 1;
	b->fs->write_error_sector = b->brq.write_sec;
	goto wake;
}

void VFS$WRITE_BUFFERS(FS *fs, int syn)
{
	BUFFER *b, *bb = NULL;
	LIST_FOR_EACH(b, &fs->dirty_buffers, BUFFER, list) {
		if (__unlikely(!(b->flags & B_DIRTY)))
			KERNEL$SUICIDE("VFS$WRITE_BUFFERS: %s: CLEAN BUFFER ON DIRTY LIST, BLK %016"__d_off_format"X, FLAGS %X", fs->filesystem_name, b->blk, b->flags);
		if (!bb || b->blk < bb->blk) bb = b;
	}
	if (!bb) return;
	if (__unlikely(bb->flags & (B_BUSY | B_WRITING))) {
		if (syn) WQ_WAIT_SYNC(&bb->wait);
		goto b1;
	}
	VFS$WRITE_BUFFER(bb, fs->syncproc);
	b1:
	if (syn) KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
	while (!LIST_EMPTY(&fs->dirty_buffers)) {
		b = LIST_STRUCT(fs->dirty_buffers.next, BUFFER, list);
		if (__unlikely(!(b->flags & B_DIRTY)))
			KERNEL$SUICIDE("VFS$WRITE_BUFFERS: %s: CLEAN BUFFER ON DIRTY LIST (2), BLK %016"__d_off_format"X, FLAGS %X", fs->filesystem_name, b->blk, b->flags);
		if (__unlikely(b->flags & (B_BUSY | B_WRITING))) {
			if (!syn) return;
			WQ_WAIT_SYNC(&b->wait);
			goto b1;
		}
		VFS$WRITE_BUFFER(b, fs->syncproc);
	}
}

void VFS$WAIT_FOR_BUFFERS(FS *fs)
{
	if (fs->write_buf_ops) WQ_WAIT_SYNC(&fs->sync_buf_done);
	KERNEL$THREAD_MAY_BLOCK();
	SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
}

WQ *VFS$FREE_SOME_BUFFERS(FS *fs)
{
	BUFFER *b;
	int f = 0;
	unsigned long to_free = (fs->n_buffers + 3) >> 2;
	while (!LIST_EMPTY(&fs->clean_buffers) && to_free) {
		to_free--;
		b = LIST_STRUCT(fs->clean_buffers.prev, BUFFER, list);
		if (b->flags & (B_BUSY | B_WRITING | B_DIRTY)) {
			if (__unlikely(b->flags & B_DIRTY))
				KERNEL$SUICIDE("VFS$FREE_SOME_BUFFERS: %s: DIRTY BUFFER ON CLEAN LIST, BLK %016"__d_off_format"X, FLAGS %X", fs->filesystem_name, b->blk, b->flags);
			DEL_FROM_LIST(&b->list);
			ADD_TO_LIST(&fs->clean_buffers, &b->list);
			if (__unlikely(!to_free)) return &b->wait;
			continue;
		}
		FREE_BUFFER(b);
		f = 1;
	}
	while (!LIST_EMPTY(&fs->dirty_buffers)) {
		b = LIST_STRUCT(fs->dirty_buffers.prev, BUFFER, list);
		if (__unlikely(!(b->flags & B_DIRTY)))
			KERNEL$SUICIDE("VFS$FREE_SOME_BUFFERS: %s: CLEAN BUFFER ON DIRTY LIST, BLK %016"__d_off_format"X, FLAGS %X", fs->filesystem_name, b->blk, b->flags);
		if (b->flags & (B_BUSY | B_WRITING)) {
			DEL_FROM_LIST(&b->list);
			ADD_TO_LIST(&fs->dirty_buffers, &b->list);
			return f ? NULL : &b->wait;
		}
		DEL_FROM_LIST(&b->list);
		ADD_TO_LIST(&fs->dirty_buffers, &b->list);
		VFS$WRITE_BUFFER(b, &KERNEL$PROC_KERNEL);
	}
	return NULL;
}

int BUFFER_INIT_GLOBAL(void)
{
	return KERNEL$CACHE_REGISTER_VM_TYPE(&buffer_vm_entity, &buffer_calls);
}

void BUFFER_TERM_GLOBAL(void)
{
	KERNEL$CACHE_UNREGISTER_VM_TYPE(buffer_vm_entity);
}


