#include <STDLIB.H>
#include <TIME.H>
#include <SPAD/HASH.H>

#include "VFS.H"

static IO_STUB VFS_PROCESS_WANTFREE;

void VFS$NULL_FNODE_CALL(FNODE *f)
{
}

void *VFS$DO_FOR_SUBTREE(FNODE *ff, void *param, int flags, void *(*op)(FNODE *f, void *param))
{
	void *r;
	int ofl;
	FNODE *f = ff;
	FNODE *pf, *of;
	sub:
	if (!(flags & DFS_POSTORDER)) if (__unlikely((r = op(f, param)) != NULL)) return r;
	if (f->flags & FNODE_DIRECTORY) {
		CHECK_DIR_LISTS(f);
		if (f->u.d.n_dirty != 0) {
			f = LIST_STRUCT(f->u.d.dirty.next, FNODE, dirent_entry);
			goto sub;
		}
		scan_clean:
		CHECK_DIR_LISTS(f);
		if (!(flags & DFS_DIRTY) && f->u.d.n_clean != 0) {
			f = LIST_STRUCT(f->u.d.clean.next, FNODE, dirent_entry);
			goto sub;
		}
	}
	go_up:
	pf = f->parent;
	of = f;
	f = LIST_STRUCT(f->dirent_entry.next, FNODE, dirent_entry);
	ofl = of->flags;
	if (flags & DFS_POSTORDER) if (__unlikely((r = op(of, param)) != NULL)) return r;
	if (__unlikely(of == ff)) return NULL;
	if (__likely(f != LIST_STRUCT(&pf->u.d.clean, FNODE, dirent_entry)) && __likely(f != LIST_STRUCT(&pf->u.d.dirty, FNODE, dirent_entry))) goto sub;
	f = pf;
	if (ofl & FNODE_DIRTYLIST) goto scan_clean;
	goto go_up;
}

static void VFS_UNDIRTY_PAGE(PAGE *p)
{
	FNODE *f;
	WQ *wq;
	while (__unlikely((wq = KERNEL$VM_UNSET_WRITEABLE(p)) != NULL)) {
		WQ_WAIT_SYNC(wq);
		LOWER_SPL(SPL_FS);
			/* the page may have disappeared */
		return;
	}
	f = p->fnode;
	p->dirty_from = __PAGE_CLUSTER_SIZE;
	p->dirty_to = 0;
	p->flags &= ~PAGE_WRITEABLE;
	DEL_FROM_LIST(&p->node_entry);
	ADD_TO_LIST_END(&f->u.h.clean, &p->node_entry);
	LOWER_SPL(SPL_FS);
	WQ_WAKE_ALL(&p->wait);
	f->fs->fsops->unaccount(f, ACCOUNT_PAGE, p);
}

void VFS_UNDIRTY_PAGES(FNODE *f)
{
	PAGE *p;
	WQ_WAKE_ALL(&f->wait);
	if (!(f->flags & FNODE_HASH)) return;
	while (!LIST_EMPTY(&f->u.h.dirty)) {
		p = LIST_STRUCT(f->u.h.dirty.next, PAGE, node_entry);
		VFS_UNDIRTY_PAGE(p);
	}
}

	/* lowers SPL to SPL_FS */
static __finline__ void DO_FREE_PAGE(PAGE *p)
{
	FS *fs;
	FNODE *f = p->fnode;
	if (p->flags & PAGE_WRITEABLE) f->fs->fsops->unaccount(f, ACCOUNT_PAGE, p);
	DEL_FROM_LIST(&p->node_entry);
	DEL_FROM_LIST(&p->hash_entry);
	if (p->flags & PAGE_WANTFREE) DEL_FROM_LIST(&p->free_entry);
	WQ_WAKE_ALL(&p->wait);
	fs = ((FNODE *)p->fnode)->fs;
#if __DEBUG >= 4
	INIT_PAGE(p, 0xa8); /* slows down too much */
#endif
	PAGEZONE_FREE(&fs->z, p, 0);
	WQ_WAKE_ALL(&fs->freemem);
	if (SPLX_BELOW(SPL_X(SPL_VSPACE), SPL_X(SPL_CACHE))) RAISE_SPL(SPL_CACHE);
	if (SPLX_BELOW(SPL_X(SPL_CACHE), SPL_X(SPL_VSPACE))) LOWER_SPL(SPL_CACHE);
	KERNEL$CACHE_REMOVE_VM_ENTITY(&p->e);
	LOWER_SPL(SPL_FS);
}

/* can free only one page, return NULL on success or wait queue on failure */
WQ *VFS$FREE_PAGE(PAGE *p)
{
	WQ *wq;
	FS *fs;
	/*__debug_printf("free_page(%Lx)", p->id);*/
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_FS)))
		KERNEL$SUICIDE("VFS$FREE_PAGE AT SPL %08X", KERNEL$SPL);
#endif
	CHECK_PAGE(p, 0);
	wq = KERNEL$VM_UNMAP_PAGE(p);
	if (__unlikely(wq != NULL) || __unlikely(p->flags & PAGE_WRITEABLE)) {
		fs = ((FNODE *)p->fnode)->fs;
		if (__likely(!(p->flags & PAGE_WANTFREE))) {
			p->flags |= PAGE_WANTFREE;
			ADD_TO_XLIST(&fs->wantfree_pages, &p->free_entry);
		}
		if (p->flags & PAGE_WRITEABLE) {
			FNODE *f = p->fnode;
			VFS$SET_DIRTY(f);
			/*__debug_printf("write fnode '%s'", f->name);*/
			WRITE_FNODE(f);
			wq = &f->wait;
		}
		if (!fs->wantfree_active) {
			WQ_WAIT(wq, &fs->wantfree, VFS_PROCESS_WANTFREE);
			fs->wantfree_active = 1;
			/*__debug_printf("%s: hook wantfree (pflags %08x, fflags %08x)...", fs->filesystem_name, p->flags, ((FNODE *)p->fnode)->flags);*/
		}
		return wq;
	}
	DO_FREE_PAGE(p);
	/* LOWER_SPL(SPL_FS); done by DO_FREE_PAGE */
	return NULL;
}

/* this is needed when extending direct-io swapper --- extending is not done
   via direct io, but clean pages need to be thrown away or they would suck
   memory */
void VFS$FREE_CLEAN_PAGES(FNODE *f)
{
	PAGE *p;
	if (__unlikely(!(f->flags & FNODE_HASH))) return;
	LIST_FOR_EACH(p, &f->u.h.clean, PAGE, node_entry) {
		PAGE *pp = LIST_STRUCT(p->node_entry.prev, PAGE, node_entry);
		if (__unlikely(!VFS$FREE_PAGE(p))) p = pp;
		else LOWER_SPL(SPL_FS);
	}
}

void VFS$WRITE_AND_FREE_FNODE_PAGES(FNODE *f)
{
	PAGE *p;
	if (__unlikely(!(f->flags & FNODE_HASH))) return;
	LIST_FOR_EACH(p, &f->u.h.dirty, PAGE, node_entry) {
		PAGE *pp = LIST_STRUCT(p->node_entry.prev, PAGE, node_entry);
		if (__unlikely(!VFS$FREE_PAGE(p))) p = pp;
		else LOWER_SPL(SPL_FS);
	}
	VFS$FREE_CLEAN_PAGES(f);
}

static DECL_IOCALL(VFS_PROCESS_WANTFREE, SPL_DEV, IORQ)
{
	PAGE *p;
	FNODE *f;
	FS *fs;
	RAISE_SPL(SPL_FS);
	fs = GET_STRUCT(RQ, FS, wantfree);
	fs->wantfree_active = 0;
	/*__debug_printf("%s: process wantfree.\n", fs->filesystem_name);*/
	while (!XLIST_EMPTY(&fs->wantfree_pages)) {
		p = LIST_STRUCT(fs->wantfree_pages.next, PAGE, free_entry);
		if (VFS$FREE_PAGE(p)) break;
	}
	LOWER_SPL(SPL_FS);
	while (!XLIST_EMPTY(&fs->wantfree_fnodes)) {
		f = LIST_STRUCT(fs->wantfree_fnodes.next, FNODE, free_entry);
		if (VFS$FREE_FNODE(f)) break;
	}
	LOWER_SPL(SPL_FS);
	/*__debug_printf("done\n");*/
	RETURN;
}

void VFS_FREE_FNODE_DATA(FNODE *f)
{
		/* !!! FIXME: check correct freeing even in case disk errors */
	if (__likely((f->flags & (FNODE_FILE | FNODE_DIRECTORY))) && __likely(!(f->flags & (FNODE_UNCOMMITTED | FNODE_SRC_UNCOMMITTED | FNODE_KILLED)))) f->fs->fsops->free_fnode(f);
	RAISE_SPL(SPL_VSPACE);
	if (f->flags & FNODE_HASH) {
		f->flags &= ~FNODE_HASH;
		LOWER_SPL(SPL_FS);
		free_hash(f, f->u.h.hash);
		WQ_WAKE_ALL(&f->fs->freemem);
	} else if (__unlikely(f->flags & FNODE_SPAGE)) {
		KERNEL$SUICIDE("VFS_FREE_FNODE_DATA: SPAGE FNODE SHOULD NOT BE HERE, FLAGS %08X", f->flags);
	} else {
		LOWER_SPL(SPL_FS);
	}
	if (f->flags & FNODE_DIRECTORY) {
		if (f->u.d.hash) {
			free_hash(f, f->u.d.hash);
			WQ_WAKE_ALL(&f->fs->freemem);
		}
	}
	f->flags &= ~(FNODE_FILE | FNODE_DIRECTORY);
	if (__unlikely((f->flags & FNODE_SYNCLIST) != 0)) {
		DEL_FROM_LIST(&f->synclist_entry);
		f->flags &= ~FNODE_SYNCLIST;
		if (f->syncproc) KERNEL$RELEASE_WRITEBACK_TAG(f->syncproc), f->syncproc = NULL;
	}
	VFS$UNSET_DIRTY(f);
	VFS$KILL_FNODE_HANDLES(f);
}

int VFS$IS_FNODE_LOCKED(FNODE *f)
{
	if (__unlikely(f->readers != 0) || __unlikely((f->flags & (FNODE_WRITELOCK | FNODE_BUSY | FNODE_WANTFREE)) != 0)) return 1;
	return 0;
}

#define free_page(p, d)							\
do {									\
	if (p->id < off) continue;					\
	if (__unlikely(p->id == off) && __unlikely(o)) {		\
		if (__likely(p->valid_to > o)) {			\
			p->valid_to = o;				\
			if (__unlikely(p->valid_from >= p->valid_to)) {	\
				p->valid_from = __PAGE_CLUSTER_SIZE;	\
				p->valid_to = 0;			\
			}						\
		}							\
		if (d) {						\
			if (__likely(p->dirty_to > o)) {		\
				p->dirty_to = o;			\
				if (p->dirty_from >= p->dirty_to) {	\
					p->dirty_from = __PAGE_CLUSTER_SIZE;\
					p->dirty_to = 0;		\
				}					\
			}						\
		}							\
		CHECK_PAGE(p, 1);					\
		LOWER_SPL(SPL_FS);					\
	} else {							\
		pp = LIST_STRUCT(p->node_entry.prev, PAGE, node_entry);	\
		RAISE_SPL(SPL_VSPACE); /* not needed, keep test happy */\
		DO_FREE_PAGE(p);					\
		/* LOWER_SPL(SPL_FS); done by DO_FREE_PAGE */		\
		p = pp;							\
	}								\
} while (0)


WQ *VFS_FREE_FILE_PAGES(FNODE *f, _u_off_t off)
{
	int o;
	WQ *wq;
	PAGE *p, *pp;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_FS)))
		KERNEL$SUICIDE("VFS_FREE_FILE_PAGES: %s: CALLED AT SPL %08X", VFS$FNODE_DESCRIPTION(f), KERNEL$SPL);
#endif
	if (__unlikely(off > f->size)) return __ERR_PTR(-EPERM);
	if (__unlikely(f->flags & FNODE_WRITELOCK)) return &f->wait;
	o = (int)off & __PAGE_CLUSTER_SIZE_MINUS_1;
	off &= ~(_u_off_t)__PAGE_CLUSTER_SIZE_MINUS_1;
	if (__unlikely(!(f->flags & FNODE_HASH))) {
		if (__likely(f->flags & FNODE_SPAGE)) {
			if (__unlikely((wq = KERNEL$VM_UNMAP_SPAGE(&f->u.s.s)) != NULL)) return wq;
			if (__likely(!o)) {
				f->flags &= ~FNODE_SPAGE;
				LOWER_SPL(SPL_FS);
				VFS_FREE_SPAGE(f);
			} else {
				char *v = KERNEL$MAP_PHYSICAL_PAGE(f->u.s.s.page);
				memset(v + f->u.s.s.offset + o, 0, (f->u.s.s.n_pages << PG_SIZE_BITS) - o);
				KERNEL$UNMAP_PHYSICAL_BANK(v);
			}
		}
		goto ret;
	}
	RAISE_SPL(SPL_VSPACE);
	f->flags &= ~FNODE_HASH;
	LOWER_SPL(SPL_FS);
	LIST_FOR_EACH(p, &f->u.h.clean, PAGE, node_entry) {
		CHECK_PAGE(p, 0);
		if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p)) != NULL)) {
			f->flags |= FNODE_HASH;
			return wq;
		}
		LOWER_SPL(SPL_FS);
	}
	LIST_FOR_EACH(p, &f->u.h.dirty, PAGE, node_entry) {
		CHECK_PAGE(p, 0);
		if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p)) != NULL)) {
			f->flags |= FNODE_HASH;
			return wq;
		}
		LOWER_SPL(SPL_FS);
	}
	LIST_FOR_EACH(p, &f->u.h.clean, PAGE, node_entry) {
		free_page(p, 0);
	}
	LIST_FOR_EACH(p, &f->u.h.dirty, PAGE, node_entry) {
		free_page(p, 1);
	}
	RAISE_SPL(SPL_VSPACE);
	f->flags |= FNODE_HASH;
	LOWER_SPL(SPL_FS);

	ret:
	time(&f->mtime);
	VFS$SET_DIRTY(f);
	/*__debug_printf("trunc: %Lx -> %Lx\n", f->size, off | o);*/
	f->size = off | (unsigned)o;
	return NULL;
}

#if __DEBUG >= 2
void CHECK_DIR_LISTS(FNODE *fp)
{
	if (__likely(!(fp->flags & FNODE_SYNCLIST)) && __unlikely(fp->syncproc != NULL))
		KERNEL$SUICIDE("CHECK_DIR_LISTS: %s: FNODE HAS NON-EMPTY SYNCPROC %p, FLAGS %X", VFS$FNODE_DESCRIPTION(fp), fp->syncproc, fp->flags);
	if (__unlikely(!(fp->flags & FNODE_DIRECTORY)))
		KERNEL$SUICIDE("CHECK_DIR_LISTS: %s: NOT A DIRECTORY, FLAGS %X", VFS$FNODE_DESCRIPTION(fp), fp->flags);
#if !(__DEBUG >= 4)
	if (__unlikely(fp->u.d.n_dirty < 0) || __unlikely(fp->u.d.n_clean < 0))
		KERNEL$SUICIDE("CHECK_DIR_LISTS: %s: SUBDIR COUNTER UNDERFLOW (CLEAN %d, DIRTY %d)", VFS$FNODE_DESCRIPTION(fp), fp->u.d.n_clean, fp->u.d.n_dirty);
	if (__unlikely(!fp->u.d.n_clean != LIST_EMPTY(&fp->u.d.clean)) || __unlikely(!fp->u.d.n_dirty != LIST_EMPTY(&fp->u.d.dirty)))
		KERNEL$SUICIDE("CHECK_DIR_LISTS: %s: BAD COUNTERS: N_CLEAN %d, EMPTY CLEAN %d, N_DIRTY %d, EMPTY DIRTY %d", VFS$FNODE_DESCRIPTION(fp), fp->u.d.n_clean, LIST_EMPTY(&fp->u.d.clean), fp->u.d.n_dirty, LIST_EMPTY(&fp->u.d.dirty));
#else
	{
		int cln, dirt, hn, i;
		FNODE *f1;
		cln = 0;
		LIST_FOR_EACH(f1, &fp->u.d.clean, FNODE, dirent_entry) {
			if (__unlikely(f1->flags & FNODE_DIRTYLIST))
				KERNEL$SUICIDE("CHECK_DIR_LISTS: %s FNODE HAS DIRTYLIST FLAG BUT IS ON CLEAN LIST, FLAGS %X", VFS$FNODE_DESCRIPTION(f1), f1->flags);
			cln++;
		}
		dirt = 0;
		LIST_FOR_EACH(f1, &fp->u.d.dirty, FNODE, dirent_entry) {
			if (__unlikely(!(f1->flags & FNODE_DIRTYLIST)))
				KERNEL$SUICIDE("CHECK_DIR_LISTS: %s FNODE HAS NOT DIRTYLIST FLAG BUT IS ON DIRTY LIST, FLAGS %X", VFS$FNODE_DESCRIPTION(f1), f1->flags);
			dirt++;
		}
		if (__unlikely(cln != fp->u.d.n_clean) || __unlikely(dirt != fp->u.d.n_dirty))
			KERNEL$SUICIDE("CHECK_DIR_LISTS: %s BAD COUNTERS: N_CLEAN %d, COUNTED %d, N_DIRTY %d, COUNTED %d", VFS$FNODE_DESCRIPTION(fp), fp->u.d.n_clean, cln, fp->u.d.n_dirty, dirt);
		hn = 0;
		if (__likely(fp->u.d.hash != NULL)) for (i = fp->u.d.hash_mask; i >= 0; i--) XLIST_FOR_EACH_UNLIKELY(f1, &fp->u.d.hash[i], FNODE, hash_entry) {
			int h = 0;
			char *e = f1->name;
			quickcasehash(e, *e, h);
			h &= fp->u.d.hash_mask;
			if (__unlikely(h != i)) KERNEL$SUICIDE("CHECK_DIR_LISTS: %s ON BAD HASH LIST %x (SHOULD BE %x), FLAGS %X, PARENT HASH MASK %x", VFS$FNODE_DESCRIPTION(f1), i, h, f1->flags, fp->u.d.hash_mask);
			hn++;
		}
		if (__unlikely(hn != fp->u.d.n_clean + fp->u.d.n_dirty))
			KERNEL$SUICIDE("CHECK_DIR_LISTS: %s HAS LIST CORRUPTED (HASH: %d, N_CLEAN: %d, N_DIRTY %d)", VFS$FNODE_DESCRIPTION(fp), hn, fp->u.d.n_clean, fp->u.d.n_dirty);
	}
#endif
}
#endif

__finline__ void VFS_FREE_EMPTY_FNODE(FNODE *f)
{
	FS *fs;
	FNODE *fp;
#if __DEBUG >= 1
	if (__unlikely(!f->e.vm_queue.next))
		KERNEL$SUICIDE("VFS_FREE_EMPTY_FNODE: %s: FNODE ALREADY FREE", VFS$FNODE_DESCRIPTION(f));
	if (__unlikely((f->flags & (FNODE_SYNCLIST | FNODE_WRITELOCK | FNODE_BUSY | FNODE_FILE | FNODE_DIRECTORY)) != 0))
		KERNEL$SUICIDE("VFS_FREE_EMPTY_FNODE: %s: FNODE HAS INVALID FLAGS %08X", VFS$FNODE_DESCRIPTION(f), f->flags);
	if (__unlikely(!XLIST_EMPTY(&f->handles)))
		KERNEL$SUICIDE("VFS_FREE_EMPTY_FNODE: %s: FREEING FNODE WITH HANDLES", VFS$FNODE_DESCRIPTION(f));
	if (__unlikely(f->depth != (f->parent ? f->parent->depth + 1 : 0)))
		KERNEL$SUICIDE("VFS_FREE_EMPTY_FNODE: %s: DEPTH MISCOUNTED: %d != %d, FLAGS %08X", VFS$FNODE_DESCRIPTION(f), f->depth, (f->parent ? f->parent->depth + 1 : 0), f->flags);
	if (__unlikely((f->readers | f->dio_writers) != 0))
		KERNEL$SUICIDE("VFS_FREE_EMPTY_FNODE: %s: REQUESTS LEAKED ON FNODE, READERS %d, DIO_WRITERS %d", VFS$FNODE_DESCRIPTION(f), f->readers, f->dio_writers);
#endif
	FNODE_OUT_OF_WANTFREE(f);
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_REMOVE_VM_ENTITY(&f->e);
	LOWER_SPL(SPL_FS);
	DEL_FROM_LIST(&f->hash_entry);
	DEL_FROM_LIST(&f->dirent_entry);
	WQ_WAKE_ALL(&f->wait);
	if (__unlikely(f->name != f->inline_name)) {
		__slow_slfree(f->name);
		f->name = f->inline_name;
	}
	if (__likely((fp = f->parent) != NULL)) {
		if (__unlikely(f->flags & FNODE_DIRTYLIST)) fp->u.d.n_dirty--;
		else fp->u.d.n_clean--;
		CHECK_DIR_LISTS(fp);
		fp->flags &= ~FNODE_COMPLETE;
	}
	fs = f->fs;
	__slfree(f);
	WQ_WAKE_ALL(&fs->freemem);
}

WQ *VFS$FREE_FNODE(FNODE *f)
{
	FS *fs = f->fs;
	FNODE *top = f;
	again:
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_FS)))
		KERNEL$SUICIDE("VFS$FREE_FNODE AT SPL %08X", KERNEL$SPL);
#endif
	/*__debug_printf("free fnode (%s,%x)", f->name, f->flags);*/
	if (__unlikely(f->flags & (FNODE_DIRTY | FNODE_BUSY | FNODE_WRITELOCK)) || __unlikely(f->readers)) {
		if (__likely(f->flags & FNODE_DIRTY)) {
			WRITE_FNODE(f);
		}
		if (!(f->flags & FNODE_WANTFREE)) {
			f->flags |= FNODE_WANTFREE;
			ADD_TO_XLIST(&fs->wantfree_fnodes, &f->free_entry);
		}
		if (!fs->wantfree_active) {
			WQ_WAIT(&f->wait, &fs->wantfree, VFS_PROCESS_WANTFREE);
			fs->wantfree_active = 1;
			/*__debug_printf("%s: hook wantfree (fnode %x)...", fs->filesystem_name, f->flags);*/
		}
		return &f->wait;
	}
	if (f->flags & FNODE_DIRECTORY) {
		CHECK_DIR_LISTS(f);
		if (__unlikely(f->u.d.n_clean != 0) || __unlikely(f->u.d.n_dirty != 0)) {
			if (!(f->flags & FNODE_WANTFREE)) {
				f->flags |= FNODE_WANTFREE;
				ADD_TO_XLIST(&fs->wantfree_fnodes, &f->free_entry);
			}
			/* if (!fs->wantfree_active) ... set at function exit */
			if (f->u.d.n_dirty != 0) f = LIST_STRUCT(f->u.d.dirty.next, FNODE, dirent_entry);
			else f = LIST_STRUCT(f->u.d.clean.next, FNODE, dirent_entry);
			goto again;
		}
	}
	if (f->flags & FNODE_HASH) {
		PAGE *p;
		while (1) {
			WQ *wq;
			RAISE_SPL(SPL_VSPACE);
			if (!LIST_EMPTY(&f->u.h.clean)) p = LIST_STRUCT(f->u.h.clean.next, PAGE, node_entry);
			else if (!LIST_EMPTY(&f->u.h.dirty)) p = LIST_STRUCT(f->u.h.dirty.next, PAGE, node_entry);
			else break;
			LOWER_SPL(SPL_FS);

			if (__unlikely((wq = VFS$FREE_PAGE(p)) != NULL)) {
				if (!(f->flags & FNODE_WANTFREE)) {
					f->flags |= FNODE_WANTFREE;
					ADD_TO_XLIST(&fs->wantfree_fnodes, &f->free_entry);
				}
				if (!fs->wantfree_active) {
					/*__debug_printf("%s: add to wantfree 2...", fs->filesystem_name);*/
					WQ_WAIT(wq, &fs->wantfree, VFS_PROCESS_WANTFREE);
					fs->wantfree_active = 1;
				}
				return wq;
			}
		}
		LOWER_SPL(SPL_FS);
	}
	if (__unlikely(f->flags & FNODE_SPAGE)) {
		WQ *wq;
		if (__unlikely((wq = KERNEL$VM_UNMAP_SPAGE(&f->u.s.s)) != NULL)) return wq;
		f->flags &= ~FNODE_SPAGE;
		LOWER_SPL(SPL_FS);
		VFS_FREE_SPAGE(f);
	}
	VFS_FREE_FNODE_DATA(f);
	if (__likely(!(f->flags & FNODE_DONT_ZAP))) {
		if (__unlikely(!f->parent)) goto dz;
		VFS_FREE_EMPTY_FNODE(f);
	} else {
		dz:
		FNODE_OUT_OF_WANTFREE(f);
		f->flags = FNODE_NEGATIVE;
		WQ_WAKE_ALL(&f->wait);
	}
	if (__unlikely(f != top)) {
		/*__debug_printf("go up.");*/
		f = top;
		goto again;
	}
	if (__unlikely(!XLIST_EMPTY(&fs->wantfree_fnodes)) && __unlikely(!fs->wantfree_active)) {
		/*WQ_WAIT(&KERNEL$FREEMEM_WAIT, &fs->wantfree, VFS_PROCESS_WANTFREE);*/
		CALL_IORQ(&fs->wantfree, VFS_PROCESS_WANTFREE);
		fs->wantfree_active = 1;
	}
	/*__debug_printf("done.");*/
	return NULL;
}

void VFS$KILL_FNODE_HANDLES(FNODE *f)
{
	while (!XLIST_EMPTY(&f->handles)) {
		HANDLE *h = LIST_STRUCT(f->handles.next, HANDLE, fnode_entry);
		KERNEL$DETACH_HANDLE(h);
	}
}

static void *kshp(FNODE *f, void *param)
{
	IORQ *rq = param;
	PAGE *p;
	VFS$KILL_FNODE_HANDLES(f);
	if (!(f->flags & FNODE_HASH)) return NULL;
	RAISE_SPL(SPL_VSPACE);
	LIST_FOR_EACH(p, &f->u.h.dirty, PAGE, node_entry) {
		WQ *wq = KERNEL$VM_UNMAP_PAGE(p);
		if (__unlikely(wq != NULL)) goto bb;
		if (__unlikely(SPL_BUSY(SPL_FS))) goto bb;
	}
	LIST_FOR_EACH(p, &f->u.h.clean, PAGE, node_entry) {
		WQ *wq = KERNEL$VM_UNMAP_PAGE(p);
		if (__unlikely(wq != NULL)) goto bb;
		if (__unlikely(SPL_BUSY(SPL_FS))) goto bb;
	}
	LOWER_SPL(SPL_FS);
	goto s;
	bb:
	LOWER_SPL(SPL_FS);
	b:
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!LIST_EMPTY(&f->u.h.dirty))) {
		WQ *wq;
		PAGE *np = LIST_STRUCT(f->u.h.dirty.next, PAGE, node_entry);
		LOWER_SPL(SPL_FS);
		if (__unlikely(!(wq = VFS$FREE_PAGE(np)))) goto b;
		WQ_WAIT_F(wq, rq);
		LOWER_SPL(SPL_FS);
		VFS$FREE_FNODE(f);
		LOWER_SPL(SPL_FS);
		return (void *)1;
	}
	if (__unlikely(!LIST_EMPTY(&f->u.h.clean))) {
		WQ *wq;
		PAGE *np = LIST_STRUCT(f->u.h.clean.next, PAGE, node_entry);
		LOWER_SPL(SPL_FS);
		if (__likely(!(wq = VFS$FREE_PAGE(np)))) goto b;
		WQ_WAIT_F(wq, rq);
		LOWER_SPL(SPL_FS);
		VFS$FREE_FNODE(f);
		LOWER_SPL(SPL_FS);
		return (void *)1;
	}
	LOWER_SPL(SPL_FS);
	s:
	return NULL;
}

void *VFS$KILL_SUBTREE_HANDLES_AND_PAGES(FNODE *f, IORQ *rq)
{
	return VFS$DO_FOR_SUBTREE(f, rq, DFS_POSTORDER, kshp);
}

#define N_FREE_RUNS	4

int VFS$FREE_SOME_DATA(FS *fs)
{
	FNODE *f;
	int freed = 0;
	int ret = 0;
	int r;
	/*__debug_printf("fsd< ");*/
	next_run:
	f = fs->root;
	subdir:
	if (f->flags & FNODE_DIRECTORY) {
		FNODE *c;
		unsigned long cl = 0, dt = 0;
		unsigned long rn;
		LIST_FOR_EACH(c, &f->u.d.clean, FNODE, dirent_entry) cl++;
		LIST_FOR_EACH(c, &f->u.d.dirty, FNODE, dirent_entry) dt++;
		if (!cl && !dt) goto free_this;
		rn = random() % (cl + dt);
		if (rn < cl) {
			LIST_FOR_EACH(c, &f->u.d.clean, FNODE, dirent_entry) if (!rn--) {
				f = c;
				goto subdir;
			}
		} else {
			rn -= cl;
			LIST_FOR_EACH(c, &f->u.d.dirty, FNODE, dirent_entry) if (!rn--) {
				f = c;
				goto subdir;
			}
		}
		KERNEL$SUICIDE("VFS$FREE_SOME_DATA: %s: FNODE: CL == %ld, DT == %ld, RN == %ld", VFS$FNODE_DESCRIPTION(f), cl, dt, rn);
	}
	if (f->flags & FNODE_HASH && random() & 15) {
		PAGE *c;
		unsigned long cl = 0, dt = 0;
		unsigned long rn;
		LIST_FOR_EACH(c, &f->u.h.clean, PAGE, node_entry) cl++;
		LIST_FOR_EACH(c, &f->u.h.dirty, PAGE, node_entry) dt++;
		if (!cl && !dt) goto free_this;
		rn = random() % (cl + dt);
		if (rn < cl) {
			LIST_FOR_EACH(c, &f->u.h.clean, PAGE, node_entry) if (!rn--) goto freep;
		} else {
			rn -= cl;
			LIST_FOR_EACH(c, &f->u.h.dirty, PAGE, node_entry) if (!rn--) goto freep;
		}
		KERNEL$SUICIDE("VFS$FREE_SOME_DATA: %s: PAGE: CL == %ld, DT == %ld, RN == %ld", VFS$FNODE_DESCRIPTION(f), cl, dt, rn);
		freep:
		/*__debug_printf("freeing page %s:%Lx:", ((FNODE *)c->fnode)->name, c->id);*/
		r = !VFS$FREE_PAGE(c), ret |= r;
		LOWER_SPL(SPL_FS);
		/*__debug_printf("%d.", r);*/
		goto free_done;
	}
	free_this:
	/*__debug_printf("freeing fnode %s:", f->name);*/
	if (__likely(f != fs->root)) {
		r = !VFS$FREE_FNODE(f), ret |= r;
		LOWER_SPL(SPL_FS);
		/*__debug_printf("%d.", r);*/
	}
	free_done:
	if (++freed < N_FREE_RUNS) goto next_run;
	/*__debug_printf("done free(%d)", ret);*/
	/*__debug_printf("%d>", ret);*/
	return ret;
}

#if __DEBUG >= 4

void CHECK_FNODE_HANDLE(FNODE *f, HANDLE *h)
{
	HANDLE *hh;
	XLIST_FOR_EACH(hh, &f->handles, HANDLE, fnode_entry) {
		if (hh->fnode_entry.prev->next != &hh->fnode_entry)
			KERNEL$SUICIDE("CHECK_FNODE_HANDLE: %s: LIST DAMAGED", VFS$FNODE_DESCRIPTION(f));
	}
	XLIST_FOR_EACH(hh, &f->handles, HANDLE, fnode_entry) {
		if (h == hh) return;
	}
	if (__unlikely(!f->parent)) return;
	KERNEL$SUICIDE("CHECK_FNODE_HANDLE: %s: HANDLE NOT IN FNODE'S LIST", VFS$FNODE_DESCRIPTION(f));
}

#endif

#if __DEBUG >= 2

void CHECK_PAGE(PAGE *p, int trunc)
{
	FNODE *f = p->fnode;
	FS *fs = f->fs;
	if (__unlikely(p->flags & PAGE_BUSY && (((unsigned)p->io_to - 1) >= __PAGE_CLUSTER_SIZE || p->io_to & fs->pageio_mask))) goto fail;
	if (__unlikely((unsigned)p->valid_from > __PAGE_CLUSTER_SIZE) ||
	    __unlikely((unsigned)p->valid_to > __PAGE_CLUSTER_SIZE) ||
	    __unlikely((unsigned)p->dirty_from > __PAGE_CLUSTER_SIZE) ||
	    __unlikely((unsigned)p->dirty_to > __PAGE_CLUSTER_SIZE)) goto fail;
	if (__unlikely(p->valid_from >= p->valid_to))
		if (__unlikely(p->valid_from != __PAGE_CLUSTER_SIZE) || __unlikely(p->valid_to != 0)) goto fail;
	if (__likely(p->dirty_from >= p->dirty_to))
		if (__unlikely(p->dirty_from != __PAGE_CLUSTER_SIZE) || __unlikely(p->dirty_to != 0)) goto fail;
	if (__unlikely(p->dirty_from < p->valid_from) || __unlikely(p->dirty_to > p->valid_to)) goto fail;
	if (__unlikely(p->valid_from & fs->pageio_mask)) goto fail;
	if (__unlikely(p->valid_to & fs->pageio_mask) && __unlikely(p->id + p->valid_to < f->size) && __unlikely(!trunc)) goto fail;
	return;
	fail:
	KERNEL$SUICIDE("CHECK_PAGE: %s: PAGEIO MASK %X, FNODE SIZE %"__64_format"X, PAGE %"__64_format"X, FLAGS %X, VALID_FROM %X, VALID_TO %X, DIRTY_FROM %X, DIRTY_TO %X, IO_TO %X", VFS$FNODE_DESCRIPTION(p->fnode), fs->pageio_mask, (__u64)f->size, (__u64)p->id, p->flags, p->valid_from, p->valid_to, p->dirty_from, p->dirty_to, p->io_to);
}

#endif


