#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/WQ.H>
#include <ARCH/TIME.H>
#include <SPAD/SYSLOG.H>
#include <STDLIB.H>

#include "VFS.H"

void VFS$DEFAULT_COMMIT(FS *fs)
{
	VFS$WRITE_BUFFERS(fs, 1);
	fs->fsops->writepages(fs, 1);
	VFS$WAIT_FOR_BUFFERS(fs);
	VFS$SYNC_DISK(fs);
}

static void DOFSSYNC(FS *fs)
{
	if (fs->flags & (FS_SOMETHING_TO_COMMIT | FS_COMMIT_FREES_DATA)) fs->flags &= ~FS_SOMETHING_TO_COMMIT, fs->fsops->commit(fs);
	WQ_WAKE_ALL(&fs->sync_done_wait);
}

static void DOWRITEPAGES(FS *fs)
{
	fs->fsops->writepages(fs, 0);
}

static __finline__ void *WRITE_FNODE_FIRST(FNODE *f, void *param)
{
	if (f->flags & FNODE_SYNCLIST) DEL_FROM_LIST(&f->synclist_entry);
	else f->flags |= FNODE_SYNCLIST;
	ADD_TO_LIST(&f->fs->synclist, &f->synclist_entry);
	return NULL;
}

static __NORET_ATTR__ void TRY_SORT_CHILDREN_MISCOUNTED(FNODE *f)
{
	FNODE *ff;
	unsigned n = 0;
	LIST_FOR_EACH(ff, &f->u.d.dirty, FNODE, dirent_entry) n++;
	KERNEL$SUICIDE("TRY_SORT_CHILDREN_MISCOUNTED: %s: CHILDREN MISCOUNTED, N_DIRTY %u, COUNTED %u", VFS$FNODE_DESCRIPTION(f), f->u.d.n_dirty, n);
}

static int COMPARE_FNODES(const void *f1_, const void *f2_)
{
	FNODE *f1 = *(FNODE **)f1_;
	FNODE *f2 = *(FNODE **)f2_;
	return strcmp(f1->name, f2->name);
}

static void TRY_SORT_CHILDREN(FNODE *f)
{
	int syncl;
	int i;
	FNODE *ff;
	FNODE **array = malloc((size_t)f->u.d.n_dirty * sizeof(FNODE *));
	if (!array) return;
	i = 0;
	ff = LIST_STRUCT(f->u.d.dirty.next, FNODE, dirent_entry);
	for (i = 0; i < f->u.d.n_dirty; i++) {
		array[i] = ff;
		ff = LIST_STRUCT(ff->dirent_entry.next, FNODE, dirent_entry);
	}
	if (__unlikely(ff != LIST_STRUCT(&f->u.d.dirty, FNODE, dirent_entry)))
		TRY_SORT_CHILDREN_MISCOUNTED(f);
	for (i = 1; i < f->u.d.n_dirty; i++) {
		if (__unlikely(COMPARE_FNODES(&array[i - 1], &array[i]) > 0)) goto do_sort;
	}
	goto free_array;
	do_sort:
	qsort(array, f->u.d.n_dirty, sizeof(FNODE *), COMPARE_FNODES);
	INIT_LIST(&f->u.d.dirty);
	syncl = 0;
	for (i = f->u.d.n_dirty - 1; i >= 0; i--) {
		ff = array[i];
		ADD_TO_LIST(&f->u.d.dirty, &ff->dirent_entry);
		syncl |= ff->flags;
		if (__likely(syncl & FNODE_SYNCLIST)) WRITE_FNODE_FIRST(ff, NULL);
	}
	free_array:
	free(array);
}

typedef struct {
	TIMER t;
	FS *fs;
} BUFTIMER_STRUCT;

static void BUFFER_TIMER(TIMER *t)
{
	BUFTIMER_STRUCT *b = GET_STRUCT(t, BUFTIMER_STRUCT, t);
	FS *fs;
	LOWER_SPL(SPL_FS);
	b->t.fn = NULL;
	fs = b->fs;
	VFS$WRITE_BUFFERS(fs, 0);
	fs->fsops->writepages(fs, 0);
	if (FS_BUFFERS_DONE(fs) && FS_WPAGES_DONE(fs)) WQ_WAKE_ALL(&fs->syncer_wait);
}

#define CAN_WAIT(fs)	(LIST_EMPTY(&(fs)->synclist) && __likely(WQ_EMPTY(&(fs)->sync_done_wait)))

static void SYNCER_WAIT(FS *fs)
{
	BUFTIMER_STRUCT t;
	if (!(fs->flags & FS_DIRTY)) {
		WQ_WAIT_SYNC(&fs->syncer_wait);
		return;
	}
	t.fs = fs;
	INIT_TIMER(&t.t);
	t.t.fn = BUFFER_TIMER;
	KERNEL$SET_TIMER(fs->buffer_flush_time, &t.t);
	WQ_WAIT_SYNC(&fs->syncer_wait);
	if (t.t.fn) {
		KERNEL$DEL_TIMER(&t.t);
	} else {
		if (__unlikely(!CAN_WAIT(fs))) return;
		fs->need_sync = 0;
		DOFSSYNC(fs);
		fs->flags &= ~FS_DIRTY;
		fs->fsops->set_dirty(fs, 0);
	}
}

long SYNCER(void *p)
{
	int r;
	int i;
	FNODE *f;
	FS *fs = p;
	RAISE_SPL(SPL_FS);
	if (fs->fsops->sync_background_init) {
		fs->flags |= FS_BACKGROUND_INIT;
		WQ_WAKE_ALL_PL(&fs->syncer_wait);
		fs->fsops->sync_background_init(fs);
		fs->flags &= ~FS_BACKGROUND_INIT;
	} else {
		WQ_WAKE_ALL_PL(&fs->syncer_wait);
	}
	while (!fs->terminate_syncer) {
		i = 0;
		if (fs->need_sync || __unlikely(!WQ_EMPTY(&fs->sync_done_wait))) {
			/* if (fs->write_error) __debug_printf("need sync %d, wq %d\n", fs->need_sync, WQ_EMPTY(&fs->sync_done_wait)); */
			fs->need_sync = 0;
			DOFSSYNC(fs);
		} else {
			DOWRITEPAGES(fs);
		/* makes fsync latencies much better */
		/* disabled --- causes too much slowdown */
			/*VFS$WRITE_BUFFERS(fs, 0);*/
		}
		if (CAN_WAIT(fs)) SYNCER_WAIT(fs);
		next_entry:
#if __DEBUG >= 1
		if (__unlikely(KERNEL$SPL != SPL_X(SPL_FS)))
			KERNEL$SUICIDE("SYNCER: %s: SPL CHANGED TO %08X", fs->filesystem_name, KERNEL$SPL);
#endif
		i++;
		if (__unlikely(!(i & 31)) && fs->n_wpages_prep >= 32) DOWRITEPAGES(fs);
		if (__unlikely((f = LIST_STRUCT(fs->synclist.next, FNODE, synclist_entry)) == LIST_STRUCT(&fs->synclist, FNODE, synclist_entry))) {
			continue;
		}

		if (f->syncproc) {
			fs->syncproc = f->syncproc, f->syncproc = NULL;
			SWITCH_PROC_ACCOUNT(fs->syncproc, SPL_X(SPL_FS));
		} else {
			fs->syncproc = &KERNEL$PROC_KERNEL, KERNEL$ACQUIRE_WRITEBACK_TAG(&KERNEL$PROC_KERNEL);
		}

		/*
		   possible states are:
			FNODE_BARRIER:	   1. commit transaction and wake
		   	FNODE_UNCOMMITTED: 1. create inode & write alloc & data
					   2. add it to directory
			FNODE_NEGATIVE:	   1. delete from directory
					   2. delete inode
			otherwise:	   1. write data & alloc
					   2. write inode
		*/
		if (__unlikely(f->flags == (FNODE_SYNC_BARRIER | FNODE_SYNCLIST)) || __unlikely(fs->need_sync & FS_SYNC_NOW) || (__unlikely(fs->need_sync & FS_SYNC_SOMETIMES) && __unlikely(i >= 50))) {
			fs->need_sync &= ~(FS_SYNC_NOW | FS_SYNC_SOMETIMES);
			DOFSSYNC(fs);
			/* someone might delete `f' from list while syncing */
			f = LIST_STRUCT(fs->synclist.next, FNODE, synclist_entry);
			if (__likely(f->flags == (FNODE_SYNC_BARRIER | FNODE_SYNCLIST))) {
				DEL_FROM_LIST(&f->synclist_entry);
				f->flags = FNODE_SYNC_BARRIER;
				WQ_WAKE_ALL_PL(&f->wait);
			}
			goto release_writeback_next_entry;
		}
#if __DEBUG >= 1
		if (__unlikely(!(f->flags & FNODE_SYNCLIST)))
			KERNEL$SUICIDE("SYNCER: %s: BAD FNODE ON SYNCLIST, FLAGS %08X", VFS$FNODE_DESCRIPTION(f), f->flags);
#endif
		if (__unlikely(!(fs->flags & FS_DIRTY))) {
			fs->flags |= FS_DIRTY;
			fs->fsops->set_dirty(fs, 1);
		/* fnode might have disappeared, so check the list again */
			goto release_writeback_next_entry;
		}
		fs->flags |= FS_SOMETHING_TO_COMMIT;

		DEL_FROM_LIST(&f->synclist_entry);
		f->flags &= ~FNODE_SYNCLIST;

		f->flags |= FNODE_WRITELOCK;
		while (__unlikely(f->readers)) WQ_WAIT_SYNC(&f->wait);

		if (f->flags & (FNODE_UNCOMMITTED | FNODE_NEGATIVE | FNODE_MOVE_SRC | FNODE_MOVE_DST)) {
			FNODE *parent = f->parent;
			parent->flags |= FNODE_WRITELOCK;
			while (__unlikely(parent->readers)) WQ_WAIT_SYNC(&parent->wait);
			if (f->flags & FNODE_UNCOMMITTED) {
				if (__unlikely(parent->flags & (FNODE_UNCOMMITTED | FNODE_SRC_UNCOMMITTED))) {
					FNODE *w = f;
					do {
						WRITE_FNODE_FIRST(w, NULL);
					} while ((w = w->parent)->flags & (FNODE_UNCOMMITTED | FNODE_SRC_UNCOMMITTED));
					goto end_unlock_parent;
				}
				VFS$UNSET_DIRTY(f);
				if (__unlikely(parent->flags & FNODE_KILLED) || __unlikely(fs->flags & FS_RO))
					goto kill_it;
				r = fs->fsops->create_fnode(f);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER KILLED FNODE: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER KILLED FNODE %s: %s", f->name, strerror(-r));
					kill_it:
					f->flags |= FNODE_KILLED;
					VFS_UNDIRTY_PAGES(f);
					goto skip_ops;
				}
				r = fs->fsops->write_fnode_data(f);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT WRITE NEW FNODE DATA: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT WRITE NEW FNODE %s DATA: %s", f->name, strerror(-r));
					VFS_UNDIRTY_PAGES(f);
					goto skip_ops;
				}
				r = fs->fsops->add_fnode_to_directory(f);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER KILLED REFERENCE: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER KILLED REFERENCE %s->%s: %s", parent->name, f->name, strerror(-r));
				}
				if (f->flags & FNODE_DIRECTORY) TRY_SORT_CHILDREN(f);
				skip_ops:
				fs->fsops->unaccount(parent, __unlikely(f->flags & FNODE_DIRECTORY) ? ACCOUNT_DIR : ACCOUNT_FILE, NULL);
				f->flags &= ~(FNODE_UNCOMMITTED | FNODE_UNCOMMITTED_PREALLOC_HINT);
			} else if (!(f->flags & (FNODE_MOVE_SRC | FNODE_MOVE_DST))) {
				if (f->flags & FNODE_DIRECTORY) {
					CHECK_DIR_LISTS(f);
					if (__unlikely((f->u.d.n_clean | f->u.d.n_dirty) != 0)) {
						VFS$DO_FOR_SUBTREE(f, NULL, DFS_PREORDER, WRITE_FNODE_FIRST);
						goto end_unlock_parent;
					}
				}
				if (__unlikely(parent->flags & FNODE_KILLED) || __unlikely(fs->flags & FS_RO))
					goto no_del;
				r = fs->fsops->remove_fnode_from_directory(f);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT REMOVE REFERENCE: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT REMOVE REFERENCE %s->%s: %s", parent->name, f->name, strerror(-r));
					goto no_del;
				}
				r = fs->fsops->delete_fnode(f);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT DELETE FNODE: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT DELETE FNODE %s: %s", f->name, strerror(-r));
				}
				no_del:
				VFS_FREE_FNODE_DATA(f);
				FNODE_OUT_OF_WANTFREE(f);
				VFS$UNSET_DIRTY(f);
				f->flags = FNODE_NEGATIVE;
				VFS_FREE_EMPTY_FNODE(f), f = NULL;
			} else {
				FNODE *ff, *ft, *of, *ftp, *ofp;
				if (f->flags & FNODE_MOVE_SRC) {
					ff = f, of = ft = f->move_link;
					if (__unlikely(!(of->flags & FNODE_MOVE_DST)) || __unlikely(of->move_link != f))
						KERNEL$SUICIDE("SYNCER: %s: BROKEN LINK BETWEEN %s(%X) AND %s(%X)", fs->filesystem_name, VFS$HIDE_NAME(f->name), f->flags, VFS$HIDE_NAME(of->name), of->flags);
				} else {
					of = ff = f->move_link, ft = f;
					if (__unlikely(!(of->flags & FNODE_MOVE_SRC)) || __unlikely(of->move_link != f))
						KERNEL$SUICIDE("SYNCER: %s: BROKEN LINK BETWEEN %s(%X) AND %s(%X)", fs->filesystem_name, VFS$HIDE_NAME(f->name), f->flags, VFS$HIDE_NAME(of->name), of->flags);
				}
				if (__unlikely(ft->parent->flags & (FNODE_UNCOMMITTED | FNODE_SRC_UNCOMMITTED))) {
					FNODE *w = ft;
					do {
						WRITE_FNODE_FIRST(w, NULL);
					} while ((w = w->parent)->flags & (FNODE_UNCOMMITTED | FNODE_SRC_UNCOMMITTED));
					goto end_unlock_parent;
				}
				if (of->flags & FNODE_SYNCLIST) {
					DEL_FROM_LIST(&of->synclist_entry);
					of->flags &= ~FNODE_SYNCLIST;
					if (of->syncproc) KERNEL$RELEASE_WRITEBACK_TAG(of->syncproc), of->syncproc = NULL;
				}
				of->flags |= FNODE_WRITELOCK;
				while (__unlikely(of->readers)) WQ_WAIT_SYNC(&of->wait);
				if ((ofp = of->parent) != parent) {
					ofp->flags |= FNODE_WRITELOCK;
					while (__unlikely(ofp->readers)) WQ_WAIT_SYNC(&ofp->wait);
				}
				ftp = ft->parent;
				if (__unlikely((ofp->flags | parent->flags) & FNODE_KILLED) || __unlikely(fs->flags & FS_RO))
					goto move_error;
				if (ft->flags & (FNODE_FILE | FNODE_DIRECTORY)) {
					if (__unlikely(ft->flags & FNODE_DIRECTORY)) {
						CHECK_DIR_LISTS(ft);
						if (__unlikely((ft->u.d.n_clean | ft->u.d.n_dirty) != 0)) {
							VFS$DO_FOR_SUBTREE(ft, NULL, DFS_PREORDER, WRITE_FNODE_FIRST);
							goto repeat_move;
						}
					}
					r = fs->fsops->remove_fnode_from_directory(ft);
					if (__unlikely(r)) {
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT REMOVE REFERENCE WHEN PROCESSING MOVE: %s", strerror(-r));
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT REMOVE REFERENCE %s->%s WHEN PROCESSING MOVE: %s", ftp->name, ft->name, strerror(-r));
						move_error:
						ff->flags &= ~(FNODE_MOVE_SRC | FNODE_MOVE_DST | FNODE_SRC_UNCOMMITTED);
						ff->flags |= FNODE_KILLED;
						ff->move_link = NULL;
						if (ft) {
							ft->flags &= ~FNODE_MOVE_DST;
							ft->flags |= FNODE_KILLED;
							ft->move_link = NULL;
						}
						VFS_UNDIRTY_PAGES(ff);
						goto skip_move_ops;
					}
					r = fs->fsops->delete_fnode(ft);
					if (__unlikely(r)) {
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT DELETE FNODE WHEN MOVING: %s", strerror(-r));
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT DELETE FNODE %s WHEN PROCESSING MOVE: %s", ft->name, strerror(-r));
					}
					VFS_FREE_FNODE_DATA(ft);
					FNODE_OUT_OF_WANTFREE(ft);
					ft->flags = FNODE_NEGATIVE | FNODE_MOVE_DST | FNODE_WRITELOCK;
				}
				VFS$UNSET_DIRTY(ft);
				VFS$UNSET_DIRTY(ff);
				if (__likely(!(ff->flags & FNODE_SRC_UNCOMMITTED))) {
					r = fs->fsops->remove_fnode_from_directory(ff);
					if (__unlikely(r)) {
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD MOVE FNODE FROM DIRECTORY: %s", strerror(-r));
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT MOVE FNODE FROM DIRECTORY %s->%s: %s", ff->parent->name, ff->name, strerror(-r));
						goto move_error;
					}
				}
				DEL_FROM_LIST(&ff->hash_entry);
				ADD_TO_LIST_AFTER(&ft->hash_entry, &ff->hash_entry);
				DEL_FROM_LIST(&ff->dirent_entry);
				if (!(ff->flags & FNODE_DIRTYLIST)) {
					ff->parent->u.d.n_clean--;
					ADD_TO_LIST_END(&ftp->u.d.clean, &ff->dirent_entry);
					ftp->u.d.n_clean++;
				} else {
					ff->parent->u.d.n_dirty--;
					ADD_TO_LIST_END(&ftp->u.d.dirty, &ff->dirent_entry);
					ftp->u.d.n_dirty++;
				}
				ff->parent = ftp;
				ff->namelen = ft->namelen;
				if (__unlikely(ft->name != ft->inline_name)) {
					if (__likely(ff->name != ff->inline_name)) {
						__slow_slfree(ff->name);
						WQ_WAKE_ALL(&ff->fs->freemem);
					}
					ff->name = ft->name;
					(ft->name = ft->inline_name)[0] = 0;
					ft->namelen = 0;
				} else strcpy(ff->name, ft->name);
				VFS$DO_FOR_SUBTREE(ff, NULL, DFS_PREORDER, FIX_DEPTH);
				ft->flags &= ~FNODE_WRITELOCK;
				VFS_FREE_EMPTY_FNODE(ft);
				ft = NULL;
				of = NULL;
				f = ff;
				ff->flags &= ~FNODE_MOVE_SRC;
				ff->move_link = NULL;
				CHECK_DIR_LISTS_3(ftp);
				if (__unlikely(ff->flags & FNODE_SRC_UNCOMMITTED)) {
					r = fs->fsops->create_fnode(ff);
					ff->flags &= ~FNODE_SRC_UNCOMMITTED;
					if (__unlikely(r)) {
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT CREATE FNODE WHEN MOVING: %s", strerror(-r));
						KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT CREATE FNODE %s WHEN MOVING: %s", ff->name, strerror(-r));
						goto move_error;
					}
				}
				r = fs->fsops->write_fnode_data(ff);
				ff->flags &= ~FNODE_UNCOMMITTED_PREALLOC_HINT;
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT WRITE FNODE DATA WHEN MOVING: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT WRITE FNODE %s DATA WHEN MOVING: %s", ff->name, strerror(-r));
					goto move_error;
				}
				r = fs->fsops->add_fnode_to_directory(ff);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER KILLED REFERENCE WHEN MOVING: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER KILLED REFERENCE %s->%s WHEN MOVING: %s", parent->name, ff->name, strerror(-r));
					goto move_error;
				}
				skip_move_ops:
				fs->fsops->unaccount(ftp, __unlikely(ff->flags & FNODE_DIRECTORY) ? ACCOUNT_DIR : ACCOUNT_FILE, NULL);
				repeat_move:
				if (ofp != parent) {
					ofp->flags &= ~FNODE_WRITELOCK;
					WQ_WAKE_ALL(&ofp->wait);
					CHECK_DIR_LISTS_3(ofp);
				}
				CHECK_DIR_LISTS_3(parent);
				if (__unlikely(of != NULL)) {
					of->flags &= ~FNODE_WRITELOCK;
					WQ_WAKE_ALL(&of->wait);
				}
			}
			end_unlock_parent:
			parent->flags &= ~FNODE_WRITELOCK;
			WQ_WAKE_ALL(&parent->wait);
		} else {
			VFS$UNSET_DIRTY(f);
			if (__likely((f->flags & (FNODE_DIRECTORY | FNODE_FILE)) != 0) && __likely(!(f->flags & FNODE_KILLED))) {
				r = fs->fsops->write_fnode_data(f);
				if (__unlikely(r)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "LAZYWRITER COULD NOT WRITE FNODE DATA: %s", strerror(-r));
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fs->filesystem_name, "LAZYWRITER COULD NOT WRITE FNODE %s DATA: %s", f->name, strerror(-r));
					goto und;
				}
			} else und:VFS_UNDIRTY_PAGES(f);
		}
		if (__likely(f != NULL)) {
			f->flags &= ~FNODE_WRITELOCK;
			WQ_WAKE_ALL(&f->wait);
		}

		release_writeback_next_entry:
		KERNEL$RELEASE_WRITEBACK_TAG(fs->syncproc);
		fs->syncproc = &KERNEL$PROC_KERNEL;

		KERNEL$THREAD_MAY_BLOCK();

		goto next_entry;
	}
	return 0;
}

static void *TIMED_SYNC(FNODE *f, void *param)
{
	if (!((f->flags ^= FNODE_TIMED_SYNC) & FNODE_TIMED_SYNC)) {
		WRITE_FNODE(f);
	}
	return NULL;
}

void SYNC_HALF(FS *fs)
{
	VFS$DO_FOR_SUBTREE(fs->root, NULL, DFS_PREORDER | DFS_DIRTY, TIMED_SYNC);
}

void SYNC_TIMER(TIMER *t)
{
	FS *fs = GET_STRUCT(t, FS, sync_timer);
	LOWER_SPL(SPL_FS);
	KERNEL$SET_TIMER(fs->sync_time, &fs->sync_timer);
	SYNC_HALF(fs);
	if (!LIST_EMPTY(&fs->synclist) || fs->flags & FS_DIRTY) {
		fs->need_sync |= FS_SYNC_IDLE;
		WQ_WAKE_ALL_PL(&fs->syncer_wait);
	}
}

static void *WRITE_OVERACCOUNTED(FNODE *f, void *param)
{
	WRITE_FNODE(f);
	if (f->flags & (FNODE_SYNCLIST | FNODE_WRITELOCK)) f->fs->owait = f;
	return NULL;
}

void *VFS$OVERACCOUNT(FNODE *root)
{
	FS *fs = root->fs;
	if (__unlikely(fs->flags & FS_BACKGROUND_INIT)) goto wait_on_sync_done;
	fs->owait = NULL;
	VFS$DO_FOR_SUBTREE(root, NULL, DFS_PREORDER, WRITE_OVERACCOUNTED);
	/* if (fs->write_error) {
		__debug_printf("owait: %p/%s, flags %x\n", fs->owait, fs->owait ? fs->owait->name : "-", fs->flags);
	} */
	if (!fs->owait) {
		if (__unlikely(fs->flags & FS_COMMIT_FREES_DATA)) goto do_sync;
		return __ERR_PTR(-ENOSPC);
	}
	if (!(fs->flags & FS_COMMIT_AFTER_OVERFLOW)) return &fs->owait->wait;
	do_sync:
	fs->need_sync |= FS_SYNC_SOMETIMES;
	WQ_WAKE_ALL_PL(&fs->syncer_wait);
	wait_on_sync_done:
	return &fs->sync_done_wait;
}

void VFS$PREPARE_CONTIG_ALLOC(FNODE *nf, int (*is_ok)(FNODE *))
{
	FNODE *pf, *f;
	if (__unlikely(is_ok(nf) <= 0)) return;
	nf->flags |= FNODE_UNCOMMITTED_PREALLOC_HINT;
	pf = nf->parent;
	LIST_FOR_EACH(f, &pf->u.d.dirty, FNODE, dirent_entry) if (__likely(f != nf) && __likely(f->flags & FNODE_UNCOMMITTED)) {
		int r;
		if (__unlikely(f->flags & FNODE_DIRECTORY)) break;
		r = is_ok(f);
		if (__unlikely(!r)) continue;
		if (__unlikely(r < 0)) break;
		f->flags |= FNODE_UNCOMMITTED_PREALLOC_HINT;
	}
	LIST_FOR_EACH_BACK(f, &pf->u.d.dirty, FNODE, dirent_entry) if (__likely(f != nf) && __likely(f->flags & FNODE_UNCOMMITTED_PREALLOC_HINT)) {
		WRITE_FNODE_FIRST(f, NULL);
	}
}
