#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VFS.H>
#include <SPAD/SYSLOG.H>
#include <STDARG.H>
#include <VALUES.H>

#include "SPADFS.H"

static int SPAD_PROCESS_OPTION(FS *fs_, const char *opt, const char *optend, const char *str);
static int SPAD_MOUNT(FS *fs_);
static void SPAD_SYNC_BACKGROUND_INIT(FS *fs_);
static void SPAD_UMOUNT(FS *fs_, int nosync);
static void SPAD_SET_DIRTY(FS *fs_, int dirty);
static int SPAD_START_NEW_TX(SPADFS *fs);

static const FSOPS spadfsops = {
	DEFAULT_VFSID,
	"SPADFS",
	FS_COMMIT_AFTER_OVERFLOW,
	sizeof(SPADFS),
	sizeof(SPADFNODE),
	SPAD_FNODE_CTOR,
	sizeof(SPADPAGEINRQ),
	SPAD_PAGEINRQ_CTOR,
	SPAD_PROCESS_OPTION,
	SPAD_MOUNT,
	SPAD_SYNC_BACKGROUND_INIT,
	SPAD_UMOUNT,
	SPAD_SET_DIRTY,
	NULL,
	SPAD_LOOKUP,
	sizeof(SPADREADDIRCOOKIE),
	SPAD_INIT_READDIR_COOKIE,
	SPAD_READDIR,
	VFS$DEFAULT_WRITEPAGES,
	SPAD_COMMIT,
	SPAD_INIT_FNODE,
	SPAD_CREATE_FNODE,
	SPAD_WRITE_FNODE_DATA,
	SPAD_ADD_FNODE_TO_DIRECTORY,
	SPAD_DELETE_FNODE,
	SPAD_REMOVE_FNODE_FROM_DIRECTORY,
	VFS$NULL_FNODE_CALL,
	SPAD_BMAP,
	SPAD_SYNC_BMAP,
	SPAD_INIT_LAST_PAGE,
	SPAD_ACCOUNT,
	SPAD_UNACCOUNT,
	SPAD_STAT,
	SPAD_STATFS,
};

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	return VFS$MOUNT(argc, argv, &spadfsops, "SPADFS.SYS");
}

__COLD_ATTR__ static int SPAD_PROCESS_OPTION(FS *fs_, const char *opt, const char *optend, const char *str)
{
	SPADFS *fs = (SPADFS *)fs_;
	if (!__strcasexcmp("DONT_MAKE_CHECKSUMS", opt, optend) && !str) fs->flags |= SPADFS_DONT_MAKE_CHECKSUMS, fs->dont_make_checksums_set = 1;
	else if (!__strcasexcmp("MAKE_CHECKSUMS", opt, optend) && !str) fs->flags &= ~SPADFS_DONT_MAKE_CHECKSUMS, fs->dont_make_checksums_set = 1;
	else if (!__strcasexcmp("DONT_CHECK_CHECKSUMS", opt, optend) && !str) fs->flags |= SPADFS_DONT_CHECK_CHECKSUMS, fs->dont_check_checksums_set = 1;
	else if (!__strcasexcmp("CHECK_CHECKSUMS", opt, optend) && !str) fs->flags &= ~SPADFS_DONT_CHECK_CHECKSUMS, fs->dont_check_checksums_set = 1;
	else if (!__strcasexcmp("NO_CHECKSUMS", opt, optend) && !str) fs->flags |= SPADFS_DONT_MAKE_CHECKSUMS | SPADFS_DONT_CHECK_CHECKSUMS, fs->dont_make_checksums_set = 1, fs->dont_check_checksums_set = 1;
	else if (!__strcasexcmp("CHECKSUMS", opt, optend) && !str) fs->flags &= ~(SPADFS_DONT_MAKE_CHECKSUMS | SPADFS_DONT_CHECK_CHECKSUMS), fs->dont_make_checksums_set = 1, fs->dont_check_checksums_set = 1;
	else return 1;
	return 0;
}

__COLD_ATTR__ void SPADFS_ERROR(SPADFS *fs, unsigned flags, char *msg, ...)
{
	if (msg) {
		va_list args;
		va_start(args, msg);
		KERNEL$SYSLOGV((__unlikely(flags & SPADFS_ERROR_SECRET) ? __SYSLOG_SECRET : 0) | (flags & (TXFLAGS_IO_READ_ERROR | TXFLAGS_IO_WRITE_ERROR) ? __SYSLOG_HW_ERROR : __SYSLOG_DATA_ERROR), fs->filesystem_name, msg, args);
		va_end(args);
	}
	fs->txflags_new |= flags;
}

__COLD_ATTR__ struct txblock *SPAD_READ_TX_BLOCK(SPADFS *fs, BUFFER **bb)
{
	struct txblock *tb;
	BUFFER *b;
	if (__unlikely((unsigned)fs->txb_sec & fs->block_mask)) goto out;
	if (__unlikely(__IS_ERR((tb = VFS$READ_BUFFER_SYNC((FS *)fs, fs->txb_sec, 1, 0, &b)))))
		return tb;
	if (__unlikely(tb->magic != CPU2SPAD32(TXBLOCK_MAGIC))) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD MAGIC ON TX BLOCK %012"__d_off_format"X", fs->txb_sec);
		goto put_err;
	}
	if (__unlikely(!BUFFER_USERMAP_TEST(b, (unsigned long)fs->txb_sec & __SECTORS_PER_PAGE_CLUSTER_MINUS_1))) {
		if (__unlikely(__byte_sum(tb, 512) != CHECKSUM_BASE)) {
			SPADFS_ERROR(fs, TXFLAGS_CHECKSUM_ERROR, "BAD CHECKSUM ON TX BLOCK %012"__d_off_format"X", fs->txb_sec);
			put_err:
			VFS$PUT_BUFFER(tb);
			return __ERR_PTR(-EFSERROR);
		}
		BUFFER_USERMAP_SET(b, (unsigned long)fs->txb_sec & __SECTORS_PER_PAGE_CLUSTER_MINUS_1);
	}
	*bb = b;
	return tb;

	out:
	SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "UNALIGNED TX BLOCK %012"__d_off_format"X (ALIGNMENT MASK %u)", fs->txb_sec, fs->block_mask);
	return __ERR_PTR(-EFSERROR);
}

__COLD_ATTR__ void SPAD_TX_BLOCK_CHECKSUM(struct txblock *tb)
{
	tb->checksum ^= CHECKSUM_BASE ^ __byte_sum(tb, 512);
}

__COLD_ATTR__ static int SPAD_INCREASE_CC(SPADFS *fs, int inc)
{
	int r = 0;
	struct txblock *tb;
	BUFFER *b;
	if (__unlikely(fs->flags & FS_RO)) return 0;
	if (__unlikely(__IS_ERR((tb = SPAD_READ_TX_BLOCK(fs, &b))))) return __PTR_ERR(tb);
	if (inc) {
		__u16 tb_cc = SPAD2CPU16_LV(&tb->cc);
		tb_cc += inc;
		CPU2SPAD16_LV(&tb->cc, tb_cc);
		/* !!! FIXME: process wrap-around */
		if (inc == 1 && __unlikely(SPAD2CPU32_LV(&fs->cct[tb_cc]))) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "CRASH COUNT TABLE IS DAMAGED (NON-ZERO AT %04X: %08X)", (unsigned)tb_cc, (unsigned)SPAD2CPU32_LV(&fs->cct[tb_cc]));
			r = -EFSERROR;
		}
		fs->cc = tb_cc - 1;
		fs->txc = SPAD2CPU32_LV(&fs->cct[fs->cc]);
	}
	CPU2SPAD32_LV(&tb->txflags, SPAD2CPU32_LV(&tb->txflags) | fs->txflags_new);
	fs->txflags |= fs->txflags_new;
	SPAD_TX_BLOCK_CHECKSUM(tb);
	VFS$MARK_BUFFER_DIRTY(b, (unsigned long)fs->txb_sec & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_block);
	VFS$WRITE_BUFFER(b, fs->syncproc);
	VFS$PUT_BUFFER_AND_WAIT(tb);
	VFS$SYNC_DISK((FS *)fs);
	return r;
}

static __finline__ int SPADFS_UPDATE_ERROR_FLAGS(SPADFS *fs)
{
	int r = 0;
	if (__unlikely(fs->txflags_new & ~fs->txflags)) {
		r = SPAD_INCREASE_CC(fs, 0);
	}
	return r;
}

__COLD_ATTR__ static int SPAD_MOUNT(FS *fs_)
{
	int i;
	int r;
	__u64 lr;
	const char *m;
	SPADFS *fs = (SPADFS *)fs_;
	BIORQ biorq;
	BIODESC desc;
	struct superblock *sb;
	struct txblock *tb;
	struct fixed_fnode_block *fx;
	struct fnode *root_fnode;
	SPADFNODE *f;
	unsigned root_fnode_pos;
	fs->cct = KERNEL$ALLOC_CONTIG_AREA(CCT_SIZE, AREA_DATA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)BIO_SECTOR_SIZE);
	if (__IS_ERR(fs->cct)) {
		r = __PTR_ERR(fs->cct);
		goto err0;
	}

	if (__IS_ERR(sb = VFS$READ_BUFFER_SYNC((FS *)fs, SUPERBLOCK_SECTOR, 1, 0, (BUFFER **)(void *)&KERNEL$LIST_END))) {
		r = __PTR_ERR(sb);
		goto err1;
	}

	if (memcmp(sb->signature, SUPERBLOCK_SIGNATURE, sizeof sb->signature)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD SUPERBLOCK SIGNATURE %02X%02X%02X%02X%02X%02X%02X%02X", sb->signature[0], sb->signature[1], sb->signature[2], sb->signature[3], sb->signature[4], sb->signature[5], sb->signature[6], sb->signature[7]);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SPADFS SUPERBLOCK NOT FOUND", fs->filesystem_name);
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}
	if (SPAD2CPU64_LV(&sb->byte_sex) != __make64(0x89ABCDEF, 0x01234567)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BYTE SEX DOES NOT MATCH, ON-DISK %016"__64_format"X, EXPECTED %016"__64_format"X", fs->filesystem_name, sb->byte_sex, CPU2SPAD64(__make64(0x89ABCDEF, 0x01234567)));
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}
	if (sb->version_major != SPADFS_VERSION_MAJOR) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD MAJOR VERSION NUMBER (DISK: %d.%d.%d, DRIVER %d.%d.%d)", fs->filesystem_name, sb->version_major, sb->version_middle, sb->version_minor, SPADFS_VERSION_MAJOR, SPADFS_VERSION_MIDDLE, SPADFS_VERSION_MINOR);
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}
	/* feature flags are used instead
	if (sb->version_middle > SPADFS_VERSION_MIDDLE) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD MIDDLE VERSION NUMBER (DISK: %d.%d.%d, DRIVER %d.%d.%d)", fs->filesystem_name, sb->version_major, sb->version_middle, sb->version_minor, SPADFS_VERSION_MAJOR, SPADFS_VERSION_MIDDLE, SPADFS_VERSION_MINOR);
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}*/
	if ((!(fs->flags & FS_RO) && SPAD2CPU32_LV(&sb->flags_compat_ro) & ~(0)) || SPAD2CPU32_LV(&sb->flags_compat_none) & ~(FLAG_COMPAT_NONE_UNIX_NAMES)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INCOMPATIBLE FEATURE FLAGS IN SUPERBLOCK: %08X/%08X/%08X/%08X", fs->filesystem_name, (unsigned)SPAD2CPU32_LV(&sb->flags_compat_fsck), (unsigned)SPAD2CPU32_LV(&sb->flags_compat_rw), (unsigned)SPAD2CPU32_LV(&sb->flags_compat_ro), (unsigned)SPAD2CPU32_LV(&sb->flags_compat_none));
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}
	if (SPAD2CPU32_LV(&sb->flags_compat_none) & FLAG_COMPAT_NONE_UNIX_NAMES) {
		fs->flags |= FS_MORE_SAME_NAMES;
	}
	if (SPAD2CPU32_LV(&sb->flags_compat_fsck) & FLAG_COMPAT_FSCK_NO_CHECKSUMS) {
		if (!fs->dont_make_checksums_set) fs->flags |= SPADFS_DONT_MAKE_CHECKSUMS;
		if (!fs->dont_check_checksums_set) fs->flags |= SPADFS_DONT_CHECK_CHECKSUMS;
	}
	if (__byte_sum(sb, 512) != CHECKSUM_BASE) {
		SPADFS_ERROR(fs, TXFLAGS_CHECKSUM_ERROR, "BAD CHECKSUM ON SUPERBLOCK: %02X", __byte_sum(sb, 512) ^ CHECKSUM_BASE);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SPADFS SUPERBLOCK DAMAGED", fs->filesystem_name);
		r = -EFSERROR;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}
	if (__unlikely((m = validate_super(sb)) != NULL)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "INVALID PARAMETERS IN SUPERBLOCK: %s", m);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SPADFS SUPERBLOCK DAMAGED", fs->filesystem_name);
		r = -EFSERROR;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}

	if ((__u64)(__d_off)(SPAD2CPU64_LV(&sb->size) * 2) != (SPAD2CPU64_LV(&sb->size) * 2)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: 48-BIT FILESYSTEM NOT SUPPORTED BY VFS LAYER --- CHANGE SIZE OF __d_off IN VFS.H AND RECOMPILE VFS.DLL AND ALL FILESYSTEM DRIVERS", fs->filesystem_name);
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}

	fs->block_mask = (1 << sb->sectors_per_block_bits) - 1;
	fs->pageio_mask = (BIO_SECTOR_SIZE << sb->sectors_per_block_bits) - 1;
	fs->pageio_bits = sb->sectors_per_block_bits + BIO_SECTOR_SIZE_BITS;
	fs->size = SPAD2CPU64_LV(&sb->size);
	fs->max_filesize = (_u_off_t)__D_OFF_TOP_BIT << BIO_SECTOR_SIZE_BITS;
	fs->max_filesize -= __PAGE_CLUSTER_SIZE;

	fs->sectors_per_block_bits = sb->sectors_per_block_bits;
	fs->sectors_per_block = 1 << fs->sectors_per_block_bits;
	fs->blocksize = BIO_SECTOR_SIZE << fs->sectors_per_block_bits;

	fs->sectors_per_cluster_bits = sb->sectors_per_cluster_bits;
	fs->sectors_per_cluster_mask = (1 << fs->sectors_per_cluster_bits) - 1;
	fs->clustersize = BIO_SECTOR_SIZE << fs->sectors_per_cluster_bits;
	fs->cluster_threshold = fs->clustersize * SPAD2CPU16_LV(&sb->cluster_threshold);

	fs->sectors_per_page_bits = sb->sectors_per_page_bits;
	fs->sectors_per_page = 1 << fs->sectors_per_page_bits;
	fs->pagesize_bits = fs->sectors_per_page_bits + BIO_SECTOR_SIZE_BITS;
	fs->pagesize = 1 << fs->pagesize_bits;

	fs->sectors_per_fnodepage_bits = sb->sectors_per_fnodepage_bits;
	fs->sectors_per_fnodepage = 1 << fs->sectors_per_fnodepage_bits;

	fs->dnode_hash_bits = fs->pagesize_bits - DNODE_PAGE_ENTRY_BITS - 1;
	fs->dnode_data_size = DNODE_PAGE_ENTRY_SIZE << fs->dnode_hash_bits;
	fs->dnode_page_sectors = DNODE_ENTRY_OFFSET + (fs->dnode_data_size << 1);
	fs->dnode_page_sectors = (fs->dnode_page_sectors + (fs->blocksize - 1)) & ~(fs->blocksize - 1);
	fs->dnode_page_sectors >>= BIO_SECTOR_SIZE_BITS;

	fs->sectors_per_group_bits = sb->sectors_per_group_bits;
	fs->group_mask = ((__d_off)1 << fs->sectors_per_group_bits) - 1;
	fs->n_groups = (fs->size + fs->group_mask) >> fs->sectors_per_group_bits;
	fs->zones[0].grp_start = 0;
	fs->zones[0].grp_n = SPAD2CPU16_LV(&sb->small_file_group);
	fs->zones[1].grp_start = SPAD2CPU16_LV(&sb->small_file_group);
	fs->zones[1].grp_n = SPAD2CPU16_LV(&sb->large_file_group) - SPAD2CPU16_LV(&sb->small_file_group);
	fs->zones[2].grp_start = SPAD2CPU16_LV(&sb->large_file_group);
	fs->zones[2].grp_n = fs->n_groups - SPAD2CPU16_LV(&sb->large_file_group);

	fs->txb_sec = SPAD2CPU64_LV(&sb->txblock);
	fs->apage_index0_sec = SPAD2CPU64_LV(&sb->apage_index[0]);
	fs->apage_index1_sec = SPAD2CPU64_LV(&sb->apage_index[1]);
	fs->cct_sec = SPAD2CPU64_LV(&sb->cct);
	fs->root_sec = SPAD2CPU64_LV(&sb->root);

	fs->n_apages = N_APAGES(fs->size, fs->pagesize_bits, fs->sectors_per_block_bits + BIO_SECTOR_SIZE_BITS);

	fs->reserve_sectors = SPAD2CPU64_LV(&sb->reserve_sectors);

	fs->max_alloc = MAXINT & ~(fs->sectors_per_block - 1);

	if (1 << fs->pagesize_bits > __PAGE_CLUSTER_SIZE) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FILESYSTEM PAGE SIZE(%d) IS LARGER THAN MACHINE PAGE SIZE(%d)", fs->filesystem_name, 1 << fs->pagesize_bits, (unsigned)__PAGE_CLUSTER_SIZE);
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto err1;
	}

	VFS$PUT_BUFFER(sb);

	if (__IS_ERR(tb = SPAD_READ_TX_BLOCK(fs, (void *)&KERNEL$LIST_END))) {
		r = __PTR_ERR(tb);
		if (r != -EIO && r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: TX BLOCK DAMAGED", fs->filesystem_name);
		goto err1;
	}
	fs->cc = SPAD2CPU16_LV(&tb->cc);
	fs->a_cc = tb->a_cc;	/* fs->a_cc/a_txc is in disk-endian */
	fs->a_txc = tb->a_txc;
	fs->txflags = SPAD2CPU32_LV(&tb->txflags);
	VFS$PUT_BUFFER(tb);

	if (__unlikely(fs->txflags & TXFLAGS_DIRTY)) SPADFS_ERROR(fs, 0, "FILESYSTEM WAS NOT CLEANLY UNMOUNTED");
	if (__unlikely(fs->txflags & TXFLAGS_IO_READ_ERROR)) SPADFS_ERROR(fs, 0, "FILESYSTEM HAS ENCOUNTERED READ I/O ERROR");
	if (__unlikely(fs->txflags & TXFLAGS_IO_WRITE_ERROR)) SPADFS_ERROR(fs, 0, "FILESYSTEM HAS ENCOUNTERED WRITE I/O ERROR");
	if (__unlikely(fs->txflags & TXFLAGS_FS_ERROR)) SPADFS_ERROR(fs, 0, "FILESYSTEM HAS ERRORS IN STRUCTURES");
	if (__unlikely(fs->txflags & TXFLAGS_CHECKSUM_ERROR)) SPADFS_ERROR(fs, 0, "FILESYSTEM STRUCTURES HAVE BAD CHECKSUMS");
	if (__unlikely(fs->txflags & TXFLAGS_EA_ERROR)) SPADFS_ERROR(fs, 0, "FILESYSTEM STRUCTURES HAS ERRORS IN EXTENDED ATTRIBUTES");
	if (__unlikely(fs->txflags & ~(TXFLAGS_DIRTY | TXFLAGS_IO_READ_ERROR | TXFLAGS_IO_WRITE_ERROR | TXFLAGS_FS_ERROR | TXFLAGS_CHECKSUM_ERROR | TXFLAGS_EA_ERROR))) SPADFS_ERROR(fs, 0, "FILESYSTEM HAS UNKNOWN ERROR FLAGS %08X", fs->txflags);
	if (__unlikely(fs->txflags) && !(fs->flags & FS_RO)) SPADFS_ERROR(fs, 0, "RUNNING SPADFSCK IS RECOMMENDED");

	biorq.h = fs->disk_handle_num;
	biorq.sec = fs->cct_sec;
	biorq.nsec = CCT_SIZE / BIO_SECTOR_SIZE;
	biorq.flags = BIO_READ;
	biorq.desc = &desc;
	biorq.proc = &KERNEL$PROC_KERNEL;
	biorq.fault_sec = -1;
	desc.v.ptr = (unsigned long)fs->cct;
	desc.v.len = CCT_SIZE;
	desc.v.vspace = &KERNEL$VIRTUAL;
	desc.next = NULL;
	SYNC_IO_CANCELABLE(&biorq, KERNEL$BIO);
	if (biorq.status < 0) {
		SPADFS_ERROR(fs, TXFLAGS_IO_READ_ERROR, "CAN'T READ CRASH COUNT TABLE AT %012"__d_off_format"X: %s", fs->cct_sec, strerror(-biorq.status));
		r = biorq.status;
		goto err15;
	}

	fs->txc = SPAD2CPU32_LV(&fs->cct[fs->cc]);

	for (i = 0; i <= fs->cc; i++) {
		if (__unlikely((SPAD2CPU32_LV(&fs->cct[i]) & 0x80000000U) != 0)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "WRONG VALUE IN CRASH COUNT TABLE AT OFFSET %d: %08X", i, (unsigned)SPAD2CPU32_LV(&fs->cct[i]));
			cct_error:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CRASH COUNT TABLE DAMAGED");
			r = -EFSERROR;
			goto err15;
		}
	}
	for (i = fs->cc + 1; i < 65536; i++) {
		if (__unlikely(SPAD2CPU32_LV(&fs->cct[i]) != 0)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "NON-ZERO VALUE IN CRASH COUNT TABLE AT OFFSET %d: %08X", i, (unsigned)SPAD2CPU32_LV(&fs->cct[i]));
			goto cct_error;
		}
	}

	fs->apage_index = KERNEL$ALLOC_CONTIG_AREA(fs->n_apages * sizeof(struct apage_index_entry) + fs->blocksize, AREA_DATA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)fs->blocksize);
	if (__IS_ERR(fs->apage_index)) {
		r = __PTR_ERR(fs->apage_index);
		goto err15;
	}
	fs->apage_index = (struct apage_index_entry *)((char *)fs->apage_index + fs->blocksize);
	CPU2SPAD64_LV(&fs->apage_index[-1].end_sector, 0);

	biorq.h = fs->disk_handle_num;
	biorq.sec = CC_VALID(fs, &fs->a_cc, &fs->a_txc) ? fs->apage_index0_sec : fs->apage_index1_sec;
	biorq.nsec = APAGE_INDEX_SECTORS(fs->n_apages, fs->blocksize);
	biorq.flags = BIO_READ;
	biorq.desc = &desc;
	biorq.proc = &KERNEL$PROC_KERNEL;
	biorq.fault_sec = -1;
	desc.v.ptr = (unsigned long)fs->apage_index;
	desc.v.len = (unsigned long)biorq.nsec << BIO_SECTOR_SIZE_BITS;
	desc.v.vspace = &KERNEL$VIRTUAL;
	desc.next = NULL;
	SYNC_IO_CANCELABLE(&biorq, KERNEL$BIO);
	if (biorq.status < 0) {
		SPADFS_ERROR(fs, TXFLAGS_IO_READ_ERROR, "CAN'T READ APAGE INDEX %d AT %012"__d_off_format"X: %s", !CC_VALID(fs, &fs->a_cc, &fs->a_txc), CC_VALID(fs, &fs->a_cc, &fs->a_txc) ? fs->apage_index0_sec : fs->apage_index1_sec, strerror(-biorq.status));
		r = biorq.status;
		goto err2;
	}

	fs->group_info = KERNEL$ALLOC_CONTIG_AREA(fs->n_groups * sizeof(struct group_info) + fs->n_apages * sizeof(struct apage_info), AREA_DATA);
	if (__IS_ERR(fs->group_info)) {
		r = __PTR_ERR(fs->group_info);
		goto err2;
	}
	fs->apage_info = (struct apage_info *)(fs->group_info + fs->n_groups);

	memset(fs->group_info, 0, fs->n_groups * sizeof(struct group_info));
	memset(fs->apage_info, 0, fs->n_apages * sizeof(struct apage_info));

	for (i = 0; i < fs->n_groups; i++) {
		if (i < fs->zones[1].grp_start) fs->group_info[i].zone = &fs->zones[0];
		else if (i < fs->zones[2].grp_start) fs->group_info[i].zone = &fs->zones[1];
		else fs->group_info[i].zone = &fs->zones[2];
	}
	lr = 0;
	for (i = 0; i < fs->n_apages; i++) {
		if (__unlikely(!SPAD2CPU64_LV(&fs->apage_index[i].apage)) || __unlikely(SPAD2CPU64_LV(&fs->apage_index[i].apage) > fs->size - fs->sectors_per_page) || __unlikely((unsigned)SPAD2CPU64_LV(&fs->apage_index[i].apage) & (fs->sectors_per_page - 1))) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "INVALID POINTER IN APAGE INDEX AT OFFSET %d: %"__64_format"X", i, (__u64)SPAD2CPU64_LV(&fs->apage_index[i].apage));
			apage_index_error:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "BAD APAGE INDEX");
			r = -EFSERROR;
			goto err3;
		}
		if (!SPAD2CPU64_LV(&fs->apage_index[i].end_sector)) break;
		if (__unlikely((unsigned)SPAD2CPU64_LV(&fs->apage_index[i].end_sector) & (fs->sectors_per_block - 1)) || __unlikely(SPAD2CPU64_LV(&fs->apage_index[i].end_sector) <= lr)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "INVALID RANGE IN APAGE INDEX AT OFFSET %d: %"__64_format"X", i, (__u64)SPAD2CPU64_LV(&fs->apage_index[i].end_sector));
			goto apage_index_error;
		}
		lr = SPAD2CPU64_LV(&fs->apage_index[i].end_sector);
	}
	if (__unlikely(lr != fs->size)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "APAGE INDEX DOES NOT COVER THE WHOLE FILESYSTEM SIZE");
		goto apage_index_error;
	}
	fs->n_active_apages = i;
	for (; i < fs->n_apages; i++) {
		if (__unlikely(fs->apage_index[i].end_sector != CPU2SPAD64(0))) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "APAGE INDEX IS CORRUPTED, NON-EMPTY ENTRY AT OFFSET %d: %"__64_format"X", i, (__u64)SPAD2CPU64_LV(&fs->apage_index[i].end_sector));
			goto apage_index_error;
		}
		if (__unlikely(!SPAD2CPU64_LV(&fs->apage_index[i].apage)) || __unlikely(SPAD2CPU64_LV(&fs->apage_index[i].apage) > fs->size - fs->sectors_per_page) || __unlikely((unsigned)SPAD2CPU64_LV(&fs->apage_index[i].apage) & (fs->sectors_per_page - 1))) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "INVALID RESERVED POINTER IN APAGE INDEX AT OFFSET %d: %"__64_format"X", i, (__u64)SPAD2CPU64_LV(&fs->apage_index[i].apage));
			goto apage_index_error;
		}
	}
	fs->min_apage_invalid_checksum = fs->n_active_apages;
	fs->max_apage_invalid_checksum = 0;

	fs->scratch_page = KERNEL$ALLOC_CONTIG_AREA(__PAGE_CLUSTER_SIZE, AREA_DATA | AREA_ALIGN, (unsigned long)__PAGE_CLUSTER_SIZE);
	if (__IS_ERR(fs->scratch_page)) {
		r = __PTR_ERR(fs->scratch_page);
		goto err3;
	}

	fs->scratch_page_2 = KERNEL$ALLOC_CONTIG_AREA(__PAGE_CLUSTER_SIZE, AREA_DATA | AREA_ALIGN, (unsigned long)__PAGE_CLUSTER_SIZE);
	if (__IS_ERR(fs->scratch_page_2)) {
		r = __PTR_ERR(fs->scratch_page_2);
		goto err4;
	}

	fx = VFS$READ_BUFFER_SYNC((FS *)fs, fs->root_sec, fs->sectors_per_block, 0, (BUFFER **)(void *)&KERNEL$LIST_END);
	if (__IS_ERR(fx)) {
		r = __PTR_ERR(fx);
		goto err5;
	}
	if (__unlikely(fx->magic != CPU2SPAD32(FIXED_FNODE_BLOCK_MAGIC))) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD MAGIC ON ROOT FIXED FNODE BLOCK %012"__d_off_format"X", fs->root_sec);
		VFS$PUT_BUFFER(fx);
		r = -EFSERROR;
		goto err5;
	}
	root_fnode_pos = FIXED_FNODE_BLOCK_FNODE1 - CC_VALID(fs, &fx->cc, &fx->txc) * (FIXED_FNODE_BLOCK_FNODE1 - FIXED_FNODE_BLOCK_FNODE0);
	root_fnode = (struct fnode *)((char *)fx + root_fnode_pos);
	if ((root_fnode->flags & (FNODE_FLAGS_DIR | FNODE_FLAGS_HARDLINK)) != FNODE_FLAGS_DIR) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD FLAGS ON ROOT FIXED FNODE BLOCK %012"__d_off_format"X", fs->root_sec);
		VFS$PUT_BUFFER(fx);
		r = -EFSERROR;
		goto err5;
	}
	if (SPAD2CPU64_LV(&*FIXED_FNODE_NLINK_PTR(root_fnode)) != 1) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD NLINK ON ROOT FIXED FNODE BLOCK %012"__d_off_format"X", fs->root_sec);
		VFS$PUT_BUFFER(fx);
		r = -EFSERROR;
		goto err5;
	}
	if (root_fnode->namelen) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "NON-EMPTY NAME ON ROOT FIXED FNODE BLOCK %012"__d_off_format"X", fs->root_sec);
		VFS$PUT_BUFFER(fx);
		r = -EFSERROR;
		goto err5;
	}
	if ((unsigned)((SPAD2CPU16_LV(&root_fnode->next) & FNODE_NEXT_SIZE) - FNODE_EA_POS(root_fnode->namelen)) > FNODE_MAX_EA_SIZE) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "INVALID EXTENDED ATTRIBUTED ON ROOT FIXED FNODE BLOCK %012"__d_off_format"X", fs->root_sec);
		VFS$PUT_BUFFER(fx);
		r = -EFSERROR;
		goto err5;
	}
	f = (SPADFNODE *)VFS$GET_ROOT_FNODE((FS *)fs);
	SPAD_GET_FNODE(f, root_fnode, fs->root_sec, root_fnode_pos);
	VFS$PUT_BUFFER(fx);

	/*if ((r = SPAD_COUNT_FREE_SPACE(fs))) goto err5;*/

	if ((r = SPAD_INCREASE_CC(fs, 1)) < 0) goto err5;
	if (fs->write_error) {
		r = -EIO;
		goto err5;
	}

	if ((r = SPAD_START_NEW_TX(fs))) goto err6;
	if (fs->write_error) {
		r = -EIO;
		goto err6;
	}
	
	/*__debug_printf("FREESPACE: %012"__d_off_format"X\n", fs->free);*/

	return 0;

	err6:
	SPAD_INCREASE_CC(fs, -1);
	err5:
	KERNEL$FREE_CONTIG_AREA(fs->scratch_page_2, __PAGE_CLUSTER_SIZE);
	err4:
	KERNEL$FREE_CONTIG_AREA(fs->scratch_page, __PAGE_CLUSTER_SIZE);
	err3:
	KERNEL$FREE_CONTIG_AREA(fs->group_info, fs->n_groups * sizeof(struct group_info) + fs->n_apages * sizeof(struct apage_info));
	err2:
	KERNEL$FREE_CONTIG_AREA((char *)fs->apage_index - fs->blocksize, fs->n_apages * sizeof(struct apage_index_entry) + fs->blocksize);
	err15:
	SPADFS_UPDATE_ERROR_FLAGS(fs);
	err1:
	KERNEL$FREE_CONTIG_AREA(fs->cct, CCT_SIZE);
	err0:
	return r;
}

__COLD_ATTR__ static void SPAD_SYNC_BACKGROUND_INIT(FS *fs_)
{
	SPAD_COUNT_FREE_SPACE((SPADFS *)fs_);
}

__COLD_ATTR__ static void SPAD_UMOUNT(FS *fs_, int nosync)
{
	SPADFS *fs = (SPADFS *)fs_;
	if (!nosync) SPAD_INCREASE_CC(fs, -1);
	KERNEL$FREE_CONTIG_AREA(fs->scratch_page_2, __PAGE_CLUSTER_SIZE);
	KERNEL$FREE_CONTIG_AREA(fs->scratch_page, __PAGE_CLUSTER_SIZE);
	KERNEL$FREE_CONTIG_AREA(fs->group_info, fs->n_groups * sizeof(struct group_info) + fs->n_apages * sizeof(struct apage_info));
	KERNEL$FREE_CONTIG_AREA((char *)fs->apage_index - fs->blocksize, fs->n_apages * sizeof(struct apage_index_entry) + fs->blocksize);
	KERNEL$FREE_CONTIG_AREA(fs->cct, CCT_SIZE);
	if (!nosync && (fs->free != fs->free_allowed || fs->overaccounted_avg)) KERNEL$SUICIDE("SPAD_UMOUNT: %s: ACCOUNTING SKEW: FREE(%012"__d_off_format"X) != FREE_ALLOWED(%012"__d_off_format"X), FREE_AT_COMMIT(%012"__d_off_format"X), OVERACCOUNTED_AVG(%012"__d_off_format"X)", fs->filesystem_name, fs->free, fs->free_allowed, fs->free_at_commit, fs->overaccounted_avg);
}

static void SPAD_SET_DIRTY(FS *fs_, int dirty)
{
}

static int SPAD_START_NEW_TX(SPADFS *fs)
{
	int r;
	__u32 v;
	if (__unlikely(SPAD2CPU32_LV(&fs->cct[fs->cc]) == 0x7fffffff)) {
		if (__unlikely(r = SPAD_INCREASE_CC(fs, 1))) return r;
	} else {
		if (__unlikely(r = SPADFS_UPDATE_ERROR_FLAGS(fs))) return r;
	}
	v = SPAD2CPU32_LV(&fs->cct[fs->cc]) + 1;
	CPU2SPAD32_LV(&fs->cct[fs->cc], v);
	fs->txc = v;
	return 0;
}

void SPAD_COMMIT(FS *fs_)
{
	BIORQ biorq;
	BIODESC desc;
	unsigned sector;
	SPADFS *fs = (SPADFS *)fs_;
	/*__debug_printf("SPAD_COMMIT\n");*/
	if (__unlikely(fs->flags & FS_RO)) goto ret;
	SPAD_DISCARD_PREALLOC(fs);
	SPAD_APAGE_FIX_CHECKSUMS(fs);
	VFS$DEFAULT_COMMIT((FS *)fs);
	if (__unlikely(fs->write_error) && __unlikely(!fs->ignore_write_errors)) goto ret;
	sector = fs->cc / (BIO_SECTOR_SIZE / sizeof(__s32));
	sector &= ~(fs->sectors_per_block - 1);
	biorq.h = fs->disk_handle_num;
	biorq.sec = fs->cct_sec + sector;
	biorq.nsec = fs->sectors_per_block;
	biorq.flags = BIO_WRITE;
	biorq.desc = &desc;
	biorq.proc = &KERNEL$PROC_KERNEL;
	biorq.fault_sec = -1;
	desc.v.ptr = (unsigned long)fs->cct + (sector << BIO_SECTOR_SIZE_BITS);
	desc.v.len = fs->blocksize;
	desc.v.vspace = &KERNEL$VIRTUAL;
	desc.next = NULL;
	SYNC_IO_CANCELABLE(&biorq, KERNEL$BIO);
	if (__unlikely(biorq.status < 0)) {
		SPADFS_ERROR(fs, TXFLAGS_IO_WRITE_ERROR, "CAN'T WRITE CRASH COUNT TABLE SECTOR %012"__d_off_format"X: %s", fs->cct_sec + sector, strerror(-biorq.status));
		fs->write_error = 1;
		fs->write_error_sector = fs->cct_sec + sector;
		/* {
			__debug_printf("fs flags: %x, need_sync %x, free: %Lx, free_allowed: %Lx, free_at_commit: %Lx \n", fs->flags, fs->need_sync, fs->free, fs->free_allowed, fs->free_at_commit);
			KERNEL$STACK_DUMP();
			KERNEL$SLEEP(JIFFIES_PER_SECOND * 5);
		} */
		goto ret;
	}
	if (__unlikely(SPAD_START_NEW_TX(fs))) {
		fs->write_error = 1;
	}
	VFS$SYNC_DISK((FS *)fs);
	ret:
	fs->free += fs->free_at_commit;
	fs->free_allowed += fs->free_at_commit;
	fs->free_at_commit = 0;
	fs->flags &= ~FS_COMMIT_FREES_DATA;
}

void SPAD_STAT(FNODE *f_, struct stat *stat)
{
#define f ((SPADFNODE *)f_)
#define fs ((SPADFS *)f->fs)
	if (__likely(!S_ISDIR(stat->st_mode)) && __unlikely(stat->st_size >= fs->cluster_threshold)) {
		stat->st_blocks = (stat->st_blocks + fs->sectors_per_cluster_mask) & ~(__u64)fs->sectors_per_cluster_mask;
	}
#undef fs
#undef f
}

#define ACCT_NEW_FILE_DIR	(__likely(dir == ACCOUNT_PAGE) ? (__likely(fs->clustersize <= __PAGE_CLUSTER_SIZE) || __unlikely(p->id < fs->cluster_threshold) ? (__SECTORS_PER_PAGE_CLUSTER << 1) : __unlikely(!((unsigned)p->id & (fs->cluster_threshold - 1))) ? (fs->sectors_per_cluster_mask + 1) * 2 : 0) : (fs->sectors_per_block + (__unlikely(dir == ACCOUNT_DIR) ? fs->sectors_per_block : 0)))
#define ACCT_AVG_SECTORS	(__likely(dir == ACCOUNT_PAGE) ? __SECTORS_PER_PAGE_CLUSTER : __unlikely(dir == ACCOUNT_DIR  ? fs->sectors_per_block : 0))

void *SPAD_ACCOUNT(FNODE *f, int dir, PAGE *p)
{
#define fs ((SPADFS *)f->fs)
	unsigned acct = ACCT_NEW_FILE_DIR;
	if (__likely(fs->free_allowed > acct + fs->reserve_sectors) && __likely(!(fs->free_allowed & __D_OFF_TOP_BIT))) {
		fs->free_allowed -= acct;
		fs->overaccounted_avg += ACCT_AVG_SECTORS;
		return NULL;
	}
	return VFS$OVERACCOUNT(fs->root);
#undef fs
}

void SPAD_UNACCOUNT(FNODE *f, int dir, PAGE *p)
{
#define fs ((SPADFS *)f->fs)
	unsigned acct = ACCT_NEW_FILE_DIR;
	fs->free_allowed += acct;
	fs->overaccounted_avg -= ACCT_AVG_SECTORS;
#undef fs
}

int SPAD_CAN_OVERALLOC(SPADFNODE *f, __u32 blocks)
{
#define fs ((SPADFS *)f->fs)
	if (__unlikely(blocks > fs->free_allowed) || __unlikely((fs->free_allowed & __D_OFF_TOP_BIT) != 0)) return 0;
	return 1;
#undef fs
}

/* algorithm from hpfs_stop_cycles in linux/fs/hpfs/super.c */
/* set (*c)[1] = 0 at first time */

int SPAD_STOP_CYCLES(__d_off key, __d_off (*c)[2])
{
	if (__unlikely((*c)[0] == key) && __unlikely((*c)[1] != 0)) return -1;
	(*c)[1]++;
	if (__likely(!(((*c)[1] - 1) & (*c)[1]))) (*c)[0] = key;
	return 0;
}

WQ *SPAD_STATFS(FS *fs_, struct statfs *stat)
{
#define fs ((SPADFS *)fs_)
	__d_off fr;
	if (__unlikely(fs->flags & FS_BACKGROUND_INIT)) return &fs->sync_done_wait;
	if (__likely(stat->f_flags & MNT_ASYNC)) stat->f_flags = (stat->f_flags & ~MNT_ASYNC) | MNT_SOFTDEP;
	stat->f_blocks = fs->size >> fs->sectors_per_block_bits;
	fr = fs->free - fs->reserve_sectors - fs->overaccounted_avg;
	if (__unlikely((fr & __D_OFF_TOP_BIT) != 0)) stat->f_bfree = 0;
	else stat->f_bfree = fr >> fs->sectors_per_block_bits;
	stat->f_bavail = stat->f_bfree;
	stat->f_files = (__u64)fs->zones[0].grp_n << (fs->sectors_per_group_bits - fs->sectors_per_block_bits);
	stat->f_ffree = fs->zones[0].freespace >> fs->sectors_per_block_bits;
	stat->f_favail = stat->f_ffree;
	stat->f_namelen = 255;
	return NULL;
#undef fs
}

