#include <SPAD/VFS.H>
#include <ARCH/MOV.H>
#include <ARCH/BSF.H>
#include <STDLIB.H>
#include <TIME.H>
#include <VALUES.H>

#include "SPADFS.H"
#include "COMMON/SFSAPAGE.H"

static int need_sync;
#define NEED_SYNC_THRESHOLD	64
/* with small threshold it went into syncing loop and seriously degradated
   per4mance :-(
   without need_sync it went into long wait under other scenarios */

static __finline__ void INCREASE_FREESPACE(SPADFS *fs, unsigned group, __u32 n)
{
#if __DEBUG >= 1
	if (__unlikely(group >= fs->n_groups)) {
		KERNEL$SUICIDE("INCREASE_FREESPACE: INVALID GROUP %u", group);
	}
#endif
	fs->group_info[group].freespace += n;
	fs->group_info[group].zone->freespace += n;
}

static __finline__ void DECREASE_FREESPACE(SPADFS *fs, unsigned group, __u32 n)
{
#if __DEBUG >= 1
	if (__unlikely(group >= fs->n_groups)) {
		KERNEL$SUICIDE("DECREASE_FREESPACE: INVALID GROUP %u", group);
	}
#endif
	fs->group_info[group].freespace -= n;
	fs->group_info[group].zone->freespace -= n;
}

static int SPLIT_GROUP_ACTION(SPADFS *fs, __d_off start, unsigned len, void (*action)(SPADFS *fs, unsigned group, __u32 n))
{
	unsigned start_group, end_group;
	start_group = start >> fs->sectors_per_group_bits;
	end_group = (start + (len - 1)) >> fs->sectors_per_group_bits;
	if (__unlikely(end_group >= fs->n_groups)) {
		range_error:
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPLIT_GROUP_ACTION: INVALID RANGE (%012"__d_off_format"X,%X)", start, len);
		return -EFSERROR;
	}
	if (__unlikely(start_group > end_group)) {
		goto range_error;
	}
	if (__unlikely(start_group == end_group)) {
		action(fs, start_group, len);
	} else {
		action(fs, start_group, fs->group_mask + 1 - (start & fs->group_mask));
		for (start_group++; start_group < end_group; start_group++) {
			action(fs, start_group, fs->group_mask + 1);
		}
		action(fs, start_group, ((start + len - 1) & fs->group_mask) + 1);
	}
	return 0;
}

#define GROUP_ACTION(fs_, start_, sectors_, action_)			\
do {									\
	if (__likely(((start_) & (fs_)->group_mask) + (sectors_) <= fs->group_mask + 1)) (action_)((fs_), (start_) >> fs->sectors_per_group_bits, (sectors_));\
	else SPLIT_GROUP_ACTION((fs_), (start_), (sectors_), (action_));\
} while (0)

static int CHECK_OTHER_MAP_BMP(struct apage_head *a, __d_off off, unsigned n_sec, __d_off *next_free)
{
	int depth = APAGE_SIZE(a->s.u.l.flags);
	unsigned Xboff = BITMAP_OFFSET(a, off);
	unsigned Xn_sec = BITMAP_LEN(a, n_sec);
	if (__unlikely(Xboff + Xn_sec <= Xboff) || __unlikely(Xboff + Xn_sec > BITMAP_SIZE(depth))) return -EFSERROR;
	do {
		if (__unlikely(BITMAP_TEST(a, Xboff))) {
			unsigned Xp = 0;
			while (++Xboff < BITMAP_SIZE(depth)) {
				if (!BITMAP_TEST(a, Xboff)) {
					Xp++;
					if (Xp >= Xn_sec) {
						*next_free = MAKE_D_OFF(a->s.u.b.start0, a->s.u.b.start1) + ((Xboff - Xp + 1) << APAGE_SECTORS_PER_BLOCK_BITS(a->s.u.b.flags));
						return -EBUSY;
					}
				} else {
					Xp = 0;
				}
			}
			return -EBUSY;
		}
		Xboff++;
	} while (--Xn_sec);
	return 0;
}

static int CHECK_OTHER_MAP(struct apage_head *a, __d_off off, unsigned n_sec, __d_off *next_free)
{
	int depth;
	if (__unlikely(!a)) return 0;
	*next_free = 0;
	depth = APAGE_SIZE(a->s.u.l.flags);
	if (__unlikely(((int)(unsigned long)a & (__PAGE_CLUSTER_SIZE_MINUS_1 >> 1)) + depth > __PAGE_CLUSTER_SIZE / 2)) {
		/* this should be filesystem error ... */
		return -EFSERROR;
	}
	if (__likely(!(a->s.u.l.flags & APAGE_BITMAP))) {
#define aa(i) (*(struct aentry *)((char *)a + (i)))
		int preblk, postblk;
		int nrec;
		preblk = SPAD_APAGE_FIND_BLOCK_BEFORE(a, off, depth);
		if (__unlikely(preblk < 0)) {
			/* this should be filesystem error ... */
			return -EFSERROR;
		}
		if (__likely(preblk != 0)) {
			if (__likely(off + n_sec <= (__d_off)SPAD2CPU64_LV(&aa(preblk).start) + SPAD2CPU32_LV(&aa(preblk).len))) {
				return 0;
			}
		}
		postblk = APTR_ALIGN(SPAD2CPU16_LV(&aa(preblk).next));
			/* postblk < depth */
		nrec = depth;
		while (__likely(postblk != 0)) {
			if (SPAD2CPU32_LV(&aa(postblk).len) >= n_sec) {
				*next_free = (__d_off)SPAD2CPU64_LV(&aa(postblk).start);
				return -EBUSY;
			}
			postblk = APTR_ALIGN(SPAD2CPU16_LV(&aa(postblk).next));
			if (__unlikely(postblk >= depth)) return -EFSERROR;
			if (__unlikely(!(nrec -= sizeof(struct aentry)))) return -EFSERROR;
		}
		return -EBUSY;
#undef aa
	} else {
		return CHECK_OTHER_MAP_BMP(a, off, n_sec, next_free);
	}
}

static int ALLOC_BLOCKS_IN_APAGE_BMP(struct alloc *al, struct apage_head *a, struct apage_head *other_a)
{
	unsigned shift = APAGE_SECTORS_PER_BLOCK_BITS(a->s.u.l.flags);
	unsigned Xbmax = BITMAP_SIZE(APAGE_SIZE(a->s.u.l.flags));
	unsigned Xboff;
	__d_off bst = MAKE_D_OFF(a->s.u.b.start0, a->s.u.b.start1);
	__d_off bottom, top;
	bottom = al->sector;
	if (bottom < bst) bottom = bst;
	top = al->top;
	if (__likely(top > bst + (Xbmax << shift))) top = bst + (Xbmax << shift);
	if (__unlikely(bottom >= top)) return -ENOSPC;
	Xboff = (bottom - bst) >> shift;
	if (top - bst < (Xbmax << shift)) Xbmax = (top - bst) >> shift;
	while (Xboff < Xbmax) {
		if (__likely(BITMAP_TEST_32_FULL(a, Xboff))) {
			Xboff = (Xboff + 32) & ~31;
			continue;
		}
		test_bit:
		if (__unlikely(!BITMAP_TEST(a, Xboff))) {
			__d_off new_bot;
			unsigned Xc;
			if (__unlikely(al->flags & ALLOC_METADATA)) {
				if ((((unsigned)bst + (Xboff << shift)) & ALLOC_MASK(al->flags)) + al->n_sectors > ALLOC_MASK(al->flags) + 1) goto align;
			} else if (((unsigned)bst + (Xboff << shift)) & ALLOC_MASK(al->flags)) {
				unsigned Xplus;
				align:
				Xplus = (ALLOC_MASK(al->flags) + 1 - (((unsigned)bst + (Xboff << shift)) & ALLOC_MASK(al->flags))) >> shift;
				if (__unlikely(!Xplus)) Xplus = 1;
				Xboff += Xplus;
				continue;
			}
			for (Xc = 1; Xc < al->n_sectors >> shift; Xc++)
				if (__unlikely(Xboff + Xc >= Xbmax) || BITMAP_TEST(a, Xboff + Xc)) {
					Xboff += Xc + 1;
					goto cont;
				}
			if (__unlikely(CHECK_OTHER_MAP(other_a, bst + (Xboff << shift), al->n_sectors, &new_bot))) {
				need_sync++;
				if (__likely(new_bot > bst + (Xboff << shift)) && __likely(new_bot < bst + (Xbmax << shift))) {
					Xboff = (new_bot - bst + ((1 << shift) - 1)) >> shift;
					continue;
				}
				return -ENOSPC;
			}
			al->sector = bst + (Xboff << shift);
			for (Xc = 0; Xc < al->n_sectors >> shift; Xc++) BITMAP_SET(a, Xboff + Xc);
			return 0;
		}
		Xboff++;
		if (__likely(Xboff < Xbmax) && __likely(Xboff & 31)) goto test_bit;
		cont:;
	}
	return -ENOSPC;
}

static __finline__ int ALLOC_BLOCK(struct aentry *a, struct alloc *al, struct apage_head *other)
{
	__d_off new_bot;
	__d_off bottom, top;
	bottom = al->sector;
	if (__likely(bottom < SPAD2CPU64_LV(&a->start))) bottom = SPAD2CPU64_LV(&a->start);
	top = al->top;
	if (__likely(top > SPAD2CPU64_LV(&a->start) + SPAD2CPU32_LV(&a->len))) top = SPAD2CPU64_LV(&a->start) + SPAD2CPU32_LV(&a->len);
	new_bottom:
	if (__unlikely(al->flags & ALLOC_METADATA))
		if (__likely(((unsigned)bottom & ALLOC_MASK(al->flags)) + al->n_sectors <= ALLOC_MASK(al->flags) + 1)) goto skip_pad;
	bottom = (bottom + ALLOC_MASK(al->flags)) & ~(__d_off)ALLOC_MASK(al->flags);
	skip_pad:
	if (__unlikely(bottom + al->n_sectors > top)) return -ENOSPC;

	if (__unlikely(CHECK_OTHER_MAP(other, bottom, al->n_sectors, &new_bot))) {
		need_sync++;
		if (__likely(new_bot > bottom)) {
			bottom = new_bot;
			goto new_bottom;
		}
		return -ENOSPC;
	}
	/* OK, finally we can safely allocate blocks */
	if (__unlikely(bottom > SPAD2CPU64_LV(&a->start))) {
		al->flags |= ALLOC_FREE_FROM;
		al->top = SPAD2CPU64_LV(&a->start);
	}
	{
#if __DEBUG >= 1
		__u32 ol;
		ol = SPAD2CPU32_LV(&a->len);
#endif
		CPU2SPAD32_LV(&a->len, SPAD2CPU32_LV(&a->len) - ((__u32)bottom - (__u32)SPAD2CPU64_LV(&a->start) + al->n_sectors));
#if __DEBUG >= 1
		if (__unlikely(SPAD2CPU32_LV(&a->len) >= ol))
			KERNEL$SUICIDE("ALLOC_BLOCK: BROKEN ALLOCATOR!!! (BOTTOM == %012"__d_off_format"X, TOP == %012"__d_off_format"X, ST == %012"__d_off_format"X, A->LEN == %08X, OL == %08X, AL->N_SECTORS == %08X", bottom, top, SPAD2CPU64_LV(&a->start), (unsigned)SPAD2CPU32_LV(&a->len), (unsigned)ol, (unsigned)al->n_sectors);
#endif
	}
	CPU2SPAD64_LV(&a->start, bottom + al->n_sectors);
	al->sector = bottom;
	return 0;
}

static int ALLOC_BLOCKS_IN_APAGE(struct alloc *al, struct apage_head *a, struct apage_head *other_a)
{
	int depth = APAGE_SIZE(a->s.u.l.flags);
	if (!(a->s.u.l.flags & APAGE_BITMAP)) {
#define aa(i) (*(struct aentry *)((char *)a + (i)))
		int nrec = depth;
		int blk = SPAD_APAGE_FIND_BLOCK_BEFORE(a, al->sector, depth);
		if (__unlikely(blk < 0)) return -EFSERROR;
		if (!blk || al->sector >= SPAD2CPU64_LV(&aa(blk).start) + SPAD2CPU32_LV(&aa(blk).len)) {
			blk = APTR_ALIGN(SPAD2CPU16_LV(&aa(blk).next)); /* valid from FIND_BLOCK_BEFORE */
			if (__unlikely(!blk)) return -ENOSPC;
		}
		next_try:
		if (SPAD2CPU32_LV(&aa(blk).len) >= al->n_sectors && 
		    __likely(!ALLOC_BLOCK(&aa(blk), al, other_a))) {
			if (__unlikely(aa(blk).len == CPU2SPAD32(0))) SPAD_DELETE_BLOCK(a, blk, depth);
			return 0;
		}
		blk = APTR_ALIGN(SPAD2CPU16_LV(&aa(blk).next));
		if (__unlikely(!blk)) return -ENOSPC;
		if (__unlikely(blk >= depth)) return -EFSERROR;
		if (__unlikely(SPAD2CPU64_LV(&aa(blk).start) >= al->top)) return -ENOSPC;
		if (__likely(nrec -= sizeof(struct aentry))) goto next_try;
		return -EFSERROR;
#undef aa
	} else {
		return ALLOC_BLOCKS_IN_APAGE_BMP(al, a, other_a);
	}
}

#define invalid_flags(flags, fs)	\
	__unlikely(((flags) & (APAGE_SIZE_BITS | APAGE_BLOCKSIZE_BITS)) != ((((fs)->sectors_per_page_bits - 1) << APAGE_SIZE_BITS_SHIFT) | ((fs)->sectors_per_block_bits << APAGE_BLOCKSIZE_BITS_SHIFT)))

#define MAP_READ	0
#define MAP_WRITE	1
#define MAP_ALLOC	2
#define MAP_NEW		3

static struct apage_head *MAP_APAGE(SPADFS *fs, unsigned ap, int flags)
{
	BUFFER *b;
	struct apage_head *a, *aa;
	__d_off sec = SPAD2CPU64_LV(&fs->apage_index[ap].apage);
	if (__unlikely((unsigned)sec & (fs->sectors_per_page - 1))) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "MISALIGNED APAGE %u AT BLOCK %012"__d_off_format"X", ap, sec);
		return __ERR_PTR(-EFSERROR);
	}
	a = VFS$READ_BUFFER_SYNC((FS *)fs, sec, fs->sectors_per_page, fs->buffer_readahead, &b);
	if (__unlikely(__IS_ERR(a))) {
		SPADFS_ERROR(fs, TXFLAGS_IO_READ_ERROR, "UNABLE TO READ APAGE %u AT BLOCK %012"__d_off_format"X: %s", ap, (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].apage), strerror(-__PTR_ERR(a)));
		return a;
	}
	if (__unlikely(flags == MAP_NEW)) {
		if (__unlikely(a->magic != CPU2SPAD16(APAGE_MAGIC))) goto bad_magic;
		VFS$MARK_BUFFER_DIRTY(b, ((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_page);
		return a;
	}

	if (__unlikely(!BUFFER_USERMAP_TEST(b, ((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1))) {
		if (__unlikely(a->magic != CPU2SPAD16(APAGE_MAGIC))) {
			bad_magic:
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD MAGIC ON APAGE %012"__d_off_format"X", b->blk | (((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
			VFS$PUT_BUFFER(a);
			return __ERR_PTR(-EFSERROR);
		}
		aa = a;
		if (!CC_VALID(fs, &a->cc, &a->txc)) aa = (struct apage_head *)((char *)a + (fs->pagesize >> 1));
		if (/*!fs->apage_info[ap].invalid_checksum &&*/ aa->s.u.l.flags & APAGE_CHECKSUM_VALID && __likely(!(fs->flags & SPADFS_DONT_CHECK_CHECKSUMS))) {
			if (__unlikely(__byte_sum(&aa->s, (fs->pagesize >> 1) - (sizeof(struct apage_head) - sizeof(struct apage_subhead))) != CHECKSUM_BASE)) {
				SPADFS_ERROR(fs, TXFLAGS_CHECKSUM_ERROR, "BAD CHECKSUM ON APAGE %012"__d_off_format"X", b->blk | (((unsigned long)aa >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
				VFS$PUT_BUFFER(a);
				return __ERR_PTR(-EFSERROR);
			}
		}
		BUFFER_USERMAP_SET(b, ((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1);
	} else {
		aa = a;
		if (!CC_VALID(fs, &a->cc, &a->txc)) aa = (struct apage_head *)((char *)a + (fs->pagesize >> 1));
	}
	if (invalid_flags(aa->s.u.l.flags, fs)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "APAGE %012"__d_off_format"X HAS INVALID FLAGS %02X", b->blk | (((unsigned long)aa >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1), aa->s.u.l.flags);
		VFS$PUT_BUFFER(a);
		return __ERR_PTR(-EFSERROR);
	}
	if (flags == MAP_ALLOC && __likely(!(aa->s.u.l.flags & APAGE_BITMAP)) && __unlikely(aa->s.u.l.last == CPU2SPAD16(0))) {
		VFS$PUT_BUFFER(a);
		/*fs->apage_info[ap].full = 1;*/
		return __ERR_PTR(-ENOSPC);
	}
	if (__unlikely(flags == MAP_READ)) return aa;
	if (__unlikely(!CC_CURRENT(fs, &a->cc, &a->txc))) {
		struct apage_head *aaa = aa;
		aa = (struct apage_head *)((unsigned long)aa ^ (fs->pagesize >> 1));
		memcpy(&aa->s, &aaa->s, (fs->pagesize >> 1) - (sizeof(struct apage_head) - sizeof(struct apage_subhead)));
		memset((char *)a + (fs->pagesize >> 1), 0, sizeof(struct apage_head) - sizeof(struct apage_subhead));
		CC_SET_CURRENT(fs, &a->cc, &a->txc);
		/*VFS$MARK_BUFFER_DIRTY(b, ((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_block); also possible -- less load on disk, more load on buffer cache */
		VFS$MARK_BUFFER_DIRTY(b, ((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, __likely(fs->sectors_per_page != fs->sectors_per_block) ? fs->sectors_per_page >> 1 : fs->sectors_per_page);
	}
	if (invalid_flags(((struct apage_head *)((unsigned long)aa ^ (fs->pagesize >> 1)))->s.u.l.flags, fs)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "APAGE %012"__d_off_format"X HAS INVALID FLAGS %02X IN INACTIVE PART", b->blk | ((((unsigned long)aa ^ (fs->pagesize >> 1)) >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1), ((struct apage_head *)((unsigned long)aa ^ (fs->pagesize >> 1)))->s.u.l.flags);
		VFS$PUT_BUFFER(a);
		return __ERR_PTR(-EFSERROR);
	}
	if (__likely(fs->sectors_per_block != fs->sectors_per_page))
		VFS$MARK_BUFFER_DIRTY(b, ((unsigned long)aa >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_page >> 1);
	else
		VFS$MARK_BUFFER_DIRTY(b, ((unsigned long)a >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_page);

	aa->s.u.l.flags &= ~APAGE_CHECKSUM_VALID;
	if (__likely(!(fs->flags & SPADFS_DONT_MAKE_CHECKSUMS))) {
		fs->apage_info[ap].invalid_checksum = 1;
		if (ap < fs->min_apage_invalid_checksum) fs->min_apage_invalid_checksum = ap;
		if (ap >= fs->max_apage_invalid_checksum) fs->max_apage_invalid_checksum = ap + 1;
	}
	return aa;
}

static __finline__ int TRY_APAGE(SPADFS *fs, unsigned ap, struct alloc *al)
{
	struct apage_head *aa, *aaa;
	int r;
	/*__debug_printf("TRY APAGE(%d)", ap);*/
	/* a -- first head
	   aa -- head, where to allocate
	   aaa -- head that should be checked for free */
	if (__unlikely(__IS_ERR((aa = MAP_APAGE(fs, ap, MAP_ALLOC)))))
		return __PTR_ERR(aa);
	aaa = (struct apage_head *)((unsigned long)aa ^ (fs->pagesize >> 1));
	need_sync = 0;
	r = ALLOC_BLOCKS_IN_APAGE(al, aa, aaa);
	if (__unlikely(need_sync >= NEED_SYNC_THRESHOLD)) fs->need_sync |= FS_SYNC_NOW;
	VFS$PUT_BUFFER(aa);
	return r;
}

static int ADDR_2_APAGE(SPADFS *fs, __d_off o)
{
	int a1, a2, a;
	a2 = fs->n_active_apages - 1;
	a1 = 0;
	again:
	a = (a1 + a2) >> 1;
	if (o < (__d_off)SPAD2CPU64_LV(&fs->apage_index[a - 1].end_sector)) a2 = a - 1;
	else if (o >= (__d_off)SPAD2CPU64_LV(&fs->apage_index[a].end_sector)) a1 = a + 1;
	else return a;
	if (__likely(a1 <= a2)) goto again;
	SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "CAN'T FIND APAGE FOR BLOCK %012"__d_off_format"X, STUCK ON %d/%d", o, a1, a2);
	return 0;
}

static int ALLOC_BLOCKS_IN_RANGE(SPADFS *fs, struct alloc *al)
{
	unsigned ap, tap;
	/*__debug_printf("ALLOC IN ZONE: %012LX-%012LX-%012LX\n", bottom, al->sector, top);*/
	if (__unlikely(al->top > fs->size)) al->top = fs->size;
	if (__unlikely(al->sector >= al->top)) return -ENOSPC;
	ap = ADDR_2_APAGE(fs, al->sector);
	if (__likely(al->top <= SPAD2CPU64_LV(&fs->apage_index[ap].end_sector))) tap = ap;
	else tap = ADDR_2_APAGE(fs, al->top - 1);
	next_ap:
	if (__likely(!TRY_APAGE(fs, ap, al))) return 0;
	ap++;
	if (__unlikely(ap <= tap)) goto next_ap;
	return -ENOSPC;
}

static unsigned GET_BLOCKLEN_AT(SPADFS *fs, __d_off block, unsigned max)
{
	unsigned ret;
	int depth;
	struct apage_head *a;
	if (__unlikely(block >= fs->size)) return 0;
	a = MAP_APAGE(fs, ADDR_2_APAGE(fs, block), MAP_READ);
	if (__IS_ERR(a)) return 0;
	depth = APAGE_SIZE(a->s.u.l.flags);
	if (__likely(!(a->s.u.l.flags & APAGE_BITMAP))) {
#define aa(i) (*(struct aentry *)((char *)a + (i)))
		int blk = SPAD_APAGE_FIND_BLOCK_BEFORE(a, block, depth);
		if (__unlikely(blk < 0)) {
			put_0:
			VFS$PUT_BUFFER(a);
			return 0;
		}
		if (block >= SPAD2CPU64_LV(&aa(blk).start) + SPAD2CPU32_LV(&aa(blk).len)) goto put_0;
		ret = SPAD2CPU64_LV(&aa(blk).start) + SPAD2CPU32_LV(&aa(blk).len) - block;
		if (__likely(ret > max)) ret = max;
		ret_ret:
		VFS$PUT_BUFFER(a);
		return ret;
#undef aa
	} else {
		unsigned Xboff = BITMAP_OFFSET(a, block);
		unsigned add = fs->sectors_per_block;
		ret = 0;
		while (ret < max && __likely(Xboff < BITMAP_SIZE(depth)) && !BITMAP_TEST(a, Xboff)) ret += add, Xboff++;
		goto ret_ret;
	}
}

/*
	al->sector -- result is returned there
	al->n_sectors -- result is returned there
	return: 0 -- ok
		-x -- error
*/

static int ALLOC_BLOCKS_IN_DIFFERENT_GROUP(SPADFNODE *fnode, struct alloc *al);
static int SKIP_GROUP(SPADFS *fs, __d_off start);
static void GOAL_OUT_OF_FS(SPADFS *fs, struct alloc *al);

int SPAD_ALLOC_BLOCKS(SPADFNODE *fnode, struct alloc *al)
{
	/* !!! FIXME: when quota will be done --- if ALLOC_NO_CHARGE, do not chage quota */
	SPADFS *fs = (SPADFS *)fnode->fs;
	/*__debug_printf("SPAD ALLOC BLOCKS(%012LX,%d)", al->sector, al->n_sectors);*/

	if (__unlikely(fs->sectors_per_block != 1)) {
		if (__unlikely((al->n_sectors & (fs->sectors_per_block - 1)) != 0)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_ALLOC_BLOCKS: %s: %012"__d_off_format"X,%08X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), al->sector, (int)al->n_sectors);
			return -EFSERROR;
		}
		al->sector &= ~(__d_off)(fs->sectors_per_block - 1);
		/* al->n_sectors = (al->n_sectors + fs->sectors_per_block - 1) & ~(fs->sectors_per_block - 1); */
	}

	if (__unlikely(al->sector >= fs->size))
		GOAL_OUT_OF_FS(fs, al);

	retry_allocation:

	if (__unlikely(al->flags & ALLOC_METADATA)) {
		al->flags |= (fs->sectors_per_page - 1) * ALLOC_MASK_1;
	} else if (__unlikely(al->n_sectors > fs->max_alloc)) {
		al->n_sectors = fs->max_alloc;
	}
	if (al->flags & ALLOC_BIG_FILE) {
		al->flags |= fs->sectors_per_cluster_mask * ALLOC_MASK_1;
	}

	if (__unlikely(al->flags & ALLOC_PARTIAL_AT_GOAL)) {
		__u32 bl = GET_BLOCKLEN_AT(fs, al->sector, al->n_sectors);
		if (__likely(bl != 0)) {
			unsigned orig_n_sectors = al->n_sectors;
			if (__unlikely(bl < al->n_sectors)) al->n_sectors = bl;
			while (1) {
				al->n_sectors &= ~(fs->sectors_per_block - 1);
				if (__unlikely(!al->n_sectors)) break;
				al->top = al->sector + al->n_sectors;
				if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) goto ok_allocated;
				al->n_sectors >>= 1;
			}
			al->n_sectors = orig_n_sectors;
		}
	}

	al->top = (al->sector + fs->group_mask + 1) & ~fs->group_mask;

	if (__unlikely(al->flags & ALLOC_SKIP_ALMOST_FULL) && SKIP_GROUP(fs, al->sector)) goto skip_current_group;
	if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) {
		ok_allocated:
		if (__unlikely(al->flags & ALLOC_FREE_FROM)) {
			int r = SPAD_FREE_BLOCKS_(fnode, al->top, al->sector - al->top, 0);
			if (__unlikely(r)) return r;
		}
		if (__unlikely(fs->free - al->n_sectors > fs->free)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_ALLOC_BLOCKS: %s: FREE BLOCK COUNT UNDERRUN: %012"__d_off_format"X < %X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), fs->free, al->n_sectors);
			return -EFSERROR;
		}
		fs->free -= al->n_sectors;
		fs->free_allowed -= al->n_sectors;
		GROUP_ACTION(fs, al->sector, al->n_sectors, DECREASE_FREESPACE);
		if (__unlikely(((unsigned)al->sector | al->n_sectors) & (fs->sectors_per_block - 1))) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "%s: UNALIGNED ALLOCATION: %012"__d_off_format"X, %08X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), al->sector, al->n_sectors);
			return -EFSERROR;
		}
		return 0;
	}
	skip_current_group:
	if (__likely(!ALLOC_BLOCKS_IN_DIFFERENT_GROUP(fnode, al))) goto ok_allocated;
	if (SPAD_DISCARD_PREALLOC((SPADFS *)fnode->fs)) goto retry_allocation;
	return -ENOSPC;
}

static int ALLOC_BLOCKS_IN_DIFFERENT_GROUP(SPADFNODE *fnode, struct alloc *al)
{
	SPADFS *fs = (SPADFS *)fnode->fs;
	__d_off orig_start = al->sector;
	struct zone *z;
	if (!(al->flags & (ALLOC_PARTIAL_AT_GOAL | ALLOC_METADATA))) {
		SPADFNODE *p = (SPADFNODE *)fnode->parent;
		unsigned group = orig_start >> fs->sectors_per_group_bits;
		if (__unlikely(!p)) goto skip_zones;
		p = (SPADFNODE *)p->parent;
		while (p && p->ctime + SPAD_PROPAGATE_HINT_TIME > time(NULL)) {
			unsigned new_group = __likely(al->flags & ALLOC_BIG_FILE) ? p->blk2_n : p->blk1_n;
			if (__unlikely(new_group != group)) {
				al->sector = (__d_off)new_group << fs->sectors_per_group_bits;
				al->top = (__d_off)(new_group + 1) << fs->sectors_per_group_bits;
				if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) {
					goto alloc_success_set_group;
				}
			}
			group = new_group;
			p = (SPADFNODE *)p->parent;
		}
		skip_zones:;
	}
	retry_less:
	if (__likely(al->flags & ALLOC_BIG_FILE)) {
		z = &fs->zones[2];
	} else if (__likely(al->flags & ALLOC_SMALL_FILE)) {
		z = &fs->zones[1];
	} else {
		z = &fs->zones[0];
	}
	if (__unlikely(__BSR(al->n_sectors) + SPAD_MAX_GROUP_PART_ALLOC_BITS >= fs->sectors_per_group_bits) && __likely(!(al->flags & ALLOC_METADATA))) {
		unsigned newsize;
		/*
		 * Allocate big allocations whereever possible, don't care
		 * about groups and don't set group hint on parent directory
		 */
		al->sector = orig_start;
		al->top = (__d_off)(z->grp_start + z->grp_n) << fs->sectors_per_group_bits;
		if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) return 0;
		al->sector = (__d_off)z->grp_start << fs->sectors_per_group_bits;
		al->top = orig_start;
		if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) return 0;
		/*
		 * Now, don't care about zones too, only avoid metadata zone
		 */
		al->sector = (__d_off)fs->zones[1].grp_start << fs->sectors_per_group_bits;
		al->top = fs->size;
		if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) return 0;
		/*
		 * Failed, so trim size and go through normal group search
		 * process.
		 */
		/* This may overflow but it doesn't matter */
		newsize = (__d_off)1 << fs->sectors_per_group_bits >> SPAD_MAX_GROUP_PART_ALLOC_BITS;
		if (__likely(newsize) && __likely(newsize < al->n_sectors)) al->n_sectors = newsize;
	}
	forward_quad_search:
	if (__likely(z->grp_n)) {
		unsigned pass;
		__d_off avg_free = z->freespace / z->grp_n;
		if (__unlikely(!avg_free)) avg_free = al->n_sectors;
		retry:
		/*
		 * 1: Use quadratic hash to find group with above-average free
		 *    space
		 * 2: Use linear search to find the same
		 * 3: Use quadratic hash to find any group that can satisfy the
		 *    request
		 * 4: Use linear search to find the same
		 */
		for (pass = 0; pass < 2; pass++) {
			unsigned i;
			unsigned grp_current = orig_start >> fs->sectors_per_group_bits;
			if (__unlikely(grp_current < z->grp_start) || __unlikely(grp_current >= z->grp_start + z->grp_n)) grp_current = z->grp_start /* + (unsigned)random() % z->grp_n */;
			i = 1;
			do {
				grp_current += __likely(!pass) ? i : 1;
				if (__unlikely(grp_current >= z->grp_start + z->grp_n)) grp_current -= z->grp_n;
				if (fs->group_info[grp_current].freespace >= avg_free) {
					al->sector = (__d_off)grp_current << fs->sectors_per_group_bits;
					al->top = (__d_off)(grp_current + 1) << fs->sectors_per_group_bits;
					if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) {
						alloc_success_set_group:
						al->flags |= ALLOC_NEW_GROUP_HINT;
						return 0;
					}
				}
				if (__likely(!pass)) i *= 2;
				else i++;
			} while (i <= z->grp_n);
		}
		if (avg_free > al->n_sectors) {
			avg_free = al->n_sectors;
			goto retry;
		}
	}
	/*
	 * Now, look in other zones. Zone order for requests is:
	 *	big files: big file zone, small file zone, metadata zone
	 *	small files: small file zone, big file zone, metadata zone
	 *	metadata zone: metadata zone, small file zone, big file zone
	 */
	try_another_zone:
	if (__likely(al->flags & ALLOC_BIG_FILE)) {
		if (__likely(z == &fs->zones[2])) z = &fs->zones[1];
		else if (__likely(z == &fs->zones[1])) z = &fs->zones[0];
		else goto failed;
	} else if (__likely(al->flags & ALLOC_SMALL_FILE)) {
		if (__likely(z == &fs->zones[1])) z = &fs->zones[2];
		else if (__likely(z == &fs->zones[2])) z = &fs->zones[0];
		else goto failed;
	} else {
		if (__likely(z == &fs->zones[0])) z = &fs->zones[1];
		else if (__likely(z == &fs->zones[1])) z = &fs->zones[2];
		else return -ENOSPC;
	}
	/*
	 * If we are going to allocate from big file zone, allocate from the
	 * beginning using quadratic and linear search. For other zones,
	 * allocate from the end if allocating kind of data that don't belong
	 * there.
	 */
	if (z == &fs->zones[2]) goto forward_quad_search;
	if (__likely(z->grp_n)) {
		/*
		 * When allocating in other zones, search groups from the end.
		 * However, allocate from the beginning of a group.
		 * Also, trim required free space, to not expand too much in
		 * a zone where this data don't belong.
		 */
		unsigned grp_current;
		__d_off avg_free = z->freespace / (z->grp_n * SPAD_AVG_FREE_DIVIDE_OTHERZONE);
		if (__unlikely(!avg_free)) avg_free = al->n_sectors;
		retry2:
		grp_current = z->grp_start + z->grp_n;
		do {
			grp_current--;
			if (fs->group_info[grp_current].freespace >= avg_free) {
				al->sector = (__d_off)grp_current << fs->sectors_per_group_bits;
				al->top = (__d_off)(grp_current + 1) << fs->sectors_per_group_bits;
				if (__likely(!ALLOC_BLOCKS_IN_RANGE(fs, al))) {
					goto alloc_success_set_group;
				}
			}
		} while (grp_current > z->grp_start);
		if (avg_free > al->n_sectors) {
			avg_free = al->n_sectors;
			goto retry2;
		}
	}
	goto try_another_zone;
	failed:
	/*
	 * Try to allocate less sectors
	 */
	al->n_sectors = (al->n_sectors >> 1) & ~(fs->sectors_per_block - 1);
	if (__unlikely(!al->n_sectors)) return -ENOSPC;
	if (__likely(ALLOC_MASK(al->flags))) {
		if (__unlikely(al->n_sectors <= ALLOC_MASK(al->flags))) al->flags = (al->flags & ~(ALLOC_MASK_MASK | ALLOC_BIG_FILE)) | ALLOC_SMALL_FILE;
		else al->n_sectors &= ~ALLOC_MASK(al->flags);
	}
	/*
	 * It might be possible to allocate more than max_alloc, however trying
	 * to keep max_alloc exact would consume too much CPU time when the
	 * filesystem is running out of space.
	 */
	if (__likely(fs->max_alloc > al->n_sectors)) fs->max_alloc = al->n_sectors;
	al->sector = orig_start;
	al->top = (al->sector + fs->group_mask + 1) & ~fs->group_mask;
	if (!ALLOC_BLOCKS_IN_RANGE(fs, al)) return 0;
	goto retry_less;
}

static void GOAL_OUT_OF_FS(SPADFS *fs, struct alloc *al)
{
	al->sector = (__d_off)fs->zones[2].grp_start << fs->sectors_per_group_bits;
	if (__unlikely(al->sector >= fs->size)) {
		al->sector = (__d_off)fs->zones[1].grp_start << fs->sectors_per_group_bits;
		if (__unlikely(al->sector >= fs->size)) {
			al->sector = 0;
		}
	}
}

/*
 * Don't create new directory in the same group if:
 *	it has below-average free space
 *	and
 *	it is more than 75% full
 */

static int SKIP_GROUP(SPADFS *fs, __d_off start)
{
	/* start is not off-disk so this won't overflow */
	unsigned grp_current = start >> fs->sectors_per_group_bits;
	struct zone *z = fs->group_info[grp_current].zone;
	__d_off avg_free;
	/* z->grp_n is not 0, otherwise we wouldn't fall into this group */
	avg_free = z->freespace / z->grp_n;
	if (fs->group_info[grp_current].freespace >= avg_free) return 0;
	return fs->group_info[grp_current].freespace << 2 < (__d_off)1 << fs->sectors_per_group_bits;
}

__COLD_ATTR__ static int CONVERT_TO_BITMAP(struct apage_head *a, __d_off start, __d_off end, __u8 *scratch)
{
#define aa(i) (*(struct aentry *)((char *)a + (i)))
	int i;
	int depth = APAGE_SIZE(a->s.u.l.flags);
	memset(scratch, 0xff, depth);
	for (i = sizeof(struct aentry); i < depth; i += sizeof(struct aentry)) {
		__d_off st;
		__u32 len = SPAD2CPU32_LV(&aa(i).len);
		unsigned Xoff, Xlen;
		unsigned shift;
		if (__unlikely(!len)) continue;	/* shoudn't happen because only full apages are converted */
		st = SPAD2CPU64_LV(&aa(i).start);
		if (__unlikely(st < start) || __unlikely(st + len < st) || __unlikely(st + len > end)) {
			_snprintf(APAGE_ERROR, __MAX_STR_LEN, "POINTER (%012"__d_off_format"X,%08X) OUT OF RANGE (%012"__d_off_format"X-%012"__d_off_format"X)", st, len, start, end);
			return -1;
		}
		shift = APAGE_SECTORS_PER_BLOCK_BITS(a->s.u.l.flags);
		if (__unlikely(((unsigned)st | (unsigned)len) & ((1 << shift) - 1))) {
			_snprintf(APAGE_ERROR, __MAX_STR_LEN, "POINTER (%012"__d_off_format"X,%08X) NOT ALIGNED", st, len);
			return -1;
		}
		Xoff = (st - start) >> shift;
		Xlen = len >> shift;
		do {
			if (__unlikely(!BITMAP_TEST(scratch, Xoff))) {
				_snprintf(APAGE_ERROR, __MAX_STR_LEN, "BLOCK %012"__d_off_format"X+%08X FREED TWICE", st, (Xoff << shift) - (unsigned)(st - start));
				return -1;
			}
			BITMAP_CLEAR(scratch, Xoff);
			Xoff++;
		} while (--Xlen);
	}
	memcpy(a + 1, scratch + sizeof(struct apage_head), depth - sizeof(struct apage_head));
	a->s.u.l.flags |= APAGE_BITMAP;
	a->s.u.b.start0 = MAKE_PART_0(start);
	a->s.u.b.start1 = MAKE_PART_1(start);
	return 0;
#undef aa
}

__COLD_ATTR__ static int SPAD_WRITE_APAGE_INDEX(SPADFS *fs)
{
	BIORQ biorq;
	BIODESC desc;
	if (__likely(!CC_CURRENT(fs, &fs->a_cc, &fs->a_txc))) {
		struct txblock *tx;
		BUFFER *b;
		if (__unlikely(__IS_ERR(tx = SPAD_READ_TX_BLOCK(fs, &b)))) {
			_snprintf(APAGE_ERROR, __MAX_STR_LEN, "UNABLE TO READ TX BLOCK: %s", strerror(-__PTR_ERR(tx)));
			return __PTR_ERR(tx);
		}
		CC_SET_CURRENT(fs, &fs->a_cc, &fs->a_txc);
		tx->a_cc = fs->a_cc;
		tx->a_txc = fs->a_txc;
		SPAD_TX_BLOCK_CHECKSUM(tx);
		VFS$MARK_BUFFER_DIRTY(b, fs->txb_sec & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_block);
		VFS$PUT_BUFFER(tx);
	}
	biorq.h = fs->disk_handle_num;
	biorq.sec = CC_VALID(fs, &fs->a_cc, &fs->a_txc) ? fs->apage_index0_sec : fs->apage_index1_sec;
	biorq.nsec = APAGE_INDEX_SECTORS(fs->n_apages, fs->blocksize);
	/*__debug_printf("WRITE INDEX: %d,%d\n", (int)biorq.sec, biorq.nsec);*/
	biorq.flags = BIO_WRITE;
	biorq.desc = &desc;
	biorq.proc = fs->syncproc;
	biorq.fault_sec = -1;
	desc.v.ptr = (unsigned long)fs->apage_index;
	desc.v.len = APAGE_INDEX_SECTORS(fs->n_apages, fs->blocksize) * BIO_SECTOR_SIZE;
	desc.v.vspace = &KERNEL$VIRTUAL;
	desc.next = NULL;
	SYNC_IO_CANCELABLE(&biorq, KERNEL$BIO);
	if (__unlikely(biorq.status < 0)) {
		_snprintf(APAGE_ERROR, __MAX_STR_LEN, "UNABLE TO WRITE APAGE INDEX: %s", strerror(-biorq.status));
		return biorq.status;
	}
	return 0;
}

__COLD_ATTR__ static int SPLIT_APAGE(SPADFS *fs, int ap, struct apage_head *a, __u8 *scratch)
{
#define aa(i) (*(struct aentry *)((char *)a + (i)))
#define	newpage0 ((struct apage_head *)scratch)
#define newpage1 ((struct apage_head *)(scratch + (__PAGE_CLUSTER_SIZE >> 1)))
	struct apage_head *aa;
	int af;
	int pref, i, n;
	__d_off split_block;
	__d_off di, pref_addr;
	__d_off p;
	unsigned b_s;
	int depth = APAGE_SIZE(a->s.u.l.flags);
	if (__unlikely(fs->n_active_apages >= fs->n_apages)) {
		_snprintf(APAGE_ERROR, __MAX_STR_LEN, "ALL %u APAGES USED", fs->n_active_apages);
		VFS$PUT_BUFFER(a);
		return -EFSERROR;
	}
	n = 0;
	for (i = depth / (2 * sizeof(struct aentry)); i; i--) {
		n = APTR_ALIGN(SPAD2CPU16_LV(&aa(n).next));
		if (__unlikely(n >= depth)) {
			_snprintf(APAGE_ERROR, __MAX_STR_LEN, "POINTER OVERRUN");
			VFS$PUT_BUFFER(a);
			return -EFSERROR;
		}
	}
	if (__unlikely(!n)) split_block = ((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector) + (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector)) >> 1;
	else split_block = SPAD2CPU64_LV(&aa(n).start);
	split_block &= ~((__d_off)fs->sectors_per_block - 1);
	b_s = (BITMAP_SIZE(depth) >> 1 << fs->sectors_per_block_bits);
	if (__unlikely(split_block - (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector) < b_s)) {
		split_block = (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector) + b_s;
		split_block += fs->sectors_per_block - 1;
		split_block &= ~((__d_off)fs->sectors_per_block - 1);
	}
	if (__unlikely((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) - split_block < b_s)) {
		split_block = (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) - b_s;
		split_block &= ~((__d_off)fs->sectors_per_block - 1);
	}
	/*
		split_block = ((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector) + (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector)) >> 1;
		split_block &= ~((__d_off)fs->sectors_per_block - 1);
	*/

	/* make new pages */
	if (SPAD_APAGE_MAKE(newpage0, depth / sizeof(struct aentry), fs->sectors_per_block_bits) ||
	    SPAD_APAGE_MAKE(newpage1, depth / sizeof(struct aentry), fs->sectors_per_block_bits)) {
		_snprintf(APAGE_ERROR, __MAX_STR_LEN, "CAN'T MAKE NEW APAGE");
		VFS$PUT_BUFFER(a);
		return -EFSERROR;
	}

	/* insert existing page into new pages */
	for (i = sizeof(struct aentry); i < depth; i += sizeof(struct aentry)) {
		int r;
		__d_off st;
		if (__unlikely(aa(i).len == CPU2SPAD32(0))) continue;
		st = SPAD2CPU64_LV(&aa(i).start);
		if (st >= split_block) r = SPAD_APAGE_FREE(newpage1, st, SPAD2CPU32_LV(&aa(i).len));
		else if (__likely(st + SPAD2CPU32_LV(&aa(i).len) <= split_block)) r = SPAD_APAGE_FREE(newpage0, st, SPAD2CPU32_LV(&aa(i).len));
		else {
			r = SPAD_APAGE_FREE(newpage0, st, split_block - st);
			r |= SPAD_APAGE_FREE(newpage1, split_block, st + SPAD2CPU32_LV(&aa(i).len) - split_block);
		}
		if (__unlikely(r)) {
			if (r > 0) _snprintf(APAGE_ERROR, __MAX_STR_LEN, "OUT OF SPACE WHEN SPLITTING APAGE");
			VFS$PUT_BUFFER(a);
			return -EFSERROR;
		}
	}

	p = ((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - (af = (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].apage) > split_block)].end_sector) + split_block) >> 1;
	pref = -1;
	di = -1;		/* warning go away */
	for (i = fs->n_active_apages; i < fs->n_apages; i++) {
		__d_off dx = (__d_off)SPAD2CPU64_LV(&fs->apage_index[i].apage) > p ? (__d_off)SPAD2CPU64_LV(&fs->apage_index[i].apage) - p : p - (__d_off)SPAD2CPU64_LV(&fs->apage_index[i].apage);
		if (pref == -1 || dx < di) {
			pref = i;
			di = dx;
		}
	}
	pref_addr = (__d_off)SPAD2CPU64_LV(&fs->apage_index[pref].apage);
	for (i = pref - 1; i >= ap + (1 - af); i--) {
		memcpy(&fs->apage_index[i + 1], &fs->apage_index[i], sizeof(struct apage_index_entry));
		memcpy(&fs->apage_info[i + 1], &fs->apage_info[i], sizeof(struct apage_info));
	}
	CPU2SPAD64_LV(&fs->apage_index[i + 1].apage, pref_addr);
	fs->apage_index[ap + 1].end_sector = fs->apage_index[ap].end_sector;
	CPU2SPAD64_LV(&fs->apage_index[ap].end_sector, split_block);
	fs->n_active_apages++;
	fs->max_apage_invalid_checksum++;
	memset(&fs->apage_info[ap], 0, sizeof(struct apage_info) * 2);
	if (!(fs->flags & SPADFS_DONT_MAKE_CHECKSUMS)) {
		fs->apage_info[ap].invalid_checksum = 1;
		fs->apage_info[ap + 1].invalid_checksum = 1;
		if (ap < fs->min_apage_invalid_checksum) fs->min_apage_invalid_checksum = ap;
		if (ap + 1 >= fs->max_apage_invalid_checksum) fs->max_apage_invalid_checksum = ap + 2;
	}
	memcpy(&a->s, &(!af ? newpage0 : newpage1)->s, depth - (int)(long)&((struct apage_head *)NULL)->s);
	aa = (struct apage_head *)((unsigned long)a ^ (fs->pagesize >> 1));
	memcpy(&(!af ? newpage0 : newpage1)->s, &aa->s, depth - (int)(long)&((struct apage_head *)NULL)->s);
	VFS$PUT_BUFFER(a);
	if (__unlikely(__IS_ERR(a = MAP_APAGE(fs, ap + (1 - af), MAP_NEW)))) {
		_snprintf(APAGE_ERROR, __MAX_STR_LEN, "CAN'T READ NEW APAGE");
		return __PTR_ERR(a);
	}
	memcpy(&a->s, &(!af ? newpage1 : newpage0)->s, depth - (int)(long)&((struct apage_head *)NULL)->s);
	aa = (struct apage_head *)((unsigned long)a ^ (fs->pagesize >> 1));
	memcpy(&aa->s, &(!af ? newpage0 : newpage1)->s, depth - (int)(long)&((struct apage_head *)NULL)->s);
	CPU2SPAD16_LV(&a->cc, fs->cc);
	CPU2SPAD32_LV(&a->txc, fs->txc);
	VFS$PUT_BUFFER(a);
	return SPAD_WRITE_APAGE_INDEX(fs);
#undef aa
#undef newpage0
#undef newpage1
}

__COLD_ATTR__ int SPAD_RESURRECT_BLOCKS(SPADFNODE *fnode, __d_off start, __u32 n_blocks)
{
	SPADFS *fs = (SPADFS *)fnode->fs;
	__d_off free_from;
	__d_off free_to;
	struct apage_head *a;
	struct alloc al;
	int r;
	unsigned ap;
	__u32 n_bl;
	if (__unlikely(!n_blocks) || __unlikely((n_blocks & (fs->sectors_per_block - 1)) != 0) || __unlikely(start <= SUPERBLOCK_SECTOR) || __unlikely(start >= fs->size) || __unlikely(start + n_blocks > fs->size)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_RESURRECT_BLOCKS: %s: %012"__d_off_format"X,%08X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), start, (int)n_blocks);
		return -EFSERROR;
	}
	retry:
	ap = ADDR_2_APAGE(fs, start);
	if ((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) < start + n_blocks) {
		__u32 s = (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) - start;
		if (__unlikely(!s) || __unlikely(s >= n_blocks)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_RESURRECT_BLOCKS: %s: %012"__d_off_format"X,%08X) - APAGE %d, APAGE_END %012"__64_format"X, S %08X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), start, (int)n_blocks, ap, (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector), s);
			return -EFSERROR;
		}
		r = SPAD_RESURRECT_BLOCKS(fnode, start, s);
		if (__unlikely(r)) return r;
		start += s;
		n_blocks -= s;
		goto retry;
	}
	if (__unlikely(__IS_ERR((a = MAP_APAGE(fs, ap, MAP_WRITE)))))
		return __PTR_ERR(a);
	al.sector = start;
	al.n_sectors = n_blocks;
	al.flags = ALLOC_SMALL_FILE;
	al.sector = start;
	al.top = start + n_blocks;
	r = ALLOC_BLOCKS_IN_APAGE(&al, a, NULL);
	if (__likely(!r)) {
		VFS$PUT_BUFFER(a);
		if (al.flags & ALLOC_FREE_FROM) {
			r = SPAD_FREE_BLOCKS_(fnode, al.top, al.sector - al.top, 0);
			if (__unlikely(r)) return r;
		}
		goto ret_0;
	}
	free_from = free_to = 0;
	n_bl = n_blocks;
	do {
		al.sector = start;
		al.n_sectors = fs->sectors_per_block;
		al.flags = ALLOC_SMALL_FILE;
		al.sector = start;
		al.top = start + fs->sectors_per_block;
		r = ALLOC_BLOCKS_IN_APAGE(&al, a, NULL);
		if (__unlikely(r)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_RESURRECT_BLOCKS: %s: COULD NOT RESURRECT BLOCK %012"__d_off_format"X: %s", VFS$FNODE_DESCRIPTION((FNODE *)fnode), start, strerror(-r));
			VFS$PUT_BUFFER(a);
			return r;
		}
		if (al.flags & ALLOC_FREE_FROM) {
			if (__unlikely(free_to != 0)) {
				SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_RESURRECT_BLOCKS: %s: FREE_FROM ALREADY SET AT %012"__d_off_format"X-%012"__d_off_format"X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), free_from, free_to);
			} else {
				free_from = al.top;
				free_to = al.sector;
			}
		}
		start += fs->sectors_per_block;
	} while (n_bl -= fs->sectors_per_block);
	start -= n_blocks;
	VFS$PUT_BUFFER(a);
	if (free_to > free_from) {
		r = SPAD_FREE_BLOCKS_(fnode, free_from, free_to - free_from, 0);
		if (__unlikely(r)) return r;
	}
	ret_0:
	if (__unlikely(fs->free_at_commit < n_blocks)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_RESURRECT_BLOCKS: %s: RESURRECTED MORE BLOCKS (%08X) THAN FREED (%012"__d_off_format"X)", VFS$FNODE_DESCRIPTION((FNODE *)fnode), (int)n_blocks, fs->free_at_commit);
	} else {
		fs->free_at_commit -= n_blocks;
		GROUP_ACTION(fs, start, n_blocks, DECREASE_FREESPACE);
	}
	return 0;
}

int SPAD_FREE_BLOCKS_(SPADFNODE *fnode, __d_off start, __d_off n_blocks, int acct)
{
	SPADFS *fs = (SPADFS *)fnode->fs;
	__u32 n_blocks_32;
	struct apage_head *a;
	int ap;
	int r;
	fs->flags |= FS_COMMIT_FREES_DATA;
	/* !n_blocks is useless, but for some reason, it improves machine code
	   (0x54 local stack with it, 0x70 without it) */
	if (__unlikely(!n_blocks) || __unlikely((((unsigned)n_blocks | (unsigned)start) & (fs->sectors_per_block - 1)) != 0) || __unlikely(start <= SUPERBLOCK_SECTOR) || __unlikely(start + n_blocks <= start) || __unlikely(start + n_blocks > fs->size)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_FREE_BLOCKS: %s: %012"__d_off_format"X,%012"__d_off_format"X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), start, n_blocks);
		return -EFSERROR;
	}
	retry:
	if (__unlikely(n_blocks & ~(__d_off)0xffffffffu)) {
		r = SPAD_FREE_BLOCKS_(fnode, start, 0xffffff80u, acct);
		if (__unlikely(r)) return r;
		start += 0xffffff80u;
		n_blocks -= 0xffffff80u;
		goto retry;
	}
	n_blocks_32 = n_blocks;
	retry_32:
	ap = ADDR_2_APAGE(fs, start);
	if ((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) < start + n_blocks_32) {
		__u32 s = (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) - start;
		if (__unlikely(!s) || __unlikely(s >= n_blocks_32)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_FREE_BLOCKS: %s: %012"__d_off_format"X,%08X - APAGE %d, APAGE_END %012"__64_format"X, S %08X", VFS$FNODE_DESCRIPTION((FNODE *)fnode), start, (unsigned)n_blocks_32, ap, (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector), s);
			return -EFSERROR;
		}
		r = SPAD_FREE_BLOCKS_(fnode, start, s, acct);
		if (__unlikely(r)) return r;
		start += s;
		n_blocks_32 -= s;
		goto retry_32;
	}
	if (fs->max_alloc + n_blocks_32 > (unsigned)MAXINT) fs->max_alloc = MAXINT & ~(fs->sectors_per_block - 1);
	else fs->max_alloc += n_blocks_32;
	if (__unlikely(__IS_ERR((a = MAP_APAGE(fs, ap, MAP_WRITE))))) return __PTR_ERR(a);
	/*fs->apage_info[ap].full = 0;*/
	free_again:
	r = SPAD_APAGE_FREE(a, start, n_blocks_32);
	if (__likely(!r)) {
		VFS$PUT_BUFFER(a);
		if (__likely(acct)) {
			fs->free_at_commit += n_blocks_32;
			GROUP_ACTION(fs, start, n_blocks_32, INCREASE_FREESPACE);
		}
		return 0;
	}
	if (__unlikely(r < 0)) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_FREE_BLOCKS: %s: APAGE ERROR FREEING BLOCK (%012"__d_off_format"X,%08X): %s", VFS$FNODE_DESCRIPTION((FNODE *)fnode), start, (unsigned)n_blocks_32, APAGE_ERROR);
		VFS$PUT_BUFFER(a);
		return r;
	}
	if (__unlikely(a->s.u.l.flags & APAGE_BITMAP))
		KERNEL$SUICIDE("SPAD_FREE_BLOCKS_: %s: ALREADY IN BITMAP FORMAT", VFS$FNODE_DESCRIPTION((FNODE *)fnode));

	if (__likely((__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector) - (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector) > BITMAP_SIZE(APAGE_SIZE(a->s.u.l.flags)) << APAGE_SECTORS_PER_BLOCK_BITS(a->s.u.l.flags))) {
		if (__unlikely(r = SPLIT_APAGE(fs, ap, a, fs->scratch_page))) {
				/* SPLIT_APAGE does VFS$PUT_BUFFER */
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_FREE_BLOCKS: %s: UNABLE TO SPLIT APAGE %d (%012"__64_format"X-%012"__64_format"X): %s", VFS$FNODE_DESCRIPTION((FNODE *)fnode), ap, (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector), (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector), APAGE_ERROR);
			return r;
		}
		goto retry_32;

	}

	if (__unlikely(r = CONVERT_TO_BITMAP(a, (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector), (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector), fs->scratch_page))) {
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "SPAD_FREE_BLOCKS: %s: UNABLE TO CONVERT APAGE %d (%012"__64_format"X-%012"__64_format"X) TO BITMAP: %s", VFS$FNODE_DESCRIPTION((FNODE *)fnode), ap, (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap - 1].end_sector), (__d_off)SPAD2CPU64_LV(&fs->apage_index[ap].end_sector), APAGE_ERROR);
		VFS$PUT_BUFFER(a);
		return r;
	}
	goto free_again;
}

void SPAD_CHARGE_QUOTA(SPADFNODE *fnode, unsigned n_blocks)
{
	/* !!! FIXME: when quotas will be implemented */
}

void SPAD_APAGE_FIX_CHECKSUMS(SPADFS *fs)
{
	unsigned i;
	if (__likely(fs->min_apage_invalid_checksum >= fs->max_apage_invalid_checksum))
		return;
	if (__unlikely(fs->max_apage_invalid_checksum > fs->n_active_apages))
		fs->max_apage_invalid_checksum = fs->n_active_apages;
	if (__unlikely(fs->min_apage_invalid_checksum < 0))
		fs->min_apage_invalid_checksum = 0;
	for (i = fs->min_apage_invalid_checksum; i < fs->max_apage_invalid_checksum; i++) {
		if (fs->apage_info[i].invalid_checksum) {
			int depth;
			struct apage_head *a;
			if (__unlikely(__IS_ERR(a = MAP_APAGE(fs, i, MAP_WRITE)))) {
				fs->apage_info[i].invalid_checksum = 0;
				continue;
			}
			fs->apage_info[i].invalid_checksum = 0;
			depth = APAGE_SIZE(a->s.u.l.flags);
			a->s.u.l.flags |= APAGE_CHECKSUM_VALID;
			a->s.u.l.checksum ^= CHECKSUM_BASE ^ __byte_sum(&a->s, depth - (int)(long)&((struct apage_head *)NULL)->s);
			VFS$PUT_BUFFER(a);
		}
	}
	fs->min_apage_invalid_checksum = fs->n_active_apages;
	fs->max_apage_invalid_checksum = 0;
}

__COLD_ATTR__ static __d_off SPAD_COUNT_APAGE(SPADFS *fs, unsigned ap, int *err)
{
	struct apage_head *a;
	__d_off free;
	int depth;
	if (__unlikely(__IS_ERR(a = MAP_APAGE(fs, ap, MAP_READ)))) {
		*err = __PTR_ERR(a);
		return 0;
	}
	free = 0;
	depth = APAGE_SIZE(a->s.u.l.flags);
	if (__likely(!(a->s.u.l.flags & APAGE_BITMAP))) {
#define aa(i) (*(struct aentry *)((char *)a + (i)))
		unsigned n = 0;
		unsigned i = depth;
		next_n:
		n = APTR_ALIGN(SPAD2CPU16_LV(&aa(n).next));
		if (n >= depth) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "ERROR IN APAGE %u: POINTER OVERRUN", ap);
			error:
			*err = -EFSERROR;
			VFS$PUT_BUFFER(a);
			return free;
		}
		if (n) {
			if (__unlikely(aa(n).len == CPU2SPAD32(0))) {
				SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "ERROR IN APAGE %u: FIELD %u IS EMPTY BUT IS ON ALLOCATED LIST", ap, n);
				goto error;
			}
			if (__unlikely(SPLIT_GROUP_ACTION(fs, SPAD2CPU64_LV(&aa(n).start), SPAD2CPU32_LV(&aa(n).len), INCREASE_FREESPACE))) goto error;
			free += SPAD2CPU32_LV(&aa(n).len);
			if (__likely(--i > 0)) goto next_n;
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "ERROR IN APAGE %u: INFINITE LOOP", ap);
			goto error;
		}
#undef aa
	} else {
		unsigned Xb;
		__d_off bst = MAKE_D_OFF(a->s.u.b.start0, a->s.u.b.start1);
		unsigned Xsize = BITMAP_SIZE(depth);
		unsigned Xrun_start = 0;
		int last_bit = 1;
		unsigned shift = fs->sectors_per_block_bits;
		for (Xb = 0; Xb < Xsize; Xb++) if (__unlikely(BITMAP_TEST(a, Xb) != last_bit)) {
			if ((last_bit ^= 1)) {
				if (__unlikely(SPLIT_GROUP_ACTION(fs, bst + (Xrun_start << shift), (Xb - Xrun_start) << shift, INCREASE_FREESPACE))) goto error;
				free += (Xb - Xrun_start) << shift;
			} else {
				Xrun_start = Xb;
			}
		}
		if (!last_bit) {
			if (__unlikely(SPLIT_GROUP_ACTION(fs, bst + (Xrun_start << shift), (Xb - Xrun_start) << shift, INCREASE_FREESPACE))) goto error;
			free += (Xb - Xrun_start) << shift;
		}
	}
	VFS$PUT_BUFFER(a);
	return free;
}

__COLD_ATTR__ int SPAD_COUNT_FREE_SPACE(SPADFS *fs)
{
	int err = 0;
	unsigned i;
	__d_off group_free;
	for (i = 0; i < fs->n_active_apages; i++) {
		__d_off free;
		/*__debug_printf("spadfs count (%d of %d)\n", i, fs->n_active_apages);*/
		free = SPAD_COUNT_APAGE(fs, i, &err);
		fs->free += free;
		fs->free_allowed += free;
		if (__likely(free != 0)) WQ_WAKE_ALL(&fs->sync_done_wait);
		if (__unlikely(err)) goto ret;
	}

	ret:
	group_free = 0;
	for (i = 0; i < fs->n_groups; i++) {
		if (__unlikely(fs->group_info[i].freespace > fs->group_mask + 1)) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "GROUP %u HAS TOO MUCH FREE SPACE (%012"__d_off_format"X), PROBABLY OVERLAPPING APAGE ENTRIES", i, fs->group_info[i].freespace);
			return -EFSERROR;
		}
		group_free += fs->group_info[i].freespace;
	}
	if (__unlikely(group_free != fs->free))
		KERNEL$SUICIDE("SPAD_COUNT_FREE_SPACE: GROUP INFO FREE SUM (%012"__d_off_format"X) != FREE SECTOR COUNT (%012"__d_off_format"X)", group_free, fs->free);
	if (__unlikely(group_free != fs->zones[0].freespace + fs->zones[1].freespace + fs->zones[2].freespace))
		KERNEL$SUICIDE("SPAD_COUNT_FREE_SPACE: ZONE INFO FREE SUM (%012"__d_off_format"X+%012"__d_off_format"X+%012"__d_off_format"X) != FREE SECTOR COUNT (%012"__d_off_format"X)", fs->zones[0].freespace, fs->zones[1].freespace, fs->zones[2].freespace, fs->free);

	return err;
}
