#include "SPADFS.H"

/* returns:
	== 0 - fnode block
	== 1 - buffer not present
	> 1 - dnode page (offset to data)
	< 0 - error
*/

int SPAD_CHECK_DFNODE(struct fnode_block *f, BUFFER *b)
{
	unsigned blk = ((unsigned long)f >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
#define spadfs	((SPADFS *)b->fs)
	if (__unlikely(!BUFFER_VALID(b, blk))) return 1;
	if (__likely(f->magic == CPU2SPAD32(FNODE_BLOCK_MAGIC))) {
		if (__unlikely(!BUFFER_USERMAP_TEST(b, blk))) {
			if (__likely(f->flags & FNODE_BLOCK_CHECKSUM_VALID) && __likely(!(spadfs->flags & SPADFS_DONT_CHECK_CHECKSUMS))) {
				if (__unlikely(__byte_sum(f, FNODE_BLOCK_SIZE) != CHECKSUM_BASE)) {
					SPADFS_ERROR(spadfs, TXFLAGS_CHECKSUM_ERROR, "BAD CHECKSUM ON FNODE BLOCK %012"__d_off_format"X", b->blk | (((unsigned long)f >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
					return -EFSERROR;
				}
			}
			BUFFER_USERMAP_SET(b, blk);
		}
		if (!(f->flags & FNODE_BLOCK_LAST)) {
			if (__unlikely(!(((unsigned long)f + FNODE_BLOCK_SIZE) & (spadfs->pagesize - 1)))) {
				SPADFS_ERROR(spadfs, TXFLAGS_FS_ERROR, "FNODE BLOCK %012"__d_off_format"X CROSSES PAGE BOUNDARY", b->blk | (((unsigned long)f >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
				return -EFSERROR;
			}
		}
		if (!(f->flags & FNODE_BLOCK_FIRST)) {
			if (__unlikely(((unsigned long)f & (spadfs->pagesize - 1)) < FNODE_BLOCK_SIZE)) {
				SPADFS_ERROR(spadfs, TXFLAGS_FS_ERROR, "FNODE BLOCK %012"__d_off_format"X DOES NOT HAVE PREVIOUS NODE", b->blk | (((unsigned long)f >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
				return -EFSERROR;
			}
		}
		return 0;
#define d ((struct dnode_page *)f)
	} else if (__likely(d->magic == CPU2SPAD32(DNODE_PAGE_MAGIC))) {
		unsigned data, o;
		if (__unlikely((d->gflags & DNODE_GFLAGS_PAGE_SIZE_BITS_MINUS_1) >> DNODE_GFLAGS_PAGE_SIZE_BITS_MINUS_1_SHIFT != spadfs->sectors_per_page_bits - 1)) {
			SPADFS_ERROR(spadfs, TXFLAGS_FS_ERROR, "DNODE %012"__d_off_format"X HAS INVALID PAGE SIZE", b->blk | (((unsigned long)d >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
			return -EFSERROR;
		}
		if (__unlikely(((unsigned long)d & (spadfs->pagesize - 1)) + (spadfs->dnode_page_sectors << BIO_SECTOR_SIZE_BITS) > spadfs->pagesize)) {
			SPADFS_ERROR(spadfs, TXFLAGS_FS_ERROR, "DNODE %012"__d_off_format"X CROSSES PAGE BOUNDARY", b->blk | (((unsigned long)d >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
			return -EFSERROR;
		}
		if (__unlikely(!BUFFERS_VALID(b, blk, spadfs->dnode_page_sectors))) return 1;
		if (CC_VALID(spadfs, &d->cc, &d->txc)) data = DNODE_ENTRY_OFFSET, o = 0;
		else data = DNODE_ENTRY_OFFSET + spadfs->dnode_data_size, o = 1;
		if (__unlikely(!BUFFER_USERMAP_TEST(b, blk))) {
			if (__likely(d->flags[o] & DNODE_CHECKSUM_VALID) && __likely(!(spadfs->flags & SPADFS_DONT_CHECK_CHECKSUMS))) {
				if (__unlikely(__byte_sum((char *)d + data, spadfs->dnode_data_size) != (CHECKSUM_BASE ^ d->checksum[o]))) {
					SPADFS_ERROR(spadfs, TXFLAGS_CHECKSUM_ERROR, "BAD CHECKSUM ON DNODE PAGE %012"__d_off_format"X", b->blk | (((unsigned long)d >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
					return -EFSERROR;
				}
			}
			BUFFER_USERMAP_SET(b, blk);
		}
		return data;
#undef d
	} else {
		SPADFS_ERROR(spadfs, TXFLAGS_FS_ERROR, "BAD MAGIC ON BLOCK %012"__d_off_format"X", b->blk | (((unsigned long)f >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
		return -EFSERROR;
	}
#undef spadfs
}

int SPAD_BEGIN_MODIFY_DNODE(SPADFS *fs, struct dnode_page *d, int valid)
{
	int newvalid;
	if (__likely(CC_CURRENT(fs, &d->cc, &d->txc))) return valid;
	newvalid = DNODE_ENTRY_OFFSET * 2 + fs->dnode_data_size - valid;
	memcpy((char *)d + newvalid, (char *)d + valid, fs->dnode_data_size);
	CC_SET_CURRENT(fs, &d->cc, &d->txc);
	return newvalid;
}

void SPAD_END_MODIFY_DNODE(BUFFER *b, struct dnode_page *d, int valid)
{
#define fs ((SPADFS *)b->fs)
	int o = valid != DNODE_ENTRY_OFFSET;
	if (!(fs->flags & SPADFS_DONT_MAKE_CHECKSUMS)) {
		d->checksum[o] = __byte_sum((char *)d + valid, fs->dnode_data_size) ^ CHECKSUM_BASE;
		d->flags[o] |= DNODE_CHECKSUM_VALID;
	} else d->flags[o] &= ~DNODE_CHECKSUM_VALID;
	VFS$MARK_BUFFER_DIRTY(b, ((unsigned long)d >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->dnode_page_sectors);
#undef fs
}

static __finline__ void SPAD_MAKE_LEAF_PAGE(struct fnode_block *b, unsigned dirblks, SPADFS *fs)
{
	struct fnode *fn;
	unsigned i = 0;
	next_block:
	CPU2SPAD32_LV(&b->magic, FNODE_BLOCK_MAGIC);
	b->flags = (FNODE_BLOCK_FIRST * !i) | (FNODE_BLOCK_LAST * (i == dirblks - 1));
	/*CPU2SPAD32_LV(&b->prev0, 0);
	CPU2SPAD16_LV(&b->prev1, 0);*/
	CPU2SPAD32_LV(&b->txc, TXC_INVALID(0));
	/*CPU2SPAD16_LV(&b->cc, 0);*/
	fn = b->fnodes;
	CPU2SPAD16_LV(&fn->next, (FNODE_MAX_SIZE & FNODE_NEXT_SIZE) | FNODE_NEXT_FREE);
	/*CPU2SPAD16_LV(&fn->cc, 0);
	CPU2SPAD32_LV(&fn->txc, 0);*/
	if (!(fs->flags & SPADFS_DONT_MAKE_CHECKSUMS)) {
		b->flags |= FNODE_BLOCK_CHECKSUM_VALID;
		b->checksum /*^*/= CHECKSUM_BASE ^ __byte_sum(b, FNODE_BLOCK_SIZE);
	}
	if (++i < dirblks) {
		b = (struct fnode_block *)((char *)b + FNODE_BLOCK_SIZE);
		goto next_block;
	}
}

struct fnode_block *SPAD_ALLOC_LEAF_PAGE(SPADFNODE *fn, __d_off hint, unsigned sectors, unsigned flags, __d_off *result)
{
#define fs ((SPADFS *)fn->fs)
	struct alloc al;
	int r;
	al.sector = hint;
	al.n_sectors = sectors;
	al.flags = ALLOC_METADATA | flags;
	r = SPAD_ALLOC_BLOCKS(fn, &al);
	if (__likely(!r)) {
		struct fnode_block *b = VFS$ALLOC_BUFFER_SYNC((FS *)fs, *result = al.sector, al.n_sectors, (void *)&KERNEL$LIST_END);
		if (__unlikely(__IS_ERR(b))) return b;
		SPAD_MAKE_LEAF_PAGE(b, al.n_sectors, fs);
		return b;
	}
	return __ERR_PTR(r);
#undef fs
}

int SPAD_ALLOC_DNODE_PAGE(SPADFNODE *fn, __d_off hint, __d_off *result, __d_off parent, __d_off ptr_init)
{
#define fs ((SPADFS *)fn->fs)
	struct alloc al;
	int r;
	al.sector = hint;
	al.n_sectors = fs->dnode_page_sectors;
	al.flags = ALLOC_METADATA;
	r = SPAD_ALLOC_BLOCKS(fn, &al);
	if (__likely(!r)) {
		BUFFER *b;
		__u32 b0, b1, b2;
		int i;
		struct dnode_page_entry *e;
		struct dnode_page *d = VFS$ALLOC_BUFFER_SYNC((FS *)fs, *result = al.sector, al.n_sectors, &b);
		if (__unlikely(__IS_ERR(d))) return __PTR_ERR(b);
		CPU2SPAD32_LV(&d->magic, DNODE_PAGE_MAGIC);
		CPU2SPAD64_LV(&d->up_dnode, parent);
		d->gflags = (fs->sectors_per_page_bits - 1) << DNODE_GFLAGS_PAGE_SIZE_BITS_MINUS_1_SHIFT;
		CPU2SPAD32_LV(&d->txc, fs->txc);
		CPU2SPAD16_LV(&d->cc, fs->cc);
		b0 = MAKE_PART_30(ptr_init);
		b1 = MAKE_PART_31(ptr_init);
		b2 = MAKE_PART_32(ptr_init);
		e = (struct dnode_page_entry *)((char *)d + DNODE_ENTRY_OFFSET);
		for (i = 1 << fs->dnode_hash_bits; i; i--) {
			e->b0 = b0, e->b1 = b1, e->b2 = b2;
			e++;
		}
		SPAD_END_MODIFY_DNODE(b, d, DNODE_ENTRY_OFFSET);
		VFS$PUT_BUFFER(d);
		return 0;
	}
	return r;
#undef fs
}

struct fnode_block *SPAD_FIND_LEAF_PAGE_FOR_HASH(SPADFNODE *fn, __d_off root, hash_t hash, BUFFER **bb, __d_off *fnode_sector, __d_off *prev_dnode, int *hash_bits, int *splitable)
{
#define fs ((SPADFS *)fn->fs)
	int r;
	BUFFER *b;
	struct fnode_block *f;
	__d_off new_page_sector = 0;
	*prev_dnode = 0;
	*hash_bits = 0;
	*splitable = -1;
	subnode:
	f = VFS$READ_BUFFER_SYNC((FS *)fs, root, 1, __likely(!*hash_bits) ? fs->buffer_readahead : 0, &b);
	*fnode_sector = root;
	again:
	if (__unlikely(__IS_ERR(f))) return f;
	r = SPAD_CHECK_DFNODE(f, b);
	if (__likely(!r)) {
		struct fnode_block *ff;
		for (ff = f; !(ff->flags & FNODE_BLOCK_LAST); ff = (struct fnode_block *)((char *)ff + FNODE_BLOCK_SIZE)) {
			r = SPAD_CHECK_DFNODE(ff, b);
			if (__likely(!r)) continue;
			VFS$PUT_BUFFER(f);
			if (__likely(r == 1)) {
				read_more:
				f = VFS$READ_BUFFER_SYNC((FS *)fs, root, __SECTORS_PER_PAGE_CLUSTER - ((unsigned long)root & __SECTORS_PER_PAGE_CLUSTER_MINUS_1), 0, &b);
				goto again;
			}
			if (r > 1) {
				SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012"__d_off_format"X", root);
				r = -EFSERROR;
			}
			return __ERR_PTR(r);
		}
		if (((char *)ff - (char *)f + FNODE_BLOCK_SIZE) > fs->sectors_per_fnodepage << BIO_SECTOR_SIZE_BITS) {
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "TOO LONG FNODE PAGE ON BLOCK %012"__d_off_format"X", root);
			VFS$PUT_BUFFER(f);
			return __ERR_PTR(-EFSERROR);
		}
		*bb = b;
		return f;
	} else if (r == 1) {
		VFS$PUT_BUFFER(f);
		goto read_more;
	} else if (__likely(r > 1)) {
		int i, j;
		struct dnode_page_entry *e;
		__d_off new_root;
		hash_t hpos;
		*prev_dnode = root;
		hpos = hash & ((1 << (fs->dnode_hash_bits)) - 1);
		e = (struct dnode_page_entry *)((char *)f + r);
		new_root = MAKE_D_OFF3(e[hpos].b0, e[hpos].b1, e[hpos].b2);
		if (__unlikely(!new_root)) {
			if (!new_page_sector) {
				int dsize;
				struct fnode_block *new_fn;
				VFS$PUT_BUFFER(f);
				dsize = fs->sectors_per_fnodepage;
				if (__unlikely(!SPAD_CAN_OVERALLOC(fn, dsize - 1)))
					alloc_1: dsize = fs->sectors_per_block;
				new_fn = SPAD_ALLOC_LEAF_PAGE(fn, root, dsize, 0, &new_page_sector);
				if (__unlikely(__IS_ERR(new_fn))) {
					if (dsize > fs->sectors_per_block) goto alloc_1;
					if (__unlikely(__PTR_ERR(new_fn) == -ENOSPC)) SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "OUT OF SPACE FOR DIRECTORY LEAF PAGE");
					return new_fn;
				}
				VFS$PUT_BUFFER(new_fn);
				goto subnode;
			} else {
				int x;
				int step;
				r = SPAD_BEGIN_MODIFY_DNODE(fs, (struct dnode_page *)f, r);
				e = (struct dnode_page_entry *)((char *)f + r);
				e[hpos].b0 = MAKE_PART_30(new_page_sector);
				e[hpos].b1 = MAKE_PART_31(new_page_sector);
				e[hpos].b2 = MAKE_PART_32(new_page_sector);
				step = 1 << fs->dnode_hash_bits;
				do {
					hpos ^= step >> 1;
					for (x = hpos; x < 1 << fs->dnode_hash_bits; x += step) {
						if (MAKE_D_OFF3(e[x].b0, e[x].b1, e[x].b2)) goto done;
					}
					for (x = hpos; x < 1 << fs->dnode_hash_bits; x += step) {
						e[x].b0 = MAKE_PART_30(new_page_sector);
						e[x].b1 = MAKE_PART_31(new_page_sector);
						e[x].b2 = MAKE_PART_32(new_page_sector);
					}
					step >>= 1;
					hpos &= ~step;
				} while (step > 1);
				done:
				SPAD_END_MODIFY_DNODE(b, (struct dnode_page *)f, r);
				VFS$PUT_BUFFER(f);
				root = new_page_sector;
				goto subnode;
			}
		}
		if (__unlikely(*hash_bits % (unsigned)fs->dnode_hash_bits) || __unlikely(*splitable != -1))
			goto bad_tree;
		j = 1 << fs->dnode_hash_bits;
		for (i = 1; i < j; i <<= 1) {
			if (MAKE_D_OFF3(e[hpos^i].b0, e[hpos^i].b1, e[hpos^i].b2) == new_root) {
				*splitable = hpos;
				break;
			}
			(*hash_bits)++;
		}
		hash >>= fs->dnode_hash_bits;
		VFS$PUT_BUFFER(f);
		if (__unlikely(*hash_bits > HASH_BITS)) goto bad_tree;
		root = new_root;
		goto subnode;
	} else {
		VFS$PUT_BUFFER(f);
		return __ERR_PTR(r);
	}

	bad_tree:
	SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD DNODE TREE ON %012"__d_off_format"X (PARENT %012"__d_off_format"X/%04X)", root, fn->fnode_address, fn->fnode_off);
	return __ERR_PTR(-EFSERROR);
#undef fs
}

int SPAD_REMOVE_RECURSIVE(SPADFNODE *fn, __d_off root, int depth)
{
#define fs ((SPADFS *)fn->fs)
	int r;
	__d_off c[2];
	int sectors;
	struct fnode_block *f;
	BUFFER *b;
	__d_off next_root;
	__d_off ln;
	unsigned hp;
	c[1] = 0;
	next_in_chain:
	next_root = 0;
	ln = 0;
	hp = 0;
	next_loop:
	f = VFS$READ_BUFFER_SYNC((FS *)fs, root, 1, fs->buffer_readahead, &b);
	again:
	if (__unlikely(__IS_ERR(f))) return __PTR_ERR(f);
	r = SPAD_CHECK_DFNODE(f, b);
	if (__likely(!r)) {
		struct fnode_block *ff;
		for (ff = f; !(ff->flags & FNODE_BLOCK_LAST); ff = (struct fnode_block *)((char *)ff + FNODE_BLOCK_SIZE)) {
			r = SPAD_CHECK_DFNODE(ff, b);
			if (__likely(!r)) continue;
			VFS$PUT_BUFFER(f);
			if (__likely(r == 1)) {
				read_more:
				f = VFS$READ_BUFFER_SYNC((FS *)fs, root, __SECTORS_PER_PAGE_CLUSTER - ((unsigned long)root & __SECTORS_PER_PAGE_CLUSTER_MINUS_1), fs->buffer_readahead, &b);
				goto again;
			}
			if (r > 1) {
				SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012"__d_off_format"X", root);
				r = -EFSERROR;
			}
			return r;
		}
		sectors = ((char *)ff - (char *)f + FNODE_BLOCK_SIZE) >> BIO_SECTOR_SIZE_BITS;
		if (CC_VALID(fs, &ff->cc, &ff->txc)) {
			next_root = MAKE_D_OFF(ff->next0, ff->next1);
		}
		VFS$PUT_BUFFER(f);
	} else if (r == 1) {
		VFS$PUT_BUFFER(f);
		goto read_more;
	} else if (__likely(r > 1)) {
		struct dnode_page_entry *e;
		if (__unlikely(depth >= HASH_BITS)) {
			VFS$PUT_BUFFER(f);
			SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "RECURSE OVERFLOW ON BLOCK %012"__d_off_format"X", root);
			return -EFSERROR;
		}
		e = (struct dnode_page_entry *)((char *)f + r);
		while (1) {
			int i;
			__d_off n = MAKE_D_OFF3(e[hp].b0, e[hp].b1, e[hp].b2);
			if (n && n != ln) {
				ln = n;
				VFS$PUT_BUFFER(f);
				r = SPAD_REMOVE_RECURSIVE(fn, n, depth + fs->dnode_hash_bits);
				if (__unlikely(r)) return r;
				i = 1 << (fs->dnode_hash_bits - 1);
				while (!((hp ^= i) & i)) if (__unlikely(!(i >>= 1))) goto brk;
				goto next_loop;
			}
			i = 1 << (fs->dnode_hash_bits - 1);
			while (!((hp ^= i) & i)) if (__unlikely(!(i >>= 1))) {
				VFS$PUT_BUFFER(f);
				goto brk;
			}
		}
		brk:
		sectors = fs->dnode_page_sectors;
	} else {
		VFS$PUT_BUFFER(f);
		return r;
	}
	VFS$CLEAN_BUFFERS_SYNC((FS *)fs, root, sectors);
	r = SPAD_FREE_BLOCKS(fn, root, sectors);
	if (__unlikely(r)) return r;
	if (next_root) {
		root = next_root;
		if (__likely(!SPAD_STOP_CYCLES(next_root, &c))) goto next_in_chain;
		SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "CYCLE DETECTED ON BLOCK %012"__d_off_format"X", next_root);
		return -EFSERROR;
	}
	return 0;
}

