/* !!! FIXME: resolve simultaneous download of two files somehow
	(probably make stats on groups when something was allocated in them)
*/
#include <SPAD/VFS.H>
#include <TIME.H>
#include <ARCH/BSF.H>

#include "SPADFS.H"

static void ANODE_ERROR_MSG(struct anode *anode, BUFFER *b, unsigned flags, char *msg);
static void BMAP_OUT_OF_RANGE_MSG(SPADPAGEINRQ *RQ, struct extent *e);
static void BMAP_NON_MONOTONIC_MSG(int off, SPADPAGEINRQ *RQ);

static __d_off size2blks(SPADFS *fs, _u_off_t size)
{
	__d_off result;
	FILE_BLOCKS(fs->blocksize, fs->clustersize, fs->cluster_threshold, size, result);
	return result;
}

static __finline__ __d_off alloc_meta_hint(SPADFS *fs, SPADFNODE *f)
{
	return f->fnode_address ? f->fnode_address : ((SPADFNODE *)f->parent)->root;
}

static __finline__ __d_off alloc_small_hint(SPADFS *fs, SPADFNODE *f)
{
	return (__d_off)((SPADFNODE *)f->parent)->blk1_n << fs->sectors_per_group_bits;
}

static __finline__ __d_off alloc_big_hint(SPADFS *fs, SPADFNODE *f)
{
	return (__d_off)((SPADFNODE *)f->parent)->blk2_n << fs->sectors_per_group_bits;
}

static void set_new_hint(SPADFNODE *f, struct alloc *al)
{
#define sfs	((SPADFS *)f->fs)
	time_t x_time = time(NULL) - SPAD_PROPAGATE_HINT_TIME;
	unsigned hint = al->sector >> sfs->sectors_per_group_bits;
	unsigned oldhint = __likely(al->flags & ALLOC_BIG_FILE) ? f->blk2_n : f->blk1_n;
	sub:
	if (__likely(al->flags & ALLOC_BIG_FILE)) f->blk2_n = hint;
	else f->blk1_n = hint;
	VFS$SET_DIRTY((FNODE *)f);
	f = (SPADFNODE *)f->parent;
	if (__likely(f != NULL) && __likely(__likely(al->flags & ALLOC_BIG_FILE) ? f->blk2_n : f->blk1_n == oldhint) && __likely(f->ctime > x_time) && __likely(f->parent != NULL)) goto sub;
#undef sfs
}

static __finline__ int CHECK_ANODE(SPADFS *fs, struct anode *anode, BUFFER *b)
{
#define sfs	((SPADFS *)fs)
	if (__unlikely(!BUFFER_USERMAP_TEST(b, ((unsigned long)anode >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1))) {
		if (__unlikely(anode->magic != CPU2SPAD32(ANODE_MAGIC))) {
			ANODE_ERROR_MSG(anode, b, TXFLAGS_FS_ERROR, "BAD MAGIC ON ANODE %012"__d_off_format"X");
			return -EFSERROR;
		}
		if (anode->flags & ANODE_CHECKSUM_VALID && __likely(!(sfs->flags & SPADFS_DONT_CHECK_CHECKSUMS))) {
			if (__unlikely(__byte_sum(anode, ANODE_SIZE) != CHECKSUM_BASE)) {
				ANODE_ERROR_MSG(anode, b, TXFLAGS_CHECKSUM_ERROR, "BAD CHECKSUM ON ANODE %012"__d_off_format"X");
				return -EFSERROR;
			}
		}
		BUFFER_USERMAP_SET(b, ((unsigned long)anode >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1);
	}
	if (__unlikely((unsigned)(anode->valid_extents - 1) >= ANODE_N_EXTENTS)) {
		ANODE_ERROR_MSG(anode, b, TXFLAGS_FS_ERROR, "ANODE %012"__d_off_format"X HAS %u VALID EXTENTS");
		return -EFSERROR;
	}
	return 0;
#undef sfs
}

#define BMAP_MAP	1
#define BMAP_EXTEND	2
#define BMAP_TRY	3

static struct anode *MAP_ANODE(SPADFS *fs, __d_off ano, BUFFER **b, int flags)
{
	struct anode *anode;
	int err;
	if (__unlikely(flags == BMAP_TRY)) {
		if (__unlikely(!(anode = VFS$GET_BUFFER((FS *)fs, ano, fs->sectors_per_block, b, &KERNEL$PROC_KERNEL)))) {
			return __ERR_PTR(-ENOENT);
		}
		goto map_ok;
	}
	if (__unlikely(__IS_ERR(anode = VFS$READ_BUFFER_SYNC((FS *)fs, ano, fs->sectors_per_block, fs->buffer_readahead, b)))) {
		SPADFS_ERROR(fs, TXFLAGS_IO_READ_ERROR, "UNABLE TO READ ANODE AT BLOCK %012"__d_off_format"X: %s", ano, strerror(-__PTR_ERR(anode)));
		return anode;
	}
	map_ok:
	if (__unlikely(err = CHECK_ANODE(fs, anode, *b))) {
		VFS$PUT_BUFFER(anode);
		return __ERR_PTR(err);
	}
	return anode;
}

static struct anode *ALLOC_ANODE(SPADFNODE *fn, __d_off hint, BUFFER **b, __d_off *result)
{
#define sfs ((SPADFS *)fn->fs)
	struct alloc al;
	struct anode *anode;
	int r;
	al.sector = hint;
	al.n_sectors = sfs->sectors_per_block;
	al.flags = ALLOC_METADATA;
	r = SPAD_ALLOC_BLOCKS(fn, &al);
	if (__unlikely(r)) return __ERR_PTR(r);
	anode = VFS$ALLOC_BUFFER_SYNC((FS *)sfs, *result = al.sector, al.n_sectors, b);
	if (__unlikely(__IS_ERR(anode))) return anode;
	CPU2SPAD32_LV(&anode->magic, ANODE_MAGIC);
	/*__debug_printf("alloc anode: %Lx\n", *result);*/
	return anode;
#undef sfs
}

#define DO_ANODE_CHECKSUM						\
do {									\
	if (!(sfs->flags & SPADFS_DONT_MAKE_CHECKSUMS)) {		\
		anode->flags |= ANODE_CHECKSUM_VALID;			\
		anode->checksum ^= CHECKSUM_BASE ^ __byte_sum(anode, ANODE_SIZE);\
	} else anode->flags &= ~ANODE_CHECKSUM_VALID;			\
} while (0)

#define DO_ANODE_DIRTY							\
do {									\
	DO_ANODE_CHECKSUM;						\
	VFS$MARK_BUFFER_DIRTY(b, (((unsigned long)anode >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1) & ~(sfs->sectors_per_block - 1), sfs->sectors_per_block);\
} while (0)

static AST_STUB SPAD_BMAP_AST;

void SPAD_BMAP(PAGEINRQ *pgin_)
{
	SPADPAGEINRQ *pgin = (SPADPAGEINRQ *)pgin_;
	SPADFNODE *f = (SPADFNODE *)pgin->fnode;
	/*__debug_printf("BMAP(%LX), %d, %d, %LX\n", pgin->off, f->blk1_n, f->blk2_n, f->root);*/
	if (__likely(pgin->off < f->blk1_n)) {
		bmap_1:
		f->file_blk = 0;
		f->disk_blk = f->blk1;
		f->run_length = f->blk1_n;
		if (__unlikely(f->blk2 == f->blk1 + f->blk1_n))
			f->run_length += f->blk2_n;
		call_done:
		if (is_bmap_invalid(f)) {
			SPAD_INVALID_BMAP_MSG(f);
			pgin->status = -EFSERROR;
		}
		VFS$BMAP_DONE((PAGEINRQ *)pgin);
		return;
	}

	if (__likely(pgin->off < f->blk1_n + f->blk2_n)) {
		if (__unlikely(f->blk2 == f->blk1 + f->blk1_n))
			goto bmap_1;
		f->file_blk = f->blk1_n;
		f->disk_blk = f->blk2;
		f->run_length = f->blk2_n;
		goto call_done;
	}

	pgin->fn = SPAD_BMAP_AST;
	pgin->depth_now = 0;
	pgin->depth_total = 0;
	pgin->dirptr = f->root;
	CALL_AST(pgin);
}

static DECL_AST(SPAD_BMAP_AST, SPL_FS, SPADPAGEINRQ)
{
#define sfs ((SPADFS *)RQ->fs)
#define sfnode ((SPADFNODE *)RQ->fnode)
	struct anode *anode;
	BUFFER *buf;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_FS), RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status != 0)) {
		VFS$BMAP_DONE((PAGEINRQ *)RQ);
		RETURN;
	}
	/* no need to test for lockup because number of cycles is limited */
	subnode:
	/*__debug_printf("BMAP_DIR(%LX)\n", RQ->dirptr);*/
	anode = VFS$GET_BUFFER(RQ->fs, RQ->dirptr, 1, &buf, RQ->tag.proc);
	if (__likely(anode != NULL)) {
		int err;
		int direct, off;
		if (__unlikely(err = CHECK_ANODE(sfs, anode, buf))) {
			VFS$PUT_BUFFER(anode);
			RQ->status = err;
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		direct = find_direct(RQ->depth_now, RQ->depth_total);
		off = find_in_anode(anode, RQ->off, anode->valid_extents);
		if (__unlikely(RQ->off >= SPAD2CPU64_LV(&anode->x[off].end_off)) || __unlikely(RQ->off < SPAD2CPU64_LV(&anode->x[off - 1].end_off))) {
			BMAP_OUT_OF_RANGE_MSG(RQ, &anode->x[off]);
			put_fserror:
			VFS$PUT_BUFFER(anode);
			fserror:
			RQ->status = -EFSERROR;
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		if (__unlikely(off >= direct)) {
			RQ->dirptr = SPAD2CPU64_LV(&anode->x[off].blk);
			update_depth(&RQ->depth_now, &RQ->depth_total, off);
			VFS$PUT_BUFFER(anode);
			goto subnode;
		}
		if (__unlikely(SPAD2CPU64_LV(&anode->x[off].end_off) <= SPAD2CPU64_LV(&anode->x[off - 1].end_off))) {
			BMAP_NON_MONOTONIC_MSG(off, RQ);
			goto put_fserror;
		}
		sfnode->file_blk = SPAD2CPU64_LV(&anode->x[off - 1].end_off);
		sfnode->disk_blk = SPAD2CPU64_LV(&anode->x[off].blk);
		sfnode->run_length = SPAD2CPU64_LV(&anode->x[off].end_off) - SPAD2CPU64_LV(&anode->x[off - 1].end_off);
		VFS$PUT_BUFFER(anode);
		if (is_bmap_invalid(sfnode)) {
			SPAD_INVALID_BMAP_MSG(sfnode);
			goto fserror;
		}
		VFS$BMAP_DONE((PAGEINRQ *)RQ);
		RETURN;
	} else {
		VFS$READ_BUFFER(RQ->dirptr, (PAGEINRQ *)RQ, RQ->fs->buffer_readahead);
		RETURN;
	}
#undef sfnode
#undef sfs
}

static int SYNC_BMAP(SPADFNODE *f, __d_off lbn, __d_off *blk, __d_off *nblks, __d_off *nback, int flags, __d_off *aptr, char *msg)
{
#define sfs ((SPADFS *)f->fs)
	int depth_now, depth_total;
	__d_off ano;
	__d_off nb;
	struct anode *anode;
	BUFFER *b;
	int direct, off;
	*aptr = 0;
	if (__likely(lbn < f->blk1_n)) {
		*blk = f->blk1 + lbn;
		*nblks = f->blk1_n - lbn;
		*nback = lbn;
		if (__unlikely(f->blk2 == f->blk1 + f->blk1_n))
			*nblks += f->blk2_n;
		if (flags == BMAP_EXTEND) {
			f->blk1_n = lbn + 1;
			f->blk2_n = 0;
			f->root = 0;
		}
		return 0;
	}
	if (__likely(lbn < f->blk1_n + f->blk2_n)) {
		*blk = f->blk2 + lbn - f->blk1_n;
		*nblks = f->blk1_n + f->blk2_n - lbn;
		*nback = lbn;
		if (__likely(f->blk2 != f->blk1 + f->blk1_n))
			*nback -= f->blk1_n;
		if (flags == BMAP_EXTEND) {
			f->blk2_n = lbn - f->blk1_n + 1;
			f->root = 0;
		}
		return 0;
	}
	ano = f->root;
	depth_now = 0;
	depth_total = 0;
	subnode:
	anode = MAP_ANODE(sfs, ano, &b, flags);
	if (__unlikely(__IS_ERR(anode))) return __PTR_ERR(anode);
	direct = find_direct(depth_now, depth_total);
	off = find_in_anode(anode, lbn, anode->valid_extents);
	if (__unlikely(lbn >= SPAD2CPU64_LV(&anode->x[off].end_off)) || __unlikely(lbn < SPAD2CPU64_LV(&anode->x[off - 1].end_off))) {
		SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "SYNC_BMAP(%s): OUT OF RANGE !(%012"__d_off_format"X <= %012"__d_off_format"X < %012"__d_off_format"X), ANODE (%012"__d_off_format"X -> %012"__d_off_format"X)", msg, SPAD2CPU64_LV(&anode->x[off - 1].end_off), lbn, SPAD2CPU64_LV(&anode->x[off].end_off), f->root, ano);
		brelse_err:
		VFS$PUT_BUFFER(anode);
		return -EFSERROR;
	}
	if (__unlikely(off >= direct)) {
		ano = SPAD2CPU64_LV(&anode->x[off].blk);
		if (flags == BMAP_EXTEND) {
			if (__unlikely(SPAD2CPU64_LV(&anode->x[off].end_off) != -1) || __unlikely(anode->valid_extents != off + 1)) {
				CPU2SPAD64_LV(&anode->x[off].end_off, -1);
				anode->valid_extents = off + 1;
				DO_ANODE_DIRTY;
			}
		}
		update_depth(&depth_now, &depth_total, off);
		VFS$PUT_BUFFER(anode);
		goto subnode;
	}
	if (__unlikely(!off) || (__unlikely(off == 2) && __unlikely(!depth_now))) *aptr = ano;
	if (__unlikely(SPAD2CPU64_LV(&anode->x[off].end_off) <= SPAD2CPU64_LV(&anode->x[off - 1].end_off))) {
		SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "SYNC_BMAP(%s): NON-MONOTONIC ANODE (%012"__d_off_format"X -> %012"__d_off_format"X), ENTRY %d", msg, f->root, ano, off);
		goto brelse_err;
	}
	nb = lbn - SPAD2CPU64_LV(&anode->x[off - 1].end_off);
	*nback = nb;
	*blk = SPAD2CPU64_LV(&anode->x[off].blk) + nb;
	*nblks = SPAD2CPU64_LV(&anode->x[off].end_off) - lbn;
	if (flags == BMAP_EXTEND) {
		if (__unlikely(SPAD2CPU64_LV(&anode->x[off].end_off) != lbn + 1) || __unlikely(anode->valid_extents != off + 1)) {
			CPU2SPAD64_LV(&anode->x[off].end_off, lbn + 1);
			anode->valid_extents = off + 1;
			DO_ANODE_DIRTY;
		}
	}
	VFS$PUT_BUFFER(anode);
	return 0;
#undef sfs
}

static int SPAD_ADD_EXTENT(SPADFNODE *f, __d_off blk, __d_off n_blks)
{
#define sfs ((SPADFS *)f->fs)
	struct anode *anode;
	__d_off ano;
	int depth_now, depth_total;
	int direct;
	__d_off ano_l;
	__d_off end_off;
	BUFFER *b;
	/*__debug_printf("add extent (%Lx,%Lx)\n", blk, n_blks);*/
	if (__likely(!f->blk1_n)) {
		unsigned mdb;
		f->blk1 = blk;
		mdb = MAX_DIRECT_BLKS(sfs->sectors_per_block);
		if (__unlikely(n_blks > mdb)) {
			f->blk1_n = mdb;
			blk += mdb;
			n_blks -= mdb;
			goto again1;
		}
		f->blk1_n = n_blks;
		return 0;
	}
	if (!f->blk2_n) {
		unsigned mdb;
		again1:
		f->blk2 = blk;
		mdb = MAX_DIRECT_BLKS(sfs->sectors_per_block);
		if (__unlikely(n_blks > mdb)) {
			f->blk2_n = mdb;
			blk += mdb;
			n_blks -= mdb;
			goto again2;
		}
		f->blk2_n = n_blks;
		return 0;
	}
	if (!f->root) {
		again2:
		anode = ALLOC_ANODE(f, alloc_meta_hint(sfs, f), &b, &f->root);
		if (__unlikely(__IS_ERR(anode))) return __PTR_ERR(anode);
		strncpy(ANODE_ROOT_NAME(anode), f->name, ANODE_ROOT_NAME_LEN);
		anode->flags |= ANODE_ROOT;
		anode->valid_extents = 3;
		CPU2SPAD64_LV(&anode->start_off, 0);
		CPU2SPAD64_LV(&anode->x[0].blk, f->blk1);
		CPU2SPAD64_LV(&anode->x[0].end_off, f->blk1_n);
		CPU2SPAD64_LV(&anode->x[1].blk, f->blk2);
		CPU2SPAD64_LV(&anode->x[1].end_off, f->blk1_n + f->blk2_n);
		CPU2SPAD64_LV(&anode->x[2].blk, blk);
		CPU2SPAD64_LV(&anode->x[2].end_off, SPAD2CPU64_LV(&anode->x[1].end_off) + n_blks);
		DO_ANODE_DIRTY;
		VFS$PUT_BUFFER(anode);
		return 0;
	}
	ano = f->root;
	ano_l = -1;
	depth_now = depth_total = 0;
	subnode:
	anode = MAP_ANODE(sfs, ano, &b, BMAP_MAP);
	if (__unlikely(__IS_ERR(anode))) return __PTR_ERR(anode);
	direct = find_direct(depth_now, depth_total);
	if (__likely(anode->valid_extents < direct)) {
		CPU2SPAD64_LV(&anode->x[anode->valid_extents].blk, blk);
		CPU2SPAD64_LV(&anode->x[anode->valid_extents].end_off, SPAD2CPU64_LV(&anode->x[anode->valid_extents - 1].end_off) + n_blks);
		anode->valid_extents++;
		DO_ANODE_DIRTY;
		VFS$PUT_BUFFER(anode);
		return 0;
	}
	if (anode->valid_extents < ANODE_N_EXTENTS) ano_l = ano;
	if (anode->valid_extents > direct) {
		ano = SPAD2CPU64_LV(&anode->x[anode->valid_extents - 1].blk);
		update_depth(&depth_now, &depth_total, anode->valid_extents - 1);
		VFS$PUT_BUFFER(anode);
		goto subnode;
	}
	if (__unlikely(ano_l == -1)) {
		SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "ANODE %012"__d_off_format"X IS FULL", f->root);
		VFS$PUT_BUFFER(anode);
		return -EFSERROR;
	}
	end_off = SPAD2CPU64_LV(&anode->x[anode->valid_extents - 1].end_off);
	VFS$PUT_BUFFER(anode);
	anode = ALLOC_ANODE(f, ano_l, &b, &ano);
	if (__unlikely(__IS_ERR(anode))) return __PTR_ERR(anode);
	anode->valid_extents = 1;
	CPU2SPAD64_LV(&anode->start_off, end_off);
	CPU2SPAD64_LV(&anode->x[0].blk, blk);
	CPU2SPAD64_LV(&anode->x[0].end_off, end_off + n_blks);
	DO_ANODE_DIRTY;
	VFS$PUT_BUFFER(anode);
	anode = MAP_ANODE(sfs, ano_l, &b, BMAP_MAP);
	if (__unlikely(__IS_ERR(anode))) return __PTR_ERR(anode);
	if (__unlikely(anode->valid_extents == ANODE_N_EXTENTS)) {
		SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "ANODE %012"__d_off_format"X FILLED UNDER US", f->root);
		VFS$PUT_BUFFER(anode);
		return -EFSERROR;
	}
	CPU2SPAD64_LV(&anode->x[anode->valid_extents].blk, ano);
	CPU2SPAD64_LV(&anode->x[anode->valid_extents].end_off, -1);
	CPU2SPAD64_LV(&anode->x[anode->valid_extents - 1].end_off, end_off);
	anode->valid_extents++;
	DO_ANODE_DIRTY;
	VFS$PUT_BUFFER(anode);
	return 0;
#undef sfs
}

static int SPAD_EXTEND_LAST_EXTENT(SPADFNODE *f, __d_off n_blks)
{
#define sfs ((SPADFS *)f->fs)
	struct anode *anode;
	__d_off ano;
	int depth_now, depth_total;
	int direct;
	BUFFER *b;
	/*__debug_printf("extend last (%Lx)\n", n_blks);*/
	if (!f->blk2_n) {
		unsigned mdb = MAX_DIRECT_BLKS(sfs->sectors_per_block);
		if (f->blk1_n + n_blks > mdb) {
			n_blks -= mdb - f->blk1_n;
			f->blk1_n = mdb;
			return SPAD_ADD_EXTENT(f, f->blk1 + mdb, n_blks);
		}
		f->blk1_n += n_blks;
		return 0;
	}
	if (!f->root) {
		unsigned mdb = MAX_DIRECT_BLKS(sfs->sectors_per_block);
		if (f->blk2_n + n_blks > mdb) {
			n_blks -= mdb - f->blk2_n;
			f->blk2_n = mdb;
			return SPAD_ADD_EXTENT(f, f->blk2 + mdb, n_blks);
		}
		f->blk2_n += n_blks;
		return 0;
	}
	ano = f->root;
	depth_now = depth_total = 0;
	subnode:
	anode = MAP_ANODE(sfs, ano, &b, BMAP_MAP);
	if (__unlikely(__IS_ERR(anode))) return __PTR_ERR(anode);
	direct = find_direct(depth_now, depth_total);
	if (anode->valid_extents <= direct) {
		CPU2SPAD64_LV(&anode->x[anode->valid_extents - 1].end_off, SPAD2CPU64_LV(&anode->x[anode->valid_extents - 1].end_off) + n_blks);
		DO_ANODE_DIRTY;
		VFS$PUT_BUFFER(anode);
		return 0;
	}
	ano = SPAD2CPU64_LV(&anode->x[anode->valid_extents - 1].blk);
	update_depth(&depth_now, &depth_total, anode->valid_extents - 1);
	VFS$PUT_BUFFER(anode);
	goto subnode;
#undef sfs
}

int SPAD_EXTEND(SPADFNODE *f, _u_off_t size)
{
#define sfs ((SPADFS *)f->fs)
	struct alloc al;
	__d_off blks, newblks;
	__d_off blk;
	int flags;
	int r;
	f->run_length = 0;
	again:
	blks = size2blks(sfs, f->disk_size);
	newblks = size2blks(sfs, size) - blks;
	flags = size > sfs->cluster_threshold ? ALLOC_BIG_FILE : ALLOC_SMALL_FILE;
	if (!newblks) goto ret;
	if (__unlikely((__s64)newblks < 0)) KERNEL$SUICIDE("SPAD_EXTEND: %s: TRUNCATING FROM %012"__64_format"X TO %012"__64_format"X", VFS$FNODE_DESCRIPTION((FNODE *)f), (__u64)f->disk_size, (__u64)size);
	if (__unlikely(newblks > 0x7fffffff)) {
		r = SPAD_EXTEND(f, f->disk_size + ((__u64)1 << 32));
		if (__unlikely(r)) return r;
		goto again;
	}
	if (blks) {
		r = SYNC_BMAP(f, blks - 1, &blk, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, BMAP_EXTEND, (void *)&KERNEL$LIST_END, "SPAD_EXTEND");
		if (__unlikely(r)) return r;
		blk++;
	} else {
		blk = 0;
	}
	if (blks && (flags == ALLOC_SMALL_FILE || f->disk_size > sfs->cluster_threshold)) {
		new_run:
		al.n_sectors =
#ifndef TEST_ALLOCATOR
				newblks;
#else
				1;
#endif
		al.flags = flags | ALLOC_PARTIAL_AT_GOAL;
		alloc_c:
		al.sector = blk;
		r = SPAD_ALLOC_BLOCKS(f, &al);
		if (__unlikely(r)) {
			no_alloc:
			SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "OUT OF SPACE FOR FILE");
			return r;
		}
	/*__debug_printf("alloc: hint %Lx, result %Lx, hintlen %Lx, resultlen %Lx\n", blk, al.sector, newblks, (__u64)al.n_sectors);*/
		add_extent:
		if (__likely(al.sector == blk)
#ifdef TEST_ALLOCATOR
			&& al.sector & 4
#endif
				) r = SPAD_EXTEND_LAST_EXTENT(f, al.n_sectors);
		else r = SPAD_ADD_EXTENT(f, al.sector, al.n_sectors);
		if (__unlikely(r)) return r;
		newblks -= al.n_sectors;
	} else {
		if (blks) {
			new_extent:
			if ((unsigned)newblks & sfs->sectors_per_cluster_mask) {
				flags = 0;
				al.flags = ALLOC_SMALL_FILE | ALLOC_PARTIAL_AT_GOAL;
				al.n_sectors = (unsigned)newblks & sfs->sectors_per_cluster_mask;
				goto alloc_c;
			}
		} else {
			f->blk1_n = 0;
			f->blk2_n = 0;
			f->root = 0;
		}
		if (__unlikely(flags & ALLOC_BIG_FILE)) {
			al.sector = alloc_big_hint(sfs, f);
		} else {
			if (__likely(f->flags & FNODE_UNCOMMITTED_PREALLOC_HINT)) {
				if (__likely(newblks <= sfs->prealloc_length)) {
					/*__debug_printf("use prealloc: %d\n", (int)newblks);*/
					al.sector = sfs->prealloc_sector;
					sfs->prealloc_sector += newblks;
					al.n_sectors = newblks;
					sfs->prealloc_length -= (unsigned)newblks;
					SPAD_CHARGE_QUOTA(f, (unsigned)al.n_sectors);
					goto add_extent;
				}
			}
			al.sector = alloc_small_hint(sfs, f);
		}
		al.n_sectors =
#ifndef TEST_ALLOCATOR
				newblks;
#else
				1;
#endif
		al.flags = flags;
		r = SPAD_ALLOC_BLOCKS(f, &al);
		if (__unlikely(r)) goto no_alloc;
		if (__unlikely(al.flags & ALLOC_NEW_GROUP_HINT)) set_new_hint((SPADFNODE *)f->parent, &al);
		goto add_extent;
	}
	if (__unlikely(newblks != 0)) {
		blk = al.sector + al.n_sectors;
		if (__unlikely(!flags)) {
			flags = ALLOC_BIG_FILE;
			goto new_extent;
		} else {
			goto new_run;
		}
	}
	ret:
	f->disk_size = size;
	return 0;
#undef sfs
}


int SPAD_TRUNCATE(SPADFNODE *f, _u_off_t size)
{
#define sfs ((SPADFS *)f->fs)
	int r;
	__d_off blks, oldblks;
	__d_off blk, bback, ano;
	f->run_length = 0;
	blks = size2blks(sfs, f->disk_size);
	oldblks = blks - size2blks(sfs, size);
	if (!oldblks) goto ret;
	if (__unlikely((__s64)oldblks < 0)) KERNEL$SUICIDE("SPAD_TRUNCATE: %s: EXTENDING FROM %012"__64_format"X TO %012"__64_format"X", VFS$FNODE_DESCRIPTION((FNODE *)f), (__u64)f->disk_size, (__u64)size);
	again:
	r = SYNC_BMAP(f, blks - 1, &blk, (void *)&KERNEL$LIST_END, &bback, BMAP_MAP, &ano, "SPAD_TRUNCATE");
	if (__unlikely(r)) goto ret_r;
	/*__debug_printf("BLKS: %d, BLK: %d, BBACK: %d\n", (int)blks, (int)blk, (int)bback);*/
	blk++;
	bback++;
	if (oldblks >= bback && __unlikely(ano != 0)) {
		VFS$CLEAN_BUFFERS_SYNC((FS *)sfs, ano, sfs->sectors_per_block);
		r = SPAD_FREE_BLOCKS(f, ano, sfs->sectors_per_block);
		if (__unlikely(r)) goto ret_r;
	}
	if (oldblks < bback) bback = oldblks;
	r = SPAD_FREE_BLOCKS(f, blk - bback, bback);
	if (__unlikely(r)) goto ret_r;
	blks -= bback;
	if ((oldblks -= bback)) goto again;
	ret:
	r = 0;
	ret_r:
	f->disk_size = size;
	return r;
#undef sfs
}

int SPAD_RESURRECT(SPADFNODE *f, _u_off_t size)
{
#define sfs ((SPADFS *)f->fs)
	__d_off blks, newblks;
	__d_off blk;
	__d_off bback, bfwd;
	__d_off ano;
	int r;
	blks = size2blks(sfs, f->disk_size);
	newblks = size2blks(sfs, size) - blks;
	if (!newblks) goto ret;
	if (__unlikely((__s64)newblks < 0)) KERNEL$SUICIDE("SPAD_RESURRECT: %s: TRUNCATING FROM %012"__64_format"X TO %012"__64_format"X", VFS$FNODE_DESCRIPTION((FNODE *)f), (__u64)f->disk_size, (__u64)size);
	again:
	r = SYNC_BMAP(f, blks, &blk, &bfwd, &bback, BMAP_MAP, &ano, "SPAD_RESURRECT");
	if (__unlikely(r)) return r;
	if (!bback && ano) {
		r = SPAD_RESURRECT_BLOCKS(f, ano, sfs->sectors_per_block);
		if (__unlikely(r)) return r;
	}
	if (bfwd > newblks) bfwd = newblks;
	SPAD_RESURRECT_BLOCKS(f, blk, bfwd);
	blks += bfwd;
	if ((newblks -= bfwd)) goto again;
	ret:
	f->disk_size = size;
	return 0;
#undef sfs
}

int SPAD_SYNC_BMAP(FNODE *f, __d_off off, int try)
{
	int r;
	__d_off blk, fwd, back;
	r = SYNC_BMAP((SPADFNODE *)f, off, &blk, &fwd, &back, __likely(!try) ? BMAP_MAP : BMAP_TRY, (void *)&KERNEL$LIST_END, __likely(!try) ? "SPAD_SYNC_BMAP" : "SPAD_SYNC_BMAP(TRY)");
	if (__unlikely(r)) return r;
	f->disk_blk = blk - back;
	f->file_blk = off - back;
	f->run_length = back + fwd;
	if (is_bmap_invalid(f)) {
		SPAD_INVALID_BMAP_MSG((SPADFNODE *)f);
		return -EFSERROR;
	}
	/*__debug_printf("OFF=%012LX BLK=%012LX BACK=%012LX FWD=%012LX\n", off, blk, back, fwd);*/
	return 0;
}

void SPAD_INIT_LAST_PAGE(FNODE *f, char *ptr, unsigned len)
{
#define sf	((SPADFNODE *)f)
#define sfs	((SPADFS *)sf->fs)
	SPADFNODE *pf;
	__u32 flags;

	/* Linux doesn't like data in the last page after end, for example
	   gcc will read beyond file end in that case */
	if (__unlikely(sfs->flags & FS_MORE_SAME_NAMES)) goto zero_end;
	if (__unlikely(sf->root != 0)) goto zero_end;

	flags = (sfs->sectors_per_block_bits << __BSF_CONST(FILE_END_SECTORS_PER_BLOCK_BITS)) | (sfs->sectors_per_cluster_bits << __BSF_CONST(FILE_END_SECTORS_PER_CLUSTER_BITS));
	if (__likely(!sf->blk2_n)) {
		struct file_end *fe;
		if (__unlikely(len < sizeof(struct file_end))) goto zero_end;
		fe = (struct file_end *)(ptr + len) - 1;
		CPU2SPAD32_LV(&fe->magic, FILE_END_MAGIC1 | flags);
		CPU2SPAD32_LV(&fe->size, (__u32)sf->size);
		len -= sizeof(struct file_end);
	} else {
		struct file_end_2 *fe;
		if (__unlikely(len < sizeof(struct file_end_2))) goto zero_end;
		fe = (struct file_end_2 *)(ptr + len) - 1;
		CPU2SPAD32_LV(&fe->magic, FILE_END_MAGIC2 | flags);
		CPU2SPAD32_LV(&fe->size, (__u32)sf->size);
		fe->run10 = MAKE_PART_0(sf->blk1);
		fe->run11 = MAKE_PART_1(sf->blk1);
		CPU2SPAD16_LV(&fe->run1n, sf->blk1_n);
		len -= sizeof(struct file_end_2);
	}
	if (__unlikely(sf->namelen > len)) {
		memcpy(ptr, sf->name, len);
		return;
	}
	memcpy(ptr + len - sf->namelen, sf->name, sf->namelen);
	len -= sf->namelen;
	pf = (SPADFNODE *)sf->parent;
	if (__unlikely(pf->namelen + 1 > len)) {
		goto zero_end;
	}
	memcpy(ptr + len - pf->namelen - 1, pf->name, pf->namelen);
	ptr[len - 1] = '/';
	len -= pf->namelen + 1;

	zero_end:
	memset(ptr, 0, len);
#undef sfs
#undef sf
}

int SPAD_TEST_CONTIG_ALLOC(FNODE *sf)
{
#define sfs	((SPADFS *)sf->fs)
	if (__unlikely(sf->size > sfs->cluster_threshold)) return 0;
	if (__unlikely(sfs->prealloc_length >= 0x80000000U)) return 0;
	sfs->prealloc_length += (((unsigned)sf->size + sfs->blocksize - 1) & ~(sfs->blocksize - 1)) >> SSIZE_BITS;
	return 1;
#undef sfs
}

void SPAD_DO_PREALLOC_(SPADFNODE *sf)
{
	/* non-zero sfs->prealloc_length is required --- already checked in SPAD_DO_PREALLOC */
#define sfs	((SPADFS *)sf->fs)
	struct alloc al;
	al.n_sectors = sfs->prealloc_length;
	sfs->prealloc_length = 0;
	al.flags = ALLOC_SMALL_FILE | ALLOC_NO_CHARGE;
	al.sector = alloc_small_hint(sfs, sf);
	if (__likely(!(SPAD_ALLOC_BLOCKS(sf, &al)))) {
/* it may happen that someone frees a file for which we preallocated
   ... and in that case we need to COMMIT to reclaim space */
		sfs->flags |= FS_COMMIT_FREES_DATA;
		sfs->prealloc_sector = al.sector;
		sfs->prealloc_length = al.n_sectors;
		if (__unlikely(al.flags & ALLOC_NEW_GROUP_HINT)) set_new_hint((SPADFNODE *)sf->parent, &al);
		/*__debug_printf("done prealloc: %d\n", sfs->prealloc_length);*/
	}
#undef sfs
}

void SPAD_DISCARD_PREALLOC_(SPADFS *sfs)
{
	/*__debug_printf("discarding preallloc\n");*/
	/* non-zero sfs->prealloc_length is required --- already checked in SPAD_DISCARD_PREALLOC */
	SPAD_FREE_BLOCKS((SPADFNODE *)sfs->root, sfs->prealloc_sector, sfs->prealloc_length);
	sfs->prealloc_length = 0;
}

__COLD_ATTR__ void SPAD_INVALID_BMAP_MSG(SPADFNODE *f)
{
	SPADFS_ERROR((SPADFS *)f->fs, TXFLAGS_FS_ERROR, "%s: %s: %012"__d_off_format"X, %012"__d_off_format"X, %012"__d_off_format"X", VFS$FNODE_DESCRIPTION((FNODE *)f), ((unsigned)f->disk_blk | (unsigned)f->file_blk | (unsigned)f->run_length) & (((SPADFS *)f->fs)->sectors_per_block - 1) ? "UNALIGNED BMAP" : "BMAP OUT OF FILESYSTEM", f->disk_blk, f->file_blk, f->run_length);
	f->disk_blk = 0;
	f->file_blk = 0;
	f->run_length = 0;
}

__COLD_ATTR__ static void ANODE_ERROR_MSG(struct anode *anode, BUFFER *b, unsigned flags, char *msg)
{
	SPADFS_ERROR((SPADFS *)b->fs, flags, msg, b->blk | (((unsigned long)anode >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1), (unsigned)anode->valid_extents);
}

__COLD_ATTR__ static void BMAP_OUT_OF_RANGE_MSG(SPADPAGEINRQ *RQ, struct extent *e)
{
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "SPAD_BMAP_AST: OUT OF RANGE !(%012"__d_off_format"X <= %012"__d_off_format"X < %012"__d_off_format"X), ANODE (%012"__d_off_format"X -> %012"__d_off_format"X)", SPAD2CPU64_LV(&e[-1].end_off), RQ->off, SPAD2CPU64_LV(&e[0].end_off), ((SPADFNODE *)RQ->fnode)->root, RQ->dirptr);
}

__COLD_ATTR__ static void BMAP_NON_MONOTONIC_MSG(int off, SPADPAGEINRQ *RQ)
{
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "SPAD_BMAP_AST: NON-MONOTONIC ANODE (%012"__d_off_format"X -> %012"__d_off_format"X), ENTRY %d", ((SPADFNODE *)RQ->fnode)->root, RQ->dirptr, off);
}
