#include <SPAD/VFS.H>
#include <SPAD/HASH.H>
#include <TIME.H>

#include "SPADFS.H"
#include "SFSFNODE.H"

static AST_STUB SPAD_LOOKUP_AST;
static AST_STUB SPAD_READDIR_AST;

void SPAD_FNODE_CTOR(struct __slhead *fs_, void *fnode_)
{
	SPADFNODE *fnode = fnode_;
	FS *fs = GET_STRUCT(fs_, FS, fnodes);
	VFS$FNODE_CTOR(&fs->fnodes, fnode);
}

void SPAD_PAGEINRQ_CTOR(struct __slhead *fs_, void *pageinrq_)
{
	SPADPAGEINRQ *pageinrq = pageinrq_;
	FS *fs = GET_STRUCT(fs_, FS, pageinrq);
	VFS$PAGEINRQ_CTOR(&fs->pageinrq, pageinrq);
}

__finline__ void SPAD_GET_FNODE(SPADFNODE *f, struct fnode *fd, __d_off addr, unsigned pos)
{
	f->fnode_address = addr;
	f->fnode_off = pos;
	f->ctime = __time32_2_time_t(SPAD2CPU32_LV(&fd->ctime));
	f->mtime = __time32_2_time_t(SPAD2CPU32_LV(&fd->mtime));
	f->ea_size = (SPAD2CPU16_LV(&fd->next) & FNODE_NEXT_SIZE) - FNODE_EA_POS(fd->namelen);
	if (__unlikely(f->ea_size)) {
		memcpy(f->ea, (char *)fd + FNODE_EA_POS(fd->namelen), f->ea_size);
	}
	FNODE_OUT_OF_WANTFREE((FNODE *)f);
	if (__likely(!(fd->flags & FNODE_FLAGS_DIR))) {
		f->flags = FNODE_FILE;
		/*f->run_length = 0;*/
		f->blk1 = MAKE_D_OFF(fd->run10, fd->run11);
		f->blk1_n = SPAD2CPU16_LV(&fd->run1n);
		f->disk_blk = f->blk1;
		f->file_blk = 0;
		f->run_length = f->blk1_n;
		f->blk2 = MAKE_D_OFF(fd->run20, fd->run21);
		f->blk2_n = SPAD2CPU16_LV(&fd->run2n);
		if (__unlikely(f->blk2 == f->blk1 + f->blk1_n))
			f->run_length += f->blk2_n;
		f->root = MAKE_D_OFF(fd->anode0, fd->anode1);
		if (__likely(f->blk1_n)) {
			if (is_bmap_invalid_nozerotest(f)) SPAD_INVALID_BMAP_MSG(f);
		}
		if (CC_VALID((SPADFS *)f->fs, &fd->cc, &fd->txc)) {
			f->size = SPAD2CPU64_LV(&fd->size0);
			/*__debug_printf("load0: size %Lx, other %Lx\n", f->size, SPAD2CPU64_LV(&fd->size1));*/
		} else {
			f->size = SPAD2CPU64_LV(&fd->size1);
			/*__debug_printf("load1: size %Lx, other %Lx\n", f->size, SPAD2CPU64_LV(&fd->size0));*/
		}
		/*__debug_printf("FNODE: %x %x - %x, %x %x - %x, %x %x\n", SPAD2CPU32_LV(&fd->run10), SPAD2CPU16_LV(&fd->run11), SPAD2CPU16_LV(&fd->run1n), SPAD2CPU32_LV(&fd->run20), SPAD2CPU16_LV(&fd->run21), SPAD2CPU16_LV(&fd->run2n), SPAD2CPU32_LV(&fd->anode0), SPAD2CPU16_LV(&fd->anode1));*/
	} else {
		if (!(f->flags & FNODE_DIRECTORY)) {
			VFS_INIT_DIR((FNODE *)f);
		}
		f->flags = FNODE_DIRECTORY;
		f->size = 0;
		f->blk1_n = SPAD2CPU16_LV(&fd->run1n);
		f->blk2_n = SPAD2CPU16_LV(&fd->run2n);
		if (CC_VALID((SPADFS *)f->fs, &fd->cc, &fd->txc))
			f->root = MAKE_D_OFF(fd->run10, fd->run11);
		else
			f->root = MAKE_D_OFF(fd->run20, fd->run21);
	}
	f->disk_size = f->size;
}

void SPAD_LOOKUP(PAGEINRQ *p)
{
	SPADPAGEINRQ *pgin = (SPADPAGEINRQ *)p;
	pgin->fn = SPAD_LOOKUP_AST;
	pgin->hash = name_hash(pgin->new_fnode->name);
	pgin->dirptr = ((SPADFNODE *)pgin->fnode)->root;
	pgin->c[1] = 0;
	CALL_AST(pgin);
}

static DECL_AST(SPAD_LOOKUP_AST, SPL_FS, SPADPAGEINRQ)
{
	BUFFER *buf;
	IORQ *caller;
	long error;
	struct fnode_block *fnode_block;

	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_FS), RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status < 0)) {
		error = RQ->status;
		goto err;
	}

	if (0) {
		subdir:
		if (0) {
			subnode:
			if (__unlikely(SPAD_STOP_CYCLES(RQ->dirptr, &RQ->c))) goto cycle;
		}
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
	}
	fnode_block = VFS$GET_BUFFER(RQ->fs, RQ->dirptr, 1, &buf, RQ->tag.proc);
	if (__likely(fnode_block != NULL)) {
		int d = SPAD_CHECK_DFNODE(fnode_block, buf);
		if (__likely(!d)) {
			unsigned size;
			struct fnode *fnode;
			next_block:
			fnode = fnode_block->fnodes;
			next_fnode:
			size = SPAD2CPU16_LV(&fnode->next) & FNODE_NEXT_SIZE;
			VALIDATE_FNODE((SPADFS *)RQ->fs, fnode_block, fnode, size, ok, skip, bad_fnode);
			ok:
			if (fnode->namelen == RQ->new_fnode->namelen) {
				int was_dir;
				SPADFNODE *fn;
				int i;
				if (__likely(!(RQ->fs->flags & FS_MORE_SAME_NAMES))) {
					for (i = fnode->namelen - 1; i >= 0; i--)
						if (FNODE_NAME(fnode)[i] != RQ->new_fnode->name[i]) goto no_match;
				} else {
					for (i = fnode->namelen - 1; i >= 0; i--)
						if (__upcasechr(FNODE_NAME(fnode)[i]) != __upcasechr(RQ->new_fnode->name[i])) goto no_match;
				}
				if (__unlikely(fnode->flags & FNODE_FLAGS_HARDLINK)) {
						/* hardlinks not supported yet */
					VFS$PUT_BUFFER(fnode_block);
					error = -EINVAL;
					goto err;
				}
				fn = (SPADFNODE *)RQ->new_fnode;
				if (fn->flags & FNODE_DIRECTORY) {
					if (__unlikely((fnode->flags & FNODE_FLAGS_DIR) == 0)) {
						VFS$PUT_BUFFER(fnode_block);
						error = -ENOTDIR;
						goto err;
					}
					was_dir = 1;
				} else was_dir = 0;
				if (__likely(RQ->fnode != NULL)) {
					RQ->fnode->readers--;
					WQ_WAKE_ALL(&RQ->fnode->wait);
#if __DEBUG >= 1
					if (__unlikely(RQ->fnode->readers < 0))
						KERNEL$SUICIDE("SPAD_LOOKUP_AST: %s: LOCK UNDERRUN: %d", VFS$FNODE_DESCRIPTION(RQ->fnode), RQ->fnode->readers);
#endif
				}
				SPAD_GET_FNODE(fn, fnode, (RQ->dirptr & ~(__d_off)__SECTORS_PER_PAGE_CLUSTER_MINUS_1) + (((unsigned long)fnode_block >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1), (char *)fnode - (char *)fnode_block);
				WQ_WAKE_ALL_PL(&fn->wait);
				VFS$PUT_BUFFER(fnode_block);

				if (was_dir) {
					RQ->new_fnode = LIST_STRUCT(fn->u.d.clean.next, FNODE, dirent_entry);
					RQ->fnode = (FNODE *)fn;
					RQ->fnode->readers++;
					RQ->hash = name_hash(RQ->new_fnode->name);
					RQ->dirptr = fn->root;
					RQ->c[1] = 0;
					goto subdir;
				}
				caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
				RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
			}
			skip:
			no_match:
			fnode = (struct fnode *)((char *)fnode + size);
			if (__likely(((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)) != 0)) goto next_fnode;
			if (!(fnode_block->flags & FNODE_BLOCK_LAST)) {
				int x;
				fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
				x = SPAD_CHECK_DFNODE(fnode_block, buf);
				if (__likely(!x)) goto next_block;
				if (__likely(x == 1)) {
					VFS$PUT_BUFFER(fnode_block);
					goto read_it;
				}
				if (x > 0) goto bad_fnode;
				goto exit_err;
			}
			if (__unlikely(CC_VALID((SPADFS *)RQ->fs, &fnode_block->cc, &fnode_block->txc))) {
				RQ->dirptr = MAKE_D_OFF(fnode_block->next0, fnode_block->next1);
				VFS$PUT_BUFFER(fnode_block);
				goto subnode;
			}
			VFS$PUT_BUFFER(fnode_block);
			error = -ENOENT;
			goto err;
		} else if (__likely(d > 1)) {
			struct dnode_page_entry *e;
			hash_t hpos;
			hpos = RQ->hash & ((1 << ((SPADFS *)RQ->fs)->dnode_hash_bits) - 1);
			RQ->hash >>= ((SPADFS *)RQ->fs)->dnode_hash_bits;
			e = (struct dnode_page_entry *)((char *)fnode_block + d + hpos * DNODE_PAGE_ENTRY_SIZE);

			RQ->dirptr = MAKE_D_OFF3(e->b0, e->b1, e->b2);
			VFS$PUT_BUFFER(fnode_block);
			if (__likely(RQ->dirptr != 0)) goto subnode;
			error = -ENOENT;
			goto err;
		} else if (__likely(d == 1)) {
			VFS$PUT_BUFFER(fnode_block);
			goto read_it;
		} else goto exit_err;
	} else {
		read_it:
			/* do not prefetch in deeper subtree */
		VFS$READ_BUFFER(RQ->dirptr, (PAGEINRQ *)RQ, __likely(!RQ->c[1]) ? RQ->fs->buffer_readahead : 0);
		RETURN;
	}

	err:
	if (__unlikely(!RQ->fnode)) {	/* this happens only during root mount */
		caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
		caller->status = error;
		RETURN_AST(caller);
	}
	if (__unlikely(error != -ENOENT) || __likely(!(RQ->wr & (O_CREAT | _O_RENAME)))) {
		/*__debug_printf("0: %d\n", error);*/
		caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, error == -ENOENT);
		caller->status = error;
		RETURN_AST(caller);
	}
	caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, 1);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);

	exit_err:
	VFS$PUT_BUFFER(fnode_block);
	error = -EFSERROR;
	goto err;

	bad_fnode:
	VFS$PUT_BUFFER(fnode_block);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012" __d_off_format "X (LOOKUP)", RQ->dirptr);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "BAD FNODE ON BLOCK %012" __d_off_format "X (LOOKUP %012" __d_off_format "X/%04X:%s/%s)", RQ->dirptr, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_address : 0, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_off : 0, RQ->fnode->name, RQ->new_fnode->name);
	error = -EFSERROR;
	goto err;

	cycle:
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "CYCLE DETECTED ON BLOCK %012" __d_off_format "X (LOOKUP)", RQ->dirptr);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "CYCLE DETECTED ON BLOCK %012" __d_off_format "X (LOOKUP %012" __d_off_format "X/%04X:%s/%s)", RQ->dirptr, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_address : 0, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_off : 0, RQ->fnode ? RQ->fnode->name : ":", RQ->new_fnode->name);
	error = -EFSERROR;
	goto err;
}

void SPAD_INIT_READDIR_COOKIE(struct readdir_buffer *buf, FNODE *f)
{
	SPADREADDIRCOOKIE *c = (SPADREADDIRCOOKIE *)buf->cookie;
	buf->cookie_size = sizeof(SPADREADDIRCOOKIE);
	c->hash = 0;
	c->chain = 0;
	c->fnode_offset = 0;	/* !!! FIXME: remove, readdir must do all-or-nothing */
}

void SPAD_READDIR(PAGEINRQ *readdir, struct readdir_buffer *buf)
{
	SPADPAGEINRQ *r = (SPADPAGEINRQ *)readdir;
	r->fn = SPAD_READDIR_AST;
	if (buf) {
#if __DEBUG >= 1
		if (__unlikely(buf->cookie_size != sizeof(SPADREADDIRCOOKIE)))
			KERNEL$SUICIDE("SPAD_READDIR: %s: COOKIE SIZE %d", VFS$FNODE_DESCRIPTION(r->fnode), buf->cookie_size);
#endif
		memcpy(&r->cookie, buf->cookie, sizeof(SPADREADDIRCOOKIE));
	} else {
		r->cookie.hash = 0;
		r->cookie.chain = 0;
		r->cookie.fnode_offset = 0;
	}
	r->hash = r->cookie.hash;
	r->dirptr = ((SPADFNODE *)r->fnode)->root;
	r->c[1] = 0;
	r->hash_bits = 0;
	r->next_hash = 0;
	r->chain_skip = r->cookie.chain;
	CALL_AST(r);
}

/*
	1. find leaf page according to cookie
	2. map output buffer ....
		fail => buffer was empty => post pageinrq
				   not empty => post success
	3. copy entries (exclude FNODE_NEGATIVE) to output
		not all entries copied
			=> buffer was not empty
				=> post success (discard new entries)
			=> buffer was empty
				=> post EMSGSIZE (caller will extend buffer)
		some entries copied => set buffer nonempty flag
	4. increase cookie in pageinrq & put new cookie into buffer
	5. unmap output buffer
	6. go to point 1. (or 2.)
*/

static DECL_AST(SPAD_READDIR_AST, SPL_FS, SPADPAGEINRQ)
{
#define COOKIE(rb)	((SPADREADDIRCOOKIE *)(void *)&(rb)->cookie)
	BUFFER *buf;
	IORQ *caller;
	long error;
	struct fnode_block *fnode_block;

	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_FS), RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status < 0)) {
		error = RQ->status;
		goto err;
	}

	if (0) {
		subnode:
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
		subnode0:
		if (__unlikely(SPAD_STOP_CYCLES(RQ->dirptr, &RQ->c))) goto cycle;
	}
	fnode_block = VFS$GET_BUFFER(RQ->fs, RQ->dirptr, 1, &buf, RQ->tag.proc);
	if (__likely(fnode_block != NULL)) {
		void *map;
		unsigned maplen;
		void (*unmap)(void *ptr);
		int d = SPAD_CHECK_DFNODE(fnode_block, buf);
		if (__likely(!d)) {
			unsigned size;
			struct fnode *fnode;
			unsigned current_offset;
			map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, RQ->fs, &maplen, &unmap);
#define rb	((struct readdir_buffer *)map)
			if (__unlikely(!map)) {
				no_map_ptr:
				VFS$PUT_BUFFER(fnode_block);
				goto pagefault;
			}
			if (__unlikely(map == (void *)1)) {
				no_map_ptr_ret:
				error = 0;
				goto err_put;
			}
			if (__unlikely(__IS_ERR(map))) {
				map_error:
				error = __PTR_ERR(map);
				goto err_put;
			}
			current_offset = 0;
			next_block:
			if (__unlikely(RQ->chain_skip != 0)) {
				goto skip_block;
			}
			fnode = fnode_block->fnodes;
			current_offset += SIZEOF_FNODE_BLOCK;
			next_fnode:
			size = SPAD2CPU16_LV(&fnode->next) & FNODE_NEXT_SIZE;
			VALIDATE_FNODE((SPADFS *)RQ->fs, fnode_block, fnode, size, ok, skip, bad_fnode_unmap);
			ok:
			if (__unlikely(current_offset <= RQ->cookie.fnode_offset)) goto skip;
			error = VFS$DO_READDIR_NAME((PAGEINRQ *)RQ, map, maplen, FNODE_NAME(fnode), fnode->namelen, fnode->flags & FNODE_FLAGS_DIR ? DT_DIR : DT_REG);
			if (__unlikely(error != 0)) {
				err_put_unmap_readdir:
				VFS$UNMAP_READDIR(map, unmap);
				goto err_put;

				bad_fnode_unmap:
				VFS$UNMAP_READDIR(map, unmap);
				goto bad_fnode;
			}
			COOKIE(rb)->fnode_offset = RQ->cookie.fnode_offset = current_offset;
			if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS)) {
				if (__unlikely(RQ->tag.proc == &KERNEL$PROC_KERNEL) || __unlikely(RQ->wr < 0)) {
					VFS$UNMAP_READDIR(map, unmap);
					VFS$PUT_BUFFER(fnode_block);
					WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
					RETURN;
				} else {
					error = -EINTR;
					goto err_put_unmap_readdir;
				}
			}
			skip:
			fnode = (struct fnode *)((char *)fnode + size);
			current_offset += size;
			if (__likely(((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)) != 0)) goto next_fnode;
			skip_block:
			if (!(fnode_block->flags & FNODE_BLOCK_LAST)) {
				int x;
				fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
				x = SPAD_CHECK_DFNODE(fnode_block, buf);
				if (__likely(!x)) goto next_block;
				if (__likely(x == 1)) {
					VFS$UNMAP_READDIR(map, unmap);
					VFS$PUT_BUFFER(fnode_block);
					goto read_it;
				}
				VFS$UNMAP_READDIR(map, unmap);
				if (x > 0) goto bad_fnode;
				goto exit_err;
			}

			if (__unlikely(CC_VALID((SPADFS *)RQ->fs, &fnode_block->cc, &fnode_block->txc))) {
				RQ->dirptr = MAKE_D_OFF(fnode_block->next0, fnode_block->next1);
				if (__likely(!RQ->chain_skip)) {
					COOKIE(rb)->chain = ++RQ->cookie.chain;
					COOKIE(rb)->fnode_offset = RQ->cookie.fnode_offset = 0;
				} else {
					RQ->chain_skip--;
				}
				VFS$UNMAP_READDIR(map, unmap);
				VFS$PUT_BUFFER(fnode_block);
				goto subnode;
			}
			goto_next_hash:
			VFS$PUT_BUFFER(fnode_block);
			if (!RQ->next_hash) {
				rb->end = 1;
				VFS$UNMAP_READDIR(map, unmap);
				if (VFS$DONE_READDIR((PAGEINRQ *)RQ)) goto repost;
				error = 0;
				goto err;
			}
			COOKIE(rb)->hash = RQ->cookie.hash = RQ->next_hash;
			COOKIE(rb)->chain = RQ->cookie.chain = 0;
			COOKIE(rb)->fnode_offset = RQ->cookie.fnode_offset = 0;
			VFS$UNMAP_READDIR(map, unmap);
#undef rb
			RQ->hash = RQ->cookie.hash;
			RQ->dirptr = ((SPADFNODE *)RQ->fnode)->root;
			RQ->c[1] = 0;
			RQ->hash_bits = 0;
			RQ->next_hash = 0;
			RQ->chain_skip = 0;
			goto subnode;
		} else if (__likely(d > 1)) {
			int i;
			struct dnode_page_entry *e;
			hash_t hpos;
			if (__unlikely(RQ->hash_bits >= HASH_BITS)) {
				VFS$PUT_BUFFER(fnode_block);
				goto deep;
			}
			hpos = RQ->hash & ((1 << ((SPADFS *)RQ->fs)->dnode_hash_bits) - 1);
			RQ->hash >>= ((SPADFS *)RQ->fs)->dnode_hash_bits;
			e = (struct dnode_page_entry *)((char *)fnode_block + d + hpos * DNODE_PAGE_ENTRY_SIZE);

			RQ->dirptr = MAKE_D_OFF3(e->b0, e->b1, e->b2);
			for (i = ((SPADFS *)RQ->fs)->dnode_hash_bits - 1; i >= 0; i--) if (!(hpos & (1 << i)) && __likely(i + RQ->hash_bits < HASH_BITS)) {
				__d_off new_off;
				e = (struct dnode_page_entry *)((char *)fnode_block + d + ((hpos & ((1 << i) - 1)) | (1 << i)) * DNODE_PAGE_ENTRY_SIZE);
				new_off = MAKE_D_OFF3(e->b0, e->b1, e->b2);
				if (__unlikely(!new_off) || new_off != RQ->dirptr) {
					RQ->next_hash = (RQ->cookie.hash & (((hash_t)1 << (RQ->hash_bits + i)) - 1)) | (hash_t)1 << (RQ->hash_bits + i);
					break;
				}
			}
			if (__likely(RQ->dirptr != 0)) {
				VFS$PUT_BUFFER(fnode_block);
				RQ->hash_bits += ((SPADFS *)RQ->fs)->dnode_hash_bits;
				goto subnode0;
			}
			map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, RQ->fs, &maplen, &unmap);
			if (__unlikely(!map)) goto no_map_ptr;
			if (__unlikely(map == (void *)1)) goto no_map_ptr_ret;
			if (__unlikely(__IS_ERR(map))) goto map_error;
			goto goto_next_hash;
		} else if (__likely(d == 1)) {
			VFS$PUT_BUFFER(fnode_block);
			goto read_it;
		} else goto exit_err;
	} else {
		read_it:
		VFS$READ_BUFFER(RQ->dirptr, (PAGEINRQ *)RQ, RQ->fs->buffer_readahead);
		RETURN;
	}

	err_put:
	VFS$PUT_BUFFER(fnode_block);
	err:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	caller->status = error;
	RETURN_AST(caller);

	repost:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);

	pagefault:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	DO_PAGEIN(caller, &((IOCTLRQ *)caller)->v, PF_RW);

	exit_err:
	VFS$PUT_BUFFER(fnode_block);
	error = -EFSERROR;
	goto err;

	bad_fnode:
	VFS$PUT_BUFFER(fnode_block);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012" __d_off_format "X (READDIR)", RQ->dirptr);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "BAD FNODE ON BLOCK %012" __d_off_format "X (READDIR %012" __d_off_format "X/%04X:%s, COOKIE %012"__64_format"X:%012"__d_off_format"X)", RQ->dirptr, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_address : 0, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_off : 0, RQ->fnode->name, (__u64)RQ->cookie.hash, RQ->cookie.chain);
	error = -EFSERROR;
	goto err;

	deep:
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "TOO DEEP DNODE TREE AT %012" __d_off_format "X (READDIR)", RQ->dirptr);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "TOO DEEP DNODE TREE AT %012" __d_off_format "X (READDIR %012" __d_off_format "X/%04X:%s, COOKIE %012"__64_format"X:%012"__d_off_format"X)", RQ->dirptr, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_address : 0, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_off : 0, RQ->fnode->name, (__u64)RQ->cookie.hash, RQ->cookie.chain);
	error = -EFSERROR;
	goto err;

	cycle:
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR, "CYCLE DETECTED ON BLOCK %012" __d_off_format "X (READDIR)", RQ->dirptr);
	SPADFS_ERROR((SPADFS *)RQ->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "CYCLE DETECTED ON BLOCK %012" __d_off_format "X (READDIR %012" __d_off_format "X/%04X:%s, COOKIE %012"__64_format"X:%012"__d_off_format"X)", RQ->dirptr, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_address : 0, RQ->fnode ? ((SPADFNODE *)RQ->fnode)->fnode_off : 0, RQ->fnode->name, (__u64)RQ->cookie.hash, RQ->cookie.chain);
	error = -EFSERROR;
	goto err;
}

