#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/LIST.H>
#include <ERRNO.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/USB.H>
#include <STDLIB.H>
#include <STDARG.H>

#include <SPAD/TTY.H>

#include "USBHID.H"

#define N_REQUESTS		3
#define HID_MEMORY_LIMIT	0x10000
#define HID_MAX_ERRORS		64

#define MAX_MOUSE_BUTTONS	32

typedef __s32 value_type;
typedef struct {
	__u32 key;
	__u8 report_id;
	__u8 current_report_id;		/* for qsort */
} key_struct;

typedef struct __hid HID;

typedef struct {
	USB_REQUEST urq;
	char posted;
	__u8 *data;
	HID *hid;
} HID_USBREQ;

struct __hid {
	USB_DEV_INTERFACE *iface;
	__u8 inum;
	char flags;
	unsigned short errors;
	USB_ENDPOINT *ctrl;
	USB_ENDPOINT *intr_in;
	unsigned current_seq;
	unsigned in_rq_size;
	LIST_HEAD *report[3];

	kbdmouse_call_t *mouse_upcall;
	struct mouse_state mouse_state;
	kbdext_call_t *kbd_upcall;
	key_struct *current_keys;
	unsigned n_current_keys;
	key_struct *last_keys;
	unsigned n_last_keys;
	unsigned n_keys;

	HID_USBREQ req[N_REQUESTS];

	LIST_HEAD collections;
	unsigned in_pkt_size;
	__u8 *report_desc;
	unsigned report_desc_size;

	void *lnte;
	void *dlrq;
	char dev_name[__MAX_STR_LEN];
	char kbd_dev_name[__MAX_STR_LEN];
};

#define HID_SHUTDOWN		1
#define HID_APP_MOUSE		2
#define HID_APP_KEYBOARD	4
#define HID_UPDATED_MOUSE	8
#define HID_UPDATED_KEYBOARD	16

typedef struct {
	__s32 logical_minimum;
	__s32 logical_maximum;
	__s32 physical_minimum;
	__s32 physical_maximum;
	__s32 unit_exponent;
	__u32 unit;
	__u8 report_size;
	__u16 report_count;
} REPORT_ATTR;

typedef struct {
	LIST_ENTRY list;	/* must match LIST_CONTAINER */
	__u8 report_id;
	__u16 usage_page;
	REPORT_ATTR a;
} GLOBAL;

typedef struct {
	char delimiter_depth;
	unsigned delimiter_branch;
	__u32 usage_minimum;
} LOCAL;

typedef struct {
	LIST_ENTRY list;	/* must match LIST_CONTAINER */
	__u32 from;
	__u32 to;
} USAGE;

typedef struct collection COLLECTION;

struct collection {
	LIST_ENTRY list;
	LIST_HEAD usages;	/* must match LIST_CONTAINER */
	COLLECTION *parent;
	__u8 type;
};

typedef struct {
	LIST_ENTRY list;
	LIST_HEAD usages;	/* must match LIST_CONTAINER */
	COLLECTION *collection;	/* may be NULL if outside collection */
	REPORT_ATTR a;
	__u32 type;
	char app;		/* HID_APP_* flags */
	value_type value[1];
} REPORT;

typedef struct {
	LIST_ENTRY list;
	LIST_HEAD usages;
} LIST_CONTAINER;

static void HID_PROCESSRQ(HID *hid, HID_USBREQ *req);
static void PROCESS_MOUSE(HID *hid, REPORT *rep);
static void PROCESS_KEYBOARD(HID *hid, REPORT *rep, __u8 report_id);
static void HID_SEND_KEYBOARD(HID *hid, __u8 report_id);
static void HID_SEND_KEY(HID *hid, __u32 key, int down);

static unsigned FREE_LIST(LIST_HEAD *head)
{
	unsigned cnt = 0;
	while (!LIST_EMPTY(head)) {
		LIST_CONTAINER *x = LIST_STRUCT(head->next, LIST_CONTAINER, list);
		DEL_FROM_LIST(&x->list);
		free(x);
		cnt++;
	}
	return cnt;
}

static void FREE_DOUBLE_LIST(LIST_HEAD *head)
{
	LIST_CONTAINER *l;
	LIST_FOR_EACH(l, head, LIST_CONTAINER, list) FREE_LIST(&l->usages);
	FREE_LIST(head);
}

static void HID_DISABLE(HID *hid)
{
	unsigned i;
	hid->flags |= HID_SHUTDOWN;
	if (hid->flags & HID_APP_KEYBOARD) {
		hid->n_current_keys = 0;
		for (i = 0; i < 256; i++) HID_SEND_KEYBOARD(hid, i);
	}
	if (hid->flags & HID_APP_MOUSE && __unlikely(hid->mouse_state.buttons)) {
		hid->mouse_state.buttons = 0;
		hid->mouse_state.rx = hid->mouse_state.ry = hid->mouse_state.rz = hid->mouse_state.wx = hid->mouse_state.wy = hid->mouse_state.wz = 0;
		hid->mouse_upcall(&hid->mouse_state);
	}
}

static __finline__ void HID_SENDRQ(HID *hid, HID_USBREQ *req, int process)
{
	do {
		if (process) HID_PROCESSRQ(hid, req);
		if (__unlikely(hid->flags & HID_SHUTDOWN)) break;
		req->posted = 1;
		/* fn set */
		req->urq.flags = USBRQ_ALLOW_SHORT;
		req->urq.v.ptr = (unsigned long)req->data;
		req->urq.v.len = hid->in_rq_size;
		/* vspace set */
		RAISE_SPL(SPL_USB);
		hid->intr_in->prepare(hid->intr_in, &req->urq);
		hid->intr_in->post(&req->urq);
		LOWER_SPL(SPL_HID);
		hid->current_seq = (hid->current_seq + 1) % N_REQUESTS;
	} while (__unlikely(!(req = &hid->req[hid->current_seq])->posted));
}

static DECL_AST(HID_GOT_REPORT, SPL_HID, USB_REQUEST)
{
	HID_USBREQ *req = GET_STRUCT(RQ, HID_USBREQ, urq);
	HID *hid = req->hid;
	unsigned seq = req - hid->req;
#if __DEBUG >= 1
	if (__unlikely(seq >= N_REQUESTS))
		KERNEL$SUICIDE("HID_GOT_REPORT: RETURNED REQUEST WITH BOGUS SEQUENNCE NUMBER %u (%p, %p)", seq, req, hid->req);
	if (__unlikely(!req->posted))
		KERNEL$SUICIDE("HID_GOT_REPORT: AST FOR NON-POSTED REQUEST %u", seq);
#endif
	req->posted = 0;
	if (__unlikely(seq != hid->current_seq)) goto ret;
	HID_SENDRQ(hid, req, 1);
	ret:
	RETURN;
}

static void HID_PROCESSRQ(HID *hid, HID_USBREQ *req)
{
	__u32 *ptr;
	__u8 ptr_bit;
	unsigned c;
	LIST_HEAD *replist;
	REPORT *report;
	__u8 report_id;
	if (__unlikely(req->urq.status < 0)) {
		inc_error:
		hid->errors++;
		if (__unlikely(hid->errors > HID_MAX_ERRORS) && __unlikely(!(hid->flags & HID_SHUTDOWN))) {
			if (__unlikely(req->urq.status != -ECONNRESET) && USB$GET_ERRORLEVEL(hid->iface) >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "DISABLING, TOO MANY ERRORS%s%s", req->urq.status < 0 ? ": " : "", req->urq.status < 0 ? strerror(-req->urq.status) : "");
			HID_DISABLE(hid);
		}
		return;
	}
	if (__unlikely(req->urq.v.len != hid->in_rq_size)) {
#if __DEBUG >= 1
		if (__unlikely(req->urq.v.len > hid->in_rq_size))
			KERNEL$SUICIDE("HID_PROCESSRQ: RETURNED MORE DATA THAN REQUESTED: %lu > %u", req->urq.v.len, hid->in_rq_size);
#endif
		memset(req->data + req->urq.v.len, 0, hid->in_rq_size - req->urq.v.len);
	}
	/*{
		int i;
		for (i = 0; i < req->urq.v.len; i++)
			__debug_printf("%02x ", req->data[i]);
		__debug_printf("\n");
	}*/
	replist = &hid->report[HID_REPORT_INPUT][0];
	ptr_bit = 0;
	report_id = 0;
	if (LIST_EMPTY(replist)) {
		replist = &hid->report[HID_REPORT_INPUT][report_id = req->data[0]];
		if (__unlikely(LIST_EMPTY(replist))) {
			if (!req->urq.v.len) {
				if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "RECEIVED ZERO-LENGTH REPORT");
			} else {
				if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "RECEIVED REPORT WITH UNKNOWN ID %02X", req->data[0]);
			}
			goto inc_error;
		}
		ptr_bit = 8;
	}
	hid->errors = 0;
	ptr = (__u32 *)req->data;
	LIST_FOR_EACH(report, replist, REPORT, list) {
		for (c = 0; c < report->a.report_count; c++) {
			unsigned val_bit = 0;
			value_type val = 0;
			if (__likely(report->a.report_size)) {
				value_type signbit;
				do {
					unsigned get_bits = report->a.report_size - val_bit;
					if (__unlikely(get_bits > 32 - ptr_bit)) get_bits = 32 - ptr_bit;
					val |= __32LE2CPU(*ptr) >> ptr_bit << val_bit;
					val_bit += get_bits;
					ptr_bit += get_bits;
					ptr = (__u32 *)((char *)ptr + ((ptr_bit >> (5 - 2)) & 4)), ptr_bit &= 31;
				} while (__unlikely(val_bit < report->a.report_size));
				val &= (2 << (report->a.report_size - 1)) - 1;
				signbit = report->a.logical_minimum >> (sizeof(report->a.logical_minimum) * 8 - 1) & (val >> (report->a.report_size - 1));
				signbit = signbit << 1 << (report->a.report_size - 1);
				val |= -signbit;
			}
			if (__likely(val >= report->a.logical_minimum) && __likely(val <= report->a.logical_maximum)) {
				report->value[c] = val;
			} else {
				if (report->type & HID_REPORT_TYPE_RELATIVE) report->value[c] = 0;
			}
		}
		if (report->app & HID_APP_MOUSE) PROCESS_MOUSE(hid, report);
		if (report->app & HID_APP_KEYBOARD) PROCESS_KEYBOARD(hid, report, report_id);
	}
#if __DEBUG >= 1
	if (((__u8 *)ptr - req->data) + ((ptr_bit + 7) >> 3) > hid->in_rq_size)
		KERNEL$SUICIDE("HID_PROCESSRQ: RAN OUT OF INPUT DATA, PTR %p, DATA %p, PTR_BIT %u, IN_RQ_SIZE %u", ptr, req->data, ptr_bit, hid->in_rq_size);
#endif
	if (__likely(hid->flags & HID_UPDATED_MOUSE)) {
		hid->flags &= ~HID_UPDATED_MOUSE;
		hid->mouse_upcall(&hid->mouse_state);
		hid->mouse_state.rx = hid->mouse_state.ry = hid->mouse_state.rz = hid->mouse_state.wx = hid->mouse_state.wy = hid->mouse_state.wz = 0;
	}
	if (hid->flags & HID_UPDATED_KEYBOARD) {
		hid->flags &= ~HID_UPDATED_KEYBOARD;
		HID_SEND_KEYBOARD(hid, report_id);
		if (__unlikely(hid->n_current_keys))
			KERNEL$SUICIDE("HID_PROCESSRQ: HID_SEND_KEYBOARD DIDN'T SEND KEY COUNT (%d)", hid->n_current_keys);
	} else {
		if (__unlikely(hid->n_current_keys))
			KERNEL$SUICIDE("HID_PROCESSRQ: KEY COUNT %d AND FLAG UPDATED_KEYBOARD NOY SET", hid->n_current_keys);
	}
}

static void PROCESS_MOUSE(HID *hid, REPORT *rep)
{
	USAGE *u;
	unsigned us, i;
	if (__unlikely(rep->type & (HID_REPORT_TYPE_CONSTANT | HID_REPORT_TYPE_BUFFERE_BYTES))) return;
	i = 0;
	LIST_FOR_EACH(u, &rep->usages, USAGE, list) for (us = u->from; us <= u->to; us++) {
		value_type val;
		if (__unlikely(i >= rep->a.report_count)) goto brk;
		val = rep->value[i];
		if (us >= HID_USAGE_BUTTON_1 && us < HID_USAGE_BUTTON_1 + MAX_MOUSE_BUTTONS) {
			unsigned btn_mask = 1 << (us - HID_USAGE_BUTTON_1);
			if (val) hid->mouse_state.buttons |= btn_mask;
			else hid->mouse_state.buttons &= ~btn_mask;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_X) {
			hid->mouse_state.rx += (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_Y) {
			hid->mouse_state.ry += (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_Z) {
			hid->mouse_state.rz += (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_RX) {
			hid->mouse_state.wx += (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_RY) {
			hid->mouse_state.wy += (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_RZ) {
			hid->mouse_state.wz += (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		} else if (us == HID_USAGE_DV_WHEEL) {
			hid->mouse_state.wy -= (unsigned)val;
			hid->flags |= HID_UPDATED_MOUSE;
		}
		i++;
	}
	brk:;
}

static void PROCESS_KEYBOARD(HID *hid, REPORT *rep, __u8 report_id)
{
	USAGE *u;
	unsigned us, i;
	if (__unlikely(rep->type & (HID_REPORT_TYPE_CONSTANT | HID_REPORT_TYPE_BUFFERE_BYTES))) return;
	hid->flags |= HID_UPDATED_KEYBOARD;
	i = 0;
	if (rep->type & HID_REPORT_TYPE_VARIABLE) {
		LIST_FOR_EACH(u, &rep->usages, USAGE, list) for (us = u->from; us <= u->to; us++) {
			if (__unlikely(i >= rep->a.report_count)) goto brk;
			if (__unlikely(rep->value[i] != 0)) {
				if (__unlikely(hid->n_current_keys >= hid->n_keys))
					KERNEL$SUICIDE("PROCESS_KEYBOARD: KEY ARRAY OVERFLOW (VARIABLE)");
				hid->current_keys[hid->n_current_keys].key = us;
				hid->current_keys[hid->n_current_keys].report_id = report_id;
				hid->n_current_keys++;
			}
			i++;
		}
	} else {
		for (; i < rep->a.report_count; i++) {
			value_type val = rep->value[i];
			LIST_FOR_EACH(u, &rep->usages, USAGE, list) if (__likely(u->from <= u->to)) {
				if (__likely(val < u->to - u->from + 1)) {
					us = u->from + val;
					goto have_key;
				} else {
					val -= u->to - u->from + 1;
				}
			}
			continue;
			have_key:
			if (__likely(us == HID_USAGE_KEYBOARD_NO_EVENT) || __unlikely(us == HID_USAGE_CONSUMER_CONTROL_NO_EVENT)) continue;
			if (__unlikely(hid->n_current_keys >= hid->n_keys))
				KERNEL$SUICIDE("PROCESS_KEYBOARD: KEY ARRAY OVERFLOW (ARRAY)");
			hid->current_keys[hid->n_current_keys].key = us;
			hid->current_keys[hid->n_current_keys].report_id = report_id;
			hid->n_current_keys++;
		}
	}
	brk:;
}

static int KEY_COMPARE(const void *k1_, const void *k2_)
{
	const key_struct *k1 = k1_;
	const key_struct *k2 = k2_;
	int q = (k2->report_id == k2->current_report_id) - (k1->report_id == k1->current_report_id);
	if (__unlikely(q)) return q;
	if (k1->key < k2->key) return -1;
	if (__likely(k1->key > k2->key)) return 1;
	return 0;
}

static void HID_SEND_KEYBOARD(HID *hid, __u8 report_id)
{
	unsigned i;
	unsigned c, l;
	key_struct *ck;
	unsigned nck;
	int err = 0;
	/*
	__debug_printf("current:");
	for (c = 0; c < hid->n_current_keys; c++) __debug_printf("%x ", hid->current_keys[c].key);
	__debug_printf("\n");
	__debug_printf("last:");
	for (c = 0; c < hid->n_last_keys; c++) __debug_printf("%x ", hid->last_keys[c].key);
	__debug_printf("\n");
	*/
	for (i = 0; i < hid->n_current_keys; i++) {
		if (__unlikely(hid->current_keys[i].key == HID_USAGE_KEYBOARD_POST_FAIL)) {
			if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
				KERNEL$SYSLOG(__SYSLOG_HW_ERROR, hid->dev_name, "KEYBOARD POST FAILURE");
			goto ret;
		}
		if (__unlikely(hid->current_keys[i].key == HID_USAGE_KEYBOARD_ROLL_OVER) ||
		    __unlikely(hid->current_keys[i].key == HID_USAGE_KEYBOARD_ERROR_UNDEFINED)) {
			err = 1;
		}
		hid->current_keys[i].current_report_id = report_id;
	}
	if (__unlikely(err)) goto ret;
	for (i = 0; i < hid->n_last_keys; i++) {
		hid->last_keys[i].current_report_id = report_id;
	}
	qsort(hid->current_keys, hid->n_current_keys, sizeof(key_struct), KEY_COMPARE);
	qsort(hid->last_keys, hid->n_last_keys, sizeof(key_struct), KEY_COMPARE);
	c = 0, l = 0;
	while ((c < hid->n_current_keys) || (l < hid->n_last_keys && hid->last_keys[l].report_id == report_id)) {
		int kc;
		if (c >= hid->n_current_keys) {
			goto kc1;
		} else if (l >= hid->n_last_keys || hid->last_keys[l].report_id != report_id) {
			goto kcm1;
		} else {
			kc = KEY_COMPARE(&hid->current_keys[c], &hid->last_keys[l]);
		}
		if (kc < 0) {
			kcm1:
			HID_SEND_KEY(hid, hid->current_keys[c].key, KBD_KEY_DOWN_SOFT_REPEAT);
			c++;
		} else if (kc > 0) {
			kc1:
			HID_SEND_KEY(hid, hid->last_keys[l].key, KBD_KEY_UP);
			l++;
		} else {
			c++, l++;
		}
	}
	if (__unlikely(hid->n_current_keys + hid->n_last_keys - l > hid->n_keys))
		KERNEL$SUICIDE("HID_SEND_KEYBOARD: KEY ARRAY OVERFLOW: n_current_keys=%u, n_last_keys=%u, c=%u, l=%u, n_keys=%u", hid->n_current_keys, hid->n_last_keys, c, l, hid->n_keys);
	memcpy(&hid->current_keys[c], &hid->last_keys[l], (hid->n_last_keys - l) * sizeof(key_struct));
	hid->n_current_keys += hid->n_last_keys - l;
	ck = hid->current_keys;
	nck = hid->n_current_keys;
	hid->current_keys = hid->last_keys;
	hid->last_keys = ck;
	hid->n_last_keys = nck;
	ret:
	hid->n_current_keys = 0;
}

#include "USBHID.I"

static void HID_SEND_KEY(HID *hid, __u32 key, int down)
{
	if (__likely((key - HID_USAGE_KEYBOARD) < sizeof HID_KEYTABLE)) {
		__u8 keycode = __secure_lookup_8(HID_KEYTABLE, sizeof HID_KEYTABLE, key - HID_USAGE_KEYBOARD);
		if (__likely(keycode)) hid->kbd_upcall(keycode, down);
	}
}

static int TEST_LIST_USAGE_VA(LIST_HEAD *l, va_list va)
{
	unsigned a;
	while ((a = va_arg(va, int))) {
		USAGE *u;
		LIST_FOR_EACH(u, l, USAGE, list) if (a >= u->from && a <= u->to) return 1;
	}
	return 0;
}

static int TEST_LIST_USAGE(LIST_HEAD *l, ...)
{
	int r;
	va_list va;
	va_start(va, l);
	r = TEST_LIST_USAGE_VA(l, va);
	va_end(va);
	return r;
}

static int TEST_COLLECTION_USAGE(COLLECTION *c, ...)
{
	for (; c; c = c->parent) {
		va_list va;
		va_start(va, c);
		if (TEST_LIST_USAGE_VA(&c->usages, va)) return 1;
		va_end(va);
	}
	return 0;
}

static int FIND_MAX_REPORT(HID *hid, int report_type, unsigned *max_size)
{
	int i;
	for (i = 0; i < HID_N_REPORTS; i++) if (__unlikely(!LIST_EMPTY(&hid->report[report_type][i]))) {
		unsigned size = !!i;
		REPORT *report;
		LIST_FOR_EACH(report, &hid->report[report_type][i], REPORT, list) {
			size += report->a.report_size * report->a.report_count;
			if (__unlikely(size > (report_type == HID_REPORT_INPUT ? hid->in_pkt_size : HID_MAX_REPORT_SIZE) * 8)) {
				if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "REPORT SIZE OVERFLOW");
				return -EPROTO;
			}
		}
		size = (size + 7) >> 3;
		if (size > *max_size) *max_size = size;
	}
	return 0;
}

static int HID_PARSE(HID *hid)
{
	int memory = 0;
	unsigned idx = 0;
	int i, q;
	int r;
	GLOBAL *glb;
	LOCAL local;
	USAGE *usage;
	COLLECTION *current_collection = NULL, *collection;
	REPORT *report;
	DECL_LIST(global_stack);
	DECL_LIST(usages);
	glb = __sync_malloc(sizeof(GLOBAL));
	if (__unlikely(!glb)) {
		ret_errno:
		r = -errno;
		goto ret;
	}
	memory += sizeof(GLOBAL);
	memset(glb, 0, sizeof(GLOBAL));
	ADD_TO_LIST(&global_stack, &glb->list);
	memset(&local, 0, sizeof local);
#define global	LIST_STRUCT(global_stack.next, GLOBAL, list)
	while (idx < hid->report_desc_size) {
		__u8 byte, type, tag, size;
		unsigned i;
		__u32 uvalue;
		__u32 svalue;
		if (__unlikely(memory > HID_MEMORY_LIMIT)) {
			if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "DEVICE STRUCTURES CONSUME TOO MUCH MEMORY");
			r = -ENOBUFS;
			goto ret;
		}
		byte = hid->report_desc[idx++];
		type = (byte >> 2) & 3;
		tag = (byte >> 4) & 15;
		size = (byte & 3) + ((byte & 3) == 3);
		if (__unlikely(idx + size > hid->report_desc_size)) {
			if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "RAN OUT OF REPORT DESCRIPTOR WHEN GETTING DATA");
			ret_eproto:
			r = -EPROTO;
			goto ret;
		}
		uvalue = 0;
		for (i = 0; i < size; i++) uvalue |= hid->report_desc[idx++] << (i << 3);
		if (size == 1) svalue = (__s8)uvalue;
		else if (size == 2) svalue = (__s16)uvalue;
		else svalue = uvalue;
		if (size == 2 && type == 3 && tag == 15) {	/* skip long items */
			idx += uvalue >> 8;
			if (__unlikely(idx > hid->report_desc_size)) {
				if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "RAN OUT OF REPORT DESCRIPTOR WHEN SKIPPING LONG ITEM");
				goto ret_eproto;
			}
			continue;
		}
		if (type == HID_ITEM_TYPE_MAIN) {
			q = HID_REPORT_FEATURE;
			switch (tag) {
				case HID_MAIN_ITEM_TAG_BEGIN_COLLECTION:
					if (__unlikely(uvalue >= 256)) {
						if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "INVALID COLLECTION %X", (unsigned)uvalue);
						goto ret_eproto;
					}
					collection = __sync_malloc(sizeof(COLLECTION));
					if (__unlikely(!collection)) goto ret_errno;
					memory += sizeof(COLLECTION);
					collection->type = uvalue;
					MOVE_LIST_HEAD(&collection->usages, &usages);
					INIT_LIST(&usages);
					collection->parent = current_collection;
					current_collection = collection;
					ADD_TO_LIST_END(&hid->collections, &collection->list);
					break;
				case HID_MAIN_ITEM_TAG_END_COLLECTION:
					if (!current_collection) {
						if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "COLLECTION STACK UNDERFLOW");
						goto ret_eproto;
					}
					current_collection = current_collection->parent;
					break;
				case HID_MAIN_ITEM_TAG_INPUT:
					q += HID_REPORT_INPUT - HID_REPORT_OUTPUT;
					/* fall through */
				case HID_MAIN_ITEM_TAG_OUTPUT:
					q += HID_REPORT_OUTPUT - HID_REPORT_FEATURE;
					/* fall through */
				case HID_MAIN_ITEM_TAG_FEATURE:
					report = __sync_malloc(sizeof(REPORT) + global->a.report_count * sizeof(value_type));
					if (__unlikely(!report)) goto ret_errno;
					memory += sizeof(REPORT) + global->a.report_count * sizeof(value_type);
					MOVE_LIST_HEAD(&report->usages, &usages);
					INIT_LIST(&usages);
					report->collection = current_collection;
					memcpy(&report->a, &global->a, sizeof(REPORT_ATTR));
					report->app = 0;
					report->type = uvalue;
					for (i = 0; i < report->a.report_count; i++) report->value[i] = 0;
					ADD_TO_LIST_END(&hid->report[q][global->report_id], &report->list);
					if (report->type & HID_REPORT_TYPE_CONSTANT || !report->a.report_count) goto skip_app_usage;
					if (TEST_COLLECTION_USAGE(report->collection, HID_USAGE_CP_POINTER, HID_USAGE_CA_MOUSE, HID_USAGE_CA_MULTI_AXIS_CONTROLLER, 0)) {
						int i;
						report->app |= HID_APP_MOUSE, hid->flags |= HID_APP_MOUSE;
						for (i = hid->mouse_state.x; i < MAX_MOUSE_BUTTONS; i++) if (TEST_LIST_USAGE(&report->usages, HID_USAGE_BUTTON_1 + i, 0)) hid->mouse_state.x = i + 1;
						if (hid->mouse_state.y < 1 && TEST_LIST_USAGE(&report->usages, HID_USAGE_DV_WHEEL, 0)) hid->mouse_state.y = 1;
						if (hid->mouse_state.z < 1 && TEST_LIST_USAGE(&report->usages, HID_USAGE_DV_X, 0)) hid->mouse_state.z = 1;
						if (hid->mouse_state.z < 2 && TEST_LIST_USAGE(&report->usages, HID_USAGE_DV_Y, 0)) hid->mouse_state.z = 2;
						if (hid->mouse_state.z < 3 && TEST_LIST_USAGE(&report->usages, HID_USAGE_DV_Z, 0)) hid->mouse_state.z = 3;
					}
					if (TEST_COLLECTION_USAGE(report->collection, HID_USAGE_CA_KEYBOARD, HID_USAGE_CA_KEYPAD, 0)) {
						report->app |= HID_APP_KEYBOARD, hid->flags |= HID_APP_KEYBOARD;
						hid->n_keys += report->a.report_count;
						memory += report->a.report_count * sizeof(key_struct) * 2;
					}
					skip_app_usage:
					break;
			}
			memset(&local, 0, sizeof local);
			memory -= FREE_LIST(&usages) * sizeof(USAGE);
			if (__unlikely(memory < 0))
				KERNEL$SUICIDE("HID_PARSE: MEMORY COUNTER UNDERFLOW: %d (MAIN)", memory);
		} else if (type == HID_ITEM_TYPE_GLOBAL) switch (tag) {
			case HID_GLOBAL_ITEM_TAG_PUSH:
				glb = __sync_malloc(sizeof(GLOBAL));
				if (__unlikely(!glb)) goto ret_errno;
				memory += sizeof(GLOBAL);
				memcpy(glb, global, sizeof(GLOBAL));
				ADD_TO_LIST(&global_stack, &glb->list);
				break;
			case HID_GLOBAL_ITEM_TAG_POP:
				glb = global;
				DEL_FROM_LIST(&glb->list);
				if (__unlikely(LIST_EMPTY(&global_stack))) {
					if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "GLOBAL STACK UNDERFLOW");
					goto ret_eproto;
				}
				free(glb);
				memory -= sizeof(GLOBAL);
				if (__unlikely(memory < 0))
					KERNEL$SUICIDE("HID_PARSE: MEMORY COUNTER UNDERFLOW (GLOBAL POP): %d", memory);
				break;
			case HID_GLOBAL_ITEM_TAG_USAGE_PAGE:
				global->usage_page = uvalue;
				break;
			case HID_GLOBAL_ITEM_TAG_LOGICAL_MINIMUM:
				global->a.logical_minimum = svalue;
				break;
			case HID_GLOBAL_ITEM_TAG_LOGICAL_MAXIMUM:
				if (global->a.logical_minimum < 0)
					global->a.logical_maximum = svalue;
				else
					global->a.logical_maximum = uvalue;
				break;
			case HID_GLOBAL_ITEM_TAG_PHYSICAL_MINIMUM:
				global->a.physical_minimum = svalue;
				break;
			case HID_GLOBAL_ITEM_TAG_PHYSICAL_MAXIMUM:
				if (global->a.physical_minimum < 0)
					global->a.physical_maximum = svalue;
				else
					global->a.physical_maximum = uvalue;
				break;
			case HID_GLOBAL_ITEM_TAG_UNIT_EXPONENT:
				global->a.unit_exponent = svalue;
				break;
			case HID_GLOBAL_ITEM_TAG_UNIT:
				global->a.unit = uvalue;
				break;
			case HID_GLOBAL_ITEM_TAG_REPORT_SIZE:
				if (__unlikely(uvalue > 32)) {
					if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "REPORT SIZE %u > 32", (unsigned)uvalue);
					goto ret_eproto;
				}
				global->a.report_size = uvalue;
				break;
			case HID_GLOBAL_ITEM_TAG_REPORT_COUNT:
				if (__unlikely(uvalue >= 0x10000)) {
					if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "REPORT COUNT %u >= 65536", (unsigned)uvalue);
					goto ret_eproto;
				}
				global->a.report_count = uvalue;
				break;
			case HID_GLOBAL_ITEM_TAG_REPORT_ID:
				if (__unlikely(!uvalue) || __unlikely(uvalue >= 256)) {
					if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "INVALID REPORT ID %u", (unsigned)uvalue);
					goto ret_eproto;
				}
				global->report_id = uvalue;
				break;
		} else if (type == HID_ITEM_TYPE_LOCAL) switch (tag) {
			case HID_LOCAL_ITEM_TAG_USAGE:
				if (__unlikely(local.delimiter_branch > 1) && __likely(local.delimiter_depth)) break;
				if (size <= 2) uvalue |= global->usage_page << 16;
				usage = __sync_malloc(sizeof(USAGE));
				if (__unlikely(!usage)) goto ret_errno;
				memory += sizeof(USAGE);
				usage->from = uvalue;
				usage->to = uvalue;
				ADD_TO_LIST_END(&usages, &usage->list);
				break;
			case HID_LOCAL_ITEM_TAG_USAGE_MINIMUM:
				if (__unlikely(local.delimiter_branch > 1) && __likely(local.delimiter_depth)) break;
				if (size <= 2) uvalue |= global->usage_page << 16;
				local.usage_minimum = uvalue;
				break;
			case HID_LOCAL_ITEM_TAG_USAGE_MAXIMUM:
				if (__unlikely(local.delimiter_branch > 1) && __likely(local.delimiter_depth)) break;
				if (size <= 2) uvalue |= global->usage_page << 16;
				if (__unlikely(local.usage_minimum > uvalue)) break;
				usage = __sync_malloc(sizeof(USAGE));
				if (__unlikely(!usage)) goto ret_errno;
				memory += sizeof(USAGE);
				usage->from = local.usage_minimum;
				usage->to = uvalue;
				ADD_TO_LIST_END(&usages, &usage->list);
				break;
			case HID_LOCAL_ITEM_TAG_DELIMITER:
				if (uvalue) {
					if (__unlikely(local.delimiter_depth)) {
						if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "NESTED DELIMITERS");
						goto ret_eproto;
					}
					local.delimiter_depth = 1;
					local.delimiter_branch++;
				} else {
					if (__unlikely(!local.delimiter_depth)) {
						if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "NESTED DELIMITERS");
						goto ret_eproto;
					}
					local.delimiter_depth = 0;
				}
				break;
		}
	}
	if (__unlikely(current_collection != NULL)) {
		if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "COLLECTION LEFT OPEN");
	}
	if (!LIST_EMPTY(&hid->report[HID_REPORT_INPUT][0])) {
		for (i = 1; i < HID_N_REPORT_TYPES; i++) if (__unlikely(!LIST_EMPTY(&hid->report[HID_REPORT_INPUT][i]))) {
			if (USB$GET_ERRORLEVEL(hid->iface) >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "BOTH NUMBERED AND UNNUMBERED INPUT REPORTS EXIST");
			goto ret_eproto;
		}
	}
	hid->in_rq_size = 0;
	if (__unlikely(r = FIND_MAX_REPORT(hid, HID_REPORT_INPUT, &hid->in_rq_size))) goto ret;
	if (__unlikely(r = FIND_MAX_REPORT(hid, HID_REPORT_OUTPUT, (void *)&KERNEL$LIST_END))) goto ret;
	if (__unlikely(r = FIND_MAX_REPORT(hid, HID_REPORT_FEATURE, (void *)&KERNEL$LIST_END))) goto ret;
	r = 0;
	ret:
	FREE_LIST(&global_stack);
	FREE_LIST(&usages);
#undef global
	return r;
}

static int HID_TEST(USB_DEV_INTERFACE *iface, long dummy)
{
	USB_DESCRIPTOR_INTERFACE *desc = USB$GET_INTERFACE_DESCRIPTOR(iface);
	/*__debug_printf("class: %02X\n", desc->bInterfaceClass);*/
	if (__unlikely(desc->bInterfaceClass != USB_CLASS_HID)) return USB_SKIP;
	return USB_ATTACH_INTERFACE;
}

static void HID_SET_IDLE(HID *hid, unsigned idle)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_HID)))
		KERNEL$SUICIDE("HID_SET_IDLE AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_USB);
	USB$SYNC_CTRL(hid->ctrl, USBHID_REQ_SET_IDLE, idle, hid->inum, 0, NULL, USBRQ_QUIET_STALL, USB$GET_CTRL_TIMEOUT(hid->iface), USB$GET_CTRL_RETRIES(hid->iface), NULL, NULL);
	LOWER_SPL(SPL_HID);
}

static void HID_FREE(HID *hid);
static int HID_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	int i, j;
	int r;
	HID *hid;
	MALLOC_REQUEST mrq;
	USB_DESCRIPTOR_HID *hid_desc;
	USB_DEV_INTERFACE *iface;
	USB_DESCRIPTOR_ENDPOINT *in_desc;
	USB_ARGS *args = NULL;
	const char *opt, *optend, *str;
	const char * const *arg = argv;
	int state = 0;
	char *keyboard = DEFAULT_KEYBOARD_NAME ":";
	static const struct __param_table params[2] = {
		"KEYBOARD", __PARAM_STRING, 1, __MAX_STR_LEN,
		NULL, 0, 0, 0,
	};
	void *vars[2];
	vars[0] = &keyboard;
	vars[1] = NULL;
	if (__unlikely(__parse_params(&arg, &state, params, vars, &opt, &optend, &str))) {
		if (__unlikely(USB$PARSE_PARAMS(opt, optend, str, &args))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBHID: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (__IS_ERR(iface = USB$ATTACH_DRIVER(args, HID_TEST, 0))) {
		r = __PTR_ERR(iface);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBHID: %s", strerror(-r));
		goto ret0;
	}
	mrq.size = sizeof(HID);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBHID: %s", strerror(-r));
		goto ret1;
	}
	hid = mrq.ptr;
	memset(hid, 0, sizeof(HID));
	INIT_LIST(&hid->collections);
	hid->iface = iface;
	hid->inum = USB$GET_INTERFACE_NUMBER(iface);
	strcpy(hid->kbd_dev_name, keyboard);
	for (i = 0; i < HID_N_REPORT_TYPES; i++) {
		mrq.size = sizeof(LIST_HEAD) * HID_N_REPORTS;
		SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(mrq.status < 0)) {
			r = mrq.status;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBHID: %s", strerror(-r));
			goto ret2;
		}
		hid->report[i] = mrq.ptr;
		for (j = 0; j < HID_N_REPORTS; j++) INIT_LIST(&hid->report[i][j]);
	}
	r = USB$GET_DEVICE_NAME(hid->dev_name, __MAX_STR_LEN, iface, args, "HID$USBHID");
	if (__unlikely(r)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBHID: CAN'T GET DEVICE NAME: %s", strerror(-r));
		goto ret2;
	}

	if (__unlikely(!(hid_desc = USB$FIND_DESCRIPTOR(iface, HID_DT, 0, USB_DESCRIPTOR_HID_MIN_SIZE)))) {
		if (USB$GET_ERRORLEVEL(iface) >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "DEVICE DOESN'T HAVE HID DESCRIPTOR");
		device_err_2:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID DEVICE", hid->dev_name);
		r = -EPROTO;
		goto ret2;
	}
	if (__unlikely(hid_desc->bLength < USB_DESCRIPTOR_HID_SIZE(hid_desc->bNumDescriptors))) {
		if (USB$GET_ERRORLEVEL(iface) >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "DEVICE HAS INVALID HID DESCRIPTOR: bLength %u, bNumDescriptors %u", hid_desc->bLength, hid_desc->bNumDescriptors);
		goto device_err_2;
	}
	/*{
		int h = creat("hiddesc.usb", 0600);
		write(h, hid_desc, hid_desc->bLength);
		close(h);
	}*/
	for (i = 0; i < hid_desc->bNumDescriptors; i++) {
		if (__likely(hid_desc->descs[i].bDescriptorType == HID_DT_REPORT)) {
			hid->report_desc_size = hid_desc->descs[i].wDescriptorLength_lo + (hid_desc->descs[i].wDescriptorLength_hi << 8);
			goto have_report;
		}
	}
	if (USB$GET_ERRORLEVEL(iface) >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "DEVICE DOESN'T HAVE REPORT DESCRIPTOR");
	goto device_err_2;
	have_report:
	mrq.size = hid->report_desc_size;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", hid->dev_name, strerror(-r));
		goto ret2;
	}
	in_desc = USB$GET_ENDPOINT_DESCRIPTOR(iface, USB_EP_INTR_IN, 0);
	if (__unlikely(!in_desc)) {
		if (USB$GET_ERRORLEVEL(iface) >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hid->dev_name, "DEVICE DOESN'T HAVE INPUT INTERRUPT ENDPOINT");
		r = -EPROTO;
		goto ret2;
	}
	hid->in_pkt_size = in_desc->wMaxPacketSize_lo + (in_desc->wMaxPacketSize_hi << 8);
	hid->report_desc = mrq.ptr;
	if (__unlikely((r = USB$REQUEST_DESCRIPTOR(iface, USB_CTRL_DIR_IN | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_INTERFACE, HID_DT_REPORT, 0, hid->inum, hid->report_desc, hid->report_desc_size, hid->report_desc_size, USB_DESC_NO_TYPECHECK)) < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID DEVICE", hid->dev_name);
		goto ret2;
	}
	/*{
		int h = creat("repdesc.usb", 0600);
		write(h, hid->report_desc, hid->report_desc_size);
		close(h);
	}*/
	if (__unlikely(r = HID_PARSE(hid))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID REPORT DESCRIPTOR", hid->dev_name);
		goto ret2;
	}
	hid->ctrl = USB$GET_DEFAULT_ENDPOINT(iface);
	if (__unlikely(__IS_ERR(hid->intr_in = USB$FIND_ENDPOINT(iface, USB_EP_INTR_IN, N_REQUESTS, hid->in_rq_size)))) {
		r = __PTR_ERR(hid->intr_in);
		hid->intr_in = NULL;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET INPUT INTERRUPT ENDPOINT: %s", hid->dev_name, strerror(-r));
		goto ret2;
	}
	for (i = 0; i < N_REQUESTS; i++) {
		HID_USBREQ *req = &hid->req[i];
		req->hid = hid;
		req->urq.fn = HID_GOT_REPORT;
		req->urq.v.vspace = &KERNEL$VIRTUAL;
		mrq.size = (hid->in_rq_size + 3) & ~3;
		SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(mrq.status < 0)) {
			r = mrq.status;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", hid->dev_name, strerror(-r));
			goto ret2;
		}
		req->data = mrq.ptr;
	}
	if (hid->n_keys) {
		mrq.size = hid->n_keys * sizeof(key_struct);
		SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(mrq.status < 0)) {
			r = mrq.status;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", hid->dev_name, strerror(-r));
			goto ret2;
		}
		hid->current_keys = mrq.ptr;
		memset(hid->current_keys, 0, hid->n_keys * sizeof(key_struct));
		mrq.size = hid->n_keys * sizeof(key_struct);
		SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(mrq.status < 0)) {
			r = mrq.status;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", hid->dev_name, strerror(-r));
			goto ret2;
		}
		hid->last_keys = mrq.ptr;
		memset(hid->last_keys, 0, hid->n_keys * sizeof(key_struct));
	}
	RAISE_SPL(SPL_HID);
	HID_SET_IDLE(hid, 0);
	LOWER_SPL(SPL_ZERO);

	if (__likely(hid->flags & HID_APP_MOUSE)) {
		if (__unlikely(r = KERNEL$DCALL(hid->kbd_dev_name, "KEYBOARD", KEYBOARD_MOUSE_GET, &hid->mouse_upcall))) {
			hid->mouse_upcall = NULL;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ATTACH MOUSE TO KEYBOARD DRIVER: %s", hid->dev_name, strerror(-r));
			goto ret2;
		}
	}
	if (hid->flags & HID_APP_KEYBOARD) {
		if (__unlikely(r = KERNEL$DCALL(hid->kbd_dev_name, "KEYBOARD", KEYBOARD_EXTKBD_GET, &hid->kbd_upcall))) {
			hid->kbd_upcall = NULL;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ATTACH KEYBOARD TO KEYBOARD DRIVER: %s", hid->dev_name, strerror(-r));
			goto ret2;
		}
	}

	r = KERNEL$REGISTER_DEVICE(hid->dev_name, "USBHID.SYS", 0, hid, NULL, NULL, NULL, NULL, HID_UNLOAD, &hid->lnte, hid->kbd_dev_name, USB$GET_CONTROLLER_NAME(iface), NULL);
	if (__unlikely(r < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", hid->dev_name, strerror(-r));
		goto ret2;
	}
	hid->dlrq = KERNEL$TSR_IMAGE();

	RAISE_SPL(SPL_HID);
	if (__likely(hid->in_rq_size)) HID_SENDRQ(hid, &hid->req[0], 0);
	LOWER_SPL(SPL_ZERO);

	strlcpy(KERNEL$ERROR_MSG(), hid->dev_name, __MAX_STR_LEN);
	USB$FREE_PARAMS(args);
	return 0;

	ret2:
	HID_FREE(hid);
	ret1:
	USB$DETACH_DRIVER(iface);
	ret0:
	USB$FREE_PARAMS(args);
	return r;
}

static void HID_FREE(HID *hid)
{
	unsigned i, j;
	int r;
	if (__likely(hid->mouse_upcall != NULL)) if (__unlikely(r = KERNEL$DCALL(hid->kbd_dev_name, "KEYBOARD", KEYBOARD_MOUSE_PUT))) KERNEL$SUICIDE("HID_FREE: CAN'T UNREGISTER MOUSE AT KEYBOARD DEVICE %s: %s", hid->kbd_dev_name, strerror(-r));
	if (hid->kbd_upcall) if (__unlikely(r = KERNEL$DCALL(hid->kbd_dev_name, "KEYBOARD", KEYBOARD_EXTKBD_PUT))) KERNEL$SUICIDE("HID_FREE: CAN'T UNREGISTER KEYBOARD AT KEYBOARD DEVICE %s: %s", hid->kbd_dev_name, strerror(-r));
	free(hid->report_desc);
	for (i = 0; i < HID_N_REPORT_TYPES; i++) if (__likely(hid->report[i] != NULL)) {
		for (j = 0; j < HID_N_REPORTS; j++) FREE_DOUBLE_LIST(&hid->report[i][j]);
		free(hid->report[i]);
	}
	FREE_DOUBLE_LIST(&hid->collections);
	for (i = 0; i < N_REQUESTS; i++) free(hid->req[i].data);
	free(hid->current_keys);
	free(hid->last_keys);
	free(hid);
}

static int HID_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	HID *hid = p;
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(hid->lnte, argv))) return r;
	RAISE_SPL(SPL_HID);
	HID_DISABLE(hid);
	while (1) {
		int i;
		int active = 0;
		for (i = 0; i < N_REQUESTS; i++) {
			if (hid->req[i].posted) {
				RAISE_SPL(SPL_USB);
				hid->intr_in->cancel(&hid->req[i].urq);
				LOWER_SPL(SPL_HID);
				active = 1;
			}
		}
		if (!active) break;
		KERNEL$SLEEP(1);
	}
	LOWER_SPL(SPL_ZERO);
	USB$DETACH_DRIVER(hid->iface);
	*release = hid->dlrq;
	HID_FREE(hid);
	return 0;
}
