#include <STDIO.H>
#include <STDLIB.H>
#include <ERRNO.H>
#include <SYS/TYPES.H>
#include <SPAD/LIST.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYSLOG.H>

#include "ROOTER.H"

/* ROOTER.CFG:
user=username,password=password,authority=fs:/bla/ble,authority=fs:/bla/ble2,cncheck=boo.cz,cnexact,device=auth$device,algorithm=eap-ttls,inner_algorithm=pap
*/

#define LINE_BUFFER	1024

DECL_LIST(config);

static __u64 config_seq = 0;

static void free_stringlist(XLIST_HEAD *head)
{
	while (!XLIST_EMPTY(head)) {
		struct stringlist *sl = LIST_STRUCT(head->next, struct stringlist, list);
		DEL_FROM_LIST(&sl->list);
		free(sl);
	}
}

static void free_config_entry(struct config *c)
{
	free(c->user);
	free(c->password);
	if (c->cert_store) {
		X509_STORE_free(c->cert_store);
	}
	free(c->cncheck);
	free(c->inner_algorithm);
	DEL_FROM_LIST(&c->list);
	free_stringlist(&c->devices);
	free_stringlist(&c->algorithms);
	free(c);
}

void clear_config(void)
{
	while (!LIST_EMPTY(&config)) free_config_entry(LIST_STRUCT(config.next, struct config, list));
}

static int ca_error;
static X509_STORE *tmp_store;

static int get_ca(const char *string, const char *end, void *null)
{
	char *file;
	X509_LOOKUP *lookup;
	int r;
	if (__unlikely(!string) || __unlikely(string == end)) return -1;
	file = xmalloc(end - string + 1);
	memcpy(file, string, end - string);
	file[end - string] = 0;
	lock_ssl();
	new_x509_store(&tmp_store);
	if (__unlikely(!tmp_store)) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, file, "TLS: CAN'T CREATE X509 STORE");
		print_ssl_errors();
		ca_error = 1;
		goto ret;
	}
	lookup = X509_STORE_add_lookup(tmp_store, X509_LOOKUP_file());
	if (__unlikely(!lookup)) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, file, "TLS: CAN'T CREATE X509 STORE LOOKUP");
		print_ssl_errors();
		ca_error = 1;
		goto ret;
	}
	r = X509_LOOKUP_load_file(lookup, file, X509_FILETYPE_PEM);
	if (__unlikely(!r)) r = X509_LOOKUP_load_file(lookup, file, X509_FILETYPE_ASN1);
	if (__unlikely(!r)) {
		fprintf(stderr, "FAILED TO LOAD SERVER CERTIFICATE %s\n", file);
		print_ssl_errors();
		ca_error = 1;
		goto ret;
	}
	clear_ssl_errors();
	ret:
	unlock_ssl();
	free(file);
	return 0;
}

static int add_to_stringlist(XLIST_HEAD *head, const char *string, const char *end)
{
	struct stringlist *sl;
	if (__unlikely(!string) || __unlikely(string == end)) return -1;
	sl = xmalloc(sizeof(struct stringlist) + (end - string));
	memcpy(sl->string, string, end - string);
	sl->string[end - string] = 0;
	ADD_TO_XLIST(head, &sl->list);
	return 0;
}

static DECL_XLIST(devices);
static DECL_XLIST(algorithms);

static int get_device(const char *string, const char *end, void *null)
{
	return add_to_stringlist(&devices, string, end);
}

static int get_algorithm(const char *string, const char *end, void *null)
{
	return add_to_stringlist(&algorithms, string, end);
}

static int fixstring(char *r)
{
	char *w;
	if (!r) return 0;
	w = r;
	do {
		while (__unlikely(*r == '%')) {
			int q = 0;
			if (r[1] >= '0' && r[1] <= '9') q += r[1] - '0';
			else if (r[1] >= 'A' && r[1] <= 'F') q += r[1] - 'A' + 10;
			else if (r[1] >= 'a' && r[1] <= 'f') q += r[1] - 'a' + 10;
			else return -1;
			q *= 16;
			if (r[2] >= '0' && r[2] <= '9') q += r[2] - '0';
			else if (r[2] >= 'A' && r[2] <= 'F') q += r[2] - 'A' + 10;
			else if (r[2] >= 'a' && r[2] <= 'f') q += r[2] - 'a' + 10;
			else return -1;
			if (__unlikely(!q)) return -1;
			*w++ = q;
			r += 3;
		}
	} while ((*w++ = *r++));
	return 0;
}

int read_config(char *file)
{
	char line_buffer[LINE_BUFFER];
	FILE *f;
	int clr_cfg = 1;
	int bad_syntax = 0;
	int bad_ca = 0;
	int line = 0;
	f = fopen(file, "r");
	if (__unlikely(!f)) return -errno;
	while (fgets(line_buffer, LINE_BUFFER, f)) {
		const char *lbp;
		struct config *entry;
		char *user = NULL, *password = NULL, *cncheck = NULL, *inner_algorithm = NULL;
		int cnexact = 0;
		int hash = 0;
		char *q;
		int state;
		static const struct __param_custom ca_custom = { get_ca, NULL };
		static const struct __param_custom dev_custom = { get_device, NULL };
		static const struct __param_custom alg_custom = { get_algorithm, NULL };
		static const struct __param_table params[10] = {
			"USER", __PARAM_NEWSTRING, 1, __MAX_STR_LEN,
			"PASSWORD", __PARAM_NEWSTRING, 1, __MAX_STR_LEN,
			"AUTHORITY", __PARAM_CUSTOM, 0, 0,
			"CNCHECK", __PARAM_NEWSTRING, 1, __MAX_STR_LEN,
			"CNEXACT", __PARAM_BOOL, ~0, 1,
			"HASH", __PARAM_BOOL, ~0, 1,
			"DEVICE", __PARAM_CUSTOM, 0, 0,
			"ALGORITHM", __PARAM_CUSTOM, 0, 0,
			"INNER_ALGORITHM", __PARAM_NEWSTRING, 1, __MAX_STR_LEN,
			NULL, 0, 0, 0,
		};
		void *vars[10];
		vars[0] = &user;
		vars[1] = &password;
		vars[2] = (void *)&ca_custom;
		vars[3] = &cncheck;
		vars[4] = &cnexact;
		vars[5] = &hash;
		vars[6] = (void *)&dev_custom;
		vars[7] = (void *)&alg_custom;
		vars[8] = &inner_algorithm;
		vars[9] = NULL;
		line++;
		if (__unlikely((q = strchr(line_buffer, '\r')) != NULL)) *q = 0;
		if (__likely((q = strchr(line_buffer, '\n')) != NULL)) *q = 0;
		if (line_buffer[0] == '#' || line_buffer[0] == ';') continue;
		if (clr_cfg) clear_config(), clr_cfg = 0;
		ca_error = 0;
		tmp_store = NULL;
		lbp = line_buffer;
		state = 0;
		if (__unlikely(__parse_extd_param(&lbp, &state, params, vars, NULL, NULL, NULL, NULL))) {
			syn_err:
			fprintf(stderr, "%s: SYNTAX ERROR AT LINE %d\n", file, line);
			bad_syntax = 1;
			err_cont:
			if (tmp_store) X509_STORE_free(tmp_store);
			free_stringlist(&devices);
			free_stringlist(&algorithms);
			free(user);
			free(password);
			free(cncheck);
			free(inner_algorithm);
			continue;
		}
		if (__unlikely(fixstring(user))) goto syn_err;
		if (__unlikely(fixstring(password))) goto syn_err;
		if (__unlikely(fixstring(cncheck))) goto syn_err;
		debug(("user: %s, pass: %s\n", user, password));
		if (__unlikely(ca_error)) {
			bad_ca = 1;
			goto err_cont;
		}
		entry = xmalloc(sizeof(struct config));
		memset(entry, 0, sizeof(struct config));
		entry->seq = config_seq;
		config_seq = (config_seq + 1) & (((__u64)1 << 63) - 1);
		entry->user = user;
		entry->password = password;
		entry->cncheck = cncheck;
		entry->inner_algorithm = inner_algorithm;
		entry->cert_store = tmp_store;
		entry->cnexact = cnexact;
		entry->hash = hash;
		INIT_XLIST(&entry->devices);
		while (!XLIST_EMPTY(&devices)) {
			struct stringlist *sl = LIST_STRUCT(devices.next, struct stringlist, list);
			DEL_FROM_LIST(&sl->list);
			ADD_TO_XLIST(&entry->devices, &sl->list);
		}
		INIT_XLIST(&entry->algorithms);
		while (!XLIST_EMPTY(&algorithms)) {
			struct stringlist *sl = LIST_STRUCT(algorithms.next, struct stringlist, list);
			DEL_FROM_LIST(&sl->list);
			ADD_TO_XLIST(&entry->algorithms, &sl->list);
		}
		ADD_TO_LIST_END(&config, &entry->list);
	}
	if (__unlikely(ferror(f))) {
		int r = -errno;
		fclose(f);
		return r;
	}
	if (clr_cfg) clear_config();
	fclose(f);
	return bad_syntax ? -EBADSYN : bad_ca ? -EINVAL : 0;
}
