/* A Methodology for the Typical Unification of Access Points and Redundancy */

#include <SPAD/LIBC.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>
#include <STRING.H>
#include <STDLIB.H>
#include <UNISTD.H>
#include <VALUES.H>

#include "ROOTER.H"

int errorlevel = 0;

unsigned priority = 100;

void *xmalloc(size_t size)
{
	void *r = malloc(size);
	if (__unlikely(!r)) {
		__critical_printf("OUT OF MEMORY (CAN'T ALLOCATE %ld BYTES)\n", (unsigned long)size);
		abort();
	}
	return r;
}

void *xrealloc(void *p, size_t size)
{
	void *r = realloc(p, size);
	if (__unlikely(!r)) {
		__critical_printf("OUT OF MEMORY (CAN'T REALLOCATE %ld TO %ld BYTES)\n", (unsigned long)__alloc_size(p), (unsigned long)size);
		abort();
	}
	return r;
}

static DECL_XLIST(instances);

static pthread_attr_t thread_attr;
static pthread_mutexattr_t thread_mutexattr;

static void create_instance(char *filename)
{
	struct instance *inst;
	int r;
	int h = open(filename, O_RDWR);
	if (__unlikely(h == -1)) return;
	inst = xmalloc(sizeof(struct instance) + strlen(filename));
	memset(inst, 0, sizeof(struct instance));
	inst->h = h;
	strcpy(inst->name, filename);
	ADD_TO_XLIST(&instances, &inst->list);
	r = pthread_create(&inst->pthread, &thread_attr, do_instance, inst);
	if (__unlikely(r)) {
		__critical_printf("ROOTER: UNABLE TO CREATE THREAD FOR %s\n", filename);
		abort();
	}
}

static char *pwd_file = "ROOTER:/ROOTER.CFG";
static char *device = NULL;

pthread_mutex_t ssl_lock;

static char logical_wake = 0;
static char logical_terminate = 0;
static WQ_DECL(logical_wq, "ROOTER$LOGICAL_WQ");
static IORQ logical_wait_rq;

static DECL_IOCALL(logical_changed, SPL_SHELL, IORQ)
{
	logical_wake = 1;
	WQ_WAKE_ALL(&logical_wq);
	if (!logical_terminate) WQ_WAIT(&KERNEL$LOGICAL_WAIT, &logical_wait_rq, logical_changed);
	else logical_terminate = 2;
	RETURN;
}

static const struct __param_table params[6] = {
	"", __PARAM_STRING, 1, __MAX_STR_LEN,
	"DEVICE", __PARAM_STRING, 1, __MAX_STR_LEN,
	"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
	"LOG_WARNINGS", __PARAM_BOOL, ~0, 2,
	"PRIORITY", __PARAM_UNSIGNED_INT, 0, MAXINT,
	NULL, 0, 0, 0,
};

static void * const vars[6] = {
	&pwd_file,
	&device,
	&errorlevel,
	&errorlevel,
	&priority,
	NULL,
};

int main(int argc, const char * const argv[])
{
	const char * const *arg = argv;
	int state = 0;
	int r, rr;
	if (__unlikely(((SSLeay() ^ OPENSSL_VERSION_NUMBER) & ~0xff0L) != 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: OPENSSL VERSION MISMATCH, BUILT AGAINS %lX, YOU HAVE %lX.", (unsigned long)OPENSSL_VERSION_NUMBER, (unsigned long)SSLeay());
		return -EINVAL;
	}
	chdir("ROOTER:/");
	if (__unlikely(r = -pthread_mutexattr_init(&thread_mutexattr))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: UNABLE TO INITIALIZE MUTEX ATTRIBUTES");
		goto ret00;
	}
	if (__unlikely(r = -pthread_mutexattr_settype(&thread_mutexattr, PTHREAD_MUTEX_ERRORCHECK))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: UNABLE TO SET CHECKING ATTRIBUTE");
		goto ret01;
	}
	if (__unlikely(r = -pthread_mutex_init(&ssl_lock, &thread_mutexattr))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: UNABLE TO INITIALIZE SSL LOCK");
		goto ret01;
	}
	if (__unlikely(r = -pthread_attr_init(&thread_attr))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: UNABLE TO INITIALIZE PTHREAD ATTRIBUTES");
		goto ret02;
	}
	if (__unlikely(r = -pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: UNABLE TO SET DETACH ATTRIBUTE");
		goto ret03;
	}
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		badsyn:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret1;
	}
	if (__unlikely(!pwd_file)) goto badsyn;
	if (__unlikely(r = read_config(pwd_file))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ROOTER: CAN'T READ CONFIGURATION FILE %s: %s", pwd_file, strerror(-r));
		goto ret1;
	}

	logical_wait_rq.status = RQS_PROCESSING;
	WQ_WAIT(&KERNEL$LOGICAL_WAIT, &logical_wait_rq, logical_changed);

	repeat:
	if (!device) {
		LOGICAL_LIST_REQUEST lrq;
		unsigned i;
		struct instance *inst;
		lrq.prefix = "AUTH$";
		SYNC_IO(&lrq, KERNEL$LIST_LOGICALS);
		if (__unlikely((r = lrq.status) < 0)) {
			fprintf(stderr, "ROOTER: CAN'T GET LOGICAL LIST: %s\n", strerror(-r));
			goto ret2;
		}
		lock_ssl();
		for (i = 0; i < lrq.n_entries; i++) {
			char *name;
			XLIST_FOR_EACH(inst, &instances, struct instance, list) {
				unsigned len = strcspn(inst->name, ":");
				if (len == strlen(lrq.entries[i]->name) && !_memcasecmp(lrq.entries[i]->name, inst->name, len)) goto found;
			}
			name = xmalloc(strlen(lrq.entries[i]->name) + 3);
			stpcpy(stpcpy(name, lrq.entries[i]->name), ":/");
			create_instance(name);
			free(name);
			found:;
		}
		unlock_ssl();
		KERNEL$FREE_LOGICAL_LIST(&lrq);
	} else {
		lock_ssl();
		if (XLIST_EMPTY(&instances)) create_instance(device);
		unlock_ssl();
	}
	RAISE_SPL(SPL_SHELL);
	if (!logical_wake) WQ_WAIT_SYNC(&logical_wq);
	logical_wake = 0;
	LOWER_SPL(SPL_ZERO);
	goto repeat;
	
	ret2:
	RAISE_SPL(SPL_SHELL);
	logical_terminate = 1;
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	while (logical_terminate != 2) KERNEL$SLEEP(1);
	LOWER_SPL(SPL_ZERO);
	ret1:
	ret03:
	if (__unlikely(rr = -pthread_attr_destroy(&thread_attr))) {
		__critical_printf("ROOTER: UNABLE TO DEINITIALIZE PTHREAD ATTRIBUTES: %s\n", strerror(-r));
		if (!r) r = rr;
	}
	ret02:
	if (__unlikely(rr = -pthread_mutex_destroy(&ssl_lock))) {
		__critical_printf("ROOTER: UNABLE TO DEINITIALIZE SSL LOCK: %s\n", strerror(-r));
		if (!r) r = rr;
	}
	ret01:
	if (__unlikely(rr = -pthread_mutexattr_destroy(&thread_mutexattr))) {
		__critical_printf("ROOTER: UNABLE TO DEINITIALIZE MUTEX ATTRIBUTES: %s\n", strerror(-r));
		if (!r) r = rr;
	}
	ret00:
	return r;
}
