#include <SPAD/AC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/DL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/ETHERNET.H>
#include <ARCH/IPCHECKS.H>
#include <SPAD/SYNC.H>

#include <SPAD/PKT.H>
#include "NET.H"

int NET$LOCAL_HANDLE;

static PROTOCOL_HANDLER *PROTOCOL_SWITCH[NET_PROTOCOL_N];

static void DUMMY_PROTOCOL(PACKET *RQ);
static IO_STUB NET_PACKET;
static IO_STUB NET_IOCTL;

static const HANDLE_OPERATIONS NET_OPERATIONS = {
	SPL_X(SPL_NET),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	NET_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	NET_PACKET,		/* pktio */
};

static void NET_NEED_CHECKSUM(PACKET *RQ);
static void NET_NEED_DEFRAG(PACKET *RQ);

static __finline__ void NET_PACKET_CORE(PACKET *RQ)
{
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_NET));
	if (__unlikely(RQ->flags & PKT_OUTPUT_CHECKSUM)) {
		if (__likely(RQ->flags & PKT_TCPUDP_CHECKSUM_OK)) {
			RQ->flags &= ~PKT_OUTPUT_CHECKSUM;
			goto cont;
		}
		NET_NEED_CHECKSUM(RQ);
	} else cont: if (__likely(!RQ->v.len)) {
		PROTOCOL_SWITCH[PKT_PROTOCOL(RQ)](RQ);
	} else {
		NET_NEED_DEFRAG(RQ);
	}
}

static DECL_IOCALL(NET_PACKET_WAKEUP, SPL_NET, PACKET)
{
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
		WQ_WAIT(&KERNEL$LOCKUP_EVENTS, RQ, NET_PACKET_WAKEUP);
		RETURN;
	}
	NET_PACKET_CORE(RQ);
	RETURN;
}

static DECL_IOCALL(NET_PACKET, SPL_NET, PACKET)
{
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS)) {
		WQ_WAIT(&KERNEL$LOCKUP_EVENTS, RQ, NET_PACKET_WAKEUP);
		RETURN;
	}
	NET_PACKET_CORE(RQ);
	RETURN;
}

static void NET_NEED_CHECKSUM(PACKET *RQ)
{
	int r;
	if (__unlikely(r = NET$CHECKSUM_PACKET(RQ))) {
		RQ->status = r;
		CALL_PKT(RQ);
	} else if (__likely(!RQ->v.len)) {
		PROTOCOL_SWITCH[PKT_PROTOCOL(RQ)](RQ);
	} else {
		NET_NEED_DEFRAG(RQ);
	}
}

static void NET_NEED_DEFRAG(PACKET *RQ)
{
	PACKET *new;
	if (__likely((new = NET$CLONE_PACKET(RQ, &NET$PKTPOOL)) != NULL)) {
		PROTOCOL_SWITCH[PKT_PROTOCOL(new)](new);
	} else {
		NET$OOM();
		RQ->status = -ENOMEM;
		CALL_PKT(RQ);
	}
}

PACKET *NET$CLONE_PACKET(PACKET *p, PKTPOOL *pool)
{
	vspace_unmap_t *unmap;
	__u8 *ind;
	PACKET *new;
	unsigned newlen;
	static __u8 aux_indirect[MAX_PACKET_LENGTH_TOP];
	static VBUF v = { SPL_X(SPL_NET), MAX_PACKET_LENGTH_TOP, aux_indirect };
	newlen = p->data_length + p->v.len;
	if (__unlikely(newlen > NET$MAX_PACKET_LENGTH))
		KERNEL$SUICIDE("NET$CLONE_PACKET: TOO LONG PACKET RECEIVED, DIRECT %u, INDIRECT %ld, MAX %d", p->data_length, p->v.len, NET$MAX_PACKET_LENGTH);
	ALLOC_PACKET(new, newlen, pool, SPL_NET, return NULL);
	new->fn = NET$FREE_PACKET;
	new->id = p->id;
	new->sender_data = &KERNEL$LIST_END;
	new->flags |= p->flags & (PKT_IPHDR_CHECKSUM_OK | PKT_TCPUDP_CHECKSUM_OK | PKT_OUTPUT_CHECKSUM | PKT_INPUT_CHECKSUM);
	new->checksum.u = p->checksum.u;
	new->data_length = newlen;
	new->addrlen = p->addrlen;
	memcpy(new->data, p->data, LL_HEADER + p->data_length);
	ind = NET$MAP_INDIRECT_PACKET(&v, &p->v, &unmap);
	if (__unlikely(__IS_ERR(ind))) {
		p->flags |= PKT_PAGING_ERROR;
		FREE_PACKET(new, &NET$PKTPOOL, SPL_NET);
		return NULL;
	}
	memcpy(new->data + LL_HEADER + p->data_length, ind, p->v.len);
	unmap(ind);
	p->status = 0;
	CALL_PKT(p);
	return new;
}

DECL_AST(NET$FREE_PACKET, SPL_NET, PACKET)
{
	PKT_AST_ENTER(RQ);
#if __DEBUG >= 1
	if (__unlikely((long)*(unsigned long *)RQ->sender_data <= 0) && RQ->sender_data != &KERNEL$LIST_END)
		KERNEL$SUICIDE("NET$FREE_PACKET: OUTSTANDING PACKET COUNT %ld", (long)*(unsigned long *)RQ->sender_data);
#endif
	(*(unsigned long *)RQ->sender_data)--;
	TRYFREE_PACKET(RQ, &NET$PKTPOOL,
		FREE_PACKET(RQ, &NET$PKTPOOL_LONG_TERM, SPL_NET);
	);
	RETURN;
}

int NET$REGISTER_PROTOCOL(PROTOCOL_HANDLER *p, unsigned type)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_NET), spl)))
		KERNEL$SUICIDE("NET$REGISTER_PROTOCOL AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_NET);
	if (__unlikely(PROTOCOL_SWITCH[type] != DUMMY_PROTOCOL)) {
		LOWER_SPLX(spl);
		return -EBUSY;
	}
	PROTOCOL_SWITCH[type] = p;
	LOWER_SPLX(spl);
	return 0;
}

void NET$UNREGISTER_PROTOCOL(unsigned type)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_NET), spl)))
		KERNEL$SUICIDE("NET$REGISTER_PROTOCOL AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_NET);
	if (__unlikely(PROTOCOL_SWITCH[type] == DUMMY_PROTOCOL))
		KERNEL$SUICIDE("NET$UNREGISTER_PROTOCOL: UNREGISTERING ALREADY FREE PROTOCOL %04X", type);
	PROTOCOL_SWITCH[type] = DUMMY_PROTOCOL;
	LOWER_SPLX(spl);
}

static void DUMMY_PROTOCOL(PACKET *RQ)
{
	unsigned typelen = __16BE2CPU(ETH_HEADER(RQ)->typelen);
	if (__likely(typelen <= ETHERNET_MTU)) {
		if (PROTOCOL_SWITCH[NET_PROTOCOL_802_3] != DUMMY_PROTOCOL) {
			PROTOCOL_SWITCH[NET_PROTOCOL_802_3](RQ);
			return;
		}
	}
	RQ->status = -ENOENT;
	CALL_PKT(RQ);
}

void PROTOCOL_INIT(void)
{
	int i;
	for (i = 0; i < NET_PROTOCOL_N; i++) {
		PROTOCOL_SWITCH[i] = DUMMY_PROTOCOL;
		KERNEL$THREAD_MAY_BLOCK();
	}
}

static DECL_IOCALL(NET_IOCTL, SPL_NET, IOCTLRQ)
{
	IF_TYPE t;
	HANDLE *h = RQ->handle;
	int r;
	if (__unlikely(h->op != &NET_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_NET));
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE:
			memset(&t, 0, sizeof t);
			t.mtu = NET$MAX_PACKET_LENGTH;
			t.addrlen = 0;
			strlcpy((char *)t.type, "LOOPBACK", sizeof(t.type));
			t.id = IDTYPE_LO;
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		default:
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
	}
}

static void net_init_root(HANDLE *h, void *data)
{
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = NULL;
	h->op = &NET_OPERATIONS;
}

void PROTOCOL_DONE(void)
{
	int i;
	for (i = 0; i < NET_PROTOCOL_N; i++) if (PROTOCOL_SWITCH[i] != DUMMY_PROTOCOL) KERNEL$SUICIDE("PROTOCOL_DONE: PROTOCOL %04X DID NOT UNREGISTER ITS HANDLER", i);
}

static THREAD_RQ trq;
static DLINITRQ *dlinitrq;
static void *lnte;

static long PACKET_DLL_INIT(void *); 
static AST_STUB PACKET_DLL_INITIALIZED;

void CHECKSUM_INIT(void);

DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	dlinitrq = RQ;
	trq.fn = PACKET_DLL_INITIALIZED;
	trq.thread_main = PACKET_DLL_INIT;
	trq.p = NULL;
	trq.error = dlinitrq->error;
	trq.cwd = NULL;
	trq.std_in = -1;
	trq.std_out = -1;
	trq.std_err = -1;
	trq.dlrq = NULL;
	trq.thread = NULL;
	trq.spawned = 0;
	RETURN_IORQ_CANCELABLE(&trq, KERNEL$THREAD, RQ);
}

static long PACKET_DLL_INIT(void *p)
{
	int r;
	OPENRQ openrq;
	RAISE_SPL(SPL_NET);
	CHECKSUM_INIT();
	if ((r = PACKET_INIT())) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT ALLOCATE PACKET BUFFERS: %s", strerror(-r));
		return r;
	}
	PROTOCOL_INIT();
	SOCKET_INIT();
	LOWER_SPL(SPL_ZERO);

	r = KERNEL$REGISTER_DEVICE(NET_INPUT_DEVICE, "NET.DLL", 0, NULL, net_init_root, NULL, NULL, NULL, NULL, &lnte, ":DMAN_SKIP", NULL);
	if (r < 0) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT REGISTER DEVICE %s: %s", NET_INPUT_DEVICE, strerror(-r));
		error_done:
		RAISE_SPL(SPL_NET);
		SOCKET_DONE();
		PROTOCOL_DONE();
		PACKET_DONE();
		return r;
	}

	openrq.flags = O_RDWR | _O_NOPOSIX | _O_NOOPEN;
	openrq.path = NET_INPUT_DEVICE ":/";
	openrq.cwd = NULL;
	SYNC_IO(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		r = openrq.status;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT OPEN DEVICE %s: %s", NET_INPUT_DEVICE, strerror(-r));
		KERNEL$UNREGISTER_DEVICE(lnte, 0);
		goto error_done;
	}

	NET$LOCAL_HANDLE = openrq.status;
	return 0;
}

static DECL_AST(PACKET_DLL_INITIALIZED, SPL_DEV, THREAD_RQ)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), dlinitrq);
	if (__unlikely(trq.status < 0)) {
		if (!trq.spawned) _snprintf(dlinitrq->error, __MAX_STR_LEN, "COULD NOT SPAWN INIT THREAD: %s", strerror(-trq.status));
		dlinitrq->status = trq.status;
		RETURN_AST(dlinitrq);
	}
	dlinitrq->status = 0;
	RETURN_AST(dlinitrq);
}

DECL_IOCALL(DLL$UNLOAD, SPL_DEV, DLINITRQ)
{
	KERNEL$FAST_CLOSE(NET$LOCAL_HANDLE);
	KERNEL$UNREGISTER_DEVICE(lnte, 0);
	RAISE_SPL(SPL_NET);
	SOCKET_DONE();
	PROTOCOL_DONE();
	PACKET_DONE();
	LOWER_SPL(SPL_DEV);
	RQ->status = 0;
	RETURN_AST(RQ);
}
