#include <SPAD/PCI.H>
#include <SPAD/PKT.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ALLOC.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/PKT.H>
#include <SPAD/IOCTL.H>
#include <STRING.H>
#include <VALUES.H>
#include <ARCH/IO.H>
#include <SPAD/IRQ.H>

#include "3C.H"
#include "3CREG.H"

			/* !!! SMPFIX: share it */
static __u8 tags = 0;	/* bit field of used tags */

static const struct {
	__u16 id;
	char *string_a;
	char *string_b;
} isa_cards[] = {
	{ 0x9150, "3C509", "3C509B" },
	{ 0x9050, "3C509-TP", "3C509B-TP" },
	{ 0x9450, "3C509-COMBO", "3C509B-COMBO" },
	{ 0x9550, "3C509-TPO", "3C509B-TPO" },
	{ 0x9850, "3C509-TPC", "3C509B-TPC" },
	{ 0x627c, "3C529", "3C529" },
	{ 0x627d, "3C529-TP", "3C529-TP" },
	{ 0x9350, "3C579", "3C579" },
	{ 0x9250, "3C579-TP", "3C579-TP" },
	{ 0x9058, "3C589-TP/COMBO", "3C589B-TP/COMBO" },
	{ 0, NULL, NULL },
};

static const struct pci_id_s pci_cards[] = {
	{ 0x10B7, 0x9000, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900-TPO ETHERLINK XL", BOOMERANG },
	{ 0x10B7, 0x9001, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900-COMBO ETHERLINK XL", BOOMERANG },
	{ 0x10B7, 0x9004, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-TPO ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9005, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-COMBO ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9006, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-TPC ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x900A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-FL ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9050, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905-TX FAST ETHERLINK XL", BOOMERANG | MII },
	{ 0x10B7, 0x9051, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905-T4 FAST ETHERLINK XL", BOOMERANG | MII },
	{ 0x10B7, 0x9055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-TX FAST ETHERLINK XL", CYCLONE | NWAY | MII_PREAMBLE },
	{ 0x10B7, 0x9056, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-T4 FAST ETHERLINK XL", CYCLONE | NWAY | MII_PREAMBLE },
	{ 0x10B7, 0x9058, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-COMBO FAST ETHERLINK XL", CYCLONE | NWAY },
	{ 0x10B7, 0x905A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-FX/SC FAST ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9200, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905C-TX FAST ETHERLINK XL", TORNADO | NWAY | MII_PREAMBLE },
	{ 0x10B7, 0x9201, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C920B-EMB INTEGRATED FAST ETHERLINK XL", TORNADO | NWAY },
	{ 0x10B7, 0x9202, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C920B-EMB-WNM INTEGRATED FAST ETHERLINK XL", TORNADO | MII },
	{ 0x10B7, 0x9210, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C920B-EMB-WNM INTEGRATED FAST ETHERLINK XL", TORNADO | NWAY },
	{ 0x10B7, 0x9800, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C980 FAST ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9805, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C980C FAST ETHERLINK XL", CYCLONE | NWAY },
	{ 0x10B7, 0x7646, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3CSOHO100-TX OFFICECONNECT", CYCLONE | NWAY },
	{ 0x10B7, 0x4500, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C450-TX HOMECONNECT", TORNADO | NWAY },
	{ 0x10B7, 0x5055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C555 FAST ETHERLINK XL", CYCLONE | EEPROM_8BIT },
	{ 0x10B7, 0x6055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C556 FAST ETHERLINK XL", TORNADO | NWAY | EEPROM_8BIT | CARDBUS | INVERT_MII_PWR | NO_EEPROM_RELOAD },
	{ 0x10B7, 0x6056, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C556B FAST ETHERLINK XL", TORNADO | NWAY | EEPROM_OFFSET_30 | CARDBUS | INVERT_MII_PWR | WNO_XCVR_PWR | NO_EEPROM_RELOAD },
	{ 0x10B7, 0x5B57, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575TX FAST ETHERLINK XL", BOOMERANG | MII | EEPROM_8BIT },
	{ 0x10B7, 0x5057, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575TX FAST ETHERLINK XL", BOOMERANG | MII | EEPROM_8BIT },
	{ 0x10B7, 0x5157, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575B FAST ETHERLINK XL", CYCLONE | NWAY | CARDBUS | EEPROM_8BIT | INVERT_LED_PWR },
	{ 0x10B7, 0x5257, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575C FAST ETHERLINK XL", TORNADO | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | MAX_COLLISION_RESET },
	{ 0x10B7, 0x6560, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656 FAST ETHERLINK XL", CYCLONE | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | INVERT_LED_PWR },
	{ 0x10B7, 0x6562, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656B FAST ETHERLINK XL", CYCLONE | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | INVERT_LED_PWR },
	{ 0x10B7, 0x6564, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656C FAST ETHERLINK XL", TORNADO | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | MAX_COLLISION_RESET },
	{ 0x10B7, 0x6564, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656C FAST ETHERLINK XL", 0 },
	{ 0x10B7, 0x1201, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C982 HYDRA PORT A", TORNADO | NWAY },
	{ 0x10B7, 0x1202, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C982 HYDRA PORT B", TORNADO | NWAY },
	{ 0, }
};

#define MEDIA_XL		0x01
#define MEDIA_V2		0x02
#define MEDIA_NEED_MII		0x04
#define MEDIA_INTERNAL_MII	0x08
#define MEDIA_NEED_COAX_CMD	0x10
#define MEDIA_CAPABLE_FDX	0x20
#define MEDIA_LINK_DETECT	0x40
#define MEDIA_100		0x80

static const struct {
	char *name;
	int flags;

	/* EL */
	__u16 config_control;
	__u16 address_config;

	/* XL */
	unsigned internal_config;
	__u16 media_options;
	__u16 media_bits;
} media[] = {
	{
		"AUTO-NEGOTIATION",
		MEDIA_XL | MEDIA_V2 | MEDIA_NEED_MII | MEDIA_INTERNAL_MII | MEDIA_CAPABLE_FDX | MEDIA_LINK_DETECT,
		0,
		0,
		X2_InternalConfig_XcvrSelect_ANE,
		X1_W3_ResetOptions_BaseTXAvailable,
		EL_XL_W4_MediaStatus_LinkBeatEnable
	}, {
		"10BASE-T",
		MEDIA_CAPABLE_FDX | MEDIA_LINK_DETECT,
		EL_W0_ConfigControl_AvailTP,
		EE_EL_AddressConfiguration_Xcvr_TP,
		X1_InternalConfig_XcvrSelect_10BASET,
		X2_W3_MediaOptions_10bTAvailable,
		EL_XL_W4_MediaStatus_JabberGuardEnable | EL_XL_W4_MediaStatus_LinkBeatEnable
	}, {
		"100BASE-FX",
		MEDIA_XL | MEDIA_CAPABLE_FDX | MEDIA_LINK_DETECT | MEDIA_100,
		0,
		0,
		X1_InternalConfig_XcvrSelect_100BASEFX,
		X2_W3_MediaOptions_BaseFXAvailable,
		EL_XL_W4_MediaStatus_LinkBeatEnable
	}, {
		"10BASE5",
		MEDIA_CAPABLE_FDX,
		EL_W0_ConfigControl_AvailAUI,
		EE_EL_AddressConfiguration_Xcvr_AUI,
		X1_InternalConfig_XcvrSelect_10AUI,
		X2_W3_MediaOptions_AuiAvailable,
		EL_XL_W4_MediaStatus_EnableSqeStats
	}, {
		"10BASE2",
		MEDIA_NEED_COAX_CMD,
		EL_W0_ConfigControl_AvailBNC,
		EE_EL_AddressConfiguration_Xcvr_BNC,
		X1_InternalConfig_XcvrSelect_10BASE2,
		X2_W3_MediaOptions_CoaxAvailable,
		0
	}, {
		"MII",
		MEDIA_XL | MEDIA_NEED_MII | MEDIA_CAPABLE_FDX,
		0,
		0,
		X1_InternalConfig_XcvrSelect_MII,
		X2_W3_MediaOptions_MiiConnector | X2_W3_MediaOptions_BaseT4Available,
		0
	}, {
		"MII-EXTERNAL",
		MEDIA_XL | MEDIA_V2 | MEDIA_NEED_MII | MEDIA_CAPABLE_FDX,
		0,
		0,
		X2_InternalConfig_XcvrSelect_MII_EXT,
		X2_W3_MediaOptions_MiiConnector | X2_W3_MediaOptions_BaseT4Available,
		0
	},
};

#define N_MEDIA			(sizeof(media) / sizeof(*media))

#define CLEAR_MEDIA_BITS	(EL_XL_W4_MediaStatus_EnableSqeStats | EL_XL_W4_MediaStatus_JabberGuardEnable | EL_XL_W4_MediaStatus_LinkBeatEnable)

#define IS_E2(xl)	(__likely((xl)->flags & (EL2)))
#define IS_XL(xl)	(__likely((xl)->flags & (BOOMERANG | CYCLONE | TORNADO)))
#define IS_V2(xl)	(__likely((xl)->flags & (CYCLONE | TORNADO)))

static __u8 READ_8(XL *xl, unsigned i)
{
#if __DEBUG >= 1
	if (__unlikely(i > REGSPACE(xl->flags) - 1)) KERNEL$SUICIDE("3C READ_8: ACCESS BEYOND REGSPACE (%X, %X)", i, REGSPACE(xl->flags));
#endif
	if (__likely(!(xl->flags2 & FLAG2_PIO))) return mmio_inb(xl->u.mem + i);
	else return io_inb((xl)->u.io + i);
}

static __u16 READ_16(XL *xl, unsigned i)
{
#if __DEBUG >= 1
	if (__unlikely(i > REGSPACE(xl->flags) - 2)) KERNEL$SUICIDE("3C READ_16: ACCESS BEYOND REGSPACE (%X, %X)", i, REGSPACE(xl->flags));
#endif
	if (__likely(!(xl->flags2 & FLAG2_PIO))) return mmio_inw(xl->u.mem + i);
	else return io_inw((xl)->u.io + i);
}

static __u32 READ_32(XL *xl, unsigned i)
{
#if __DEBUG >= 1
	if (__unlikely(i > REGSPACE(xl->flags) - 4)) KERNEL$SUICIDE("3C READ_32: ACCESS BEYOND REGSPACE (%X, %X)", i, REGSPACE(xl->flags));
#endif
	if (__likely(!(xl->flags2 & FLAG2_PIO))) return mmio_inl(xl->u.mem + i);
	else return io_inl((xl)->u.io + i);
}

static void WRITE_8(XL *xl, unsigned i, __u8 v)
{
#if __DEBUG >= 1
	if (__unlikely(i > REGSPACE(xl->flags) - 1)) KERNEL$SUICIDE("3C WRITE_8: ACCESS BEYOND REGSPACE (%X, %X)", i, REGSPACE(xl->flags));
#endif
	if (__likely(!(xl->flags2 & FLAG2_PIO))) return mmio_outb(xl->u.mem + i, v);
	else return io_outb((xl)->u.io + i, v);
}

static void WRITE_16(XL *xl, unsigned i, __u16 v)
{
#if __DEBUG >= 1
	if (__unlikely(i > REGSPACE(xl->flags) - 2)) KERNEL$SUICIDE("3C WRITE_16: ACCESS BEYOND REGSPACE (%X, %X)", i, REGSPACE(xl->flags));
#endif
	if (__likely(!(xl->flags2 & FLAG2_PIO))) return mmio_outw(xl->u.mem + i, v);
	else return io_outw((xl)->u.io + i, v);
}

static void WRITE_32(XL *xl, unsigned i, __u32 v)
{
#if __DEBUG >= 1
	if (__unlikely(i > REGSPACE(xl->flags) - 4)) KERNEL$SUICIDE("3C WRITE_32: ACCESS BEYOND REGSPACE (%X, %X)", i, REGSPACE(xl->flags));
#endif
	if (__likely(!(xl->flags2 & FLAG2_PIO))) return mmio_outl(xl->u.mem + i, v);
	else return io_outl((xl)->u.io + i, v);
}

static int SUPPORTED_MEDIA(XL *xl, int flags)
{
	if (flags & MEDIA_XL && !IS_XL(xl)) return 0;
	if (flags & MEDIA_V2 && !IS_V2(xl)) return 0;
	return 1;
}

static int FIND_MEDIA_BY_INTERNAL_CONFIG(XL *xl, __u16 address_config, __u32 internal_config)
{
	int i;
	if (!IS_XL(xl)) {
		for (i = 0; i < N_MEDIA; i++) {
			if (SUPPORTED_MEDIA(xl, media[i].flags) && !((media[i].address_config ^ address_config) & EE_EL_AddressConfiguration_Xcvr)) return i;
		}
	} else {
		__u32 media_type = internal_config & (!IS_V2(xl) ? X1_InternalConfig_XcvrSelect : X2_InternalConfig_XcvrSelect);
		for (i = 0; i < N_MEDIA; i++) {
			if (SUPPORTED_MEDIA(xl, media[i].flags) && media[i].internal_config == media_type) return i;
		}
	}
	return -1;
}

static int AUTOSELECT_MEDIA(XL *xl, int last_media)
{
	int wrap = 0;
	int i = last_media;
	while (1) {
		i++;
		if (__unlikely(i >= N_MEDIA)) {
			i = 0, wrap++;
			if (__unlikely(wrap == 2)) return -1;
		}
		if (!SUPPORTED_MEDIA(xl, media[i].flags)) continue;
		if (!IS_XL(xl)) {
			if (xl->config_control & media[i].config_control) return i;
		} else {
			if (xl->media_options & media[i].media_options) return i;
		}
	}
}

#include "3CFN.I"

#include "MII.I"

static __finline__ void MII_DELAY(XL *xl)
{
	READ_16(xl, XL_W4_PhysicalMgmt);
}

static void MII_SYNC(XL *xl)
{
	int i;
	for (i = 0; i < 32; i++) {
		WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtData | XL_W4_PhysicalMgmt_MgmtDir);
		MII_DELAY(xl);
		WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtData | XL_W4_PhysicalMgmt_MgmtDir | XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
}

__u16 MII_READ(void *p, unsigned idx)
{
	unsigned cmd, ret;
	int i;
	XL *xl = p;
	if (__unlikely(xl->phy_id < 0)) return 0xffff;
	WINDOW(xl, 4);
	if (__unlikely(xl->flags2 & FLAG2_MII_PREAMBLE_RUNTIME)) MII_SYNC(xl);
	cmd = (0xf6 << 10) | (xl->phy_id << 5) | idx;
	for (i = 14; i >= 0; i--) {
		unsigned val = (cmd & (1 << i) ? XL_W4_PhysicalMgmt_MgmtData : 0) | XL_W4_PhysicalMgmt_MgmtDir;
		WRITE_16(xl, XL_W4_PhysicalMgmt, val);
		MII_DELAY(xl);
		WRITE_16(xl, XL_W4_PhysicalMgmt, val | XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
	ret = 0;
	for (i = 19; i > 0; i--) {
		unsigned rd;
		WRITE_16(xl, XL_W4_PhysicalMgmt, 0);
		MII_DELAY(xl);
		rd = READ_16(xl, XL_W4_PhysicalMgmt);
		ret = (ret << 1) | !!(rd & XL_W4_PhysicalMgmt_MgmtData);
		WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
	if (__unlikely(ret & 0x20000)) return 0xffff;
	return ret >> 1;
}

void MII_WRITE(void *p, unsigned idx, __u16 val)
{
	unsigned cmd;
	int i;
	XL *xl = p;
	if (__unlikely(xl->phy_id < 0)) return;
	WINDOW(xl, 4);
	if (__unlikely(xl->flags2 & FLAG2_MII_PREAMBLE_RUNTIME)) MII_SYNC(xl);
	cmd = 0x50020000 | (xl->phy_id << 23) | (idx << 18) | val;
	for (i = 31; i >= 0; i--) {
		unsigned val = (cmd & (1 << i) ? XL_W4_PhysicalMgmt_MgmtData : 0) | XL_W4_PhysicalMgmt_MgmtDir;
		WRITE_16(xl, XL_W4_PhysicalMgmt, val);
		MII_DELAY(xl);
		WRITE_16(xl, XL_W4_PhysicalMgmt, val | XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
	for (i = 1; i >= 0; i--) {
		WRITE_16(xl, XL_W4_PhysicalMgmt, 0);
		MII_DELAY(xl);
		WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
}

static int MII_PROBE(XL *xl)
{
	__u16 bmsr;
	xl->flags2 |= FLAG2_MII_PREAMBLE_RUNTIME;
	bmsr = MII_READ(xl, MII_BMSR);
	if (!bmsr || bmsr == 0xffff) return 0;
	if (!(xl->flags & MII_PREAMBLE) && bmsr & 0x40) xl->flags2 &= ~FLAG2_MII_PREAMBLE_RUNTIME;
	return 1;
}

static void XL_SET_MAC_CONTROL(XL *xl, int fdx);
static long RESET_THREAD(void *xl_);
static AST_STUB RESET_THREAD_DONE;
static void CHECK_MEDIA(XL *xl);

static void EXTRACT_ADDRESS(XL *xl, unsigned offset)
{
	xl->address[0] = xl->eeprom[offset] >> 8;
	xl->address[1] = xl->eeprom[offset];
	xl->address[2] = xl->eeprom[offset + 1] >> 8;
	xl->address[3] = xl->eeprom[offset + 1];
	xl->address[4] = xl->eeprom[offset + 2] >> 8;
	xl->address[5] = xl->eeprom[offset + 2];
}

static void RESET_1(XL *xl)
{
	WAIT(xl, NULL);
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetInterruptEnable);
	if (!IS_XL(xl)) {
		/* full reset disconnects the adapter from ISA bus */
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxReset);
		WAIT(xl, "TX RESET");
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_RxReset);
		WAIT(xl, "RX RESET");
	} else {
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_GlobalReset | (xl->flags & NO_EEPROM_RELOAD ? EL_XL_Cmd_GlobalReset_AismReset : 0) | EL_XL_Cmd_GlobalReset_NetworkReset);
	}
}

static int RESET_2(XL *xl)
{
	KERNEL$SLEEP((JIFFIES_PER_SECOND + 9) / 10);
	if (WAIT(xl, "RESET")) return 1;
	return 0;
}

static void RESET_3(XL *xl, int loading)
{
	if (__likely(!(xl->flags2 & FLAG2_MEDIA_UNKNOWN))) {
		if (!IS_XL(xl)) {
			__u16 address_config;
			WINDOW(xl, 0);
			address_config = READ_16(xl, EL_W0_AddressConfiguration);
			address_config &= ~EE_EL_AddressConfiguration_Xcvr;
			address_config |= media[xl->media].address_config;
			WRITE_16(xl, EL_W0_AddressConfiguration, address_config);
		} else {
			__u32 internal_config;
			WINDOW(xl, 3);
			internal_config = READ_32(xl, XL_W3_InternalConfig);
			if (!IS_V2(xl)) internal_config &= ~X1_InternalConfig_XcvrSelect;
			else internal_config &= ~X2_InternalConfig_XcvrSelect;
			internal_config |= media[xl->media].internal_config;
			WRITE_32(xl, XL_W3_InternalConfig, internal_config);
		}
	}
	if (__unlikely(xl->flags & CARDBUS)) {
		__u16 ro;
		WINDOW(xl, 2);
		ro = READ_16(xl, X2_W2_ResetOptions) & ~(X2_W2_ResetOptions_CB_LedPwr | X2_W2_ResetOptions_CB_MiiPwr);
		if (!!(xl->flags & INVERT_MII_PWR) ^ (loading == -1)) ro |= X2_W2_ResetOptions_CB_MiiPwr;
		if (!!(xl->flags & INVERT_LED_PWR) ^ (loading == -1)) ro |= X2_W2_ResetOptions_CB_LedPwr;
		WRITE_16(xl, X2_W2_ResetOptions, ro);
		if (xl->flags & WNO_XCVR_PWR) {
			WINDOW(xl, 0);
			WRITE_16(xl, 0, 0x0800);
		}
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
	}
	if (__likely(!(xl->flags2 & FLAG2_MEDIA_UNKNOWN))) {
		__u16 media_status;
		WINDOW(xl, 4);
		media_status = READ_16(xl, EL_XL_W4_MediaStatus);
		media_status &= ~CLEAR_MEDIA_BITS;
		media_status |= media[xl->media].media_bits;
		WRITE_16(xl, EL_XL_W4_MediaStatus, media_status);
		if (__unlikely(media[xl->media].flags & MEDIA_NEED_COAX_CMD)) {
			WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_EnableDcConverter);
			KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
			WAIT(xl, "ENABLE DC CONVERTER");
		} else {
			WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_DisableDcConverter);
			KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
			WAIT(xl, "DISABLE DC CONVERTER");
		}
		/*__debug_printf("media: %04x\n", READ_16(xl, EL_XL_W4_MediaStatus));*/
	}
	xl->phy_id = -1;
	if (__unlikely(xl->flags2 & FLAG2_MEDIA_UNKNOWN) || media[xl->media].flags & MEDIA_NEED_MII /*|| xl->flags & NWAY*/) {
		int i;
		if (__likely(!(xl->flags2 & FLAG2_MEDIA_UNKNOWN)) && media[xl->media].flags & MEDIA_INTERNAL_MII) {
			xl->phy_id = INTERNAL_PHY_ID;
			if (__likely(MII_PROBE(xl))) goto have_mii;
		}
		for (i = 0; i < N_PHY; i++) {
			xl->phy_id = i;
			if (__likely(MII_PROBE(xl))) goto have_mii;
		}
		xl->phy_id = -1;
		have_mii:;
	}
	if (!IS_XL(xl)) {
		__u16 netdiag;
		int duplex = 0;
		if (xl->bmcr & BMCR_FULLDPLX) duplex = 1;
		if (xl->bmcr & BMCR_ANENABLE && xl->eeprom[EE_EL_XL_Software_Information] & EE_EL_XL_Software_Information_FullDuplex) duplex = 1;
		if (!(media[xl->media].flags & MEDIA_CAPABLE_FDX)) duplex = 0;
		WINDOW(xl, 4);
		netdiag = READ_16(xl, EL_XL_W4_NetworkDiagnostic);
		netdiag &= ~EL_XL_W4_NetworkDiagnostic_ExternalLoopback;
		netdiag |= EL_XL_W4_NetworkDiagnostic_ExternalLoopback * duplex;
		WRITE_16(xl, EL_XL_W4_NetworkDiagnostic, netdiag);
		xl->flags2 &= ~FLAG2_FULL_DUPLEX_ENABLED;
		xl->flags2 |= FLAG2_FULL_DUPLEX_ENABLED * duplex;
	} else {
		XL_SET_MAC_CONTROL(xl, xl->bmcr & BMCR_FULLDPLX && (xl->flags2 & FLAG2_MEDIA_UNKNOWN || media[xl->media].flags & MEDIA_CAPABLE_FDX));
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_RxReset | (xl->flags & (EL2 | BOOMERANG | CYCLONE | TORNADO) ? E2_XL_Cmd_RxReset_NetworkRxReset : 0));
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
		WAIT(xl, "RX RESET");
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxReset | (IS_XL(xl) || IS_E2(xl) ? E2_XL_Cmd_TxReset_NetworkTxReset : 0));
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
		WAIT(xl, "TX RESET");
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
		if (xl->phy_id != -1) {
			MII_WRITE(xl, MII_BMCR, xl->bmcr);

			/*__debug_printf("bmcr: %04x\n", MII_READ(xl, MII_BMCR));
			MII_WRITE(xl, MII_ADVERTISE, 0x5e1);
			__debug_printf("advertise: %04x\n", MII_READ(xl, MII_ADVERTISE));
			__debug_printf("expansion: %04x\n", MII_READ(xl, MII_EXPANSION));
			__debug_printf("quick status: %04x\n", MII_READ(xl, 31));*/
		}
		XL_SET_MAC_CONTROL(xl, xl->bmcr & BMCR_FULLDPLX && (xl->flags2 & FLAG2_MEDIA_UNKNOWN || media[xl->media].flags & MEDIA_CAPABLE_FDX));
	}
	if (!IS_XL(xl)) {
		__u16 res;
		WINDOW(xl, 0);
		res = READ_16(xl, EL_W0_ResourceConfiguration);
		res &= ~EE_EL_ResourceConfiguration_IRQ;
		res |= xl->irq << __BSF_CONST(EE_EL_ResourceConfiguration_IRQ);
		WRITE_16(xl, EL_W0_ResourceConfiguration, res);
		/*
		This causes that the card loses IO configuration occasionally
		res = READ_16(xl, EL_W0_ConfigControl);
		res |= EL_W0_ConfigControl_ENA;
		WRITE_16(xl, EL_W0_ConfigControl, res);
		*/
	}
}

static void XL_SET_MAC_CONTROL(XL *xl, int fdx)
{
	__u16 mac_ctrl;
#if __DEBUG >= 1
	if (!IS_XL(xl))
		KERNEL$SUICIDE("XL_SET_MAC_CONTROL: NOT XL CARD, FLAGS %X, FLAGS2 %X", xl->flags, xl->flags2);
#endif
	WINDOW(xl, 3);
	mac_ctrl = READ_16(xl, XL_W3_MacControl) & ~(XL_W3_MacControl_FullDuplexEnable | X2_W3_MacControl_FlowControlEnable | XL_W3_MacControl_DeferExtendedEnable | XL_W3_MacControl_DeferTimerSelect | X2_W3_MacControl_ExtendAfterCollision);
	if (fdx) {
		xl->flags2 |= FLAG2_FULL_DUPLEX_ENABLED;
		mac_ctrl |= XL_W3_MacControl_FullDuplexEnable;
		if (IS_V2(xl) && xl->flags & FLOW_CONTROL) {
			mac_ctrl |= X2_W3_MacControl_FlowControlEnable;
		}
	} else {
		xl->flags2 &= ~FLAG2_FULL_DUPLEX_ENABLED;
		if (xl->defer_timer) {
			mac_ctrl |= XL_W3_MacControl_DeferExtendedEnable;
			mac_ctrl |= ((xl->defer_timer >> 4) << __BSF_CONST(XL_W3_MacControl_DeferTimerSelect)) & XL_W3_MacControl_DeferTimerSelect;
		}
		if (IS_V2(xl) && xl->flags & DEFER_ONLY_COLLISION) {
			mac_ctrl |= X2_W3_MacControl_ExtendAfterCollision;
		}
	}
	WRITE_16(xl, XL_W3_MacControl, mac_ctrl);
}

static void READ_EEPROM(XL *xl)
{
	unsigned i, w;
	__u16 eeprom_cmd, checksum;
	if (xl->flags & EEPROM_8BIT) eeprom_cmd = (EL_XL_W0_EepromCommand_ReadRegister << 2) | EL_XL_W0_EepromCommand_DataOffset;
	else if (xl->flags & EEPROM_OFFSET_30) eeprom_cmd = EL_XL_W0_EepromCommand_ReadRegister | EL_XL_W0_EepromCommand_DataOffset;
	else eeprom_cmd = EL_XL_W0_EepromCommand_ReadRegister;
	WINDOW(xl, 0);
	for (i = 0; i < EEPROM_SIZE; i++) {
		WRITE_16(xl, EL_XL_W0_EepromCommand, eeprom_cmd + i);
		for (w = 0; w < 15; w++) {
			KERNEL$SLEEP((JIFFIES_PER_SECOND + 499) / 500);
			if (!(READ_16(xl, EL_XL_W0_EepromCommand) & EL_XL_W0_EepromCommand_EepromBusy)) break;
		}
		xl->eeprom[i] = READ_16(xl, EL_XL_W0_EepromData);
	}
	checksum = 0;
	for (i = 0; i < (IS_XL(xl) ? 0x18 : IS_E2(xl) ? 0x40 : 0x10); i++) checksum ^= xl->eeprom[i];
	checksum ^= checksum >> 8;
	checksum &= 0xff;
	if (IS_XL(xl) && checksum) {
		for (i = 0x18; i < 0x21; i++) checksum ^= xl->eeprom[i];
		checksum ^= checksum >> 8;
		checksum &= 0xff;
	}
	if (checksum && !(xl->flags & TORNADO))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "INVALID EEPROM CHECKSUM");
}

int INCREASE_TX_THRESH(XL *xl)
{
	if (__unlikely(xl->tx_thresh >= PKT_SIZE)) return 1;
	xl->tx_thresh += XL_INCREMENT_TX_THRESH;
	if (__unlikely(xl->tx_thresh > PKT_SIZE)) xl->tx_thresh = PKT_SIZE;
	return 0;
}

void SET_TX_THRESH(XL *xl)
{
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetTxStartThresh | ((xl->tx_thresh + 3) >> (IS_XL(xl) ? 2 : 0)));
}

static void CONTINUE_RESET(XL *xl)
{
	unsigned i;
	unsigned int_mask;
	WINDOW(xl, 2);
	for (i = 0; i < ETHERNET_ADDRLEN; i++)
		WRITE_8(xl, EL_XL_W2_StationAddressLo + i, xl->address[i]);
	if (IS_XL(xl)) for (i = 0; i < ETHERNET_ADDRLEN; i += 2)
		WRITE_16(xl, XL_W2_StationMaskLo + i, 0);
	
	if (IS_XL(xl)) {
		for (i = 0; i < N_RX_DESCS; i++) {
			RX_DESC(xl, i)->upd.UpNextPtr = __32CPU2LE(xl->desc_dmaaddr + RX_DESCS_OFFSET + ((i + 1) & (N_RX_DESCS - 1)) * SIZEOF_XL_RXDESC);
			RX_DESC(xl, i)->upd.UpPktStatus = __32CPU2LE(XL_UPS_UpComplete);
			RX_DESC(xl, i)->frags[0].UpFragLen = __32CPU2LE(PKT_SIZE | XL_UFL_UpFragLast);
		}
		for (i = 0; i < xl->n_recv; i++) {
			RX_DESC(xl, (xl->first_recv + i) & (N_RX_DESCS - 1))->upd.UpPktStatus = __32CPU2LE(0);
		}
		XL_DESTROY_SENT_PACKETS(xl);
		xl->first_sent = 0;
		xl->n_sent = 0;
		if (!IS_V2(xl)) WRITE_8(xl, X1_DM_TxFreeThresh, (PKT_SIZE + 255) >> 8);
	}
	SET_TX_THRESH(xl);
	if (IS_V2(xl)) WRITE_16(xl, EL_XL_IntStatusCmd, EM_X2_Cmd_SetTxReclaimThresh | xl->tx_reclaim_thresh);
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetRxEarlyThresh | ((PKT_SIZE + 3) >> 2));

	UPDATE_FILTER(xl);

	if (IS_XL(xl)) {
		WRITE_16(xl, EL_XL_IntStatusCmd, XL_Cmd_DnStall);
		WAIT(xl, "DOWNLOAD STALL");
		WRITE_16(xl, EL_XL_IntStatusCmd, XL_Cmd_UpStall);
		WAIT(xl, "UPLOAD STALL");

		WRITE_32(xl, XL_DM_DmaCtrl, XL_DM_DmaCtrl_UpEarlyRxEnable);
		if (IS_V2(xl) && !(xl->eeprom[EE_XL_Software_Information_2] & EE_XL_Software_Information_2_FixedMWIBug)) {
			WRITE_32(xl, XL_DM_DmaCtrl, READ_32(xl, XL_DM_DmaCtrl) | X2_DM_DmaCtrl_DefeatMWI);
		}

		if (XMIT_POLLING(xl)) {
			TX_DESC(xl, N_TX_DESCS - 1)->upd.DnNextPtr = __32CPU2LE(0);
			TX_DESC(xl, N_TX_DESCS - 1)->upd.FrameStartHeader = __32CPU2LE(X2_FSH_DpdEmpty);
			WRITE_32(xl, XL_DM_DnListPtr, xl->desc_dmaaddr + TX_DESCS_OFFSET + (N_TX_DESCS - 1) * SIZEOF_XL_TXDESC);
		} else {
			WRITE_32(xl, XL_DM_DnListPtr, 0);
		}
		WRITE_32(xl, XL_DM_UpListPtr, xl->desc_dmaaddr + RX_DESCS_OFFSET + xl->first_recv * SIZEOF_XL_RXDESC);

	}

	WRITE_16(xl, EL_XL_IntStatusCmd,
		EL_XL_Cmd_AcknowledgeInterrupt |
		EL_XL_Cmd_AcknowledgeInterrupt_InterruptLatchAck |
		EL_XL_Cmd_AcknowledgeInterrupt_RxEarlyAck |
		EL_XL_Cmd_AcknowledgeInterrupt_IntRequestedAck |
		(!IS_XL(xl) ?
			EL_Cmd_AcknowledgeInterrupt_TxComplete | EL_Cmd_AcknowledgeInterrupt_TxAvailable | EL_XL_Cmd_AcknowledgeInterrupt_RxComplete | EL_XL_Cmd_AcknowledgeInterrupt_IntRequestedAck
		:
			XL_Cmd_AcknowledgeInterrupt_DnCompleteAck | XL_Cmd_AcknowledgeInterrupt_UpCompleteAck
		));
	int_mask = EL_XL_IntStatus_HostError | EL_XL_IntStatus_TxComplete;
	if (!IS_XL(xl)) int_mask |= EL_IntStatus_TxAvailable | EL_XL_IntStatus_RxComplete;
	else int_mask |= XL_IntStatus_DnComplete | XL_IntStatus_UpComplete;
	xl->flags2 &= ~FLAG2_LINK_INTERRUPT;
	if (IS_V2(xl) && __likely(!(xl->flags & NO_LINK_INTR)) && __likely(!(xl->flags2 & FLAG2_MEDIA_UNKNOWN)) && __likely(media[xl->media].flags & MEDIA_INTERNAL_MII) && __likely(xl->phy_id == INTERNAL_PHY_ID)) {
		int_mask |= XL_IntStatus_LinkEvent;
		xl->flags2 |= FLAG2_LINK_INTERRUPT;
	}
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetIndicationEnable | int_mask);
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetInterruptEnable | int_mask);

	if (IS_XL(xl)) {
		if (RECV_POLLING(xl)) WRITE_8(xl, X2_DM_UpPoll, RECV_POLL_INTERVAL);
		if (XMIT_POLLING(xl)) WRITE_8(xl, X2_DM_DnPoll, XMIT_POLL_INTERVAL);

		WRITE_16(xl, EL_XL_IntStatusCmd, XL_Cmd_DnUnStall);
		WRITE_16(xl, EL_XL_IntStatusCmd, XL_Cmd_UpUnStall);
	}

	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxEnable);
	WAIT(xl, "TX ENABLE");
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_RxEnable);
	WAIT(xl, "RX ENABLE");
	THE_PCMCIA_PEOPLE_ARE_IDIOTS(xl);

	CHECK_MEDIA(xl);

	KERNEL$DEL_TIMER(&xl->media_timer);
	KERNEL$SET_TIMER(INITIAL_LINK_EXPIRE_TIME > LINK_CHECK_INITIAL_TIME ? LINK_CHECK_INITIAL_TIME : INITIAL_LINK_EXPIRE_TIME, &xl->media_timer);

	xl->flags2 &= ~(FLAG2_RECEIVED_PACKET | FLAG2_RECEIVED_PACKET_2);
	xl->change_media_time = KERNEL$GET_JIFFIES() + INITIAL_LINK_EXPIRE_TIME;

	WINDOW(xl, 1);
}

void UPDATE_FILTER(XL *xl)
{
	unsigned i, j;
	unsigned filter = EL_XL_Cmd_SetRxFilter_ReceiveIndividual | EL_XL_Cmd_SetRxFilter_ReceiveBroadcast;
	if (__unlikely((xl->flags2 & (FLAG2_MEDIA_AUTOSELECT | FLAG2_RECEIVED_PACKET)) == FLAG2_MEDIA_AUTOSELECT) && xl->phy_id == -1 && !(media[xl->media].flags & MEDIA_LINK_DETECT)) goto promisc;
	if (__unlikely(!WQ_EMPTY(&xl->mcast_table[MCAST_PROMISC]))) {
		promisc:
		filter |= EL_XL_Cmd_SetRxFilter_ReceiveAllFrames | EL_XL_Cmd_SetRxFilter_ReceiveMulticast;
		xl->mcast_state = STATUS_MULTICAST_PROMISC;
		goto set_filter;
	}
	if (__unlikely(!WQ_EMPTY(&xl->mcast_table[MCAST_ALL]))) {
		all_mc:
		filter |= EL_XL_Cmd_SetRxFilter_ReceiveMulticast;
		xl->mcast_state = STATUS_MULTICAST_ALL;
		goto set_filter;
	}
	xl->mcast_state = STATUS_MULTICAST_NONE;
	for (i = 0; i < N_HW_MCAST; i++) if (__unlikely(!WQ_EMPTY(&xl->mcast_table[i]))) {
		if (!IS_V2(xl)) goto all_mc;
		if (!(filter & X2_Cmd_SetRxFilter_ReceiveMulticastHash)) {
/* Temporarily receive all multicasts, while we clear and reinitialize the hash.
We don't know if the hash is 6-bit or 8-bit, so we must do it this way. */
			WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetRxFilter | filter | EL_XL_Cmd_SetRxFilter_ReceiveMulticast);
			for (j = 0; j < N_HW_MCAST; j++) {
				WRITE_16(xl, EL_XL_IntStatusCmd, X2_Cmd_ClearHashFilterBit | j);
			}
			xl->mcast_state = STATUS_MULTICAST_SOME;
		}
		filter |= X2_Cmd_SetRxFilter_ReceiveMulticastHash;
		WRITE_16(xl, EL_XL_IntStatusCmd, X2_Cmd_SetHashFilterBit | i);
	}
	set_filter:
	WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_SetRxFilter | filter);
}

static int INIT(XL *xl)
{
	char a1[ETHERNET_ADDRESS_STR_LEN];

	xl->tx_thresh = XL_INIT_TX_THRESH;
	xl->tx_reclaim_thresh = XL_INIT_TX_RECLAIM_THRESH;

	RESET_1(xl);
	if (RESET_2(xl)) return -EIO;
	READ_EEPROM(xl);

	if (!(xl->flags2 & FLAG2_ADDRESS_OVERRIDE)) {
		EXTRACT_ADDRESS(xl, EE_EL_XL_OEM_Node_Address_0);
		if (__unlikely(!NET$VALID_ETHERNET_ADDRESS(xl->address))) {
			EXTRACT_ADDRESS(xl, EE_EL_XL_3Com_Node_Address_0);
			if (__unlikely(!NET$VALID_ETHERNET_ADDRESS(xl->address))) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "INVALID ETHERNET ADDRESS: %s", NET$PRINT_ETHERNET_ADDRESS(a1, xl->address));
				return -EIO;
			}
		}
	}
	/*{
		int i;
		for (i = 0; i < 0x40; i++) {
			__debug_printf("%04x ", xl->eeprom[i]);
		}
		__debug_printf("\n");
	}*/
	return 0;
}

void RESET_DEQUEUE(XL *xl)
{
	if (__unlikely(xl->flags2 & (FLAG2_RESETTING | FLAG2_DEAD))) return;
	xl->flags2 |= FLAG2_RESETTING;
	RESET_1(xl);
	xl->reset_thread.fn = RESET_THREAD_DONE;
	xl->reset_thread.thread_main = RESET_THREAD;
	xl->reset_thread.p = xl;
	xl->reset_thread.error = NULL;
	xl->reset_thread.cwd = NULL;
	xl->reset_thread.std_in = -1;
	xl->reset_thread.std_out = -1;
	xl->reset_thread.std_err = -1;
	xl->reset_thread.dlrq = NULL;
	xl->reset_thread.spawned = 0;
	/* xl->reset_thread.thread already set */
	CALL_IORQ(&xl->reset_thread, KERNEL$THREAD);
}

static long RESET_THREAD(void *xl_)
{
	XL *xl = xl_;
	RAISE_SPL(SPL_XL);
	RESET_2(xl);
	RESET_3(xl, 1);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static DECL_AST(RESET_THREAD_DONE, SPL_XL, THREAD_RQ)
{
	XL *xl = GET_STRUCT(RQ, XL, reset_thread);
	CONTINUE_RESET(xl);
	if (__unlikely(xl->flags2 & FLAG2_RESETTING_INTR)) {
		KERNEL$UNMASK_IRQ(xl->irq_ctrl);
	}
	xl->flags2 &= ~(FLAG2_RESETTING | FLAG2_RESETTING_INTR);
	if (__unlikely(!IS_XL(xl))) EL_DEQUEUE(xl);
	else if (__unlikely(xl->flags2 & FLAG2_PIO)) PIO_DEQUEUE(xl);
	else MMIO_DEQUEUE(xl);
	RETURN;
}

static void CHECK_MEDIA(XL *xl)
{
	LINK_STATE_PHYS ls;
	if (__unlikely(xl->flags2 & FLAG2_DEAD)) {
		memset(&ls, 0, sizeof ls);
		ls.flags |= LINK_STATE_ADAPTER_DEAD;
	} else {
		if (__likely(xl->phy_id != -1)) {
			MII_GET_MODE(xl, &ls);
			if (ls.desc && xl->media_options & X2_W3_MediaOptions_BaseT4Available) {
				memset(ls.desc, 0, sizeof ls.desc);
				strlcpy(ls.desc, "100BASE-T4", sizeof ls.desc);
			}
			XL_SET_MAC_CONTROL(xl, ls.flags & LINK_STATE_FULL_DUPLEX);
	/* Tx/Rx should be reset here, but neither Linux nor FreeBSD do it.
	   Unfortunatelly, the manual is unclear about what bits should be set
	   in reset mask (resetting the whole card is not good idea, because it
	   would change media status and loop back here).
	   Don't reset for now, and see if someone reports problems.
	*/
		} else {
			memset(&ls, 0, sizeof ls);
			if (xl->flags2 & FLAG2_MEDIA_UNKNOWN) {
				ls.flags |= LINK_STATE_UNKNOWN;
			} else {
				strlcpy(ls.desc, media[xl->media].name, sizeof(ls.desc));
				if (media[xl->media].flags & MEDIA_LINK_DETECT) {
					__u16 media_status;
					WINDOW(xl, 4);
					media_status = READ_16(xl, EL_XL_W4_MediaStatus);
					if (media_status & EL_XL_W4_MediaStatus_LinkDetect) ls.flags |= LINK_STATE_UP;
				} else {
					ls.flags |= LINK_STATE_UNKNOWN;
				}
				if (media[xl->media].flags & MEDIA_100) {
					ls.speed = 100000000;
				} else {
					ls.speed = 10000000;
				}
			}
			if (xl->flags2 & FLAG2_FULL_DUPLEX_ENABLED) ls.flags |= LINK_STATE_FULL_DUPLEX;
			else ls.flags |= LINK_STATE_HALF_DUPLEX;
		}
		if (xl->flags2 & FLAG2_MEDIA_AUTOSELECT) ls.flags |= LINK_STATE_MEDIA_AUTOSELECT;
	}
	if (__unlikely(memcmp(&xl->link_state, &ls, sizeof(LINK_STATE_PHYS)))) {
		memcpy(&xl->link_state, &ls, sizeof(LINK_STATE_PHYS));
		WQ_WAKE_ALL_PL(&xl->link_state_wait);
	}
}

static void AUTOSELECT_LINK_CHANGE(XL *xl)
{
	if (__unlikely(xl->flags2 & (FLAG2_DEAD | FLAG2_RESETTING))) return;
	xl->media = AUTOSELECT_MEDIA(xl, xl->media);
	RESET_DEQUEUE(xl);
}

static void MEDIA_TIMER(TIMER *t)
{
	XL *xl = GET_STRUCT(t, XL, media_timer);
	u_jiffies_lo_t j, tm;
	LOWER_SPL(SPL_XL);
	j = KERNEL$GET_JIFFIES_LO();
	if (j - xl->n_sent_decay >= N_SENT_DECAY_TIME) {
		xl->n_sent_decay = j;
		if (xl->max_sent > DEFAULT_SENT) xl->max_sent--;
	}
	tm = LINK_CHECK_TIME(xl);
	if (__unlikely(xl->flags2 & FLAG2_RESETTING)) goto skip_check;
	CHECK_MEDIA(xl);
	if (!(xl->flags2 & FLAG2_MEDIA_AUTOSELECT)) goto skip_check;
	if (__likely(xl->link_state.flags & LINK_STATE_UP)) {
		xl->change_media_time = j + MONITORED_LINK_EXPIRE_TIME;
	} else if (__unlikely(!(xl->link_state.flags & LINK_STATE_UNKNOWN))) {
		if ((jiffies_lo_t)(j - xl->change_media_time) >= 0) {
			AUTOSELECT_LINK_CHANGE(xl);
			goto skip_check;
		}
	} else if (__unlikely(!(xl->flags2 & FLAG2_RECEIVED_PACKET))) {
		if ((jiffies_lo_t)(j - xl->change_media_time) >= 0) {
			AUTOSELECT_LINK_CHANGE(xl);
			goto skip_check;
		}
	} else {
		if (__likely(xl->flags2 & FLAG2_RECEIVED_PACKET_2)) {
			xl->flags2 &= ~FLAG2_RECEIVED_PACKET_2;
			xl->change_media_time = j + UNMONITORED_LINK_PROMISC_TIME;
		} else if ((jiffies_lo_t)(j - xl->change_media_time) >= 0) {
			xl->flags2 &= ~FLAG2_RECEIVED_PACKET;
			UPDATE_FILTER(xl);
			xl->change_media_time = j + UNMONITORED_LINK_EXPIRE_TIME;
		}
	}
	if (xl->change_media_time - j < tm) tm = xl->change_media_time - j;
	skip_check:
	KERNEL$SET_TIMER(tm, &xl->media_timer);
}

static void TIMEOUT_RESET(XL *xl);

static void TRANSMIT_TIMER(TIMER *t)
{
	XL *xl = GET_STRUCT(t, XL, timer);
	LOWER_SPL(SPL_XL);
	SET_TIMER_NEVER(&xl->timer);
	if (__unlikely(xl->flags2 & (FLAG2_RESETTING | FLAG2_DEAD))) return;
	if (__unlikely(!IS_XL(xl)) || __unlikely(xl->n_sent)) TIMEOUT_RESET(xl);
}

int REPORT_TX_ERROR(XL *xl, __u8 tx_status)
{
	int r = 0;
	if (tx_status & EL_XL_TxStatus_TxReclaimError) {
		if (xl->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "OUT OF WINDOW COLLISION");
		r = 1;
	}
	if (tx_status & EL_XL_TxStatus_TxStatusOverflow) {
		if (xl->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, xl->dev_name, "TX STATUS STACK OVERFLOW");
		r = 1;
	}
	if (tx_status & EL_XL_TxStatus_MaxCollisions) {
		if (xl->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, xl->dev_name, "EXCESSIVE COLLISIONS");
		r = 1;
	}
	if (tx_status & EL_XL_TxStatus_TxUnderrun) {
		if (__unlikely(INCREASE_TX_THRESH(xl)))
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "TX UNDERRUN WITH MAXIMUM THRESHOLD");
		r = 1;
	}
	if (tx_status & EL_XL_TxStatus_TxJabber) {
		if (xl->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "JABBER DETECTED");
		r = 1;
	}
	if (tx_status & EL_XL_TxStatus_InterruptRequested) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNEXPECTED INTERRUPT REQUESTED");
		r = 1;
	}
	if (__unlikely(!r)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNKNOWN TX ERROR, STATUS %02X", tx_status);
		return 1;
	}
	return 0;
}

int XL_IRQ_SPECIAL(XL *xl, __u16 status)
{
	if (__unlikely(status == 0xffff)) {
		xl->flags2 |= FLAG2_DEAD;
		return 1;
	}
	if (__likely(status & XL_IntStatus_LinkEvent)) {
		MII_READ(xl, MII_EXPANSION);	/* ack for 3C905B */
		CHECK_MEDIA(xl);
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_AcknowledgeInterrupt | EL_XL_Cmd_AcknowledgeInterrupt_InterruptLatchAck | X3_Cmd_AcknowledgeInterrupt_LinkEventAck);	/* ack for 3C905C */
	}
	if (__unlikely(status & EL_XL_IntStatus_HostError)) {
		__u32 dmactrl = READ_32(xl, XL_DM_DmaCtrl);
		if ((dmactrl & (XL_DM_DmaCtrl_TargetAbort | XL_DM_DmaCtrl_MasterAbort)) == XL_DM_DmaCtrl_TargetAbort) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, xl->dev_name, "PCI TARGET ABORT");
		if ((dmactrl & (XL_DM_DmaCtrl_TargetAbort | XL_DM_DmaCtrl_MasterAbort)) == XL_DM_DmaCtrl_MasterAbort) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, xl->dev_name, "PCI MASTER ABORT");
		if (dmactrl != 0xffffffff) KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "HOST ERROR, DMACTRL %08X", dmactrl);
		reset_it:
		RESET_DEQUEUE(xl);
		return 1;
	}
	if (__likely(status & EL_XL_IntStatus_TxComplete)) {
		__u8 tx_status = READ_8(xl, XL_DM_TxStatus);
		WRITE_8(xl, XL_DM_TxStatus, 0);
		if (REPORT_TX_ERROR(xl, tx_status)) goto reset_it;
		if (tx_status & EL_XL_TxStatus_TxReclaimError) {
			xl->tx_reclaim_thresh += XL_INCREMENT_TX_RECLAIM_THRESH;
			if (xl->tx_reclaim_thresh > XL_MAX_TX_RECLAIM_THRESH) xl->tx_reclaim_thresh = XL_MAX_TX_RECLAIM_THRESH;
			if (IS_V2(xl)) {
				WRITE_16(xl, EL_XL_IntStatusCmd, EM_X2_Cmd_SetTxReclaimThresh | xl->tx_reclaim_thresh);
				WAIT(xl, "SET TX RECLAIM THRESH");
			}
		}
		if (tx_status & (EL_XL_TxStatus_TxUnderrun | EL_XL_TxStatus_TxJabber)) goto reset_it;
		if (tx_status & EL_XL_TxStatus_MaxCollisions && xl->flags & MAX_COLLISION_RESET) {
			WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxReset | E2_XL_Cmd_TxReset_FifoTxReset | XL_Cmd_TxReset_DnTxReset);
			KERNEL$UDELAY(1000);
			WAIT(xl, "TX RESET (EXCEPT FIFO AND DOWNLOAD)");
		}
		SET_TX_THRESH(xl);
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxEnable);
		WAIT(xl, "TX ENABLE");
	}
	return 0;
}

void XL_RECV_ERROR(XL *xl, __u32 up_status)
{
	int r = 0;
	up_status = __32LE2CPU(up_status);
	if (up_status & XL_UPS_UpOverrun) {
		if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_HW_WARNING, xl->dev_name, "RECEIVE FIFO OVERRUN");
		r = 1;
	}
	if (up_status & XL_UPS_RuntFrame) {
		if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "RUNT PACKET");
		r = 1;
	}
	if (up_status & XL_UPS_AlignmentError) {
		if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "ALIGNMENT ERROR");
		r = 1;
	}
	if (up_status & XL_UPS_CrcError) {
		if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, xl->dev_name, "CRC ERROR");
		r = 1;
	}
	if (up_status & (XL_UPS_OversizedFrame | XL_UPS_UpOverflow)) {
		if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "TOO LONG PACKET");
		r = 1;
	}
	if (__unlikely(!r)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNKNOWN RECEIVE ERROR, STATUS %08X", up_status);
	}
}

static void TIMEOUT_RESET(XL *xl)
{
	if (__unlikely(READ_16(xl, EL_XL_IntStatusCmd) == 0xffff)) xl->flags2 |= FLAG2_DEAD;
	if (!(xl->flags2 & FLAG2_DEAD))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "TRANSMIT TIMEOUT");
	RESET_DEQUEUE(xl);
}

DECL_IOCALL(IOCTL, SPL_XL, IOCTLRQ)
{
	int r;
	HANDLE *h = RQ->handle;
	XL *xl;
	if (__unlikely(h->op != &EL_OPERATIONS) &&
	    __unlikely(h->op != &PIO_OPERATIONS) &&
	    __unlikely(h->op != &MMIO_OPERATIONS))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_XL));
	xl = h->fnode;
	if (__unlikely(xl->flags2 & FLAG2_RESETTING)) {
		WQ_WAIT_F(&xl->link_state_wait, RQ);
		RETURN;
	}
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			ETHERNET_FILL_IF_TYPE(&t);
			t.id = xl->pkt_id;
			memcpy(t.addr, xl->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			CHECK_MEDIA(xl);
			if (!memcmp(&xl->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&xl->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			CHECK_MEDIA(xl);
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &xl->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			xl->link_state.seq++;
			WQ_WAKE_ALL_PL(&xl->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_MULTICAST: {
			if (__unlikely(RQ->v.len != 0)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			RQ->status = xl->mcast_state;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SET_MULTICAST: {
			__u8 mcast[ETHERNET_ADDRLEN];
			unsigned bit;
			int need_refresh;
			if (__unlikely(RQ->param == PARAM_MULTICAST_PROMISC)) {
				bit = MCAST_PROMISC;
				goto have_bit;
			} else if (__unlikely(RQ->param == PARAM_MULTICAST_ALL)) {
				bit = MCAST_ALL;
				goto have_bit;
			}
			if (__unlikely(RQ->param != PARAM_MULTICAST_ADDRESS)) {
				RQ->status = -ENOOP;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, mcast, ETHERNET_ADDRLEN)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			bit = NET$ETHERNET_CRC(mcast, ETHERNET_ADDRLEN) & (N_HW_MCAST - 1);
			have_bit:
			need_refresh = WQ_EMPTY(&xl->mcast_table[bit]);
			WQ_WAIT_F(&xl->mcast_table[bit], RQ);
			if (need_refresh) UPDATE_FILTER(xl);
			RETURN;
		}
		case IOCTL_IF_RESET_MULTICAST: {
			UPDATE_FILTER(xl);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
	RETURN;
}

static void XL_DESTROY_RECV_PACKETS(XL *xl)
{
	RAISE_SPL(SPL_XL);
	if (!IS_XL(xl)) {
		NET$PKTPOOL_FREE(&xl->pool);
	} else {
		while (xl->n_recv) {
			PACKET *p = RX_DESC(xl, xl->first_recv)->p;
			XL_UNMAP_RECV_PACKET(xl, p, xl->first_recv);
			FREE_PACKET(p, NULL, SPL_XL);
			xl->first_recv = (xl->first_recv + 1) & (N_RX_DESCS - 1);
			xl->n_recv--;
		}
	}
	LOWER_SPL(SPL_ZERO);
#if __DEBUG >= 2
	if (__unlikely(xl->recv_maps))
		KERNEL$SUICIDE("XL_DESTROY_RECV_PACKETS: RECEIVE MAPS LEAKED: %d", xl->recv_maps);
#endif
}

void XL_DESTROY_SENT_PACKETS(XL *xl)
{
	while (xl->n_sent) {
		PACKET *p;
		XL_UNMAP_SENT_PACKET(xl, TX_DESC(xl, xl->first_sent));
		p = TX_DESC(xl, xl->first_sent)->p;
		p->status = -EAGAIN;
		CALL_PKT(p);
		xl->first_sent = (xl->first_sent + 1) & (N_RX_DESCS - 1);
		xl->n_sent--;
	}
}

static void INIT_ROOT(HANDLE *h, void *data)
{
	XL *xl = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = xl;
	h->op = !IS_XL(xl) ? &EL_OPERATIONS : xl->flags2 & FLAG2_PIO ? &PIO_OPERATIONS : &MMIO_OPERATIONS;
}

static __u16 EL_ID_READ_EEPROM(io_t probe_io, unsigned idx)
{
	unsigned result, i;
	io_outb(probe_io, PROBE_EL_READ_EEPROM | idx);
	KERNEL$SLEEP((JIFFIES_PER_SECOND + 249) / 250);
	result = 0;
	for (i = 0; i < 16; i++) {
		result <<= 1;
		result |= io_inb(probe_io) & 1;
	}
	return result;
}

static int UNLOAD(void *p, void **release, const char * const argv[]);
static void DROP_TAG(__u8 tag);
static void XL_DESTROY_RECV_PACKETS(XL *xl);

#define IRQ_FLAGS(xl)	((xl)->bus != BUS_ISA ? IRQ_REQUEST_SHARED : IRQ_REQUEST_EXCLUSIVE)

int main(int argc, const char * const argv[])
{
	__s8 bus = -1;
	int errorlevel = 0;
	__u8 tag = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	unsigned long isa_probe = 0;
	char *net_input = NET_INPUT_DEVICE ":";
	char *address_str = NULL;
	char pio = 0;
	char *media_string = NULL;
	int bmcr = BMCR_ANENABLE;
	int defer_timer = 0;
	int extra_flags = 0;
	const char *opt, *optend, *str;
	int r;
	char dev_name[__MAX_STR_LEN];
	char *chip_name;
	unsigned long flags;
	unsigned flags2;
	__u8 *mem = NULL;	/* against warning */
	unsigned long io = 0;
	int irq = -1;
	__u16 address_config;
	__u32 internal_config;
	union {
		char a1[__MAX_STR_LEN];
		OPENRQ openrq;
		DLLZRQ lz;
		VDESC desc;
	} u;
	int i;
	XL *xl;
	const char * const *arg = argv;
	int state = 0;
	static const struct __param_table params[19] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2,
		"INPUT", __PARAM_STRING, 1, MAXINT,
		"ADDRESS", __PARAM_STRING, 1, MAXINT,
		"PIO", __PARAM_BOOL, ~0, 1,
		"MEDIA", __PARAM_STRING, 1, MAXINT,
		"FULL_DUPLEX", __PARAM_BOOL, BMCR_ANENABLE | BMCR_FULLDPLX, BMCR_FULLDPLX,
		"HALF_DUPLEX", __PARAM_BOOL, BMCR_ANENABLE | BMCR_FULLDPLX, 0,
		"10M", __PARAM_BOOL, BMCR_ANENABLE | BMCR_SPEED100, 0,
		"100M", __PARAM_BOOL, BMCR_ANENABLE | BMCR_SPEED100, BMCR_SPEED100,
		"AUTO", __PARAM_BOOL, BMCR_ANENABLE | BMCR_SPEED100 | BMCR_FULLDPLX, BMCR_ANENABLE,
		"DEFER_TIMER", __PARAM_INT, 0, (XL_W3_MacControl_DeferTimerSelect >> __BSF_CONST(XL_W3_MacControl_DeferTimerSelect) << 4) + 15 + 1,
		"DEFER_ONLY_AFTER_COLLISION", __PARAM_BOOL, DEFER_ONLY_COLLISION, DEFER_ONLY_COLLISION,
		"PAUSE_RECEIVE", __PARAM_BOOL, FLOW_CONTROL, FLOW_CONTROL,
		"NO_LINK_INTERRUPT", __PARAM_BOOL, NO_LINK_INTR, NO_LINK_INTR,
		"NO_TX_CHECKSUM", __PARAM_BOOL, NO_TX_CHECKSUM, NO_TX_CHECKSUM,
		"IO", __PARAM_UNSIGNED_LONG, 0x200, 0x3e1,
		"IRQ", __PARAM_INT, 0, 16,
		NULL, 0, 0, 0,
	};
	void *vars[19];
	static const struct __param_table isa_params[2] = {
		"PROBEIO", __PARAM_UNSIGNED_LONG, 0x100, 0x200,
		NULL, 0, 0, 0,
	};
	void *isa_vars[2];

	PKT_CHECK_VERSION;

	vars[0] = &errorlevel;
	vars[1] = &errorlevel;
	vars[2] = &net_input;
	vars[3] = &address_str;
	vars[4] = &pio;
	vars[5] = &media_string;
	vars[6] = &bmcr;
	vars[7] = &bmcr;
	vars[8] = &bmcr;
	vars[9] = &bmcr;
	vars[10] = &bmcr;
	vars[11] = &defer_timer;
	vars[12] = &extra_flags;
	vars[13] = &extra_flags;
	vars[14] = &extra_flags;
	vars[15] = &extra_flags;
	vars[16] = &io;
	vars[17] = &irq;
	vars[18] = NULL;
	isa_vars[0] = &isa_probe;
	isa_vars[1] = NULL;

	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (!__strcasexcmp("ISA", opt, optend)) {
			bus = BUS_ISA;
			if (str) {
				int xstate = 0;
				if (__parse_extd_param(&str, &xstate, isa_params, isa_vars, NULL, NULL, NULL, NULL)) goto bads;
			}
		} else if (!__strcasexcmp("PCI", opt, optend)) {
			u.lz.handle = KERNEL$DLRQ();
			u.lz.caller = NULL;
			strcpy(u.lz.modname, "PCI");
			SYNC_IO_CANCELABLE(&u.lz, KERNEL$DL_LOAD_LAZY);
			if (u.lz.status < 0) {
				if (u.lz.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: PCI NOT ENABLED: %s", u.lz.error);
				r = u.lz.status;
				goto ret0;
			}
			if (!PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
				bus = BUS_PCI;
			}
		} else {
			bads:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: SYNTAX ERROR");
			badsm:
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (bus == -1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: SPECIFY /ISA OR /PCI");
		goto badsm;
	}
	if (io & 0xf) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: BAD IO PORT");
		goto badsm;
	}

	if (bus == BUS_ISA) {
		unsigned probe_tries = 0;
		io_t probe_io;
		int i;
		IO_RANGE probe_range;
		__u16 lrs_state, dev_id, sw2, reg;
		retry_isa_probe:
		if (!isa_probe) {
			for (probe_io = 0x110; probe_io < 0x200; probe_io += 0x10) {
				if (probe_io == 0x170 || probe_io == 0x1f0) continue;	/* hard disk port */
				probe_range.start = probe_io;
				probe_range.len = 1;
				probe_range.name = "3C";
				if (KERNEL$REGISTER_IO_RANGE(&probe_range)) continue;
				io_outb(probe_io, 0x00);
				io_outb(probe_io, 0xff);
				if (!(io_inb(probe_io) & 0x01)) {
					KERNEL$UNREGISTER_IO_RANGE(&probe_range);
					continue;
				}
				goto found_probe_io;
			}
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: NO ISA PROBE PORT FOUND");
			r = -ENXIO;
			goto ret0;
		} else {
			probe_io = isa_probe;
			probe_range.start = probe_io;
			probe_range.len = 1;
			probe_range.name = "3C";
			if ((r = KERNEL$REGISTER_IO_RANGE(&probe_range))) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: CAN'T REGISTER ISA PROBE PORT AT "IO_FORMAT": %s", (io_t)probe_io, strerror(-r));
				goto ret0;
			}
			io_outb(probe_io, 0x00);
			io_outb(probe_io, 0xff);
			if (!(io_inb(probe_io) & 0x01)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: ISA PROBE PORT AT "IO_FORMAT" BUSY", (io_t)probe_io);
				r = -EBUSY;
				ret0_probe:
				KERNEL$UNREGISTER_IO_RANGE(&probe_range);
				goto ret0;
			}
		}
		found_probe_io:
		io_outb(probe_io, 0x00);
		io_outb(probe_io, 0x00);
		lrs_state = 0xff;
		for (i = 0; i < 255; i++) {
			io_outb(probe_io, lrs_state);
			lrs_state <<= 1;
			lrs_state = lrs_state & 0x100 ? lrs_state ^ 0xcf : lrs_state;
		}
		if (tag) {
			io_outb(probe_io, PROBE_EL_TEST_TAG | tag);
			io_outb(probe_io, PROBE_EL_SET_TAG);
		} else if (!tags) {
			io_outb(probe_io, PROBE_EL_SET_TAG);
		} else {
			io_outb(probe_io, PROBE_EL_TEST_TAG);
		}
		/*{
			for (i = 0; i < 0x40; i++) {
				__debug_printf("%02X: %04X\n", i, EL_ID_READ_EEPROM(probe_io, i));
			}
		}*/
		if (EL_ID_READ_EEPROM(probe_io, EE_EL_XL_ManufacturerId) != EE_EL_XL_ManufacturerIdValue) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: NO ISA DEVICE FOUND%s", probe_tries ? " (RETRIED)" : "");
			r = -ENODEV;
			goto ret0_probe;
		}
			/* take the card with lowest MAC address */
		for (i = EE_EL_XL_3Com_Node_Address_0; i < EE_EL_XL_3Com_Node_Address_0 + 3; i++) EL_ID_READ_EEPROM(probe_io, i);
		for (i = EE_EL_XL_OEM_Node_Address_0; i < EE_EL_XL_OEM_Node_Address_0 + 3; i++) EL_ID_READ_EEPROM(probe_io, i);

		if (!io) io = 0x200 + (((EL_ID_READ_EEPROM(probe_io, EE_EL_AddressConfiguration) & EE_EL_AddressConfiguration_Address) >> __BSF_CONST(EE_EL_AddressConfiguration_Address)) << 4);
		if (irq < 0) irq = (EL_ID_READ_EEPROM(probe_io, EE_EL_ResourceConfiguration) & EE_EL_ResourceConfiguration_IRQ) >> __BSF_CONST(EE_EL_ResourceConfiguration_IRQ);
		dev_id = EL_ID_READ_EEPROM(probe_io, EE_EL_XL_DeviceId);
		sw2 = EL_ID_READ_EEPROM(probe_io, EE_EL_Software_Information_2);
		if (!tag) tag = __BSF(~tags & ~1);
		if (tag == 8) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: ALL TAGS ARE IN USE");
			r = -ENODEV;
			goto ret0_probe;
		}
		tags |= 1 << tag;
		io_outb(probe_io, PROBE_EL_SET_TAG | tag);
		io_outb(probe_io, ((io >> 4) & 0x1f) | PROBE_EL_ACTIVATE);
		KERNEL$UNREGISTER_IO_RANGE(&probe_range);

		_snprintf(dev_name, sizeof(dev_name), "PKT$3C@" IO_ID, (io_t)io);
		/* My 3c509 card has a bug that toggling ENA in
		   EL_W0_ConfigControl occasionally drops I/O space
		   configuration.
		   It can be reproduced when loading and unloading the driver
		   in a loop.
		   Unfortunatelly, toggling ENA is required, otherwise
		   interrupts won't work.
		   If the I/O space is dropped, retry ISA activation.
		   We must remember tag to find the card, without I/O space
		   there is no other access to the card than tag.
		*/
		probe_range.start = io;
		probe_range.len = REGSPACE_3C5X9;
		probe_range.name = dev_name;
		if ((r = KERNEL$REGISTER_IO_RANGE(&probe_range))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, probe_range.start, probe_range.start + probe_range.len - 1, strerror(-r));
			goto ret0;
		}
		/*__debug_printf("status 1: %04x\n", io_inw(io + EL_XL_IntStatusCmd));*/
		io_outw(io + EL_XL_IntStatusCmd, EL_XL_Cmd_SetRegisterWindow);
		io_outw(io + EL_W0_ConfigControl, io_inw(io + EL_W0_ConfigControl) | EL_W0_ConfigControl_ENA);
		/*__debug_printf("status 2: %04x\n", io_inw(io + EL_XL_IntStatusCmd));*/
		reg = io_inw(io + EL_XL_IntStatusCmd);
		KERNEL$UNREGISTER_IO_RANGE(&probe_range);
		if (reg == 0xffff) {
			if (++probe_tries <= ISA_PROBE_RETRIES) {
				goto retry_isa_probe;
			}
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "UNABLE TO MAP IO SPACE");
			r = -EIO;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INITIALIZATION FAILED", dev_name);
			goto ret0;
		}

		if (probe_tries)
			KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, dev_name, "ISA PROBE WAS RETRIED %d TIMES", probe_tries);

		pio = 2;
		if ((sw2 & EE_EL_Software_Information_2_Revision) == EE_EL_Software_Information_2_Revision_B << __BSF_CONST(EE_EL_Software_Information_2_Revision)) flags = EL2;
		else flags = EL;
		flags2 = FLAG2_PIO;
		chip_name = !(flags & EL2) ? "3C509 (UNKNOWN)" : "3C509B (UNKNOWN)";
		for (i = 0; isa_cards[i].string_a; i++) {
			if (isa_cards[i].id == dev_id) {
				chip_name = !(flags & EL2) ? isa_cards[i].string_a : isa_cards[i].string_b;
				break;
			}
		}
	} else if (bus == BUS_PCI) {
		flags2 = 0;
		if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: NO PCI DEVICE FOUND");
			r = -ENODEV;
			goto ret0;
		}
		_snprintf(dev_name, sizeof(dev_name), "PKT$3C@" PCI_ID_FORMAT, id);

		if (!(flags & (CYCLONE | TORNADO))) pio = 1;
		if (!pio) {
			PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
			mem = PCI$MAP_MEM_RESOURCE(id, 1, REGSPACE(flags));
			if (!mem) pio = 1;
			if (__IS_ERR(mem)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
				r = __PTR_ERR(mem);
				goto ret1;
			}
		}
		if (pio) {
			PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
			io = PCI$READ_IO_RESOURCE(id, 0);
			if (!io) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO%s RESOURCE", flags & (CYCLONE | TORNADO) ? " OR MEMORY" : "");
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO RESOURCE", dev_name);
				r = -ENXIO;
				goto ret1;
			}
			flags2 = FLAG2_PIO;
		}
		irq = PCI$READ_INTERRUPT_LINE(id);
	} else {
		KERNEL$SUICIDE("3C: UNKNOWN BUS %d", bus);
	}

	xl = KERNEL$ALLOC_CONTIG_AREA(XL_SIZE, AREA_DATA | AREA_PCIDMA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)DESC_ALIGN);
	if (__IS_ERR(xl)) {
		r = __PTR_ERR(xl);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE DMA DESCRIPTORS", dev_name);
		goto ret2;
	}
	memset(xl, 0, sizeof(XL));
	xl->flags = flags | extra_flags;
	xl->flags2 = flags2;
	xl->errorlevel = errorlevel;
	xl->bus = bus;
	xl->window = 8;
	xl->tag = tag;
	xl->id = id;
	xl->irq = irq;
	INIT_PKTPOOL(&xl->pool, PKT_SIZE - ETHERNET_HEADER + 3);
	xl->vbuf.ptr = xl->packet_tmp;
	xl->vbuf.len = ETHERNET_MTU;
	xl->vbuf.spl = SPL_X(SPL_XL);
	strcpy(xl->dev_name, dev_name);
	if (pio) {
		xl->u.io = io;
		xl->range.start = io;
		xl->range.len = REGSPACE(flags);
		xl->range.name = xl->dev_name;
		if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&xl->range))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, xl->range.start, xl->range.start + xl->range.len - 1, strerror(-r));
			goto ret25;
		}
	} else {
		xl->u.mem = mem;
	}

	if (bus == BUS_PCI) xl->pkt_id = IDTYPE_PCI | (xl->id & IDTYPE_MASK);
	else xl->pkt_id = IDTYPE_IOPORT | (xl->u.io & IDTYPE_MASK);

	WQ_INIT(&xl->link_state_wait, "3C$LINK_STATE_WAIT");
	for (i = 0; i < N_MCAST; i++) WQ_INIT(&xl->mcast_table[i], "RTL$MCAST_TABLE");
	xl->max_sent = DEFAULT_SENT;
	xl->timer.fn = TRANSMIT_TIMER;
	INIT_TIMER(&xl->timer);
	SET_TIMER_NEVER(&xl->timer);
	xl->media_timer.fn = MEDIA_TIMER;
	INIT_TIMER(&xl->media_timer);
	SET_TIMER_NEVER(&xl->media_timer);
	xl->phy_id = -1;
	xl->bmcr = bmcr;
	xl->defer_timer = defer_timer;
	if (!IS_V2(xl)) xl->flags |= NO_TX_CHECKSUM;

	if (IS_XL(xl)) {
		VDMA dma;
		u.desc.ptr = (unsigned long)DESCS_START(xl);
		u.desc.len = DESCS_LEN;
		u.desc.vspace = &KERNEL$VIRTUAL;
		dma.spl = SPL_X(SPL_ZERO);
		RAISE_SPL(SPL_VSPACE);
		xl->desc_dmaunlock = KERNEL$VIRTUAL.op->vspace_dmalock(&u.desc, PF_READ | PF_WRITE, &dma);
		xl->desc_dmaaddr = dma.ptr;
	}

	if (!IS_XL(xl) && xl->bmcr & BMCR_SPEED100) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: THIS CARD DOESN'T SUPPORT 100M", xl->dev_name);
		r = -EOPNOTSUPP;
		goto ret3;
	}

	xl->reset_thread.thread = KERNEL$ALLOC_THREAD_SYNC();
	if (__IS_ERR(xl->reset_thread.thread)) {
		r = __PTR_ERR(xl->reset_thread.thread);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE THREAD: %s", xl->dev_name, strerror(-r));
		xl->reset_thread.thread = NULL;
		goto ret3;
	}

	if (address_str) {
		if (NET$PARSE_ETHERNET_ADDRESS(address_str, xl->address) || !NET$VALID_ETHERNET_ADDRESS(xl->address)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: INVALID ADDRESS: %s", address_str);
			r = -EBADSYN;
			goto ret3;
		}
		xl->flags2 |= FLAG2_ADDRESS_OVERRIDE;
	}

	if (xl->flags & CARDBUS) {
/* it is expected that if xl->flags & CARDBUS then xl->cb_mem is accessible */
		__u8 *cb_mem = PCI$MAP_MEM_RESOURCE(id, 2, REGSPACE_CB);
		if (!cb_mem) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO CARDBUS MEMORY RESOURCE");
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO CARDBUS MEMORY RESOURCE", dev_name);
			r = -ENXIO;
			goto ret3;
		}
		if (__IS_ERR(cb_mem)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP CARDBUS MEMORY RESOURCE: %s", dev_name, strerror(-__PTR_ERR(cb_mem)));
			r = __PTR_ERR(cb_mem);
			goto ret3;
		}
		xl->cb_mem = cb_mem;
	}

	if ((r = NETQUE$ALLOC_QUEUE(&xl->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", dev_name, strerror(-r));
		goto ret3;
	}

	if (IS_XL(xl)) {
		if ((r = KERNEL$RESERVE_BOUNCE(dev_name, N_RX_DESCS * N_RX_FRAGS + N_TX_DESCS * N_TX_FRAGS, PKT_SIZE, 0))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT RESERVE BOUNCE BUFFERS: %s", dev_name, strerror(-r));
			goto ret4;
		}
	}

	xl->first_recv = 0;
	xl->n_recv = 0;
	xl->first_sent = 0;
	xl->n_sent = 0;
	if (!IS_XL(xl)) {
		if ((r = NET$PKTPOOL_RESERVE(&xl->pool))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET POOL: %s", dev_name, strerror(-r));
			goto ret5;
		}
	} else {
		for (i = 0; i < N_RX_DESCS; i++) {
			PACKET *p;
			RAISE_SPL(SPL_XL);
			alloc_again:
			ALLOC_PACKET(p, PKT_SIZE - ETHERNET_HEADER, NULL, SPL_XL, {
				if ((r = NET$MEMWAIT_SYNC())) {
					if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE PACKETS", dev_name);
					goto ret5;
				}
				goto alloc_again;
			});
			XL_MAP_RECV_PACKET(xl, p, i);
			xl->n_recv = i + 1;
			LOWER_SPL(SPL_ZERO);
		}
	}

	u.openrq.flags = O_WRONLY;
	u.openrq.path = net_input;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", dev_name, net_input, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto ret5;
	}
	xl->packet_input = u.openrq.status;
	xl->outstanding = 0;

	r = INIT(xl);
	if (r) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INITIALIZATION FAILED", dev_name);
		goto ret6;
	}

	if (xl->bus == BUS_PCI && !(xl->eeprom[EE_XL_Software_Information_2] & EE_XL_Software_Information_2_FixedMWIBug)) {
		__u16 cmd = PCI$READ_CONFIG_WORD(id, PCI_COMMAND);
		if (cmd & PCI_COMMAND_INVALIDATE) {
			cmd &= ~PCI_COMMAND_INVALIDATE;
			PCI$WRITE_CONFIG_WORD(id, PCI_COMMAND, cmd);
		}
	}

	address_config = 0;
	internal_config = 0;
	if (!IS_XL(xl)) {
		WINDOW(xl, 0);
		xl->config_control = READ_16(xl, EL_W0_ConfigControl);
		address_config = xl->eeprom[EE_EL_AddressConfiguration];
		/*__debug_printf("addr: %04X\n", READ_16(xl, EL_W0_AddressConfiguration));
		__debug_printf("res: %04X\n", READ_16(xl, EL_W0_ResourceConfiguration));
		__debug_printf("config control %04x\n", xl->config_control);*/
	} else {
		WINDOW(xl, 3);
		xl->media_options = READ_16(xl, X2_W3_MediaOptions);
		internal_config = READ_32(xl, XL_W3_InternalConfig);
	}

	if (media_string) {
		if (!_strcasecmp(media_string, "AUTOSELECT")) goto media_auto;
		if (!_strcasecmp(media_string, "UNKNOWN")) goto media_unknown;
		if (!_strcasecmp(media_string, "DEFAULT")) goto media_default;
		for (i = 0; i < N_MEDIA; i++) {
			if (!_strcasecmp(media[i].name, media_string)) {
				if (!SUPPORTED_MEDIA(xl, media[i].flags)) {
					_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: THIS CARD DOESN'T SUPPORT MEDIA \"%s\"", dev_name, media_string);
					r = -EOPNOTSUPP;
					goto ret6;
				}
				xl->media = i;
				goto media_ok;
			}
		}
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID MEDIA STRING \"%s\"", dev_name, media_string);
		r = -EBADSYN;
		goto ret6;
	} else {
		media_default:
		if (!IS_XL(xl)) {
			if (!(address_config & EE_EL_AddressConfiguration_AutoSelect)) {
				xl->media = FIND_MEDIA_BY_INTERNAL_CONFIG(xl, address_config, internal_config);
				if (xl->media != -1) goto media_ok;
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNKNOWN MEDIA IN ADDRESSCONFIG REGISTER (%04X), ASSUMING AUTOSELECT", address_config);
			}
		} else {
			if (!(internal_config & XL_InternalConfig_AutoSelect)) {

				xl->media = FIND_MEDIA_BY_INTERNAL_CONFIG(xl, address_config, internal_config);
				if (xl->media != -1) goto media_ok;
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNKNOWN MEDIA IN INTERNALCONFIG REGISTER (%08X), ASSUMING AUTOSELECT", internal_config);
			}
			if (!(xl->media_options & (X2_W3_MediaOptions_BaseT4Available | X2_W3_MediaOptions_BaseTXAvailable | X2_W3_MediaOptions_BaseFXAvailable | X2_W3_MediaOptions_10bTAvailable | X2_W3_MediaOptions_CoaxAvailable | X2_W3_MediaOptions_AuiAvailable | X2_W3_MediaOptions_MiiConnector)))
				xl->media_options = X2_W3_MediaOptions_MiiConnector;
			if (xl->media_options & X2_W3_MediaOptions_BaseTXAvailable)
				xl->media_options &= ~X2_W3_MediaOptions_10bTAvailable;
		}
		media_auto:
		xl->flags2 |= FLAG2_MEDIA_AUTOSELECT;
		xl->media = AUTOSELECT_MEDIA(xl, -1);
		if (xl->media != -1) {
			if (xl->media == AUTOSELECT_MEDIA(xl, xl->media)) xl->flags2 &= ~FLAG2_MEDIA_AUTOSELECT;
		} else {
			xl->flags2 &= ~FLAG2_MEDIA_AUTOSELECT;
			xl->media = FIND_MEDIA_BY_INTERNAL_CONFIG(xl, address_config, internal_config);
			if (xl->media != -1) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "AUTOSELECT COULDN'T FIND MEDIA, ASSUMING %s", media[xl->media].name);
				goto media_ok;
			}
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "AUTOSELECT COULDN'T FIND MEDIA, LEAVING MEDIA SETTINGS UNCHANGED");
			media_unknown:
			xl->media = 0;
			xl->flags2 |= FLAG2_MEDIA_UNKNOWN;
		}
	}
	media_ok:

	RESET_3(xl, 1);

	_printf("%s: %s", dev_name, chip_name);
	if (bus == BUS_PCI) _printf(" ON PCI: %s", PCI$ID(u.a1, id));
	_printf("\n");
	_printf("%s: %s %"__64_format"X, IRQ %d\n", dev_name, pio ? "IO" : "MEM", pio ? (__u64)io : PCI$READ_MEM_RESOURCE(id, 1, NULL), irq);
	_printf("%s: ADDRESS %s%s\n", dev_name, NET$PRINT_ETHERNET_ADDRESS(u.a1, xl->address), xl->flags2 & FLAG2_ADDRESS_OVERRIDE ? " (OVERRIDEN)" : "");

	RAISE_SPL(SPL_XL);
	xl->irq_ast.fn = !IS_XL(xl) ? EL_IRQ : pio ? PIO_IRQ : MMIO_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(irq, &xl->irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_FLAGS(xl), NULL, &xl->irq_ast, dev_name)) < 0) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret6;
	}
	CONTINUE_RESET(xl);
	LOWER_SPL(SPL_ZERO);

	r = KERNEL$REGISTER_DEVICE(dev_name, "3C.SYS", 0, xl, INIT_ROOT, NULL, NULL, NULL, UNLOAD, &xl->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret7;
	}
	xl->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret7:
	KERNEL$MASK_IRQ(xl->irq_ctrl);
	RAISE_SPL(SPL_XL);
	while (xl->flags2 & FLAG2_RESETTING) KERNEL$SLEEP(1);
	xl->flags2 |= FLAG2_DEAD;
	KERNEL$DEL_TIMER(&xl->timer);
	KERNEL$DEL_TIMER(&xl->media_timer);
	RESET_1(xl);
	RESET_2(xl);
	RESET_3(xl, -1);
	XL_DESTROY_SENT_PACKETS(xl);
	KERNEL$RELEASE_IRQ(xl->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_FLAGS(xl), NULL, &xl->irq_ast);
	LOWER_SPL(SPL_ZERO);
	ret6:
	while (xl->outstanding) KERNEL$SLEEP(1);
	KERNEL$FAST_CLOSE(xl->packet_input);
	ret5:
	XL_DESTROY_RECV_PACKETS(xl);
	if (IS_XL(xl)) {
		KERNEL$UNRESERVE_BOUNCE(xl->dev_name, N_RX_DESCS * N_RX_FRAGS + N_TX_DESCS * N_TX_FRAGS, PKT_SIZE, 0);
	}
	ret4:
	NETQUE$FREE_QUEUE(xl->queue);
	ret3:
	if (xl->reset_thread.thread) KERNEL$FREE_THREAD(xl->reset_thread.thread);
	if (xl->cb_mem) PCI$UNMAP_MEM_RESOURCE(id, xl->cb_mem, REGSPACE_CB);
	if (IS_XL(xl)) {
		xl->desc_dmaunlock(xl->desc_dmaaddr);
	}
	WQ_WAKE_ALL(&xl->link_state_wait);
	if (xl->flags2 & FLAG2_PIO) KERNEL$UNREGISTER_IO_RANGE(&xl->range);
	ret25:
	KERNEL$FREE_CONTIG_AREA(xl, XL_SIZE);
	ret2:
	if (!pio) PCI$UNMAP_MEM_RESOURCE(id, mem, REGSPACE(flags));
	ret1:
	if (bus == BUS_ISA) {
		io_outw(io + EL_XL_IntStatusCmd, EL_XL_Cmd_GlobalReset);
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 10);
	}
	if (bus == BUS_PCI) PCI$FREE_DEVICE(id);
	ret0:
	DROP_TAG(tag);
	return r;
}

static int UNLOAD(void *p, void **release, const char * const argv[])
{
	unsigned i;
	int r;
	XL *xl = p;
	if ((r = KERNEL$DEVICE_UNLOAD(xl->lnte, argv))) return r;
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&xl->mcast_table[i]);
	KERNEL$MASK_IRQ(xl->irq_ctrl);
	RAISE_SPL(SPL_XL);
	while (xl->flags2 & FLAG2_RESETTING) KERNEL$SLEEP(1);
	xl->flags2 |= FLAG2_DEAD;
	KERNEL$DEL_TIMER(&xl->timer);
	KERNEL$DEL_TIMER(&xl->media_timer);
	RESET_1(xl);
	RESET_2(xl);
	RESET_3(xl, -1);
	XL_DESTROY_SENT_PACKETS(xl);
	KERNEL$RELEASE_IRQ(xl->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_FLAGS(xl), NULL, &xl->irq_ast);
	LOWER_SPL(SPL_ZERO);
	while (xl->outstanding) KERNEL$SLEEP(1);
	KERNEL$FAST_CLOSE(xl->packet_input);
	XL_DESTROY_RECV_PACKETS(xl);
	if (IS_XL(xl)) {
		KERNEL$UNRESERVE_BOUNCE(xl->dev_name, N_RX_DESCS * N_RX_FRAGS + N_TX_DESCS * N_TX_FRAGS, PKT_SIZE, 0);
	}
	NETQUE$FREE_QUEUE(xl->queue);
	if (xl->cb_mem) PCI$UNMAP_MEM_RESOURCE(xl->id, xl->cb_mem, REGSPACE_CB);
	if (IS_XL(xl)) {
		xl->desc_dmaunlock(xl->desc_dmaaddr);
	}
	if (!(xl->flags2 & FLAG2_PIO)) PCI$UNMAP_MEM_RESOURCE(xl->id, xl->u.mem, REGSPACE(xl->flags));
	if (xl->bus == BUS_PCI) PCI$FREE_DEVICE(xl->id);
	*release = xl->dlrq;
	WQ_WAKE_ALL(&xl->link_state_wait);
	KERNEL$FREE_THREAD(xl->reset_thread.thread);
	if (xl->bus == BUS_ISA) {
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_GlobalReset);
		KERNEL$SLEEP((JIFFIES_PER_SECOND + 9) / 10);
	}
	if (xl->flags2 & FLAG2_PIO) KERNEL$UNREGISTER_IO_RANGE(&xl->range);
	DROP_TAG(xl->tag);
	KERNEL$FREE_CONTIG_AREA(xl, XL_SIZE);
	return 0;
}

static void DROP_TAG(__u8 tag)
{
	if (!tag) return;
	if (tag >= 8)
		KERNEL$SUICIDE("DROP_TAG: INVALID TAG %d, TAGS %02X", tag, tags);
	if (!(tags & (1 << tag)))
		KERNEL$SUICIDE("DROP_TAG: TAG %d NOT ACTIVE, TAGS %02X", tag, tags);
	tags &= ~(1 << tag);
}

DECL_IOCALL(DLL$UNLOAD, SPL_XL, DLINITRQ)
{
	if (tags) KERNEL$SUICIDE("DLL$UNLOAD: 3COM TAGS LEAKED: %02X", tags);
	RQ->status = 0;
	RETURN_AST(RQ);
}
