#include <ARCH/IO.H>
#include <ARCH/MOV.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/PKT.H>
#include <SPAD/IOCTL.H>

#include "3C.H"
#include "3CREG.H"

#define MAX_INTERRUPT_WORK	16
#define MAX_RECEIVE_LOOPS	128
#define MAX_TXSTATUS_LOOPS	40

#define READ_8(xl, i)		io_inb((xl)->u.io + (i))
#define READ_16(xl, i)		io_inw((xl)->u.io + (i))
#define READ_32(xl, i)		io_inl((xl)->u.io + (i))
#define WRITE_8(xl, i, v)	io_outb((xl)->u.io + (i), (v))
#define WRITE_16(xl, i, v)	io_outw((xl)->u.io + (i), (v))
#define WRITE_32(xl, i, v)	io_outl((xl)->u.io + (i), (v))

#include "3CFN.I"

static IO_STUB EL_PACKET;
static int EL_TX(XL *xl, PACKET *p);
static AST_STUB EL_PACKET_RETURNED;
static int EL_SLOW_IRQ(XL *xl, __u16 status);
static void EL_RX_ERROR(XL *xl, __u16 status);

const HANDLE_OPERATIONS EL_OPERATIONS = {
	SPL_X(SPL_XL),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	IOCTL,			/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	EL_PACKET,		/* pktio */
};

static DECL_IOCALL(EL_PACKET, SPL_XL, PACKET)
{
	XL *xl;
	HANDLE *h;
	__CHECK_PKT(RQ, "EL_PACKET");
	h = RQ->handle;
	if (__unlikely(h->op != &EL_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_XL));
	xl = h->fnode;
	if (__unlikely(xl->flags2 & (FLAG2_RESETTING | FLAG2_DEAD))) {
		if (xl->flags2 & FLAG2_DEAD) {
			RQ->status = -ENODEV;
			goto ret_pkt;
		}
		goto enq;
	}
	if (__unlikely(RQ->data_length + RQ->v.len > PKT_SIZE - ETHERNET_HEADER)) {
		RQ->status = -EMSGSIZE;
		ret_pkt:
		RETURN_PKT(RQ);
	}
	if (__likely(!NETQUE$QUEUE_EMPTY(xl->queue))) {
		enq:
		NETQUE$ENQUEUE_PACKET(xl->queue, RQ);
	} else if (__unlikely(EL_TX(xl, RQ))) {
		KERNEL$DEL_TIMER(&xl->timer);
		KERNEL$SET_TIMER(TX_TIMEOUT, &xl->timer);
		goto enq;
	}
	RETURN;
}

static int EL_TX(XL *xl, PACKET *p)
{
	unsigned len;
	__u8 *ind;
	vspace_unmap_t *unmap;
	WINDOW(xl, 1);
	READ_16(xl, EL_X1_W1_TxFree);
	if (__unlikely(READ_16(xl, EL_X1_W1_TxFree) < ETHERNET_HEADER + p->data_length + p->v.len + FIFO_OVERHEAD)) {
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_Cmd_SetTxAvailableThresh | (PKT_SIZE + FIFO_OVERHEAD));
		return 1;
	}
	CHECKSUM_PACKET(p, {
		p->status = r_;
		goto ret_pkt;
	});
	ind = NET$MAP_INDIRECT_PACKET(&xl->vbuf, &p->v, &unmap);
	if (__unlikely(__IS_ERR(ind))) {
		p->flags |= PKT_PAGING_ERROR;
		p->status = __PTR_ERR(ind);
		goto ret_pkt;
	}
	len = ETHERNET_HEADER + p->data_length + p->v.len;
	WRITE_16(xl, EL_W1_Pio, len);
	WRITE_16(xl, EL_W1_Pio, 0);
	len = (len + 3) & ~3;
	if (__likely(!ind)) {
		io_outsl(xl->u.io + EL_W1_Pio, &p->data[LL_HEADER - ETHERNET_HEADER], len >> 2);
	} else {
		io_outsw(xl->u.io + EL_W1_Pio, &p->data[LL_HEADER - ETHERNET_HEADER], (ETHERNET_HEADER + p->data_length) >> 1);
		len -= ETHERNET_HEADER + p->data_length;
		io_outsl(xl->u.io + EL_W1_Pio, ind, len >> 2);
		if (__unlikely(len & 2)) io_outsw(xl->u.io + EL_W1_Pio, ind + len - 2, 1);
		unmap(ind);
	}
	p->status = 0;
	ret_pkt:
	CALL_PKT(p);
	return 0;
}

void EL_DEQUEUE(XL *xl)
{
	int first = 1;
	PACKET *p;
	dequeue_next:
	p = NETQUE$DEQUEUE(xl->queue);
	if (!p) {
		disable_timer:
		KERNEL$DEL_TIMER(&xl->timer);
		SET_TIMER_NEVER(&xl->timer);
		return;
	}
	__CHECK_PKT(p, "EL_DEQUEUE");
	if (__unlikely(EL_TX(xl, p))) {
		if (__unlikely(first)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNABLE TO SEND PACKET, TXFREE %04X", READ_16(xl, EL_X1_W1_TxFree));
			p->status = -EIO;
			CALL_PKT(p);
			NETQUE$DISCARD_PACKETS(xl->queue, -EIO);
			goto disable_timer;
		} else {
			NETQUE$REQUEUE_DEQUEUED_PACKET(xl->queue, p);
			KERNEL$DEL_TIMER(&xl->timer);
			KERNEL$SET_TIMER(TX_TIMEOUT, &xl->timer);
		}
		return;
	}
	first = 0;
	goto dequeue_next;
}

static int EL_RX(XL *xl)
{
	unsigned loops = MAX_RECEIVE_LOOPS + 1;
	__u16 rx_status;
	WINDOW(xl, 1);
	while ((__s16)(rx_status = READ_16(xl, EL_X1_W1_RxStatus)) > 0) {
		if (__unlikely(!--loops)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "STUCK IN RECEIVE LOOP, RX STATUS %04X", rx_status);
			RESET_DEQUEUE(xl);
			return 1;
		}
		if (__unlikely(rx_status & EL_X1_W1_RxStatus_RxError)) {
			EL_RX_ERROR(xl, rx_status);
		} else {
			PACKET *p;
			unsigned pkt_len = rx_status & EL_W1_RxStatus_RxBytes;
			if (__unlikely(pkt_len - 60 > PKT_SIZE - 60)) {
				if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "INVALID PACKET LENGTH %d", pkt_len);
				goto skip;
			}
			ALLOC_PACKET(p, pkt_len - ETHERNET_HEADER + 3, &xl->pool, SPL_XL, {
				if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_SW_WARNING, xl->dev_name, "OUT OF MEMORY FOR PACKET");
				goto skip;
			});
			p->data_length = pkt_len - ETHERNET_HEADER;
			io_insl(xl->u.io + EL_W1_Pio, p->data + LL_HEADER - ETHERNET_HEADER, (pkt_len + 3) >> 2);
			p->addrlen = ETHERNET_ADDRLEN;
			p->fn = EL_PACKET_RETURNED;
			p->sender_data = xl;
			p->h = xl->packet_input;
			p->id = xl->pkt_id;
			xl->outstanding++;
			CALL_IORQ(p, KERNEL$PKTIO);
		}
		skip:
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_Cmd_DiscardTopPacket);
		WAIT(xl, "DISCARD TOP PACKET");
	}
	return 0;
}

static DECL_AST(EL_PACKET_RETURNED, SPL_XL, PACKET)
{
	XL *xl;
	PKT_AST_ENTER(RQ);
	xl = RQ->sender_data;
	FREE_PACKET(RQ, &xl->pool, SPL_XL);
	xl->outstanding--;
	RETURN;
}

DECL_IRQ_AST(EL_IRQ, SPL_XL, AST)
{
	XL *xl = GET_STRUCT(RQ, XL, irq_ast);
	__u16 status;
	unsigned work;
	if (__unlikely(xl->flags2 & (FLAG2_RESETTING | FLAG2_DEAD))) {
		if (xl->flags2 & FLAG2_RESETTING) {
			xl->flags2 |= FLAG2_RESETTING_INTR;
			goto ret;
		}
		goto end;
	}
	work = MAX_INTERRUPT_WORK + 1;
	while ((status = READ_16(xl, EL_XL_IntStatusCmd)) & (EL_XL_IntStatus_InterruptLatch | EL_XL_IntStatus_HostError | EL_XL_IntStatus_TxComplete | EL_IntStatus_TxAvailable | EL_XL_IntStatus_RxComplete)) {
		if (__unlikely(!--work)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "TOO MUCH WORK IN INTERRUPT, STATUS %04X", status);
			RESET_DEQUEUE(xl);
			goto end;
		}
		WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_AcknowledgeInterrupt | (status & ~(EL_XL_IntStatus_CmdInProgress | EL_XL_IntStatus_WindowNumber)));
		if (__unlikely(status == 0xffff)) {
			xl->flags2 |= FLAG2_DEAD;
			goto end;
		}
		if (__likely(status & EL_XL_IntStatus_RxComplete)) {
			if (__unlikely(EL_RX(xl))) goto end;
		}
		if (__likely(status & EL_IntStatus_TxAvailable)) {
			EL_DEQUEUE(xl);
		}
		if (__unlikely(status & (EL_XL_IntStatus_HostError | EL_XL_IntStatus_TxComplete))) {
			if (EL_SLOW_IRQ(xl, status)) goto end;
		}
	}

	end:
	KERNEL$UNMASK_IRQ(xl->irq_ctrl);
	ret:
	RETURN;
}

static int EL_SLOW_IRQ(XL *xl, __u16 status)
{
	if (__unlikely(status & EL_XL_IntStatus_HostError)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "HOST ERROR, STATUS %04X", status);
		reset_it:
		RESET_DEQUEUE(xl);
		return 1;
	}
	if (status & EL_XL_IntStatus_TxComplete) {
		__u8 tx_status;
		unsigned loops;
		WINDOW(xl, 1);
		loops = MAX_TXSTATUS_LOOPS + 1;
		while ((tx_status = READ_8(xl, EL_X1_W1_TxStatus)) & EL_X1_W1_TxStatus_Complete) {
			if (__unlikely(!--loops)) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "STUCK IN TRANSMIT STATUS LOOP, TX STATUS %02X", tx_status);
				goto reset_it;
			}
			if (REPORT_TX_ERROR(xl, tx_status)) goto reset_it;
			if (tx_status & (EL_X1_W1_TxStatus_Underrun | EL_X1_W1_TxStatus_Jabber)) {
				WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxReset);
				WAIT(xl, "TX RESET");
			}
			if (tx_status & (EL_X1_W1_TxStatus_TxStatusOverflow | EL_X1_W1_TxStatus_MaxCollisions | EL_X1_W1_TxStatus_Underrun | EL_X1_W1_TxStatus_Jabber)) {
				WRITE_16(xl, EL_XL_IntStatusCmd, EL_XL_Cmd_TxEnable);
			}
			SET_TX_THRESH(xl);
			WRITE_8(xl, EL_X1_W1_TxStatus, 0);
		}
	}
	return 0;
}

static void EL_RX_ERROR(XL *xl, __u16 status)
{
	if (status == 0xffff) return;
	if (xl->flags2 & FLAG2_MEDIA_UNKNOWN) return;	/* default media produces a lot of errors --- do not flood log with them */
	switch (status & EL_W1_RxStatus_Code) {
		case EL_W1_RxStatus_Code_Overrun:
			if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_HW_WARNING, xl->dev_name, "RECEIVE OVERRUN");
			break;
		case EL_W1_RxStatus_Code_Oversized:
			if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "TOO LONG PACKET");
			break;
		case EL_W1_RxStatus_Code_DribbleBits:
			if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "DRIBBLE BITS");
			break;
		case EL_W1_RxStatus_Code_RuntPacket:
			if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "RUNT PACKET");
			break;
		case EL_W1_RxStatus_Code_FramingError:
			if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "FRAMING ERROR");
			break;
		case EL_W1_RxStatus_Code_CrcError:
			if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "CRC ERROR");
			break;
		default:
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, xl->dev_name, "UNKNOWN RECEIVE STATUS %04X", status);
			break;
	}
}
