#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/TIME.H>
#include <SPAD/IOCTL.H>
#include <VALUES.H>

#include "IPGREG.H"
#include "IPG.H"

/* partially based on

  Intel PRO/1000 Linux driver
  Copyright(c) 1999 - 2006 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  Linux NICS <linux.nics@intel.com>
  e1000-devel Mailing List <e1000-devel@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497
*/

#define PHY_RESET_DISABLE	0

static const struct pci_id_s pci_cards[] = {
	{ 0x8086, 0x1000, PCI_ANY_ID, PCI_ANY_ID, 0x02, 0xff, "82542 2.0", MAC_82542_rev2_0 | (MEDIA_FIBER << MEDIA_SHIFT) },
	{ 0x8086, 0x1000, PCI_ANY_ID, PCI_ANY_ID, 0x03, 0xff, "82542 2.1", MAC_82542_rev2_1 | (MEDIA_FIBER << MEDIA_SHIFT) },
	{ 0x8086, 0x1001, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82543GC FIBER", MAC_82543 },
	{ 0x8086, 0x1004, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82543GC COPPER", MAC_82543 },
	{ 0x8086, 0x1008, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82544EI COPPER", MAC_82544 },
	{ 0x8086, 0x1009, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82544EI FIBER", MAC_82544 },
	{ 0x8086, 0x100c, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82544GC COPPER", MAC_82544 },
	{ 0x8086, 0x100d, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82544GC LOM", MAC_82544 },
	{ 0x8086, 0x100e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82540EM", MAC_82540 },
	{ 0x8086, 0x100f, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82545EM COPPER", MAC_82545 },
	{ 0x8086, 0x1010, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546EB COPPER", MAC_82546 },
	{ 0x8086, 0x1011, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82545EM FIBER", MAC_82545 },
	{ 0x8086, 0x1012, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546EB FIBER", MAC_82546 },
	{ 0x8086, 0x1013, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541EI", MAC_82541 },
	{ 0x8086, 0x1014, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541ER LOM", MAC_82541 },
	{ 0x8086, 0x1015, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82540EM LOM", MAC_82540 },
	{ 0x8086, 0x1016, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82540EP LOM", MAC_82540 },
	{ 0x8086, 0x1017, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82540EP", MAC_82540 },
	{ 0x8086, 0x1018, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541EI MOBILE", MAC_82541 },
	{ 0x8086, 0x1019, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82547EI", MAC_82547 },
	{ 0x8086, 0x101a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82547EI MOBILE", MAC_82547 },
	{ 0x8086, 0x101d, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546EB QUAD COPPER", MAC_82546 },
	{ 0x8086, 0x101e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82540EP LP", MAC_82540 },
	{ 0x8086, 0x1026, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82545GM COPPER", MAC_82545_rev_3 },
	{ 0x8086, 0x1027, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82545GM FIBER", MAC_82545_rev_3 },
	{ 0x8086, 0x1028, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82545GM SERDES", MAC_82545_rev_3 | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x1049, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IGP M AMT", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x104a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IGP AMT", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x104b, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IGP C", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x104c, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IFE", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x104d, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IGP M", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x105e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB COPPER", MAC_82571 },
	{ 0x8086, 0x105f, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB FIBER", MAC_82571 },
	{ 0x8086, 0x1060, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB SERDES", MAC_82571 | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x1075, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82547GI", MAC_82547_rev_2 },
	{ 0x8086, 0x1076, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541GI", MAC_82541_rev_2 },
	{ 0x8086, 0x1077, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541GI MOBILE", MAC_82541_rev_2 },
	{ 0x8086, 0x1078, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541ER", MAC_82541_rev_2 },
	{ 0x8086, 0x1079, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546GB COPPER", MAC_82546_rev_3 },
	{ 0x8086, 0x107a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546GB FIBER", MAC_82546_rev_3 },
	{ 0x8086, 0x107b, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546GB SERDES", MAC_82546_rev_3 | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x107c, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82541GI LF", MAC_82541_rev_2 },
	{ 0x8086, 0x107d, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82572EI COPPER", MAC_82572 },
	{ 0x8086, 0x107e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82572EI FIBER", MAC_82572 },
	{ 0x8086, 0x107f, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82572EI SERDES", MAC_82572 | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x108a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546GB PCIE", MAC_82546_rev_3 },
	{ 0x8086, 0x108b, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82573E", MAC_82573 | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x108c, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82573E IAMT", MAC_82573 | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x1096, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "80003ES2LAN COPPER DPT", MAC_80003es2lan },
	{ 0x8086, 0x1098, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "80003ES2LAN SERDES DPT", MAC_80003es2lan | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x1099, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546GB QUAD COPPER", MAC_82546_rev_3 },
	{ 0x8086, 0x109a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82573L", MAC_82573 | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x10a4, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB QUAD COPPER", MAC_82571 },
	{ 0x8086, 0x10a5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB QUAD FIBER", MAC_82571 },
	{ 0x8086, 0x10b5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82546GB QUAD COPPER KSP3", MAC_82546_rev_3 },
	{ 0x8086, 0x10b9, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82572EI", MAC_82572 },
	{ 0x8086, 0x10ba, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "80003ES2LAN COPPER SPT", MAC_80003es2lan },
	{ 0x8086, 0x10bb, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "80003ES2LAN SERDES SPT", MAC_80003es2lan | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x10bc, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB QUAD COPPER LOWPROFILE", MAC_82571 },
	{ 0x8086, 0x10c4, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IFE GT", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x10c5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ICH8 IFE G", MAC_ich8lan | (MEDIA_COPPER << MEDIA_SHIFT) },
	{ 0x8086, 0x10d5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571PT QUAD COPPER", MAC_82571 },
	{ 0x8086, 0x10d9, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB SERDES DUAL", MAC_82571 | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0x8086, 0x10da, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "82571EB SERDES QUAD", MAC_82571 | (MEDIA_SERDES << MEDIA_SHIFT) },
	{ 0 },
};

void IPG_WRITE_IO(IPG *ipg, unsigned reg, __u32 val)
{
	if (__unlikely(!ipg->io)) {
		IPG_WRITE(ipg, reg, val);
		return;
	}
	io_outl(ipg->io, reg);
	io_outl(ipg->io + 4, val);
}

__u16 ICH8_READ_16(IPG *ipg, unsigned reg)
{
	if (__unlikely(!ipg->flash_mem))
		KERNEL$SUICIDE("ICH8_READ_16: NO FLASH PRESENT, MAC TYPE %d", ipg->mac_type);
	return mmio_inw(ipg->flash_mem + reg);
}

__u32 ICH8_READ_32(IPG *ipg, unsigned reg)
{
	if (__unlikely(!ipg->flash_mem))
		KERNEL$SUICIDE("ICH8_READ_32: NO FLASH PRESENT, MAC TYPE %d", ipg->mac_type);
	return mmio_inl(ipg->flash_mem + reg);
}

void ICH8_WRITE_16(IPG *ipg, unsigned reg, __u16 val)
{
	if (__unlikely(!ipg->flash_mem))
		KERNEL$SUICIDE("ICH8_WRITE_16: NO FLASH PRESENT, MAC TYPE %d", ipg->mac_type);
	mmio_outw(ipg->flash_mem + reg, val);
}

void ICH8_WRITE_32(IPG *ipg, unsigned reg, __u32 val)
{
	if (__unlikely(!ipg->flash_mem))
		KERNEL$SUICIDE("ICH8_WRITE_32: NO FLASH PRESENT, MAC TYPE %d", ipg->mac_type);
	mmio_outl(ipg->flash_mem + reg, val);
}


void MSLEEP(int syn, unsigned val)
{
	if (__unlikely((unsigned)syn > 1)) KERNEL$SUICIDE("MSLEEP(%d, %u)", syn, val);
	if (syn) KERNEL$SLEEP(((__u64)val * JIFFIES_PER_SECOND + 999) / 1000);
	else KERNEL$UDELAY(val * 1000);
}

static void IPG_SET_MULTICAST_INTERNAL(IPG *ipg);

static void IPG_SET_MULTICAST(IPG *ipg)
{
	if (IPG_RX_RESET_WORKAROUND(ipg)) {
		IPG_ERROR_RESET(ipg);
	} else {
		IPG_SET_MULTICAST_INTERNAL(ipg);
	}
}

static void IPG_SET_MULTICAST_INTERNAL(IPG *ipg)
{
	int i, j;
	__u32 rctl;
	ipg->mcast_state = STATUS_MULTICAST_NONE;
	rctl = IPG_READ(ipg, IPG_RCTL);
	rctl &= ~(IPG_RCTL_UPE | IPG_RCTL_MPE | IPG_RCTL_MO);
	rctl |= IPG_RCTL_BAM | (MULTICAST_OFFSET << __BSF_CONST(IPG_RCTL_MO));
	if (__unlikely(!WQ_EMPTY(&ipg->mcast_table[MCAST_PROMISC]))) {
		rctl |= IPG_RCTL_UPE | IPG_RCTL_MPE;
		ipg->mcast_state = STATUS_MULTICAST_PROMISC;
		goto set_rctl;
	}
	if (__unlikely(!WQ_EMPTY(&ipg->mcast_table[MCAST_ALL]))) {
		rctl |= IPG_RCTL_MPE;
		ipg->mcast_state = STATUS_MULTICAST_ALL;
		goto set_rctl;
	}
	for (i = (ipg->mac_type < MAC_ich8lan ? IPG_MTA_BITS - 32 : IPG_MTA_BITS_ICH8 - 32); i >= 0; i -= 32) {
		__u32 reg = 0;
		for (j = 0; j < 32; j++) {
			reg |= !WQ_EMPTY(&ipg->mcast_table[i + j]) << j;
		}
		IPG_WRITE(ipg, IPG_MTA + i / 8, reg);
		if (__unlikely(reg)) ipg->mcast_state = STATUS_MULTICAST_SOME;
	}
	set_rctl:
	IPG_WRITE(ipg, IPG_RCTL, rctl);
}

void IPG_SET_ITR(IPG *ipg, int ips)
{
	__u32 val;
	if (!ips) val = 0;
	else {
		val = 1000000000 / (ips * 256);
		if (val > 65535) val = 65535;
	}
	IPG_WRITE(ipg, IPG_ITR, ips);
}

static void IPG_PCIEX_MASTER_DISABLE(IPG *ipg)
{
	unsigned i;
	__u32 ctrl = IPG_READ(ipg, IPG_CTRL);
	ctrl |= IPG_CTRL_GIO_MASTER_DISABLE;
	IPG_WRITE(ipg, IPG_CTRL, ctrl);
	for (i = 0; i < 800; i++) {
		if (!(IPG_READ(ipg, IPG_STATUS) & IPG_STATUS_GIO_MASTER_ENABLE)) return;
		KERNEL$UDELAY(100);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "MASTER REQUESTS STUCK");
}

static void IPG_CLEAR_COUNTERS(IPG *ipg)
{
	IPG_READ(ipg, IPG_STAT_CRCERRS);
	IPG_READ(ipg, IPG_STAT_SYMERRS);
	IPG_READ(ipg, IPG_STAT_MPC);
	IPG_READ(ipg, IPG_STAT_SCC);
	IPG_READ(ipg, IPG_STAT_ECOL);
	IPG_READ(ipg, IPG_STAT_MCC);
	IPG_READ(ipg, IPG_STAT_LATECOL);
	IPG_READ(ipg, IPG_STAT_COLC);
	IPG_READ(ipg, IPG_STAT_DC);
	IPG_READ(ipg, IPG_STAT_SEC);
	IPG_READ(ipg, IPG_STAT_RLEC);
	IPG_READ(ipg, IPG_STAT_XONRXC);
	IPG_READ(ipg, IPG_STAT_XONTXC);
	IPG_READ(ipg, IPG_STAT_XOFFRXC);
	IPG_READ(ipg, IPG_STAT_XOFFTXC);
	IPG_READ(ipg, IPG_STAT_FCRUC);
	IPG_READ(ipg, IPG_STAT_PTC1522);
	IPG_READ(ipg, IPG_STAT_GPRC);
	IPG_READ(ipg, IPG_STAT_BPRC);
	IPG_READ(ipg, IPG_STAT_MPRC);
	IPG_READ(ipg, IPG_STAT_GPTC);
	IPG_READ(ipg, IPG_STAT_GORCL);
	IPG_READ(ipg, IPG_STAT_GORCH);
	IPG_READ(ipg, IPG_STAT_GOTCL);
	IPG_READ(ipg, IPG_STAT_GOTCH);
	IPG_READ(ipg, IPG_STAT_RNBC);
	IPG_READ(ipg, IPG_STAT_RUC);
	IPG_READ(ipg, IPG_STAT_RFC);
	IPG_READ(ipg, IPG_STAT_ROC);
	IPG_READ(ipg, IPG_STAT_RJC);
	IPG_READ(ipg, IPG_STAT_TORL);
	IPG_READ(ipg, IPG_STAT_TORH);
	IPG_READ(ipg, IPG_STAT_TOTL);
	IPG_READ(ipg, IPG_STAT_TOTH);
	IPG_READ(ipg, IPG_STAT_TPR);
	IPG_READ(ipg, IPG_STAT_TPT);
	IPG_READ(ipg, IPG_STAT_MPTC);
	IPG_READ(ipg, IPG_STAT_BPTC);
	if (ipg->mac_type < MAC_ich8lan) {
		IPG_READ(ipg, IPG_STAT_PRC64);
		IPG_READ(ipg, IPG_STAT_PRC127);
		IPG_READ(ipg, IPG_STAT_PRC255);
		IPG_READ(ipg, IPG_STAT_PRC511);
		IPG_READ(ipg, IPG_STAT_PRC1023);
		IPG_READ(ipg, IPG_STAT_PRC1522);
		IPG_READ(ipg, IPG_STAT_PTC64);
		IPG_READ(ipg, IPG_STAT_PTC127);
		IPG_READ(ipg, IPG_STAT_PTC255);
		IPG_READ(ipg, IPG_STAT_PTC511);
		IPG_READ(ipg, IPG_STAT_PTC1023);
	}
	if (ipg->mac_type < MAC_82543) return;
	IPG_READ(ipg, IPG_STAT_ALGNERRC);
	IPG_READ(ipg, IPG_STAT_RXERRC);
	IPG_READ(ipg, IPG_STAT_TNCRS);
	IPG_READ(ipg, IPG_STAT_CEXTERR);
	IPG_READ(ipg, IPG_STAT_TSCTC);
	IPG_READ(ipg, IPG_STAT_TSCTFC);
	if (ipg->mac_type < MAC_82544) return;
	IPG_READ(ipg, IPG_STAT_MGTPRC);
	IPG_READ(ipg, IPG_STAT_MGTPDC);
	IPG_READ(ipg, IPG_STAT_MGTPTC);
	if (ipg->mac_type < MAC_82547_rev_2) return;
	IPG_READ(ipg, IPG_STAT_IAC);
	IPG_READ(ipg, IPG_STAT_ICRXOC);
	if (ipg->mac_type >= MAC_ich8lan) return;
	IPG_READ(ipg, IPG_STAT_ICRXPTC);
	IPG_READ(ipg, IPG_STAT_ICRXATC);
	IPG_READ(ipg, IPG_STAT_ICTXPTC);
	IPG_READ(ipg, IPG_STAT_ICTXATC);
	IPG_READ(ipg, IPG_STAT_ICTXQEC);
	IPG_READ(ipg, IPG_STAT_ICTXQMTC);
	IPG_READ(ipg, IPG_STAT_ICRXDMTC);
}

static int IPG_ARC_SUBSYSTEM_VALID(IPG *ipg)
{
	if (ipg->mac_type >= MAC_ich8lan) return 1;
	if (ipg->mac_type >= MAC_82571) {
		__u32 fwsm = IPG_READ(ipg, IPG_FWSM);
		if (fwsm & IPG_FWSM_MODE_MASK) return 1;
		return 0;
	}
	return 0;
}

static void IPG_MNG_PASS_THROUGH(IPG *ipg)
{
	if (ipg->mac_type >= MAC_82541) {
		__u32 manc = IPG_READ(ipg, IPG_MANC);
		if (!(manc & (IPG_MANC_RCV_TCO_EN | IPG_MANC_EN_MAC_ADDR_FILTER))) return;
		if (IPG_ARC_SUBSYSTEM_VALID(ipg)) {
			__u32 fwsm = IPG_READ(ipg, IPG_FWSM);
			__u32 factps = IPG_READ(ipg, IPG_FACTPS);
			if ((fwsm & IPG_FWSM_MODE_MASK) == IPG_FWSM_MODE_PT && !(factps & IPG_FACTPS_MNGCG))
				ipg->en_mng_pt = 1;
		} else {
			if (manc & IPG_MANC_SMBUS_EN && !(manc & IPG_MANC_ASF_EN))
				ipg->en_mng_pt = 1;
		}
	}
}

static void IPG_RESET_ADAPTIVE(IPG *ipg)
{
	ipg->current_ifs_val = 0;
	IPG_WRITE(ipg, IPG_AIFS, 0);
}

static void IPG_RESET(IPG *ipg, int syn)
{
	__u32 ctrl;
	__u32 pba;

	if (IPG_RX_RESET_WORKAROUND(ipg)) {
		ipg->pci_cmd = PCI$READ_CONFIG_WORD(ipg->id, PCI_COMMAND);
		if (ipg->pci_cmd & PCI_COMMAND_INVALIDATE) {
			PCI$WRITE_CONFIG_WORD(ipg->id, PCI_COMMAND, ipg->pci_cmd & ~PCI_COMMAND_INVALIDATE);
		}
	}

	/* e1000_reset */
	switch (ipg->mac_type) {
		case MAC_82547:
		case MAC_82547_rev_2:
			pba = IPG_PBA_30K;
			break;
		case MAC_82571:
		case MAC_82572:
		case MAC_80003es2lan:
			pba = IPG_PBA_38K;
			break;
		case MAC_82573:
			pba = IPG_PBA_12K;
			break;
		case MAC_ich8lan:
			pba = IPG_PBA_8K;
			break;
		default:
			pba = IPG_PBA_48K;
			break;
	}
	if (ipg->mtu > 8192 && ipg->mac_type != MAC_82573) {
		pba -= IPG_PBA_8K;
	}
	ipg->fc_high_water_mark = ((pba * 9216) / 10) & 0xFFF8;
	ipg->fc_low_water_mark = ipg->fc_high_water_mark - 8;
	IPG_WRITE(ipg, IPG_PBA, pba);

	/* e1000_reset_hw */
	if (ipg->bus_type == BUS_PCI_EX) {
		IPG_PCIEX_MASTER_DISABLE(ipg);
	}
	IPG_WRITE(ipg, IPG_IMC, ~0);
	IPG_WRITE(ipg, IPG_RCTL, 0);
	IPG_WRITE(ipg, IPG_TCTL, IPG_TCTL_PSP);
	IPG_FLUSH(ipg);
	IPG_READ(ipg, IPG_ICR);
	MSLEEP(syn, 10);
	ctrl = IPG_READ(ipg, IPG_CTRL);
	if (ipg->mac_type == MAC_82541 || ipg->mac_type == MAC_82547) {
		IPG_WRITE(ipg, IPG_CTRL, ctrl | IPG_CTRL_PHY_RST);
		MSLEEP(syn, 5);
	}
	if (ipg->mac_type == MAC_82573) {
		unsigned i;
		__u32 extcnf_ctrl = IPG_READ(ipg, IPG_EXTCNF_CTRL);
		extcnf_ctrl |= IPG_EXTCNF_CTRL_MDIO_SW_OWNERSHIP;
		for (i = 0; i < 10; i++) {
			IPG_WRITE(ipg, IPG_EXTCNF_CTRL, extcnf_ctrl);
			extcnf_ctrl = IPG_READ(ipg, IPG_EXTCNF_CTRL);
			if (extcnf_ctrl & IPG_EXTCNF_CTRL_MDIO_SW_OWNERSHIP) break;
			extcnf_ctrl |= IPG_EXTCNF_CTRL_MDIO_SW_OWNERSHIP;
			MSLEEP(syn, 2);
		}
	}
	if (ipg->mac_type >= MAC_ich8lan) {
		IPG_WRITE(ipg, IPG_PBA, IPG_PBA_8K);
		IPG_WRITE(ipg, IPG_PBS, IPG_PBA_16K);
	}
	if (ipg->mac_type == MAC_82544 ||
	    ipg->mac_type == MAC_82540 ||
	    ipg->mac_type == MAC_82545 ||
	    ipg->mac_type == MAC_82546 ||
	    ipg->mac_type == MAC_82541 ||
	    ipg->mac_type == MAC_82541_rev_2) {
		IPG_WRITE_IO(ipg, IPG_CTRL, ctrl | IPG_CTRL_RST);
	} else if (ipg->mac_type == MAC_82545_rev_3 ||
		   ipg->mac_type == MAC_82546_rev_3) {
		IPG_WRITE(ipg, IPG_CTRL_DUP, ctrl | IPG_CTRL_RST);
	} else if (ipg->mac_type >= MAC_ich8lan) {
		if (!PHY_RESET_DISABLE && !IPG_CHECK_PHY_RESET_BLOCK(ipg)) {
			ctrl |= IPG_CTRL_PHY_RST;
		}
		IPG_GET_SOFTWARE_FLAG(ipg);
		IPG_WRITE(ipg, IPG_CTRL, ctrl | IPG_CTRL_RST);
		MSLEEP(syn, 5);
	} else {
		IPG_WRITE(ipg, IPG_CTRL, ctrl | IPG_CTRL_RST);
	}
	/* reload eeprom */
	if (ipg->mac_type <= MAC_82544) {
		__u32 ctrl_ext;
		KERNEL$UDELAY(10);
		ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		ctrl_ext |= IPG_CTRL_EXT_EE_RST;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
		IPG_FLUSH(ipg);
		MSLEEP(syn, 2);
	} else if (ipg->mac_type >= MAC_82541 && ipg->mac_type <= MAC_82547_rev_2) {
		MSLEEP(syn, 20);
	} else if (ipg->mac_type >= MAC_82571) {
		unsigned i;
		if (ipg->mac_type == MAC_82573 && ipg->eeprom.type == EEPROM_FLASH) {
			__u32 ctrl_ext;
			KERNEL$UDELAY(10);
			ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
			ctrl_ext |= IPG_CTRL_EXT_EE_RST;
			IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
			IPG_FLUSH(ipg);
		}
		for (i = 0; i < 10; i++) {
			if (IPG_READ(ipg, IPG_EECD) & IPG_EECD_AUTO_RD) goto ee_loaded;
			MSLEEP(syn, 1);
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "EEPROM AUTOLOAD NOT COMPLETED");
		ee_loaded:
		if (ipg->mac_type == MAC_82573) {
			MSLEEP(syn, 25);
		}
	} else {
		MSLEEP(syn, 5);
	}
	if (ipg->mac_type >= MAC_82540 && ipg->mac_type <= MAC_82547_rev_2) {
		IPG_WRITE(ipg, IPG_MANC, IPG_READ(ipg, IPG_MANC) & ~IPG_MANC_ARP_EN);
	}
	IPG_PHY_DETECT(ipg);
	if (ipg->mac_type == MAC_82541 || ipg->mac_type == MAC_82547) {
		__u32 led_ctl;
		IPG_PHY_INIT_SCRIPT(ipg, syn);
		led_ctl = IPG_READ(ipg, IPG_LEDCTL);
		led_ctl &= ~IPG_LEDCTL_LED1_MOD;
		led_ctl |= 3 << __BSF_CONST(IPG_LEDCTL_LED1_MOD);
		led_ctl &= ~IPG_LEDCTL_LED3_MOD;
		led_ctl |= 7 << __BSF_CONST(IPG_LEDCTL_LED3_MOD);
		IPG_WRITE(ipg, IPG_LEDCTL, led_ctl);
	}
	IPG_WRITE(ipg, IPG_IMC, ~0);
	IPG_FLUSH(ipg);
	IPG_READ(ipg, IPG_ICR);
	if (ipg->mac_type >= MAC_ich8lan) {
		IPG_WRITE(ipg, IPG_KABGTXD, IPG_READ(ipg, IPG_KABGTXD) | IPG_KABGTXD_BGSQLBIAS);
	}

	/* e1000_reset continue */
	if (ipg->mac_type >= MAC_82544) IPG_WRITE(ipg, IPG_WUC, 0);
}

static void IPG_INIT_HW(IPG *ipg, int syn)
{
	/* e1000_init_hw */
	if (ipg->mac_type >= MAC_ich8lan) {
		if (ipg->rev_id < 3 || (ipg->dev_id != 0x1049 && ipg->dev_id != 0x104d))
			IPG_WRITE(ipg, IPG_STATUS, IPG_READ(ipg, IPG_STATUS) & ~0x80000000);
	}
	/* e1000_initialize_hardware_bits */
	if (ipg->mac_type >= MAC_82571) {
		__u32 reg_tarc0, reg_tarc1, reg_txdctl, reg_tctl, reg_ctrl, reg_ctrl_ext;
		reg_tarc0 = IPG_READ(ipg, IPG_TARC0);
		reg_tarc0 &= ~((1 << 30) | (1 << 29) | (1 << 28) | (1 << 27));

		reg_txdctl = IPG_READ(ipg, IPG_TXDCTL);
		reg_txdctl |= IPG_TXDCTL_COUNT_DESC;
		IPG_WRITE(ipg, IPG_TXDCTL, reg_txdctl);
		reg_txdctl = IPG_READ(ipg, IPG_TXDCTL1);
		reg_txdctl |= IPG_TXDCTL_COUNT_DESC;
		IPG_WRITE(ipg, IPG_TXDCTL1, reg_txdctl);
		switch (ipg->mac_type) {
			case MAC_82571:
			case MAC_82572:
				reg_tarc0 |= (1 << 26) | (1 << 25) | (1 << 24) | (1 << 23);
				reg_tarc1 = IPG_READ(ipg, IPG_TARC1);
				reg_tarc1 &= ~((1 << 30) | (1 << 29));
				reg_tarc1 |= (1 << 26) | (1 << 25) | (1 << 24);
				set_tarc1:
				reg_tctl = IPG_READ(ipg, IPG_TCTL);
				if (reg_tctl & IPG_TCTL_MULR) {
					reg_tarc1 &= ~(1 << 28);
				} else {
					reg_tarc1 |= 1 << 28;
				}
				IPG_WRITE(ipg, IPG_TARC1, reg_tarc1);
				break;
			case MAC_82573:
				reg_ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
				reg_ctrl_ext &= ~(1 << 23);
				reg_ctrl_ext |= 1 << 22;
				reg_ctrl = IPG_READ(ipg, IPG_CTRL);
				reg_ctrl &= ~(1 << 29);
				IPG_WRITE(ipg, IPG_CTRL, reg_ctrl);
				IPG_WRITE(ipg, IPG_CTRL_EXT, reg_ctrl_ext);
				break;
			case MAC_80003es2lan:
				if (ipg->media == MEDIA_FIBER || ipg->media == MEDIA_SERDES) reg_tarc0 &= ~(1 << 30);
				reg_tarc1 = IPG_READ(ipg, IPG_TARC1);
				goto set_tarc1;
			case MAC_ich8lan:
				if (ipg->rev_id < 3 || (ipg->dev_id != 0x1049 && ipg->dev_id != 0x104d)) reg_tarc0 |= (1 << 29) | (1 << 28);
				reg_ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
				reg_ctrl_ext |= 1 << 22;
				IPG_WRITE(ipg, IPG_CTRL_EXT, reg_ctrl_ext);
				reg_tarc0 |= (1 << 27) | (1 << 26) | (1 << 24) | (1 << 23);
				reg_tarc1 = IPG_READ(ipg, IPG_TARC1);
				reg_tarc1 |= (1 << 30) | (1 << 26) | (1 << 24);
				goto set_tarc1;
		}
		IPG_WRITE(ipg, IPG_TARC0, reg_tarc0);
	}
	/* e1000_init_hw continue */
	if (ipg->mac_type != MAC_82545_rev_3 && ipg->mac_type != MAC_82546_rev_3 && ipg->bus_type == BUS_PCIX) {
		__u16 pcix_cmd = PCI$READ_CONFIG_WORD(ipg->id, PCIX_COMMAND_REGISTER);
		__u16 pcix_stat_hi = PCI$READ_CONFIG_WORD(ipg->id, PCIX_STATUS_REGISTER_HI);
		__u16 cmd_mmrbc = (pcix_cmd & PCIX_COMMAND_MMRBC_MASK) >> __BSF_CONST(PCIX_COMMAND_MMRBC_MASK);
		__u16 stat_mmrbc = (pcix_stat_hi & PCIX_STATUS_HI_MMRBC_MASK) >> __BSF_CONST(PCIX_STATUS_HI_MMRBC_MASK);
		if (stat_mmrbc == 3) stat_mmrbc = 2;
		if (cmd_mmrbc > stat_mmrbc) {
			pcix_cmd &= ~PCIX_COMMAND_MMRBC_MASK;
			pcix_cmd |= stat_mmrbc << __BSF_CONST(PCIX_COMMAND_MMRBC_MASK);
			PCI$WRITE_CONFIG_WORD(ipg->id, PCIX_COMMAND_REGISTER, pcix_cmd);
		}
	}
	if (ipg->mac_type >= MAC_ich8lan) MSLEEP(syn, 15);

	IPG_SETUP_LINK(ipg, syn);

	if (ipg->mac_type >= MAC_82540) {
		__u32 txdctl = IPG_READ(ipg, IPG_TXDCTL);
		txdctl = (txdctl & ~IPG_TXDCTL_WTHRESH) | (1 << __BSF_CONST(IPG_RXDCTL_WTHRESH)) | IPG_RXDCTL_GRAN;
		IPG_WRITE(ipg, IPG_TXDCTL, txdctl);
	}
	if (ipg->mac_type >= MAC_82571 && ipg->mac_type != MAC_82573) {
		__u32 txdctl = IPG_READ(ipg, IPG_TXDCTL1);
		txdctl = (txdctl & ~IPG_TXDCTL_WTHRESH) | (1 << __BSF_CONST(IPG_RXDCTL_WTHRESH)) | IPG_RXDCTL_GRAN;
		IPG_WRITE(ipg, IPG_TXDCTL1, txdctl);
	}
	if (ipg->mac_type == MAC_80003es2lan) {
		__u32 tctl, tctl_ext, tipg, fflt;
		tctl = IPG_READ(ipg, IPG_TCTL);
		tctl |= IPG_TCTL_RTLC;
		IPG_WRITE(ipg, IPG_TCTL, tctl);
		tctl_ext = IPG_READ(ipg, IPG_TCTL_EXT);
		tctl_ext = (tctl_ext & ~IPG_TCTL_EXT_GCEX_MASK) | IPG_TCTL_EXT_GCEX_DEFAULT;
		IPG_WRITE(ipg, IPG_TCTL_EXT, tctl_ext);
		tipg = IPG_READ(ipg, IPG_TIPG);
		tipg = (tipg & ~IPG_TIPG_IPGT) | IPG_TIPG_IPGT_DEFAULT_80003ES2LAN_1000;
		IPG_WRITE(ipg, IPG_TIPG, tipg);
		fflt = IPG_READ(ipg, IPG_FFLT + 4);
		fflt &= ~0x00100000;
		IPG_WRITE(ipg, IPG_FFLT + 4, fflt);
	}
	if (ipg->mac_type == MAC_82573) {
		__u32 gcr = IPG_READ(ipg, IPG_GCR);
		gcr |= IPG_GCR_L1_ACT_WITHOUT_L0S_RX;
		IPG_WRITE(ipg, IPG_GCR, gcr);
	}
	IPG_CLEAR_COUNTERS(ipg);
	if (ipg->mac_type >= MAC_ich8lan && ipg->bus_type == BUS_PCI_EX) {
		/* e1000_set_pci_ex_no_snoop(PCI_EX_NO_SNOOP_ALL) */
		IPG_WRITE(ipg, IPG_GCR, IPG_GCR_RXD_NO_SNOOP | IPG_GCR_RXDSCW_NO_SNOOP | IPG_GCR_RXDSCR_NO_SNOOP | IPG_GCR_TXD_NO_SNOOP | IPG_GCR_TXDSCW_NO_SNOOP | IPG_GCR_TXDSCR_NO_SNOOP);
	}
	if (ipg->mac_type >= MAC_ich8lan || ipg->dev_id == 0x1099 || ipg->dev_id == 0x10b5) {
		__u32 ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		ctrl_ext |= IPG_CTRL_EXT_RO_DIS;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
	}
	/* e1000_reset continue */
	IPG_RESET_ADAPTIVE(ipg);
	if (ipg->mac_type == MAC_82571 || ipg->mac_type == MAC_82572) {
		__u16 phy_pwr;
		if (IPG_READ_PHY(ipg, IGP02E1000_PHY_POWER_MGMT, &phy_pwr)) goto skip_pwr;
		phy_pwr &= ~IGP02E1000_PHY_POWER_MGMT_SPD;
		IPG_WRITE_PHY(ipg, IGP02E1000_PHY_POWER_MGMT, phy_pwr);
		skip_pwr:;
	}
	/* e1000_get_hw_control */
	if (ipg->mac_type == MAC_82573) {
		__u32 swsm = IPG_READ(ipg, IPG_SWSM);
		IPG_WRITE(ipg, IPG_SWSM, swsm | IPG_SWSM_DRV_LOAD);
	} else if (ipg->mac_type >= MAC_82571) {
		__u32 ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext | IPG_CTRL_EXT_DRV_LOAD);
	}
}

static void IPG_OPEN(IPG *ipg)
{
	unsigned i;
	__u32 tipg, tctl, rctl;
	IPG_POWERUP_PHY(ipg);

	if (IPG_RX_RESET_WORKAROUND(ipg)) {
		rctl = IPG_READ(ipg, IPG_RCTL);
		rctl |= IPG_RCTL_RST;
		IPG_WRITE(ipg, IPG_RCTL, rctl);
		IPG_FLUSH(ipg);
		MSLEEP(0, 5);
	}

	for (i = 0; i < (ipg->mac_type < MAC_ich8lan ? IPG_RA_ENTRIES * 8 : IPG_RA_ENTRIES_ICH8 * 8); i += 8) {
		IPG_WRITE(ipg, IPG_RAH + i, 0);
		IPG_FLUSH(ipg);
		IPG_WRITE(ipg, IPG_RAL + i, 0);
		IPG_FLUSH(ipg);
	}
	IPG_WRITE(ipg, IPG_RAL, ipg->address[0] | (ipg->address[1] << 8) | (ipg->address[2] << 16) | (ipg->address[3] << 24));
	IPG_FLUSH(ipg);
	IPG_WRITE(ipg, IPG_RAH, ipg->address[4] | (ipg->address[5] << 8) | IPG_RAH_AV);
	IPG_FLUSH(ipg);
	if (IPG_RAR_WORKAROUND(ipg)) {
		IPG_WRITE(ipg, IPG_RAL + 8 * (IPG_RA_ENTRIES - 1), ipg->address[0] | (ipg->address[1] << 8) | (ipg->address[2] << 16) | (ipg->address[3] << 24));
		IPG_FLUSH(ipg);
		IPG_WRITE(ipg, IPG_RAH + 8 * (IPG_RA_ENTRIES - 1), ipg->address[4] | (ipg->address[5] << 8) | IPG_RAH_AV);
		IPG_FLUSH(ipg);
	}

	if (ipg->mac_type < MAC_ich8lan) {
		if (ipg->mac_type < MAC_82545_rev_3)
			IPG_WRITE(ipg, IPG_VET, 0);
		for (i = 0; i < 128 + 4; i += 4)
			IPG_WRITE(ipg, IPG_VFTA + i, 0);
		IPG_FLUSH(ipg);
	}

	if (IPG_RX_RESET_WORKAROUND(ipg)) {
		rctl = IPG_READ(ipg, IPG_RCTL);
		rctl &= ~IPG_RCTL_RST;
		IPG_WRITE(ipg, IPG_RCTL, rctl);
		IPG_FLUSH(ipg);
		MSLEEP(0, 5);
		if (ipg->pci_cmd & PCI_COMMAND_INVALIDATE)
			PCI$WRITE_CONFIG_WORD(ipg->id, PCI_COMMAND, ipg->pci_cmd);
	}

	/* e1000_configure */
	IPG_SET_MULTICAST_INTERNAL(ipg);

	/* e1000_init_manageability */
	if (ipg->en_mng_pt) {
		__u32 manc = IPG_READ(ipg, IPG_MANC);
		manc &= ~IPG_MANC_ARP_EN;
		if (ipg->mac_type >= MAC_82571) {
			__u32 manc2h = IPG_READ(ipg, IPG_MANC2H);
			manc |= IPG_MANC_EN_MNG2HOST;
			manc2h |= IPG_MANC2H_PORT_623 | IPG_MANC2H_PORT_624;
			IPG_WRITE(ipg, IPG_MANC2H, manc2h);
		}
		IPG_WRITE(ipg, IPG_MANC, manc);
	}
	/* e1000_configure_tx */
	IPG_WRITE(ipg, IPG_TDLEN, (ipg->sent_mask + 1) * sizeof(IPG_TX_DESC));
	IPG_WRITE(ipg, IPG_TDBAH, ipg->desc_dmaaddr >> 32);
	IPG_WRITE(ipg, IPG_TDBAL, ipg->desc_dmaaddr);
	IPG_WRITE(ipg, IPG_TDH, 0);
	IPG_WRITE(ipg, IPG_TDT, 0);
	IPG_WRITE(ipg, IPG_TDH + ipg->tdht_add, ipg->first_sent);
	IPG_WRITE(ipg, IPG_TDT + ipg->tdht_add, (ipg->first_sent + ipg->n_sent) & ipg->sent_mask);

	if (ipg->mac_type < MAC_82571 && (ipg->media == MEDIA_FIBER || ipg->media == MEDIA_SERDES))
		tipg = IPG_TIPG_IPGT_DEFAULT_FIBER;
	else
		tipg = IPG_TIPG_IPGT_DEFAULT_COPPER;
	if (ipg->mac_type < MAC_82543) {
		tipg = IPG_TIPG_IPGT_DEFAULT_82542;
		tipg |= IPG_TIPG_IPGR1_DEFAULT_82542 << __BSF_CONST(IPG_TIPG_IPGR1);
		tipg |= IPG_TIPG_IPGR2_DEFAULT_82542 << __BSF_CONST(IPG_TIPG_IPGR2);
	} else if (ipg->mac_type == MAC_80003es2lan) {
		tipg |= IPG_TIPG_IPGR1_DEFAULT_80003ES2LAN << __BSF_CONST(IPG_TIPG_IPGR1);
		tipg |= IPG_TIPG_IPGR2_DEFAULT_80003ES2LAN << __BSF_CONST(IPG_TIPG_IPGR2);
	} else {
		tipg |= IPG_TIPG_IPGR1_DEFAULT_82543 << __BSF_CONST(IPG_TIPG_IPGR1);
		tipg |= IPG_TIPG_IPGR2_DEFAULT_82543 << __BSF_CONST(IPG_TIPG_IPGR2);
	}
	IPG_WRITE(ipg, IPG_TIPG, tipg);
	IPG_WRITE(ipg, IPG_TIDV, TX_INT_DELAY);
	if (IPG_ABS_TIMERS(ipg))
		IPG_WRITE(ipg, IPG_TADV, TX_ABS_INT_DELAY);

	tctl = IPG_READ(ipg, IPG_TCTL);
	tctl &= ~IPG_TCTL_EN;
	IPG_WRITE(ipg, IPG_TCTL, tctl);
	tctl &= ~IPG_TCTL_CT;
	tctl |= IPG_TCTL_PSP | IPG_TCTL_RTLC | (IPG_TCTL_CT_DEFAULT << __BSF_CONST(IPG_TCTL_CT));

	if (ipg->mac_type == MAC_82571 || ipg->mac_type == MAC_82572) {
		__u32 tarc = IPG_READ(ipg, IPG_TARC0);
		tarc |= 1 << 21;
		IPG_WRITE(ipg, IPG_TARC0, tarc);
	}
	if (ipg->mac_type == MAC_80003es2lan) {
		__u32 tarc = IPG_READ(ipg, IPG_TARC0);
		tarc |= 1;
		IPG_WRITE(ipg, IPG_TARC0, tarc);
		tarc = IPG_READ(ipg, IPG_TARC1);
		tarc |= 1;
		IPG_WRITE(ipg, IPG_TARC1, tarc);
	}
	
	IPG_SET_COLLISION_DIST(ipg);

	IPG_WRITE(ipg, IPG_TCTL, tctl);

	/* e1000_setup_rctl */

	rctl = IPG_READ(ipg, IPG_RCTL);
	rctl &= ~IPG_RCTL_EN;
	IPG_WRITE(ipg, IPG_RCTL, rctl);

	rctl = IPG_READ(ipg, IPG_RCTL);
	rctl &= ~IPG_RCTL_LBM;
	rctl &= ~IPG_RCTL_RDMTS;
	rctl |= IPG_RCTL_RDMTS_HALF;
	if (ipg->mtu <= ETHERNET_MTU) rctl &= ~IPG_RCTL_LPE;
	else rctl |= IPG_RCTL_LPE;
	rctl &= ~(IPG_RCTL_BSIZE | IPG_RCTL_BSEX);
	if (IPG_TBI(ipg) || ipg->errorlevel)
		rctl |= IPG_RCTL_SBP;
	switch (ipg->rx_pkt_size) {
		case 256: rctl |= IPG_RCTL_BSIZE_256; break;
		case 512: rctl |= IPG_RCTL_BSIZE_512; break;
		case 1024: rctl |= IPG_RCTL_BSIZE_1024; break;
		case 2048: rctl |= IPG_RCTL_BSIZE_2048; break;
		case 4096: rctl |= IPG_RCTL_BSIZE_4096; break;
		case 8192: rctl |= IPG_RCTL_BSIZE_8192; break;
		case 16384: rctl |= IPG_RCTL_BSIZE_16384; break;
		default:
			KERNEL$SUICIDE("IPG_OPEN: INVALID RECEIVE PACKET SIZE %d", ipg->rx_pkt_size);
	}
	IPG_WRITE(ipg, IPG_RCTL, rctl);

	/* e1000_configure_rx */
	if (!IPG_ABS_TIMERS(ipg)) {
		IPG_WRITE(ipg, IPG_RDTR, 0);
	} else {
		IPG_WRITE(ipg, IPG_RDTR, RX_INT_DELAY);
		IPG_WRITE(ipg, IPG_RADV, RX_ABS_INT_DELAY);
	}
	IPG_SET_ITR(ipg, ipg->ips_100);

	if (ipg->mac_type >= MAC_82571) {
		__u32 ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		ctrl_ext |= IPG_CTRL_EXT_INT_TIMER_CLR;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
		IPG_FLUSH(ipg);
	}
	IPG_WRITE(ipg, IPG_RDLEN, (ipg->recv_mask + 1) * sizeof(IPG_RX_DESC));
	IPG_WRITE(ipg, IPG_RDBAH, ipg->desc_dmaaddr >> 32);
	IPG_WRITE(ipg, IPG_RDBAL, ipg->desc_dmaaddr + N_TX_DESCS * sizeof(IPG_TX_DESC));
	IPG_WRITE(ipg, IPG_RDH, 0);
	IPG_WRITE(ipg, IPG_RDT, 0);
	IPG_WRITE(ipg, IPG_RDH + ipg->tdht_add, ipg->first_recv);
	IPG_WRITE(ipg, IPG_RDT + ipg->tdht_add, (ipg->first_recv + ipg->n_recv) & ipg->recv_mask);

	if (IPG_RX_CHECKSUM(ipg)) {
		__u32 rxcsum = IPG_READ(ipg, IPG_RXCSUM);
		rxcsum |= IPG_RXCSUM_IPOFLD | IPG_RXCSUM_TUOFLD;
		rxcsum &= ~IPG_RXCSUM_PCSD;
		rxcsum &= ~IPG_RXCSUM_PCSS;
		rxcsum |= IPG_SUM_START << __BSF_CONST(IPG_RXCSUM_PCSS);
		IPG_WRITE(ipg, IPG_RXCSUM, rxcsum);
		IPG_FLUSH(ipg);
		rxcsum = IPG_READ(ipg, IPG_RXCSUM);
		if ((rxcsum & IPG_RXCSUM_PCSS) == (IPG_SUM_START << __BSF_CONST(IPG_RXCSUM_PCSS))) {
			ipg->rx_csum_general = 0;
		} else {
			ipg->rx_csum_general = 1;
		}
	}
	if (ipg->mac_type == MAC_82573)
		IPG_WRITE(ipg, IPG_ERT, 0x100);

	rctl = IPG_READ(ipg, IPG_RCTL);
	rctl |= IPG_RCTL_EN;
	IPG_WRITE(ipg, IPG_RCTL, rctl);
	IPG_FLUSH(ipg);
	tctl |= IPG_TCTL_EN;
	IPG_WRITE(ipg, IPG_TCTL, tctl);
	IPG_FLUSH(ipg);

	IPG_WRITE(ipg, IPG_IMS, INTERRUPT_MASK(ipg));
	IPG_FLUSH(ipg);

	IPG_WRITE(ipg, IPG_ICS, IPG_ICR_LSC);
	ipg->get_link_status = 1;

	/* !!! WARNING: can't modify media_timer here because it may be called
	from the timer with timer unset */

	IPG_LINK_CHANGED(ipg, 0);
}

void IPG_TIMEOUT(IPG *ipg)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, ipg->dev_name, "TRANSMIT TIMEOUT");
	IPG_ERROR_RESET(ipg);
}

void IPG_ERROR_RESET(IPG *ipg)
{
	IPG_RESET(ipg, 0);
	IPG_INIT_HW(ipg, 0);
	IPG_TEAR_DOWN_SENT_PACKETS(ipg);
	IPG_REINIT_RX_RING(ipg);
	IPG_OPEN(ipg);
	IPG_REAP_TX(ipg);
}

static void IPG_DISABLE(IPG *ipg)
{
	IPG_WRITE(ipg, IPG_IMC, ~0);
	IPG_FLUSH(ipg);
	IPG_READ(ipg, IPG_ICR);
	IPG_RESET(ipg, 0);
	IPG_POWERDOWN_PHY(ipg);
	/* e1000_release_manageability */
	if (ipg->en_mng_pt) {
		__u32 manc = IPG_READ(ipg, IPG_MANC);
		manc |= IPG_MANC_ARP_EN;
		if (ipg->mac_type >= MAC_82571)
			manc &= IPG_MANC_EN_MNG2HOST;
		IPG_WRITE(ipg, IPG_MANC, manc);
	}
	/* e1000_release_hw_control */
	if (ipg->mac_type == MAC_82573) {
		__u32 swsm = IPG_READ(ipg, IPG_SWSM);
		IPG_WRITE(ipg, IPG_SWSM, swsm & ~IPG_SWSM_DRV_LOAD);
	} else if (ipg->mac_type >= MAC_82571) {
		__u32 ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext & ~IPG_CTRL_EXT_DRV_LOAD);
	}
}

static void INIT_ROOT(HANDLE *h, void *data)
{
	IPG *ipg = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = ipg;
	h->op = &IPG_OPERATIONS;
}

static int UNLOAD(void *p, void **release, const char * const argv[]);

#ifndef IPG_HIGH_INTERRUPT
#define IRQ_PARAMS	IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &ipg->irq_ast
#else
#define IRQ_PARAMS	 IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_SHARED, IPG_IRQ_RT, ipg
#endif

int main(int argc, const char * const argv[])
{
	int r;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	unsigned long flags;
	int ips = -1, ips_100 = -1;
	char *net_input = NET_INPUT_DEVICE ":";
	char *address_str = NULL;
	int mtu = ETHERNET_MTU;
	int flow_control = FC_DEFAULT;
	int link_mode = LINK_AUTO;
	char *chip_name;
	int errorlevel = 0;
	const char *opt, *optend, *str;
	char dev_name[__MAX_STR_LEN];
	union {
		char gstr[__MAX_STR_LEN];
		OPENRQ openrq;
	} u;
	void *mem;
	int irq;
	IPG *ipg;
	VDESC v;
	VDMA64 dma;
	unsigned i;
	int res, bar;
	const char * const *arg = argv;
	int state = 0;
	static const struct __param_table params[18] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2,
		"IPS", __PARAM_INT, IPS_MIN, IPS_MAX,
		"IPS_100M", __PARAM_INT, IPS_MIN, IPS_MAX,
		"INPUT", __PARAM_STRING, 1, MAXINT,
		"ADDRESS", __PARAM_STRING, 1, MAXINT,
		"MTU", __PARAM_INT, ETHERNET_MTU, ETHERNET_JUMBO_MTU+1,
		"PAUSE_RECEIVE", __PARAM_BOOL, ~FC_TX_PAUSE, FC_RX_PAUSE,
		"NO_PAUSE_RECEIVE", __PARAM_BOOL, ~FC_TX_PAUSE, 0,
		"PAUSE_SEND", __PARAM_BOOL, ~FC_RX_PAUSE, FC_TX_PAUSE,
		"NO_PAUSE_SEND", __PARAM_BOOL, ~FC_RX_PAUSE, 0,
		"PAUSE_DEFAULT", __PARAM_BOOL, ~0, FC_DEFAULT,
		"10M", __PARAM_BOOL, ~LINK_FD, 0,
		"100M", __PARAM_BOOL, ~LINK_FD, LINK_100,
		"HALF_DUPLEX", __PARAM_BOOL, ~LINK_100, 0,
		"FULL_DUPLEX", __PARAM_BOOL, ~LINK_100, LINK_FD,
		"AUTO", __PARAM_BOOL, ~0, LINK_AUTO,
		NULL, 0, 0, 0,
	};
	void *vars[18];

	PKT_CHECK_VERSION;

	vars[0] = &errorlevel;
	vars[1] = &errorlevel;
	vars[2] = &ips;
	vars[3] = &ips_100;
	vars[4] = &net_input;
	vars[5] = &address_str;
	vars[6] = &mtu;
	vars[7] = &flow_control;
	vars[8] = &flow_control;
	vars[9] = &flow_control;
	vars[10] = &flow_control;
	vars[11] = &flow_control;
	vars[12] = &link_mode;
	vars[13] = &link_mode;
	vars[14] = &link_mode;
	vars[15] = &link_mode;
	vars[16] = &link_mode;
	vars[17] = NULL;
	
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "IPG: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (__PAGE_CLUSTER_SIZE > 65536) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "IPG: MAXIMUM PAGE SIZE IS 65536, ACTUAL PAGE SIZE IS %d", (unsigned)__PAGE_CLUSTER_SIZE);
		r = -EOPNOTSUPP;
		goto ret0;
	}
	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "IPG: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	_snprintf(dev_name, sizeof(dev_name), "PKT$IPG@" PCI_ID_FORMAT, id);
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	mem = PCI$MAP_MEM_RESOURCE(id, 0, IPG_REGSPACE);
	if (!mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
		r = __PTR_ERR(mem);
		goto ret1;
	}
	irq = PCI$READ_INTERRUPT_LINE(id);
	ipg = KERNEL$ALLOC_CONTIG_AREA(IPG_SIZE, AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA64 | AREA_ALIGN, (unsigned long)IPG_ALIGN);
	if (__IS_ERR(ipg)) {
		r = __PTR_ERR(ipg);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE DMA BUFFER: %s", dev_name, strerror(-r));
		goto ret2;
	}
	memset(ipg, 0, sizeof(IPG));

	strcpy(ipg->dev_name, dev_name);
	ipg->mem = mem;
	ipg->irq = irq;
	ipg->id = id;
	ipg->chip_name = chip_name;
	ipg->mtu = mtu;
	ipg->flow_control = flow_control;
	ipg->link_mode = link_mode;
	ipg->dev_id = PCI$READ_CONFIG_BYTE(id, PCI_DEVICE_ID);
	ipg->rev_id = PCI$READ_CONFIG_BYTE(id, PCI_REVISION_ID);
	ipg->mac_type = flags & MAC_MASK;

	memset(&ipg->link_state, 0, sizeof ipg->link_state);
	WQ_INIT(&ipg->link_state_wait, "IPG$LINK_STATE_WAIT");

	for (i = 0; i < N_MCAST; i++) WQ_INIT(&ipg->mcast_table[i], "IPG$MCAST_TABLE");
	ipg->errorlevel = errorlevel;
	ipg->ips = ips;
	ipg->ips_100 = ips_100;
	if (ipg->ips == -1) {
		ipg->ips = IPS_DEFAULT_1G;
		if (ipg->ips_100 == -1) ipg->ips_100 = IPS_DEFAULT_100M;
	}
	if (ipg->ips_100 == -1) ipg->ips_100 = ipg->ips;

	v.ptr = (unsigned long)TX_DESCS(ipg);
	v.len = N_TX_DESCS * sizeof(IPG_TX_DESC) + N_RX_DESCS * sizeof(IPG_RX_DESC);
	v.vspace = &KERNEL$VIRTUAL;
	dma.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	ipg->desc_unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&v, PF_READ | PF_WRITE, &dma);
	ipg->desc_dmaaddr = dma.ptr;

	ipg->rx_pkt_size = 256;
	while (ipg->mtu + ETHERNET_HEADER + 4 > ipg->rx_pkt_size) {
		ipg->rx_pkt_size <<= 1;
		if (ipg->rx_pkt_size > 16384) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID MTU", dev_name);
			r = -ERANGE;
			goto ret3;
		}
	}

	ipg->recv_mask = N_RX_DESCS - 1;
	ipg->sent_mask = N_TX_DESCS - 1;
	if (ipg->mac_type < MAC_82544) {
		if (ipg->recv_mask > 255) ipg->recv_mask = 255;
		if (ipg->sent_mask > 255) ipg->sent_mask = 255;
	}
	if (ipg->mac_type < MAC_82543) ipg->tdht_add = IPG_TDH1 - IPG_TDH;
	ipg->max_sent = DEFAULT_MAX_SENT;

	if (!(ipg->media = (flags >> MEDIA_SHIFT) & MEDIA_MASK)) {
		__u32 status = IPG_READ(ipg, IPG_STATUS);
		if (status & IPG_STATUS_TBIMODE) {
			ipg->media = MEDIA_FIBER;
		} else {
			ipg->media = MEDIA_COPPER;
		}
	}
	if (ipg->mac_type >= MAC_ich8lan) {
		ipg->flash_mem = PCI$MAP_MEM_RESOURCE(id, 1, ICH8_FLASH_REGSPACE);
		if (!ipg->flash_mem) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO FLASH MEM RESOURCE");
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO FLASH MEM RESOURCE", dev_name);
			r = -ENXIO;
			goto ret3;
		}
		if (__IS_ERR(ipg->flash_mem)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP FLASH MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(ipg->flash_mem)));
			r = __PTR_ERR(ipg->flash_mem);
			ipg->flash_mem = NULL;
			goto ret3;
		}
	}
	for (res = -1; (bar = PCI$NEXT_RESOURCE(id, &res)) >= 0; ) {
		if ((bar & PCI_BASE_ADDRESS_SPACE) == PCI_BASE_ADDRESS_SPACE_IO) {
			ipg->io = PCI$READ_IO_RESOURCE(id, res);
			if (ipg->io) {
				ipg->range.start = ipg->io;
				ipg->range.len = IPG_IO_REGSPACE;
				ipg->range.name = ipg->dev_name;
				if (KERNEL$REGISTER_IO_RANGE(&ipg->range)) ipg->io = 0;
				break;
			}
		}
	}

	if (ipg->mac_type <= MAC_82542_rev2_1) {
		ipg->bus_type = BUS_PCI;
	} else if (ipg->mac_type >= MAC_82571) {
		ipg->bus_type = BUS_PCI_EX;
	} else {
		if (IPG_READ(ipg, IPG_STATUS) & IPG_STATUS_PCIX_MODE) ipg->bus_type = BUS_PCIX;
		else ipg->bus_type = BUS_PCI;
	}

	IPG_MNG_PASS_THROUGH(ipg);

	if ((r = IPG_INIT_EEPROM(ipg))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INIT EEPROM", dev_name);
		goto ret3;
	}

	RAISE_SPL(SPL_IPG);
	IPG_RESET(ipg, 1);
	IPG_INIT_HW(ipg, 1);
	LOWER_SPL(SPL_IPG);

	if (ipg->media == MEDIA_COPPER && !ipg->phy.type) {
		r = -EIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INIT PHY", dev_name);
		goto ret4;
	}

	/* FC default, e1000_setup_link */
	if (ipg->flow_control == FC_DEFAULT) {
		__u16 c2;
		if (ipg->mac_type >= MAC_ich8lan || ipg->mac_type == MAC_82573) {
			ipg->flow_control = FC_FULL;
		} else if (IPG_READ_EEPROM(ipg, EEPROM_INIT_CONTROL2_REG, 1, &c2)) {
			ipg->flow_control = FC_NONE;
		} else if (!(c2 & EEPROM_INIT_CONTROL2_REG_PAUSE_MASK)) {
			ipg->flow_control = FC_NONE;
		} else if ((c2 & EEPROM_INIT_CONTROL2_REG_PAUSE_MASK) == EEPROM_INIT_CONTROL2_REG_ASM_DIR) {
			ipg->flow_control = FC_TX_PAUSE;
		} else {
			ipg->flow_control = FC_FULL;
		}
	}

	if (!address_str) {
		for (i = 0; i < ETHERNET_ADDRLEN / 2; i++) {
			__u16 data;
			if ((r = IPG_READ_EEPROM(ipg, i, 1, &data))) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT READ MAC ADDRESS", dev_name);
				goto ret4;
			}
			ipg->address[i * 2] = data;
			ipg->address[i * 2 + 1] = data >> 8;
		}
		if (ipg->mac_type == MAC_82546 ||
		    ipg->mac_type == MAC_82546_rev_3 ||
		    ipg->mac_type == MAC_82571 ||
		    ipg->mac_type == MAC_80003es2lan) {
			if (IPG_READ(ipg, IPG_STATUS) & IPG_STATUS_FUNCTION_1) ipg->address[5] ^= 1;
		}
		if (!NET$VALID_ETHERNET_ADDRESS(ipg->address)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INVALID ETHERNET ADDRESS: %s", NET$PRINT_ETHERNET_ADDRESS(u.gstr, ipg->address));
			r = -EIO;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID ETHERNET ADDRESS", dev_name);
			goto ret4;
		}
	} else {
		if (NET$PARSE_ETHERNET_ADDRESS(address_str, ipg->address) || !NET$VALID_ETHERNET_ADDRESS(ipg->address)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID ADDRESS: %s", dev_name, address_str);
			r = -EBADSYN;
			goto ret4;
		}
	}
	_printf("%s: %s ON %s: %s\n", dev_name, chip_name, ipg->bus_type == BUS_PCIX ? "PCI-X" : ipg->bus_type == BUS_PCI_EX ? "PCI-EX" : "PCI", PCI$ID(u.gstr, id));
	_printf("%s: MEM %"__64_format"X", dev_name, PCI$READ_MEM_RESOURCE(id, 0, NULL));
	if (ipg->mac_type >= MAC_ich8lan)
		_printf(", MEM2 %"__64_format"X", PCI$READ_MEM_RESOURCE(id, 1, NULL));
	if (ipg->io)
		_printf(", IO "IO_FORMAT, ipg->io);
	_printf(", IRQ %d\n", irq);
	_printf("%s: ADDRESS %s%s\n", dev_name, NET$PRINT_ETHERNET_ADDRESS(u.gstr, ipg->address), address_str ? " (OVERRIDEN)" : "");

	if ((r = NETQUE$ALLOC_QUEUE(&ipg->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", dev_name, strerror(-r));
		goto ret4;
	}

	r = KERNEL$RESERVE_BOUNCE(dev_name, ipg->recv_mask, ipg->rx_pkt_size, BOUNCE_DMA64);
	if (r) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE RECEIVE BOUNCE BUFFERS: %s", dev_name, strerror(-r));
		goto ret41;
	}
	r = KERNEL$RESERVE_BOUNCE(dev_name, ipg->sent_mask, ipg->mtu + sizeof(struct eth_header), BOUNCE_DMA64);
	if (r) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE RECEIVE BOUNCE BUFFERS: %s", dev_name, strerror(-r));
		goto ret42;
	}

	for (i = 0; i < ipg->recv_mask; i++) {
		PACKET *p;
		RAISE_SPL(SPL_IPG);
		alloc_again:
		ALLOC_PACKET(p, ipg->rx_pkt_size - ETHERNET_HEADER, NULL, SPL_IPG, {
			if (__unlikely(r = NET$MEMWAIT_SYNC())) {
				if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE PACKETS", dev_name);
				goto ret5;
			}
			goto alloc_again;
		});
		IPG_ADD_RECV_PACKET(ipg, p);
		LOWER_SPL(SPL_ZERO);
	}

	u.openrq.flags = O_WRONLY;
	u.openrq.path = net_input;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", dev_name, net_input, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto ret5;
	}
	ipg->packet_input = u.openrq.status;
	ipg->outstanding = 0;

	RAISE_SPL(SPL_IPG);
	ipg->irq_ast.fn = IPG_IRQ;
#ifdef IPG_HIGH_INTERRUPT
	ipg->icr = 0;
#endif
	if ((r = KERNEL$REQUEST_IRQ(irq, &ipg->irq_ctrl, IRQ_PARAMS, dev_name)) < 0) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret6;
	}

	IPG_OPEN(ipg);

	ipg->media_timer.fn = ipg->media == MEDIA_COPPER ? IPG_COPPER_MEDIA_TIMER : IPG_FIBER_MEDIA_TIMER;
	INIT_TIMER(&ipg->media_timer);
	KERNEL$SET_TIMER(0, &ipg->media_timer);

	LOWER_SPL(SPL_ZERO);

	r = KERNEL$REGISTER_DEVICE(dev_name, "IPG.SYS", 0, ipg, INIT_ROOT, NULL, NULL, NULL, UNLOAD, &ipg->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret7;
	}

	ipg->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);

	return 0;

	ret7:
	KERNEL$MASK_IRQ(ipg->irq_ctrl);
	RAISE_SPL(SPL_IPG);
	KERNEL$DEL_TIMER(&ipg->media_timer);
	KERNEL$RELEASE_IRQ(ipg->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_PARAMS);
	IPG_WRITE(ipg, IPG_IMC, ~0);
	IPG_FLUSH(ipg);
	IPG_READ(ipg, IPG_ICR);
	LOWER_SPL(SPL_ZERO);
	ret6:
	KERNEL$FAST_CLOSE(ipg->packet_input);
	ret5:
	KERNEL$UNRESERVE_BOUNCE(ipg->dev_name, ipg->sent_mask, ipg->mtu + sizeof(struct eth_header), BOUNCE_DMA64);
	ret42:
	KERNEL$UNRESERVE_BOUNCE(ipg->dev_name, ipg->recv_mask, ipg->rx_pkt_size, BOUNCE_DMA64);
	ret41:
	NETQUE$FREE_QUEUE(ipg->queue);
	ret4:
	RAISE_SPL(SPL_IPG);
	IPG_DISABLE(ipg);
	ret3:
	RAISE_SPL(SPL_IPG);
	IPG_TEAR_DOWN_RECV_PACKETS(ipg);
	IPG_TEAR_DOWN_SENT_PACKETS(ipg);
	LOWER_SPL(SPL_ZERO);
	ipg->desc_unlock(ipg->desc_dmaaddr);
	if (ipg->flash_mem) PCI$UNMAP_MEM_RESOURCE(id, ipg->flash_mem, ICH8_FLASH_REGSPACE);
	if (ipg->io) KERNEL$UNREGISTER_IO_RANGE(&ipg->range);
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&ipg->mcast_table[i]);
	WQ_WAKE_ALL(&ipg->link_state_wait);
	KERNEL$FREE_CONTIG_AREA(ipg, IPG_SIZE);
	ret2:
	PCI$UNMAP_MEM_RESOURCE(id, mem, IPG_REGSPACE);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int UNLOAD(void *p, void **release, const char * const argv[])
{
	unsigned i;
	int r;
	IPG *ipg = p;
	if ((r = KERNEL$DEVICE_UNLOAD(ipg->lnte, argv))) return r;
	KERNEL$MASK_IRQ(ipg->irq_ctrl);
	RAISE_SPL(SPL_IPG);
	KERNEL$DEL_TIMER(&ipg->media_timer);
	KERNEL$RELEASE_IRQ(ipg->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_PARAMS);
	IPG_DISABLE(ipg);
	LOWER_SPL(SPL_ZERO);
	KERNEL$FAST_CLOSE(ipg->packet_input);
	KERNEL$UNRESERVE_BOUNCE(ipg->dev_name, ipg->sent_mask, ipg->mtu + sizeof(struct eth_header), BOUNCE_DMA64);
	KERNEL$UNRESERVE_BOUNCE(ipg->dev_name, ipg->recv_mask, ipg->rx_pkt_size, BOUNCE_DMA64);
	NETQUE$FREE_QUEUE(ipg->queue);
	RAISE_SPL(SPL_IPG);
	IPG_TEAR_DOWN_RECV_PACKETS(ipg);
	IPG_TEAR_DOWN_SENT_PACKETS(ipg);
	LOWER_SPL(SPL_ZERO);
	ipg->desc_unlock(ipg->desc_dmaaddr);
	if (ipg->flash_mem) PCI$UNMAP_MEM_RESOURCE(ipg->id, ipg->flash_mem, ICH8_FLASH_REGSPACE);
	if (ipg->io) KERNEL$UNREGISTER_IO_RANGE(&ipg->range);
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&ipg->mcast_table[i]);
	WQ_WAKE_ALL(&ipg->link_state_wait);
	PCI$UNMAP_MEM_RESOURCE(ipg->id, ipg->mem, IPG_REGSPACE);
	PCI$FREE_DEVICE(ipg->id);
	*release = ipg->dlrq;
	KERNEL$FREE_CONTIG_AREA(ipg, IPG_SIZE);
	return 0;
}

static unsigned IPG_HASH_MULTICAST(IPG *ipg, __u8 addr[ETHERNET_ADDRLEN])
{
	unsigned val;
	switch (MULTICAST_OFFSET) {
	case 0:
		if (ipg->mac_type >= MAC_ich8lan)
			val = ((addr[4] >> 6) | (addr[5] << 2));
		else
			val = ((addr[4] >> 4) | (addr[5] << 4));
		break;
	case 1:
		if (ipg->mac_type >= MAC_ich8lan)
			val = ((addr[4] >> 5) | (addr[5] << 3));
		else
			val = ((addr[4] >> 3) | (addr[5] << 5));
		break;
	case 2:
		if (ipg->mac_type >= MAC_ich8lan)
			val = ((addr[4] >> 4) | (addr[5] << 4));
		else
			val = ((addr[4] >> 2) | (addr[5] << 6));
		break;
	case 3:
		if (ipg->mac_type >= MAC_ich8lan)
			val = ((addr[4] >> 2) | (addr[5] << 6));
		else
			val = ((addr[4] >> 0) | (addr[5] << 8));
		break;
	}
	if (ipg->mac_type >= MAC_ich8lan)
		val &= 0x3ff;
	else
		val |= 0x3ff;

	return val;
}

DECL_IOCALL(IPG_IOCTL, SPL_IPG, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	IPG *ipg;
	int r;
	if (__unlikely(h->op != &IPG_OPERATIONS))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_IPG));
	ipg = h->fnode;
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			ETHERNET_FILL_IF_TYPE(&t);
			t.id = IPG_ID(ipg);
			memcpy(t.addr, ipg->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (!memcmp(&ipg->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&ipg->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &ipg->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			ipg->link_state.seq++;
			WQ_WAKE_ALL_PL(&ipg->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_MULTICAST: {
			if (__unlikely(RQ->v.len != 0)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			RQ->status = ipg->mcast_state;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SET_MULTICAST: {
			__u8 mcast[ETHERNET_ADDRLEN];
			unsigned bit;
			int need_refresh;
			if (__unlikely(RQ->param == PARAM_MULTICAST_PROMISC)) {
				bit = MCAST_PROMISC;
				goto have_bit;
			} else if (__unlikely(RQ->param == PARAM_MULTICAST_ALL)) {
				bit = MCAST_ALL;
				goto have_bit;
			}
			if (__unlikely(RQ->param != PARAM_MULTICAST_ADDRESS)) {
				goto enoop;
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, mcast, ETHERNET_ADDRLEN)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			have_bit:
			bit = IPG_HASH_MULTICAST(ipg, mcast);
			need_refresh = WQ_EMPTY(&ipg->mcast_table[bit]);
			WQ_WAIT_F(&ipg->mcast_table[bit], RQ);
			if (need_refresh) IPG_SET_MULTICAST(ipg);
			RETURN;
		}
		case IOCTL_IF_RESET_MULTICAST: {
			IPG_SET_MULTICAST(ipg);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		default: {
			enoop:
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

