#include <SPAD/SYSLOG.H>

#include "IPGREG.H"
#include "IPG.H"

static IO_STUB IPG_PACKET;

const HANDLE_OPERATIONS IPG_OPERATIONS = {
	SPL_X(SPL_IPG),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	IPG_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	IPG_PACKET,		/* pktio */
};

void IPG_ADD_RECV_PACKET(IPG *ipg, PACKET *p)
{
	VDMA64 vdma;
	VDESC desc;
	unsigned idx;
	IPG_RX_DESC *rx;
	p->addrlen = ETHERNET_ADDRLEN;
	desc.ptr = (unsigned long)p->data + LL_HEADER - sizeof(struct eth_header);
	desc.len = ipg->rx_pkt_size;
	desc.vspace = &KERNEL$VIRTUAL;
	vdma.spl = SPL_X(SPL_IPG);
	RAISE_SPL(SPL_VSPACE);
	p->sender_data = KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_WRITE, &vdma);
	idx = (ipg->first_recv + ipg->n_recv++) & ipg->recv_mask;
	ipg->recv_packets[idx] = p;
	rx = &RX_DESCS(ipg)[idx];
	rx->addr = __64CPU2LE(p->v.ptr = vdma.ptr);
	rx->status = 0;
}

static __finline__ void IPG_ACK_RECV_PACKET(IPG *ipg)
{
	__barrier();
	IPG_WRITE(ipg, IPG_RDT + ipg->tdht_add, (ipg->first_recv + ipg->n_recv) & ipg->sent_mask);
}

static __finline__ void IPG_UNMAP_RECV_PACKET(IPG *ipg, PACKET *p)
{
	((vspace_dma64unlock_t *)p->sender_data)(p->v.ptr);
}

static __finline__ void IPG_UNMAP_TX_DESC(IPG *ipg, unsigned idx)
{
	ipg->xmit_info[idx].unlock(__64LE2CPU(TX_DESCS(ipg)[idx].legacy.addr));
}

static void IPG_SEND(IPG *ipg, PACKET *p)
{
	IPG_TX_DESC *tx;
	VDESC desc;
	VDMA64 vdma;
	unsigned pos;
	desc.ptr = (unsigned long)p->data + LL_HEADER - sizeof(struct eth_header);
	desc.len = p->data_length + sizeof(struct eth_header);
	desc.vspace = &KERNEL$VIRTUAL;
	pos = (ipg->first_sent + ipg->n_sent) & ipg->sent_mask;
	vdma.spl = SPL_X(SPL_IPG);
	RAISE_SPL(SPL_VSPACE);
	ipg->xmit_info[pos].unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_READ, &vdma);
	set_tx:
	tx = &TX_DESCS(ipg)[pos];
	tx->legacy.addr = __64CPU2LE(vdma.ptr);
	tx->legacy.len = __16CPU2LE(vdma.len);
	tx->legacy.cmd = 0;
	ipg->xmit_info[pos].p = NULL;
	while (__unlikely(vdma.len > IPG_MAX_TX_DESC_SIZE)) {
		vspace_dma64unlock_t *unlock;
		tx->legacy.len = __16CPU2LE(IPG_MAX_TX_DESC_SIZE);
		vdma.len -= IPG_MAX_TX_DESC_SIZE;
		vdma.ptr += IPG_MAX_TX_DESC_SIZE;
		unlock = ipg->xmit_info[pos].unlock;
		ipg->xmit_info[pos].unlock = KERNEL$NULL_VSPACE_DMA64UNLOCK;
		pos = (pos + 1) & ipg->sent_mask;
		ipg->xmit_info[pos].unlock = unlock;
		goto set_tx;
	}
	if (__unlikely(p->v.len != 0)) {
		pos = (pos + 1) & ipg->sent_mask;
		RAISE_SPL(SPL_VSPACE);
		ipg->xmit_info[pos].unlock = p->v.vspace->op->vspace_dma64lock(&p->v, PF_READ, &vdma);
		if (__unlikely(!ipg->xmit_info[pos].unlock)) {
			p->flags |= PKT_PAGING_ERROR;
			p->status = -EVSPACEFAULT;
			while (pos != ((ipg->first_sent + ipg->n_sent) & ipg->sent_mask)) {
				pos = (pos - 1) & ipg->sent_mask;
				IPG_UNMAP_TX_DESC(ipg, pos);
			}
			CALL_PKT(p);
			return;
		}
		p->v.ptr += vdma.len;
		p->v.len -= vdma.len;
		goto set_tx;
	}
	/* We could theoretically drop CMD_RS here to avoid PCI bus writes
	   and cacheline invalidation ...
	   on 82545 it works, on 82571 it doesn't */
	tx->legacy.cmd = IPG_TX_DESC_LEGACY_CMD_EOP | IPG_TX_DESC_LEGACY_CMD_IFCS | IPG_TX_DESC_LEGACY_CMD_RS | IPG_TX_DESC_LEGACY_CMD_IDE;
	if (__unlikely(!IPG_ABS_TIMERS(ipg))) {
		if (__unlikely(ipg->mac_type < MAC_82543)) {
			tx->legacy.cmd |= IPG_TX_DESC_LEGACY_CMD_RPS;
			tx->legacy.cmd &= ~IPG_TX_DESC_LEGACY_CMD_RS;
		}
		tx->legacy.cmd &= ~IPG_TX_DESC_LEGACY_CMD_IDE;
	}
	ipg->xmit_info[pos].p = p;
	if (__likely(p->flags & PKT_OUTPUT_CHECKSUM)) {
		tx->legacy.cmd |= IPG_TX_DESC_LEGACY_CMD_IC;
		tx->legacy.css = CHECKSUM_FROM(p) + sizeof(struct eth_header);
		tx->legacy.cso = CHECKSUM_POS(p) + sizeof(struct eth_header);
	}
	ipg->n_sent = (pos + 1 - ipg->first_sent) & ipg->sent_mask;
	__barrier();
	IPG_WRITE(ipg, IPG_TDT + ipg->tdht_add, (ipg->first_sent + ipg->n_sent) & ipg->sent_mask);
	ipg->tx_last_time = KERNEL$GET_JIFFIES_LO();
}

static DECL_IOCALL(IPG_PACKET, SPL_IPG, PACKET)
{
	HANDLE *h = RQ->handle;
	IPG *ipg;
	__CHECK_PKT(RQ, "IPG_PACKET");
	if (__unlikely(h->op != &IPG_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_IPG));
	ipg = h->fnode;
	if (__unlikely(RQ->data_length + RQ->v.len > ipg->mtu)) {
		RQ->status = -EMSGSIZE;
		RETURN_AST(RQ);
	}
	RQ->status = 0;
	if (ipg->n_sent < ipg->max_sent) IPG_SEND(ipg, RQ);
	else NETQUE$ENQUEUE_PACKET(ipg->queue, RQ);
	RETURN;
}

void IPG_REAP_TX(IPG *ipg)
{
	PACKET *p;
	unsigned pos = IPG_READ(ipg, IPG_TDH + ipg->tdht_add);
	if (__unlikely(pos > ipg->sent_mask)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, ipg->dev_name, "TX HEAD OUT OF RANGE: %X, MAX %X", pos, ipg->sent_mask);
		IPG_ERROR_RESET(ipg);
		return;
	}
	if (__unlikely(ipg->first_sent != pos)) {
		do {
			if (__unlikely(!ipg->n_sent)) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, ipg->dev_name, "TX HEAD OUT OF USED AREA: %X, HEAD %X", pos, ipg->first_sent);
				IPG_ERROR_RESET(ipg);
				return;
			}
			ipg->n_sent--;
			IPG_UNMAP_TX_DESC(ipg, ipg->first_sent);
			if (__likely((p = ipg->xmit_info[ipg->first_sent].p) != NULL)) {
				p->status = 0;
				CALL_PKT(p);
			}
			ipg->first_sent = (ipg->first_sent + 1) & ipg->sent_mask;
		} while (ipg->first_sent != pos);
		ipg->tx_last_time = KERNEL$GET_JIFFIES_LO();
	}
	if (__unlikely(NETQUE$QUEUE_EMPTY(ipg->queue)))
		return;
	if (__unlikely(!ipg->n_sent)) {
		if (__likely(ipg->max_sent < ipg->sent_mask - SEND_DESCS)) {
			ipg->max_sent++;
			/*__debug_printf("max sent adjust: %d\n", ipg->max_sent);*/
		}
	}
	while (ipg->n_sent < ipg->max_sent) {
		p = NETQUE$DEQUEUE(ipg->queue);
		if (__unlikely(!p)) break;
		IPG_SEND(ipg, p);
	}
}

static DECL_AST(IPG_PACKET_RETURNED, SPL_IPG, PACKET)
{
	IPG *ipg;
	PKT_AST_ENTER(RQ);
	ipg = RQ->sender_data;
	ipg->outstanding--;
	if (__unlikely(ipg->n_recv < ipg->recv_mask)) {
		RQ->flags &= PKT_REUSE_FLAGS;
		IPG_ADD_RECV_PACKET(ipg, RQ);
		IPG_ACK_RECV_PACKET(ipg);
	} else {
		FREE_PACKET(RQ, NULL, SPL_IPG);
	}
	RETURN;
}

static int IPG_TEST_TBI(IPG *ipg, PACKET *p, IPG_RX_DESC *rx);

static void IPG_PROCESS_RX(IPG *ipg)
{
	char split_packet = 0;
	PACKET *p;
	__barrier();
	while (__likely(ipg->n_recv)) {
		IPG_RX_DESC *rx = &RX_DESCS(ipg)[ipg->first_recv];
		if (__unlikely(!(rx->status & IPG_RX_DESC_STATUS_DD)))
			break;
		p = ipg->recv_packets[ipg->first_recv];
		ipg->first_recv = (ipg->first_recv + 1) & ipg->recv_mask;
		ipg->n_recv--;
		IPG_UNMAP_RECV_PACKET(ipg, p);
		PKT_PREFETCH_1(p);
		if (ipg->rx_csum_general) {
			p->checksum.u = MKCHECKSUM(IPG_SUM_START, __16LE2CPU(rx->csum));
			p->flags |= PKT_INPUT_CHECKSUM;
		}
		if (__likely(!(rx->status & IPG_RX_DESC_STATUS_IXSM))) {
			if (__likely(rx->status & IPG_RX_DESC_STATUS_IPCS) && __likely(!(rx->errors & IPG_RX_DESC_ERRORS_IPE))) p->flags |= PKT_IPHDR_CHECKSUM_OK;
			if (__likely(rx->status & (IPG_RX_DESC_STATUS_TCPCS | IPG_RX_DESC_STATUS_UDPCS)) && __likely(!(rx->errors & IPG_RX_DESC_ERRORS_TCPE))) p->flags |= PKT_TCPUDP_CHECKSUM_OK;
		}
		p->fn = IPG_PACKET_RETURNED;
		p->sender_data = ipg;
		p->h = ipg->packet_input;
		p->id = IPG_ID(ipg);
		ipg->outstanding++;
		p->data_length = __16LE2CPU(rx->len) - sizeof(struct eth_header) - 4;
		if (__unlikely(!(rx->status & IPG_RX_DESC_STATUS_EOP))) {
			split_packet = 1;
			pkt_err:
			p->status = -EIO;
			CALL_PKT(p);
			continue;
		}
		if (__unlikely(split_packet)) {
			split_packet = 0;
			goto pkt_err;
		}
		if (__unlikely(rx->errors & IPG_RX_DESC_ERRORS_MASK)) {
			if (__unlikely(IPG_TEST_TBI(ipg, p, rx))) goto pkt_err;
		}
		if (__unlikely(p->data_length > ipg->mtu)) {
			goto pkt_err;
		}
		PKT_PREFETCH_2(p);
		CALL_IORQ(p, KERNEL$PKTIO);
	}
	while (ipg->n_recv < ipg->recv_mask) {
		ALLOC_PACKET(p, ipg->rx_pkt_size - ETHERNET_HEADER, NULL, SPL_IPG, {
			goto no_mem;
		});
		IPG_ADD_RECV_PACKET(ipg, p);
	}
	no_mem:;
	IPG_ACK_RECV_PACKET(ipg);
}

#ifdef IPG_HIGH_INTERRUPT

DECL_RT_IRQ_HANDLER(IPG_IRQ_RT)
{
	IPG *ipg = DATA;
	__u32 icr = IPG_READ(ipg, IPG_ICR);
	if (__unlikely(!icr)) {
		goto i_ret;
	}
	if (__unlikely(ipg->icr)) {
		__OR32(&ipg->icr, icr);
		i_ret:
		IRQ_RETURN;
	}
	ipg->icr = icr;
	IRQ_POST_AST(&ipg->irq_ast);
}

#endif

DECL_IRQ_AST(IPG_IRQ, SPL_IPG, AST)
{
	IPG *ipg = GET_STRUCT(RQ, IPG, irq_ast);
	__u32 icr;
#ifndef IPG_HIGH_INTERRUPT
	icr = IPG_READ(ipg, IPG_ICR);
#else
	do
		icr = ipg->icr;
	while (__unlikely(__CMPXCHG32(&ipg->icr, icr, 0)));
#endif
	/*if (icr) __debug_printf("irq: %08x\n", icr);*/
	if (__unlikely(ipg->mac_type == MAC_82547) || __unlikely(ipg->mac_type == MAC_82547_rev_2))
		IPG_WRITE(ipg, IPG_IMC, ~0);
	if (__likely(icr & (IPG_ICR_TXDW | IPG_ICR_TXQE)))
		IPG_REAP_TX(ipg);
	if (__likely(icr & (IPG_ICR_RXDMT0 | IPG_ICR_RXTO)))
		IPG_PROCESS_RX(ipg);
	if (__unlikely(icr & (IPG_ICR_LSC | IPG_ICR_RXSEQ))) {
		ipg->get_link_status = 1;
		KERNEL$DEL_TIMER(&ipg->media_timer);
		KERNEL$SET_TIMER(0, &ipg->media_timer);
	}
	if (__unlikely(ipg->mac_type == MAC_82547) || __unlikely(ipg->mac_type == MAC_82547_rev_2))
		IPG_WRITE(ipg, IPG_IMS, INTERRUPT_MASK(ipg));
#ifndef IPG_HIGH_INTERRUPT
	KERNEL$UNMASK_IRQ(ipg->irq_ctrl);
#endif
	RETURN;
}

static __finline__ void IPG_MEDIA_TIMER_COMMON(IPG *ipg)
{
	u_jiffies_lo_t j;

	j = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(j - ipg->last_decay_time > IPG_DECAY_TIME)) {
		ipg->last_decay_time = j;
		if (__likely(ipg->max_sent > DEFAULT_MAX_SENT)) {
			ipg->max_sent--;
			/*__debug_printf("decay: %d\n", ipg->max_sent);*/
		}
	}
	if (__unlikely(j - ipg->tx_last_time > IPG_TX_TIMEOUT && ipg->n_sent)) {
		/*__debug_printf("reap from timer\n");*/
		IPG_REAP_TX(ipg);
		j = KERNEL$GET_JIFFIES_LO();
		if (__unlikely(j - ipg->tx_last_time > IPG_TX_TIMEOUT && ipg->n_sent)) {
			IPG_TIMEOUT(ipg);
		}
	}
	if (IPG_RAR_WORKAROUND(ipg)) {
		IPG_WRITE(ipg, IPG_RAL, ipg->address[0] | (ipg->address[1] << 8) | (ipg->address[2] << 16) | (ipg->address[3] << 24));
		IPG_FLUSH(ipg);
		IPG_WRITE(ipg, IPG_RAH, ipg->address[4] | (ipg->address[5] << 8) | IPG_RAH_AV);
		IPG_FLUSH(ipg);
	}
}

void IPG_COPPER_MEDIA_TIMER(TIMER *t)
{
	__u32 status;
	int link_up;
	IPG *ipg;
	int link_chg;
	__u16 phy_status;

	LOWER_SPL(SPL_IPG);

	ipg = GET_STRUCT(t, IPG, media_timer);

	IPG_MEDIA_TIMER_COMMON(ipg);

	link_chg = ipg->get_link_status;

	/* e1000_watchdog */

	/* e1000_check_for_link */

	status = IPG_READ(ipg, IPG_STATUS);

	if (__likely(!ipg->get_link_status))
		goto link_done;
	if (IPG_READ_PHY(ipg, PHY_STATUS, &phy_status)) goto link_done;
	if (IPG_READ_PHY(ipg, PHY_STATUS, &phy_status)) goto link_done;
	if (phy_status & MII_SR_LINK_STATUS) {
		ipg->get_link_status = 0;
		if ((__unlikely(ipg->mac_type == MAC_82543) || __unlikely(ipg->mac_type == MAC_82544)) && !(ipg->link_mode & (LINK_AUTO | LINK_100))) {
			__u32 icr;
			IPG_WRITE(ipg, IPG_IMC, ~0);
			IPG_POLARITY_REVERSAL_WORKAROUND(ipg, 0);
			icr = IPG_READ(ipg, IPG_ICR);
			IPG_WRITE(ipg, IPG_ICS, icr & ~IPG_ICR_LSC);
			IPG_WRITE(ipg, IPG_IMS, INTERRUPT_MASK(ipg));
		}
	}

	link_done:
	link_up = !!(status & IPG_STATUS_LU);
	if (__unlikely(link_chg) || __unlikely(link_up * LINK_STATE_UP != (ipg->link_state.flags & LINK_STATE_UP))) {
		IPG_LINK_CHANGED(ipg, link_up);
	}

	KERNEL$SET_TIMER(IPG_MEDIA_TIME, &ipg->media_timer);
}

void IPG_FIBER_MEDIA_TIMER(TIMER *t)
{
	__u32 ctrl, status, signal, rxcw, txcw;
	int link_up;
	IPG *ipg;
	int link_chg;

	LOWER_SPL(SPL_IPG);

	ipg = GET_STRUCT(t, IPG, media_timer);

	IPG_MEDIA_TIMER_COMMON(ipg);

	link_chg = 0;

	/* e1000_watchdog */

	/* e1000_check_for_link */

	status = IPG_READ(ipg, IPG_STATUS);
	ctrl = IPG_READ(ipg, IPG_CTRL);
	rxcw = IPG_READ(ipg, IPG_RXCW);
	txcw = IPG_READ(ipg, IPG_TXCW);
	signal = ipg->mac_type > MAC_82544 ? IPG_CTRL_SDP1_DATA : 0;

	if (__likely(!(ctrl & IPG_CTRL_SLU)) && __unlikely(!(status & IPG_STATUS_LU)) && __unlikely(!(rxcw & IPG_RXCW_RXCONFIG)) && ((ipg->media == MEDIA_FIBER && __likely((ctrl & IPG_CTRL_SDP1_DATA) == signal)) || ipg->media == MEDIA_SERDES)) {
		if (!ipg->fiber_autoneg_failed) {
			ipg->fiber_autoneg_failed = 1;
			goto link_done;
		}
		txcw &= ~IPG_TXCW_ANE;
		IPG_WRITE(ipg, IPG_TXCW, txcw);
		ctrl |= IPG_CTRL_SLU | IPG_CTRL_FD;
		IPG_WRITE(ipg, IPG_CTRL, ctrl);
		link_chg = 1;
	} else if (__unlikely(ctrl & IPG_CTRL_SLU) && __unlikely(rxcw & IPG_RXCW_RXCONFIG)) {
		txcw |= IPG_TXCW_ANE;
		IPG_WRITE(ipg, IPG_TXCW, txcw);
		ctrl &= ~IPG_CTRL_SLU;
		IPG_WRITE(ipg, IPG_CTRL, ctrl);
		ipg->fiber_autoneg_failed = 0;
		link_chg = 1;
	}

	link_done:
	if (__unlikely(ipg->media == MEDIA_SERDES) && __unlikely(!(txcw & IPG_TXCW_ANE))) {
		KERNEL$UDELAY(10);
		rxcw = IPG_READ(ipg, IPG_RXCW);
		link_up = (rxcw & (IPG_RXCW_RXSYNCHRONIZE | IPG_RXCW_RXCONFIGINVALID)) == IPG_RXCW_RXSYNCHRONIZE;
	} else {
		link_up = !!(status & IPG_STATUS_LU);
	}
	if (__unlikely(link_chg) || __unlikely(link_up * LINK_STATE_UP != (ipg->link_state.flags & LINK_STATE_UP))) {
		IPG_LINK_CHANGED(ipg, link_up);
	}

	KERNEL$SET_TIMER(IPG_MEDIA_TIME, &ipg->media_timer);
}

static int IPG_TEST_TBI(IPG *ipg, PACKET *p, IPG_RX_DESC *rx)
{
	if (__unlikely(!IPG_TBI(ipg)))
		goto packet_error;
	if (__unlikely((rx->errors & IPG_RX_DESC_ERRORS_MASK) != IPG_RX_DESC_ERRORS_CE))
		goto packet_error;
	if (__unlikely(p->data[LL_HEADER + p->data_length + 4 - 1] != 0x0f))
		goto packet_error;
	/* TODO: check CRC ? Linux doesn't do it */
	p->data_length--;
	return 0;

	packet_error:
	if (rx->errors & IPG_RX_DESC_ERRORS_CE)
		if (ipg->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, ipg->dev_name, "CRC ERROR");
	if (rx->errors & IPG_RX_DESC_ERRORS_SE)
		if (ipg->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ipg->dev_name, "SYMBOL ERROR");
	if (rx->errors & IPG_RX_DESC_ERRORS_SEQ)
		if (ipg->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ipg->dev_name, "SEQUENCE ERROR");
	if (rx->errors & IPG_RX_DESC_ERRORS_CXE)
		if (ipg->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ipg->dev_name, "CARRIER EXTENSION ERROR");
	if (rx->errors & IPG_RX_DESC_ERRORS_RXE)
		if (ipg->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ipg->dev_name, "RECEIVE ERROR");
	return -1;
}

void IPG_TEAR_DOWN_RECV_PACKETS(IPG *ipg)
{
	unsigned j;
	for (j = 0; j < ipg->n_recv; j++) {
		PACKET *p = ipg->recv_packets[(ipg->first_recv + j) & ipg->recv_mask];
		IPG_UNMAP_RECV_PACKET(ipg, p);
		FREE_PACKET(p, NULL, SPL_IPG);
	}
}

void IPG_TEAR_DOWN_SENT_PACKETS(IPG *ipg)
{
	unsigned i;
	for (i = 0; i < ipg->n_sent; ipg++) {
		PACKET *p;
		unsigned idx = (ipg->first_sent + i) & ipg->sent_mask;
		IPG_UNMAP_TX_DESC(ipg, idx);
		if (__likely((p = ipg->xmit_info[idx].p) != NULL)) {
			p->status = -EAGAIN;
			CALL_PKT(p);
		}
	}
	ipg->first_sent = 0;
	ipg->n_sent = 0;
}

void IPG_REINIT_RX_RING(IPG *ipg)
{
	unsigned j;
	for (j = 0; j < ipg->n_recv; j++) {
		IPG_RX_DESC *rx = &RX_DESCS(ipg)[(ipg->first_recv + j)& ipg->recv_mask];
		rx->status = 0;
	}
}

