#include <SPAD/SYSLOG.H>

#include "IPGREG.H"
#include "IPG.H"

static int IPG_READ_EEPROM_EERD(IPG *ipg, unsigned offset, unsigned len, __u16 *data);
static int IPG_READ_EEPROM_ICH8(IPG *ipg, unsigned offset, unsigned len, __u16 *data);
static int IPG_READ_ICH8_DATA(IPG *ipg, unsigned offset, unsigned len, __u16 *data);
static int IPG_ICH8_FLASH_CYCLE_INIT(IPG *ipg);
static int IPG_ICH8_FLASH_CYCLE(IPG *ipg, unsigned timeout);

static __u16 IPG_SHIFT_IN_EE_BITS(IPG *ipg, unsigned count);
static void IPG_SHIFT_OUT_EE_BITS(IPG *ipg, __u16 data, unsigned count);
static void IPG_STANDBY_EEPROM(IPG *ipg);
static int IPG_SPI_EEPROM_READY(IPG *ipg);

static int IPG_ACQUIRE_EEPROM(IPG *ipg);
static void IPG_RELEASE_EEPROM(IPG *ipg);
static int IPG_80003_GET_SWSM(IPG *ipg);
static void IPG_80003_PUT_SWSM(IPG *ipg);
static int IPG_GET_HW_EEPROM_SEMAPHORE(IPG *ipg);
static void IPG_PUT_HW_EEPROM_SEMAPHORE(IPG *ipg);

int IPG_INIT_EEPROM(IPG *ipg)
{
	__u32 eecd = IPG_READ(ipg, IPG_EECD);
	if (ipg->mac_type <= MAC_82544) {
		ipg->eeprom.type = EEPROM_MICROWIRE;
		ipg->eeprom.word_size = 64;
		ipg->eeprom.opcode_bits = 3;
		ipg->eeprom.delay_usec = 50;
		ipg->eeprom.use_eerd = 0;
	} else if (ipg->mac_type <= MAC_82546_rev_3) {
		ipg->eeprom.type = EEPROM_MICROWIRE;
		ipg->eeprom.opcode_bits = 3;
		ipg->eeprom.delay_usec = 50;
		if (eecd & IPG_EECD_EE_SIZE) {
			ipg->eeprom.word_size = 256;
			ipg->eeprom.address_bits = 8;
		} else {
			ipg->eeprom.word_size = 64;
			ipg->eeprom.address_bits = 6;
		}
	} else if (ipg->mac_type <= MAC_82547_rev_2) {
		if (eecd & IPG_EECD_EE_TYPE) {
			ipg->eeprom.type = EEPROM_SPI;
			ipg->eeprom.opcode_bits = 8;
			ipg->eeprom.delay_usec = 1;
			if (eecd & IPG_EECD_EE_SIZE_2) {
				ipg->eeprom.page_size = 32;
				ipg->eeprom.address_bits = 16;
			} else {
				ipg->eeprom.page_size = 8;
				ipg->eeprom.address_bits = 8;
			}
		} else {
			ipg->eeprom.type = EEPROM_MICROWIRE;
			ipg->eeprom.opcode_bits = 3;
			ipg->eeprom.delay_usec = 50;
			if (eecd & IPG_EECD_EE_SIZE_2) {
				ipg->eeprom.word_size = 256;
				ipg->eeprom.address_bits = 8;
			} else {
				ipg->eeprom.word_size = 64;
				ipg->eeprom.address_bits = 6;
			}
		}
		ipg->eeprom.use_eerd = 0;
	} else if (ipg->mac_type <= MAC_82572) {
		ipg->eeprom.type = EEPROM_SPI;
		ipg->eeprom.opcode_bits = 8;
		ipg->eeprom.delay_usec = 1;
		if (eecd & IPG_EECD_EE_SIZE_2) {
			ipg->eeprom.page_size = 32;
			ipg->eeprom.address_bits = 16;
		} else {
			ipg->eeprom.page_size = 8;
			ipg->eeprom.address_bits = 8;
		}
		ipg->eeprom.use_eerd = 0;
	} else if (ipg->mac_type == MAC_82573) {
		ipg->eeprom.type = EEPROM_SPI;
		ipg->eeprom.opcode_bits = 8;
		ipg->eeprom.delay_usec = 1;
		if (eecd & IPG_EECD_EE_SIZE_2) {
			ipg->eeprom.page_size = 32;
			ipg->eeprom.address_bits = 16;
		} else {
			ipg->eeprom.page_size = 8;
			ipg->eeprom.address_bits = 8;
		}
		ipg->eeprom.use_eerd = 1;
		if ((eecd & IPG_EECD_NVADDS) == IPG_EECD_NVADDS) {
			ipg->eeprom.type = EEPROM_FLASH;
			ipg->eeprom.word_size = 2048;
			eecd &= ~IPG_EECD_AUPDEN;
			IPG_WRITE(ipg, IPG_EECD, eecd);
		}
	} else if (ipg->mac_type == MAC_80003es2lan) {
		ipg->eeprom.type = EEPROM_SPI;
		ipg->eeprom.opcode_bits = 8;
		ipg->eeprom.delay_usec = 1;
		if (eecd & IPG_EECD_EE_SIZE_2) {
			ipg->eeprom.page_size = 32;
			ipg->eeprom.address_bits = 16;
		} else {
			ipg->eeprom.page_size = 8;
			ipg->eeprom.address_bits = 8;
		}
		ipg->eeprom.use_eerd = 1;
	} else if (ipg->mac_type == MAC_ich8lan) {
		unsigned flash_size = ICH8_READ_32(ipg, ICH8_FLASH_GFPREG);
		ipg->eeprom.type = EEPROM_ICH8;
		ipg->eeprom.use_eerd = 0;
		ipg->eeprom.word_size = ICH8_FLASH_WORDS;
		ipg->flash_base_addr = (flash_size & ICH8_GFPREG_BASE_MASK) * ICH8_FLASH_SECTOR_SIZE;
		ipg->flash_bank_size = ((flash_size >> 16) & ICH8_GFPREG_BASE_MASK) + 1;
		ipg->flash_bank_size -= (flash_size & ICH8_GFPREG_BASE_MASK);
		ipg->flash_bank_size *= ICH8_FLASH_SECTOR_SIZE;
		ipg->flash_bank_size /= 2 * sizeof(__u16);
	} else KERNEL$SUICIDE("IPG_INIT_EEPROM: UNKNOWN MAC %d", ipg->mac_type);
	if (ipg->eeprom.type == EEPROM_SPI) {
		__u16 eeprom_size;
		if (ipg->mac_type <= MAC_82547_rev_2) {
			int r;
			ipg->eeprom.word_size = 64;
			if ((r = IPG_READ_EEPROM(ipg, EEPROM_CFG, 1, &eeprom_size))) return r;
			eeprom_size = (eeprom_size & EEPROM_SIZE_MASK) >> EEPROM_SIZE_SHIFT;
			eeprom_size += !!eeprom_size;
		} else {
			eeprom_size = (eecd & IPG_EECD_SIZE_EX_MASK) >> __BSF_CONST(IPG_EECD_SIZE_EX_MASK);
		}
		ipg->eeprom.word_size = 1 << EEPROM_WORD_SIZE_SHIFT << eeprom_size;
	}
	return 0;
}

int IPG_READ_EEPROM(IPG *ipg, unsigned offset, unsigned len, __u16 *data)
{
	int r;
	if (__unlikely(!len)) return 0;
	if (__unlikely(offset + len > ipg->eeprom.word_size)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "AN ATTEMPT TO READ OUT OF EEPROM, OFFSET %u, LENGTH %u, EEPROM SIZE %u", offset, len, (unsigned)ipg->eeprom.word_size);
		return -EINVAL;
	}
	if (ipg->eeprom.use_eerd) {
		return IPG_READ_EEPROM_EERD(ipg, offset, len, data);
	}
	if (ipg->eeprom.type == EEPROM_ICH8) {
		return IPG_READ_EEPROM_ICH8(ipg, offset, len, data);
	}
	if (__unlikely(r = IPG_ACQUIRE_EEPROM(ipg))) return r;
	if (ipg->eeprom.type == EEPROM_MICROWIRE) {
		do {
			IPG_SHIFT_OUT_EE_BITS(ipg, EEPROM_READ_OPCODE_MICROWIRE, ipg->eeprom.opcode_bits);
			IPG_SHIFT_OUT_EE_BITS(ipg, offset, ipg->eeprom.address_bits);
			*data = IPG_SHIFT_IN_EE_BITS(ipg, 16);
			IPG_STANDBY_EEPROM(ipg);
		} while (++offset, ++data, --len);
	} else if (ipg->eeprom.type == EEPROM_SPI) {
		__u8 read_opcode;
		if (__unlikely(r = IPG_SPI_EEPROM_READY(ipg))) {
			IPG_RELEASE_EEPROM(ipg);
			return r;
		}
		IPG_STANDBY_EEPROM(ipg);
		read_opcode = EEPROM_READ_OPCODE_SPI;
		if (ipg->eeprom.address_bits == 8 && offset >= 128) read_opcode |= EEPROM_A8_OPCODE_SPI;
		IPG_SHIFT_OUT_EE_BITS(ipg, read_opcode, ipg->eeprom.opcode_bits);
		IPG_SHIFT_OUT_EE_BITS(ipg, offset * 2, ipg->eeprom.address_bits);
		do {
			__u16 word = IPG_SHIFT_IN_EE_BITS(ipg, 16);
			word = (word >> 8) | (word << 8);
			*data = word;
		} while (++offset, ++data, --len);
	} else KERNEL$SUICIDE("IPG_READ_EEPROM: UNKNOWN EEPROM TYPE %u", (unsigned)ipg->eeprom.type);
	IPG_RELEASE_EEPROM(ipg);
	return 0;
}

static int IPG_READ_EEPROM_EERD(IPG *ipg, unsigned offset, unsigned len, __u16 *data)
{
	do {
		unsigned i;
		IPG_WRITE(ipg, IPG_EERD, (offset << __BSF_CONST(IPG_EERD_ADDR_2)) | IPG_EERD_START);
		for (i = 0; i < 100000; i++) {
			if (IPG_READ(ipg, IPG_EERD) & IPG_EERD_DONE_2) goto eerd_ok;
			KERNEL$UDELAY(5);
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "TIMEOUT WHEN READING EEPROM AT OFFSET %u", offset);
		return -ETIMEDOUT;
		eerd_ok:;
		*data = IPG_READ(ipg, IPG_EERD) >> __BSF_CONST(IPG_EERD_DATA);
	} while (++offset, ++data, --len);
	return 0;
}

static int IPG_READ_EEPROM_ICH8(IPG *ipg, unsigned offset, unsigned len, __u16 *data)
{
	int r;
	unsigned flash_bank = !!(IPG_READ(ipg, IPG_EECD) & IPG_EECD_SEC1VAL);
	unsigned bank_offset = flash_bank * ipg->flash_bank_size * 2;
	if (__unlikely(r = IPG_GET_SOFTWARE_FLAG(ipg))) return r;
	do {
		if (__unlikely(r = IPG_READ_ICH8_DATA(ipg, bank_offset + offset * 2, 2, data))) return r;
	} while (++offset, ++data, --len);
	IPG_RELEASE_SOFTWARE_FLAG(ipg);
	return 0;
}

static int IPG_READ_ICH8_DATA(IPG *ipg, unsigned offset, unsigned len, __u16 *data)
{
	unsigned i;
	unsigned flash_linear_address = (ICH8_FLASH_LINEAR_ADDR_MASK & offset) + ipg->flash_base_addr;
	for (i = 0; i < 10; i++) {
		int r;
		__u32 hsfctl, hsfsts;
		KERNEL$UDELAY(1);
		if (__unlikely(r = IPG_ICH8_FLASH_CYCLE_INIT(ipg))) return r;
		hsfctl = ICH8_READ_16(ipg, ICH8_FLASH_HSFCTL);
		hsfctl = (hsfctl & ~ICH8_FLASH_HSFCTL_FLDBCOUNT) | ((len - 1) << __BSF_CONST(ICH8_FLASH_HSFCTL_FLDBCOUNT));
		hsfctl = (hsfctl & ~ICH8_FLASH_HSFCTL_FLCYCLE) | ICH8_FLASH_HSFCTL_FLCYCLE_READ;
		ICH8_WRITE_16(ipg, ICH8_FLASH_HSFCTL, hsfctl);
		ICH8_WRITE_32(ipg, ICH8_FLASH_FADDR, flash_linear_address);
		if (__likely(!(r = IPG_ICH8_FLASH_CYCLE(ipg, 5000)))) {
			*data = ICH8_READ_32(ipg, ICH8_FLASH_FDATA0) & (len == 1 ? 0xff : 0xffff);
			return 0;
		}
		hsfsts = ICH8_READ_16(ipg, ICH8_FLASH_HSFSTS);
		if (!(hsfsts & (ICH8_FLASH_HSFSTS_FLCDONE | ICH8_FLASH_HSFSTS_FLCERR))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ipg->dev_name, "FLASH CYCLE DID NOT COMPLETE");
			return r;
		}
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "TOO MANY UNSUCESSFUL FLASH TRIES");
	return -EIO;
}

static int IPG_ICH8_FLASH_CYCLE_INIT(IPG *ipg)
{
	unsigned i;
	__u32 hsfsts = ICH8_READ_16(ipg, ICH8_FLASH_HSFSTS);
	if (__unlikely(!(hsfsts & ICH8_FLASH_HSFSTS_FLDESVALID))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, ipg->dev_name, "FLASH DESCRIPTOR INVALID: %04X", hsfsts);
		return -EIO;
	}
	hsfsts |= ICH8_FLASH_HSFSTS_FLCERR | ICH8_FLASH_HSFSTS_DAEL;
	ICH8_WRITE_16(ipg, ICH8_FLASH_HSFSTS, hsfsts);
	for (i = 0; i < 5000; i++) {
		if (__likely(!(hsfsts & ICH8_FLASH_HSFSTS_FLCINPROG))) {
			hsfsts |= ICH8_FLASH_HSFSTS_FLCDONE;
			ICH8_WRITE_16(ipg, ICH8_FLASH_HSFSTS, hsfsts);
			return 0;
		}
		KERNEL$UDELAY(1);
		hsfsts = ICH8_READ_16(ipg, ICH8_FLASH_HSFSTS);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "FLASH CONTROLLER BUSY");
	return -ETIMEDOUT;
}

static int IPG_ICH8_FLASH_CYCLE(IPG *ipg, unsigned timeout)
{
	unsigned i;
	__u32 hsfsts;
	__u32 hsfctl = ICH8_READ_16(ipg, ICH8_FLASH_HSFCTL);
	hsfctl |= ICH8_FLASH_HSFCTL_FLCGO;
	ICH8_WRITE_16(ipg, ICH8_FLASH_HSFCTL, hsfctl);
	for (i = 0; i < timeout; i++) {
		hsfsts = ICH8_READ_16(ipg, ICH8_FLASH_HSFSTS);
		if (hsfsts & ICH8_FLASH_HSFSTS_FLCDONE) {
			return hsfsts & ICH8_FLASH_HSFSTS_FLCERR ? -EIO : 0;
		}
		KERNEL$UDELAY(1);
	}
	return -ETIMEDOUT;
}

static void IPG_EE_CLK(IPG *ipg, __u32 *eecd, int rais)
{
	*eecd = (*eecd & ~IPG_EECD_SK) | (rais * IPG_EECD_SK);
	IPG_WRITE(ipg, IPG_EECD, *eecd);
	IPG_FLUSH(ipg);
	KERNEL$UDELAY(ipg->eeprom.delay_usec);
}

static __u16 IPG_SHIFT_IN_EE_BITS(IPG *ipg, unsigned count)
{
	__u16 data;
	__u32 eecd = IPG_READ(ipg, IPG_EECD);
	eecd &= ~(IPG_EECD_DO | IPG_EECD_DI);
	data = 0;
	do {
		IPG_EE_CLK(ipg, &eecd, 1);
		eecd = IPG_READ(ipg, IPG_EECD);
		eecd &= ~IPG_EECD_DI;
		data <<= 1;
		data |= !!(eecd & IPG_EECD_DO);
		IPG_EE_CLK(ipg, &eecd, 0);
	} while (--count);
	return data;
}

static void IPG_SHIFT_OUT_EE_BITS(IPG *ipg, __u16 data, unsigned count)
{
	unsigned mask = 1 << (count - 1);
	__u32 eecd = IPG_READ(ipg, IPG_EECD);
	if (ipg->eeprom.type == EEPROM_MICROWIRE) {
		eecd &= ~IPG_EECD_DO;
	} else if (ipg->eeprom.type == EEPROM_SPI) {
		eecd |= IPG_EECD_DO;
	} else KERNEL$SUICIDE("IPG_SHIFT_OUT_EE_BITS: UNKNOWN EEPROM TYPE %u", (unsigned)ipg->eeprom.type);
	do {
		eecd &= ~IPG_EECD_DI;
		if (data & mask) eecd |= IPG_EECD_DI;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
		IPG_EE_CLK(ipg, &eecd, 1);
		IPG_EE_CLK(ipg, &eecd, 0);
	} while (mask >>= 1);
	eecd &= ~IPG_EECD_DI;
	IPG_WRITE(ipg, IPG_EECD, eecd);
}

static void IPG_STANDBY_EEPROM(IPG *ipg)
{
	__u32 eecd = IPG_READ(ipg, IPG_EECD);
	if (ipg->eeprom.type == EEPROM_MICROWIRE) {
		eecd &= ~(IPG_EECD_CS | IPG_EECD_SK);
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
		eecd |= IPG_EECD_SK;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
		eecd |= IPG_EECD_CS;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
		eecd &= ~IPG_EECD_SK;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
	} else if (ipg->eeprom.type == EEPROM_SPI) {
		eecd |= IPG_EECD_CS;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
		eecd &= ~IPG_EECD_CS;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
	} else KERNEL$SUICIDE("IPG_STANDBY_EEPROM: UNKNOWN EEPROM TYPE %u", (unsigned)ipg->eeprom.type);
}

static int IPG_SPI_EEPROM_READY(IPG *ipg)
{
	unsigned i;
	for (i = 0; i < 1000; i++) {
		__u8 status;
		IPG_SHIFT_OUT_EE_BITS(ipg, EEPROM_RDSR_OPCODE_SPI, ipg->eeprom.opcode_bits);
		status = IPG_SHIFT_IN_EE_BITS(ipg, 8);
		if (__likely(!(status & EEPROM_STATUS_RDY_SPI))) return 0;
		KERNEL$UDELAY(5);
		IPG_STANDBY_EEPROM(ipg);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "EEPROM NOT READY");
	return -ETIMEDOUT;
}

static int IPG_ACQUIRE_EEPROM(IPG *ipg)
{
	int r;
	__u32 eecd;
	if (__unlikely(r = IPG_SWFW_SYNC_ACQUIRE(ipg, IPG_SW_FW_SYNC_EEP_SM))) return r;
	eecd = IPG_READ(ipg, IPG_EECD);
	if (ipg->mac_type >= MAC_82540 && ipg->mac_type != MAC_82573) {
		unsigned i;
		eecd |= IPG_EECD_EE_REQ;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		for (i = 0; i < 1000; i++) {
			eecd = IPG_READ(ipg, IPG_EECD);
			if (eecd & IPG_EECD_EE_GNT) goto granted;
			KERNEL$UDELAY(5);
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "EEPROM ACCESS NOT GRANTED");
		eecd &= ~IPG_EECD_EE_REQ;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_SWFW_SYNC_RELEASE(ipg, IPG_SW_FW_SYNC_EEP_SM);
		return -ETIMEDOUT;
		granted:;
	}
	if (ipg->eeprom.type == EEPROM_MICROWIRE) {
		eecd &= ~(IPG_EECD_DI | IPG_EECD_SK);
		IPG_WRITE(ipg, IPG_EECD, eecd);
		eecd |= IPG_EECD_CS;
		IPG_WRITE(ipg, IPG_EECD, eecd);
	}
	if (ipg->eeprom.type == EEPROM_SPI) {
		eecd &= ~(IPG_EECD_CS | IPG_EECD_SK);
		IPG_WRITE(ipg, IPG_EECD, eecd);
		KERNEL$UDELAY(1);
	}
	return 0;
}

static void IPG_RELEASE_EEPROM(IPG *ipg)
{
	__u32 eecd = IPG_READ(ipg, IPG_EECD);
	if (ipg->eeprom.type == EEPROM_MICROWIRE) {
		eecd &= ~(IPG_EECD_CS | IPG_EECD_DI);
		IPG_WRITE(ipg, IPG_EECD, eecd);
		eecd |= IPG_EECD_SK;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
		eecd &= ~IPG_EECD_SK;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
	}
	if (ipg->eeprom.type == EEPROM_SPI) {
		eecd |= IPG_EECD_CS;
		eecd &= ~IPG_EECD_SK;
		IPG_WRITE(ipg, IPG_EECD, eecd);
		KERNEL$UDELAY(ipg->eeprom.delay_usec);
	}
	if (ipg->mac_type >= MAC_82540) {
		eecd &= ~IPG_EECD_EE_REQ;
		IPG_WRITE(ipg, IPG_EECD, eecd);
	}
	IPG_SWFW_SYNC_RELEASE(ipg, IPG_SW_FW_SYNC_EEP_SM);
}

int IPG_SWFW_SYNC_ACQUIRE(IPG *ipg, __u16 mask)
{
	__u32 sw_fw_sync;
	unsigned i;
	if (ipg->mac_type >= MAC_ich8lan) return IPG_GET_SOFTWARE_FLAG(ipg);
	if (ipg->mac_type != MAC_80003es2lan) {
		if (ipg->mac_type < MAC_82571) return 0;
		return IPG_GET_HW_EEPROM_SEMAPHORE(ipg);
	}
	for (i = 0; i < 200; i++) {
		int r;
		if (__unlikely(r = IPG_80003_GET_SWSM(ipg))) return r;
		sw_fw_sync = IPG_READ(ipg, IPG_SW_FW_SYNC);
		if (__likely(!(sw_fw_sync & (mask | (mask << 16))))) {
			sw_fw_sync |= mask;
			IPG_WRITE(ipg, IPG_SW_FW_SYNC, sw_fw_sync);
			IPG_80003_PUT_SWSM(ipg);
			return 0;
		}
		IPG_80003_PUT_SWSM(ipg);
		KERNEL$UDELAY(5);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "CAN'T ACCESS RESOURCE %04X, SW_FW_SYNC %08X", mask, sw_fw_sync);
	return -ETIMEDOUT;
}

void IPG_SWFW_SYNC_RELEASE(IPG *ipg, __u16 mask)
{
	char acq;
	__u32 sw_fw_sync;
	if (ipg->mac_type >= MAC_ich8lan) {
		IPG_RELEASE_SOFTWARE_FLAG(ipg);
		return;
	}
	if (ipg->mac_type != MAC_80003es2lan) {
		if (ipg->mac_type < MAC_82571) return;
		IPG_PUT_HW_EEPROM_SEMAPHORE(ipg);
		return;
	}
	acq = !IPG_80003_GET_SWSM(ipg);
	sw_fw_sync = IPG_READ(ipg, IPG_SW_FW_SYNC);
	sw_fw_sync &= ~mask;
	IPG_WRITE(ipg, IPG_SW_FW_SYNC, sw_fw_sync);
	if (__likely(acq)) IPG_80003_PUT_SWSM(ipg);
}

static int IPG_80003_GET_SWSM(IPG *ipg)
{
	unsigned i;
	for (i = 0; i < ipg->eeprom.word_size + 1; i++) {
		__u32 swsm = IPG_READ(ipg, IPG_SWSM);
		if (!(swsm & IPG_SWSM_SMBI)) return 0;
		KERNEL$UDELAY(1000);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "CAN'T GET SOFTWARE SEMAPHORE");
	return -ETIMEDOUT;
}

static void IPG_80003_PUT_SWSM(IPG *ipg)
{
	__u32 swsm = IPG_READ(ipg, IPG_SWSM);
	swsm &= ~(IPG_SWSM_SMBI | IPG_SWSM_SWESMBI);
	IPG_WRITE(ipg, IPG_SWSM, swsm);
}

static int IPG_GET_HW_EEPROM_SEMAPHORE(IPG *ipg)
{
	unsigned i;
	for (i = 0; i < ipg->eeprom.word_size + 1; i++) {
		__u32 swsm = IPG_READ(ipg, IPG_SWSM);
		swsm |= IPG_SWSM_SWESMBI;
		IPG_WRITE(ipg, IPG_SWSM, swsm);
		swsm = IPG_READ(ipg, IPG_SWSM);
		if (swsm & IPG_SWSM_SWESMBI) return 0;
		KERNEL$UDELAY(50);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "CAN'T GET EEPROM SEMAPHORE");
	return -ETIMEDOUT;
}

static void IPG_PUT_HW_EEPROM_SEMAPHORE(IPG *ipg)
{
	__u32 swsm = IPG_READ(ipg, IPG_SWSM);
	swsm &= ~IPG_SWSM_SWESMBI;
	IPG_WRITE(ipg, IPG_SWSM, swsm);
}

int IPG_GET_SOFTWARE_FLAG(IPG *ipg)
{
	unsigned i;
	if (__unlikely(ipg->mac_type < MAC_ich8lan))
		KERNEL$SUICIDE("IPG_GET_SOFTWARE_FLAG: BAD MAC TYPE: %d", ipg->mac_type);
	for (i = 0; i < 100; i++) {
		IPG_WRITE(ipg, IPG_EXTCNF_CTRL, IPG_READ(ipg, IPG_EXTCNF_CTRL) | IPG_EXTCNF_CTRL_SWFLAG);
		if (IPG_READ(ipg, IPG_EXTCNF_CTRL) & IPG_EXTCNF_CTRL_SWFLAG) return 0;
		KERNEL$UDELAY(1000);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "SOFTWARE FLAG LOCKED TOO LONG");
	return -ETIMEDOUT;
}

void IPG_RELEASE_SOFTWARE_FLAG(IPG *ipg)
{
	if (__unlikely(ipg->mac_type < MAC_ich8lan))
		KERNEL$SUICIDE("IPG_GET_SOFTWARE_FLAG: BAD MAC TYPE: %d", ipg->mac_type);
	IPG_WRITE(ipg, IPG_EXTCNF_CTRL, IPG_READ(ipg, IPG_EXTCNF_CTRL) & ~IPG_EXTCNF_CTRL_SWFLAG);
}

