#include <SPAD/SYSLOG.H>

#include "IPGREG.H"
#include "IPG.H"

#define FC_PAUSE_TIME		0x0680

static void IPG_SETUP_COPPER_LINK(IPG *ipg, int syn);
static void IPG_SETUP_FIBER_SERDES_LINK(IPG *ipg, int syn);

void IPG_SETUP_LINK(IPG *ipg, int syn)
{
	__u16 c2;
	if (IPG_CHECK_PHY_RESET_BLOCK(ipg)) return;
	if (ipg->mac_type == MAC_82543) {
		__u32 ctrl_ext;
		IPG_READ_EEPROM(ipg, EEPROM_INIT_CONTROL2_REG, 1, &c2);
		ctrl_ext = (c2 & EEPROM_INIT_CONTROL2_REG_SWPDIO_EXT) << 4;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
	}
	if (ipg->media == MEDIA_COPPER) IPG_SETUP_COPPER_LINK(ipg, syn);
	else IPG_SETUP_FIBER_SERDES_LINK(ipg, syn);
	if (ipg->mac_type < MAC_ich8lan) {
		IPG_WRITE(ipg, IPG_FCT, 0x00008808);
		IPG_WRITE(ipg, IPG_FCAH, 0x00000100);
		IPG_WRITE(ipg, IPG_FCAL, 0x00c28001);
	}
	if (ipg->mac_type == MAC_80003es2lan) {
		IPG_WRITE(ipg, IPG_FCTTV, 0xffff);
	} else {
		IPG_WRITE(ipg, IPG_FCTTV, FC_PAUSE_TIME);
	}
	if (!(ipg->flow_control & FC_TX_PAUSE)) {
		IPG_WRITE(ipg, IPG_FCRTL, 0);
		IPG_WRITE(ipg, IPG_FCRTH, 0);
	} else {
		IPG_WRITE(ipg, IPG_FCRTL, ipg->fc_low_water_mark | IPG_FCRTL_XONE);
		IPG_WRITE(ipg, IPG_FCRTH, ipg->fc_high_water_mark);
	}
}

static void COPPER_LINK_PRECONFIG(IPG *ipg, int syn);
static void IGP_PHY_COPPER_SETUP(IPG *ipg, int syn);
static void M88_PHY_COPPER_SETUP(IPG *ipg, int syn);
static void GGP_PHY_COPPER_SETUP(IPG *ipg, int syn);

static void IPG_SETUP_COPPER_LINK(IPG *ipg, int syn)
{
	__u16 adv, t1000, phy_ctrl;
	if (ipg->mac_type == MAC_80003es2lan || ipg->mac_type >= MAC_ich8lan) {
		__u16 data;
		IPG_WRITE_KMRN(ipg, KUMCTRLSTA_OFFSET_TIMEOUTS, 0xFFFF);
		if (IPG_READ_KMRN(ipg, KUMCTRLSTA_OFFSET_INB_PARAM, &data)) goto skip_1;
		data |= 0x3f;
		IPG_WRITE_KMRN(ipg, KUMCTRLSTA_OFFSET_INB_PARAM, data);
		skip_1:;
	}
	COPPER_LINK_PRECONFIG(ipg, syn);
	if (ipg->mac_type == MAC_80003es2lan) {
		IPG_WRITE_KMRN(ipg, KUMCTRLSTA_OFFSET_INB_CTRL, KUMCTRLSTA_INB_CTRL_DIS_PADDING | KUMCTRLSTA_INB_CTRL_LINK_STATUS_TX_TIMEOUT_DEFAULT);
	}
	if (ipg->phy.type == PHY_IGP || ipg->phy.type == PHY_IGP_2 || ipg->phy.type == PHY_IGP_3) IGP_PHY_COPPER_SETUP(ipg, syn);
	if (ipg->phy.type == PHY_M88) M88_PHY_COPPER_SETUP(ipg, syn);
	if (ipg->phy.type == PHY_GG82563) GGP_PHY_COPPER_SETUP(ipg, syn);
	if (ipg->link_mode == LINK_AUTO) {
		/* e1000_copper_link_autoneg */
		/* e1000_phy_setup_autoneg */
		if (IPG_READ_PHY(ipg, PHY_AUTONEG_ADV, &adv)) goto skip_phy;
		t1000 = 0;
		if (ipg->phy.type != PHY_IFE) {
			if (IPG_READ_PHY(ipg, PHY_1000T_CTRL, &t1000)) goto skip_phy;
		}
		adv |= PHY_AUTONEG_ADV_10T_HD_CAPS | PHY_AUTONEG_ADV_10T_FD_CAPS | PHY_AUTONEG_ADV_100TX_HD_CAPS | PHY_AUTONEG_ADV_100TX_FD_CAPS;
		t1000 &= ~(PHY_1000T_CTRL_HD_CAPS | PHY_1000T_CTRL_FD_CAPS);
		if (ipg->phy.type != PHY_IFE) {
			t1000 |= PHY_1000T_CTRL_FD_CAPS;
		}
		adv &= ~(PHY_AUTONEG_ADV_PAUSE | PHY_AUTONEG_ADV_ASM_DIR);
		if (ipg->flow_control == FC_TX_PAUSE) adv |= PHY_AUTONEG_ADV_PAUSE;
		if (ipg->flow_control == FC_RX_PAUSE || ipg->flow_control == FC_FULL) adv |= PHY_AUTONEG_ADV_PAUSE | PHY_AUTONEG_ADV_ASM_DIR;
		if (IPG_WRITE_PHY(ipg, PHY_AUTONEG_ADV, adv)) goto skip_phy;
		if (ipg->phy.type != PHY_IFE) {
			if (IPG_WRITE_PHY(ipg, PHY_1000T_CTRL, t1000)) goto skip_phy;
		}
		skip_phy:
		if (IPG_READ_PHY(ipg, PHY_CTRL, &phy_ctrl)) goto skip_phy_2;
		phy_ctrl |= PHY_CTRL_AUTO_NEG_EN | PHY_CTRL_RESTART_AUTO_NEG;
		if (IPG_WRITE_PHY(ipg, PHY_CTRL, phy_ctrl)) goto skip_phy_2;
		skip_phy_2:;
		/* e1000_copper_link_autoneg continue */
	} else {
		__u32 ipg_ctrl;
		/* e1000_phy_force_speed_duplex */
		ipg_ctrl = IPG_READ(ipg, IPG_CTRL);
		ipg_ctrl |= IPG_CTRL_FRCSPD | IPG_CTRL_FRCDPLX;
		ipg_ctrl &= ~(IPG_CTRL_SPEED | IPG_CTRL_FD | IPG_CTRL_ASDE);
		if (IPG_READ_PHY(ipg, PHY_CTRL, &phy_ctrl)) goto skip_phy_fixed;
		phy_ctrl &= ~(PHY_CTRL_AUTO_NEG_EN | PHY_CTRL_FULL_DUPLEX | PHY_CTRL_SPEED_SELECT_LSB | PHY_CTRL_SPEED_SELECT_MSB);
		if (ipg->link_mode & LINK_FD) {
			ipg_ctrl |= IPG_CTRL_FD;
			phy_ctrl |= PHY_CTRL_FULL_DUPLEX;
		}
		if (ipg->link_mode & LINK_100) {
			ipg_ctrl |= IPG_CTRL_SPEED_100;
			phy_ctrl |= PHY_CTRL_SPEED_SELECT_LSB;
		}
		IPG_SET_COLLISION_DIST(ipg);
		IPG_WRITE(ipg, IPG_CTRL, ipg_ctrl);
		if (ipg->phy.type == PHY_M88 || ipg->phy.type == PHY_GG82563) {
			__u16 spec_ctrl;
			if (IPG_READ_PHY(ipg, M88E1000_PHY_SPEC_CTRL, &spec_ctrl)) goto skip_phy_spec;
			spec_ctrl &= ~M88E1000_PHY_SPEC_CTRL_AUTO_X_MODE;
			IPG_WRITE_PHY(ipg, M88E1000_PHY_SPEC_CTRL, spec_ctrl);
			phy_ctrl |= PHY_CTRL_RESET;
			skip_phy_spec:;
		} else if (ipg->phy.type == PHY_IFE) {
			__u16 mdix_ctrl;
			if (IPG_READ_PHY(ipg, IFE_PHY_MDIX_CONTROL, &mdix_ctrl)) goto skip_mdix;
			mdix_ctrl &= ~(IFE_PHY_MDIX_CONTROL_AUTO_MDIX | IFE_PHY_MDIX_CONTROL_FORCE_MDIX);
			IPG_WRITE_PHY(ipg, IFE_PHY_MDIX_CONTROL, mdix_ctrl);
			skip_mdix:;
		} else {
			__u16 port_ctrl;
			if (IPG_READ_PHY(ipg, IGP01E1000_PHY_PORT_CTRL, &port_ctrl)) goto no_port_ctrl;
			port_ctrl &= ~(IGP01E1000_PHY_PORT_CTRL_AUTO_MDIX | IGP01E1000_PHY_PORT_CTRL_FORCE_MDI_MDIX);
			IPG_WRITE_PHY(ipg, IGP01E1000_PHY_PORT_CTRL, port_ctrl);
			no_port_ctrl:;
		}
		if (IPG_WRITE_PHY(ipg, PHY_CTRL, phy_ctrl)) goto skip_phy_fixed;
		KERNEL$UDELAY(1);
		if (ipg->phy.type == PHY_M88) {
			__u16 phy_spec_ctrl, ext_phy_spec_ctrl;
			if (IPG_READ_PHY(ipg, M88E1000_EXT_PHY_SPEC_CTRL, &ext_phy_spec_ctrl)) goto skip_m88;
			ext_phy_spec_ctrl |= M88E1000_EXT_PHY_SPEC_CTRL_TX_CLK_25;
			if (IPG_WRITE_PHY(ipg, M88E1000_EXT_PHY_SPEC_CTRL, ext_phy_spec_ctrl)) goto skip_m88;
			if (IPG_READ_PHY(ipg, M88E1000_PHY_SPEC_CTRL, &phy_spec_ctrl)) goto skip_m88;
			phy_spec_ctrl |= M88E1000_PHY_SPEC_CTRL_ASSERT_CRS_ON_TX;
			if (IPG_WRITE_PHY(ipg, M88E1000_PHY_SPEC_CTRL, phy_spec_ctrl)) goto skip_m88;
			skip_m88:;
			IPG_POLARITY_REVERSAL_WORKAROUND(ipg, syn);
		} else if (ipg->phy.type == PHY_GG82563) {
			__u16 mac_spec;
			if (IPG_READ_PHY(ipg, GG82563_PHY_MAC_SPEC_CTRL, &mac_spec)) goto skip_gg;
			mac_spec &= ~GG82563_PHY_MAC_SPEC_CTRL_TX_CLK_MASK;
			if (ipg->link_mode & LINK_100) {
				mac_spec |= GG82563_PHY_MAC_SPEC_CTRL_TX_CLK_100MBPS_25MHZ;
			} else {
				mac_spec |= GG82563_PHY_MAC_SPEC_CTRL_TX_CLK_10MBPS_2_5MHZ;
			}
			mac_spec |= GG82563_PHY_MAC_SPEC_CTRL_ASSERT_CRS_ON_TX;
			if (IPG_WRITE_PHY(ipg, GG82563_PHY_MAC_SPEC_CTRL, mac_spec)) goto skip_gg;
			skip_gg:;
		}
		skip_phy_fixed:;
	}
	/*{
		unsigned i;
		__debug_printf("phy (%d)\n", ipg->link_mode);
		for (i = 0; i < 0x20; i++) {
			__u16 val = 0xaaaa;
			IPG_READ_PHY(ipg, i, &val);
			__debug_printf("%04x ", val);
		}
		__debug_printf("\n");
	}*/
	/* e1000_copper_link_postconfig ... is it really needed? What if it
	   starts with link down? */
}

static void COPPER_LINK_PRECONFIG(IPG *ipg, int syn)
{
	__u32 ctrl = IPG_READ(ipg, IPG_CTRL);
	if (ipg->mac_type >= MAC_82544) {
		ctrl |= IPG_CTRL_SLU;
		ctrl &= ~(IPG_CTRL_FRCSPD | IPG_CTRL_FRCDPLX);
		IPG_WRITE(ipg, IPG_CTRL, ctrl);
	} else {
		ctrl |= IPG_CTRL_FRCSPD | IPG_CTRL_FRCDPLX | IPG_CTRL_SLU;
		IPG_WRITE(ipg, IPG_CTRL, ctrl);
		IPG_PHY_HW_RESET(ipg, syn);
	}
	/* e1000_set_phy_mode */
	if (ipg->mac_type == MAC_82545_rev_3) {
		__u16 eeprom_class;
		if (IPG_READ_EEPROM(ipg, EEPROM_PHY_CLASS_WORD, 1, &eeprom_class)) goto skip_class;
		if (eeprom_class != EEPROM_RESERVED_WORD && eeprom_class & EEPROM_PHY_CLASS_A) {
			if (IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0xb << 5), 0x8104)) goto skip_class;
		}
		skip_class:;
	}

	if (ipg->mac_type == MAC_82545_rev_3 || ipg->mac_type == MAC_82546_rev_3) {
		__u16 data;
		if (IPG_READ_PHY(ipg, M88E1000_PHY_SPEC_CTRL, &data)) goto skip_phy;
		data |= 0x00000008;
		IPG_WRITE_PHY(ipg, M88E1000_PHY_SPEC_CTRL, data);
		skip_phy:;
	}
}

static void IGP_PHY_COPPER_SETUP(IPG *ipg, int syn)
{
	__u16 phy_data;
	IPG_PHY_RESET(ipg, syn);
	MSLEEP(syn, 15);
	if (ipg->mac_type < MAC_ich8lan) {
		/* Someone at Intel is nuts. This same code is already at two
		   different places. I don't know which one is needed.
		   They likely don't know either */
		__u32 led_ctl = IPG_READ(ipg, IPG_LEDCTL);
		led_ctl &= ~IPG_LEDCTL_LED1_MOD;
		led_ctl |= 3 << __BSF_CONST(IPG_LEDCTL_LED1_MOD);
		IPG_WRITE(ipg, IPG_LEDCTL, led_ctl);
	}
	if (ipg->phy.type == PHY_IGP) {
		/* e1000_set_d3_lplu_state */
		if (ipg->mac_type == MAC_82541_rev_2 || ipg->mac_type == MAC_82547_rev_2) {
			if (IPG_READ_PHY(ipg, IGP01E1000_GMII_FIFO, &phy_data)) goto no_d3_lpu;
			phy_data &= ~IGP01E1000_GMII_FIFO_FLEX_SPD;
			IPG_WRITE_PHY(ipg, IGP01E1000_GMII_FIFO, phy_data);
		} else if (ipg->mac_type >= MAC_ich8lan) {
			__u32 phy_ctrl = IPG_READ(ipg, IPG_PHY_CTRL);
			phy_ctrl &= ~IPG_PHY_CTRL_NOND0A_LPLU;
			IPG_WRITE(ipg, IPG_PHY_CTRL, phy_ctrl);
		} else {
			if (IPG_READ_PHY(ipg, IGP02E1000_PHY_POWER_MGMT, &phy_data)) goto no_d3_lpu;
			phy_data &= ~IGP02E1000_PHY_POWER_MGMT_D3_LPLU;
			IPG_WRITE_PHY(ipg, IGP02E1000_PHY_POWER_MGMT, phy_data);
		}
		no_d3_lpu:;
	}
	if (ipg->mac_type >= MAC_82571) {
		/* e1000_set_d0_lplu_state */
		if (ipg->mac_type >= MAC_ich8lan) {
			__u32 phy_ctrl = IPG_READ(ipg, IPG_PHY_CTRL);
			phy_ctrl &= ~IPG_PHY_CTRL_D0A_LPLU;
			IPG_WRITE(ipg, IPG_PHY_CTRL, phy_ctrl);
		} else {
			if (IPG_READ_PHY(ipg, IGP02E1000_PHY_POWER_MGMT, &phy_data)) goto no_d0_lpu;
			phy_data &= ~IGP02E1000_PHY_POWER_MGMT_D0_LPLU;
			IPG_WRITE_PHY(ipg, IGP02E1000_PHY_POWER_MGMT, phy_data);
		}
		no_d0_lpu:;
	}
	if (IPG_READ_PHY(ipg, IGP01E1000_PHY_PORT_CTRL, &phy_data)) goto no_mdix;
	if (ipg->mac_type == MAC_82541 || ipg->mac_type == MAC_82547) {
		phy_data &= ~(IGP01E1000_PHY_PORT_CTRL_AUTO_MDIX | IGP01E1000_PHY_PORT_CTRL_FORCE_MDI_MDIX);
	} else {
		phy_data |= IGP01E1000_PHY_PORT_CTRL_AUTO_MDIX;
	}
	IPG_WRITE_PHY(ipg, IGP01E1000_PHY_PORT_CTRL, phy_data);
	no_mdix:
	if (ipg->link_mode == LINK_AUTO) {
		if (IPG_READ_PHY(ipg, PHY_1000T_CTRL, &phy_data)) goto no_ms;
		phy_data &= ~PHY_1000T_CTRL_MS_ENABLE;
		IPG_WRITE_PHY(ipg, PHY_1000T_CTRL, phy_data);
		no_ms:;
	}
}

static void M88_PHY_COPPER_SETUP(IPG *ipg, int syn)
{
	__u16 phy_spec_ctrl, phy_ext_spec_ctrl;
	if (IPG_READ_PHY(ipg, M88E1000_PHY_SPEC_CTRL, &phy_spec_ctrl)) return;
	phy_spec_ctrl |= M88E1000_PHY_SPEC_CTRL_ASSERT_CRS_ON_TX | M88E1000_PHY_SPEC_CTRL_AUTO_X_MODE;
	phy_spec_ctrl &= ~M88E1000_PHY_SPEC_CTRL_POLARITY_REVERSAL;
	if (IPG_WRITE_PHY(ipg, M88E1000_PHY_SPEC_CTRL, phy_spec_ctrl)) return;
	if (ipg->phy.revision < M88E1011_I_REV_4) {
		if (IPG_READ_PHY(ipg, M88E1000_EXT_PHY_SPEC_CTRL, &phy_ext_spec_ctrl)) return;
		phy_ext_spec_ctrl |= M88E1000_EXT_PHY_SPEC_CTRL_TX_CLK_25;
		if (ipg->phy.revision == M88E1011_I_REV_2 && ipg->phy.id == M88E1111_I_PHY_ID) {
			phy_ext_spec_ctrl &= ~M88EC018_EXT_PHY_SPEC_CTRL_DOWNSHIFT_COUNTER_MASK;
			phy_ext_spec_ctrl |= M88EC018_EXT_PHY_SPEC_CTRL_DOWNSHIFT_COUNTER_5X;
		} else {
			phy_ext_spec_ctrl &= ~(M88E1000_EXT_PHY_SPEC_CTRL_MASTER_DOWNSHIFT_MASK | M88E1000_EXT_PHY_SPEC_CTRL_SLAVE_DOWNSHIFT_MASK);
			phy_ext_spec_ctrl |= M88E1000_EXT_PHY_SPEC_CTRL_MASTER_DOWNSHIFT_1X | M88E1000_EXT_PHY_SPEC_CTRL_SLAVE_DOWNSHIFT_1X;
		}
		if (IPG_WRITE_PHY(ipg, M88E1000_EXT_PHY_SPEC_CTRL, phy_ext_spec_ctrl)) return;
	}
	IPG_PHY_RESET(ipg, syn);
}

static void GGP_PHY_COPPER_SETUP(IPG *ipg, int syn)
{
	__u16 mac_spec;
	__u16 spec, spec_2;
	__u16 in_b, pwr, km;
	__u32 ctrl_ext;
	if (IPG_READ_PHY(ipg, GG82563_PHY_MAC_SPEC_CTRL, &mac_spec)) return;
	mac_spec |= GG82563_PHY_MAC_SPEC_CTRL_ASSERT_CRS_ON_TX | GG82563_PHY_MAC_SPEC_CTRL_TX_CLK_1000MBPS_25MHZ;
	if (IPG_WRITE_PHY(ipg, GG82563_PHY_MAC_SPEC_CTRL, mac_spec)) return;
	if (IPG_READ_PHY(ipg, GG82563_PHY_SPEC_CTRL, &spec)) return;
	spec |= GG82563_PHY_SPEC_CTRL_CROSSOVER_MODE_AUTO;
	spec &= ~GG82563_PHY_SPEC_CTRL_POLARITY_REVERSAL_DISABLE;
	if (IPG_WRITE_PHY(ipg, GG82563_PHY_SPEC_CTRL, spec)) return;
	IPG_PHY_RESET(ipg, syn);
	if (ipg->mac_type == MAC_80003es2lan) {
		if (IPG_WRITE_KMRN(ipg, KUMCTRLSTA_OFFSET_FIFO_CTRL, KUMCTRLSTA_FIFO_CTRL_RX_BYPASS | KUMCTRLSTA_FIFO_CTRL_TX_BYPASS)) return;
		if (IPG_READ_PHY(ipg, GG82563_PHY_SPEC_CTRL, &spec_2)) return;
		spec_2 &= ~GG82563_PHY_SPEC_CTRL_2_REVERSE_AUTO_NEG;
		if (IPG_WRITE_PHY(ipg, GG82563_PHY_SPEC_CTRL, spec_2)) return;
		ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		ctrl_ext &= ~IPG_CTRL_EXT_LINK_MODE;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
		if (IPG_READ_PHY(ipg, GG82563_PHY_PWR_MGMT_CTRL, &pwr)) return;
		if (IPG_CHECK_IAMT(ipg)) {
			pwr |= GG82563_PHY_PWR_MGMT_CTRL_ENABLE_ELECTRICAL_IDLE;
			if (IPG_WRITE_PHY(ipg, GG82563_PHY_PWR_MGMT_CTRL, pwr)) return;
			if (IPG_READ_PHY(ipg, GG82563_PHY_KMRN_MODE_CTRL, &km)) return;
			km &= ~GG82563_PHY_KMRN_MODE_CTRL_PASS_FALSE_CARRIER;
			if (IPG_WRITE_PHY(ipg, GG82563_PHY_KMRN_MODE_CTRL, km)) return;
		}
		if (IPG_READ_PHY(ipg, GG82563_PHY_INBAND_CTRL, &in_b)) return;
		in_b |= KUMCTRLSTA_INB_CTRL_DIS_PADDING;
		if (IPG_WRITE_PHY(ipg, GG82563_PHY_INBAND_CTRL, in_b)) return;
	}
}

static void IPG_SETUP_FIBER_SERDES_LINK(IPG *ipg, int syn)
{
	__u32 ctrl, txcw;
	if (ipg->mac_type == MAC_82571 || ipg->mac_type == MAC_82572)
		IPG_WRITE(ipg, IPG_SCTL, IPG_SCTL_DISABLE_LOOPBACK);
	ctrl = IPG_READ(ipg, IPG_CTRL);
	ctrl &= ~IPG_CTRL_LRST;
	if ((ipg->mac_type == MAC_82545_rev_3 || ipg->mac_type == MAC_82546_rev_3)) {
		__u16 gen_ctrl;
		if (ipg->media == MEDIA_SERDES) {
			/* e1000_adjust_serdes_amplitude */
			__u16 ee_ampl;
			if (IPG_READ_EEPROM(ipg, EEPROM_SERDES_AMPLITUDE, 1, &ee_ampl) || ee_ampl == EEPROM_RESERVED_WORD) goto skip_ampl;
			ee_ampl &= EEPROM_SERDES_AMPLITUDE_MASK;
			IPG_WRITE_PHY(ipg, M88E1000_PHY_EXT_CTRL, ee_ampl);
			skip_ampl:;
		}
		/* e1000_set_vco_speed */
		if (IPG_READ_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x5 << 5), &gen_ctrl)) goto skip_vco;
		gen_ctrl &= ~0x100;
		if (IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x5 << 5), gen_ctrl)) goto skip_vco;
		if (IPG_READ_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x4 << 5), &gen_ctrl)) goto skip_vco;
		gen_ctrl |= 0x00000800;
		if (IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x4 << 5), gen_ctrl)) goto skip_vco;
		skip_vco:;
	}
	IPG_SET_COLLISION_DIST(ipg);
	txcw = IPG_TXCW_ANE | IPG_TXCW_FD;
	if (ipg->flow_control == FC_TX_PAUSE) txcw |= IPG_TXCW_ASM_DIR;
	if (ipg->flow_control == FC_RX_PAUSE || ipg->flow_control == FC_FULL) txcw |= IPG_TXCW_PAUSE_MASK;
	IPG_WRITE(ipg, IPG_TXCW, txcw);
	IPG_WRITE(ipg, IPG_CTRL, ctrl);
	MSLEEP(syn, 1);
	/* don't check for link here, should be done in the timer */
}

int IPG_POLARITY_REVERSAL_WORKAROUND(IPG *ipg, int syn)
{
	unsigned i;
	int r;
	if (__likely(ipg->link_mode & (LINK_100 | LINK_AUTO))) return 0;
	if (__likely(ipg->mac_type != MAC_82544) && __likely(ipg->mac_type != MAC_82543)) return 0;
	if (__unlikely(r = IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x19 << 5), 0xffff))) return r;
	for (i = 0; i < 20; i++) {
		__u16 mii_st;
/* should it be really read twice or did person with compulsive disorder write it */
		if (__unlikely(r = IPG_READ_PHY(ipg, PHY_STATUS, &mii_st))) return r;
		if (__unlikely(r = IPG_READ_PHY(ipg, PHY_STATUS, &mii_st))) return r;
		if (!(mii_st & MII_SR_LINK_STATUS)) break;
		MSLEEP(syn, 100);
	}
	MSLEEP(syn, 200);
	if (__unlikely(r = IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x19 << 5), 0xfff0))) return r;
	MSLEEP(syn, 50);
	if (__unlikely(r = IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x19 << 5), 0xff00))) return r;
	MSLEEP(syn, 50);
	if (__unlikely(r = IPG_WRITE_PHY(ipg, M88E1000_PHY_GEN_CONTROL | (0x19 << 5), 0x0000))) return r;
	for (i = 0; i < 20; i++) {
		__u16 mii_st;
		if (__unlikely(r = IPG_READ_PHY(ipg, PHY_STATUS, &mii_st))) return r;
		if (__unlikely(r = IPG_READ_PHY(ipg, PHY_STATUS, &mii_st))) return r;
		if (mii_st & MII_SR_LINK_STATUS) break;
		MSLEEP(syn, 100);
	}
	return 1;
}

void IPG_SET_COLLISION_DIST(IPG *ipg)
{
	__u32 tctl;
	tctl = IPG_READ(ipg, IPG_TCTL);
	tctl &= ~IPG_TCTL_COLD;
	if (ipg->mac_type < MAC_82543) tctl |= 64 << __BSF_CONST(IPG_TCTL_COLD);
	else tctl |= 63 << __BSF_CONST(IPG_TCTL_COLD);
	IPG_WRITE(ipg, IPG_TCTL, tctl);
	IPG_FLUSH(ipg);
}

static void IPG_CONFIG_MAC_TO_PHY(IPG *ipg)
{
	__u32 ctrl;
	__u16 phy_status;
	if (__likely(ipg->mac_type >= MAC_82544))
		goto cd;
	ctrl = IPG_READ(ipg, IPG_CTRL);
	ctrl |= IPG_CTRL_FRCSPD | IPG_CTRL_FRCDPLX;
	ctrl &= ~(IPG_CTRL_ILOS | IPG_CTRL_SPEED | IPG_CTRL_FD);
	if (IPG_READ_PHY(ipg, M88E1000_PHY_SPEC_STATUS, &phy_status)) return;
	if (phy_status & M88E1000_PHY_SPEC_STATUS_DPLX) ctrl |= IPG_CTRL_FD;
	if ((phy_status & M88E1000_PHY_SPEC_STATUS_SPEED) == M88E1000_PHY_SPEC_STATUS_1000MBS) ctrl |= IPG_CTRL_SPEED_1000;
	if ((phy_status & M88E1000_PHY_SPEC_STATUS_SPEED) == M88E1000_PHY_SPEC_STATUS_100MBS) ctrl |= IPG_CTRL_SPEED_100;
	if ((phy_status & M88E1000_PHY_SPEC_STATUS_SPEED) == M88E1000_PHY_SPEC_STATUS_10MBS) ctrl |= IPG_CTRL_SPEED_10;
	IPG_WRITE(ipg, IPG_CTRL, ctrl);
	cd:
	IPG_SET_COLLISION_DIST(ipg);
}

static void IPG_SET_FC(IPG *ipg, unsigned fc)
{
	__u32 ctrl = IPG_READ(ipg, IPG_CTRL);
	ctrl &= ~(IPG_CTRL_RFCE | IPG_CTRL_TFCE);
	if (fc & FC_RX_PAUSE) ctrl |= IPG_CTRL_RFCE;
	if (fc & FC_TX_PAUSE) ctrl |= IPG_CTRL_TFCE;
	if (ipg->mac_type == MAC_82542_rev2_0) ctrl &= ~IPG_CTRL_TFCE;
	IPG_WRITE(ipg, IPG_CTRL, ctrl);
}

static void IPG_CONFIGURE_FC(IPG *ipg)
{
	__u16 status, adv, lp;
	unsigned pause;
	if (ipg->media == MEDIA_FIBER || ipg->media == MEDIA_SERDES || (ipg->media == MEDIA_COPPER && ipg->link_mode != LINK_AUTO)) {
		dflt:
		IPG_SET_FC(ipg, ipg->flow_control);
		return;
	}
	if (IPG_READ_PHY(ipg, PHY_STATUS, &status)) goto dflt;
	if (IPG_READ_PHY(ipg, PHY_STATUS, &status)) goto dflt;
	if (!(status & MII_SR_AUTONEG_COMPLETE)) goto dflt;
	if (IPG_READ_PHY(ipg, PHY_AUTONEG_ADV, &adv)) goto dflt;
	if (IPG_READ_PHY(ipg, PHY_LP_ABILITY, &lp)) goto dflt;
	if ((adv & PHY_AUTONEG_ADV_PAUSE) &&
	    (lp & PHY_LP_ABILITY_PAUSE)) {
		pause = FC_TX_PAUSE | FC_RX_PAUSE;
	} else if (!(adv & PHY_AUTONEG_ADV_PAUSE) &&
	            (adv & PHY_AUTONEG_ADV_ASM_DIR) &&
		    (lp & PHY_LP_ABILITY_PAUSE) &&
		    (lp & PHY_LP_ABILITY_ASM_DIR)) {
		pause = FC_TX_PAUSE;
	} else if ((adv & PHY_AUTONEG_ADV_PAUSE) &&
		   (adv & PHY_AUTONEG_ADV_ASM_DIR) &&
		  !(lp & PHY_LP_ABILITY_PAUSE) &&
		   (lp & PHY_LP_ABILITY_ASM_DIR)) {
		pause = FC_RX_PAUSE;
	} else {
		pause = FC_RX_PAUSE;
	}
	if (!(ipg->flow_control & FC_TX_PAUSE)) pause &= ~FC_TX_PAUSE;
	if (ipg->link_state.flags & LINK_STATE_HALF_DUPLEX) pause = 0;
	IPG_SET_FC(ipg, pause);
}

static void IPG_CONFIGURE_KMRN(IPG *ipg)
{
	int giga = ipg->link_state.speed >= 1000000000;
	__u16 mode_ctrl;
	__u32 tipg;
	IPG_WRITE_KMRN(ipg, KUMCTRLSTA_OFFSET_HD_CTRL, !giga ? KUMCTRLSTA_OFFSET_HD_CTRL_10_100 : KUMCTRLSTA_OFFSET_HD_CTRL_1000);
	tipg = IPG_READ(ipg, IPG_TIPG);
	tipg &= ~IPG_TIPG_IPGT;
	tipg |= giga ? IPG_TIPG_IPGT_DEFAULT_80003ES2LAN_1000 : IPG_TIPG_IPGT_DEFAULT_80003ES2LAN_10_100;
	IPG_WRITE(ipg, IPG_TIPG, tipg);
	if (!IPG_READ_PHY(ipg, GG82563_PHY_KMRN_MODE_CTRL, &mode_ctrl)) {
		if (!giga && !(ipg->link_state.flags & LINK_STATE_HALF_DUPLEX))
			mode_ctrl |= GG82563_PHY_KMRN_MODE_CTRL_PASS_FALSE_CARRIER;
		else
			mode_ctrl &= ~GG82563_PHY_KMRN_MODE_CTRL_PASS_FALSE_CARRIER;
		IPG_WRITE_PHY(ipg, GG82563_PHY_KMRN_MODE_CTRL, mode_ctrl);
	}
}

static void IPG_KMRN_LOCK_LOSS_WORKAROUND(IPG *ipg)
{
	__u16 status;
	__u32 phy_ctrl;
	int i;
	if (IPG_READ_PHY(ipg, PHY_STATUS, &status)) return;
	if (IPG_READ_PHY(ipg, PHY_STATUS, &status)) return;
	if (!(status & MII_SR_LINK_STATUS)) return;
	for (i = 0;; i++) {
		__u16 kmrn_diag;
		if (IPG_READ_PHY(ipg, IGP3_KMRN_DIAG, &kmrn_diag)) return;
		if (IPG_READ_PHY(ipg, IGP3_KMRN_DIAG, &kmrn_diag)) return;
		if (!(kmrn_diag & IGP3_KMRN_DIAG_PCS_LOCK_LOSS)) return;
		if (i == 9) break;
		IPG_PHY_HW_RESET(ipg, 0);
		KERNEL$UDELAY(5000);
	}
	phy_ctrl = IPG_READ(ipg, IPG_PHY_CTRL);
	phy_ctrl |= IPG_PHY_CTRL_GBE_DISABLE | IPG_PHY_CTRL_NOND0A_GBE_DISABLE;
	IPG_WRITE(ipg, IPG_PHY_CTRL, phy_ctrl);
}

void IPG_LINK_CHANGED(IPG *ipg, int link)
{
	LINK_STATE_PHYS ls;
	memset(&ls, 0, sizeof ls);
	if (__unlikely(!link)) goto set_link;
	IPG_CONFIG_MAC_TO_PHY(ipg);
	ls.flags |= LINK_STATE_UP;
	if (ipg->mac_type >= MAC_82543) {
		__u32 status = IPG_READ(ipg, IPG_STATUS);
		if (status & IPG_STATUS_SPEED_1000) {
			ls.speed = 1000000000;
			strlcpy(ls.desc, "1000BASE", sizeof ls.desc);
		} else if (status & IPG_STATUS_SPEED_100) {
			ls.speed = 100000000;
			strlcpy(ls.desc, "100BASE", sizeof ls);
		} else {
			ls.speed = 10000000;
			strlcpy(ls.desc, "10BASE", sizeof ls);
		}
		if (status & IPG_STATUS_FD) {
			ls.flags |= LINK_STATE_FULL_DUPLEX;
		} else {
			ls.flags |= LINK_STATE_HALF_DUPLEX;
		}
	} else {
		ls.speed = 1000000000;
		strlcpy(ls.desc, "1000BASE", sizeof ls.desc);
		ls.flags |= LINK_STATE_FULL_DUPLEX;
	}
	if (ipg->media == MEDIA_COPPER) {
		if (ls.speed == 100000000) strlcat(ls.desc, "-TX", sizeof ls.desc);
		else strlcat(ls.desc, "-T", sizeof ls.desc);
	}
	if (ipg->phy.type == PHY_IGP && ls.speed <= 100000000) {
		__u16 val;
		if (IPG_READ_PHY(ipg, PHY_AUTONEG_EXP, &val)) goto done_phy;
		if (!(val & PHY_AUTONEG_EXP_LP_NWAY_CAPS)) {
			hd:
			ls.flags &= ~LINK_STATE_FULL_DUPLEX;
			ls.flags |= LINK_STATE_HALF_DUPLEX;
			goto done_phy;
		}
		if (IPG_READ_PHY(ipg, PHY_LP_ABILITY, &val)) goto done_phy;
		if (ls.speed == 100000000 && !(val & PHY_AUTONEG_ADV_100TX_FD_CAPS)) goto hd;
		if (ls.speed == 10000000 && !(val & PHY_AUTONEG_ADV_10T_FD_CAPS)) goto hd;
	}
	done_phy:
	if (ipg->media == MEDIA_COPPER) {
		if (ipg->link_mode == LINK_AUTO) {
			__u16 status;
			IPG_READ_PHY(ipg, PHY_STATUS, &status);
			if (!IPG_READ_PHY(ipg, PHY_STATUS, &status)) {
				if (!(status & MII_SR_AUTONEG_COMPLETE)) {
					ls.flags |= LINK_STATE_AUTO_NEGOTIATION_FAILED;
				} else {
					ls.flags |= LINK_STATE_AUTO_NEGOTIATION;
				}
			}
		}
	} else {
		if (IPG_READ(ipg, IPG_TXCW) & IPG_TXCW_ANE)
			ls.flags |= LINK_STATE_AUTO_NEGOTIATION;
		else
			ls.flags |= LINK_STATE_AUTO_NEGOTIATION_FAILED;
	}
	set_link:
	if (memcmp(&ipg->link_state, &ls, sizeof(LINK_STATE_PHYS))) {
		memcpy(&ipg->link_state, &ls, sizeof(LINK_STATE_PHYS));
		WQ_WAKE_ALL(&ipg->link_state_wait);
		IPG_SET_ITR(ipg, ls.speed >= 1000000000 ? ipg->ips : ipg->ips_100);
	}
	if (!link) {
		return;
	}
	if (ipg->mac_type == MAC_80003es2lan && ipg->media == MEDIA_COPPER) {
		IPG_CONFIGURE_KMRN(ipg);
	}
	IPG_CONFIGURE_FC(ipg);
	if (ipg->phy.type == PHY_IGP_3 && ls.speed >= 1000000000) {
		IPG_KMRN_LOCK_LOSS_WORKAROUND(ipg);
	}
}
