#include <SPAD/SYSLOG.H>

#include "IPGREG.H"
#include "IPG.H"

int IPG_CHECK_PHY_RESET_BLOCK(IPG *ipg)
{
	if (ipg->mac_type <= MAC_82547_rev_2) {
		return 0;
	}
	if (ipg->mac_type == MAC_ich8lan) {
		return IPG_READ(ipg, IPG_FWSM) & IPG_FWSM_RSPCIPHY ? 0 : -EBUSY;
	}
	return IPG_READ(ipg, IPG_MANC) & IPG_MANC_BLK_PHY_RST_ON_IDE ? -EBUSY : 0;
}

static void IPG_MDI_CLK(IPG *ipg, __u32 *ctrl, int rais)
{
	*ctrl = (*ctrl & ~IPG_CTRL_MDC) | (rais * IPG_CTRL_MDC);
	IPG_WRITE(ipg, IPG_CTRL, *ctrl);
	IPG_FLUSH(ipg);
	KERNEL$UDELAY(10);
}

static __u32 IPG_SHIFT_IN_MDI_BITS(IPG *ipg, unsigned count)
{
	__u32 data;
	__u32 ctrl = IPG_READ(ipg, IPG_CTRL);
	ctrl &= ~(IPG_CTRL_MDIO_DIR | IPG_CTRL_MDIO);
	IPG_WRITE(ipg, IPG_CTRL, ctrl);
	IPG_FLUSH(ipg);
	IPG_MDI_CLK(ipg, &ctrl, 1);
	IPG_MDI_CLK(ipg, &ctrl, 0);
	data = 0;
	do {
		IPG_MDI_CLK(ipg, &ctrl, 1);
		ctrl = IPG_READ(ipg, IPG_CTRL);
		data <<= 1;
		data |= !!(ctrl & IPG_CTRL_MDIO);
		IPG_MDI_CLK(ipg, &ctrl, 0);
	} while (--count);
	IPG_MDI_CLK(ipg, &ctrl, 1);
	IPG_MDI_CLK(ipg, &ctrl, 0);
	return data;
}

static void IPG_SHIFT_OUT_MDI_BITS(IPG *ipg, __u32 data, unsigned count)
{
	__u32 mask = 1 << (count - 1);
	__u32 ctrl = IPG_READ(ipg, IPG_CTRL);
	ctrl |= IPG_CTRL_MDIO_DIR | IPG_CTRL_MDC_DIR;
	do {
		ctrl &= ~IPG_CTRL_MDIO;
		if (data & mask) ctrl |= IPG_CTRL_MDIO;
		IPG_WRITE(ipg, IPG_CTRL, ctrl);
		IPG_FLUSH(ipg);
		KERNEL$UDELAY(10);
		IPG_MDI_CLK(ipg, &ctrl, 1);
		IPG_MDI_CLK(ipg, &ctrl, 0);
	} while (mask >>= 1);
}

static int IPG_READ_PHY_INTERNAL(IPG *ipg, unsigned addr, __u16 *data)
{
	const unsigned phy_addr = 1;
	if (ipg->mac_type >= MAC_82544) {
		int i;
		__u32 mdic = ((addr << __BSF_CONST(IPG_MDIC_REGADD)) | (phy_addr << __BSF_CONST(IPG_MDIC_PHYADD)) | IPG_MDIC_OP_READ);
		IPG_WRITE(ipg, IPG_MDIC, mdic);
		for (i = 0; i < 100; i++) {
			KERNEL$UDELAY(50);
			mdic = IPG_READ(ipg, IPG_MDIC);
			if (mdic & IPG_MDIC_R) goto read_mdi;
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "MDI READ TIMEOUT, ADDR %X", addr);
		return -ETIMEDOUT;
		read_mdi:
		if (__unlikely(mdic & IPG_MDIC_E)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "MDI READ ERROR, ADDR %X, MDIC %08X", addr, mdic);
			return -EIO;
		}
		*data = mdic;
		return 0;
	} else {
		__u32 mdic;
		IPG_SHIFT_OUT_MDI_BITS(ipg, ~0, 32);
		mdic = (addr | (phy_addr << 5) | (2 << 10) | (1 << 12));
		IPG_SHIFT_OUT_MDI_BITS(ipg, mdic, 14);
		*data = IPG_SHIFT_IN_MDI_BITS(ipg, 16);
		return 0;
	}
}

static int IPG_WRITE_PHY_INTERNAL(IPG *ipg, unsigned addr, __u16 data)
{
	const unsigned phy_addr = 1;
	if (ipg->mac_type >= MAC_82544) {
		int i;
		__u32 mdic = (data | (addr << __BSF_CONST(IPG_MDIC_REGADD)) | (phy_addr << __BSF_CONST(IPG_MDIC_PHYADD)) | IPG_MDIC_OP_WRITE);
		IPG_WRITE(ipg, IPG_MDIC, mdic);
		for (i = 0; i < 1000; i++) {
			KERNEL$UDELAY(5);
			mdic = IPG_READ(ipg, IPG_MDIC);
			if (mdic & IPG_MDIC_R) goto wrote_mdi;
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "MDI WRITE TIMEOUT, ADDR %X", addr);
		return -ETIMEDOUT;
		wrote_mdi:
		return 0;
	} else {
		__u32 mdic;
		IPG_SHIFT_OUT_MDI_BITS(ipg, ~0, 32);
		mdic = (data | (2 << 16) | (addr << 18) | (phy_addr << 23) | (1 << 28) | (1 << 30));
		IPG_SHIFT_OUT_MDI_BITS(ipg, mdic, 32);
		return 0;
	}
}

static int ACQUIRE_PHY_LOCK(IPG *ipg, __u16 *swfw)
{
	if (ipg->mac_type == MAC_80003es2lan && IPG_READ(ipg, IPG_STATUS) & IPG_STATUS_FUNCTION_1) *swfw = IPG_SW_FW_SYNC_PHY1_SM;
	else *swfw = IPG_SW_FW_SYNC_PHY0_SM;
	return IPG_SWFW_SYNC_ACQUIRE(ipg, *swfw);
}

static int IPG_PHY_ACCESS(IPG *ipg, unsigned addr, __u16 *data, int w)
{
	__u16 old_page;
	int r;
	__u16 swfw;
	if (__unlikely(r = ACQUIRE_PHY_LOCK(ipg, &swfw))) return r;
	if ((ipg->phy.type == PHY_IGP || ipg->phy.type == PHY_IGP_2 || ipg->phy.type == PHY_IGP_3) && __unlikely(addr > MAX_PHY_MULTI_PAGE_REG)) {
		if (__unlikely(r = IPG_WRITE_PHY_INTERNAL(ipg, IGP01E1000_PHY_PAGE_SELECT, addr))) goto ret;
	} else if (ipg->phy.type == PHY_GG82563) {
		if ((addr & MAX_PHY_REG_ADDRESS) < GG82563_MIN_ALT_REG) {
			if (__unlikely(r = IPG_WRITE_PHY_INTERNAL(ipg, GG82563_PHY_PAGE_SELECT, addr >> 5))) goto ret;
		} else {
			if (__unlikely(r = IPG_WRITE_PHY_INTERNAL(ipg, GG82563_PHY_PAGE_SELECT_ALT, addr >> 5))) goto ret;
		}
	} else if (ipg->phy.type == PHY_M88) {
		if (__unlikely((addr & MAX_PHY_REG_ADDRESS) == M88E1000_PHY_GEN_CONTROL)) {
			if (__unlikely(r = IPG_READ_PHY_INTERNAL(ipg, M88E1000_PHY_PAGE_SELECT, &old_page))) goto ret;
			if (__unlikely(r = IPG_WRITE_PHY_INTERNAL(ipg, M88E1000_PHY_PAGE_SELECT, addr >> 5))) goto ret;
		}
	}
	if (!w) r = IPG_READ_PHY_INTERNAL(ipg, addr & MAX_PHY_REG_ADDRESS, data);
	else r = IPG_WRITE_PHY_INTERNAL(ipg, addr & MAX_PHY_REG_ADDRESS, *data);
	if (ipg->phy.type == PHY_M88) {
		if ((addr & MAX_PHY_REG_ADDRESS) == M88E1000_PHY_GEN_CONTROL) {
			/* I'm not sure if returning old page is necessary,
			   Intel driver does it inconsistently (sometimes not,
			   sometimes returns zero, sometimes old value */
			IPG_WRITE_PHY_INTERNAL(ipg, M88E1000_PHY_PAGE_SELECT, old_page);
		}
	}
	ret:
	IPG_SWFW_SYNC_RELEASE(ipg, swfw);
	return r;
}

int IPG_READ_PHY(IPG *ipg, unsigned addr, __u16 *data)
{
	return IPG_PHY_ACCESS(ipg, addr, data, 0);
}

int IPG_WRITE_PHY(IPG *ipg, unsigned addr, __u16 data)
{
	return IPG_PHY_ACCESS(ipg, addr, &data, 1);
}

void IPG_PHY_DETECT(IPG *ipg)
{
	__u16 id;
	if (ipg->media != MEDIA_COPPER) return;
	if (ipg->phy.type) return;
	if (ipg->mac_type == MAC_82571 || ipg->mac_type == MAC_82572) {
		ipg->phy.id = IGP01E1000_I_PHY_ID;
		ipg->phy.type = PHY_IGP_2;
		return;
	}
	if (ipg->mac_type == MAC_80003es2lan) ipg->phy.type = PHY_GG82563;
	/*{
		int i;
		for (i = 0; i < 0x20; i++) {
			IPG_READ_PHY(ipg, i, &id);
			__debug_printf("%02x: %04x\n", i, id);
		}
	}*/
	if (IPG_READ_PHY(ipg, PHY_ID1, &id)) goto phy_err;
	ipg->phy.id = id << 16;
	KERNEL$UDELAY(20);
	if (IPG_READ_PHY(ipg, PHY_ID2, &id)) goto phy_err;
	ipg->phy.id |= id & ~0xF;
	ipg->phy.revision = id & 0xF;
	switch (ipg->mac_type) {
		case MAC_82543:
			if (ipg->phy.id != M88E1000_E_PHY_ID) goto invl_phy;
			break;
		case MAC_82544:
			if (ipg->phy.id != M88E1000_I_PHY_ID) goto invl_phy;
			break;
		case MAC_82540:
		case MAC_82545:
		case MAC_82545_rev_3:
		case MAC_82546:
		case MAC_82546_rev_3:
			if (ipg->phy.id != M88E1011_I_PHY_ID) goto invl_phy;
			break;
		case MAC_82541:
		case MAC_82541_rev_2:
		case MAC_82547:
		case MAC_82547_rev_2:
			if (ipg->phy.id != IGP01E1000_I_PHY_ID) goto invl_phy;
			break;
		case MAC_82573:
			if (ipg->phy.id != M88E1111_I_PHY_ID) goto invl_phy;
			break;
		case MAC_80003es2lan:
			if (ipg->phy.id != GG82563_E_PHY_ID) goto invl_phy;
			break;
		case MAC_ich8lan:
			if (ipg->phy.id != IGP03E1000_E_PHY_ID &&
			    ipg->phy.id != IFE_E_PHY_ID &&
			    ipg->phy.id != IFE_PLUS_E_PHY_ID &&
			    ipg->phy.id != IFE_C_E_PHY_ID) goto invl_phy;
			break;
		default:
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ipg->dev_name, "MAC TYPE %s DOESN'T SUPPORT COPPER", ipg->chip_name);
			goto phy_err;
	}
	switch (ipg->phy.id) {
		case M88E1000_E_PHY_ID:
		case M88E1000_I_PHY_ID:
		case M88E1011_I_PHY_ID:
		case M88E1111_I_PHY_ID:
			ipg->phy.type = PHY_M88;
			break;
		case IGP01E1000_I_PHY_ID:
			ipg->phy.type = PHY_IGP;
			break;
		case IGP03E1000_E_PHY_ID:
			ipg->phy.type = PHY_IGP_3;
			break;
		case IFE_E_PHY_ID:
		case IFE_PLUS_E_PHY_ID:
		case IFE_C_E_PHY_ID:
			ipg->phy.type = PHY_IFE;
			break;
		case GG82563_E_PHY_ID:
			ipg->phy.type = PHY_GG82563;
			break;
	}
	return;

	invl_phy:
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "INVALID PHY ID %08X, REV %01X, MAC TYPE %s", ipg->phy.id, ipg->phy.revision, ipg->chip_name);
	phy_err:
	ipg->phy.type = 0;
}

void IPG_PHY_INIT_SCRIPT(IPG *ipg, int syn)
{
	char saved;
	__u16 phy_saved_data;
	if (!(ipg->mac_type == MAC_82541 || ipg->mac_type == MAC_82541_rev_2 || ipg->mac_type == MAC_82547 || ipg->mac_type == MAC_82547_rev_2)) return;
	MSLEEP(syn, 20);
	saved = !IPG_READ_PHY(ipg, 0x2f5b, &phy_saved_data);
	IPG_WRITE_PHY(ipg, 0x2f5b, 0x0003);
	MSLEEP(syn, 20);
	IPG_WRITE_PHY(ipg, 0x0000, 0x0140);
	MSLEEP(syn, 5);
	if (ipg->mac_type == MAC_82541 || ipg->mac_type == MAC_82547) {
		IPG_WRITE_PHY(ipg, 0x1f95, 0x0001);
		IPG_WRITE_PHY(ipg, 0x1f71, 0xbd21);
		IPG_WRITE_PHY(ipg, 0x1f79, 0x0018);
		IPG_WRITE_PHY(ipg, 0x1f30, 0x1600);
		IPG_WRITE_PHY(ipg, 0x1f31, 0x0014);
		IPG_WRITE_PHY(ipg, 0x1f32, 0x161c);
		IPG_WRITE_PHY(ipg, 0x1f94, 0x0003);
		IPG_WRITE_PHY(ipg, 0x1f96, 0x003f);
		IPG_WRITE_PHY(ipg, 0x2010, 0x0008);
	} else {
		IPG_WRITE_PHY(ipg, 0x1f73, 0x0099);
	}
	IPG_WRITE_PHY(ipg, 0x0000, 0x3300);
	MSLEEP(syn, 20);
	if (saved) IPG_WRITE_PHY(ipg, 0x2f5b, phy_saved_data);
	if (ipg->mac_type == MAC_82547) {
		__u16 fused, fine, coarse;
		if (!IPG_READ_PHY(ipg, 0x20D1, &fused)) return;
		if (!(fused & 0x0100)) {
			if (!IPG_READ_PHY(ipg, 0x20D0, &fused)) return;
			fine = fused & 0x0f80;
			coarse = fused & 0x0070;
			if (coarse > 0x0040) {
				coarse -= 0x0010;
				fine = fine - 0x0080;
			} else if (coarse == 0x0040) {
				fine = fine - 0x0500;
			}
			fused = (fused & 0xf000) | (fine & 0x0f80) | (coarse & 0x0070);
			IPG_WRITE_PHY(ipg, 0x20dc, fused);
			IPG_WRITE_PHY(ipg, 0x20de, 0x0002);
		}
	}
}

int IPG_PHY_HW_RESET(IPG *ipg, int syn)
{
	int r;
	unsigned timeout;
	__u32 cfg_mask;
	if (IPG_CHECK_PHY_RESET_BLOCK(ipg)) return 0;
	if (ipg->mac_type >= MAC_82544) {
		__u32 ctrl;
		__u16 swfw;
		if (__unlikely(r = ACQUIRE_PHY_LOCK(ipg, &swfw))) return r;
		ctrl = IPG_READ(ipg, IPG_CTRL);
		IPG_WRITE(ipg, IPG_CTRL, ctrl | IPG_CTRL_PHY_RST);
		IPG_FLUSH(ipg);
		if (ipg->mac_type < MAC_82571) {
			MSLEEP(syn, 10);
		} else {
			KERNEL$UDELAY(100);
		}
		IPG_WRITE(ipg, IPG_CTRL, ctrl);
		IPG_FLUSH(ipg);
		if (ipg->mac_type >= MAC_82571) {
			MSLEEP(syn, 10);
		}
		IPG_SWFW_SYNC_RELEASE(ipg, swfw);
	} else {
		__u32 ctrl_ext = IPG_READ(ipg, IPG_CTRL_EXT);
		ctrl_ext |= IPG_CTRL_EXT_SDP4_IODIR;
		ctrl_ext &= ~IPG_CTRL_EXT_SDP4_DATA;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
		IPG_FLUSH(ipg);
		MSLEEP(syn, 10);
		ctrl_ext |= IPG_CTRL_EXT_SDP4_DATA;
		IPG_WRITE(ipg, IPG_CTRL_EXT, ctrl_ext);
		IPG_FLUSH(ipg);
	}
	KERNEL$UDELAY(150);
	if (ipg->mac_type == MAC_82541 || ipg->mac_type == MAC_82547) {
		__u32 led_ctl = IPG_READ(ipg, IPG_LEDCTL);
		led_ctl &= ~IPG_LEDCTL_LED1_MOD;
		led_ctl |= 3 << __BSF_CONST(IPG_LEDCTL_LED1_MOD);
		IPG_WRITE(ipg, IPG_LEDCTL, led_ctl);
	}

	/* e1000_get_phy_cfg_done */
	timeout = 100;
	cfg_mask = IPG_EEMNGCTL_CFG_DONE;
	switch (ipg->mac_type) {
		case MAC_80003es2lan:
			if (IPG_READ(ipg, IPG_STATUS) & IPG_STATUS_FUNCTION_1)
				cfg_mask = IPG_EEMNGCTL_CFG_DONE_PORT_1;
		case MAC_82571:
		case MAC_82572:
			while (1) {
				if (IPG_READ(ipg, IPG_EEMNGCTL) & cfg_mask)
					break;
				if (!--timeout) {
					KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ipg->dev_name, "MNG CONFIGURATION HAS NOT COMPLETED");
					break;
				}
			}
			break;
		default:
			MSLEEP(syn, 10);
	}

	if (ipg->mac_type == MAC_80003es2lan) {
		__u32 swsm = IPG_READ(ipg, IPG_SWSM);
		swsm &= ~IPG_SWSM_SMBI;
		IPG_WRITE(ipg, IPG_SWSM, swsm);
	}

	/* e1000_init_lcd_from_nvm? --- we have no power transitions so
	   hopefully it is not needed */
	
	return 0;
}

int IPG_PHY_RESET(IPG *ipg, int syn)
{
	int r;
	if (IPG_CHECK_PHY_RESET_BLOCK(ipg)) return 0;
	if (ipg->phy.type == PHY_IGP ||
	    ipg->phy.type == PHY_IGP_2 ||
	    ipg->phy.type == PHY_IGP_3 ||
	    ipg->phy.type == PHY_IFE) {
		if ((r = IPG_PHY_HW_RESET(ipg, syn))) return r;
	} else {
		__u16 phy_ctrl;
		if ((r = IPG_READ_PHY(ipg, PHY_CTRL, &phy_ctrl))) return r;
		phy_ctrl |= PHY_CTRL_RESET;
		if ((r = IPG_WRITE_PHY(ipg, PHY_CTRL, phy_ctrl))) return r;
		KERNEL$UDELAY(1);
	}
	if (ipg->phy.type == PHY_IGP || ipg->phy.type == PHY_IGP_2) IPG_PHY_INIT_SCRIPT(ipg, syn);
	return 0;
}

int IPG_READ_KMRN(IPG *ipg, unsigned addr, __u16 *data)
{
	int r;
	__u16 swfw;
	if (__unlikely(r = ACQUIRE_PHY_LOCK(ipg, &swfw))) return r;
	IPG_WRITE(ipg, IPG_KUMCTRLSTA, (addr << __BSF_CONST(IPG_KUMCTRLSTA_OFFSET)) | IPG_KUMCTRLSTA_REN);
	KERNEL$UDELAY(2);
	*data = IPG_READ(ipg, IPG_KUMCTRLSTA);
	IPG_SWFW_SYNC_RELEASE(ipg, swfw);
	return 0;
}

int IPG_WRITE_KMRN(IPG *ipg, unsigned addr, __u16 data)
{
	int r;
	__u16 swfw;
	if (__unlikely(r = ACQUIRE_PHY_LOCK(ipg, &swfw))) return r;
	IPG_WRITE(ipg, IPG_KUMCTRLSTA, (addr << __BSF_CONST(IPG_KUMCTRLSTA_OFFSET)) | data);
	KERNEL$UDELAY(2);
	IPG_SWFW_SYNC_RELEASE(ipg, swfw);
	return 0;
}

int IPG_CHECK_IAMT(IPG *ipg)
{
	__u32 fwsm = IPG_READ(ipg, IPG_FWSM);
	return (fwsm & IPG_FWSM_MODE_MASK) == (ipg->mac_type >= MAC_ich8lan ? IPG_FWSM_MODE_IAMT_ICH : IPG_FWSM_MODE_IAMT);
}

void IPG_POWERUP_PHY(IPG *ipg)
{
	if (ipg->media == MEDIA_COPPER) {
		__u16 phy_ctrl;
		if (!IPG_READ_PHY(ipg, PHY_CTRL, &phy_ctrl))
			IPG_WRITE_PHY(ipg, PHY_CTRL, phy_ctrl & ~PHY_CTRL_POWER_DOWN);
	}
}

void IPG_POWERDOWN_PHY(IPG *ipg)
{
	if (ipg->mac_type >= MAC_82540 && ipg->media == MEDIA_COPPER) {
		__u16 phy_ctrl;
		if (ipg->mac_type < MAC_82571) {
			if (IPG_READ(ipg, IPG_MANC) & IPG_MANC_SMBUS_EN)
				return;
		} else {
			if (IPG_CHECK_IAMT(ipg))
				return;
			if (IPG_CHECK_PHY_RESET_BLOCK(ipg))
				return;
		}
		if ((IPG_READ_PHY(ipg, PHY_CTRL, &phy_ctrl))) return;
		phy_ctrl |= PHY_CTRL_POWER_DOWN;
		IPG_WRITE_PHY(ipg, PHY_CTRL, phy_ctrl);
	}
}
