#include <SPAD/LIBC.H>
#include <STDLIB.H>
#include <VALUES.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/THREAD.H>
#include <SPAD/SYSLOG.H>

#include <SPAD/PIO.H>
#include <SPAD/PKT.H>
#include <SPAD/ETHERNET.H>

#define FAST_TRIES			10
/* 3000 - 4000 --- severe problems with Linux (many timeouts)
   5000 - 9000 --- sporadic problems with Linux
   50000 --- problems with Linux with 100HZ timer and no IRQ
   75000 --- sporadic problems with Linux with 100HZ timer and no IRQ */
#define DEFAULT_NIBBLE_TIMEOUT		100000
#define DEFAULT_TRIGGER_TIMEOUT		100000
/* Experimentally obtained with Linux.
   With 19000, it gets timeouts.
   With 30000, it gets sporadic timeouts. */
#define DEFAULT_SCHEDULE_TIMEOUT	35000
/* 250000 gets "interrupt in error state" messages on Linux with 100HZ timer
   and no parport irq.
   350000 gets "interrupt in error state" when Linux is loaded */
#define DEFAULT_ERROR_TIMEOUT		500000
#define COLLISION_BACKOFF		200
#define TRANSMIT_BACKOFF		300
#define RECEIVE_LIVELOCK		25
#define TX_TRIGGER_FLIP			1000
#define HOLD_ON				20000

#define DEFAULT_PMAC_TIMEOUT		2000000
#define PMAC_INIT_WAIT			(JIFFIES_PER_SECOND * 3 + JIFFIES_PER_SECOND / 10)
#define PMAC_KNOCKDOWN_WAIT		(JIFFIES_PER_SECOND * 1)

#define PLIP_TIMER			(JIFFIES_PER_SECOND * 10)

#define PROTOCOL_PLIP			1
#define PROTOCOL_PMAC			2

typedef struct __plip PLIP;

struct __plip {
	struct pio_client client;
	
	IORQ event;

	int (*dispatcher)(PLIP *);

	unsigned tries;

	unsigned rx_count;
	unsigned rx_length;
	PACKET *rx_packet;

	unsigned tx_count;
	PACKET *tx_packet;

	unsigned char protocol;
	unsigned char plip_connected;

	unsigned short mtu;
	__u8 checksum;
	char send_timed;

	__u8 pmac_recv_byte;
	__u8 pmac_recv_state;
	__u8 pmac_send_byte;
	__u8 pmac_send_state;

#define PMAC_STATE_IDLE		1
#define PMAC_STATE_HEAD		2
#define PMAC_STATE_LEN_LO	3
#define PMAC_STATE_LEN_HI	4
#define PMAC_STATE_DATA		5

	int packet_input;
	int outstanding;
	MTX *irq_event;
	PKTQUEUE *queue;
	PKTPOOL pool;

	u_jiffies_lo_t schedule_timeout;

	unsigned nibble_timeout;
	unsigned trigger_timeout;

	unsigned receive_livelock;

	u_jiffies_lo_t error_time;
	u_jiffies_lo_t pmac_probe_period;
	u_jiffies_lo_t error_timeout;
	u_jiffies_lo_t pmac_timeout;
	char error_irq;

	char errorlevel;
	char terminate;
	char acquired;

	TIMER timer;

	LINK_STATE link_state;
	WQ link_state_wait;

	__u8 address[ETHERNET_ADDRLEN];
	char dev_name[__MAX_STR_LEN];
	char pio_name[__MAX_STR_LEN];
	void *lnte;
	void *dlrq;
};

#define WAIT_NONE		-1
#define WAIT_DEFAULT		-2
#define WAIT_IRQ		-3
#define WAIT_YIELD		-4
#define WAIT_RESET		-5

static IO_STUB plip_ioctl;
static IO_STUB plip_packet;

static const HANDLE_OPERATIONS PLIP_OPERATIONS = {
	SPL_X(SPL_PLIP),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	plip_ioctl,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	plip_packet,		/* pktio */
};

static int plip_dispatcher_receive(PLIP *plip);
static int plip_dispatcher_receive_length(PLIP *plip);
static int plip_dispatcher_receive_data(PLIP *plip);
static int plip_dispatcher_receive_data_dummy(PLIP *plip);
static int plip_dispatcher_send(PLIP *plip);
static void plip_set_nibble(PLIP *plip, __u8 data);
static int plip_dispatcher_send_data(PLIP *plip);
static AST_STUB packet_returned;
static int plip_hold_on(PLIP *plip);
static int plip_error(PLIP *plip);
static int plip_error_dispatcher(PLIP *plip);

static int pmac_dispatcher_connect(PLIP *plip);
static int pmac_dispatcher_connect_slave_1(PLIP *plip);
static int pmac_dispatcher_connect_slave_2(PLIP *plip);
static int pmac_dispatcher_connect_slave_3(PLIP *plip);
static int pmac_dispatcher_connect_slave_4(PLIP *plip);
static int pmac_dispatcher_connect_slave_wait(PLIP *plip);
static int pmac_dispatcher_slave_prepare_irq(PLIP *plip);
static int pmac_dispatcher_slave_prepare_irq_2(PLIP *plip);
static int pmac_dispatcher_slave_irq(PLIP *plip);
static int pmac_dispatcher_slave_ack_irq(PLIP *plip);
static int pmac_dispatcher_slave_hi(PLIP *plip);
static int pmac_dispatcher_slave_lo(PLIP *plip);
static int pmac_dispatcher_connect_master_1(PLIP *plip);
static int pmac_dispatcher_connect_master_2(PLIP *plip);
static int pmac_dispatcher_connect_master_3(PLIP *plip);
static int pmac_dispatcher_connect_master_wait(PLIP *plip);
static int pmac_dispatcher_master_prepare_irq(PLIP *plip);
static int pmac_dispatcher_master_irq(PLIP *plip);
static int pmac_dispatcher_master_hi(PLIP *plip);
static int pmac_dispatcher_master_lo(PLIP *plip);
static int pmac_received_byte(PLIP *plip);
static void pmac_byte_to_send(PLIP *plip);

static void plip_reset_timer(PLIP *plip, u_jiffies_lo_t t);
static void plip_get_link_state(PLIP *plip);

static void free_packets(PLIP *plip)
{
	if (__unlikely(plip->rx_packet != NULL)) {
		plip->rx_packet->status = -EIO;
		CALL_PKT(plip->rx_packet);
		plip->rx_packet = NULL;
	}
	if (__unlikely(plip->tx_packet != NULL)) {
		plip->tx_packet->status = -EIO;
		CALL_PKT(plip->tx_packet);
		plip->tx_packet = NULL;
	}
}

static int dispatcher_idle(PLIP *plip)
{
	int r;
	__u8 st;
	if (__unlikely(((unsigned long)plip->rx_packet | (unsigned long)plip->tx_packet) != 0)) free_packets(plip);
	if (__unlikely(!plip->acquired)) {
		plip->irq_event = plip->client.op->acquire(&plip->client);
		/*__debug_printf("attempt acquire %p\n", plip->irq_event);*/
		if (__unlikely(plip->irq_event != NULL)) goto discard_queue;
		plip->acquired = 1;
	}
	if (__unlikely(r = plip->client.op->set_mode(&plip->client, PIO_ECR_MODE_STANDARD | PIO_MODE_NONBLOCK))) {
		KERNEL$SUICIDE("dispatcher_idle: CAN'T SET STANDARD MODE: %d", r);
	}
	if (__unlikely(plip->client.op->want_release(&plip->client))) {
		/*__debug_printf("release\n");*/
		plip->client.op->set_data(&plip->client, 0xff);
		plip->client.op->release(&plip->client);
		plip->acquired = 0;
		plip->irq_event = plip->client.op->acquire(&plip->client);
	/*__debug_printf("attempt acquire 2 %p\n", plip->irq_event);*/
		if (__likely(plip->irq_event != NULL)) goto discard_queue;
	}

	plip->client.op->set_data(&plip->client,
#ifdef PREFER_PMAC
		!plip->plip_connected && plip->protocol & PROTOCOL_PMAC
#else
		!(plip->protocol & PROTOCOL_PLIP)
#endif
		? 0x01 : 0x00);

	plip->irq_event = plip->client.op->enable_irq(&plip->client);
	st = plip->client.op->get_status(&plip->client);

	if (st == 0x40) {
		if (__unlikely(!plip->plip_connected) && plip->protocol & PROTOCOL_PMAC) {
			/* Don't know if this is plip or pmac IRQ */
			NETQUE$DISCARD_PACKETS(plip->queue, -ENETDOWN);
			r = WAIT_YIELD;
			goto ret;
		}
		plip->error_time = 0;
		if (__unlikely(++plip->receive_livelock > RECEIVE_LIVELOCK)) {
			if (!NETQUE$QUEUE_EMPTY(plip->queue) && plip->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE LIVELOCK");
			NETQUE$DISCARD_PACKETS(plip->queue, -EIO);
			plip->receive_livelock = 0;
		}
		plip->tries = 0;
		plip->dispatcher = plip_dispatcher_receive;
		r = plip_dispatcher_receive(plip);
		goto ret;
	} else if (!st) {
		PACKET *p;
		if (!(plip->protocol & PROTOCOL_PLIP))
			goto discard_queue;
		plip->plip_connected = 1;
		packet_retry:
		plip->error_time = 0;
		if (__likely((p = NETQUE$DEQUEUE(plip->queue)) != NULL)) {
			__CHECK_PKT(p, "dispatcher_idle");
			DEFRAG_PACKET(p, {
				r = -ENOMEM;
				return_pkt:
				p->status = r;
				CALL_PKT(p);
				goto packet_retry;
			});
			CHECKSUM_PACKET(p, {
				r = r_;
				goto return_pkt;
			});
			plip->tx_packet = p;
			plip->tries = 0;
			plip->dispatcher = plip_dispatcher_send;
			r = plip_dispatcher_send(plip);
			goto ret;
		}
	} else {
		if (plip_hold_on(plip) && plip->protocol & PROTOCOL_PLIP) {
			r = WAIT_YIELD;
			goto ret;
		}
		plip->plip_connected = 0;
		if (st == 0x08 && plip->protocol & PROTOCOL_PMAC) {
			plip->receive_livelock = 0;
			NETQUE$DISCARD_PACKETS(plip->queue, -ENETDOWN);

			/*__debug_printf("pmac connect\n");*/

			plip->error_time = KERNEL$GET_JIFFIES_LO();
			plip->pmac_probe_period = JIFFIES_PER_SECOND * 3 / 2 + random() % (JIFFIES_PER_SECOND * 3 / 2);
			plip->dispatcher = pmac_dispatcher_connect;
			r = WAIT_YIELD;
			goto ret;
		}
		if (st == 0x80) {
			/* left pmac session ... knock it down */
			plip->client.op->set_data(&plip->client, random() & 1 ? 0x11 : 0x08);
			plip_reset_timer(plip, PMAC_KNOCKDOWN_WAIT);
		}
		discard_queue:
		plip->receive_livelock = 0;
		NETQUE$DISCARD_PACKETS(plip->queue, -ENETDOWN);
	}
	r = WAIT_IRQ;
	ret:
	plip_get_link_state(plip);
	return r;
}

static int plip_dispatcher_receive(PLIP *plip)
{
	/* Receive Linux weirdness:
		Linux sets 0x08 on data lines and waits for seeing 0x01.
		If it temporarily times out, it sets 0x00 and expects 0x00.

		After 10 temporary timeouts, it reports error and abandons
		the packet.

		We must reproduce it here, fiddling with lines like Linux wants.
		When we are here, we never know if Linux started sending us data
		or if it just temporarily timed out. When it sets something
		different from 0x00 and 0x08, we are sure that it is data.

		TODO: should we use both IRQ and YIELD to wait?
		--- doesn't seem necessary --- in 10 retries we eventually get
		in sync with the party.
	*/
	__u8 st = plip->client.op->get_status(&plip->client);
	if (st == 0x40) {
		plip->client.op->set_data(&plip->client, 0x01);
	} else if (!st) {
		plip->client.op->set_data(&plip->client, 0x00);
	} else {
		plip->tries = 0;
		plip->rx_length = 0;
		plip->rx_count = 0;
		plip->dispatcher = plip_dispatcher_receive_length;
		return plip_dispatcher_receive_length(plip);
	}
	KERNEL$UDELAY(1);
	plip->tries++;
	if (__unlikely(plip->tries >= plip->trigger_timeout)) {
		if (plip->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE TRIGGER TIMEOUT (STATUS %02X)", st);
		return plip_error(plip);
	}
	return WAIT_NONE;
}

static int plip_receive_nibble(PLIP *plip)
{
	__u8 st;
	while (!((((st = plip->client.op->get_status(&plip->client)) >> 7) ^ plip->rx_count) & 1)) {
		plip->tries++;
		if (__unlikely(plip->tries >= FAST_TRIES)) {
			if (__unlikely(plip->tries >= plip->nibble_timeout)) {
				if (!plip->rx_count && st == 0x08) {
					plip->plip_connected = 0;
					return WAIT_RESET;
				}
				if (plip->errorlevel >= 2) {
					if (plip->rx_count >= 4) KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE TIMEOUT (NIBBLE %u OF %u, STATUS %02X)", plip->rx_count, (plip->rx_length << 1) + 6, st);
					else KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE TIMEOUT (NIBBLE %u, STATUS %02X)", plip->rx_count, st);
				}
				return plip_error(plip);
			}
			KERNEL$UDELAY(1);
			return WAIT_NONE;
		}
	}
	plip->client.op->set_data(&plip->client, (~plip->rx_count & 1) << 4);
	plip->tries = 0;
	return (st >> 3) & 0x0f;
}

static int plip_dispatcher_receive_length(PLIP *plip)
{
	int r;
	PACKET *p;
	if (__unlikely((r = plip_receive_nibble(plip)) < 0)) return r;
	plip->rx_length |= r << (plip->rx_count++ << 2);
	if (__unlikely(plip->rx_count < 4)) return WAIT_NONE;
	if (__unlikely(plip->rx_length - ETHERNET_HEADER > plip->mtu)) {
		if (plip->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "INVALID PACKET LENGTH: %04X", plip->rx_length);
		return plip_error(plip);
	}
	if (__unlikely(plip->rx_packet != NULL))
		KERNEL$SUICIDE("plip_dispatcher_receive_length: RX PACKET ALREADY PRESENT");
	ALLOC_PACKET(p, plip->rx_length - ETHERNET_HEADER + 1, &plip->pool, SPL_PLIP, {
		plip->dispatcher = plip_dispatcher_receive_data_dummy;
		return WAIT_NONE;
	});
	p->data_length = plip->rx_length - ETHERNET_HEADER;
	p->addrlen = ETHERNET_ADDRLEN;
	p->fn = packet_returned;
	p->sender_data = plip;
	p->h = plip->packet_input;
	p->id = plip->client.net_id;
	plip->dispatcher = plip_dispatcher_receive_data;
	plip->rx_packet = p;
	plip->checksum = 0;
	plip->outstanding++;
	return WAIT_NONE;
}

static int plip_dispatcher_receive_data(PLIP *plip)
{
	int r;
	PACKET *p = plip->rx_packet;
	if (__unlikely((r = plip_receive_nibble(plip)) < 0)) return r;
	if (!(plip->rx_count & 1)) {
		p->data[LL_HEADER - ETHERNET_HEADER + (plip->rx_count >> 1) - 2] = r;
	} else {
		__u8 byte;
		byte = p->data[LL_HEADER - ETHERNET_HEADER + (plip->rx_count >> 1) - 2] |= r << 4;
		if (__unlikely(plip->rx_count == (plip->rx_length << 1) + 5)) {
			plip->client.op->set_data(&plip->client, 0x00);
			if (__unlikely(plip->checksum != byte)) {
				if (plip->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "CHECKSUM MISMATCH, COMPUTED %02X, RECEIVED %02X", plip->checksum, byte);
				return WAIT_RESET;
			}
			/*{
				int i;
				__debug_printf("receive:");
				for (i = -ETHERNET_HEADER; i < p->data_length; i++) __debug_printf(" %02x", p->data[LL_HEADER + i]);
				__debug_printf("\n");
			}*/
			CALL_IORQ(p, KERNEL$PKTIO);
			plip->rx_packet = NULL;
			return WAIT_RESET;
		} else {
			plip->checksum += byte;
		}
	}
	plip->rx_count++;
	return WAIT_NONE;
}

static int plip_dispatcher_receive_data_dummy(PLIP *plip)
{
	int r;
	if (__unlikely((r = plip_receive_nibble(plip)) < 0)) return r;
	if (__unlikely(plip->rx_count == (plip->rx_length << 1) + 5)) {
		plip->client.op->set_data(&plip->client, 0x00);
		return WAIT_RESET;
	}
	plip->rx_count++;
	return WAIT_NONE;
}

static DECL_IOCALL(plip_event, SPL_PLIP, IORQ)
{
	PLIP *plip = GET_STRUCT(RQ, PLIP, event);
	int r;
	u_jiffies_lo_t last_block = KERNEL$GET_JIFFIES_LO();
	plip->irq_event = NULL;
	while ((r = plip->dispatcher(plip)) >= WAIT_DEFAULT) {
		if (__unlikely(KERNEL$GET_JIFFIES_LO() - last_block > plip->schedule_timeout) || (KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_KERNEL_IORQS && r == WAIT_DEFAULT)) {
			/*if (plip->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "YIELDING");*/
			r = WAIT_YIELD;
			break;
		}
	}
	if (__unlikely(plip->terminate)) {
		plip->terminate = 2;
		RETURN;
	}

	if (r == WAIT_RESET)
		plip->dispatcher = dispatcher_idle;

	if (r == WAIT_IRQ) {
		MTX_WAIT(plip->irq_event, &plip->event, plip_event);
	} else {
		WQ_WAIT(&KERNEL$LOCKUP_EVENTS, &plip->event, plip_event);
	}
	RETURN;
}

static DECL_IOCALL(plip_packet, SPL_PLIP, PACKET)
{
	HANDLE *h = RQ->handle;
	PLIP *plip;
	__CHECK_PKT(RQ, "plip_packet");
	if (__unlikely(h->op != &PLIP_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_PLIP));
	plip = h->fnode;
	NETQUE$ENQUEUE_PACKET(plip->queue, RQ);
	if (__likely(plip->irq_event != NULL)) MTX_SET(plip->irq_event, 0);
	RETURN;
}

int plip_dispatcher_send(PLIP *plip)
{
	__u8 st;
	if (!plip->tries) {
		st = plip->client.op->get_status(&plip->client);
		if (__unlikely(st == 0x40)) goto maybe_retry;
		plip->client.op->set_data(&plip->client, 0x08);
	}
	KERNEL$UDELAY(1);
	st = plip->client.op->get_status(&plip->client);
	if (__likely(st & 0x08)) {
		st = plip->client.op->get_status(&plip->client);
		if (!(st & 0x08)) goto unstable_status;
		plip->send_timed = 0;
		plip->receive_livelock = 0;
		plip->tx_count = 0;
		plip->checksum = 0;
		plip_set_nibble(plip, plip->tx_packet->data_length + ETHERNET_HEADER);
		plip->dispatcher = plip_dispatcher_send_data;
		return WAIT_NONE;
	}
	unstable_status:
	if (__unlikely(st == 0x40)) {
		plip->client.op->set_data(&plip->client, 0x00);
		/*if (plip->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "TRANSMIT COLLISION");*/
		KERNEL$UDELAY(random() % COLLISION_BACKOFF);
		maybe_retry:
		/* Retry packet once */
		if (__likely(plip->send_timed ^= 1)) {
			NETQUE$REQUEUE_DEQUEUED_PACKET(plip->queue, plip->tx_packet);
			plip->tx_packet = NULL;
		}
		plip->client.op->set_data(&plip->client, 0x00);
		return WAIT_RESET;
	}
	plip->tries++;
	if (__unlikely(plip->tries >= plip->trigger_timeout)) {
		if (plip->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "TRANSMIT TRIGGER TIMEOUT (STATUS %02X)", st);
		return plip_error(plip);
	}
	if (!(plip->tries % TX_TRIGGER_FLIP)) {
	/* Linux has some race condition, it sometimes times out with (1,c0)
	   probably because it didn't check interrupt bit after enabling
	   interrupts. Flip the interrupt bit sometimes to trigger the interrupt
	   on the other side. */
		plip->client.op->set_data(&plip->client, 0x00);
		KERNEL$UDELAY(1);
		plip->client.op->set_data(&plip->client, 0x08);
	}
	return WAIT_NONE;
}

static void plip_set_nibble(PLIP *plip, __u8 data)
{
	__u8 d;
	plip->tries = 0;
	d = (data & 0x0f) | ((plip->tx_count & 1) << 4);
	plip->client.op->set_data(&plip->client, d);
	plip->client.op->set_data(&plip->client, d ^ 0x10);
}

static int wait_nibble(PLIP *plip)
{
	__u8 st;
	while (!((((st = plip->client.op->get_status(&plip->client)) >> 7) ^ plip->tx_count) & 1)) {
		plip->tries++;
		if (__unlikely(plip->tries >= FAST_TRIES)) {
			if (__unlikely(plip->tries >= plip->nibble_timeout)) {
				if (plip->errorlevel >= 2)
					KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "TRANSMIT TIMEOUT (NIBBLE %u OF %u, STATUS %02X)", plip->tx_count, (plip->tx_packet->data_length + ETHERNET_HEADER + 3) << 1, st);
				return plip_error(plip);
			}
			KERNEL$UDELAY(1);
			return WAIT_NONE;
		}
	}
	plip->tx_count++;
	return 0;
}

static int plip_dispatcher_send_data(PLIP *plip)
{
	__u8 ni;
	int r;
	if (__unlikely((r = wait_nibble(plip)) < 0)) return r;
	if (__unlikely(plip->tx_count <= 4)) {
		if (plip->tx_count < 4) {
			ni = (plip->tx_packet->data_length + ETHERNET_HEADER) >> (plip->tx_count << 2);
		} else {
			ni = plip->tx_packet->data[LL_HEADER - ETHERNET_HEADER];
		/* Linux IP stack hates packets with multicast bit set */
			if (__likely(PKT_PROTOCOL(plip->tx_packet) == NET_PROTOCOL_IP)) ni &= 0xfe;
			plip->checksum = ni;
		}
	} else if (__likely(plip->tx_count < (plip->tx_packet->data_length + ETHERNET_HEADER + 2) << 1)) {
		ni = plip->tx_packet->data[LL_HEADER - ETHERNET_HEADER + (plip->tx_count >> 1) - 2];
		plip->checksum += ni & ((plip->tx_count & 1) - 1);
		ni >>= ((plip->tx_count & 1) << 2);
	} else if (plip->tx_count < (plip->tx_packet->data_length + ETHERNET_HEADER + 3) << 1) {
		ni = plip->checksum >> ((plip->tx_count & 1) << 2);
	} else {
		plip->client.op->set_data(&plip->client, 0x00);
		plip->tx_packet->status = 0;
		CALL_PKT(plip->tx_packet);
		plip->tx_packet = NULL;
			/* Give other party a chance to send */
		KERNEL$UDELAY(TRANSMIT_BACKOFF);
		return WAIT_RESET;
	}
	plip_set_nibble(plip, ni);
	return WAIT_NONE;
}

/*
static void plip_fast_irq(struct pio_client *client)
{
	PLIP *plip = GET_STRUCT(client, PLIP, client);
	__u8 st = plip->client.op->get_status(&plip->client);
	if (__likely(st == 0x40)) {
		plip->intr_receive = 1;
		plip->client.op->set_data(&plip->client, 0x01);
	}
}
*/

static DECL_AST(packet_returned, SPL_PLIP, PACKET)
{
	PLIP *plip;
	PKT_AST_ENTER(RQ);
	plip = RQ->sender_data;
	FREE_PACKET(RQ, &plip->pool, SPL_PLIP);
	plip->outstanding--;
	RETURN;
}

/* If invalid status is seen for this amount of time, hold on and do not assume
   link down. Linux does it in IRQ-less mode. */

static int plip_hold_on(PLIP *plip)
{
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	if (plip->error_time) {
		u_jiffies_lo_t t = j - plip->error_time;
		u_jiffies_lo_t eb;
		USEC_2_JIFFIES_LO(HOLD_ON, eb);
		return t < TIMEOUT_JIFFIES(eb);
	} else {
		plip->error_time = j;
		return 1;
	}
}

/* Error recovery on Linux is tricky --- Linux kernel sets 0x00 and waits in
   error state until it sees 0x00 on data lines --- unfortunatelly we don't know
   when it noticed our 0x00 and left error state.

   If we start sending too early, Linux kernel will notice 0x08 on data lines
   and will wait in error state.

   So wait for some time and hope that Linux noticed us.

   The other party may start sending to us, when it happens, exit error state.
*/

static int plip_error(PLIP *plip)
{
	plip->client.op->set_data(&plip->client, 0x00);
	plip->error_irq = plip->client.op->get_status(&plip->client) == 0x40;
	plip->dispatcher = plip_error_dispatcher;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_NONE;
}

static int plip_error_dispatcher(PLIP *plip)
{
	u_jiffies_lo_t t = KERNEL$GET_JIFFIES_LO() - plip->error_time;
	if (__unlikely(t > plip->error_timeout)) {
		/*__debug_printf("reset error state: %x / %x\n", t, plip->error_time);*/
		plip->error_time = 0;
		return WAIT_RESET;
	}
	if (__likely(plip->client.op->get_status(&plip->client) != 0x40)) {
		plip->error_irq = 0;
	} else {
		if (!plip->error_irq)  {
			plip->error_time = 0;
			return WAIT_RESET;
		}
	}
	return WAIT_YIELD;
}


static void pmac_print_timeout(PLIP *plip, const char *state)
{
	if (!state) return;
	if (plip->errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "PMAC TIMEOUT IN STATE %s, STATUS %02X, TX STATE %d, RX STATE %d", state, plip->client.op->get_status(&plip->client), plip->pmac_send_state, plip->pmac_recv_state);
}

#define test_pmac_timeout(plip, str, action)				\
do {									\
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - (plip)->error_time > (plip)->pmac_timeout)) {								\
		pmac_print_timeout(plip, str);				\
		{							\
			action;						\
		}							\
		__unreachable_code();					\
	}								\
} while (0)

static int pmac_dispatcher_connect(PLIP *plip)
{
	__u8 st;
	int state;
	if (__unlikely(((unsigned long)plip->rx_packet | (unsigned long)plip->tx_packet) != 0)) free_packets(plip);
	state = (KERNEL$GET_JIFFIES_LO() - plip->error_time) % (plip->pmac_probe_period * 2) >= plip->pmac_probe_period;
	plip->pmac_send_byte = 0;
	plip->client.op->set_data(&plip->client, !state ? 0x01 : 0x08);
	st = plip->client.op->get_status(&plip->client);
	/*__debug_printf("connect: %d, %02x\n", state, st);*/
	if (st == 0x08) return WAIT_YIELD;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x08) return WAIT_YIELD;
	if (st == 0x40) {
		plip->client.op->set_data(&plip->client, 0x09);
		plip->dispatcher = pmac_dispatcher_connect_slave_1;
		plip->error_time = KERNEL$GET_JIFFIES_LO();
		return WAIT_DEFAULT;
	} else if (__likely(st == 0x48)) {
		plip->client.op->set_data(&plip->client, 0x0a);
		plip->dispatcher = pmac_dispatcher_connect_master_1;
		plip->error_time = KERNEL$GET_JIFFIES_LO();
		return WAIT_DEFAULT;
	} else {
		return WAIT_RESET;
	}
}

static int pmac_dispatcher_connect_slave_1(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x40) {
		test_pmac_timeout(plip, NULL, return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(st != 0x50)) return WAIT_RESET;
	plip->client.op->set_data(&plip->client, 0x0b);
	plip->dispatcher = pmac_dispatcher_connect_slave_2;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_slave_2(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x50) {
		test_pmac_timeout(plip, "SLAVE CONNECT 2", return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(st != 0x60)) return WAIT_RESET;
	plip->client.op->set_data(&plip->client, 0x0d);
	plip->dispatcher = pmac_dispatcher_connect_slave_3;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_slave_3(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x60) {
		test_pmac_timeout(plip, "SLAVE CONNECT 3", return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(st != 0x70)) return WAIT_RESET;
	plip->client.op->set_data(&plip->client, 0x10);
	plip->dispatcher = pmac_dispatcher_connect_slave_4;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_slave_4(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x70) {
		test_pmac_timeout(plip, "SLAVE CONNECT 4", return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(st != 0x80)) return WAIT_RESET;
	/*__debug_printf("negotiation complete - slave\n");*/
	plip->dispatcher = pmac_dispatcher_connect_slave_wait;
	plip_get_link_state(plip);
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_slave_wait(PLIP *plip)
{
	if (__likely(KERNEL$GET_JIFFIES_LO() - plip->error_time < PMAC_INIT_WAIT))
		return WAIT_YIELD;
	plip->dispatcher = pmac_dispatcher_slave_irq;
	return WAIT_DEFAULT;
}

#define LOOP_DECL		\
	unsigned loop;

#define LOOP_START		\
	loop = FAST_TRIES;	\
	retry:

#define LOOP_TEST		\
	while (__likely(--loop)) goto retry;

static int pmac_dispatcher_slave_prepare_irq(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (st != 0x78 && st != 0x00) {
		LOOP_TEST;
		test_pmac_timeout(plip, "SLAVE PREPARE IRQ", return WAIT_RESET;);
		return WAIT_NONE;
	}
	if (st == 0x78) {
		plip->pmac_send_state = PMAC_STATE_HEAD;
		plip->dispatcher = pmac_dispatcher_slave_hi;
		goto ret;
	}
	plip->client.op->set_data(&plip->client, 0x10);
	plip->dispatcher = pmac_dispatcher_slave_prepare_irq_2;
	ret:
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_NONE;
}

static int pmac_dispatcher_slave_prepare_irq_2(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (!st) {
		LOOP_TEST;
		test_pmac_timeout(plip, "SLAVE PREPARE IRQ 2", return WAIT_RESET;);
		return WAIT_NONE;
	}
	/*__debug_printf("sleep\n");*/
	plip->dispatcher = pmac_dispatcher_slave_irq;
	return WAIT_NONE;
}

static int pmac_dispatcher_slave_irq(PLIP *plip)
{
	__u8 st;
	plip->pmac_recv_state = PMAC_STATE_IDLE;
	plip->pmac_send_state = PMAC_STATE_IDLE;
	if (!NETQUE$QUEUE_EMPTY(plip->queue)) {
		pmac_byte_to_send(plip);
		if (__likely(plip->pmac_send_state > PMAC_STATE_HEAD)) {
			plip->client.op->set_data(&plip->client, 0x00 | (plip->pmac_send_byte >> 4));
			plip->dispatcher = pmac_dispatcher_slave_ack_irq;
			goto ret;
		}
	}
	plip->irq_event = plip->client.op->enable_irq(&plip->client);
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(!(st & 0x40))) {
		return WAIT_IRQ;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__likely(st == 0x70)) {
		plip->client.op->set_data(&plip->client, 0x00);
		plip->pmac_send_state = PMAC_STATE_HEAD;
		plip->pmac_send_byte = 0;
		plip->dispatcher = pmac_dispatcher_slave_ack_irq;
	} else {
		if (st != 0xf8 && st != 0x40)
			if (plip->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "INVALID PMAC SLAVE IRQ STATUS %02X", st);
		return WAIT_RESET;
	}
	ret:
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_NONE;
}

static int pmac_dispatcher_slave_ack_irq(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (st != 0x78 && st != 0x00) {
		LOOP_TEST;
		test_pmac_timeout(plip, "SLAVE ACK IRQ", return WAIT_RESET;);
		return WAIT_NONE;
	}
	plip->dispatcher = pmac_dispatcher_slave_hi;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_NONE;
}

static int pmac_dispatcher_slave_hi(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	if (plip->pmac_recv_state <= PMAC_STATE_HEAD && __unlikely(plip->pmac_send_state <= PMAC_STATE_IDLE)) {
		plip->dispatcher = pmac_dispatcher_slave_prepare_irq;
		plip->error_time = KERNEL$GET_JIFFIES_LO();
		return WAIT_NONE;
	}
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (st & 0x80) {
		LOOP_TEST;
		test_pmac_timeout(plip, "SLAVE HI", return WAIT_RESET;);
		return WAIT_NONE;
	}
	/*{ static int x = 0; if (!(++x & 0xfff)) __debug_printf("s/hi: loops: %d\n", loop); }*/
	st = plip->client.op->get_status(&plip->client);
	plip->pmac_recv_byte = (st << 1) & 0xf0;
	plip->client.op->set_data(&plip->client, 0x10 | (plip->pmac_send_byte & 0xf));
	plip->dispatcher = pmac_dispatcher_slave_lo;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return pmac_dispatcher_slave_lo(plip);
}

static int pmac_dispatcher_slave_lo(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (!(st & 0x80)) {
		LOOP_TEST;
		test_pmac_timeout(plip, "SLAVE LO", return WAIT_RESET;);
		return WAIT_NONE;
	}
	/*{ static int x = 0; if (!(++x & 0xfff)) __debug_printf("s/lo: loops: %d\n", loop); }*/
	st = plip->client.op->get_status(&plip->client);
	plip->pmac_recv_byte |= (st >> 3) & 0xf;
	pmac_byte_to_send(plip);
	plip->client.op->set_data(&plip->client, 0x00 | (plip->pmac_send_byte >> 4));
	plip->dispatcher = pmac_dispatcher_slave_hi;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return pmac_received_byte(plip);
}

static int pmac_dispatcher_connect_master_1(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x48) {
		test_pmac_timeout(plip, NULL, return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(st != 0x58)) return WAIT_RESET;
	plip->client.op->set_data(&plip->client, 0x0c);
	plip->dispatcher = pmac_dispatcher_connect_master_2;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_master_2(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x58) {
		test_pmac_timeout(plip, "MASTER CONNECT 2", return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(st != 0x68)) return WAIT_RESET;
	plip->client.op->set_data(&plip->client, 0x0e);
	plip->dispatcher = pmac_dispatcher_connect_master_3;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_master_3(PLIP *plip)
{
	__u8 st;
	st = plip->client.op->get_status(&plip->client);
	if (st != 0x78 && st != 0x80) {
		test_pmac_timeout(plip, "MASTER CONNECT 3", return WAIT_RESET;);
		return WAIT_DEFAULT;
	}
	plip->client.op->set_data(&plip->client, 0x10);
	/*__debug_printf("negotiation complete - master\n");*/
	plip->dispatcher = pmac_dispatcher_connect_master_wait;
	plip_get_link_state(plip);
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_connect_master_wait(PLIP *plip)
{
	if (__likely(KERNEL$GET_JIFFIES_LO() - plip->error_time < PMAC_INIT_WAIT))
		return WAIT_YIELD;
	plip->dispatcher = pmac_dispatcher_master_irq;
	return WAIT_DEFAULT;
}

static int pmac_dispatcher_master_prepare_irq(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (st != 0x80) {
		LOOP_TEST;
		test_pmac_timeout(plip, "MASTER PREPARE IRQ", return WAIT_RESET;);
		return WAIT_NONE;
	}
	plip->client.op->set_data(&plip->client, 0x10);
	plip->dispatcher = pmac_dispatcher_master_irq;
	return WAIT_NONE;
}

static int pmac_dispatcher_master_irq(PLIP *plip)
{
	__u8 st;
	plip->pmac_recv_state = PMAC_STATE_IDLE;
	plip->pmac_send_state = PMAC_STATE_IDLE;
	if (!NETQUE$QUEUE_EMPTY(plip->queue)) {
		pmac_byte_to_send(plip);
		if (__likely(plip->pmac_send_state > PMAC_STATE_HEAD)) {
			plip->client.op->set_data(&plip->client, 0x0e);
			plip->dispatcher = pmac_dispatcher_master_hi;
			goto ret;
		}
	}
	plip->irq_event = plip->client.op->enable_irq(&plip->client);
	st = plip->client.op->get_status(&plip->client);
	if (__unlikely(!(st & 0x40))) {
		return WAIT_IRQ;
	}
	st = plip->client.op->get_status(&plip->client);
	if (__likely(st == 0x78)) {
		plip->pmac_send_state = PMAC_STATE_HEAD;
		plip->pmac_send_byte = 0;
		plip->dispatcher = pmac_dispatcher_master_hi;
	} else {
		if (st != 0xf8 && st != 0x40)
			if (plip->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "INVALID PMAC MASTER IRQ STATUS %02X", st);
		return WAIT_RESET;
	}
	ret:
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_NONE;
}

static int pmac_dispatcher_master_hi(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (st & 0x80) {
		LOOP_TEST;
		test_pmac_timeout(plip, "MASTER HI", return WAIT_RESET;);
		return WAIT_NONE;
	}
	/*{ static int x = 0; if (!(++x & 0xfff)) __debug_printf("m/hi: loops: %d\n", loop); }*/
	st = plip->client.op->get_status(&plip->client);
	plip->pmac_recv_byte = (st << 1) & 0xf0;
	plip->client.op->set_data(&plip->client, 0x00 | (plip->pmac_send_byte >> 4));
	if (__unlikely(plip->pmac_recv_state <= PMAC_STATE_HEAD) && __unlikely(plip->pmac_send_state <= PMAC_STATE_IDLE) && !st) {
		plip->dispatcher = pmac_dispatcher_master_prepare_irq;
		plip->error_time = KERNEL$GET_JIFFIES_LO();
		return WAIT_NONE;
	}
	plip->dispatcher = pmac_dispatcher_master_lo;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return pmac_dispatcher_master_lo(plip);
}

static int pmac_dispatcher_master_lo(PLIP *plip)
{
	__u8 st;
	LOOP_DECL;
	LOOP_START;
	st = plip->client.op->get_status(&plip->client);
	if (!(st & 0x80)) {
		LOOP_TEST;
		test_pmac_timeout(plip, "MASTER LO", return WAIT_RESET;);
		return WAIT_NONE;
	}
	/*{ static int x = 0; if (!(++x & 0xfff)) __debug_printf("m/lo: loops: %d\n", loop); }*/
	st = plip->client.op->get_status(&plip->client);
	plip->pmac_recv_byte |= (st >> 3) & 0xf;
	plip->client.op->set_data(&plip->client, 0x10 | (plip->pmac_send_byte & 0xf));
	pmac_byte_to_send(plip);
	plip->dispatcher = pmac_dispatcher_master_hi;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return pmac_received_byte(plip);
}

static int pmac_received_byte(PLIP *plip)
{
	PACKET *p;
	if (__likely(plip->pmac_recv_state == PMAC_STATE_DATA)) {
		if (__likely((p = plip->rx_packet) != NULL)) {
			p->data[LL_HEADER - ETHERNET_HEADER + plip->rx_count] = plip->pmac_recv_byte;
		}
		plip->rx_count++;
		if (__unlikely(plip->rx_count == plip->rx_length)) {
			plip->pmac_recv_state = PMAC_STATE_HEAD;
			/*{
				int i;
				__debug_printf("pmac receive:");
				for (i = -ETHERNET_HEADER; i < p->data_length; i++) {
					__debug_printf(" %02x", p->data[LL_HEADER + i]);
				}
				__debug_printf("\n");
			}*/
			if (__likely(p != NULL)) {
				CALL_IORQ(p, KERNEL$PKTIO);
				plip->rx_packet = NULL;
			}
		}
	} else if (__likely(plip->pmac_recv_state <= PMAC_STATE_HEAD)) {
		if (__likely(!plip->pmac_recv_byte)) {
			plip->pmac_recv_state = PMAC_STATE_IDLE;
			return WAIT_DEFAULT;
		}
		if (__unlikely(plip->pmac_recv_byte != 0xff) &&
		    __unlikely(plip->pmac_recv_byte != 0xef)) {
			if (plip->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "INVALID PMAC HEADER BYTE %02X", plip->pmac_recv_byte);
			return WAIT_RESET;
		}
		plip->pmac_recv_state = PMAC_STATE_LEN_LO;
	} else if (plip->pmac_recv_state == PMAC_STATE_LEN_LO) {
		plip->rx_length = plip->pmac_recv_byte;
		plip->pmac_recv_state = PMAC_STATE_LEN_HI;
	} else if (plip->pmac_recv_state == PMAC_STATE_LEN_HI) {
		plip->rx_length |= plip->pmac_recv_byte << 8;
		plip->rx_count = 0;
		plip->pmac_recv_state = PMAC_STATE_DATA;
		if (__unlikely(plip->rx_length - ETHERNET_HEADER > plip->mtu)) {
			if (plip->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "INVALID PACKET LENGTH: %04X", plip->rx_length);
			return WAIT_RESET;
		}
		if (__unlikely(plip->rx_packet != NULL))
			KERNEL$SUICIDE("pmac_received_byte: RX PACKET ALREADY PRESENT");
		ALLOC_PACKET(p, plip->rx_length - ETHERNET_HEADER + 1, &plip->pool, SPL_PLIP, {
			return WAIT_DEFAULT;
		});
		p->data_length = plip->rx_length - ETHERNET_HEADER;
		p->addrlen = ETHERNET_ADDRLEN;
		p->fn = packet_returned;
		p->sender_data = plip;
		p->h = plip->packet_input;
		p->id = plip->client.net_id;
		plip->rx_packet = p;
		plip->outstanding++;
	} else {
		KERNEL$SUICIDE("pmac_received_byte: INVALID STATE %d", plip->pmac_recv_state);
	}
	return WAIT_DEFAULT;
}

static void pmac_byte_to_send(PLIP *plip)
{
	PACKET *p;
	if (__likely(plip->pmac_send_state == PMAC_STATE_DATA)) {
		plip->pmac_send_byte = plip->tx_packet->data[LL_HEADER - ETHERNET_HEADER + plip->tx_count];
		plip->tx_count++;
		if (__unlikely(plip->tx_count == plip->tx_packet->data_length + ETHERNET_HEADER)) {
			plip->tx_packet->status = 0;
			CALL_PKT(plip->tx_packet);
			plip->tx_packet = NULL;
			plip->pmac_send_state = PMAC_STATE_HEAD;
		}
	} else if (__likely(plip->pmac_send_state <= PMAC_STATE_HEAD)) {
		plip->pmac_send_state = PMAC_STATE_IDLE;
		plip->pmac_send_byte = 0x00;
		packet_retry:
		if (plip->tx_packet != NULL)
			KERNEL$SUICIDE("pmac_byte_to_send: TX PACKET ALREADY PRESENT");
		if (__unlikely((p = NETQUE$DEQUEUE(plip->queue)) != NULL)) {
			int r;
			__CHECK_PKT(p, "pmac_byte_to_send");
			DEFRAG_PACKET(p, {
				r = -ENOMEM;
				return_pkt:
				p->status = r;
				CALL_PKT(p);
				goto packet_retry;
			});
			CHECKSUM_PACKET(p, {
				r = r_;
				goto return_pkt;
			});
			plip->tx_packet = p;
			plip->pmac_send_byte = 0xff;
			plip->pmac_send_state = PMAC_STATE_LEN_LO;
		}
	} else if (plip->pmac_send_state == PMAC_STATE_LEN_LO) {
		plip->pmac_send_byte = plip->tx_packet->data_length + ETHERNET_HEADER;
		plip->pmac_send_state = PMAC_STATE_LEN_HI;
	} else if (plip->pmac_send_state == PMAC_STATE_LEN_HI) {
		plip->pmac_send_byte = (plip->tx_packet->data_length + ETHERNET_HEADER) >> 8;
		plip->pmac_send_state = PMAC_STATE_DATA;
		plip->tx_count = 0;
	} else {
		KERNEL$SUICIDE("pmac_byte_to_send: INVALID STATE %d", plip->pmac_send_state);
	}
}

static void plip_get_link_state(PLIP *plip)
{
	LINK_STATE_PHYS ls;
	memset(&ls, 0, sizeof(LINK_STATE_PHYS));
	if (!plip->acquired) {
	} else if (plip->dispatcher == dispatcher_idle) {
		__u8 st = plip->client.op->get_status(&plip->client);
		if (!st || st == 0x40 || plip_hold_on(plip)) goto plip;
	} else if (plip->dispatcher == pmac_dispatcher_connect) {
		pmac_connecting:
		ls.flags |= LINK_STATE_CONNECTING;
		goto id_pmac;
	} else if (plip->dispatcher == pmac_dispatcher_connect_slave_1 ||
		   plip->dispatcher == pmac_dispatcher_connect_slave_2 ||
		   plip->dispatcher == pmac_dispatcher_connect_slave_3 ||
		   plip->dispatcher == pmac_dispatcher_connect_slave_4 ||
		   plip->dispatcher == pmac_dispatcher_connect_slave_wait) {
		ls.flags |= LINK_STATE_SLAVE;
		goto pmac_connecting;
	} else if (plip->dispatcher == pmac_dispatcher_connect_master_1 ||
		   plip->dispatcher == pmac_dispatcher_connect_master_2 ||
		   plip->dispatcher == pmac_dispatcher_connect_master_3 ||
		   plip->dispatcher == pmac_dispatcher_connect_master_wait) {
		ls.flags |= LINK_STATE_MASTER;
		goto pmac_connecting;
	} else if (plip->dispatcher == pmac_dispatcher_slave_prepare_irq ||
		   plip->dispatcher == pmac_dispatcher_slave_prepare_irq_2 ||
		   plip->dispatcher == pmac_dispatcher_slave_irq ||
		   plip->dispatcher == pmac_dispatcher_slave_ack_irq ||
		   plip->dispatcher == pmac_dispatcher_slave_hi ||
		   plip->dispatcher == pmac_dispatcher_slave_lo) {
		strlcpy(ls.desc, "OS/2 PMAC", sizeof ls.desc);
		ls.flags |= LINK_STATE_UP | LINK_STATE_FULL_DUPLEX | LINK_STATE_SLAVE;
	} else if (plip->dispatcher == pmac_dispatcher_master_prepare_irq ||
		   plip->dispatcher == pmac_dispatcher_master_irq ||
		   plip->dispatcher == pmac_dispatcher_master_hi ||
		   plip->dispatcher == pmac_dispatcher_master_lo) {
		ls.flags |= LINK_STATE_UP | LINK_STATE_FULL_DUPLEX | LINK_STATE_MASTER;
		id_pmac:
		strlcpy(ls.desc, "OS/2 PMAC", sizeof ls.desc);
	} else {
		plip:
		if (plip->protocol & PROTOCOL_PLIP) {
			ls.flags |= LINK_STATE_UP | LINK_STATE_HALF_DUPLEX;
			strlcpy(ls.desc, "DOS/LINUX PLIP", sizeof ls.desc);
		}
	}
	if (__unlikely(memcmp(&ls, &plip->link_state, sizeof(LINK_STATE_PHYS)))) {
		WQ_WAKE_ALL_PL(&plip->link_state_wait);
		memcpy(&plip->link_state, &ls, sizeof(LINK_STATE_PHYS));
	}
}

static void plip_reset_timer(PLIP *plip, u_jiffies_lo_t t)
{
	KERNEL$DEL_TIMER(&plip->timer);
	KERNEL$SET_TIMER(t, &plip->timer);
}

static void plip_timer(TIMER *t)
{
	PLIP *plip = GET_STRUCT(t, PLIP, timer);
	LOWER_SPL(SPL_PLIP);
	KERNEL$SET_TIMER(PLIP_TIMER, &plip->timer);
	plip_get_link_state(plip);
	if (__likely(plip->irq_event != NULL)) MTX_SET(plip->irq_event, 0);
}


static DECL_IOCALL(plip_ioctl, SPL_PLIP, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	PLIP *plip;
	int r;
	if (__unlikely(h->op != &PLIP_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_PLIP));
	plip = h->fnode;
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			t.mtu = plip->mtu;
			t.addrlen = ETHERNET_ADDRLEN;
			t.arptype = htons(1);
			strlcpy(t.type, "PLIP", sizeof(t.type));
			t.id = plip->client.net_id;
			memcpy(t.addr, plip->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			plip_get_link_state(plip);
			if (!memcmp(&plip->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&plip->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			plip_get_link_state(plip);
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &plip->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			plip->link_state.seq++;
			WQ_WAKE_ALL_PL(&plip->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static void plip_init_root(HANDLE *h, void *data)
{
	PLIP *plip = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = plip;
	h->op = &PLIP_OPERATIONS;
}

static int PLIP_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	int r;
	char *lpt = NULL;
	int errorlevel = 0;
	char *address_str = NULL;
	int mtu = ETHERNET_MTU;
	int nibble_timeout = DEFAULT_NIBBLE_TIMEOUT;
	int trigger_timeout = DEFAULT_TRIGGER_TIMEOUT;
	int schedule_timeout = DEFAULT_SCHEDULE_TIMEOUT;
	int error_timeout = DEFAULT_ERROR_TIMEOUT;
	int pmac_timeout = DEFAULT_PMAC_TIMEOUT;
	char *net_input = NET_INPUT_DEVICE ":";
	int protocol = PROTOCOL_PLIP | PROTOCOL_PMAC;
	PLIP *plip;
	const char * const *arg;
	int state;
	union {
		MALLOC_REQUEST mrq;
		OPENRQ openrq;
	} u;
	static const struct __param_table protocol_params[4] = {
		"AUTO", __PARAM_BOOL, ~0, PROTOCOL_PLIP | PROTOCOL_PMAC,
		"PLIP", __PARAM_BOOL, ~0, PROTOCOL_PLIP,
		"PMAC", __PARAM_BOOL, ~0, PROTOCOL_PMAC,
		NULL, 0, 0, 0,
	};
	static const struct __param_table params[13] = {
		"LPT", __PARAM_STRING, 1, __MAX_STR_LEN,
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_WARNINGS", __PARAM_BOOL, ~1, 2,
		"ADDRESS", __PARAM_STRING, 1, MAXINT,
		"MTU", __PARAM_INT, 576, ETHERNET_JUMBO_MTU+1,
		"INPUT", __PARAM_STRING, 1, MAXINT,
		"PROTOCOL", __PARAM_EXTD_ONE, (long)&protocol_params, 0,
		"TIMEOUT", __PARAM_INT, FAST_TRIES, 1000001,
		"TRIGGER_TIMEOUT", __PARAM_INT, FAST_TRIES, 1000001,
		"SCHEDULE_TIMEOUT", __PARAM_INT, 0, 1000001,
		"ERROR_TIMEOUT", __PARAM_INT, 0, 1000001,
		"PMAC_TIMEOUT", __PARAM_INT, 0, 10000001,
		NULL, 0, 0, 0,
	};
	void *vars[13];
	void *protocol_vars[4];

	PKT_CHECK_VERSION;

	vars[0] = &lpt;
	vars[1] = &errorlevel;
	vars[2] = &errorlevel;
	vars[3] = &address_str;
	vars[4] = &mtu;
	vars[5] = &net_input;
	vars[6] = &protocol_vars;
	vars[7] = &nibble_timeout;
	vars[8] = &trigger_timeout;
	vars[9] = &schedule_timeout;
	vars[10] = &error_timeout;
	vars[11] = &pmac_timeout;
	vars[12] = NULL;
	protocol_vars[0] = &protocol;
	protocol_vars[1] = &protocol;
	protocol_vars[2] = &protocol;
	protocol_vars[3] = NULL;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}

	u.mrq.size = sizeof(PLIP);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: COULD NOT ALLOCATE MEMORY: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret0;
	}
	plip = u.mrq.ptr;
	memset(plip, 0, sizeof(PLIP));
	plip->protocol = protocol;
	plip->mtu = mtu;
	plip->errorlevel = errorlevel;
	plip->nibble_timeout = nibble_timeout;
	plip->trigger_timeout = trigger_timeout;
	USEC_2_JIFFIES_LO(schedule_timeout, plip->schedule_timeout);
	plip->schedule_timeout = TIMEOUT_JIFFIES(plip->schedule_timeout);
	USEC_2_JIFFIES_LO(error_timeout, plip->error_timeout);
	plip->error_timeout = TIMEOUT_JIFFIES(plip->error_timeout);
	USEC_2_JIFFIES_LO(pmac_timeout, plip->pmac_timeout);
	plip->pmac_timeout = TIMEOUT_JIFFIES(plip->pmac_timeout);

	if (address_str) {
		if (NET$PARSE_ETHERNET_ADDRESS(address_str, plip->address) || !NET$VALID_ETHERNET_ADDRESS(plip->address)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: INVALID ADDRESS: %s", address_str);
			r = -EBADSYN;
			goto ret1;
		}
	} else {
		memset(plip->address, 0xfe, sizeof plip->address);
	}

	plip->client.size_of_pio_operations = sizeof(struct pio_operations);
	plip->client.size_of_pio_client = sizeof(struct pio_client);
	if (!lpt) {
		size_t n;
		LOGICAL_LIST_REQUEST lrq;
		lrq.prefix = "PIO$";
		SYNC_IO_CANCELABLE(&lrq, KERNEL$LIST_LOGICALS);
		if (lrq.status < 0) {
			if (lrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: CAN'T LIST LOGICAL NAMES");
			r = lrq.status;
			goto ret1;
		}
		for (n = 0; n < lrq.n_entries; n++) {
			if (!(r = KERNEL$DCALL(lrq.entries[n]->name, "PIO", PIO_CMD_ATTACH_MOST_TIME_NOACQ, &plip->client))) {
				strcpy(plip->pio_name, lrq.entries[n]->name);
				KERNEL$FREE_LOGICAL_LIST(&lrq);
				goto ok;
			}
			if (r == -EINTR) {
				KERNEL$FREE_LOGICAL_LIST(&lrq);
				goto ret1;
			}
		}
		if (!lrq.n_entries) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: NO PARALLEL PORTS"), r = -ENXIO;
		else _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ALL PARALLEL PORTS ARE LOCKED"), r = -EBUSY;
		KERNEL$FREE_LOGICAL_LIST(&lrq);
		goto ret1;
	} else {
		if ((r = KERNEL$DCALL(lpt, "PIO", PIO_CMD_ATTACH_MOST_TIME_NOACQ, &plip->client))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: CAN'T ATTACH TO PARALLEL PORT DRIVER %s: %s", lpt, strerror(-r));
			goto ret1;
		}
		strcpy(plip->pio_name, lpt);
		__check_logical_name(plip->pio_name, 1);
	}
	ok:
	_snprintf(plip->dev_name, sizeof(plip->dev_name), "PKT$PLIP@%s", plip->pio_name);

	/*if (fastirq) plip->fast_irq = plip_fast_irq;*/
	/*if ((r = plip->client.op->set_mode(&plip->client, PIO_ECR_MODE_STANDARD))) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCAPABILITY, plip->dev_name, "CAN'T SET STANDARD MODE: %s", strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T SET STANDARD MODE", plip->dev_name);
		goto ret2;
	}*/
	WQ_INIT(&plip->link_state_wait, "PLIP$LINK_STATE_WAIT");
	INIT_PKTPOOL(&plip->pool, mtu);
	if ((r = NET$PKTPOOL_RESERVE(&plip->pool))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET POOL: %s", plip->dev_name, strerror(-r));
		goto ret2;
	}
	if ((r = NETQUE$ALLOC_QUEUE(&plip->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", plip->dev_name, strerror(-r));
		goto ret3;
	}
	u.openrq.flags = O_WRONLY;
	u.openrq.path = net_input;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", plip->dev_name, net_input, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto ret4;
	}
	plip->packet_input = u.openrq.status;

	r = KERNEL$REGISTER_DEVICE(plip->dev_name, "PLIP.SYS", 0, plip, plip_init_root, NULL, NULL, NULL, PLIP_UNLOAD, &plip->lnte, plip->pio_name, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", plip->dev_name, strerror(-r));
		goto ret6;
	}
	/*__debug_printf("status: %02x\n", plip->client.op->get_status(&plip->client));*/

	INIT_TIMER(&plip->timer);
	plip->timer.fn = plip_timer;
	KERNEL$SET_TIMER(PLIP_TIMER, &plip->timer);

	plip->dispatcher = dispatcher_idle;
	CALL_IORQ(&plip->event, plip_event);

	plip->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), plip->dev_name);
	return 0;

	ret6:
	KERNEL$FAST_CLOSE(plip->packet_input);
	ret4:
	NETQUE$FREE_QUEUE(plip->queue);
	ret3:
	NET$PKTPOOL_FREE(&plip->pool);
	ret2:
	WQ_WAKE_ALL(&plip->link_state_wait);
	if (plip->acquired) plip->client.op->set_data(&plip->client, 0xff);
	KERNEL$DCALL(plip->pio_name, "PIO", PIO_CMD_DETACH, &plip->client);
	ret1:
	free(plip);
	ret0:
	return r;
}

static int PLIP_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	PLIP *plip = p;
	if ((r = KERNEL$DEVICE_UNLOAD(plip->lnte, argv))) return r;
	RAISE_SPL(SPL_PLIP);
	NETQUE$DISCARD_PACKETS(plip->queue, -ENOLNM);
	if (__likely(plip->irq_event != NULL)) MTX_SET(plip->irq_event, 0);
	plip->terminate = 1;
	while (plip->terminate == 1) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SLEEP(1);
		RAISE_SPL(SPL_PLIP);
	}
	KERNEL$DEL_TIMER(&plip->timer);
	free_packets(plip);
	while (plip->outstanding) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SLEEP(1);
		RAISE_SPL(SPL_PLIP);
	}
	LOWER_SPL(SPL_ZERO);
	KERNEL$FAST_CLOSE(plip->packet_input);
	NETQUE$FREE_QUEUE(plip->queue);
	WQ_WAKE_ALL(&plip->link_state_wait);
	NET$PKTPOOL_FREE(&plip->pool);
	if (plip->acquired) plip->client.op->set_data(&plip->client, 0xff);
	KERNEL$DCALL(plip->pio_name, "PIO", PIO_CMD_DETACH, &plip->client);
	*release = plip->dlrq;
	free(plip);
	return 0;
}
