#include <SPAD/LIBC.H>
#include <SYS/TYPES.H>
#include <STDLIB.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/PCI.H>
#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <ARCH/TIME.H>
#include <SPAD/TIMER.H>
#include <ENDIAN.H>
#include <SPAD/PKT.H>
#include <SPAD/ETHERNET.H>
#include <NETINET/IP.H>
#include <SPAD/IOCTL.H>
#include <SPAD/RANDOM.H>
#include <VALUES.H>

#include "SK.H"

static const struct pci_id_s pci_cards[] = {

/* SKGE */
	{0x1148, 0x4300, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SYSKONNECT GENESIS", 0 },
/* SKY1 */
	{0x10b7, 0x1700, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C940", 0 },
	{0x10b7, 0x80eb, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C940B", 0 },
	{0x1148, 0x4320, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SYSKONNECT YUKON", 0 },
	{0x1186, 0x4b01, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DGE530-T", 0 },
	{0x1186, 0x4c00, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DGE560-T", 0 },
	{0x11ab, 0x4320, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL YUKON", 0 },
	{0x11ab, 0x5005, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "BELKIN F5D5005", 0 },
	{0x1371, 0x434e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "CNET GIGACARD", 0 },
	{0x1737, 0x1032, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "LINKSYS EG1032", 0 },
	{0x1737, 0x1064, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "LINKSYS EG1064", 0 },
/* SKY2 */
	{0x1148, 0x9000, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SYSKONNECT SK-9S", 0 },
	{0x1148, 0x9e00, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SYSKONNECT SK-9E", 0 },
	{0x1186, 0x4b00, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DGE530-T", 0 },
	{0x1186, 0x4001, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DGE550-SX", 0 },
	{0x1186, 0x4b02, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DGE560-SX", 0 },
	{0x1186, 0x4b03, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DGE550-T", 0 },
	{0x11ab, 0x4340, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8021", 0 },
	{0x11ab, 0x4341, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8022", 0 },
	{0x11ab, 0x4342, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8061", 0 },
	{0x11ab, 0x4343, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8062", 0 },
	{0x11ab, 0x4344, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8021", 0 },
	{0x11ab, 0x4345, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8022", 0 },
	{0x11ab, 0x4346, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8061", 0 },
	{0x11ab, 0x4347, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8062", 0 },
	{0x11ab, 0x4350, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8035", 0 },
	{0x11ab, 0x4351, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8035", 0 },
	{0x11ab, 0x4352, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8038", 0 },
	{0x11ab, 0x4353, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8039", 0 },
	{0x11ab, 0x4354, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8040", 0 },
	{0x11ab, 0x4355, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8040-T", 0 },
	{0x11ab, 0x4356, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88EC033", 0 },
	{0x11ab, 0x4357, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8042", 0 },
	{0x11ab, 0x435a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8048", 0 },
	{0x11ab, 0x4360, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8052", 0 },
	{0x11ab, 0x4361, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8050", 0 },
	{0x11ab, 0x4362, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8053", 0 },
	{0x11ab, 0x4363, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8055", 0 },
	{0x11ab, 0x4364, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8056", 0 },
	{0x11ab, 0x4365, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8070", 0 },
	{0x11ab, 0x4366, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88EC036", 0 },
	{0x11ab, 0x4367, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88EC032", 0 },
	{0x11ab, 0x4368, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88EC034", 0 },
	{0x11ab, 0x4369, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88EC042", 0 },
	{0x11ab, 0x436a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8058", 0 },
	{0x11ab, 0x436b, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8071", 0 },
	{0x11ab, 0x436c, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8072", 0 },
	{0x11ab, 0x436d, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8055", 0 },
	{0x11ab, 0x4370, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8075", 0 },
	{0x11ab, 0x4380, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "MARVELL 88E8057", 0 },
	{ 0 },
};

#define SK_PORTLOCKED		1	/* flags in HANDLE */

/* XMAC PHY IO */
__u16 XM_PHY_READ(MAC *mac, __u8 reg)
{
	__u16 val;
	XM_WRITE_16(mac, XM_PHY_ADDR, reg | mac->sk->phyaddr);
	val = XM_READ_16(mac, XM_PHY_DATA);
	if (__unlikely(mac->sk->phytype != SK_EPROM_1_PHY_XMAC)) {
		u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj = j;
		while (!(XM_READ_16(mac, XM_MMU_CMD) & XM_MMU_PHY_RDY)) {
			if (__unlikely(jj - j > TIMEOUT_JIFFIES(PHY_TIMEOUT))) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "XMACII PHY READ TIMEOUT, PORT %c", PORT_ID(mac));
				return 0;
			}
			jj = KERNEL$GET_JIFFIES_LO();
		}
		val = XM_READ_16(mac, XM_PHY_DATA);
	}
	return val;
}

void XM_PHY_WRITE(MAC *mac, __u8 reg, __u16 val)
{
	if (__unlikely(mac->sk->phytype != SK_EPROM_1_PHY_XMAC)) {
		u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj = j;
		while (__unlikely((XM_READ_16(mac, XM_MMU_CMD) & XM_MMU_PHY_BUSY))) {
			if (__unlikely(jj - j > TIMEOUT_JIFFIES(PHY_TIMEOUT))) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "XMACII PHY TIMEOUT BEFORE WRITE, PORT %c", PORT_ID(mac));
				return;
			}
			jj = KERNEL$GET_JIFFIES_LO();
		}
	}
	XM_WRITE_16(mac, XM_PHY_ADDR, reg | mac->sk->phyaddr);
	XM_WRITE_16(mac, XM_PHY_DATA, val);
	if (__unlikely(mac->sk->phytype != SK_EPROM_1_PHY_XMAC)) {
		u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj = j;
		while ((XM_READ_16(mac, XM_MMU_CMD) & XM_MMU_PHY_BUSY)) {
			if (__unlikely(jj - j > TIMEOUT_JIFFIES(PHY_TIMEOUT))) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "XMACII PHY TIMEOUT AFTER WRITE, PORT %c", PORT_ID(mac));
				return;
			}
			jj = KERNEL$GET_JIFFIES_LO();
		}
	}
}

/* GMAC PHY IO */

__u16 GM_PHY_READ(MAC *mac, __u8 reg)
{
	u_jiffies_lo_t j;
	__u16 val = (PHY_ADDR_MARV << GM_SMI_CT_PHY_A_MSK_SHIFT) | (reg << GM_SMI_CT_REG_A_MSK_SHIFT) | GM_SMI_CT_OP_RD;
	GM_WRITE_16(mac, GM_SMI_CTRL, val);
	j = KERNEL$GET_JIFFIES_LO();
	while (1) {
		u_jiffies_lo_t jj = KERNEL$GET_JIFFIES_LO();
		__u16 r = GM_READ_16(mac, GM_SMI_CTRL);
		if (__unlikely(r == 0xffff)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "UNABLE TO READ SMI REGISTER WHILE READING FROM GMAC PHY, PORT %c", PORT_ID(mac));
			return 0;
		}
		if (__unlikely(!(r & GM_SMI_CT_OP_RD))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "GMAC PHY READ IMPOSSIBLE, PORT %c, REG %02X", PORT_ID(mac), reg);
			return 0;
		}
		if ((r ^ val) == GM_SMI_CT_RD_VAL) break;
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(PHY_TIMEOUT))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "GMAC PHY READ TIMEOUT, PORT %c, REG %02X", PORT_ID(mac), reg);
			return 0;
		}
		KERNEL$UDELAY(10);
	}
	val = GM_READ_16(mac, GM_SMI_DATA);
	SK_READ_32(mac, SK_GMAC_TS_VALUE);
	return val;
}

void GM_PHY_WRITE(MAC *mac, __u8 reg, __u16 val)
{
	u_jiffies_lo_t j;
	GM_WRITE_16(mac, GM_SMI_DATA, val);
	if (__unlikely(GM_READ_16(mac, GM_SMI_DATA) != val)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "GMAC PHY WRITE IMPOSSIBLE, PORT %c, REG %02X, VAL %04X", PORT_ID(mac), reg, val);
		return;
	}
	GM_WRITE_16(mac, GM_SMI_CTRL, (PHY_ADDR_MARV << GM_SMI_CT_PHY_A_MSK_SHIFT) | (reg << GM_SMI_CT_REG_A_MSK_SHIFT));
	j = KERNEL$GET_JIFFIES_LO();
	while (1) {
		u_jiffies_lo_t jj = KERNEL$GET_JIFFIES_LO();
		__u16 r = GM_READ_16(mac, GM_SMI_CTRL);
		if (__unlikely(r == 0xffff)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "UNABLE TO READ SMI REGISTER WHILE WRITING TO GMAC PHY, PORT %c, REG %02X, VAL %04X", PORT_ID(mac), reg, val);
			return;
		}
		if (!(r & GM_SMI_CT_BUSY)) break;
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(PHY_TIMEOUT))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, mac->sk->dev_name, "GMAC PHY WRITE TIMEOUT, PORT %c, REG %02X, VAL %04X", PORT_ID(mac), reg, val);
			return;
		}
		KERNEL$UDELAY(10);
	}
	SK_READ_32(mac, SK_GMAC_TS_VALUE);
}

#define SK_TESTMODE_ON(sk)	SK_WRITE_8((sk), SK_TEST_CTRL_1, SK_TEST_CTRL_1_EN_CONFIG_ON)
#define SK_TESTMODE_OFF(sk)	SK_WRITE_8((sk), SK_TEST_CTRL_1, SK_TEST_CTRL_1_EN_CONFIG_OFF)

void SK_BASIC_RESET(SK *sk)
{
	unsigned ctrl_stat, pci_status;
	__u8 rev;

	SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + 0x80, 0);
	SK_WRITE_8(sk, SK_CS, SK_CS_SW_RESET_CLEAR);

	rev = SK_READ_8(sk, SK_CHIP_REVISION);
	ctrl_stat = SK_READ_16(sk, SK_CS);
	if (Y2_REV(rev)) {
		if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EX) {
			__u16 st = SK_READ_16(sk, SK_Y2_CCSR);
			st &= ~(SK_Y2_CCSR_AHB_RST | SK_Y2_CCSR_CPU_RST_MODE | SK_Y2_CCSR_UC_STATE_MSK);
			SK_WRITE_16(sk, SK_Y2_CCSR, st);
		} else {
			SK_WRITE_32(sk, SK_ASF_STAT_CMD, SK_ASF_STAT_CMD_RESET);
		}
		SK_WRITE_16(sk, SK_CS, SK_Y2_CS_ASF_DISABLE);
	}
	SK_WRITE_8(sk, SK_CS, SK_CS_SW_RESET_SET);
	KERNEL$UDELAY(10);
	SK_WRITE_8(sk, SK_CS, SK_CS_SW_RESET_CLEAR);
	pci_status = SK_READ_16(sk, (Y2_REV(rev) ? SK_Y2_CONFIGURATION_FILE : SK_CONFIGURATION_FILE) + PCI_STATUS);
	SK_TESTMODE_ON(sk);
	SK_WRITE_16(sk, (Y2_REV(rev) ? SK_Y2_CONFIGURATION_FILE : SK_CONFIGURATION_FILE) + PCI_STATUS, pci_status | PCI_STATUS_DETECTED_PARITY | PCI_STATUS_SIG_SYSTEM_ERROR | PCI_STATUS_REC_MASTER_ABORT | PCI_STATUS_REC_TARGET_ABORT | PCI_STATUS_PARITY);
	SK_TESTMODE_OFF(sk);
	SK_WRITE_8(sk, SK_CS, SK_CS_MASTER_RESET_CLEAR);
	if (!Y2_REV(rev)) {
		SK_WRITE_16(sk, SK_CS, ctrl_stat & (SK_CS_CLK_RUN_ENA | SK_CS_CLK_RUN_RST | SK_CS_CLK_RUN_HOT));
	} else {
		__u32 our_status = SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + PCI_SK_OUR_STATUS);
		if (!(our_status & PCI_SK_OUR_STATUS_PCI_X)) {
			SK_TESTMODE_ON(sk);
			SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + PCI_SK_PEX_UNC_ERR_STAT, 0xffffffffU);
			SK_TESTMODE_OFF(sk);
		}
	}
}

static void SK_GET_LINK(MAC *mac);
static void XM_SET_MULTICAST_HASH(MAC *mac);
static void GM_SET_MULTICAST_HASH(MAC *mac);

void SK_CONTINUE_RESET(SK *sk)
{
	int i, j, k, l;
	int rambuffers;
	__u32 imr, hwimr;

	/* reset GMAC link */
	if (sk->chiptype != CHIPTYPE_GENESIS) {
		for (i = 0; i < sk->n_macs; i++) {
			SK_WRITE_16(sk, SK_GMAC_LINK_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_LINK_CTRL_RESET_SET);
			SK_WRITE_16(sk, SK_GMAC_LINK_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_LINK_CTRL_RESET_CLEAR);
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EX ||
			    sk->chipsubtype == CHIPSUBTYPE_YUKON_SUPR) {
				SK_WRITE_16(sk, SK_GMAC_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_Y2_GMAC_CTRL_BRETR_ON | SK_Y2_GMAC_CTRL_BMACSEC_TX_ON | SK_Y2_GMAC_CTRL_BMACSEC_RX_ON);
			}
		}
	}

	/* start descriptor poll timer */
	SK_WRITE_32(sk, SK_DP_TIMER_INIT_VAL, 0x00f00000);
	SK_WRITE_8(sk, SK_DP_TIMER_CTRL, SK_DP_TIMER_CTRL_START);

	if (sk->chiptype == CHIPTYPE_GENESIS) {
		/* start blink source counter */
		SK_WRITE_32(sk, SK_BLINK_CNTR_INIT_VAL, sk->host_clock * BLINK_SOURCE_TIME);
		SK_WRITE_8(sk, SK_BLINK_CNTR_CONTROL, SK_BLINK_CNTR_CONTROL_START);

		/* initialize MAC arbiter */
		SK_WRITE_16(sk, SK_MA_CTRL, SK_MA_CTRL_RESET_CLEAR);
		SK_WRITE_8(sk, SK_MA_TIMEOUT_INIT_RX, MAC_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_8(sk, SK_MA_TIMEOUT_INIT_RX + ADD_MAC2_SK_MA_TIMEOUT_TIMER, MAC_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_8(sk, SK_MA_TIMEOUT_INIT_TX, MAC_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_8(sk, SK_MA_TIMEOUT_INIT_TX + ADD_MAC2_SK_MA_TIMEOUT_TIMER, MAC_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_8(sk, SK_MA_RECOVERY_INIT_RX, 0);
		SK_WRITE_8(sk, SK_MA_RECOVERY_INIT_RX + ADD_MAC2_SK_MA_RECOVERY_TIMER, 0);
		SK_WRITE_8(sk, SK_MA_RECOVERY_INIT_TX, 0);
		SK_WRITE_8(sk, SK_MA_RECOVERY_INIT_TX + ADD_MAC2_SK_MA_RECOVERY_TIMER, 0);

		/* initialize packet arbiter */
		SK_WRITE_16(sk, SK_PA_CTRL, SK_PA_RESET_CLEAR);
		SK_WRITE_16(sk, SK_PA_TIMEOUT_INIT_RX, PKT_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_16(sk, SK_PA_TIMEOUT_INIT_RX + ADD_MAC2_SK_PA_TIMEOUT, PKT_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_16(sk, SK_PA_TIMEOUT_INIT_TX, PKT_ARBITER_TIMEOUT_AT_52);
		SK_WRITE_16(sk, SK_PA_TIMEOUT_INIT_TX + ADD_MAC2_SK_PA_TIMEOUT, PKT_ARBITER_TIMEOUT_AT_52);
		/* enable if jumbo frames not used */
		if (sk->macs[0].mtu <= ETHERNET_MTU && (sk->n_macs == 1 || sk->macs[1].mtu <= ETHERNET_MTU)) {
			SK_WRITE_16(sk, SK_PA_CTRL, SK_PA_CTRL_EN_TIMEOUT_TX1_ON | (sk->n_macs == 2 ? SK_PA_CTRL_EN_TIMEOUT_TX2_ON : 0));
		}
	} else {
		/* time stamp timer */
		SK_WRITE_8(sk, SK_GMAC_TS_CTRL, SK_GMAC_TS_CTRL_START);
	}
	for (i = 0; i < sk->n_macs; i++) {
		MAC *mac = &sk->macs[i];
		/* initialize tx arbiters */
		if (mac->queues == 2) {
			SK_WRITE_32(sk, SK_TA_INT_TIMER_INIT + i * ADD_MAC2_TRANSMIT_ARBITER, TX_ARBITER_TIMER * sk->host_clock);
			SK_WRITE_32(sk, SK_TA_LIMIT_CNT_INIT + i * ADD_MAC2_TRANSMIT_ARBITER, TX_ARBITER_LIMIT);
			SK_WRITE_8(sk, SK_TA_CTRL + i * ADD_MAC2_TRANSMIT_ARBITER, SK_TA_CTRL_FORCE_SYNC_OFF | SK_TA_CTRL_EN_ALLOC | SK_TA_CTRL_RATE_CONTROL_START | SK_TA_CTRL_OPERATIONAL_ON);
		} else {
			SK_WRITE_8(sk, SK_TA_CTRL + i * ADD_MAC2_TRANSMIT_ARBITER, SK_TA_CTRL_OPERATIONAL_ON);
		}
	}
	/* initialize ram interface arbiter */
	if (sk->chiptype == CHIPTYPE_YUKON_2) {
		rambuffers = sk->n_macs;
	} else {
		rambuffers = 1;
	}
	for (i = 0; i < rambuffers; i++) {
		SK_WRITE_8(sk, SK_RI_CTRL + i * ADD_RB2_SK_RI_CTRL, SK_RI_CTRL_RESET_CLEAR);
		for (j = 0; j < 2; j++) {
			SK_WRITE_8(sk, SK_RI_TIMEOUT_WRITE_RX + i * ADD_RB2_SK_RI_CTRL + j * ADD_MAC2_SK_RI_TIMEOUT, RI_ARBITER_TIMEOUT_AT_52);
			SK_WRITE_8(sk, SK_RI_TIMEOUT_WRITE_TXA + i * ADD_RB2_SK_RI_CTRL + j * ADD_MAC2_SK_RI_TIMEOUT, RI_ARBITER_TIMEOUT_AT_52);
			SK_WRITE_8(sk, SK_RI_TIMEOUT_WRITE_TXS + i * ADD_RB2_SK_RI_CTRL + j * ADD_MAC2_SK_RI_TIMEOUT, RI_ARBITER_TIMEOUT_AT_52);
			SK_WRITE_8(sk, SK_RI_TIMEOUT_READ_RX + i * ADD_RB2_SK_RI_CTRL + j * ADD_MAC2_SK_RI_TIMEOUT, RI_ARBITER_TIMEOUT_AT_52);
			SK_WRITE_8(sk, SK_RI_TIMEOUT_READ_TXA + i * ADD_RB2_SK_RI_CTRL + j * ADD_MAC2_SK_RI_TIMEOUT, RI_ARBITER_TIMEOUT_AT_52);
			SK_WRITE_8(sk, SK_RI_TIMEOUT_READ_TXS + i * ADD_RB2_SK_RI_CTRL + j * ADD_MAC2_SK_RI_TIMEOUT, RI_ARBITER_TIMEOUT_AT_52);
		}
	}
	if (sk->chiptype == CHIPTYPE_YUKON_2) {
		memset(&sk->y2_status, 0, sizeof sk->y2_status);
		sk->stat_sum_1 = sk->stat_sum_2 = 0;
		sk->first_stat = 0;
		/* initialize status BMU */
		SK_WRITE_32(sk, SK_STAT_CTRL, SK_STAT_CTRL_RESET_SET);
		SK_WRITE_32(sk, SK_STAT_CTRL, SK_STAT_CTRL_RESET_CLEAR);
		SK_WRITE_32(sk, SK_STAT_LIST_ADDR_LO, (__u32)sk->stat_dma64addr);
		SK_WRITE_32(sk, SK_STAT_LIST_ADDR_HI, sk->stat_dma64addr >> 32);
		SK_WRITE_16(sk, SK_STAT_LAST_IDX, N_STATUS_DESCRIPTORS - 1);
		if (Y2_BUG_43_418(sk)) {
			SK_WRITE_16(sk, SK_STAT_TX_IDX_TH, 0xfff);
			SK_WRITE_8(sk, SK_STAT_FIFO_WM, 0x21);
			SK_WRITE_8(sk, SK_STAT_FIFO_ISR_WM, 0x7);
			SK_WRITE_32(sk, SK_STAT_TX_TIMER_INI, 10 * sk->host_clock);
		} else {
	/* This prevents BMU hangs. No one knows why. Should be
	   checked against more recent version of Linux driver */
			SK_WRITE_16(sk, SK_STAT_TX_IDX_TH, 0xa);
			SK_WRITE_8(sk, SK_STAT_FIFO_WM, 0x10);
			if (Y2_BUG_4109(sk)) {
				SK_WRITE_8(sk, SK_STAT_FIFO_ISR_WM, 0x4);
			} else {
				SK_WRITE_8(sk, SK_STAT_FIFO_ISR_WM, 0x10);
			}
			SK_WRITE_32(sk, SK_STAT_TX_TIMER_INI, 1000 * sk->host_clock);
		}
		SK_WRITE_32(sk, SK_STAT_ISR_TIMER_INI, 20 * sk->host_clock);
		SK_WRITE_32(sk, SK_STAT_LEV_TIMER_INI, 100 * sk->host_clock);
		SK_WRITE_32(sk, SK_STAT_CTRL, SK_STAT_CTRL_OPERATIONAL_ON);
		SK_WRITE_8(sk, SK_STAT_TX_TIMER_CTRL, SK_STAT_TX_TIMER_CTRL_START);
		SK_WRITE_8(sk, SK_STAT_LEV_TIMER_CTRL, SK_STAT_LEV_TIMER_CTRL_START);
		SK_WRITE_8(sk, SK_STAT_ISR_TIMER_CTRL, SK_STAT_ISR_TIMER_CTRL_START);
	}
	/* initialization of ports */
	for (i = 0; i < 2; i++) {
		MAC *mac = &sk->macs[i];
		unsigned ramsize;
		unsigned ramoffs;
		unsigned csize;
		__u16 rxwm, txwm;
		unsigned n_tmp_recv_packets, n;
		if (i >= sk->n_macs) {
			SK_WRITE_8(sk, SK_RRB_CTRL + i * ADD_MAC2_SK_RRB, SK_RRB_CTRL_RESET_SET);
			SK_WRITE_8(sk, SK_TRB_CTRL + i * ADD_MAC2_SK_TRB, SK_TRB_CTRL_RESET_SET);
			SK_WRITE_8(sk, SK_TRB_CTRL + i * ADD_MAC2_SK_TRB + ADD_ASYNC_SK_TRB, SK_TRB_CTRL_RESET_SET);
			continue;
		}
		if (sk->chiptype == CHIPTYPE_GENESIS) {
			/* initialize rx LED */
			SK_WRITE_32(sk, SK_RX_LED_CNT_INIT_VAL + i * ADD_MAC2_SK_RMF, LED_TIME * sk->host_clock);
			SK_WRITE_8(sk, SK_RX_LED_CNT_CTRL + i * ADD_MAC2_SK_RMF, SK_RX_LED_CNT_CTRL_START);
			/* initialize tx LED */
			SK_WRITE_32(sk, SK_TX_LED_CNT_INIT_VAL + i * ADD_MAC2_SK_TMF, LED_TIME * sk->host_clock);
			SK_WRITE_8(sk, SK_TX_LED_CNT_CTRL + i * ADD_MAC2_SK_TMF, SK_TX_LED_CNT_CTRL_START);

			/* init XMAC */

			/* clear XMAC reset */
			SK_WRITE_16(sk, SK_TMF_MAC_CTRL + i * ADD_MAC2_SK_TMF, SK_TMF_MAC_CTRL_RESET_CLEAR);
			if (sk->phytype != SK_EPROM_1_PHY_XMAC) {
				__u32 gp = SK_READ_32(sk, SK_GENERAL_PURPOSE);
				gp |= 0x101 << (i << 1);
				SK_WRITE_32(sk, SK_GENERAL_PURPOSE, gp);
				XM_WRITE_16(mac, XM_HW_CFG, XM_HW_GMII_MD);
				mac->phyid1 = XM_PHY_READ(mac, PHY_XMAC_ID1);
				XM_WRITE_16(mac, XM_MMU_CMD, XM_READ_16(mac, XM_MMU_CMD) | XM_MMU_NO_PRE);
				switch (mac->phyid1) {
					case PHY_BCOM_ID1_C0:
					XM_PHY_WRITE(mac,0x18,0x0c20);
					XM_PHY_WRITE(mac,0x17,0x0012);
					XM_PHY_WRITE(mac,0x15,0x1104);
					XM_PHY_WRITE(mac,0x17,0x0013);

					XM_PHY_WRITE(mac,0x15,0x0404);
					XM_PHY_WRITE(mac,0x17,0x8006);
					XM_PHY_WRITE(mac,0x15,0x0132);
					XM_PHY_WRITE(mac,0x17,0x8006);

					XM_PHY_WRITE(mac,0x15,0x0232);
					XM_PHY_WRITE(mac,0x17,0x800D);
					XM_PHY_WRITE(mac,0x15,0x000F);
					XM_PHY_WRITE(mac,0x18,0x0420);
					break;

					case PHY_BCOM_ID1_A1:
					XM_PHY_WRITE(mac,0x18,0x0c20);
					XM_PHY_WRITE(mac,0x17,0x0012);
					XM_PHY_WRITE(mac,0x15,0x1204);
					XM_PHY_WRITE(mac,0x17,0x0013);
					XM_PHY_WRITE(mac,0x15,0x0A04);
					XM_PHY_WRITE(mac,0x18,0x0420);
					break;
				}
				XM_PHY_WRITE(mac, PHY_BCOM_AUX_CTRL, XM_PHY_READ(mac, PHY_BCOM_AUX_CTRL) | PHY_B_AC_DIS_PM);
			}
			XM_READ_16(mac, XM_ISRC);

			/* init XMAC PHY */
			if (sk->phytype == SK_EPROM_1_PHY_XMAC) {
				if (mac->duplex == DUPLEX_FULL) {
					XM_PHY_WRITE(mac, PHY_XMAC_CTRL, PHY_CT_DUP_MD);
				} else if (mac->duplex == DUPLEX_HALF) {
					XM_PHY_WRITE(mac, PHY_XMAC_CTRL, 0);
				} else {
					XM_PHY_WRITE(mac, PHY_XMAC_AUNE_ADV, PHY_X_AN_FD | PHY_X_P_BOTH_MD);
					XM_PHY_WRITE(mac, PHY_XMAC_CTRL, PHY_CT_ANE | PHY_CT_RE_CFG);
				}
			} else if (sk->phytype == SK_EPROM_1_PHY_BCOM) {
				__u16 ctrl1 = PHY_CT_SP1000;
				__u16 ctrl2 = 0;
				__u16 ctrl3 = PHY_SEL_TYPE;
				__u16 ctrl4 = PHY_B_PEC_EN_LTR;
				__u16 ctrl5 = PHY_B_AC_TX_TST;
				if (mac->duplex == DUPLEX_FULL) {
					ctrl1 |= PHY_CT_DUP_MD;
				} else if (mac->duplex == DUPLEX_HALF) {
				} else {
					ctrl2 |= PHY_B_1000C_RD | PHY_B_1000C_AFD;
					ctrl3 |= PHY_B_P_BOTH_MD;
					ctrl1 |= PHY_CT_ANE | PHY_CT_RE_CFG;
				}
				XM_PHY_WRITE(mac, PHY_BCOM_1000T_CTRL, ctrl2);
				XM_PHY_WRITE(mac, PHY_BCOM_AUNE_ADV, ctrl3);
				if (mac->mtu > ETHERNET_MTU) {
					ctrl4 |= PHY_B_PEC_HIGH_LA;
					ctrl5 |= PHY_B_AC_LONG_PACK;
					XM_PHY_WRITE(mac, PHY_BCOM_AUX_CTRL, ctrl5);
				}
				XM_PHY_WRITE(mac, PHY_BCOM_P_EXT_CTRL, ctrl4);
				XM_PHY_WRITE(mac, PHY_BCOM_CTRL, ctrl1);
			}

			/* setup MAC address */
			if (!mac->force_address) {
				mac->address[0] = SK_READ_16(mac, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR);
				mac->address[1] = SK_READ_16(mac, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 2);
				mac->address[2] = SK_READ_16(mac, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 4);
				mac->force_address = 1;
			}
			/*if (mac->force_address == 2) {
				SK_TESTMODE_ON(sk);
				SK_WRITE_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR, mac->address[0]);
				SK_WRITE_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 2, mac->address[1]);
				SK_WRITE_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 4, mac->address[2]);
				SK_TESTMODE_OFF(sk);
			}*/
			XM_WRITE_16(mac, XM_SA, mac->address[0]);
			XM_WRITE_16(mac, XM_SA + 2, mac->address[1]);
			XM_WRITE_16(mac, XM_SA + 4, mac->address[2]);

			XM_WRITE_16(mac, XM_RX_HI_WM, SK_XM_RX_HI_WM);
			XM_WRITE_16(mac, XM_TX_THR, mac->mtu > ETHERNET_MTU ? SK_XM_THR_JUMBO : sk->n_macs == 1 ? SK_XM_THR_SL : SK_XM_THR_MULL);
			XM_WRITE_16(mac, XM_TX_CMD, XM_TX_AUTO_PAD);
			XM_WRITE_16(mac, XM_RX_CMD, XM_RX_STRIP_FCS | XM_RX_LENERR_OK | (mac->mtu > ETHERNET_MTU ? XM_RX_BIG_PK_OK : 0) | (mac->duplex == DUPLEX_HALF ? XM_RX_DIS_CEXT : 0));
			XM_WRITE_32(mac, XM_MODE, XM_MD_RX_RUNT | XM_MD_RX_IRLE | XM_MD_RX_LONG | XM_MD_RX_CRCE | XM_MD_RX_ERR | XM_MD_CSA | XM_MD_CAA);
			XM_SET_MULTICAST_HASH(mac);
			XM_WRITE_32(mac, XM_RX_EV_MSK, XMR_OK_LO_OV | XMR_OK_HI_OV);
			XM_WRITE_32(mac, XM_TX_EV_MSK, XMT_OK_LO_OV | XMT_OK_HI_OV);
			
			/* initialize rx MAC fifo */
			SK_WRITE_8(sk, SK_RMF_CTRL + i * ADD_MAC2_SK_RMF, SK_RMF_CTRL_RESET_CLEAR);
			SK_WRITE_16(sk, SK_RMF_MAC_CTRL + i * ADD_MAC2_SK_RMF, SK_RMF_MAC_CTRL_RXVAL_PTCH_ON | (mac->mtu > GENESIS_FIFO_SIZE ? SK_RMF_MAC_CTRL_EN_FLUSHRX_ON : 0));
			SK_WRITE_8(sk, SK_RMF_CTRL + i * ADD_MAC2_SK_RMF, SK_RMF_CTRL_OPERATIONAL_ON);
			/* initialize tx MAC fifo */
			SK_WRITE_8(sk, SK_TMF_CTRL + i * ADD_MAC2_SK_TMF, SK_TMF_CTRL_RESET_CLEAR);
			SK_WRITE_16(sk, SK_TMF_MAC_CTRL + i * ADD_MAC2_SK_TMF, SK_TMF_MAC_CTRL_EN_RECOV_ON | SK_TMF_MAC_CTRL_TXRDY_PTCH_ON | SK_TMF_MAC_CTRL_EN_FLUSHTX_ON);
			SK_WRITE_8(sk, SK_TMF_CTRL + i * ADD_MAC2_SK_TMF, SK_TMF_CTRL_OPERATIONAL_ON);
		} else {
			__u16 gp, ser, ctrl, c1000baset, ana, gpa, led_ctrl, led_over;
			/* initialize GMAC */
			/* reset GMAC */
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE_A3) {
				__u32 gp = SK_READ_32(sk, SK_GENERAL_PURPOSE);
				gp |= (1 << (9 + 16)) | (1 << 9);
				SK_WRITE_32(sk, SK_GENERAL_PURPOSE, gp);
			}
			SK_WRITE_8(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_SET);
			SK_WRITE_8(sk, SK_GMAC_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_CTRL_RESET_SET);
			/* clear GMAC reset */
			if (sk->chiptype == CHIPTYPE_YUKON_2) {
				SK_WRITE_8(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_CLEAR);
				SK_WRITE_8(sk, SK_GMAC_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_CTRL_RESET_CLEAR);
			} else {
				__u32 rst_data;
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE_A3) {
					__u32 gp = SK_READ_32(sk, SK_GENERAL_PURPOSE);
					gp |= 1 << (9 + 16);
					gp &= ~(1 << 9);
					SK_WRITE_32(sk, SK_GENERAL_PURPOSE, gp);
				}
				rst_data = SK_GPHY_CTRL_INT_POL_HI | SK_GPHY_CTRL_DIS_FC | SK_GPHY_CTRL_DIS_SLEEP | SK_GPHY_CTRL_ENA_XC | SK_GPHY_CTRL_ANEG_ADV_ALL_M | SK_GPHY_CTRL_ENA_PAUSE;
				if (sk->copper) {
					rst_data |= SK_GPHY_CTRL_HWCFG_GMII_COP;
				} else {
					rst_data |= SK_GPHY_CTRL_HWCFG_GMII_FIB;
				}
				SK_WRITE_32(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, rst_data | SK_GPHY_CTRL_RESET_SET);
				SK_WRITE_32(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, rst_data | SK_GPHY_CTRL_RESET_CLEAR);
				SK_WRITE_32(sk, SK_GMAC_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_CTRL_RESET_CLEAR | SK_GMAC_CTRL_PAUSE_ON);
			}
			if (Y2_BUG_472(sk) && i) {
				u_jiffies_lo_t j1, j2;
				j1 = j2 = KERNEL$GET_JIFFIES_LO();
				SK_WRITE_8(sk, SK_GPHY_CTRL + 0 * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_CLEAR);
				do {
					if (j2 - j1 > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 10 + 1)) {
						KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PHY RESET DOESN'T STABILIZE");
						break;
					}
					j2 = KERNEL$GET_JIFFIES_LO();
					SK_WRITE_8(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_SET);
					SK_WRITE_8(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_CLEAR);
				} while (GM_PHY_READ(mac, PHY_MARV_ID0) != PHY_MARV_ID0_VAL ||
					GM_PHY_READ(mac, PHY_MARV_ID1) != PHY_MARV_ID1_Y2 ||
					GM_PHY_READ(mac, PHY_MARV_INT_MASK));
			}

			/* sky2_phy_power_up */
			if (sk->chiptype == CHIPTYPE_YUKON_2) {
				__u32 x;
				SK_TESTMODE_ON(sk);
				x = SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + 0x40);
				x &= ~(1 << (26 + i));
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_XL && sk->chiprev > SK_CONFIGURATION_XL_1) x |= 1 << (28 + i);
				SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + 0x40, x);
				SK_TESTMODE_OFF(sk);
				SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + 0x40);
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_FE) {
					GM_PHY_WRITE(mac, PHY_MARV_CTRL, PHY_CT_ANE);
				} else if (ADV_POWER_CTL(sk)) {
					SK_WRITE_8(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_CLEAR);
				}
			}

			SK_READ_16(sk, SK_GMAC_IRQ_SRC + i * ADD_MAC2_SK_GMAC_GPHY);
			/* initialize Marvell phy */
			if (mac->duplex == DUPLEX_AUTO && !NEWER_PHY(sk)) {
				__u16 ext = GM_PHY_READ(mac, PHY_MARV_EXT_CTRL);
				ext &= ~(PHY_M_EC_M_DSC_MSK | PHY_M_EC_S_DSC_MSK | PHY_M_EC_MAC_S_MSK);
				ext |= MAC_TX_CLK_25_MHZ << PHY_M_EC_MAC_S_MSK_SHIFT;
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EC) {
					ext |= (2 << PHY_M_EC_M_DSC_2_SHIFT) | PHY_M_EC_DOWN_S_ENA;
				} else {
					ext |= (0 << PHY_M_EC_M_DSC_MSK_SHIFT) | (1 << PHY_M_EC_S_DSC_MSK_SHIFT);
				}
				GM_PHY_WRITE(mac, PHY_MARV_EXT_CTRL, ext);
			}
			if (sk->chiptype == CHIPTYPE_YUKON_2) {
				__u16 ctrl = GM_PHY_READ(mac, PHY_MARV_PHY_CTRL);
				if (sk->copper) {
					if (!GIGABIT(sk)) {
						/*ctrl &= ~(PHY_M_PC_EN_DET_MSK | PHY_M_PC_MDIX_MSK);*/
						ctrl |= PHY_M_PC_ENA_AUTO << (PHY_M_PC_MDIX_MSK_SHIFT - 1);
						if (sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P && sk->chiprev == SK_CONFIGURATION_FE_P_A0) {
							__u16 spec = GM_PHY_READ(mac, PHY_MARV_FE_SPEC_2);
							spec |= PHY_M_FESC_SEL_CL_A;
							GM_PHY_WRITE(mac, PHY_MARV_FE_SPEC_2, spec);
						}
					} else {
						ctrl &= ~(PHY_M_PC_EN_DET_MSK | PHY_M_PC_MDIX_MSK);
						ctrl |= PHY_M_PC_ENA_AUTO << PHY_M_PC_MDIX_MSK_SHIFT;
						if (mac->duplex == DUPLEX_AUTO && NEWER_PHY(sk)) {
							ctrl &= PHY_M_PC_DSC_MSK;
							ctrl |= (2 << PHY_M_PC_DSC_MSK_SHIFT) | PHY_M_PC_DOWN_S_ENA;
						}
					}
				} else {
					ctrl &= ~PHY_M_PC_MDIX_MSK;
				}
				GM_PHY_WRITE(mac, PHY_MARV_PHY_CTRL, ctrl);
			}
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_XL && !sk->copper) {
				GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 2);
				ctrl = GM_PHY_READ(mac, PHY_MARV_PHY_CTRL);
				ctrl &= ~PHY_M_MAC_MD_MSK;
				ctrl |= PHY_M_MAC_MD_1000BX << PHY_M_MAC_MD_MSK_SHIFT;
				GM_PHY_WRITE(mac, PHY_MARV_PHY_CTRL, ctrl);
				if (sk->media == 'P') {
					GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 1);
					ctrl = GM_PHY_READ(mac, PHY_MARV_PHY_CTRL);
					ctrl |= PHY_M_FIB_SIGD_POL;
					GM_PHY_WRITE(mac, PHY_MARV_PHY_CTRL, ctrl);
				}
				GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 0);
			}
			ctrl = GM_PHY_READ(mac, PHY_MARV_CTRL);
			if (mac->duplex != DUPLEX_AUTO) {
				ctrl &= ~PHY_CT_ANE;
			}
			ctrl |= PHY_CT_RESET;
			GM_PHY_WRITE(mac, PHY_MARV_CTRL, ctrl);
			ctrl = 0;
			c1000baset = 0;
			ana = PHY_SEL_TYPE;
			gp = 0;
			if (mac->duplex != DUPLEX_AUTO) {
				gp |= GM_GPCR_AU_ALL_DIS;
				c1000baset |= PHY_M_1000C_MSE;
				if (mac->duplex == DUPLEX_FULL) {
					ctrl |= PHY_CT_DUP_MD;
					gp |= GM_GPCR_DUP_FULL;
				}
				if (mac->speed == SPEED_1000) {
					ctrl |= PHY_CT_SP1000;
					gp |= GM_GPCR_SPEED_1000;
				} else if (mac->speed == SPEED_100) {
					ctrl |= PHY_CT_SP100;
					gp |= GM_GPCR_SPEED_100;
				}
				SK_WRITE_8(sk, SK_GMAC_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_CTRL_PAUSE_ON);
				ctrl |= PHY_CT_RESET;
			} else {
				if (sk->chiptype == CHIPTYPE_YUKON)
					gp |= GM_GPCR_SPEED_100 | GM_GPCR_SPEED_1000 | GM_GPCR_DUP_FULL;
				if (sk->copper) {
					c1000baset |= PHY_M_1000C_AFD;
					c1000baset |= PHY_M_1000C_AHD;
					ana |= PHY_M_AN_100_FD | PHY_M_AN_10_FD;
					ana |= PHY_M_AN_100_HD | PHY_M_AN_10_HD;
					ana |= PHY_B_P_BOTH_MD;
				} else {
					ana |= PHY_M_AN_1000X_AFD;
					ana |= PHY_M_AN_1000X_AHD;
					ana |= PHY_M_P_BOTH_MD_X;
				}
				ctrl |= PHY_CT_ANE | PHY_CT_RE_CFG;
			}
			GM_WRITE_16(mac, GM_GP_CTRL, gp);
			if (GIGABIT(sk))
				GM_PHY_WRITE(mac, PHY_MARV_1000T_CTRL, c1000baset);
			GM_PHY_WRITE(mac, PHY_MARV_AUNE_ADV, ana);
			GM_PHY_WRITE(mac, PHY_MARV_CTRL, ctrl);

			/* PHY fixups */
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P) {
				ctrl |= PHY_M_PC_ENA_LIP_NP;
				ctrl &= ~(PHY_M_PC_ENA_ENE_DT | PHY_M_PC_DIS_SCRAMB);
				GM_PHY_WRITE(mac, PHY_MARV_CTRL, ctrl);
			}
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EC_U ||
			    sk->chipsubtype == CHIPSUBTYPE_YUKON_UL_2) {
				GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 255);
				GM_PHY_WRITE(mac, 0x18, 0xaa99);
				GM_PHY_WRITE(mac, 0x17, 0x2011);
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EC_U) {
					GM_PHY_WRITE(mac, 0x18, 0xa204);
					GM_PHY_WRITE(mac, 0x17, 0x2002);
				}
				GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 0);
			}
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P &&
			    sk->chiprev == SK_CONFIGURATION_FE_P_A0) {
				GM_PHY_WRITE(mac, PHY_MARV_PAGE_ADDR, 17);
				GM_PHY_WRITE(mac, PHY_MARV_PAGE_DATA, 0x3f60);
			}


			/* LEDs */
			led_ctrl = PULS_170MS << PHY_M_LEDC_PULS_MSK_SHIFT;
			led_over = 0;
			switch (sk->chipsubtype) {
				__u16 l_par;
				case CHIPSUBTYPE_YUKON_FE:
					led_ctrl |= BLINK_340MS << (PHY_M_LEDC_BL_R_MSK_SHIFT + 1);
					l_par = GM_PHY_READ(mac, PHY_MARV_FE_LED_PAR);
					l_par &= ~(0xf << PHY_M_FELP_LED1_CTRL_SHIFT);
					l_par |= LED_PAR_CTRL_ACT_BL << PHY_M_FELP_LED1_CTRL_SHIFT;
					GM_PHY_WRITE(mac, PHY_MARV_FE_LED_PAR, l_par);
					break;
				case CHIPSUBTYPE_YUKON_FE_P:
					l_par = LED_PAR_CTRL_ACT_BL << PHY_M_FELP_LED2_CTRL_SHIFT |
						LED_PAR_CTRL_LINK << PHY_M_FELP_LED1_CTRL_SHIFT |
						LED_PAR_CTRL_SPEED << PHY_M_FELP_LED0_CTRL_SHIFT;
					GM_PHY_WRITE(mac, PHY_MARV_FE_LED_PAR, l_par);
					break;
				case CHIPSUBTYPE_YUKON_XL:
					GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 3);
					GM_PHY_WRITE(mac, PHY_MARV_PHY_CTRL,
						1 << PHY_M_LEDC_LOS_MSK_SHIFT |
						7 << PHY_M_LEDC_INIT_MSK_SHIFT |
						7 << PHY_M_LEDC_STA1_MSK_SHIFT |
						7 << PHY_M_LEDC_STA0_MSK_SHIFT);
					GM_PHY_WRITE(mac, PHY_MARV_PHY_STAT,
						4 << PHY_M_POLC_LS1M_MSK_SHIFT |
						4 << PHY_M_POLC_IS0M_MSK_SHIFT |
						2 << PHY_M_POLC_LOS_MSK_SHIFT |
						2 << PHY_M_POLC_INIT_MSK_SHIFT |
						2 << PHY_M_POLC_STA1_MSK_SHIFT |
						2 << PHY_M_POLC_STA0_MSK_SHIFT);
					GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 0);
					break;
				case CHIPSUBTYPE_YUKON_EC_U:
				case CHIPSUBTYPE_YUKON_EX:
				case CHIPSUBTYPE_YUKON_SUPR:
					GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 3);
					GM_PHY_WRITE(mac, PHY_MARV_PHY_CTRL,
						1 << PHY_M_LEDC_LOS_MSK_SHIFT |
						8 << PHY_M_LEDC_INIT_MSK_SHIFT |
						7 << PHY_M_LEDC_STA1_MSK_SHIFT |
						7 << PHY_M_LEDC_STA0_MSK_SHIFT);
					GM_PHY_WRITE(mac, PHY_MARV_INT_MASK, led_ctrl | BLINK_340MS << PHY_M_LEDC_BL_R_MSK_SHIFT);
					GM_PHY_WRITE(mac, PHY_MARV_EXT_ADR, 0);
					break;
				default:
					if (sk->chiptype == CHIPTYPE_YUKON)
						led_ctrl |= BLINK_84MS << PHY_M_LEDC_BL_R_MSK_SHIFT;
					else
						led_ctrl |= BLINK_340MS << PHY_M_LEDC_BL_R_MSK_SHIFT;
					led_ctrl |= PHY_M_LEDC_TX_CTRL | PHY_M_LEDC_DP_CTRL;
					if (sk->chipsubtype >= CHIPSUBTYPE_YUKON_EC) {
						led_over |= MO_LED_OFF << PHY_M_LED_MO_RX_SHIFT;
					}
					break;
			}
			if (sk->chipsubtype != CHIPSUBTYPE_YUKON_EX &&
			    sk->chipsubtype < CHIPSUBTYPE_YUKON_SUPR) {
				GM_PHY_WRITE(mac, PHY_MARV_LED_CTRL, led_ctrl);
				if (mac->duplex != DUPLEX_AUTO && mac->speed == SPEED_100)
					led_over |= 3 << PHY_M_LED_MO_100_SHIFT;
				if (led_over)
					GM_PHY_WRITE(mac, PHY_MARV_LED_OVER, led_over);
			}

			/* reset GMAC counter */
			gpa = GM_READ_16(mac, GM_PHY_ADDR);
			GM_WRITE_16(mac, GM_PHY_ADDR, gpa | GM_PAR_MIB_CLR);
			for (j = 0; j < GM_MIB_CNT_SIZE; j++) {
				GM_READ_16(mac, GM_MIB_CNT_BASE + 8*i);
			}
			GM_WRITE_16(mac, GM_PHY_ADDR, gpa);

			/* transmit control register */
			GM_WRITE_16(mac, GM_TX_CTRL, TX_COL_DEF << GM_TXCR_COL_THR_MSK_SHIFT);
			/* receive control register */
			GM_WRITE_16(mac, GM_RX_CTRL, GM_RXCR_UCF_ENA | GM_RXCR_CRC_DIS);
			GM_SET_MULTICAST_HASH(mac);
			/* transmit flow control */
			GM_WRITE_16(mac, GM_TX_FLOW_CTRL, 0xffff);
			/* transmit parameter */
			GM_WRITE_16(mac, GM_TX_PARAM, (TX_JAM_LEN_DEF << GM_TXPA_JAMLEN_MSK_SHIFT) | (TX_JAM_IPG_DEF << GM_TXPA_JAMIPG_MSK_SHIFT) | (TX_IPG_JAM_DEF << GM_TXPA_JAMDAT_MSK_SHIFT));
			/* serial mode register */
			ser = (DATA_BLIND_DEF << GM_SMOD_DATABL_MSK_SHIFT) | GM_SMOD_VLAN_ENA | (IPG_DATA_DEF << GM_SMOD_IPG_MSK_SHIFT);
			if (mac->mtu > ETHERNET_MTU) ser |= GM_SMOD_JUMBO_ENA;
			GM_WRITE_16(mac, GM_SERIAL_MODE, ser);
			/* MAC address */
/* I don't understand it --- the adresses for two macs overlap! */
			/*
			GM_WRITE_16(mac, GM_SRC_ADDR_1L, SK_READ_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR));
			GM_WRITE_16(mac, GM_SRC_ADDR_1M, SK_READ_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 2));
			GM_WRITE_16(mac, GM_SRC_ADDR_1H, SK_READ_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 4));
			GM_WRITE_16(mac, GM_SRC_ADDR_2L, SK_READ_16(sk, SK_MAC_ADDR_1 + i * ADD_MAC2_SK_MAC_ADDR));
			GM_WRITE_16(mac, GM_SRC_ADDR_2M, SK_READ_16(sk, SK_MAC_ADDR_1 + i * ADD_MAC2_SK_MAC_ADDR + 2));
			GM_WRITE_16(mac, GM_SRC_ADDR_2H, SK_READ_16(sk, SK_MAC_ADDR_1 + i * ADD_MAC2_SK_MAC_ADDR + 4));
			*/
			if (!mac->force_address) {
				mac->address[0] = SK_READ_16(mac, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR);
				mac->address[1] = SK_READ_16(mac, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 2);
				mac->address[2] = SK_READ_16(mac, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 4);
				mac->force_address = 1;
			}
			/*if (mac->force_address == 2) {
				SK_TESTMODE_ON(sk);
				SK_WRITE_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR, mac->address[0]);
				SK_WRITE_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 2, mac->address[1]);
				SK_WRITE_16(sk, SK_MAC_ADDR_2 + i * ADD_MAC2_SK_MAC_ADDR + 4, mac->address[2]);
				SK_TESTMODE_OFF(sk);
			}*/
			/*__debug_printf("a1: %04x:%04x:%04x\n", SK_READ_16(sk, SK_MAC_ADDR_1), SK_READ_16(sk, SK_MAC_ADDR_1 + 2), SK_READ_16(sk, SK_MAC_ADDR_1 + 4));
			__debug_printf("a2: %04x:%04x:%04x\n", SK_READ_16(sk, SK_MAC_ADDR_2), SK_READ_16(sk, SK_MAC_ADDR_2 + 2), SK_READ_16(sk, SK_MAC_ADDR_2 + 4));
			__debug_printf("a3: %04x:%04x:%04x\n", SK_READ_16(sk, SK_MAC_ADDR_3), SK_READ_16(sk, SK_MAC_ADDR_3 + 2), SK_READ_16(sk, SK_MAC_ADDR_3 + 4));*/
			GM_WRITE_16(mac, GM_SRC_ADDR_1L, mac->address[0]);
			GM_WRITE_16(mac, GM_SRC_ADDR_1M, mac->address[1]);
			GM_WRITE_16(mac, GM_SRC_ADDR_1H, mac->address[2]);
			GM_WRITE_16(mac, GM_SRC_ADDR_2L, mac->address[0]);
			GM_WRITE_16(mac, GM_SRC_ADDR_2M, mac->address[1]);
			GM_WRITE_16(mac, GM_SRC_ADDR_2H, mac->address[2]);

			GM_WRITE_16(mac, GM_TX_IRQ_MSK, 0);
			GM_WRITE_16(mac, GM_RX_IRQ_MSK, 0);
			GM_WRITE_16(mac, GM_TR_IRQ_MSK, 0);

			/* initialize rx GMAC fifo --- SkGeInitMacFifo */
			if (Y2_BUG_4115(sk)) {
				SK_WRITE_16(sk, SK_RX_GMF_FLUSH_MASK + i * ADD_MAC2_SK_RX_GMF, 0);
			} else if (sk->chiptype != CHIPTYPE_YUKON_2) {
				SK_WRITE_16(sk, SK_RX_GMF_FLUSH_MASK + i * ADD_MAC2_SK_RX_GMF, GMR_FS_CRC_ERR | GMR_FS_LONG_ERR | GMR_FS_MII_ERR | GMR_FS_BAD_FC | GMR_FS_GOOD_FC | GMR_FS_JABBER);
			} else {
				SK_WRITE_16(sk, SK_RX_GMF_FLUSH_MASK + i * ADD_MAC2_SK_RX_GMF, GMR_FS_RX_FF_OV | GMR_FS_CRC_ERR | GMR_FS_FRAGMENT | GMR_FS_LONG_ERR | GMR_FS_MII_ERR | GMR_FS_BAD_FC | GMR_FS_UN_SIZE | GMR_FS_JABBER);
			}
			SK_WRITE_8(sk, SK_RX_GMF_CTRL + i * ADD_MAC2_SK_RX_GMF, SK_RX_GMF_CTRL_RESET_CLEAR);
			SK_WRITE_32(sk, SK_RX_GMF_CTRL + i * ADD_MAC2_SK_RX_GMF,
				SK_RX_GMF_CTRL_OPERATIONAL_ON |
				(sk->chipsubtype != CHIPSUBTYPE_YUKON_LITE_A0 ? SK_RX_GMF_CTRL_FIFO_FLUSH_ON : 0) |
				(sk->chipsubtype == CHIPSUBTYPE_YUKON_EX || sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P ? SK_RX_GMF_CTRL_RX_OVER_ON : 0));
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P &&
			    sk->chiprev == SK_CONFIGURATION_FE_P_A0)
				SK_WRITE_16(sk, SK_RX_GMF_FLUSH_THRESH + i * ADD_MAC2_SK_RX_GMF, 0x178);
			else
				SK_WRITE_16(sk, SK_RX_GMF_FLUSH_THRESH + i * ADD_MAC2_SK_RX_GMF, GMAC_FIFO_FLUSH_THRESH + 1);
			if (sk->chiptype == CHIPTYPE_YUKON_2) {
				unsigned trunc_thresh;
				trunc_thresh = mac->mtu + ETHERNET_HEADER;
				trunc_thresh = (trunc_thresh + 7) & ~7;
				trunc_thresh = (trunc_thresh - 8) / 4;
				if (trunc_thresh > 0x1ff) {
					SK_WRITE_32(sk, SK_RX_GMF_CTRL + i * ADD_MAC2_SK_RX_GMF, SK_RX_GMF_CTRL_TRUNC_OFF);
				} else {
					SK_WRITE_32(sk, SK_RX_GMF_TRUNC_THRESH + i * ADD_MAC2_SK_RX_GMF, trunc_thresh);
					SK_WRITE_32(sk, SK_RX_GMF_CTRL + i * ADD_MAC2_SK_RX_GMF, SK_RX_GMF_CTRL_TRUNC_ON);
				}
			}
			/* initialize tx GMAC fifo */
			SK_WRITE_8(sk, SK_TX_GMF_CTRL + i * ADD_MAC2_SK_TX_GMF, SK_TX_GMF_CTRL_RESET_CLEAR);
			SK_WRITE_16(sk, SK_TX_GMF_CTRL + i * ADD_MAC2_SK_TX_GMF, SK_TX_GMF_CTRL_OPERATIONAL_ON);
			if (!sk->ramsize) {
				SK_WRITE_16(sk, SK_RX_GMF_LP_THRESH + i * ADD_MAC2_SK_RX_GMF, 768 / 8);
				SK_WRITE_16(sk, SK_RX_GMF_UP_THRESH + i * ADD_MAC2_SK_RX_GMF, 1024 / 8);
				/* set store & forward */
				if ((sk->chipsubtype == CHIPSUBTYPE_YUKON_EX && sk->chiprev != SK_CONFIGURATION_EX_A0) ||
				    sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P ||
				    sk->chipsubtype == CHIPSUBTYPE_YUKON_SUPR) {
					if (mac->mtu <= ETHERNET_MTU)
						SK_WRITE_32(sk, SK_TX_GMF_CTRL + i * ADD_MAC2_SK_TX_GMF, SK_Y2_TX_GMF_CTRL_JUMBO_DIS | SK_Y2_TX_GMF_CTRL_STFW_ENA);
					else SK_WRITE_32(sk, SK_TX_GMF_CTRL + i * ADD_MAC2_SK_TX_GMF, SK_Y2_TX_GMF_CTRL_JUMBO_ENA | SK_Y2_TX_GMF_CTRL_STFW_ENA);
				} else {
					if (mac->mtu <= ETHERNET_MTU)
						SK_WRITE_32(sk, SK_TX_GMF_CTRL + i * ADD_MAC2_SK_TX_GMF, SK_Y2_TX_GMF_CTRL_STFW_ENA);
					else {
						int j;
						SK_WRITE_32(sk, SK_TX_GMF_AE_THRESH+ i * ADD_MAC2_SK_TX_GMF, 0x00800070);
						SK_WRITE_32(sk, SK_TX_GMF_CTRL + i * ADD_MAC2_SK_TX_GMF, SK_Y2_TX_GMF_CTRL_STFW_DIS);
						for (j = 0; j < 2; j++)
							mac->xmit_queue[j].no_stfw = 1;
					}
				}
				/* disable dynamic watermark */
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P && sk->chiprev == SK_CONFIGURATION_FE_P_A0) {
					__u16 x = SK_READ_16(sk, SK_TX_GMF_END_ADDR + i * ADD_MAC2_SK_TX_GMF);
					x &= ~SK_TX_GMF_END_ADDR_DYN;
					SK_WRITE_16(sk, SK_TX_GMF_END_ADDR + i * ADD_MAC2_SK_TX_GMF, x);
				}
			}
		}


		/* initialize ram interface --- SkGeInitRamBufs */
		/* split ram equally between ports
		   in case of two tx queues, split ram 4:2:2 (rx:txs:txa)
		   in case of one tx queue, split it 5:3 (rx:txa) */
		ramsize = sk->ramsize / sk->n_macs;
		ramoffs = sk->ramoffs + i * ramsize;
		SK_WRITE_8(sk, SK_RRB_CTRL + i * ADD_MAC2_SK_RRB, SK_RRB_CTRL_RESET_CLEAR);
		if (mac->queues == 2) {
			csize = ramsize / 2;
			csize = (csize + 8191) & ~8191;
		} else {
			csize = ramsize * 5 / 8;
			csize = (csize + 8191) & ~8191;
		}
		if (csize >= ramsize) {
			csize = ramsize / 2;
			csize = (csize + 1023) & ~1023;
		}

		/* receive queue */
		if (sk->ramsize) {
			SK_WRITE_32(sk, SK_RRB_START_ADDR + i * ADD_MAC2_SK_RRB, ramoffs / 8);
			SK_WRITE_32(sk, SK_RRB_END_ADDR + i * ADD_MAC2_SK_RRB, (ramoffs + csize) / 8 - 1);
			SK_WRITE_32(sk, SK_RRB_WRITE_PTR + i * ADD_MAC2_SK_RRB, ramoffs / 8);
			SK_WRITE_32(sk, SK_RRB_READ_PTR + i * ADD_MAC2_SK_RRB, ramoffs / 8);
			SK_WRITE_32(sk, SK_RRB_UPPER_THRESH_PP + i * ADD_MAC2_SK_RRB, (csize - UPPER_PAUSE_THRESH) / 8);
			if (csize > LOWER_PAUSE_THRESH)
				SK_WRITE_32(sk, SK_RRB_LOWER_THRESH_PP + i * ADD_MAC2_SK_RRB, (csize - LOWER_PAUSE_THRESH) / 8);
			else
				SK_WRITE_32(sk, SK_RRB_LOWER_THRESH_PP + i * ADD_MAC2_SK_RRB, (csize - LOWER_LOWER_PAUSE_THRESH) / 8);
		}
		SK_WRITE_8(sk, SK_RRB_CTRL + i * ADD_MAC2_SK_RRB, SK_RRB_CTRL_OPERATIONAL_ON);
		ramsize -= csize;
		ramoffs += csize;

		/* tx queues */
		for (j = 0; j < 2; j++) {
			if (mac->queues == 2 || j == 1) {
				SK_WRITE_8(sk, SK_TRB_CTRL + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, SK_TRB_CTRL_RESET_CLEAR);
				if (!j) {
					csize = ramsize / 2;
					csize = (csize + 8191) & ~8191;
					if (csize >= ramsize) {
						csize = ramsize / 2;
						csize = (csize + 1023) & ~1023;
					}
				} else {
					csize = ramsize;
				}
				if (sk->ramsize) {
					SK_WRITE_32(sk, SK_TRB_START_ADDR + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, ramoffs / 8);
					SK_WRITE_32(sk, SK_TRB_END_ADDR + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, (ramoffs + ramsize) / 8 - 1);
					SK_WRITE_32(sk, SK_TRB_WRITE_PTR + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, ramoffs / 8);
					SK_WRITE_32(sk, SK_TRB_READ_PTR + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, ramoffs / 8);
					if (sk->chiptype != CHIPTYPE_GENESIS || mac->mtu > GENESIS_FIFO_SIZE) SK_WRITE_8(sk, SK_TRB_CTRL + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, SK_TRB_CTRL_ST_FWD_ON);
				}
				SK_WRITE_8(sk, SK_TRB_CTRL + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, SK_TRB_CTRL_OPERATIONAL_ON);
				ramsize -= csize;
				ramoffs += csize;
			} else {
				SK_WRITE_8(sk, SK_TRB_CTRL + i * ADD_MAC2_SK_TRB + j * ADD_ASYNC_SK_TRB, SK_TRB_CTRL_RESET_SET);
			}
		}
		/* init BMU descriptors */
		if (sk->chiptype != CHIPTYPE_YUKON_2) {
			SKGE_TEAR_DOWN_RECV_PACKETS(mac, 0);
			SKGE_TEAR_DOWN_SENT_PACKETS(mac);
		} else {
			SKY2_TEAR_DOWN_RECV_PACKETS(mac, 0);
			SKY2_TEAR_DOWN_SENT_PACKETS(mac);
		}
		k = 0;
		l = N_DESCRIPTORS - 1;
		while (k < l) {
			if (mac->recv_packets[k]) k++;
			else if (!mac->recv_packets[l]) l--;
			else mac->recv_packets[k] = mac->recv_packets[l], mac->recv_packets[l] = NULL;
		}
		n_tmp_recv_packets = k;
		mac->first_recv = 0;
		if (sk->chiptype != CHIPTYPE_YUKON_2) {
			memset(mac->xmit_queue[0].u.xmit, 0, sizeof(mac->xmit_queue[0].u.xmit));
			memset(mac->xmit_queue[1].u.xmit, 0, sizeof(mac->xmit_queue[1].u.xmit));
			memset(mac->u.recv, 0, sizeof(mac->u.recv));
			for (j = 0; j < N_DESCRIPTORS; j++) {
				mac->xmit_queue[0].u.xmit[j].ntdadr = (__u32)mac->xmit_queue[0].snd_dma64addr + ((j + 1) & (N_DESCRIPTORS - 1)) * sizeof(XMIT_DESC);
				mac->xmit_queue[1].u.xmit[j].ntdadr = (__u32)mac->xmit_queue[1].snd_dma64addr + ((j + 1) & (N_DESCRIPTORS - 1)) * sizeof(XMIT_DESC);
				mac->u.recv[j].nrdadr = (__u32)mac->rcv_dma64addr + ((j + 1) & (N_DESCRIPTORS - 1)) * sizeof(RECV_DESC);
				mac->u.recv[j].tcp_sum_start_1 = SK_TCP_SUM_START_1 + ETHERNET_HEADER;
				mac->u.recv[j].tcp_sum_start_2 = SK_TCP_SUM_START_2 + ETHERNET_HEADER;
			}
			for (j = 0; j < mac->n_recv; j++) {
				SKGE_MAP_RCV_PACKET(mac, mac->recv_packets[j], j);
			}
		} else {
			memcpy(&sk->tmp_recv_packets, &mac->recv_packets, n_tmp_recv_packets * sizeof(PACKET *));
			memset(&mac->recv_packets, 0, sizeof mac->recv_packets);
			mac->n_recv = 0;
		}
		/* init BMU */
		rxwm = BMU_RX_WATERMARK;
		txwm = BMU_TX_WATERMARK;
		if (sk->chiptype != CHIPTYPE_YUKON_2) {
			if (!(SK_READ_32(sk, SK_CS) & (SK_CS_BUS_CLOCK | SK_CS_SLOT_SIZE))) {
				rxwm = BMU_PCI33_RX_WATERMARK;
				txwm = BMU_PCI33_TX_WATERMARK;
			}
		} else {
			if (!(SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + PCI_SK_OUR_STATUS) & PCI_SK_OUR_STATUS_PCI_X)) {
				rxwm = BMU_PEX_RX_WATERMARK;
				txwm = BMU_PEX_TX_WATERMARK;
			}
		}
		if (sk->chiptype != CHIPTYPE_YUKON_2) {
			SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_RX_CS_RESET_DESC_CLEAR | SK_RX_CS_RESET_FIFO_CLEAR | SK_RX_CS_RUN_PFI_SM | SK_RX_CS_RUN_SUPERVISOR_SM | SK_RX_CS_RUN_DESC_READ_SM | SK_RX_CS_RUN_DESC_WRITE_SM | SK_RX_CS_RUN_TRANSFER_SM);
			SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_RX_CS_CLR_IRQ_PAR | SK_RX_CS_CLR_IRQ_EOB | SK_RX_CS_CLR_IRQ_EOF | SK_RX_CS_CLR_IRQ_ERR);
			SK_WRITE_32(sk, SK_RX_FLAG + i * ADD_MAC2_SK_RX, rxwm);
			SK_WRITE_32(sk, SK_RX_CUR_DESC_LO + i * ADD_MAC2_SK_RX, (__u32)mac->rcv_dma64addr);
			SK_WRITE_32(sk, SK_RX_CUR_DESC_HI + i * ADD_MAC2_SK_RX, mac->rcv_dma64addr >> 32);
			SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_RX_CS_EN_POLLING_ON | SK_RX_CS_START_RX);	/* start it */
			for (j = 2 - mac->queues; j < 2; j++) {
				SK_WRITE_32(sk, SK_TX_CS + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_TX_CS_RESET_DESC_CLEAR | SK_TX_CS_RESET_FIFO_CLEAR | SK_TX_CS_RUN_PFI_SM | SK_TX_CS_RUN_SUPERVISOR_SM | SK_TX_CS_RUN_DESC_READ_SM | SK_TX_CS_RUN_DESC_WRITE_SM | SK_TX_CS_RUN_TRANSFER_SM);
				SK_WRITE_32(sk, SK_TX_CS + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_TX_CS_CLR_IRQ_EOB | SK_TX_CS_CLR_IRQ_EOF | SK_TX_CS_CLR_IRQ_ERR);
				SK_WRITE_32(sk, SK_TX_FLAG + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, txwm);
				SK_WRITE_32(sk, SK_TX_CUR_DESC_LO + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, (__u32)mac->xmit_queue[j].snd_dma64addr);
				SK_WRITE_32(sk, SK_TX_CUR_DESC_HI + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, mac->xmit_queue[j].snd_dma64addr >> 32);
				SK_WRITE_32(sk, SK_TX_CS + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_TX_CS_EN_POLLING_ON);	/* start it */
			}
		} else {
			/* rx prefetch unit */
			SK_WRITE_32(sk, SK_Y2_RX_PREF_CTRL + i * ADD_MAC2_SK_RX, SK_Y2_RX_PREF_CTRL_RESET_SET);
			SK_WRITE_32(sk, SK_Y2_RX_PREF_CTRL + i * ADD_MAC2_SK_RX, SK_Y2_RX_PREF_CTRL_RESET_CLEAR);
			SK_WRITE_16(sk, SK_Y2_RX_PREF_LAST_IDX + i * ADD_MAC2_SK_RX, N_DESCRIPTORS - 1);
			SK_WRITE_32(sk, SK_Y2_RX_PREF_ADDR_LO + i * ADD_MAC2_SK_RX, (__u32)mac->rcv_dma64addr);
			SK_WRITE_32(sk, SK_Y2_RX_PREF_ADDR_HI + i * ADD_MAC2_SK_RX, mac->rcv_dma64addr >> 32);
			SK_WRITE_32(sk, SK_Y2_RX_PREF_CTRL + i * ADD_MAC2_SK_RX, SK_Y2_RX_PREF_CTRL_OPERATIONAL_ON);
			/* rx queue */
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EC_U &&
			    (sk->chiprev == SK_CONFIGURATION_EC_U_A1 ||
			     sk->chiprev == SK_CONFIGURATION_EC_U_B0))
				SK_WRITE_32(sk, SK_Y2_RX_TEST, SK_Y2_RX_TEST_RAM_DIS);
			SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_Y2_RX_CS_FIFO_RESET | SK_Y2_RX_CS_OP_OFF | SK_Y2_RX_CS_CLEAR_RESET);
			SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_Y2_RX_CS_CLR_IRQ_PAR | SK_Y2_RX_CS_CLR_IRQ_CHK | SK_Y2_RX_CS_START_RX | SK_Y2_RX_CS_FIFO_ENA | SK_Y2_RX_CS_OP_ON);
			SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_Y2_RX_CS_FIFO_OP_ON);
			SK_WRITE_32(sk, SK_Y2_RX_WATERMARK, rxwm);
			for (j = 2 - mac->queues; j < 2; j++) {
				/* tx prefetch unit */
				SK_WRITE_32(sk, SK_Y2_TX_PREF_CTRL + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_PREF_CTRL_RESET_SET);
				SK_WRITE_32(sk, SK_Y2_TX_PREF_CTRL + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_PREF_CTRL_RESET_CLEAR);
				SK_WRITE_16(sk, SK_Y2_TX_PREF_LAST_IDX + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, N_DESCRIPTORS - 1);
				SK_WRITE_32(sk, SK_Y2_TX_PREF_ADDR_LO + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, (__u32)mac->xmit_queue[j].snd_dma64addr);
				SK_WRITE_32(sk, SK_Y2_TX_PREF_ADDR_HI + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, mac->xmit_queue[j].snd_dma64addr >> 32);
				SK_WRITE_32(sk, SK_Y2_TX_PREF_CTRL + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_PREF_CTRL_OPERATIONAL_ON);
				/* tx queue */
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EX &&
				    sk->chiprev == SK_CONFIGURATION_EX_B0)
					SK_WRITE_32(sk, SK_Y2_TX_TEST, SK_Y2_TX_TEST_AUTO_OFF);
				if (sk->chipsubtype == CHIPSUBTYPE_YUKON_EC_U &&
				    sk->chiprev == SK_CONFIGURATION_EC_U_A0)
					SK_WRITE_32(sk, SK_Y2_TX_ALIGNMENT, 0x1a0);
				SK_WRITE_32(sk, SK_TX_CS + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_CS_FIFO_RESET | SK_Y2_TX_CS_OP_OFF | SK_Y2_TX_CS_CLEAR_RESET);
				SK_WRITE_32(sk, SK_TX_CS + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_CS_CLR_IRQ_TCP | SK_Y2_TX_CS_CLR_IRQ_CHK | SK_Y2_TX_CS_START_TX | SK_Y2_TX_CS_FIFO_ENA | SK_Y2_TX_CS_OP_ON);
				SK_WRITE_32(sk, SK_TX_CS + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_CS_FIFO_OP_ON);
				SK_WRITE_32(sk, SK_Y2_TX_WATERMARK + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, txwm);
				mac->xmit_queue[j].checksum_cache = -1;
			}
#ifdef SK_ENABLE_RX_CHECKSUM
			if (!NO_RX_CSUM(sk) && !NEW_LE(sk)) {
				SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_Y2_RX_CS_CHKSUM_ENA);
				mac->u.y2_recv[0].rxch.tcp_sum_start_1 = SK_TCP_SUM_START_1 + ETHERNET_HEADER;
				mac->u.y2_recv[0].rxch.tcp_sum_start_2 = SK_TCP_SUM_START_2 + ETHERNET_HEADER;
				mac->u.y2_recv[0].rxch.reserved1 = 0;
				mac->u.y2_recv[0].rxch.lock = 0;
				__write_barrier();
				mac->u.y2_recv[0].rxch.opcode = Y2_OP_TCPSTART | Y2_OP_OWNER;
				mac->n_recv = 1;
			} else {
				SK_WRITE_32(sk, SK_RX_CS + i * ADD_MAC2_SK_RX, SK_Y2_RX_CS_CHKSUM_DIS);
			}
#endif
			n = 0;
			while (!SK_RCVQ_FULL(mac) && n < n_tmp_recv_packets) {
				PACKET *p = sk->tmp_recv_packets[n++];
				SKY2_ADD_RCV_PACKET(mac, p);
			}
			while (n < n_tmp_recv_packets) {
				PACKET *p = sk->tmp_recv_packets[n++];
				FREE_PACKET(p, NULL, SPL_SK);
			}
			SKY2_KICK_RECEIVE(mac);
		}
	}
	/* LED */
	if (sk->chiptype != CHIPTYPE_YUKON_2) {
		SK_WRITE_8(sk, SK_LED, SK_LED_ON);
	} else {
		SK_WRITE_16(sk, SK_CS, SK_Y2_CS_LED_STAT_ON);
	}
	/* timer */
	SK_WRITE_8(sk, SK_TIMER_CONTROL, SK_TIMER_CONTROL_STOP);
	SK_WRITE_8(sk, SK_TIMER_CONTROL, SK_TIMER_CONTROL_CLEAR_IRQ);
	SK_WRITE_32(sk, SK_TIMER_INIT_VAL, TIMER_INTERVAL * 1000000 * sk->host_clock);
	SK_WRITE_8(sk, SK_TIMER_CONTROL, SK_TIMER_CONTROL_START);
	/* setup IMR */
	if (sk->chiptype != CHIPTYPE_YUKON_2) {
		imr = SK_IRQ_HW_ERROR | SK_IRQ_I2C_READY | SK_IRQ_SW | SK_IRQ_EXTERNAL_REG | SK_IRQ_TIMER;
		if (sk->chiptype == CHIPTYPE_YUKON) {
			if (SK_READ_32(sk, SK_ISR) & SK_IRQ_HW_ERROR) {
				if (SK_READ_32(sk, SK_IHESR) & SK_HWIRQ_SENSOR) {
					imr &= ~SK_IRQ_HW_ERROR;
				}
			}
		}
		imr |= SK_IRQ_EOF_RX1 | SK_IRQ_CHCK_RX1 | SK_IRQ_EOF_AS_TX1 | SK_IRQ_CHCK_AS_TX1 | SK_IRQ_PKT_TOUT_RX1 | SK_IRQ_PKT_TOUT_TX1 | SK_IRQ_MAC_1 | SK_IRQ_LINK_SYNC_MAC_1;
		if (sk->macs[0].queues == 2) imr |= SK_IRQ_EOF_S_TX1 | SK_IRQ_CHCK_S_TX1;
		if (sk->n_macs == 2) {
			imr |= SK_IRQ_EOF_RX2 | SK_IRQ_CHCK_RX2 | SK_IRQ_EOF_AS_TX2 | SK_IRQ_CHCK_AS_TX2 | SK_IRQ_PKT_TOUT_RX2 | SK_IRQ_PKT_TOUT_TX2 | SK_IRQ_MAC_2 | SK_IRQ_LINK_SYNC_MAC_2;
			if (sk->macs[1].queues == 2) imr |= SK_IRQ_EOF_S_TX2 | SK_IRQ_CHCK_S_TX2;
		}
		hwimr = SK_HWIRQ_SENSOR | SK_HWIRQ_TS_TIMER_OVERFLOW | SK_HWIRQ_PAR_WR_RAM | SK_HWIRQ_PAR_RD_RAM | SK_HWIRQ_STATUS | SK_HWIRQ_MASTER_ERROR;
		hwimr |= SK_HWIRQ_PAR_RX_1 | SK_HWIRQ_PAR_MAC_1 | SK_HWIRQ_NO_STATUS_MAC_1 | SK_HWIRQ_NO_TIMESTAMP_MAC_1;
		if (sk->n_macs == 2) {
			hwimr |= SK_HWIRQ_PAR_RX_2 | SK_HWIRQ_PAR_MAC_2 | SK_HWIRQ_NO_STATUS_MAC_2 | SK_HWIRQ_NO_TIMESTAMP_MAC_2;
		}
	} else {
		imr = SK_Y2_IRQ_HW_ERROR | SK_Y2_IRQ_STAT_BMU | SK_Y2_IRQ_POLL_CHECK | SK_Y2_IRQ_ASF | SK_Y2_IRQ_I2C_READY | SK_Y2_IRQ_SW | SK_Y2_IRQ_TIMER;
		imr |= SK_Y2_IRQ_RX1 | SK_Y2_IRQ_TXA1 | SK_Y2_IRQ_PHY1 | SK_Y2_IRQ_MAC1;
		if (sk->macs[0].queues == 2) imr |= SK_Y2_IRQ_TXS1;
		if (sk->n_macs == 2) {
			imr |= SK_Y2_IRQ_RX2 | SK_Y2_IRQ_TXA2 | SK_Y2_IRQ_PHY2 | SK_Y2_IRQ_MAC2;
			if (sk->macs[1].queues == 2) imr |= SK_Y2_IRQ_TXS2;
		}
		hwimr = SK_Y2_HWIRQ_MASTER_ERROR | SK_Y2_HWIRQ_STATUS_EXCEPTION;
		hwimr |= SK_Y2_HWIRQ_PAR_RD_RAM_1 | SK_Y2_HWIRQ_PAR_WR_RAM_1 | SK_Y2_HWIRQ_PAR_MAC_1 | SK_Y2_HWIRQ_PAR_RX_1 | SK_Y2_HWIRQ_TCP_A1;
		if (sk->macs[0].queues == 2) hwimr |= SK_Y2_HWIRQ_TCP_S1;
		if (sk->n_macs == 2) {
			hwimr |= SK_Y2_HWIRQ_PAR_RD_RAM_2 | SK_Y2_HWIRQ_PAR_WR_RAM_2 | SK_Y2_HWIRQ_PAR_MAC_2 | SK_Y2_HWIRQ_PAR_RX_2 | SK_Y2_HWIRQ_TCP_A2;
			if (sk->macs[1].queues == 2) hwimr |= SK_Y2_HWIRQ_TCP_S2;
		}
		if (PCI$FIND_CAPABILITY(sk->id, -1, PCI_CAP_ID_EXP) >= 0) {
			SK_WRITE_32(sk, SK_Y2_AER_CONFIGURATION_FILE + PCI_ERR_UNCOR_STATUS, 0xffffffff);
			if (!(SK_READ_32(sk, SK_IHESR) & SK_Y2_HWIRQ_PCI_EXPRESS))
				hwimr |= SK_Y2_HWIRQ_PCI_EXPRESS;
		}
	}
	sk->imr = imr;
	SK_WRITE_32(sk, SK_IMR, imr);
	SK_READ_32(sk, SK_IMR);
	sk->hwimr = hwimr;
	SK_WRITE_32(sk, SK_IHEMR, hwimr);
	SK_READ_32(sk, SK_IHEMR);
	/* initialize xmacii / gmac */
	if (sk->chiptype == CHIPTYPE_GENESIS) {
		for (i = 0; i < sk->n_macs; i++) {
			MAC *mac = &sk->macs[i];
			__u16 mmu;
			__u16 isrc = XM_IS_INP_ASS | XM_IS_LIPA_RC | XM_IS_RX_PAGE | XM_IS_AND | XM_IS_TXF_UR;
			if (sk->phytype != SK_EPROM_1_PHY_XMAC) {
				isrc &= ~XM_IS_INP_ASS;
			}
			XM_WRITE_16(mac, XM_IMSK, ~isrc);
			mmu = XM_READ_16(mac, XM_MMU_CMD);
			if (sk->phytype != SK_EPROM_1_PHY_XMAC && mac->duplex != DUPLEX_HALF) {
				mmu |= XM_MMU_GMII_FD;
			}
			if (sk->phytype == SK_EPROM_1_PHY_BCOM) {
				__u16 aux = XM_PHY_READ(mac, PHY_BCOM_AUX_CTRL);
				XM_PHY_WRITE(mac, PHY_BCOM_AUX_CTRL, aux & ~PHY_B_AC_DIS_PM);
				XM_PHY_WRITE(mac, PHY_BCOM_INT_MASK, ~(PHY_B_IS_AN_PR | PHY_B_IS_LST_CHANGE));
			}
			mmu |= XM_MMU_ENA_RX | XM_MMU_ENA_TX;
			XM_WRITE_16(mac, XM_MMU_CMD, mmu);
		}
	} else {
		for (i = 0; i < sk->n_macs; i++) {
			MAC *mac = &sk->macs[i];
			__u16 gp;
			SK_WRITE_8(sk, SK_GMAC_IRQ_MASK + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_IRQ_SRC_TX_CNT_OVER | SK_GMAC_IRQ_SRC_RX_CNT_OVER | SK_GMAC_IRQ_SRC_TX_FIFO_UNDER | SK_GMAC_IRQ_SRC_RX_FIFO_OVER);
			gp = GM_READ_16(mac, GM_GP_CTRL);
			gp |= GM_GPCR_RX_ENA | GM_GPCR_TX_ENA;
			GM_WRITE_16(mac, GM_GP_CTRL, gp);
			GM_READ_16(mac, GM_GP_CTRL);
			if (mac->duplex == DUPLEX_AUTO)
				GM_PHY_WRITE(mac, PHY_MARV_INT_MASK, PHY_M_IS_AN_COMPL | PHY_M_IS_LSP_CHANGE | PHY_M_IS_LST_CHANGE | PHY_M_IS_DUP_CHANGE);
			else
				GM_PHY_WRITE(mac, PHY_MARV_INT_MASK, PHY_M_IS_LSP_CHANGE | PHY_M_IS_LST_CHANGE | PHY_M_IS_DUP_CHANGE);
		}
	}
	for (i = 0; i < sk->n_macs; i++) {
		MAC *mac = &sk->macs[i];
		for (j = 2 - mac->queues; j < 2; j++) {
			KERNEL$DEL_TIMER(&mac->xmit_queue[j].timer);
			SET_TIMER_NEVER(&mac->xmit_queue[j].timer);
		}
		if (sk->chiptype != CHIPTYPE_YUKON_2) SKGE_START_XMIT(mac);
		else SKY2_START_XMIT(mac);
		SK_GET_LINK(mac);
	}
#ifdef SK_HIGH_INTERRUPT
	sk->isr = 0;
#endif
	__rw_barrier();
}

static void SK_SHUTDOWN_RESET(SK *sk)
{
	int i, j;
	if (sk->chiptype != CHIPTYPE_GENESIS) {
		for (i = 0; i < sk->n_macs; i++) {
			SK_WRITE_8(sk, SK_GPHY_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GPHY_CTRL_RESET_SET);
			SK_WRITE_8(sk, SK_GMAC_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_CTRL_RESET_SET);
			SK_WRITE_16(sk, SK_GMAC_LINK_CTRL + i * ADD_MAC2_SK_GMAC_GPHY, SK_GMAC_LINK_CTRL_RESET_SET);
			SK_WRITE_8(sk, SK_RX_GMF_CTRL + i * ADD_MAC2_SK_RX_GMF, SK_RX_GMF_CTRL_RESET_SET);
			if (sk->chiptype == CHIPTYPE_YUKON_2) {
				SK_WRITE_32(sk, SK_Y2_RX_PREF_CTRL + i * ADD_MAC2_SK_RX, SK_Y2_RX_PREF_CTRL_RESET_SET);
				for (j = 2 - sk->macs[i].queues; j < 2; j++) {
					SK_WRITE_32(sk, SK_Y2_TX_PREF_CTRL + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX, SK_Y2_TX_PREF_CTRL_RESET_SET);
				}
			}
		}
	}
	if (sk->chiptype == CHIPTYPE_YUKON_2) {
		SK_WRITE_32(sk, SK_STAT_CTRL, SK_STAT_CTRL_RESET_SET);
	}
}

static void XM_SET_MULTICAST_HASH(MAC *mac)
{
	unsigned i;
	__u8 hash[8];
	__u16 mode = XM_READ_32(mac, XM_MODE);
	mode &= ~(XM_MD_ENA_HASH | XM_MD_ENA_PROM);
	if (__unlikely(!WQ_EMPTY(&mac->mcast_table[MCAST_PROMISC]))) {
		mode |= XM_MD_ENA_PROM;
		mac->mcast_state = STATUS_MULTICAST_PROMISC;
		goto set_mode;
	}
	if (__unlikely(!WQ_EMPTY(&mac->mcast_table[MCAST_ALL]))) {
		memset(hash, 0xff, sizeof hash);
		mode |= XM_MD_ENA_HASH;
		mac->mcast_state = STATUS_MULTICAST_ALL;
		goto set_hash;
	}
	memset(hash, 0, sizeof hash);
	mac->mcast_state = STATUS_MULTICAST_NONE;
	for (i = 0; i < 64; i++) {
		if (__unlikely(!WQ_EMPTY(&mac->mcast_table[i]))) {
			hash[i / 8] |= 1 << (i & 7);
			mode |= XM_MD_ENA_HASH;
			mac->mcast_state = STATUS_MULTICAST_SOME;
		}
	}
	set_hash:
	if (mode & XM_MD_ENA_HASH) {
		for (i = 0; i < 8; i += 2)
			XM_WRITE_16(mac, XM_HSM + i, hash[i] | (hash[i + 1] << 8));
	}
	set_mode:
	XM_WRITE_32(mac, XM_MODE, mode);
}

static unsigned SK_MULTICAST_HASH_BIT(MAC *mac, const __u8 *addr)
{
	if (__unlikely(mac->sk->chiptype == CHIPTYPE_GENESIS)) {
		return ~NET$ETHERNET_CRC_LE(addr, ETHERNET_ADDRLEN) & 63;
	} else {
		return NET$ETHERNET_CRC(addr, ETHERNET_ADDRLEN) & 63;
	}
}

static void GM_SET_MULTICAST_HASH(MAC *mac)
{
	int multicast_bit;
	static const __u8 pause_mac_addr[ETHERNET_ADDRLEN] = { 0x1, 0x80, 0xc2, 0x0, 0x0, 0x1 };
	unsigned i;
	__u8 hash[8];
	__u16 rx_ctrl = GM_READ_16(mac, GM_RX_CTRL);
	rx_ctrl &= ~(GM_RXCR_UCF_ENA | GM_RXCR_MCF_ENA);
	if (__unlikely(!WQ_EMPTY(&mac->mcast_table[MCAST_PROMISC]))) {
		mac->mcast_state = STATUS_MULTICAST_PROMISC;
		goto set_rx_ctrl;
	}
	rx_ctrl |= GM_RXCR_UCF_ENA;
	if (__unlikely(!WQ_EMPTY(&mac->mcast_table[MCAST_ALL]))) {
		memset(hash, 0xff, sizeof hash);
		rx_ctrl |= GM_RXCR_MCF_ENA;
		mac->mcast_state = STATUS_MULTICAST_ALL;
		goto set_rx_hash;
	}
	multicast_bit = SK_MULTICAST_HASH_BIT(mac, pause_mac_addr);
	memset(hash, 0, sizeof hash);
	mac->mcast_state = STATUS_MULTICAST_NONE;
	for (i = 0; i < 64; i++) {
		if (__unlikely(!WQ_EMPTY(&mac->mcast_table[i]))) {
			mac->mcast_state = STATUS_MULTICAST_SOME;
			goto set_bit;
		}
		if (i == multicast_bit) {
			set_bit:
			hash[i / 8] |= 1 << (i & 7);
			rx_ctrl |= GM_RXCR_MCF_ENA;
		}
	}
	set_rx_hash:
	if (rx_ctrl & GM_RXCR_MCF_ENA) {
		GM_WRITE_16(mac, GM_MC_ADDR_H1, hash[0] | (hash[1] << 8));
		GM_WRITE_16(mac, GM_MC_ADDR_H2, hash[2] | (hash[3] << 8));
		GM_WRITE_16(mac, GM_MC_ADDR_H3, hash[4] | (hash[5] << 8));
		GM_WRITE_16(mac, GM_MC_ADDR_H4, hash[6] | (hash[7] << 8));
	}
	set_rx_ctrl:
	GM_WRITE_16(mac, GM_RX_CTRL, rx_ctrl);
}

static void SK_SETUP_IRQ_MODERATION(SK *sk)
{
	int i;
	int g = 0;
	int ips;
	for (i = 0; i < sk->n_macs; i++) {
		if (sk->macs[i].link_state.speed > 100000000 && sk->macs[i].link_state.flags & (LINK_STATE_UP | LINK_STATE_UNKNOWN)) g = 1;
	}
	ips = g ? sk->ips : sk->ips_100;
	SK_WRITE_32(sk, SK_IRQM_TIMER_CONTROL, SK_IRQM_TIMER_CONTROL_STOP);
	SK_WRITE_32(sk, SK_IHEMMR, 0);
	if (!ips) {
		SK_WRITE_32(sk, SK_IMMR, 0);
		return;
	}
	if (sk->chiptype != CHIPTYPE_YUKON_2)
		SK_WRITE_32(sk, SK_IMMR, sk->imr & (SK_IRQ_LINK_SYNC_MAC_1 | SK_IRQ_LINK_SYNC_MAC_2 | SK_IRQ_EOF_RX1 | SK_IRQ_EOF_RX2 | SK_IRQ_EOF_S_TX1 | SK_IRQ_EOF_AS_TX1 | SK_IRQ_EOF_S_TX2 | SK_IRQ_EOF_AS_TX2));
	else
		SK_WRITE_32(sk, SK_IMMR, sk->imr & SK_Y2_IRQ_STAT_BMU);
	SK_WRITE_32(sk, SK_IRQM_TIMER_INIT_VAL, 1000000U * sk->host_clock / ips);
	SK_WRITE_32(sk, SK_IRQM_TIMER_CONTROL, SK_IRQM_TIMER_CONTROL_START);
}

void SK_MAC_IRQ(MAC *mac)
{
	if (mac->chiptype == CHIPTYPE_GENESIS) {
		XM_READ_16(mac, XM_ISRC);
	} else {
		__u8 gmistat = SK_READ_8(mac, SK_GMAC_IRQ_SRC + (ADD_MAC2_SK_GMAC_GPHY & mac->mac_2));
		if (gmistat & SK_GMAC_IRQ_SRC_TX_CNT_OVER) {
			GM_READ_16(mac, GM_TX_IRQ_SRC);
		}
		if (gmistat & SK_GMAC_IRQ_SRC_RX_CNT_OVER) {
			GM_READ_16(mac, GM_RX_IRQ_SRC);
		}
		if (gmistat & (SK_GMAC_IRQ_SRC_RX_CNT_OVER | SK_GMAC_IRQ_SRC_TX_CNT_OVER)) {
			GM_READ_16(mac, GM_RX_IRQ_SRC);
			SK_READ_16(mac, SK_RAP);
		}
		if (gmistat & SK_GMAC_IRQ_SRC_TX_FIFO_UNDER) {
			SK_WRITE_8(mac, SK_TX_GMF_CTRL + (ADD_MAC2_SK_TX_GMF & mac->mac_2), SK_TX_GMF_CTRL_CLR_TX_UNDERRUN);
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, mac->sk->dev_name, "PORT %c: TRANSMIT FIFO UNDERRUN", PORT_ID(mac));
		}
		if (gmistat & SK_GMAC_IRQ_SRC_RX_FIFO_OVER) {
			SK_WRITE_8(mac, SK_RX_GMF_CTRL + (ADD_MAC2_SK_RX_GMF & mac->mac_2), SK_RX_GMF_CTRL_CLR_RX_OVERRUN);
			if (mac->sk->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_HW_WARNING, mac->sk->dev_name, "PORT %c: RECEIVE FIFO OVERRUN", PORT_ID(mac));
		}
	}
}

static void GET_LINK_TECHNOLOGY(MAC *mac, LINK_STATE_PHYS *l, char m[3])
{
	m[0] = SK_READ_8(mac, SK_PMD_TYPE);
	if (m[0] < 32 || m[0] >= 127 || m[0] == '1') m[0] = 'T';
	m[1] = 'X', m[2] = 0;
	if (l->speed == 10000000) m[1] = 0;
	if (l->speed == 1000000000 && m[0] == 'T') m[1] = 0;
}

static void SK_GET_LINK(MAC *mac)
{
	LINK_STATE_PHYS *const l = (LINK_STATE_PHYS *)(void *)&mac->link_state;
	char m[3];
	memset(l, 0, sizeof(LINK_STATE_PHYS));
	if (mac->chiptype != CHIPTYPE_GENESIS) {
		__u16 aux_stat = GM_PHY_READ(mac, PHY_MARV_PHY_STAT);
		/*__debug_printf("phyi...: %04x\n", aux_stat);*/
		if (__unlikely(aux_stat & PHY_M_PS_JABBER)) l->flags |= LINK_STATE_JABBER;
		if (__unlikely(!(aux_stat & PHY_M_PS_LINK_UP))) goto ret;
		l->flags |= LINK_STATE_UP;
		if (aux_stat & PHY_M_PS_SPEED_1000 && GIGABIT(mac)) l->speed = 1000000000, strlcpy(l->desc, "1000BASE-", sizeof l->desc);
		else if (aux_stat & PHY_M_PS_SPEED_100) l->speed = 100000000, strlcpy(l->desc, "100BASE-", sizeof l->desc);
		else l->speed = 10000000, strlcpy(l->desc, "10BASE-", sizeof l->desc);
		GET_LINK_TECHNOLOGY(mac, l, m);
		strlcat(l->desc, m, sizeof l->desc);
		if (aux_stat & PHY_M_PS_FULL_DUP) l->flags |= LINK_STATE_FULL_DUPLEX;
		else l->flags |= LINK_STATE_HALF_DUPLEX;
	
		if (mac->duplex != DUPLEX_AUTO) goto ret;
	
		l->flags |= LINK_STATE_AUTO_NEGOTIATION;

		if (mac->chiptype == CHIPTYPE_YUKON_2) {
			__u16 advert, lpa;
			advert = GM_PHY_READ(mac, PHY_MARV_AUNE_ADV);
			lpa = GM_PHY_READ(mac, PHY_MARV_AUNE_LP);
			if (!mac->sk->copper) {
				advert &= ~0x0c00;
				if (advert & 0x0080)
					advert |= 0x0400;
				if (advert & 0x0100)
					advert |= 0x0800;
				lpa &= ~0x0c00;
				if (lpa & 0x0080)
					lpa |= 0x0400;
				if (lpa & 0x0100)
					lpa |= 0x0800;
			}
			aux_stat &= ~PHY_M_PS_PAUSE_MSK;
			if (advert & 0x0400) {
				if (lpa & 0x0400)
					aux_stat |= PHY_M_PS_PAUSE_MSK;
				else if (advert & 0x0800)
					aux_stat |= PHY_M_PS_RX_P_EN;
			} else if (advert & 0x0800 && (lpa & 0x0c00) == 0x0c000) {
				aux_stat |= PHY_M_PS_TX_P_EN;
			}
		}

		if (l->speed < 1000000000 && l->flags & LINK_STATE_HALF_DUPLEX && mac->chipsubtype != CHIPSUBTYPE_YUKON_EC_U && mac->chipsubtype != CHIPSUBTYPE_YUKON_EX)
			aux_stat &= ~PHY_M_PS_PAUSE_MSK;
		if (aux_stat & PHY_M_PS_TX_P_EN)
			SK_WRITE_8(mac, SK_GMAC_CTRL + (mac->mac_2 & ADD_MAC2_SK_GMAC_GPHY), SK_GMAC_CTRL_PAUSE_ON);
		else
			SK_WRITE_8(mac, SK_GMAC_CTRL + (mac->mac_2 & ADD_MAC2_SK_GMAC_GPHY), SK_GMAC_CTRL_PAUSE_OFF);
	} else {
		if (mac->sk->phytype == SK_EPROM_1_PHY_XMAC) {
			if (mac->duplex == DUPLEX_AUTO) {
				__u16 lpab = XM_PHY_READ(mac, PHY_XMAC_AUNE_LP);
				__u16 resab = XM_PHY_READ(mac, PHY_XMAC_RES_ABI);
				if (lpab & PHY_X_AN_RFB) {
					l->flags |= LINK_STATE_REMOTE_FAULT;
					goto ret;
				}
				if ((resab & (PHY_X_RS_HD | PHY_X_RS_FD)) == PHY_X_RS_FD) l->flags |= LINK_STATE_FULL_DUPLEX | LINK_STATE_AUTO_NEGOTIATION | LINK_STATE_UP;
				else if ((resab & (PHY_X_RS_HD | PHY_X_RS_FD)) == PHY_X_RS_HD) l->flags |= LINK_STATE_HALF_DUPLEX | LINK_STATE_AUTO_NEGOTIATION | LINK_STATE_UP;
				else l->flags |= LINK_STATE_AUTO_NEGOTIATION_FAILED;
			}
		} else {
			if (mac->duplex == DUPLEX_AUTO) {
				__u16 lpab = XM_PHY_READ(mac, PHY_BCOM_AUNE_LP);
				__u16 aux_stat = XM_PHY_READ(mac, PHY_BCOM_AUX_STAT);
				if (lpab & PHY_B_AN_RF) {
					l->flags |= LINK_STATE_REMOTE_FAULT;
					goto ret;
				}
				if ((aux_stat & (PHY_B_RES_1000HD | PHY_B_RES_1000FD)) == PHY_B_RES_1000FD) l->flags |= LINK_STATE_FULL_DUPLEX | LINK_STATE_AUTO_NEGOTIATION | LINK_STATE_UP;
				else if (((aux_stat & (PHY_B_RES_1000HD | PHY_B_RES_1000FD)) == PHY_B_RES_1000HD)) l->flags |= LINK_STATE_HALF_DUPLEX | LINK_STATE_AUTO_NEGOTIATION | LINK_STATE_UP;
				else l->flags |= LINK_STATE_AUTO_NEGOTIATION_FAILED;
			}
		}
		if (mac->duplex == DUPLEX_FULL) l->flags |= LINK_STATE_FULL_DUPLEX | LINK_STATE_UNKNOWN;
		if (mac->duplex == DUPLEX_HALF) l->flags |= LINK_STATE_HALF_DUPLEX | LINK_STATE_UNKNOWN;
		l->speed = 1000000000;
		GET_LINK_TECHNOLOGY(mac, l, m);
		_snprintf(l->desc, sizeof l->desc, "1000BASE-%s", m);
	}
	ret:
	SK_WRITE_8(mac, SK_LS_LINK_LED, l->flags & LINK_STATE_UP ?
		SK_LS_LINK_LED_ON | SK_LS_LINK_LED_LINK_SYNC_OFF | SK_LS_LINK_LED_BLINKING_OFF :
		SK_LS_LINK_LED_OFF);
	WQ_WAKE_ALL_PL(&mac->link_state_wait);
	SK_SETUP_IRQ_MODERATION(mac->sk);
}

void SK_GPHY_IRQ(MAC *mac)
{
	GM_PHY_READ(mac, PHY_MARV_INT_STAT);
	SK_GET_LINK(mac);
}

void SK_TIMER_FN(SK *sk)
{
	int i;
	for (i = 0; i < sk->n_macs; i++) {
		TX_QUEUE *q = sk->macs[i].queues == 1 ? &sk->macs[i].xmit_queue[1] : &sk->macs[i].xmit_queue[0];
		if (q->max_sent > DEFAULT_SEND) q->max_sent--;
	}
}

void SK_TX_TIMEOUT(TX_QUEUE *q)
{
	MAC *mac = q->mac;
	SK *sk = mac->sk;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, sk->dev_name, "PORT %c: TRANSMIT TIMEOUT ON %s QUEUE", PORT_ID(mac), QUEUE_ID(q));
	SK_BASIC_RESET(sk);
	SK_CONTINUE_RESET(sk);
}

void SK_BAD_STATUS(MAC *mac, __u32 stat, unsigned len)
{
	int r;
	SK *sk = mac->sk;
	if (__unlikely(sk->chiptype == CHIPTYPE_GENESIS)) {
		r = 0;
		if (stat & XMR_FS_CEX_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: CARRIER EXTENSION ERROR", PORT_ID(mac));
			r = 1;
		}
		if (stat & XMR_FS_LEN_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: IN-RANGE LENGTH ERROR", PORT_ID(mac));
			r = 1;
		}
		if (stat & XMR_FS_FRA_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: FRAMING ERROR", PORT_ID(mac));
			r = 1;
		}
		if (stat & XMR_FS_RUNT) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: RUNT PACKET", PORT_ID(mac));
			r = 1;
		}
		if (stat & XMR_FS_FCS_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: FRAME CHECK SEQ", PORT_ID(mac));
			r = 1;
		}
		if (r) return;
		if (stat & (XMR_FS_2L_VLAN | XMR_FS_LNG_ERR | XMR_FS_MCTRL)) return;
		r = 0;
		if (stat & XMR_FS_COL_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: COLLISION ERROR", PORT_ID(mac));
			r = 1;
		}
		if (stat & XMR_FS_CAR_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: CARRIER EVENT ERROR", PORT_ID(mac));
			r = 1;
		}
		if (__likely(r)) return;
		if (__unlikely((stat >> XMR_FS_LEN_SHIFT) != len)) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: RECEIVE LENGTH MISMATCH (XMAC %d, BMU %d)", PORT_ID(mac), stat >> XMR_FS_LEN_SHIFT, len);
			return;
		}
	} else {
		r = 0;
		if (stat & GMR_FS_CRC_ERR) {
			if (sk->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, sk->dev_name, "PORT %c: CRC ERROR", PORT_ID(mac));
			r = 1;
		}
		if (stat & GMR_FS_FRAGMENT) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: FRAGMENT", PORT_ID(mac));
			r = 1;
		}
		if (stat & GMR_FS_LONG_ERR) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: TOO LONG PACKET", PORT_ID(mac));
			r = 1;
		}
		if (stat & GMR_FS_MII_ERR) {
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: RECEIVE MII ERROR", PORT_ID(mac));
			r = 1;
		}
		if (stat & GMR_FS_BAD_FC) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: BAD FLOW CONTROL PACKET", PORT_ID(mac));
			r = 1;
		}
		if (stat & GMR_FS_UN_SIZE) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: RUNT PACKET", PORT_ID(mac));
			r = 1;
		}
		if (stat & GMR_FS_JABBER) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: JABBER DETECTED", PORT_ID(mac));
			r = 1;
		}
		if (__unlikely(!r) && stat & GMR_FS_RX_FF_OV) {
			if (sk->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_HW_WARNING, sk->dev_name, "PORT %c: RECEIVE FIFO OVERRUN", PORT_ID(mac));
			r = 1;
		}
		if (__likely(r)) return;
		if (stat & GMR_FS_GOOD_FC) return;
		if (__unlikely(!(stat & GMR_FS_RX_OK))) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: UNKNOWN ERROR (STATUS %08X)", PORT_ID(mac), stat);
			return;
		}
		if (__unlikely((stat >> GMR_FS_LEN_SHIFT) != len)) {
			if (sk->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sk->dev_name, "PORT %c: RECEIVE LENGTH MISMATCH (GMAC %d, BMU %d)", PORT_ID(mac), stat >> GMR_FS_LEN_SHIFT, len);
			return;
		}
	}
}

static void SK_INIT_ROOT(HANDLE *h, void *p)
{
	SK *sk = p;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = &sk->macs[0];
	h->op = sk->chiptype != CHIPTYPE_YUKON_2 ? &SKGE_OPERATIONS : &SKY2_OPERATIONS;
}

void *SK_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	MAC *mac = h->fnode;
	if (__likely(!strcmp(str, "^PORT=B"))) {
		if (__unlikely((h->flags & SK_PORTLOCKED) != 0)) return __ERR_PTR(-EPERM);
		if (__unlikely(mac->sk->n_macs == 1)) return __ERR_PTR(-ENODEV);
		h->flags |= SK_PORTLOCKED;
		h->fnode = mac + 1;
		return NULL;
	}
	if (__likely(!strcmp(str, "^PORT=A"))) {
		if (__unlikely((h->flags & SK_PORTLOCKED) != 0)) return __ERR_PTR(-EPERM);
		h->flags |= SK_PORTLOCKED;
		return NULL;
	}
	return *str == '^' ? __ERR_PTR(-EBADMOD) : __ERR_PTR(-ENOTDIR);
}

DECL_IOCALL(SK_IOCTL, SPL_SK, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	MAC *mac;
	int r;
	if (h->op != &SKGE_OPERATIONS && __unlikely(h->op != &SKY2_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_SK));
	mac = h->fnode;
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			ETHERNET_FILL_IF_TYPE(&t);
			t.mtu = mac->mtu;
			t.id = (IDTYPE_PCI | (mac->sk->id & IDTYPE_MASK)) ^ mac->mac_2;
			t.addr[0] = mac->address[0];
			t.addr[1] = mac->address[0] >> 8;
			t.addr[2] = mac->address[1];
			t.addr[3] = mac->address[1] >> 8;
			t.addr[4] = mac->address[2];
			t.addr[5] = mac->address[2] >> 8;
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (!memcmp(&mac->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&mac->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &mac->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			mac->link_state.seq++;
			WQ_WAKE_ALL_PL(&mac->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_MULTICAST: {
			if (__unlikely(RQ->v.len != 0)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			RQ->status = mac->mcast_state;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SET_MULTICAST: {
			__u8 mcast[ETHERNET_ADDRLEN];
			unsigned bit;
			int need_refresh;
			if (__unlikely(RQ->param == PARAM_MULTICAST_PROMISC)) { 
				bit = MCAST_PROMISC;
				goto have_bit;
			} else if (__unlikely(RQ->param == PARAM_MULTICAST_ALL)) {				bit = MCAST_ALL;
				goto have_bit;
			}
			if (__unlikely(RQ->param != PARAM_MULTICAST_ADDRESS)) {
				RQ->status = -ENOOP;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, mcast, ETHERNET_ADDRLEN)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			bit = SK_MULTICAST_HASH_BIT(mac, mcast);
			have_bit:
			need_refresh = WQ_EMPTY(&mac->mcast_table[bit]);
			WQ_WAIT_F(&mac->mcast_table[bit], RQ);
			if (need_refresh) {
				set_mcast:
				if (mac->sk->chiptype == CHIPTYPE_GENESIS) {
					XM_SET_MULTICAST_HASH(mac);
				} else {
					GM_SET_MULTICAST_HASH(mac);
				}
			}
			RETURN;
		}
		case IOCTL_IF_RESET_MULTICAST: {
			RQ->status = 0;
			CALL_AST(RQ);
			goto set_mcast;
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static int SK_UNLOAD(void *p, void **release, const char * const argv[]);
static void SK_DESTRUCT(SK *sk);

#ifndef SK_HIGH_INTERRUPT
#define IRQ_PARAMS	IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &sk->irq_ast
#else
#define IRQ_PARAMS	IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_SHARED, sk->chiptype != CHIPTYPE_YUKON_2 ? SKGE_IRQ_RT : SKY2_IRQ_RT, sk
#endif

int main(int argc, const char * const argv[])
{
	int errorlevel = 0;
	int r;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	const char *opt, *optend, *str;
	char *chip_name;
	unsigned long flags;
	char dev_name[__MAX_STR_LEN];
	SK0 sk0;
	SK *sk;
	int irq;
	unsigned chip_id, chip_rev, mac_cfg, n_macs, ramid, phy;
	int ips = -1, ips_100 = -1;
	char *net_input = NULL, *net_input_a = NULL, *net_input_b = NULL;
	int queues = 0, queues_a = 0, queues_b = 0;
	int mtu = 0, mtu_a = 0, mtu_b = 0;
	char *address_str_a = NULL, *address_str_b = NULL;
	int duplex = 0, duplex_a = 0, duplex_b = 0;
	int speed = 0, speed_a = 0, speed_b = 0;
	int port_2_defined;
	int i, j;
#ifndef USE_MMIO
	IO_RANGE range;
#endif
	union {
		OPENRQ openrq;
		char gstr[__MAX_STR_LEN];
	} u;
	const char * const *arg = argv;
	int state = 0;
	static const struct __param_table params[56] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2,
		"IPS", __PARAM_INT, 0, IPS_MAX,
		"IPS_100M", __PARAM_INT, 0, IPS_MAX,
		"INPUT", __PARAM_STRING, 1, MAXINT,
		"INPUT_A", __PARAM_STRING, 1, MAXINT,
		"INPUT_B", __PARAM_STRING, 1, MAXINT,
		"QUEUES", __PARAM_INT, 1, 2+1,
		"QUEUES_A", __PARAM_INT, 1, 2+1,
		"QUEUES_B", __PARAM_INT, 1, 2+1,
		"MTU", __PARAM_INT, ETHERNET_MTU, ETHERNET_JUMBO_MTU+1,
		"MTU_A", __PARAM_INT, ETHERNET_MTU, ETHERNET_JUMBO_MTU+1,
		"MTU_B", __PARAM_INT, ETHERNET_MTU, ETHERNET_JUMBO_MTU+1,
		"JUMBO", __PARAM_BOOL, ~0, ETHERNET_JUMBO_MTU,
		"JUMBO_A", __PARAM_BOOL, ~0, ETHERNET_JUMBO_MTU,
		"JUMBO_B", __PARAM_BOOL, ~0, ETHERNET_JUMBO_MTU,
		"ADDRESS", __PARAM_STRING, 1, MAXINT,
		"ADDRESS_A", __PARAM_STRING, 1, MAXINT,
		"ADDRESS_B", __PARAM_STRING, 1, MAXINT,
		"FULL_DUPLEX", __PARAM_BOOL, ~0, DUPLEX_FULL,
		"FULL_DUPLEX", __PARAM_BOOL, SPEED_AUTO_BIT, 0,
		"FULL_DUPLEX_A", __PARAM_BOOL, ~0, DUPLEX_FULL,
		"FULL_DUPLEX_A", __PARAM_BOOL, SPEED_AUTO_BIT, 0,
		"FULL_DUPLEX_B", __PARAM_BOOL, ~0, DUPLEX_FULL,
		"FULL_DUPLEX_B", __PARAM_BOOL, SPEED_AUTO_BIT, 0,
		"HALF_DUPLEX", __PARAM_BOOL, ~0, DUPLEX_HALF,
		"HALF_DUPLEX", __PARAM_BOOL, SPEED_AUTO_BIT, 0,
		"HALF_DUPLEX_A", __PARAM_BOOL, ~0, DUPLEX_HALF,
		"HALF_DUPLEX_A", __PARAM_BOOL, SPEED_AUTO_BIT, 0,
		"HALF_DUPLEX_B", __PARAM_BOOL, ~0, DUPLEX_HALF,
		"HALF_DUPLEX_B", __PARAM_BOOL, SPEED_AUTO_BIT, 0,
		"10M", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"10M", __PARAM_BOOL, ~0, SPEED_10,
		"10M_A", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"10M_A", __PARAM_BOOL, ~0, SPEED_10,
		"10M_B", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"10M_B", __PARAM_BOOL, ~0, SPEED_10,
		"100M", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"100M", __PARAM_BOOL, ~0, SPEED_100,
		"100M_A", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"100M_A", __PARAM_BOOL, ~0, SPEED_100,
		"100M_B", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"100M_B", __PARAM_BOOL, ~0, SPEED_100,
		"1G", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"1G", __PARAM_BOOL, ~0, SPEED_1000,
		"1G_A", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"1G_A", __PARAM_BOOL, ~0, SPEED_1000,
		"1G_B", __PARAM_BOOL, DUPLEX_AUTO_BIT, 0,
		"1G_B", __PARAM_BOOL, ~0, SPEED_1000,
		"AUTO", __PARAM_BOOL, ~0, DUPLEX_AUTO,
		"AUTO", __PARAM_BOOL, ~0, SPEED_AUTO,
		"AUTO_A", __PARAM_BOOL, ~0, DUPLEX_AUTO,
		"AUTO_A", __PARAM_BOOL, ~0, SPEED_AUTO,
		"AUTO_B", __PARAM_BOOL, ~0, DUPLEX_AUTO,
		"AUTO_B", __PARAM_BOOL, ~0, SPEED_AUTO,
		NULL, 0, 0, 0,
	};
	void *vars[56];

	PKT_CHECK_VERSION;

	vars[0] = &errorlevel;
	vars[1] = &errorlevel;
	vars[2] = &ips;
	vars[3] = &ips_100;
	vars[4] = &net_input;
	vars[5] = &net_input_a;
	vars[6] = &net_input_b;
	vars[7] = &queues;
	vars[8] = &queues_a;
	vars[9] = &queues_b;
	vars[10] = &mtu;
	vars[11] = &mtu_a;
	vars[12] = &mtu_b;
	vars[13] = &mtu;
	vars[14] = &mtu_a;
	vars[15] = &mtu_b;
	vars[16] = &address_str_a;
	vars[17] = &address_str_a;
	vars[18] = &address_str_b;
	vars[19] = &duplex;
	vars[20] = &speed;
	vars[21] = &duplex_a;
	vars[22] = &speed_a;
	vars[23] = &duplex_b;
	vars[24] = &speed_b;
	vars[25] = &duplex;
	vars[26] = &speed;
	vars[27] = &duplex_a;
	vars[28] = &speed_a;
	vars[29] = &duplex_b;
	vars[30] = &speed_b;
	vars[31] = &duplex;
	vars[32] = &speed;
	vars[33] = &duplex_a;
	vars[34] = &speed_a;
	vars[35] = &duplex_b;
	vars[36] = &speed_b;
	vars[37] = &duplex;
	vars[38] = &speed;
	vars[39] = &duplex_a;
	vars[40] = &speed_a;
	vars[41] = &duplex_b;
	vars[42] = &speed_b;
	vars[43] = &duplex;
	vars[44] = &speed;
	vars[45] = &duplex_a;
	vars[46] = &speed_a;
	vars[47] = &duplex_b;
	vars[48] = &speed_b;
	vars[49] = &duplex;
	vars[50] = &speed;
	vars[51] = &duplex_a;
	vars[52] = &speed_a;
	vars[53] = &duplex_b;
	vars[54] = &speed_b;
	vars[55] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			bads:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SK: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	port_2_defined = net_input_b || queues_b || mtu_b || address_str_b || duplex_b || speed_b;
	if (queues) {
		if (queues_a || queues_b) goto bads;
		queues_a = queues_b = queues;
	}
	if (mtu) {
		if (mtu_a || mtu_b) goto bads;
		mtu_a = mtu_b = mtu;
	}
	if (!mtu_a) mtu_a = DEFAULT_MTU;
	if (!mtu_b) mtu_b = DEFAULT_MTU;
	if (__unlikely(mtu_a > NET$MAX_PACKET_LENGTH)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SK: TOO HIGH MTU (%d > %d)", mtu_a, NET$MAX_PACKET_LENGTH);
		r = -EMSGSIZE;
		goto ret0;
	}
	if (__unlikely(mtu_b > NET$MAX_PACKET_LENGTH)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SK: TOO HIGH MTU (%d > %d)", mtu_b, NET$MAX_PACKET_LENGTH);
		r = -EMSGSIZE;
		goto ret0;
	}
	if (net_input) {
		if (net_input_a || net_input_b) goto bads;
		net_input_a = net_input_b = net_input;
	}
	if (!net_input_a) net_input_a = NET_INPUT_DEVICE ":";
	if (!net_input_b) net_input_b = NET_INPUT_DEVICE ":";
	if (duplex || speed) {
		if (duplex_a || duplex_b || speed_a || speed_b) goto bads;
		duplex_a = duplex_b = duplex;
		speed_a = speed_b = speed;
	}
	if (!duplex_a) {
		if (!speed_a || speed_a == SPEED_AUTO) duplex_a = DUPLEX_AUTO;
		else duplex_a = DEFAULT_DUPLEX;
	}
	if (!duplex_b) {
		if (!speed_b || speed_b == SPEED_AUTO) duplex_b = DUPLEX_AUTO;
		else duplex_b = DEFAULT_DUPLEX;
	}
	if (!speed_a) {
		if (duplex_a == DUPLEX_AUTO) speed_a = SPEED_AUTO;
		else speed_a = DEFAULT_SPEED;
	}
	if (!speed_b) {
		if (duplex_b == DUPLEX_AUTO) speed_b = SPEED_AUTO;
		else speed_b = DEFAULT_SPEED;
	}
	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SK: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	_snprintf(dev_name, sizeof(dev_name), "PKT$SK@" PCI_ID_FORMAT, id);
#ifndef USE_MMIO
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	sk0.io = PCI$READ_IO_RESOURCE(id, 1);
	if (!sk0.io) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE, RECOMPILE DRIVER WITH USE_MMIO", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	range.start = sk0.io;
	range.len = SK_IO_REGSPACE;
	range.name = dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, range.start, range.start + range.len - 1, strerror(-r));
		goto ret1;
	}
#else
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	sk0.mem = PCI$MAP_MEM_RESOURCE(id, 0, SK_REGSPACE);
	if (!sk0.mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(sk0.mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(sk0.mem)));
		r = __PTR_ERR(sk0.mem);
		goto ret1;
	}
#endif
	irq = PCI$READ_INTERRUPT_LINE(id);
#if defined(__ENDIAN_BIG)
#if defined(__ENDIAN_LITTLE)
	both endians used ???
#endif
	PCI$WRITE_CONFIG_DWORD(id, PCI_SK_OUR_2, PCI$READ_CONFIG_DWORD(id, PCI_SK_OUR_2) | PCI_SK_OUR_2_REV_BYTE_DESC);
#elif defined(__ENDIAN_LITTLE)
	PCI$WRITE_CONFIG_DWORD(id, PCI_SK_OUR_2, PCI$READ_CONFIG_DWORD(id, PCI_SK_OUR_2) & ~PCI_SK_OUR_2_REV_BYTE_DESC);
#else
	no endian !
#endif

	SK_BASIC_RESET((SK *)(void *)&sk0);
	chip_id = SK_READ_8(&sk0, SK_CHIP_REVISION);
	if (!Y2_REV(chip_id)) {
		mac_cfg = SK_READ_8(&sk0, SK_CONFIGURATION);
		n_macs = mac_cfg & SK_CONFIGURATION_SINGLE_MAC ? 1 : 2;
		chip_rev = mac_cfg & SK_CONFIGURATION_REV_MASK;
	} else {
		mac_cfg = SK_READ_8(&sk0, SK_EPROM_2);
		n_macs = (mac_cfg & SK_EPROM_2_DUAL_MAC) == SK_EPROM_2_DUAL_MAC && !(SK_READ_8(&sk0, SK_Y2_CLK_GATE) & SK_Y2_CLK_GATE_STATUS_LNK2) ? 2 : 1;
		mac_cfg = SK_READ_8(&sk0, SK_CONFIGURATION);
		chip_rev = mac_cfg & SK_CONFIGURATION_REV_MASK;
	}

	if (n_macs == 2) {
		int cap = PCI$FIND_CAPABILITY(id, -1, PCI_CAP_ID_PCIX);
		if (cap >= 0) {
			__u16 cmd = PCI$READ_CONFIG_WORD(id, cap + PCI_X_CMD);
			cmd &= ~PCI_X_CMD_MAX_SPLIT;
			PCI$WRITE_CONFIG_WORD(id, cap + PCI_X_CMD, cmd);
		}
	}

	if (n_macs == 1 && port_2_defined) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: THIS CARD HAS ONLY ONE PORT", dev_name);
		r = -EINVAL;
		goto ret2;
	}

	if (!queues_a) queues_a = HAVE_SYNC_QUEUE(chip_id) ? DEFAULT_QUEUES : 1;
	if (!queues_b) queues_b = HAVE_SYNC_QUEUE(chip_id) ? DEFAULT_QUEUES : 1;

	if (!HAVE_SYNC_QUEUE(chip_id) && (queues_a == 2 || queues_b == 2)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: YUKON 2 CHIPS DON'T HAVE TWO TRANSMIT QUEUES", dev_name);
		r = -EINVAL;
		goto ret2;
	}
	if (chip_id == SK_CHIP_REVISION_GENESIS && ((speed_a != SPEED_AUTO && speed_a != SPEED_1000) || (speed_b != SPEED_AUTO && speed_b != SPEED_1000))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: GENESIS CHIP DOESN'T SUPPORT OTHER SPEEDS THAN 1G", dev_name);
		r = -EINVAL;
		goto ret2;
	}
	if ((chip_id == SK_CHIP_REVISION_YUKON_FE || chip_id == SK_CHIP_REVISION_YUKON_FE_P) && (speed_a == SPEED_1000 || speed_b == SPEED_1000)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: YUKON FE CHIP DOESN'T SUPPORT 1G SPEED", dev_name);
		r = -EINVAL;
		goto ret2;
	}
	if ((chip_id == SK_CHIP_REVISION_YUKON_FE || chip_id == SK_CHIP_REVISION_YUKON_FE_P) && (mtu_a > ETHERNET_MTU || mtu_b > ETHERNET_MTU)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: YUKON FE CHIP DOESN'T SUPPORT JUMBO FRAMES", dev_name);
		r = -EINVAL;
		goto ret2;
	}


	sk = KERNEL$ALLOC_CONTIG_AREA(sizeof(SK) - (2 - n_macs) * sizeof(MAC), AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA | AREA_ALIGN, (unsigned long)SK_ALIGN);
	if (__IS_ERR(sk)) {
		r = __PTR_ERR(sk);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE DMA DESCRIPTORS", dev_name);
		goto ret2;
	}
#ifndef USE_MMIO
	sk->io = sk0.io;
#else
	sk->mem = sk0.mem;
#endif
	sk->id = id;
	sk->packet_offset = Y2_REV(chip_id) ? Y2_RECV_PACKET_OFFSET : 0;
	sk->n_macs = n_macs;
	sk->errorlevel = errorlevel;
	strcpy(sk->dev_name, dev_name);
	/*{
		for (i = 0; i < 16384; i += 4) {
			if (!(i & 31)) __debug_printf("%04x: ", i);
			if (i >= 0x180 && i < 0x1c00) __debug_printf("--------");
			else __debug_printf("%08x", SK_READ_32(sk, i));
			if (!((i + 4) & 31)) __debug_printf("\n");
		}
	}*/
	if (Y2_REV(chip_id)) {
		VDMA64 vdma;
		VDESC desc;
		desc.ptr = (unsigned long)&sk->y2_status;
		desc.len = sizeof sk->y2_status;
		desc.vspace = &KERNEL$VIRTUAL;
		vdma.spl = SPL_X(SPL_ZERO);
		RAISE_SPL(SPL_VSPACE);
		sk->stat_dma64unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_READ | PF_WRITE, &vdma);
		sk->stat_dma64addr = vdma.ptr;
	} else {
		sk->stat_dma64unlock = NULL;
	}
	for (i = 0; i < n_macs; i++) {
		MAC *mac = &sk->macs[i];
		VDMA64 vdma;
		VDESC desc;
#ifndef USE_MMIO
		mac->io = sk->io;
#else
		mac->mem = sk->mem;
#endif
		mac->mac_2 = !i ? 0 : -1;
		mac->sk = sk;
		mac->queues = !i ? queues_a : queues_b;
		mac->mtu = !i ? mtu_a : mtu_b;
		mac->duplex = !i ? duplex_a : duplex_b;
		mac->speed = !i ? speed_a : speed_b;
		mac->first_recv = 0;
		mac->n_recv = 0;
		mac->outstanding = 0;
		mac->bounce_reserved = 0;
		for (j = 0; j < 2; j++) {
#ifndef USE_MMIO
			mac->xmit_queue[j].io = sk->io;
#else
			mac->xmit_queue[j].mem = sk->mem;
#endif
			mac->xmit_queue[j].no_stfw = 0;
			mac->xmit_queue[j].cs_reg = (!Y2_REV(chip_id) ? SK_TX_CS : SK_Y2_TX_PUT_IDX) + i * ADD_MAC2_SK_TX + j * ADD_ASYNC_SK_TX;
			mac->xmit_queue[j].first_sent = 0;
			mac->xmit_queue[j].n_sent = 0;
			if (mac->queues == 1 || !j)
				mac->xmit_queue[j].max_sent = DEFAULT_SEND;
			else
				mac->xmit_queue[j].max_sent = !Y2_REV(chip_id) ? SKGE_MAX_SEND : SKY2_MAX_SEND;
			mac->xmit_queue[j].mac = mac;
			mac->xmit_queue[j].queue_2 = !j ? 0 : -1;
			if (!Y2_REV(chip_id)) {
				desc.ptr = (unsigned long)&mac->xmit_queue[j].u.xmit;
				desc.len = sizeof mac->xmit_queue[j].u.xmit;
			} else {
				desc.ptr = (unsigned long)&mac->xmit_queue[j].u.y2_xmit;
				desc.len = sizeof mac->xmit_queue[j].u.y2_xmit;
			}
			desc.vspace = &KERNEL$VIRTUAL;
			vdma.spl = SPL_X(SPL_ZERO);
			RAISE_SPL(SPL_VSPACE);
			mac->xmit_queue[j].snd_dma64unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_READ | PF_WRITE, &vdma);
			mac->xmit_queue[j].snd_dma64addr = vdma.ptr;
#if __DEBUG >= 2
			mac->xmit_queue[j].snd_maps = 0;
#endif
		}

/* will be rewritten with real values --- initialize just in case of exit */
		mac->packet_input = -1;
		mac->queue = NULL;
		mac->force_address = 0;
		if (!Y2_REV(chip_id)) {
			desc.ptr = (unsigned long)&mac->u.recv;
			desc.len = sizeof mac->u.recv;
		} else {
			desc.ptr = (unsigned long)&mac->u.y2_recv;
			desc.len = sizeof mac->u.y2_recv;
		}
		desc.vspace = &KERNEL$VIRTUAL;
		vdma.spl = SPL_X(SPL_ZERO);
		RAISE_SPL(SPL_VSPACE);
		mac->rcv_dma64unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_READ | PF_WRITE, &vdma);
		mac->rcv_dma64addr = vdma.ptr;
		WQ_INIT(&mac->link_state_wait, "SK$LINK_STATE_WAIT");
		memset(&mac->link_state, 0, sizeof(LINK_STATE));
		for (j = 0; j < N_MCAST; j++) WQ_INIT(&mac->mcast_table[j], "SK$MCAST_TABLE");
		mac->mcast_state = STATUS_MULTICAST_NONE;
#if __DEBUG >= 2
		mac->rcv_maps = 0;
#endif
		memset(&mac->recv_packets, 0, sizeof mac->recv_packets);
		for (j = 2 - mac->queues; j < 2; j++) {
			mac->xmit_queue[j].timer.fn = !Y2_REV(chip_id) ? SKGE_TIMER_FN : SKY2_TIMER_FN;
			INIT_TIMER(&mac->xmit_queue[j].timer);
			SET_TIMER_NEVER(&mac->xmit_queue[j].timer);
			mac->xmit_queue[j].timeout_cycles = SKGE_TX_TIMEOUT_CYCLES;
		}
	}

	ramid = SK_READ_8(sk, SK_EPROM_0);

	sk->chiprev = chip_rev;
	if (chip_id == SK_CHIP_REVISION_GENESIS) {
		sk->chiptype = CHIPTYPE_GENESIS;
		sk->chipsubtype = 0;
		sk->host_clock = GENESIS_HOST_CLOCK;
		switch (ramid) {
			case 1:
				sk->ramsize = 512 * 1024;
				sk->ramoffs = 0;
				break;
			case 2:
				sk->ramsize = 1024 * 1024;
				sk->ramoffs = 0;
				break;
			case 3:
				sk->ramsize = 1024 * 1024;
				sk->ramoffs = 512 * 1024;
				break;
			case 4:
				sk->ramsize = 2048 * 1024;
				sk->ramoffs = 0;
				break;
			default:
				KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "UNKNOWN RAM SIZE ID %02X", ramid);
				cant_init:
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CARD CAN'T BE INITIALIZED", dev_name);
				r = -ENXIO;
				goto ret3;
		}
	} else {
		__u32 ac;
		if (ramid || Y2_REV(chip_id)) sk->ramsize = ramid * 4096;
		else sk->ramsize = 128 * 1024;
		sk->ramoffs = 0;
		if (!Y2_REV(chip_id)) {
			sk->chiptype = CHIPTYPE_YUKON;
			sk->chipsubtype = 0;
			sk->host_clock = YUKON_HOST_CLOCK;
			if (chip_id == SK_CHIP_REVISION_YUKON_LITE) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_LITE;
			}
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE) {
				if (chip_rev == SK_CONFIGURATION_LITE_A1) sk->chipsubtype = CHIPSUBTYPE_YUKON_LITE_A1;
				if (chip_rev == SK_CONFIGURATION_LITE_A3) sk->chipsubtype = CHIPSUBTYPE_YUKON_LITE_A3;
			}
			ac = SK_READ_32(sk, SK_EPROM_ADDR_COUNTER);
			SK_WRITE_8(sk, SK_EPROM_ADDR_COUNTER + 3, 0xff);
			if (SK_READ_8(sk, SK_EPROM_ADDR_COUNTER + 3)) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_LITE_A0;
			}
			SK_WRITE_32(sk, SK_EPROM_ADDR_COUNTER, ac);
		} else {
			__u32 our_status;
			sk->chiptype = CHIPTYPE_YUKON_2;
			sk->chipsubtype = 0;
			if (chip_id == SK_CHIP_REVISION_YUKON_XL) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_XL;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_EC_U) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_EC_U;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_EX) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_EX;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_EC) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_EC;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_FE) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_FE;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_FE_P) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_FE_P;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_SUPR) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_SUPR;
			} else if (chip_id == SK_CHIP_REVISION_YUKON_UL_2) {
				sk->chipsubtype = CHIPSUBTYPE_YUKON_UL_2;
			}
			our_status = SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + PCI_SK_OUR_STATUS);
			if (our_status & PCI_SK_OUR_STATUS_PCI_X) {
				SK_TESTMODE_ON(sk);
				if (!SK_READ_8(sk, SK_Y2_CONFIGURATION_FILE + PCI_CACHE_LINE_SIZE)) SK_WRITE_8(sk, SK_Y2_CONFIGURATION_FILE + PCI_CACHE_LINE_SIZE, 2);
				if (our_status & PCI_SK_OUR_STATUS_PCIX) {
					SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + PCI_SK_OUR_1, SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + PCI_SK_OUR_1) | PCI_Y2_OUR_1_CLS_OPT);
				}
				SK_TESTMODE_OFF(sk);
			}
			SK_TESTMODE_ON(sk);
			SK_WRITE_32(sk, SK_Y2_CLK_CTRL, SK_Y2_CLK_CTRL_DISABLE);
			SK_TESTMODE_OFF(sk);
			/*__debug_printf("y2: subtype %d, clock %d\n", sk->chipsubtype, sk->host_clock);*/
			switch (sk->chipsubtype) {
				case CHIPSUBTYPE_YUKON_FE:
					sk->host_clock = YUKON_2_FE_PEX_HOST_CLOCK;
					break;
				case CHIPSUBTYPE_YUKON_FE_P:
					sk->host_clock = YUKON_2_FE_P_PEX_HOST_CLOCK;
					break;
				case CHIPSUBTYPE_YUKON_XL:
					sk->host_clock = YUKON_2_PCIX_HOST_CLOCK;
					break;
				default:
					sk->host_clock = YUKON_2_PEX_HOST_CLOCK;
					break;
			}
			if (!sk->ramsize)
				sk->packet_offset = 0;
		}
		SK_WRITE_8(sk, SK_POWER, SK_POWER_VAUX_ENA | SK_POWER_VCC_ENA | SK_POWER_VAUX_OFF | SK_POWER_VCC_ON);
		if (Y2_REV(chip_id)) {
			if (sk->chipsubtype == CHIPSUBTYPE_YUKON_XL && sk->chiprev > SK_CONFIGURATION_XL_0) {
				SK_WRITE_8(sk, SK_Y2_CLK_GATE, SK_Y2_CLK_GATE_PCI_CLK_LNK1 | SK_Y2_CLK_GATE_COR_CLK_LNK1 | SK_Y2_CLK_GATE_GAT_CLK_LNK1 | SK_Y2_CLK_GATE_PCI_CLK_LNK2 | SK_Y2_CLK_GATE_COR_CLK_LNK2 | SK_Y2_CLK_GATE_GAT_CLK_LNK2);
			} else {
				SK_WRITE_8(sk, SK_Y2_CLK_GATE, 0);
			}
		}
		if (ADV_POWER_CTL(sk)) {
			__u32 x;
			SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + 0x80, 0);
			x = SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + 0x84);
			x &= 0xf000;
			SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + 0x84, x);
			x = SK_READ_32(sk, SK_Y2_CONFIGURATION_FILE + 0x88);
			x &= 0x18000000;
			SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + 0x88, x);
			SK_WRITE_32(sk, SK_Y2_CONFIGURATION_FILE + 0x94, 0);
			x = SK_READ_32(sk, SK_GENERAL_PURPOSE);
			x |= SK_GENERAL_PURPOSE_STAT_RACE_D;
			SK_WRITE_32(sk, SK_GENERAL_PURPOSE, x);
			SK_READ_32(sk, SK_GENERAL_PURPOSE);
		}
	}

	for (i = 0; i < n_macs; i++) {
		MAC *mac = &sk->macs[i];
		mac->chiptype = sk->chiptype;
		mac->chipsubtype = sk->chipsubtype;
		mac->chiprev = sk->chiprev;
		mac->recv_packet_size = mac->mtu;
		if (sk->packet_offset) {
			mac->recv_packet_size += sk->packet_offset;
			mac->recv_packet_size += ETHERNET_HEADER;
			mac->recv_packet_size -= Y2_COPY_FIXED;
			mac->recv_packet_size = (mac->recv_packet_size + Y2_COPY_BLOCK - 1);
			mac->recv_packet_size -= mac->recv_packet_size % Y2_COPY_BLOCK;
			mac->recv_packet_size += Y2_COPY_FIXED;
			mac->recv_packet_size -= ETHERNET_HEADER;
		}
		{
			unsigned r2 = mac->recv_packet_size;
			r2 += ETHERNET_HEADER;
			r2 = (r2 + Y2_RECV_ALIGN_TAIL - 1) & ~(Y2_RECV_ALIGN_TAIL - 1);
			r2 -= ETHERNET_HEADER;
			r2 += sk->packet_offset;
			if (r2 > mac->recv_packet_size) mac->recv_packet_size = r2;
		}
		for (j = 0; j < 2; j++) {
			mac->xmit_queue[j].chiptype = sk->chiptype;
			mac->xmit_queue[j].chipsubtype = sk->chipsubtype;
			mac->xmit_queue[j].chiprev = sk->chiprev;
			mac->xmit_queue[j].new_checksum = !!NEW_CHECKSUM(sk);
		}
	}

	if (!SUPPORT_EC_A1 && (sk)->chipsubtype == CHIPSUBTYPE_YUKON_EC && (sk)->chiprev == SK_CONFIGURATION_EC_A1) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "ENGINEERING SAMPLE YUKON EC A1 NOT SUPPORTED");
		goto cant_init;
	}

	sk->ips = ips;
	sk->ips_100 = ips_100;
	if (sk->ips == -1) {
		sk->ips = !Y2_BUG_420(sk) ? IPS_DEFAULT_1G : 0;
		if (sk->ips_100 == -1) sk->ips_100 = !Y2_BUG_420(sk) ? IPS_DEFAULT_100M : 0;
	}
	if (sk->ips_100 == -1) sk->ips_100 = sk->ips;

	sk->media = SK_READ_8(sk, SK_PMD_TYPE);
	sk->copper = sk->media == 'T' || sk->media == '1';
	if (!Y2_REV(chip_id)) phy = SK_READ_8(sk, SK_EPROM_1) & 0xf;
	else phy = 0;

	if (sk->chiptype == CHIPTYPE_GENESIS) {
		switch (phy) {
			case SK_EPROM_1_PHY_XMAC:
				sk->phyaddr = SK_XMACII_PHY_ADDR_XMAC;
				break;
			case SK_EPROM_1_PHY_BCOM:
				sk->phyaddr = SK_XMACII_PHY_ADDR_BCOM;
				break;
			/*
			case SK_EPROM_1_PHY_LONE:
				sk->phyaddr = SK_XMACII_PHY_ADDR_LONE;
				break;
			case SK_EPROM_1_PHY_NAT:
				sk->phyaddr = SK_XMACII_PHY_ADDR_NAT;
				break;
			*/
			default:
				KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "UNKNOWN PHY TYPE %02X", phy);
				goto cant_init;
		}
	} else if (!Y2_REV(chip_id)) {
		if (phy < SK_EPROM_1_PHY_MARV_COPPER) phy = SK_EPROM_1_PHY_MARV_COPPER, sk->copper = 1;
		if (phy != SK_EPROM_1_PHY_MARV_COPPER) {
			if (!sk->copper) phy = SK_EPROM_1_PHY_MARV_FIBER;
			else phy = SK_EPROM_1_PHY_MARV_COPPER;
		}
		sk->phyaddr = 0;
		SK_WRITE_32(sk, SK_I2C_HW_IRQ, SK_I2C_HW_IRQ_CLEAR);
	} else {
		if (sk->media != 'L' && sk->media != 'S' && sk->media != 'P') phy = SK_EPROM_1_PHY_MARV_COPPER, sk->copper = 1;
		else phy = SK_EPROM_1_PHY_MARV_FIBER, sk->copper = 0;
		sk->phyaddr = 0;
		SK_WRITE_32(sk, SK_I2C_HW_IRQ, SK_I2C_HW_IRQ_CLEAR);
	}
	/*
	if (sk->chiptype == CHIPTYPE_YUKON) {
		__u8 test = SK_READ_8(sk, SK_EPROM_3);
		if (test & SK_EPROM_3_SELF_TEST_RESULT) {
			KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "SELF TEST FAILED, RESULT %02X", test);
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SELF TEST FAILED", dev_name);
			r = -EIO;
			goto ret3;
		}
	}
	*/
	sk->phytype = phy;

	for (i = 0; i < n_macs; i++) {
		MAC *mac = &sk->macs[i];
		char *a = !i ? address_str_a : address_str_b;
		if (a) {
			__u8 addr[6];
			if (NET$PARSE_ETHERNET_ADDRESS(a, addr) || !NET$VALID_ETHERNET_ADDRESS(addr)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SK: INVALID ADDRESS: %s", a);
				r = -EBADSYN;
				goto ret3;
			}
			mac->force_address = 1;
			mac->address[0] = addr[0] + (addr[1] << 8);
			mac->address[1] = addr[2] + (addr[3] << 8);
			mac->address[2] = addr[4] + (addr[5] << 8);
		}

		r = KERNEL$RESERVE_BOUNCE(dev_name, N_DESCRIPTORS * (1 + mac->queues), mac->mtu + ETHERNET_HEADER, BOUNCE_DMA64);
		if (r) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE RECEIVE BOUNCE BUFFERS FOR MAC %d: %s", dev_name, i, strerror(-r));
			goto ret3;
		}
		mac->bounce_reserved = 1;

		if (sk->chiptype != CHIPTYPE_YUKON_2) {
			mac->max_n_recv = N_DESCRIPTORS;
			for (j = 0; j < N_DESCRIPTORS; j++) {
				PACKET *p;
				RAISE_SPL(SPL_SK);
				again_alloc:
				ALLOC_PACKET(p, mac->mtu, NULL, SPL_SK, {
					if (__unlikely(r = NET$MEMWAIT_SYNC())) {
						if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE PACKETS", dev_name);
						goto ret3;
					}
					goto again_alloc;
				});
				mac->recv_packets[j] = p;
				mac->n_recv++;
				SKGE_MAP_RCV_PACKET(mac, p, j);
				LOWER_SPL(SPL_ZERO);
			}
		} else {
			if (!Y2_BUG_420(sk)) {
				mac->max_n_recv = N_DESCRIPTORS - 2;
			} else {
				mac->max_n_recv = MAX_420_RECV_DESC / sk->n_macs;
			}
			while (!SK_RCVQ_FULL(mac)) {
				PACKET *p;
				RAISE_SPL(SPL_SK);
				again_alloc_2:
				ALLOC_PACKET(p, mac->recv_packet_size, NULL, SPL_SK, {
					if (__unlikely(r = NET$MEMWAIT_SYNC())) {
						if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE PACKETS", dev_name);
						goto ret3;
					}
					goto again_alloc_2;
				});
				SKY2_ADD_RCV_PACKET(mac, p);
				LOWER_SPL(SPL_ZERO);
			}
		}
	}

	for (i = 0; i < n_macs; i++) {
		MAC *mac = &sk->macs[i];
		u.openrq.flags = O_WRONLY;
		u.openrq.path = !i ? net_input_a : net_input_b;
		u.openrq.cwd = KERNEL$CWD();
		SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
		if (u.openrq.status < 0) {
			if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", dev_name, net_input, strerror(-u.openrq.status));
			r = u.openrq.status;
			goto ret3;
		}
		mac->packet_input = u.openrq.status;
		if (__unlikely(r = NETQUE$ALLOC_QUEUE(&mac->queue))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", dev_name, strerror(-r));
			goto ret3;
		}
	}

#ifndef USE_MMIO
	sk->range.start = range.start;
	sk->range.len = range.len;
	sk->range.name = sk->dev_name;
	KERNEL$UNREGISTER_IO_RANGE(&range);
	range.name = NULL;
	if ((r = KERNEL$REGISTER_IO_RANGE(&sk->range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, sk->range.start, sk->range.start + sk->range.len - 1, strerror(-r));
		goto ret3;
	}
#endif

	/*memset(&sk->rnd_ctx, 0, sizeof sk->rnd_ctx);*/
	sk->irq_ast.fn = sk->chiptype != CHIPTYPE_YUKON_2 ? SKGE_IRQ : SKY2_IRQ;
	RAISE_SPL(SPL_SK);
#ifdef SK_HIGH_INTERRUPT
	sk->isr = 0;
#endif
	if ((r = KERNEL$REQUEST_IRQ(irq, &sk->irq_ctrl, IRQ_PARAMS, dev_name)) < 0) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret35;
	}
	SK_CONTINUE_RESET(sk);
	LOWER_SPL(SPL_ZERO);
	for (i = 0; i < n_macs; i++) {
		__u8 addr[6];
		addr[0] = sk->macs[i].address[0] & 0xff;
		addr[1] = sk->macs[i].address[0] >> 8;
		addr[2] = sk->macs[i].address[1] & 0xff;
		addr[3] = sk->macs[i].address[1] >> 8;
		addr[4] = sk->macs[i].address[2] & 0xff;
		addr[5] = sk->macs[i].address[2] >> 8;
		if (!NET$VALID_ETHERNET_ADDRESS(addr)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INVALID ADDRESS FOR PORT %c: %s", PORT_ID(&sk->macs[i]), NET$PRINT_ETHERNET_ADDRESS(u.gstr, addr));
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID ETHERNET ADDRESS", dev_name);
			r = -EIO;
			goto ret4;
		}
	}

	r = KERNEL$REGISTER_DEVICE(dev_name, "SK.SYS", 0, sk, SK_INIT_ROOT, NULL, NULL, NULL, SK_UNLOAD, &sk->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret4;
	}
	sk->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);

	/*_printf("found %s at %x\n", chip_name, (int)id);
	_printf("chip %02x, mac %02x\n", chip_id, mac_cfg);
	_printf("ramid %02x\n", ramid);
	_printf("pmd %c\n", sk->media);
	_printf("phy %02x\n", SK_READ_8(sk, SK_EPROM_1));
	_printf("skcs %08x\n", SK_READ_32(sk, SK_CS));*/

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(u.gstr, id));
#ifndef USE_MMIO
	_printf("%s: IO " IO_FORMAT ", IRQ %d, CHIP %s, REV %X\n", dev_name, sk->io, irq,
#else
	_printf("%s: MEM %"__64_format"X, IRQ %d, CHIP %s, REVISION %X\n", dev_name, PCI$READ_MEM_RESOURCE(id, 0, NULL), irq,
#endif
		sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE ? "YUKON LITE" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE_A0 ? "YUKON LITE A0" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE_A1 ? "YUKON LITE A1" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_LITE_A3 ? "YUKON LITE A3" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_XL ? "YUKON 2 XL" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_EC_U ? "YUKON 2 EC ULTRA" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_EX ? "YUKON 2 EX" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_EC ? "YUKON 2 EC" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_FE ? "YUKON 2 FE" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_FE_P ? "YUKON 2 FE+" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_SUPR ? "YUKON 2 SUPREME" :
		sk->chipsubtype == CHIPSUBTYPE_YUKON_UL_2 ? "YUKON 2 ULTRA 2" :
		sk->chiptype == CHIPTYPE_GENESIS ? "GENESIS" :
		sk->chiptype == CHIPTYPE_YUKON ? "YUKON" :
		sk->chiptype == CHIPTYPE_YUKON_2 ? "YUKON 2" :
		"UNKNOWN",
		sk->chiprev >> 4);
	_printf("%s: ADDRESS %02X:%02X:%02X:%02X:%02X:%02X%s\n", dev_name, sk->macs[0].address[0] & 0xff, sk->macs[0].address[0] >> 8, sk->macs[0].address[1] & 0xff, sk->macs[0].address[1] >> 8, sk->macs[0].address[2] & 0xff, sk->macs[0].address[2] >> 8, address_str_a ? " (OVERRIDEN)" : "");
	if (sk->n_macs == 2) _printf("%s:/^PORT=B ADDRESS %02X:%02X:%02X:%02X:%02X:%02X%s\n", dev_name, sk->macs[1].address[0] & 0xff, sk->macs[1].address[0] >> 8, sk->macs[1].address[1] & 0xff, sk->macs[1].address[1] >> 8, sk->macs[1].address[2] & 0xff, sk->macs[1].address[2] >> 8, address_str_b ? " (OVERRIDEN)" : "");

	return 0;

	ret4:
	KERNEL$MASK_IRQ(sk->irq_ctrl);
	SK_DESTRUCT(sk);
	KERNEL$RELEASE_IRQ(sk->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_PARAMS);
#ifndef USE_MMIO
	KERNEL$UNREGISTER_IO_RANGE(&sk->range);
#endif
	if (0) {
		ret35:
#ifndef USE_MMIO
		KERNEL$UNREGISTER_IO_RANGE(&sk->range);
#endif
		ret3:
		SK_DESTRUCT(sk);
	}
	KERNEL$FREE_CONTIG_AREA(sk, sizeof(SK) - (2 - n_macs) * sizeof(MAC));
	ret2:
#ifdef USE_MMIO
	PCI$UNMAP_MEM_RESOURCE(id, sk0.mem, SK_REGSPACE);
#else
	if (range.name) KERNEL$UNREGISTER_IO_RANGE(&range);
#endif
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static void SK_DESTRUCT(SK *sk)
{
	int i, j;
	RAISE_SPL(SPL_SK);
	SK_BASIC_RESET(sk);
	SK_SHUTDOWN_RESET(sk);
	for (i = 0; i < sk->n_macs; i++) {
		sk->macs[i].max_n_recv = 0;
		for (j = 2 - sk->macs[i].queues; j < 2; j++)
			KERNEL$DEL_TIMER(&sk->macs[i].xmit_queue[j].timer);
	}
	LOWER_SPL(SPL_ZERO);
	while (sk->macs[0].outstanding || (sk->n_macs == 2 && sk->macs[1].outstanding)) {
		if (sk->macs[0].outstanding < 0) KERNEL$SUICIDE("SK_DESTRUCT: PORT A: OUTSTANDING %d", sk->macs[0].outstanding);
		if (sk->n_macs == 2 && sk->macs[1].outstanding < 0) KERNEL$SUICIDE("SK_DESTRUCT: PORT B: OUTSTANDING %d", sk->macs[1].outstanding);
		KERNEL$SLEEP(1);
	}
	for (i = 0; i < sk->n_macs; i++) {
		RAISE_SPL(SPL_SK);
		if (sk->chiptype != CHIPTYPE_YUKON_2) {
			SKGE_TEAR_DOWN_RECV_PACKETS(&sk->macs[i], 1);
			SKGE_TEAR_DOWN_SENT_PACKETS(&sk->macs[i]);
		} else {
			SKY2_TEAR_DOWN_RECV_PACKETS(&sk->macs[i], 1);
			SKY2_TEAR_DOWN_SENT_PACKETS(&sk->macs[i]);
		}
		LOWER_SPL(SPL_ZERO);
		sk->macs[i].rcv_dma64unlock(sk->macs[i].rcv_dma64addr);
		for (j = 0; j < 2; j++) {
			sk->macs[i].xmit_queue[j].snd_dma64unlock(sk->macs[i].xmit_queue[j].snd_dma64addr);
		}
		if (sk->macs[i].packet_input != -1)
			KERNEL$FAST_CLOSE(sk->macs[i].packet_input);
		if (sk->macs[i].queue)
			NETQUE$FREE_QUEUE(sk->macs[i].queue);
		if (sk->macs[i].bounce_reserved)
			KERNEL$UNRESERVE_BOUNCE(sk->dev_name, N_DESCRIPTORS * (1 + sk->macs[i].queues), sk->macs[i].mtu + ETHERNET_HEADER, BOUNCE_DMA64);
#if __DEBUG >= 2
		if (__unlikely(sk->macs[i].rcv_maps | sk->macs[i].xmit_queue[0].snd_maps | sk->macs[i].xmit_queue[1].snd_maps))
			KERNEL$SUICIDE("SK_DESTRUCT: %d RECEIVE, %d SYNC SEND, %d ASYNC SEND MAPS LEAKED ON MAC %d", sk->macs[i].rcv_maps, sk->macs[i].xmit_queue[0].snd_maps, sk->macs[i].xmit_queue[1].snd_maps, i);
#endif
		WQ_WAKE_ALL(&sk->macs[i].link_state_wait);
		for (j = 0; j < N_MCAST; j++) WQ_WAKE_ALL(&sk->macs[i].mcast_table[j]);
	}
	if (sk->stat_dma64unlock) sk->stat_dma64unlock(sk->stat_dma64addr);
}

static int SK_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	SK *sk = p;
	if ((r = KERNEL$DEVICE_UNLOAD(sk->lnte, argv))) return r;
	KERNEL$MASK_IRQ(sk->irq_ctrl);
	SK_DESTRUCT(sk);
	KERNEL$RELEASE_IRQ(sk->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_PARAMS);
#ifdef USE_MMIO
	PCI$UNMAP_MEM_RESOURCE(sk->id, sk->mem, SK_REGSPACE);
#else
	KERNEL$UNREGISTER_IO_RANGE(&sk->range);
#endif
	PCI$FREE_DEVICE(sk->id);
	*release = sk->dlrq;
	KERNEL$FREE_CONTIG_AREA(sk, sizeof(SK) - (2 - sk->n_macs) * sizeof(MAC));
	return 0;
}

