#include <SPAD/SYSLOG.H>
#include <STRING.H>
#include "MCAST.H"

#include "TCPIP.H"

#define ARP_LOCK		1
#define ARP_WAIT		2
#define ARP_INVALID		4

static int arp_outstanding = 0;

typedef struct {
	LIST_ENTRY hash;
	LIST_ENTRY list;
	in_addr_t host;
	unsigned long id;
	int flags;
	__u8 pad[3];
	__u8 hwaddr[MAX_ADDR_LEN + 1];
	LIST_HEAD queue;
	jiffies_lo_t last_used;
	int tries;
	TIMER timer;
} ARP;

#define ARP_HASH_SIZE	4096

static XLIST_HEAD arp_hash[ARP_HASH_SIZE];
static DECL_LIST(all_arps);

#define ARP_HASH(host)	(ntohl((host)) & (ARP_HASH_SIZE - 1))

static struct __slhead arp_slab;

static DECL_TIMER(arp_reaper);

unsigned ARP_TIME_TO_LIVE = ARP_INIT_TIME_TO_LIVE;

static void ARP_RESEND(TIMER *t);
static void ARP_DEL(ARP *a, int err);
static void ARP_DEL_PACKETS(ARP *a, int err);
static void ARP_SEND_PACKETS(ARP *a);

static void ARP_CTOR(struct __slhead *g, void *a_)
{
	ARP *a = a_;
	INIT_LIST(&a->queue);
	INIT_TIMER(&a->timer);
	a->timer.fn = ARP_RESEND;
}

static void INVALID_ARP_PACKET(PACKET *p)
{
	static char a1[__MAX_STR_LEN];
	static char a2[__MAX_STR_LEN];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "INVALID ARP PACKET RECEIVED: DSTADDR %s, SRCADDR %s, LENGTH %X, HWTYPE %04X, PROTOCOL %04X, HLEN %02X, PLEN %02X, OPERATION %04X", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), p->data_length, ntohs(arp(p)->hwtype), ntohs(arp(p)->protocol), arp(p)->hlen, arp(p)->plen, ntohs(arp(p)->operation));
}

static void ARP_DOUBLE(in_addr_t host, int hw1len, __u8 *hw1addr, int hw2len, __u8 *hw2addr)
{
	static char a1[16];
	static char a2[__MAX_STR_LEN];
	static char a3[__MAX_STR_LEN];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "TWO HOSTS RESPONDED WITH ARP FOR IP ADDRESS %s: %s AND %s", PRINT_IP(a1, host), NET$PRINT_HWADDR(a2, hw1addr, hw1len), NET$PRINT_HWADDR(a3, hw2addr, hw2len));
}

static void ARP_RESPONSE(PACKET *p)
{
	ARP *a;
	in_addr_t host;
	unsigned h;
	memcpy(&host, arpaddr(p) + p->addrlen, sizeof(in_addr_t));
	h = ARP_HASH(host);
	XLIST_FOR_EACH(a, &arp_hash[h], ARP, hash) {
		if (__likely(a->host == host) && __likely(a->id == p->id)) {
			if (__unlikely((a->flags & (ARP_LOCK | ARP_WAIT)) != ARP_LOCK)) {
				if (__unlikely(a->hwaddr[0] != arp(p)->hlen) || __unlikely(memcmp(a->hwaddr + 1, arpaddr(p), a->hwaddr[0]))) {
					ARP_DOUBLE(host, a->hwaddr[0], a->hwaddr + 1, arp(p)->hlen, arpaddr(p));
					a->flags &= ~ARP_LOCK;
					a->flags |= ARP_INVALID;
					ARP_SEND_PACKETS(a);
					return;
				}
				return;
			}
			memcpy(a->hwaddr + 1, arpaddr(p), a->hwaddr[0] = arp(p)->hlen);
			KERNEL$DEL_TIMER(&a->timer);
			if (__likely(!local_security)) {
				a->flags &= ~ARP_LOCK;
				ARP_SEND_PACKETS(a);
			} else {
				a->flags |= ARP_WAIT;
				KERNEL$SET_TIMER(1, &a->timer);
			}
			return;
		}
	}
}

void ARP_INPUT(PACKET *p)
{
	iplog(p);
	if (__unlikely(p->data_length < sizeof(struct arphdr))) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "TOO SHORT ARP PACKET, LENGTH %X", p->data_length);
		goto drop;
	}
	if (__unlikely(arp(p)->protocol != NET_PROTOCOL_IP) ||
	    __unlikely(arp(p)->hlen != p->addrlen) ||
	    __unlikely(arp(p)->plen != sizeof(in_addr_t)) ||
	    __unlikely(p->data_length < sizeof(struct arphdr) + 2 * sizeof(in_addr_t) + (arp(p)->plen << 1))) {
		if (arp(p)->protocol != NET_PROTOCOL_IP) goto drop;
		if (!arp(p)->hlen) goto drop;	/* invalid packet, but some hosts send it */
		INVALID_ARP_PACKET(p);
		goto drop;
	}
	if (arp(p)->operation == ARP_OP_ARP_REQUEST) {
		IF_ARP_REQUEST(p);
	} else if (__likely(arp(p)->operation == ARP_OP_ARP_RESPONSE)) {
		ARP_RESPONSE(p);
	} else if (__likely(arp(p)->operation == ARP_OP_RARP_REQUEST) ||
	           __likely(arp(p)->operation == ARP_OP_RARP_RESPONSE)) {
		goto drop;
	} else {
		INVALID_ARP_PACKET(p);
	}
	drop:
	p->status = 0;
	CALL_PKT(p);
}

static int ARP_ZAP_ENTRY(void)
{
	ARP *a;
	if (__unlikely(LIST_EMPTY(&all_arps))) return -1;
	a = LIST_STRUCT(all_arps.prev, ARP, list);
	ARP_DEL(a, -EAGAIN);
	return 0;
}

static void ARP_SEND_PACKETS(ARP *a)
{
	XLIST_HEAD q;
	INIT_XLIST(&q);
	while (!LIST_EMPTY(&a->queue)) {
		PACKET *p = PKT_Q_RQ(a->queue.next);
		DEL_FROM_LIST(PKT_Q_ENTRY(p));
		ADD_TO_XLIST(&q, PKT_Q_ENTRY(p));
		arp_outstanding--;
	}
	while (!XLIST_EMPTY(&q)) {
		PACKET *p = PKT_Q_RQ(q.next);
		DEL_FROM_LIST(PKT_Q_ENTRY(p));
		if (p->fn == &NET$FREE_PACKET) (*(unsigned long *)p->sender_data)--;
		((int (*)(PACKET *))p->tmp3)(p);
	}
}

static void ARP_DEL_PACKETS(ARP *a, int err)
{
	while (__unlikely(!LIST_EMPTY(&a->queue))) {
		PACKET *p = PKT_Q_RQ(a->queue.next);
		DEL_FROM_LIST(PKT_Q_ENTRY(p));
		arp_outstanding--;
		if (err != -ENETDOWN)
			ICMP_SEND(p, ICMP_TYPECODE(ICMP_UNREACH, ICMP_UNREACH_HOST), 0);
		/*{
			unsigned long off2_ = 0;
			const char *str2_ = KERNEL$DL_GET_SYMBOL_NAME((p)->fn, &off2_, 0);
			__debug_printf("arp del (%d): DATA SIZE %d, sender_data %p, AST %s:%lx\n", err, p->data_length, p->sender_data, str2_ ? str2_ : "?", off2_);;
		}*/
		p->status = err;
		CALL_PKT(p);
	}
}

void ARP_ZAP_PACKETS(void)
{
	ARP *a;
	LIST_FOR_EACH(a, &all_arps, ARP, list) ARP_DEL_PACKETS(a, -EAGAIN);
	/* Once more for ICMPs generated */
	LIST_FOR_EACH(a, &all_arps, ARP, list) ARP_DEL_PACKETS(a, -EAGAIN);
	NET$OOM();
}

static __finline__ void ARP_SEND_QUERY(ARP *a)
{
	IF_ARP_SEND_QUERY(a->id, a->host);
}

static void ARP_DEL(ARP *a, int err)
{
	if (__unlikely(a->flags & ARP_LOCK)) KERNEL$DEL_TIMER(&a->timer);
	DEL_FROM_LIST(&a->hash);
	DEL_FROM_LIST(&a->list);
	/* Warning, order is important --- ARP_DEL_PACKETS can send ICMPs and
	   generate new ARP entries --- unlink entry so that it won't be reused
	   while deleting it */
	ARP_DEL_PACKETS(a, err);
	__slfree(a);
}

static void ARP_RESEND(TIMER *t)
{
	ARP *a;
	LOWER_SPL(SPL_NET);
	a = GET_STRUCT(t, ARP, timer);
	if (__likely(a->flags & ARP_WAIT)) {
		a->flags &= ~(ARP_LOCK | ARP_WAIT);
		ARP_SEND_PACKETS(a);
		return;
	}
	if (__unlikely(++a->tries >= ARP_TRIES)) {
		a->flags &= ~ARP_LOCK;
		ARP_DEL(a, -ENETUNREACH);
		return;
	}
	ARP_SEND_QUERY(a);
	KERNEL$SET_TIMER(ARP_TIMEOUT, &a->timer);
}

static __u8 *ARP_FIND_SLOW(unsigned long id, in_addr_t host, PACKET *p);
static __u8 *ARP_ADD_TO_QUEUE(ARP *a, PACKET *p);
static __u8 *ARP_FIND_MCAST(in_addr_t host, PACKET *p);

__u8 *ARP_FIND(unsigned long id, in_addr_t host, PACKET *p, int (*call)(PACKET *p))
{
	ARP *a;
	unsigned h;
	p->tmp3 = (unsigned long)call;
	if (__unlikely(TCPIP_MULTICAST_ADDRESS(host))) return ARP_FIND_MCAST(host, p);
	h = ARP_HASH(host);
	XLIST_FOR_EACH(a, &arp_hash[h], ARP, hash) {
		if (__likely(a->host == host) && __likely(a->id == id)) {
			if (__likely(!(a->flags & (ARP_LOCK | ARP_INVALID)))) {
				return a->hwaddr;
			}
			if (__unlikely(a->flags & ARP_INVALID)) {
				ICMP_SEND(p, ICMP_TYPECODE(ICMP_UNREACH, ICMP_UNREACH_HOST), 0);
				p->status = -EINVAL;
				CALL_PKT(p);
				return NULL;
			}
			return ARP_ADD_TO_QUEUE(a, p);
		}
	}
	return ARP_FIND_SLOW(id, host, p);
}

static __u8 *ARP_FIND_SLOW(unsigned long id, in_addr_t host, PACKET *p)
{
	ARP *a;
	unsigned h;
	while (__unlikely(!(a = __slalloc(&arp_slab)))) {
		if (__unlikely(ARP_ZAP_ENTRY())) {
			p->status = -EAGAIN;
			CALL_PKT(p);
			return NULL;
		}
	}
	a->host = host;
	a->id = id;
	a->flags = ARP_LOCK;
	a->last_used = KERNEL$GET_JIFFIES_LO();
	h = ARP_HASH(host);
	ADD_TO_XLIST(&arp_hash[h], &a->hash);
	ADD_TO_LIST(&all_arps, &a->list);
	ARP_SEND_QUERY(a);
	a->tries = 0;
	KERNEL$SET_TIMER(ARP_TIMEOUT, &a->timer);
	return ARP_ADD_TO_QUEUE(a, p);
}

static __u8 *ARP_ADD_TO_QUEUE(ARP *a, PACKET *p)
{
	unsigned long call = p->tmp3;
	DUP_PACKET(p, &NET$PKTPOOL_LONG_TERM, {
		NET$DELAYED_OOM();
		p->status = -ENOMEM;
		CALL_PKT(p);
		return NULL;
	});
	p->tmp3 = call;
	ADD_TO_LIST(&a->queue, PKT_Q_ENTRY(p));
	arp_outstanding++;
	if (__unlikely(arp_outstanding > ARP_MAX_OUTSTANDING)) ARP_ZAP_PACKETS();
	return NULL;
}

static __u8 *ARP_FIND_MCAST(in_addr_t host, PACKET *p)
{
	static __u8 reply[MAX_ADDR_LEN + 1];
	reply[0] = p->addrlen;
	if (__unlikely(p->addrlen != MCAST_PHYS_ADDRLEN)) {
		memset(reply + 1, 0xff, MAX_ADDR_LEN);
		return reply;
	}
	MCAST_PHYS_ADDR(host, reply + 1);
	return reply;
}

static void ARP_REAPER(TIMER *t)
{
	ARP *a;
	u_jiffies_lo_t j;
	LOWER_SPL(SPL_NET);
	j = KERNEL$GET_JIFFIES_LO();
	LIST_FOR_EACH(a, &all_arps, ARP, list) if (__unlikely(j - a->last_used > TIMEOUT_JIFFIES(ARP_TIME_TO_LIVE))) {
		ARP *aa = LIST_STRUCT(a->list.prev, ARP, list);
		ARP_DEL(a, -EAGAIN);
		a = aa;
	}
	KERNEL$SET_TIMER(ARP_TIME_TO_LIVE, &arp_reaper);
}

void ARP_UPDATE(PACKET *p)
{
	ARP *a;
	in_addr_t host = IP_FIND_ROUTE(ip(p)->ip_src.s_addr);
	unsigned h = ARP_HASH(host);
	XLIST_FOR_EACH(a, &arp_hash[h], ARP, hash) {
		if (__likely(a->host == host)) {
			if (__likely(!(a->flags & (ARP_LOCK | ARP_INVALID)))) {
				if (__likely(a->hwaddr[0] == 6) && __likely(p->addrlen == 6) && __likely(!memcmp(PKT_SRCADDR(p, p->addrlen), a->hwaddr + 1, 6))) reval: a->last_used = KERNEL$GET_JIFFIES_LO();
				else if (__likely(a->hwaddr[0] == p->addrlen) && __likely(!memcmp(PKT_SRCADDR(p, p->addrlen), a->hwaddr + 1, p->addrlen))) goto reval;
			}
			return;
		}
	}
}

int ARP_INIT(void)
{
	int i, r;
	for (i = 0; i < ARP_HASH_SIZE; i++) INIT_XLIST(&arp_hash[i]);
	KERNEL$SLAB_INIT(&arp_slab, sizeof(ARP), 0, VM_TYPE_WIRED_MAPPED, ARP_CTOR, &NET$MEMORY_AVAIL, "TCPIP$ARP");
	if (__unlikely(r = KERNEL$SLAB_RESERVE(&arp_slab, 1))) {
		KERNEL$SLAB_DESTROY(&arp_slab);
		return 0;
	}
	arp_reaper.fn = ARP_REAPER;
	KERNEL$SET_TIMER(ARP_TIME_TO_LIVE, &arp_reaper);
	return 0;
}

void ARP_DONE(void)
{
	int i;
	KERNEL$DEL_TIMER(&arp_reaper);
	while (!LIST_EMPTY(&all_arps)) {
		ARP *a = LIST_STRUCT(all_arps.next, ARP, list);
		ARP_DEL(a, -ENETDOWN);
	}
	for (i = 0; i < ARP_HASH_SIZE; i++) if (!XLIST_EMPTY(&arp_hash[i]))
		KERNEL$SUICIDE("ARP_DONE: ARP LEAK");
	KERNEL$SLAB_DESTROY(&arp_slab);
}
