/* !!! FIXME: don't send packets during ifconfig */

#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/IOCTL.H>
#include <STDLIB.H>
#include <UNISTD.H>
#include <ARCH/IPCHECKS.H>
#include <SPAD/SYSLOG.H>
#include <SYS/PARAM.H>
#include <VALUES.H>
#include "MCAST.H"

#include "TCPIP.H"

#define SLEEP_WARNING	(10 * (JIFFIES_PER_SECOND >> KERNEL$JIFFIES_STEP_BITS))

/*
dctl net$inet ifconfig pkt$ne: 195.113.27.228 /netmask=255.255.252.0
dctl net$inet ifconfig pkt$ne: 195.113.27.228 /pointtopoint=195.113.27.229
dctl net$inet ifconfig pkt$ne: /dhcp
dctl net$inet ifconfig pkt$ne: 195.113.27.228 /dhcp
dctl net$inet ifconfig
dctl net$inet ifconfig pkt$ne:
dctl net$inet ifconfig pkt$ne: /down

dctl net$inet route /add 1.2.3.4 /netmask=255.255.255.0 195.113.25.222
dctl net$inet route /add default 195.113.25.222
dctl net$inet route /del 1.2.3.4 /netmask=255.255.255.0
dctl net$inet route /del default
*/

#define IFF_PTP			0x00000001
#define IFF_DHCP		0x00000002
#define IFF_DHCP_UNBOUND	0x00000004
#define IFF_DHCP_CONFLICT	0x00000008
#define IFF_DHCP_ONLY_BOOTP	0x00000010
#define IFF_IGMP2		0x00000020
#define IFF_GOING_UP		0x00000040
#define IFF_GOING_DOWN		0x00000080

#define IFF_DHCP_SECURE		0x10000000
#define IFF_DHCP_DISTRUST	0x20000000
#define IFF_DHCP_DISTRUST_LOCAL	0x40000000
#define IFF_DHCP_REQUEST_ADDR	0x80000000

typedef struct {
	LIST_ENTRY list;
	int handle;
	int flags;
	long outstanding;
	in_addr_t addr;
	in_addr_t netmask;
	in_addr_t pointtopoint;
	IF_TYPE type;
	IOCTLRQ linkchg_ioctl;
	char linkchg_ioctl_posted;
	LINK_STATE link_state;
	LINK_STATE link_state_prev;
} IF;

#define IF_DHCP(iff)	((DHCP *)((iff) + 1))
#define DHCP_IF(dhcp)	((IF *)(dhcp) - 1)

static unsigned n_interfaces = 0;

static DECL_LIST(interfaces);

typedef struct {
	in_addr_t addr;
	in_addr_t netmask;
	in_addr_t gw;
	IF *iff;
} ROUTE;

#define NO_GW		((in_addr_t)-1)

static ROUTE *routes = NULL;
static unsigned n_routes = 0;

int IFF_GATE = 0;
int IFF_PROXYARP = 0;

#define IS_LOOPBACK(str)	(!_memcasecmp(str, NET_INPUT_DEVICE ":/", strlen(NET_INPUT_DEVICE ":/")))

static void add_route(ROUTE *routes, unsigned *n_routes, in_addr_t dest_addr, in_addr_t netmask_addr, in_addr_t gw_addr, IF *iff);
static int allocate_routes(ROUTE **routes, unsigned n_routes);

int GET_IP(char *str, in_addr_t *addr, char **end)
{
	in_addr_t a;
	long num;
	char *p;
	if (__unlikely(!str)) return -1;
	if (__unlikely(!(p = strchr(str, '.')))) return -1;
	if (__unlikely(__get_number(str, p, 0, &num)) || __unlikely((unsigned long)num >= 256)) return -1;
	a = num;
	str = p + 1;
	if (__unlikely(!(p = strchr(str, '.')))) return -1;
	if (__unlikely(__get_number(str, p, 0, &num)) || __unlikely((unsigned long)num >= 256)) return -1;
	a = (a << 8) | num;
	str = p + 1;
	if (__unlikely(!(p = strchr(str, '.')))) return -1;
	if (__unlikely(__get_number(str, p, 0, &num)) || __unlikely((unsigned long)num >= 256)) return -1;
	a = (a << 8) | num;
	str = p + 1;
	p = str + strspn(str, "0123456789");
	if (__unlikely(__get_number(str, p, 0, &num)) || __unlikely((unsigned long)num >= 256)) return -1;
	a = (a << 8) | num;
	*addr = htonl(a);
	if (end) {
		*end = p;
	} else {
		if (__unlikely(*p)) return -1;
	}
	return 0;
}

int IS_NETMASK(in_addr_t a)
{
	a = ntohl(a);
	if (__unlikely((~a >> 1) & a)) return 0;
	return 1;
}

static in_addr_t default_netmask(in_addr_t a)
{
	in_addr_t netmask;
	if (IN_CLASSA(ntohl(a))) netmask = htonl(IN_CLASSA_NET);
	else if (IN_CLASSB(ntohl(a))) netmask = htonl(IN_CLASSB_NET);
	else if (IN_CLASSC(ntohl(a))) netmask = htonl(IN_CLASSC_NET);
	else if (IN_CLASSD(ntohl(a))) netmask = htonl(IN_CLASSD_NET);
	else netmask = htonl(IN_CLASSD_NET);
	return netmask;
}

static __finline__ int is_in_interface(IF *iff, in_addr_t a)
{
	if (__unlikely(iff->flags & IFF_DHCP_UNBOUND)) return 0;
	if (__unlikely(a == iff->addr)) return 2;
	if (__likely(!(iff->flags & IFF_PTP))) {
		return (a & iff->netmask) == (iff->addr & iff->netmask);
	}
	return a == iff->pointtopoint;
}

static int interfaces_conflict(IF *iff, int flags, in_addr_t addr, in_addr_t netmask, in_addr_t pointtopoint)
{
	if (__unlikely(iff->flags & IFF_DHCP_UNBOUND)) return 0;
	if (__likely(!(flags & IFF_PTP))) {
		if (__likely(!(iff->flags & IFF_PTP))) {
			in_addr_t cnm = iff->netmask & netmask;
			if (__unlikely((addr & cnm) == (iff->addr & cnm))) return 1;
		} else {
			if (__unlikely((iff->addr & netmask) == (addr & netmask))) return 1;
			if (__unlikely((iff->pointtopoint & netmask) == (addr & netmask))) return 1;
		}
	} else {
		if (__unlikely(is_in_interface(iff, addr))) return 1;
		if (__unlikely(is_in_interface(iff, pointtopoint))) return 1;
	}
	return 0;
}

char *PRINT_IP(char str[16], in_addr_t a)
{
	a = ntohl(a);
	_snprintf(str, 16, "%u.%u.%u.%u", (a >> 24) & 0xff, (a >> 16) & 0xff, (a >> 8) & 0xff, a & 0xff);
	return str;
}

static void print_if(IF *iff)
{
	unsigned i, j;
	char a1[__MAX_STR_LEN + 2];
	LINK_STATE_PHYS ls;
	union {
		IOCTLRQ io;
		OPENRQ op;
	} u;
	_printf("%s: ", KERNEL$HANDLE_PATH(iff->handle));
	if (__likely(!(iff->flags & IFF_DHCP_UNBOUND))) {
		_printf("ADDRESS %s", PRINT_IP(a1, iff->addr));
		if (__unlikely(iff->flags & IFF_PTP)) _printf(" POINT-TO-POINT %s", PRINT_IP(a1, iff->pointtopoint));
		else _printf(" NETMASK %s", PRINT_IP(a1, iff->netmask));
		if (iff->flags & IFF_DHCP) _printf(" %s", DHCP_MODE(IF_DHCP(iff)));
		if (iff->flags & IFF_IGMP2) _printf(" IGMP 2");
	} else {
		_printf("%s %s", DHCP_MODE(IF_DHCP(iff)), __likely(!(iff->flags & IFF_DHCP_CONFLICT)) ? DHCP_STATUS(IF_DHCP(iff)) : "RETURNED CONFLICTING ADDRESS");
	}
	_printf("\nTYPE %s", iff->type.type);
	if (iff->type.addrlen) _printf(" HWADDR %s", NET$PRINT_HWADDR(a1, iff->type.addr, iff->type.addrlen));
	_printf(" MTU %d", iff->type.mtu);
	u.io.h = iff->handle;
	u.io.v.ptr = 0;
	u.io.v.len = 0;
	u.io.v.vspace = &KERNEL$VIRTUAL;
	u.io.ioctl = IOCTL_IF_GET_MULTICAST;
	u.io.param = 0;
	SYNC_IO(&u.io, KERNEL$IOCTL);
	if (u.io.status == STATUS_MULTICAST_SOME) _printf(" MULTICAST");
	if (u.io.status == STATUS_MULTICAST_ALL) _printf(" MULTICAST ALL");
	if (u.io.status == STATUS_MULTICAST_PROMISC) _printf(" PROMISCUOUS");
	_printf("\n");
	u.io.h = iff->handle;
	u.io.v.ptr = (unsigned long)&ls;
	u.io.v.len = sizeof ls;
	u.io.v.vspace = &KERNEL$VIRTUAL;
	u.io.ioctl = IOCTL_IF_GET_LINK;
	u.io.param = 0;
	SYNC_IO(&u.io, KERNEL$IOCTL);
	if (__likely(u.io.status >= 0)) {
		_printf("%s\n", NET$PRINT_LINK_STATE(a1, &ls));
	}
	_snprintf(a1, __MAX_STR_LEN, "AUTH$EAP@%s", KERNEL$HANDLE_PATH(iff->handle));
	for (i = 0, j = 0; a1[i]; i++) {
		if (a1[i] != ':' &&
		    a1[i] != '/' &&
		    a1[i] != '=' &&
		    a1[i] != '.')
			a1[j++] = a1[i];
	}
	a1[j++] = ':';
	a1[j++] = '/';
	a1[j++] = 0;
	u.op.flags = O_RDONLY;
	u.op.path = a1;
	u.op.cwd = NULL;
	SYNC_IO(&u.op, KERNEL$OPEN);
	if (__unlikely(u.op.status >= 0)) {
		int eap_h = u.op.status;
		u.io.h = eap_h;
		u.io.v.ptr = (unsigned long)&a1;
		u.io.v.len = __MAX_STR_LEN;
		u.io.v.vspace = &KERNEL$VIRTUAL;
		u.io.ioctl = IOCTL_EAP_GET_STATUS;
		u.io.param = 0;
		SYNC_IO(&u.io, KERNEL$IOCTL);
		if (__likely(u.io.status >= 0)) {
			_printf("EAP %s\n", a1);
		}
		close(eap_h);
	}
}

static void if_monitor_link(IF *iff, int get);

static DECL_AST(if_linkchg_ast, SPL_NET, IOCTLRQ)
{
	IF *iff = GET_STRUCT(RQ, IF, linkchg_ioctl);
	iff->linkchg_ioctl_posted = 0;
	if (__unlikely(iff->flags & IFF_GOING_DOWN)) RETURN;
	if (__unlikely(iff->linkchg_ioctl.status < 0)) {
		iff->linkchg_ioctl_posted = 1;
		iff->linkchg_ioctl.status = RQS_PROCESSING;
		WQ_WAIT(&KERNEL$LOGICAL_WAIT, &iff->linkchg_ioctl, KERNEL$IOCTL);
		RETURN;
	}
	if (iff->linkchg_ioctl.ioctl == IOCTL_IF_WAIT_LINK) {
		if_monitor_link(iff, 1);
	} else {
		if (__likely(!(iff->flags & IFF_GOING_UP))) if (iff->link_state.flags & LINK_STATE_UP || (iff->link_state.flags & LINK_STATE_UNKNOWN && (!(iff->link_state_prev.flags & (LINK_STATE_UP | LINK_STATE_UNKNOWN)) || iff->link_state_prev.seq != iff->link_state.seq))) {
			if (iff->flags & IFF_DHCP) DHCP_NOTIFY_LINK_CHANGE(IF_DHCP(iff));
			IGMP_NOTIFY_LINK_CHANGE(iff->handle);
		}
		memcpy(&iff->link_state_prev, &iff->link_state, sizeof iff->link_state);
		if_monitor_link(iff, 0);
		/*{
			char buf[__MAX_STR_LEN];
			__debug_printf("link state: %s, seq %Ld\n", NET$PRINT_LINK_STATE(buf, (LINK_STATE_PHYS *)&iff->link_state), iff->link_state.seq);
		}*/
	}
	RETURN;
}

static void if_monitor_link(IF *iff, int get)
{
	if (__unlikely(iff->linkchg_ioctl_posted)) return;
	iff->linkchg_ioctl.fn = if_linkchg_ast;
	iff->linkchg_ioctl.h = iff->handle;
	iff->linkchg_ioctl.ioctl = get ? IOCTL_IF_GET_LINK : IOCTL_IF_WAIT_LINK;
	iff->linkchg_ioctl.param = 0;
	iff->linkchg_ioctl.v.ptr = (unsigned long)&iff->link_state;
	iff->linkchg_ioctl.v.len = sizeof iff->link_state;
	iff->linkchg_ioctl.v.vspace = &KERNEL$VIRTUAL;
	iff->linkchg_ioctl_posted = 1;
	CALL_IORQ(&iff->linkchg_ioctl, KERNEL$IOCTL);
}

static IF *find_if_by_name(char *name)
{
	char *nn;
	IF *iff;
	OPENRQ rq;
	rq.flags = _O_NOACCESS | _O_NOOPEN;
	rq.path = name;
	rq.cwd = KERNEL$CWD();
	SYNC_IO(&rq, KERNEL$OPEN);
	if (__unlikely(rq.status < 0) || __unlikely(!(nn = KERNEL$HANDLE_PATH(rq.status)))) nn = name;
	LIST_FOR_EACH(iff, &interfaces, IF, list) {
		char *c = KERNEL$HANDLE_PATH(iff->handle);
		if (__likely(c != NULL) && __likely(!_strcasecmp(c, nn))) {
			if (__likely(rq.status >= 0)) KERNEL$FAST_CLOSE(rq.status);
			return iff;
		}
	}
	if (__likely(rq.status >= 0)) KERNEL$FAST_CLOSE(rq.status);
	return NULL;
}

static IF *find_if_by_id(unsigned long id)
{
	IF *iff;
	LIST_FOR_EACH(iff, &interfaces, IF, list)
		if (iff->type.id == id)
			return iff;
	return NULL;
}

static void delete_if(IF *iff, IF *nif);

static int add_if(char *name, int flags, in_addr_t addr, in_addr_t netmask, in_addr_t pointtopoint, int mtu, IF *del_if)
{
	int r;
	IF *iff, *liff;
	int h, i;
	char *hp;
	union {
		OPENRQ op;
		IOCTLRQ io;
		MALLOC_REQUEST mrq;
	} u;
#if 0
	if (__likely(!(flags & IFF_DHCP))) LIST_FOR_EACH(iff, &interfaces, IF, list) {
		if (__unlikely(iff == del_if)) continue;
		if (__unlikely(interfaces_conflict(iff, flags, addr, netmask, pointtopoint))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ADDRESS FOR INTERFACE %s CONFLICTS WITH INTERFACE %s", net_name, name, KERNEL$HANDLE_PATH(iff->handle));
			r = -EBUSY;
			goto err0;
		}
	}
#endif
	u.op.flags = O_RDWR;
	if (__likely(strlen(name) <= 512)) {
		strcpy((char *)(u.op.path = alloca(strlen(name) + 1)), name);
		__upcase((char *)u.op.path);
	} else {
		/*u.op.path = name;*/
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: TOO LONG INTERFACE NAME", net_name);
		r = -ENAMETOOLONG;
		goto err0;
	}
	u.op.cwd = NULL;
	SYNC_IO_CANCELABLE(&u.op, KERNEL$OPEN);
	if (__unlikely(u.op.status < 0)) {
		if (u.op.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T OPEN INTERFACE %s: %s", net_name, name, strerror(-u.op.status));
		r = u.op.status;
		goto err0;
	}
	h = u.op.status;
	hp = KERNEL$HANDLE_PATH(h);
	if (!IS_LOOPBACK(hp) && __unlikely(r = KERNEL$REGISTER_DEPENDENCY(tcpip_lnte, hp))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER DEPENDENCY FOR INTERFACE %s: %s", net_name, name, strerror(-r));
		goto err1;
	}
	u.mrq.size = sizeof(IF);
	if (__unlikely(flags & IFF_DHCP)) u.mrq.size += DHCP_STRUCT_SIZE;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC INTERFACE STRUCTURE: %s", net_name, strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err2;
	}
	iff = u.mrq.ptr;
	memset(iff, 0, sizeof(IF));
	iff->handle = h;
	iff->flags = (flags & (IFF_PTP | IFF_DHCP | IFF_DHCP_ONLY_BOOTP | IFF_IGMP2)) | IFF_GOING_UP;
	if (__unlikely(flags & IFF_DHCP)) iff->flags |= IFF_DHCP_UNBOUND;
	iff->addr = addr;
	iff->netmask = netmask;
	iff->pointtopoint = pointtopoint;
	u.io.h = h;
	u.io.ioctl = IOCTL_IF_GET_TYPE;
	u.io.param = 0;
	u.io.v.ptr = (unsigned long)&iff->type;
	u.io.v.len = sizeof(iff->type);
	u.io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&u.io, KERNEL$IOCTL);
	if (__unlikely(u.io.status < 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET TYPE OF INTERFACE %s: %s", net_name, name, strerror(-u.io.status));
		r = u.io.status;
		goto err3;
	}
	if (__unlikely(iff->type.addrlen > sizeof(iff->type.addr))) {
		invalid_reply:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INTERFACE %s HAS INVALID IF_GETTYPE REPLY", net_name, name);
		r = -EINVAL;
		goto err3;
	}
	for (i = iff->type.addrlen; i < sizeof(iff->type.addr); i++)
		if (__unlikely(iff->type.addr[i])) goto invalid_reply;
	if (__unlikely(iff->type.addrlen > MAX_ADDR_LEN)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INTERFACE %s REPORTS BOGUS ADDRESS LENGTH %d", net_name, name, iff->type.addrlen);
		r = -EINVAL;
		goto err3;
	}
	if ((liff = find_if_by_id(iff->type.id)) && liff != del_if) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INTERFACE %s ALREADY CONFIGURED WITH NAME %s", net_name, name, KERNEL$HANDLE_PATH(liff->handle));
		r = -EEXIST;
		goto err3;
	}
	if (__unlikely(mtu)) iff->type.mtu = mtu;
	if (__unlikely(iff->type.mtu < IP_MIN_MTU)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INTERFACE %s REPORTS BOGUS MTU %d", net_name, name, iff->type.mtu);
		r = -EINVAL;
		goto err3;
	}
	if (__unlikely(iff->type.mtu > NET$MAX_PACKET_LENGTH)) iff->type.mtu = NET$MAX_PACKET_LENGTH;
	if (__unlikely(r = allocate_routes(&routes, n_routes + DHCP_RESERVED_ROUTES + 1 + (n_interfaces + 1) * 2))) {
		goto err3;
	}
	RAISE_SPL(SPL_NET);
	n_interfaces++;
	ADD_TO_LIST_END(&interfaces, &iff->list);
	/* routes must be atomically in sync with interfaces --- otherwise
	   packets destined for internal networks may leak into internet */
	if (!(flags & IFF_DHCP)) {
		if (__likely(!(flags & IFF_PTP))) {
			add_route(routes, &n_routes, addr & netmask, netmask, NO_GW, iff);
		} else {
			add_route(routes, &n_routes, addr, -1, NO_GW, iff);
			add_route(routes, &n_routes, pointtopoint, -1, NO_GW, iff);
		}
	}
	if_monitor_link(iff, 1);
	if (__unlikely(del_if != NULL)) {
		delete_if(del_if, iff);
	}
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(flags & IFF_DHCP)) {
		DHCP_INIT(IF_DHCP(iff), &iff->type, iff->handle, flags & IFF_DHCP_REQUEST_ADDR, addr, flags & IFF_DHCP_ONLY_BOOTP, flags & IFF_DHCP_SECURE, flags & IFF_DHCP_DISTRUST ? 2 : flags & IFF_DHCP_DISTRUST_LOCAL ? 1 : 0);
	}
	RAISE_SPL(SPL_NET);
	iff->flags &= ~IFF_GOING_UP;
	LOWER_SPL(SPL_ZERO);
	return 0;

	err3:
	free(iff);
	err2:
	hp = KERNEL$HANDLE_PATH(h);
	if (!IS_LOOPBACK(hp))
		KERNEL$UNREGISTER_DEPENDENCY(tcpip_lnte, hp);
	err1:
	close(h);
	err0:
	return r;
}

static void IF_DHCP_UNSET_NOCHECK(DHCP *dhcp);

int IF_DHCP_SETUP(DHCP *dhcp, in_addr_t addr, in_addr_t netmask, int valid_netmask, in_addr_t broadcast, int valid_broadcast)
{
	static char a1[16], a2[16];
	IF *test_iff;
	IF *iff = DHCP_IF(dhcp);
	if (__unlikely(!(iff->flags & IFF_DHCP_UNBOUND))) IF_DHCP_UNSET_NOCHECK(IF_DHCP(iff));
	if (__unlikely(!valid_netmask)) netmask = default_netmask(addr);
	LIST_FOR_EACH(test_iff, &interfaces, IF, list) {
		if (__unlikely(interfaces_conflict(test_iff, 0, addr, netmask, 0))) {
			if (test_iff->flags & IFF_DHCP && !(test_iff->link_state.flags & LINK_STATE_UP) && (iff->link_state.flags & LINK_STATE_UP || (iff->link_state.flags & LINK_STATE_UNKNOWN && !(test_iff->link_state.flags & LINK_STATE_UNKNOWN)))) {
				IF_DHCP_UNSET_NOCHECK(IF_DHCP(test_iff));
				DHCP_NOTIFY_LINK_CHANGE(IF_DHCP(test_iff));
				continue;
			}
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): %s REPORTS ADDRESS %s AND MASK %s WHICH CONFLICTS WITH INTERFACE %s", KERNEL$HANDLE_PATH(iff->handle), DHCP_MODE(IF_DHCP(iff)), PRINT_IP(a1, addr), PRINT_IP(a2, netmask), KERNEL$HANDLE_PATH(test_iff->handle));
			iff->flags |= IFF_DHCP_CONFLICT;
			TCPIP_SOCKETS_CHECK();
			return 1;
		}
	}
	iff->addr = addr;
	iff->netmask = netmask;
	iff->flags &= ~(IFF_DHCP_UNBOUND | IFF_DHCP_CONFLICT);
	add_route(routes, &n_routes, addr & netmask, netmask, NO_GW, iff);
	IGMP_CHECK_MULTICASTS();
	TCPIP_SOCKETS_CHECK();
	return 0;
}

void IF_DHCP_SETUP_GW(DHCP *dhcp, in_addr_t gw)
{
	static char a1[16];
	IF *test_iff;
	IF *iff = DHCP_IF(dhcp);
	LIST_FOR_EACH(test_iff, &interfaces, IF, list) {
		if (__likely(is_in_interface(test_iff, gw) == 1)) {
			add_route(routes, &n_routes, 0, 0, gw, test_iff);
			IGMP_CHECK_MULTICASTS();
			return;
		}
	}
	KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): %s RETURNES UNREACHABLE GATEWAY %s. IGNORED", KERNEL$HANDLE_PATH(iff->handle), DHCP_MODE(IF_DHCP(iff)), PRINT_IP(a1, gw));
}

void IF_DHCP_SETUP_ROUTE(DHCP *dhcp, in_addr_t dst, in_addr_t mask, in_addr_t gw)
{
	static char a1[16];
	IF *test_iff;
	IF *iff = DHCP_IF(dhcp);
	if (__unlikely(__alloc_size(routes) < (n_routes + 2) * sizeof(ROUTE))) {
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): TOO MANY ROUTES REQUESTED. SOME IGNORED", KERNEL$HANDLE_PATH(iff->handle));
		return;
	}
	LIST_FOR_EACH(test_iff, &interfaces, IF, list) {
		if (__likely(is_in_interface(test_iff, gw) == 1)) {
			add_route(routes, &n_routes, dst, mask, gw, test_iff);
			IGMP_CHECK_MULTICASTS();
			return;
		}
	}
	KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): %s RETURNES UNREACHABLE ROUTER %s. IGNORED", KERNEL$HANDLE_PATH(iff->handle), DHCP_MODE(IF_DHCP(iff)), PRINT_IP(a1, gw));
}

static void unroute_if(IF *iff, IF *nif);

void IF_DHCP_UNSET(DHCP *dhcp)
{
	IF_DHCP_UNSET_NOCHECK(dhcp);
	TCPIP_SOCKETS_CHECK();
}

static void IF_DHCP_UNSET_NOCHECK(DHCP *dhcp)
{
	IF *iff = DHCP_IF(dhcp);
	unroute_if(iff, NULL);
	iff->addr = 0;
	iff->netmask = 0;
	iff->flags &= ~IFF_DHCP_CONFLICT;
	iff->flags |= IFF_DHCP_UNBOUND;
	IGMP_CANCEL_IF_ASYNC(iff->handle);
}

static void unroute_if(IF *iff, IF *nif)
{
	unsigned i, j;
	for (i = 0, j = 0; i < n_routes; i++) {
		if (__unlikely(routes[i].iff == iff)) {
			if (nif && routes[i].gw != NO_GW) {
				if (__likely(is_in_interface(nif, routes[i].gw) == 1)) {
					routes[i].iff = nif;
					goto ok;
				}
			}
			continue;
		}
		ok:
		if (i != j) memcpy(&routes[j], &routes[i], sizeof(ROUTE));
		j++;
	}
	n_routes -= i - j;
}

/* Lowers SPL to ZERO. May be called at any spl up to SPL_NET */

static void delete_if(IF *iff, IF *nif)
{
	unsigned cnt;
	char *hp;
	/* must delete the interface before blocking */
	RAISE_SPL(SPL_NET);
	if (__unlikely(iff->flags & IFF_DHCP)) DHCP_DESTROY(IF_DHCP(iff));
	if (__unlikely(!n_interfaces))
		KERNEL$SUICIDE("delete_if: n_interfaces UNDERFLOW");
	n_interfaces--;
	DEL_FROM_LIST(&iff->list);
	unroute_if(iff, nif);
	iff->flags |= IFF_GOING_DOWN;
	TCPIP_SOCKETS_CHECK();	/* remove sockets before block */
	IGMP_CANCEL_IF(iff->handle);	/* does block */
	LOWER_SPL(SPL_ZERO);
	cnt = 0;
	while (iff->linkchg_ioctl_posted) {
		KERNEL$CIO((IORQ *)(void *)&iff->linkchg_ioctl);
		KERNEL$SLEEP(1);
		if (__unlikely(++cnt == SLEEP_WARNING)) {
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, net_name, "WAITING FOR LINKCHG CANCEL ON IF %s", KERNEL$HANDLE_PATH(iff->handle));
		}
	}
	cnt = 0;
	while (__unlikely(iff->outstanding)) {
		KERNEL$SLEEP(1);
		if (__unlikely(++cnt == SLEEP_WARNING)) {
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, net_name, "WAITING FOR %ld OUTSTANDING PACKETS ON IF %s", iff->outstanding, KERNEL$HANDLE_PATH(iff->handle));
		}
	}
	cnt = 0;
	while (__unlikely(TCPIP_SOCKETS_OUSTANDING_PACKETS(iff->handle))) {
		KERNEL$SLEEP(1);
		if (__unlikely(++cnt == SLEEP_WARNING))
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, net_name, "WAITING FOR OUTSTANDING TCP PACKETS ON IF %s", KERNEL$HANDLE_PATH(iff->handle));
	}
	hp = KERNEL$HANDLE_PATH(iff->handle);
	if (!IS_LOOPBACK(hp) && tcpip_lnte)
		KERNEL$UNREGISTER_DEPENDENCY(tcpip_lnte, hp);
	close(iff->handle);
	free(iff);
}

static void print_route(ROUTE *r, int implicit)
{
	static char a1[16], a2[16], a3[16];
	if (r->gw == NO_GW) {
		if (r->netmask != (in_addr_t)-1) _printf("NET %s MASK %s IF %s\n", PRINT_IP(a1, r->addr), PRINT_IP(a2, r->netmask), KERNEL$HANDLE_PATH(r->iff->handle));
		else _printf("HOST %s IF %s\n", PRINT_IP(a1, r->addr), KERNEL$HANDLE_PATH(r->iff->handle));
	} else {
		if (!r->netmask) _printf("DEFAULT GW %s IF %s\n", PRINT_IP(a3, r->gw), KERNEL$HANDLE_PATH(r->iff->handle));
		else if (r->netmask != (in_addr_t)-1) _printf("NET %s MASK %s GW %s IF %s\n", PRINT_IP(a1, r->addr), PRINT_IP(a2, r->netmask), PRINT_IP(a3, r->gw), KERNEL$HANDLE_PATH(r->iff->handle));
		else _printf("HOST %s GW %s IF %s\n", PRINT_IP(a1, r->addr), PRINT_IP(a3, r->gw), KERNEL$HANDLE_PATH(r->iff->handle));
	}
}

static int allocate_routes(ROUTE **routes, unsigned n_routes)
{
	ROUTE *new_routes, *old_routes;
	size_t sz, rsz;
	MALLOC_REQUEST mrq;
	if (__likely((sz = __alloc_size(*routes)) >= (rsz = n_routes * sizeof(ROUTE)))) {
		if (__unlikely(sz < (rsz >> 3))) {
			sz = rsz;
			goto ok_shrink;
		}
		return 0;
	}
	ok_shrink:
	mrq.size = rsz;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC ROUTE STRUCTURE: %s", net_name, strerror(-mrq.status));
		return mrq.status;
	}
	new_routes = mrq.ptr;
	old_routes = *routes;
	RAISE_SPL(SPL_NET);
	memcpy(new_routes, old_routes, MIN(sz, rsz));
	*routes = new_routes;
	LOWER_SPL(SPL_ZERO);
	free(old_routes);
	return 0;
}

static void add_route(ROUTE *routes, unsigned *n_routes, in_addr_t dest_addr, in_addr_t netmask_addr, in_addr_t gw_addr, IF *iff)
{
	unsigned i;
	unsigned nr;
	int spl;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_NET);
	nr = *n_routes;
	for (i = 0; i < nr; i++) {
		if (__unlikely(ntohl(routes[i].netmask) > ntohl(netmask_addr))) continue;
		if (__likely(routes[i].netmask == netmask_addr)) {
			if (__unlikely(routes[i].addr == dest_addr)) {
				routes[i].gw = gw_addr;
				routes[i].iff = iff;
				return;
			}
			continue;
		}
		break;
	}
	if (__unlikely(__alloc_size(routes) < (nr + 1) * sizeof(ROUTE)))
		KERNEL$SUICIDE("add_route: ROUTE TABLE NOT EXTENDED, SIZE %ld, NEEDED %ld", (unsigned long)__alloc_size(routes), (unsigned long)((nr + 1) * sizeof(ROUTE)));
	memmove(&routes[i + 1], &routes[i], (nr - i) * sizeof(ROUTE));
	routes[i].addr = dest_addr;
	routes[i].netmask = netmask_addr;
	routes[i].gw = gw_addr;
	routes[i].iff = iff;
	*n_routes = nr + 1;
	LOWER_SPLX(spl);
}

in_addr_t IP_DEFAULT_MULTICAST_IF(void)
{
	unsigned i;
	for (i = 0; i < n_routes; i++) {
		if (__likely(routes[i].netmask == 0)) {
			return routes[i].iff->addr;
		}
	}
	return htonl(INADDR_ANY);
}

int IP_IF_HANDLE(in_addr_t addr)
{
	IF *iff = TCPIP_BINDABLE_ADDRESS(addr);
	if (__unlikely(!iff)) return -1;
	return iff->handle;
}

IF_TYPE *IP_IF_TYPE(in_addr_t addr)
{
	IF *iff = TCPIP_BINDABLE_ADDRESS(addr);
	if (__unlikely(!iff)) return NULL;
	return &iff->type;
}

int IP_IF_IGMP_VERSION(in_addr_t addr)
{
	IF *iff = TCPIP_BINDABLE_ADDRESS(addr);
	if (__unlikely(!iff)) return -1;
	return iff->flags & IFF_IGMP2 ? 2 : 1;
}

void *TCPIP_BINDABLE_ADDRESS(in_addr_t addr)
{
	IF *iff;
	LIST_FOR_EACH(iff, &interfaces, IF, list) {
		if (iff->addr == addr && __likely(!(iff->flags & IFF_DHCP_UNBOUND))) return iff;
	}
	return NULL;
}

static IF *IP_ROUTE(in_addr_t addr, in_addr_t *to)
{
	unsigned i;
	for (i = 0; i < n_routes; i++) {
		if (__unlikely((addr & routes[i].netmask) == routes[i].addr)) {
			in_addr_t gw = routes[i].gw;
			if (gw == NO_GW) gw = addr;
			*to = gw;
			return routes[i].iff;
		}
	}
	return NULL;
}

in_addr_t IP_FIND_ROUTE(in_addr_t to)
{
	in_addr_t gw;
	if (__likely(IP_ROUTE(to, &gw) != NULL)) return gw;
	return htonl(INADDR_ANY);
}

in_addr_t IP_FIND_LOCAL_ADDRESS(in_addr_t addr)
{
	IF *iff = IP_ROUTE(addr, (in_addr_t *)(void *)&KERNEL$LIST_END);
	if (__unlikely(!iff)) return htonl(INADDR_ANY);
	return iff->addr;
}

int IP_FIND_MTU(in_addr_t addr)
{
	IF *iff = IP_ROUTE(addr, (in_addr_t *)(void *)&KERNEL$LIST_END);
	if (__unlikely(!iff)) return IP_MSS;
	return iff->type.mtu;
}

int IP_IS_BROADCAST(in_addr_t addr)
{
	IF *iff = IP_ROUTE(addr, (in_addr_t *)(void *)&KERNEL$LIST_END);
	if (__unlikely(!iff)) return -1;
	return addr == (iff->addr | ~iff->netmask);
}

static __u16 ip_id_pool[IP_ID_POOL_SIZE];

int IP_SEND_PACKET(PACKET *p)
{
	unsigned hash, len;
	__u8 *dest_addr;
	in_addr_t gw;
	IF *iff;
	len = p->data_length;
	ip(p)->ip_len = htons(len);
	if (len <= IP_MIN_MTU) {
		*(__u32 *)(void *)&ip(p)->ip_id = 0;
	} else {
		hash = ntohl(ip(p)->ip_dst.s_addr) & (IP_ID_POOL_SIZE - 1);
		*(__u32 *)(void *)&ip(p)->ip_id = ++ip_id_pool[hash]
#ifdef __BIG_ENDIAN
			<< 16
#endif
		;
	}
	IP_HEADER_MAKE_CHECKSUM((__u8 *)ip(p));
	p->flags |= PKT_IPHDR_CHECKSUM_OK;
	if (__unlikely(TCPIP_MULTICAST_ADDRESS(ip(p)->ip_dst.s_addr))) {
		gw = ip(p)->ip_dst.s_addr;
		if (__unlikely(!(iff = TCPIP_BINDABLE_ADDRESS(ip(p)->ip_src.s_addr)))) goto unreach_noif;
	} else if (__unlikely(!(iff = IP_ROUTE(ip(p)->ip_dst.s_addr, &gw)))) {
		unreach_noif:
		if (p->fn == &NET$FREE_PACKET) p->sender_data = &KERNEL$LIST_END;
		unreach:
		p->status = -EHOSTUNREACH;
		CALL_PKT(p);
		return -EHOSTUNREACH;
	}
	if (p->fn == &NET$FREE_PACKET) (*(long *)(p->sender_data = &iff->outstanding))++;
	PKT_PROTOCOL(p) = NET_PROTOCOL_IP;
	if (__unlikely(ip(p)->ip_dst.s_addr == iff->addr)) {
		p->h = NET$LOCAL_HANDLE;
		p->addrlen = 0;
		goto skip_addr;
	}
	p->h = iff->handle;
	p->addrlen = iff->type.addrlen;
	if (__unlikely(iff->flags & IFF_PTP)) {
		if (__unlikely(iff->type.addrlen)) {
			SET_BROADCAST_DST(p);
			goto sa;
		}
	} else if (__likely(p->addrlen)) {
		if (__unlikely(!(dest_addr = ARP_FIND(iff->type.id, gw, p, IP_SEND_PACKET)))) return 0;
		dest_addr++;
		if (__likely(iff->type.addrlen == 6)) {	/* fast path for ethernet */
			if (__unlikely(dest_addr[-1] != 6)) goto unreach;
			memcpy(PKT_DSTADDR(p, 6), dest_addr, 6);
			memcpy(PKT_SRCADDR(p, 6), iff->type.addr, 6);
		} else {
			if (__unlikely(dest_addr[-1] != iff->type.addrlen)) goto unreach;
			memcpy(PKT_DSTADDR(p, iff->type.addrlen), dest_addr, iff->type.addrlen);
			sa:
			memcpy(PKT_SRCADDR(p, iff->type.addrlen), iff->type.addr, iff->type.addrlen);
		}
	}
	skip_addr:
	if (__unlikely(p->data_length > iff->type.mtu)) {
		IP_FRAGMENT(p, iff->type.mtu, &iff->outstanding);
		return 0;
	}
	iplog(p);
	p->id = IDTYPE_LO;
	CALL_IORQ(p, KERNEL$PKTIO);
	return 0;
}

static int IP_INPUT_LOCAL(PACKET *p)
{
	IF *iff;
	p->flags |= PKT_IPHDR_CHECKSUM_OK;
	if (__unlikely(!(iff = IP_ROUTE(ip(p)->ip_dst.s_addr, (in_addr_t *)(void *)&KERNEL$LIST_END)))) {
		unreach:
		p->status = -EHOSTUNREACH;
		CALL_PKT(p);
		return -EHOSTUNREACH;
	}
	if (__unlikely(ip(p)->ip_dst.s_addr != iff->addr)) {
		goto unreach;
	}
	p->h = NET$LOCAL_HANDLE;
	p->addrlen = 0;
	p->id = IDTYPE_LO;
	CALL_IORQ(p, KERNEL$PKTIO);
	return 0;
}

static void INVALID_ICMP_ROUTE(PACKET *p)
{
	static char a1[16];
	static char a2[16];
	struct icmp *icmp = (struct icmp *)((__u8 *)ip(p) + IP_HLEN(ip(p)->ip_vhl));
	if (errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "ATTEMPTING TO ROUTE ICMP PACKET SUITABLE ONLY FOR LAN, SRC %s, DST %s, TYPE %02X, CODE %02X", PRINT_IP(a1, ip(p)->ip_src.s_addr), PRINT_IP(a2, ip(p)->ip_dst.s_addr), icmp->icmp_type, icmp->icmp_code);
}

int IP_ROUTE_PACKET(PACKET *p)
{
	__u8 ttl;
	__u8 *dest_addr;
	in_addr_t gw;
	IF *iff;
	if (__unlikely(ip(p)->ip_p == IPPROTO_ICMP) && __likely(p->data_length >= IP_HLEN(ip(p)->ip_vhl) + ICMP_LEN)) {
		struct icmp *icmp = (struct icmp *)((__u8 *)ip(p) + IP_HLEN(ip(p)->ip_vhl));
		if (__unlikely(icmp->icmp_type == ICMP_REDIRECT) ||
		    __unlikely(icmp->icmp_type == ICMP_ROUTERADVERT) ||
		    __unlikely(icmp->icmp_type == ICMP_ROUTERSOLICIT) ||
		    __unlikely(icmp->icmp_type == ICMP_IREQ) ||
		    __unlikely(icmp->icmp_type == ICMP_IREQREPLY) ||
		    __unlikely(icmp->icmp_type == ICMP_MASKREQ) ||
		    __unlikely(icmp->icmp_type == ICMP_MASKREPLY)) {
			INVALID_ICMP_ROUTE(p);
			if (p->fn == &NET$FREE_PACKET) p->sender_data = &KERNEL$LIST_END;
			p->status = -EINVAL;
			CALL_PKT(p);
			return -EINVAL;
		}
	}
	iff = IP_ROUTE(ip(p)->ip_dst.s_addr, &gw);
	if (__unlikely(!iff)) {
		if (p->fn == &NET$FREE_PACKET) p->sender_data = &KERNEL$LIST_END;
		unreach:
		ICMP_SEND(p, ICMP_TYPECODE(ICMP_UNREACH, ICMP_UNREACH_NET), 0);
		p->status = -EHOSTUNREACH;
		CALL_PKT(p);
		return -EHOSTUNREACH;
	}
	if (p->fn == &NET$FREE_PACKET) (*(long *)(p->sender_data = &iff->outstanding))++;
	p->addrlen = iff->type.addrlen;
	PKT_PROTOCOL(p) = NET_PROTOCOL_IP;
	if (__unlikely(iff->flags & IFF_PTP)) {
		if (__unlikely(iff->type.addrlen)) {
			SET_BROADCAST_DST(p);
			goto sa;
		}
	} else if (__likely(p->addrlen)) {
		if (__unlikely(!(dest_addr = ARP_FIND(iff->type.id, gw, p, IP_ROUTE_PACKET)))) return 0;
		dest_addr++;
		if (__likely(iff->type.addrlen == 6)) {	/* fast path for ethernet */
			if (__unlikely(dest_addr[-1] != 6)) goto unreach;
			memcpy(PKT_DSTADDR(p, 6), dest_addr, 6);
			memcpy(PKT_SRCADDR(p, 6), iff->type.addr, 6);
		} else {
			if (__unlikely(dest_addr[-1] != iff->type.addrlen)) goto unreach;
			memcpy(PKT_DSTADDR(p, iff->type.addrlen), dest_addr, iff->type.addrlen);
			sa:
			memcpy(PKT_SRCADDR(p, iff->type.addrlen), iff->type.addr, iff->type.addrlen);
		}
	}
	ttl = ip(p)->ip_ttl;
	if (__unlikely(ttl <= 1)) {
		ICMP_SEND(p, ICMP_TYPECODE(ICMP_TIMXCEED, ICMP_TIMXCEED_INTRANS), 0);
		p->status = -ETIMEDOUT;
		CALL_PKT(p);
		return -ETIMEDOUT;
	}
	ip(p)->ip_ttl = ttl - 1;
	IP_HEADER_MAKE_CHECKSUM((__u8 *)ip(p));
	p->flags |= PKT_IPHDR_CHECKSUM_OK;
	if (__unlikely(ip(p)->ip_dst.s_addr == iff->addr)) {
		p->h = NET$LOCAL_HANDLE;
	} else {
		p->h = iff->handle;
	}
	if (__unlikely(p->data_length > iff->type.mtu)) {
		IP_FRAGMENT(p, iff->type.mtu, &iff->outstanding);
		return 0;
	}
	p->id = IDTYPE_LO;
	CALL_IORQ(p, KERNEL$PKTIO);
	return 0;
}

static void IF_SEND_ARP(IF *iff, PACKET *op, in_addr_t dst_host, in_addr_t src_host)
{
	PACKET *p;
	__u8 *data;
	ALLOC_PACKET(p, sizeof(struct arphdr) + 2 * sizeof(in_addr_t) + 2 * MAX_ADDR_LEN, &NET$PKTPOOL, SPL_NET, {
		ARP_ZAP_PACKETS();
		return;
	});
	arp(p)->hwtype = iff->type.arptype;
	arp(p)->protocol = NET_PROTOCOL_IP;
	arp(p)->hlen = iff->type.addrlen;
	arp(p)->plen = sizeof(in_addr_t);
	arp(p)->operation = op ? ARP_OP_ARP_RESPONSE : ARP_OP_ARP_REQUEST;
	data = arpaddr(p);
	data = mempcpy(data, iff->type.addr, iff->type.addrlen);
	data = mempcpy(data, &src_host, sizeof(in_addr_t));
	if (op) memcpy(data, PKT_SRCADDR(op, iff->type.addrlen), iff->type.addrlen);
	else memset(data, 0x00, iff->type.addrlen);
	data += iff->type.addrlen;
	data = mempcpy(data, &dst_host, sizeof(in_addr_t));
	p->data_length = data - (__u8 *)arp(p);
	p->addrlen = iff->type.addrlen;
	PKT_PROTOCOL(p) = NET_PROTOCOL_ARP;
	if (op) memcpy(PKT_DSTADDR(p, iff->type.addrlen), PKT_SRCADDR(op, op->addrlen), iff->type.addrlen);
	else SET_BROADCAST_DST(p);
	memcpy(PKT_SRCADDR(p, iff->type.addrlen), iff->type.addr, iff->type.addrlen);
	iplog(p);
	p->fn = NET$FREE_PACKET;
	(*(long *)(p->sender_data = &iff->outstanding))++;
	p->h = iff->handle;
	p->id = IDTYPE_LO;
	CALL_IORQ(p, KERNEL$PKTIO);
}

void IF_ARP_REQUEST(PACKET *op)
{
	in_addr_t source;
	in_addr_t target;
	int hlen;
	IF *iff = find_if_by_id(op->id);
	if (__unlikely(!iff) || __unlikely(op->addrlen != iff->type.addrlen)) return;
	/* some hosts send bogus source IP address, so don't check it */
	hlen = arp(op)->hlen;
	memcpy(&source, arpaddr(op) + hlen, sizeof(in_addr_t));
	memcpy(&target, arpaddr(op) + sizeof(in_addr_t) + (hlen << 1), sizeof(in_addr_t));
	if (__likely(iff->addr != target)) {
		IF *tif;
		if (__likely(!IFF_PROXYARP)) return;
		tif = IP_ROUTE(target, (void *)&KERNEL$LIST_END);
		if (__unlikely(!tif) || __likely(tif == iff)) return;
		/* When proxy ARP is on and we route to a different interface,
		   reply to an ARP */
	}
	IF_SEND_ARP(iff, op, source, target);
}

void IF_ARP_SEND_QUERY(unsigned long id, in_addr_t host)
{
	IF *iff = find_if_by_id(id);
	if (__unlikely(!iff)) return;
	IF_SEND_ARP(iff, NULL, host, iff->addr);
}

static void SRC_ADDR_DOESNT_MATCH(PACKET *p, __u8 *addr)
{
	static char a1[__MAX_STR_LEN];
	static char a2[__MAX_STR_LEN];
	static char a3[__MAX_STR_LEN];
	static char a4[16];
	static char a5[16];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "PACKET FROM INVALID HW ADDRESS, DSTHWADDR %s, SRCHWADDR %s, EXPECTED SRCHWADDR %s, DSTADDR %s, SRCADDR %s", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a3, addr + 1, *addr), PRINT_IP(a4, ip(p)->ip_dst.s_addr), PRINT_IP(a5, ip(p)->ip_src.s_addr));
}

static void UNKNOWN_SRC_ADDR(PACKET *p)
{
	static char a1[__MAX_STR_LEN];
	static char a2[__MAX_STR_LEN];
	static char a4[16];
	static char a5[16];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "PACKET FROM UNREACHABLE IP ADDRESS, DSTHWADDR %s, SRCHWADDR %s, DSTADDR %s, SRCADDR %s", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), PRINT_IP(a4, ip(p)->ip_dst.s_addr), PRINT_IP(a5, ip(p)->ip_src.s_addr));
}

static void BAD_INTERFACE(PACKET *p)
{
	static char a1[__MAX_STR_LEN];
	static char a2[__MAX_STR_LEN];
	static char a4[16];
	static char a5[16];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "PACKET CAME FROM BAD INTERFACE, DSTHWADDR %s, SRCHWADDR %s, DSTADDR %s, SRCADDR %s", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), PRINT_IP(a4, ip(p)->ip_dst.s_addr), PRINT_IP(a5, ip(p)->ip_src.s_addr));
}

static void BAD_DST_ADDR(PACKET *p)
{
	static char a1[__MAX_STR_LEN];
	static char a2[__MAX_STR_LEN];
	static char a4[16];
	static char a5[16];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DESTINATION ADDRESS DOESN'T MATCH INTERFACE ADDRESS, DSTHWADDR %s, SRCHWADDR %s, DSTADDR %s, SRCADDR %s", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), PRINT_IP(a4, ip(p)->ip_dst.s_addr), PRINT_IP(a5, ip(p)->ip_src.s_addr));
}

static int CHECK_MCAST_ADDRESS(PACKET *p);

int ARP_CHECK_SECURITY(PACKET *p)
{
	__u8 *addr;
	in_addr_t host;
	IF *iff;
	if (__unlikely(TCPIP_MULTICAST_ADDRESS(ip(p)->ip_dst.s_addr))) {
		return CHECK_MCAST_ADDRESS(p);
	}
	if (__unlikely(!(iff = IP_ROUTE(ip(p)->ip_src.s_addr, &host)))) {
		UNKNOWN_SRC_ADDR(p);
		goto drop;
	}
	if (__unlikely(iff->type.id != p->id)) {
		if (__unlikely(p->id != IDTYPE_LO)) {
			BAD_INTERFACE(p);
			goto drop;
		}
		return 0;
	}
	if (__unlikely(iff->addr != ip(p)->ip_dst.s_addr) && __unlikely(!IFF_GATE)) {
		BAD_DST_ADDR(p);
		goto drop;
	}
	if (__likely(local_security < 2)) return 0;
	if (__unlikely(!p->addrlen)) return 0;
	if (__unlikely(!iff->type.addrlen) || __unlikely(iff->flags & IFF_PTP)) return 0;
	/*{
		unsigned long off_ = 0;
		const char *str_ = KERNEL$DL_GET_SYMBOL_NAME((p)->alloc_ptr, &off_, 0);
		unsigned long off2_ = 0;
		const char *str2_ = KERNEL$DL_GET_SYMBOL_NAME((p)->fn, &off2_, 0);
		__debug_printf("arp find: FREE PACKET, PACKET SIZE %d, DATA SIZE %d, FLAGS %X, ALLOCATED AT %s+%lX, AST %s+%lX\n", (p)->length, (p)->data_length, (p)->flags, str_ ? str_ : "?", off_, str2_ ? str2_ : "?", off2_);
	}*/
	if (__likely((addr = ARP_FIND(p->id, host, p, IP_INPUT_LOCAL)) != NULL)) {
		if (__unlikely(addr[0] != p->addrlen) || __unlikely(memcmp(addr + 1, PKT_SRCADDR(p, p->addrlen), p->addrlen))) {
			SRC_ADDR_DOESNT_MATCH(p, addr);
			goto drop;
		}
		return 0;
	}
	return -1;

	drop:
	p->status = 0;
	CALL_PKT(p);
	return -1;
}

static int CHECK_MCAST_ADDRESS(PACKET *p)
{
	__u8 mc[MCAST_PHYS_ADDRLEN];
	if (__unlikely(p->addrlen != MCAST_PHYS_ADDRLEN)) return 0;
	TEST_BROADCAST_PACKET(p, return 0);
	MCAST_PHYS_ADDR(ip(p)->ip_dst.s_addr, mc);
	if (__unlikely(memcmp(PKT_DSTADDR(p, MCAST_PHYS_ADDRLEN), mc, MCAST_PHYS_ADDRLEN))) goto drop;
	return 0;

	drop:
	p->status = 0;
	CALL_PKT(p);
	return -1;
}

#if 0
static void ARP_BAD_INTERFACE(PACKET *p, in_addr_t sender, in_addr_t target)
{
	static char a1[__MAX_STR_LEN];
	static char a2[__MAX_STR_LEN];
	static char a4[16];
	static char a5[16];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "ARP PACKET CAME FROM BAD INTERFACE, DSTHWADDR %s, SRCHWADDR %s, DSTADDR %s, SRCADDR %s", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), PRINT_IP(a4, target), PRINT_IP(a5, sender));
}
#endif

static __finline__ void DHCP_MAKE_CHECKSUM(PACKET *p)
{
	p->flags |= PKT_OUTPUT_CHECKSUM | PKT_OUTPUT_CHECKSUM_UDP | PKT_TCPUDP_CHECKSUM_OK;
	p->checksum.u = MKCHECKSUM(sizeof(struct ip) - 2 * sizeof(in_addr_t), sizeof(struct ip) + 6);
	dhcp(p)->udp_checksum = TCPUDP_MAGIC_CHECKSUM(IPPROTO_UDP, sizeof(struct dhcp) - sizeof(struct ip));
}

void IF_SEND_DHCP_PACKET(DHCP *dhcp, PACKET *p)
{
	IF *iff = DHCP_IF(dhcp);
	DHCP_MAKE_CHECKSUM(p);
	IP_HEADER_MAKE_CHECKSUM((__u8 *)ip(p));
	iplog(p);
	p->fn = NET$FREE_PACKET;
	(*(long *)(p->sender_data = &iff->outstanding))++;
	p->h = iff->handle;
	p->id = IDTYPE_LO;
	CALL_IORQ(p, KERNEL$PKTIO);
}

void TCPIP_IF_SHUTDOWN(void)
{
	IF *iff;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_NET)))
		KERNEL$SUICIDE("TCPIP_IF_SHUTDOWN AT SPL %08X", KERNEL$SPL);
	while (!LIST_EMPTY(&interfaces)) {
		iff = LIST_STRUCT(interfaces.next, IF, list);
		LOWER_SPL(SPL_ZERO);
		delete_if(iff, NULL);
		RAISE_SPL(SPL_NET);
	}
	if (__unlikely(n_interfaces))
		KERNEL$SUICIDE("TCPIP_IF_SHUTDOWN: n_interfaces LEAKED (%u)", n_interfaces);
	n_routes = 0;
	free(routes);
}

#define IFC_DOWN		1
#define IFC_DHCP		2
#define IFC_BOOTP		4

int TCPIP_IFCONFIG(const char * const argv[])
{
	const char * const *arg;
	int state;
	IF *iff;
	char *iname = NULL, *addr = NULL, *netmask = NULL, *pointtopoint = NULL;
	in_addr_t addr_addr, netmask_addr, pointtopoint_addr;
	int flags = 0;
	int mtu = 0;
	int dhcp_secure = 0, dhcp_distrust = 0, dhcp_distrust_local = 0;
	int igmp = 0;
	int flg = 0;
	static const struct __param_table ext_params[4] = {
		"SECURE", __PARAM_BOOL, ~0, 1,
		"DISTRUST", __PARAM_BOOL, ~0, 1,
		"DISTRUST_LOCAL", __PARAM_BOOL, ~0, 1,
		NULL, 0, 0, 0,
	};
	void *ext_vars[4];
	static const struct __param_table params[12] = {
		"", __PARAM_STRING, 1, MAXINT,
		"", __PARAM_STRING, 1, MAXINT,
		"NETMASK", __PARAM_STRING, 1, MAXINT,
		"POINTTOPOINT", __PARAM_STRING, 1, MAXINT,
		"MTU", __PARAM_INT, IP_MIN_MTU, MAX_PACKET_LENGTH_TOP + 1,
		"DOWN", __PARAM_BOOL, IFC_DOWN, IFC_DOWN,
		"DHCP", __PARAM_BOOL_X, IFC_DHCP, IFC_DHCP,
		"BOOTP", __PARAM_BOOL_X, IFC_BOOTP, IFC_BOOTP,
		"DHCP", __PARAM_EXTD, (long)&ext_params, 0,
		"BOOTP", __PARAM_EXTD, (long)&ext_params, 0,
		"IGMP", __PARAM_INT, 1, 3,
		NULL, 0, 0, 0,
	};
	void *vars[12];
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("TCPIP_IFCONFIG AT SPL %08X", KERNEL$SPL);
	vars[0] = &iname;
	vars[1] = &addr;
	vars[2] = &netmask;
	vars[3] = &pointtopoint;
	vars[4] = &mtu;
	vars[5] = &flags;
	vars[6] = &flags;
	vars[7] = &flags;
	vars[8] = &ext_vars;
	vars[9] = &ext_vars;
	vars[10] = &igmp;
	vars[11] = NULL;
	ext_vars[0] = &dhcp_secure;
	ext_vars[1] = &dhcp_distrust;
	ext_vars[2] = &dhcp_distrust_local;
	ext_vars[3] = NULL;
	arg = argv;
	state = 0;
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: IFCONFIG: SYNTAX ERROR", net_name);
		return -EBADSYN;
		bad_a:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: IFCONFIG: BAD ADDRESS", net_name);
		return -EINVAL;
	}
	if (__unlikely(!iname)) {
		if (__unlikely(netmask != NULL) || __unlikely(pointtopoint != NULL) || __unlikely(flags | mtu | igmp)) goto bads;
		LIST_FOR_EACH(iff, &interfaces, IF, list) print_if(iff);
		return 0;
	}
	if (igmp == 2) flg |= IFF_IGMP2;
	if (__unlikely(flags & (IFC_DHCP | IFC_BOOTP))) {
		flg |= IFF_DHCP;
		if (__unlikely((flags & (IFC_DHCP | IFC_BOOTP)) == (IFC_DHCP | IFC_BOOTP)) || __unlikely(flags & IFC_DOWN)) goto bads;
		if (__unlikely(flags & IFC_BOOTP)) flg |= IFF_DHCP_ONLY_BOOTP;
		if (dhcp_secure) flg |= IFF_DHCP_SECURE;
		if (dhcp_distrust) flg |= IFF_DHCP_DISTRUST;
		if (dhcp_distrust_local) flg |= IFF_DHCP_DISTRUST_LOCAL;
		if (__unlikely(netmask != NULL) || __unlikely(pointtopoint != NULL)) goto bads;
		addr_addr = 0;
		if (__unlikely(addr != NULL)) {
			if (__unlikely(flags & IFC_BOOTP)) goto bads;
			if (__unlikely(GET_IP(addr, &addr_addr, NULL))) goto bads;
			if (__unlikely(INAPPROPRIATE_IF_ADDRESS(addr_addr))) goto bad_a;
			flg |= IFF_DHCP_REQUEST_ADDR;
		}
		iff = find_if_by_name(iname);
		if (__unlikely(iff != NULL)) delete_if(iff, NULL);
		return add_if(iname, flg, addr_addr, 0, 0, mtu, NULL);
	}
	if (!addr) {
		if (__unlikely(netmask != NULL) || __unlikely(pointtopoint != NULL) || __unlikely(mtu | igmp)) goto bads;
		if (__unlikely(!(iff = find_if_by_name(iname)))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INTERFACE %s NOT FOUND", net_name, iname);
			return -ENOENT;
		}
		if (__unlikely(!(flags & IFC_DOWN))) print_if(iff);
		else delete_if(iff, NULL);
		return 0;
	}
	if (__unlikely(flags & IFC_DOWN)) goto bads;
	if (__unlikely(GET_IP(addr, &addr_addr, NULL))) goto bads;
	if (__unlikely(INAPPROPRIATE_IF_ADDRESS(addr_addr))) goto bad_a;
	iff = find_if_by_name(iname);
	if (__unlikely(pointtopoint != NULL)) {
		flg |= IFF_PTP;
		if (__unlikely(netmask != NULL)) goto bads;
		if (__unlikely(GET_IP(pointtopoint, &pointtopoint_addr, NULL))) goto bads;
		if (__unlikely(INAPPROPRIATE_IF_ADDRESS(pointtopoint_addr))) goto bad_a;
		return add_if(iname, flg, addr_addr, 0, pointtopoint_addr, mtu, iff);
	}
	if (__unlikely(!netmask)) {
		netmask_addr = default_netmask(addr_addr);
	} else {
		if (__unlikely(GET_IP(netmask, &netmask_addr, NULL))) goto bads;
	}
	if (__unlikely(!IS_NETMASK(netmask_addr))) goto bads;
	return add_if(iname, flg, addr_addr, netmask_addr, 0, mtu, iff);
}

int TCPIP_ROUTE(const char * const argv[])
{
	const char * const *arg;
	int state;
	int add;
	char *adddel = NULL, *dest = NULL, *netmask = NULL, *gw = NULL;
	in_addr_t dest_addr, netmask_addr, gw_addr;
	unsigned i;
	int r;
	IF *iff;
	static char a1[16];
	static const struct __param_table params[5] = {
		"", __PARAM_STRING, 1, __MAX_STR_LEN,
		"", __PARAM_STRING, 1, MAXINT,
		"NETMASK", __PARAM_STRING, 1, MAXINT,
		"", __PARAM_STRING, 1, MAXINT,
		NULL, 0, 0, 0,
	};
	void *vars[5];
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("TCPIP_ROUTE AT SPL %08X", KERNEL$SPL);
	vars[0] = &adddel;
	vars[1] = &dest;
	vars[2] = &netmask;
	vars[3] = &gw;
	vars[4] = NULL;
	arg = argv;
	state = 0;
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ROUTE: SYNTAX ERROR", net_name);
		return -EBADSYN;
	}
	if (__unlikely(!adddel)) {
		if (__unlikely(dest != NULL) || __unlikely(netmask != NULL)) goto bads;
		for (i = 0; i < n_routes; i++) print_route(&routes[i], 0);
		return 0;
	}
	if (!_strcasecmp(adddel, "ADD")) add = 1;
	else if (!_strcasecmp(adddel, "DEL") || !_strcasecmp(adddel, "DELETE")) add = 0;
	else goto bads;
	if (add) {
		if (__unlikely(GET_IP(gw, &gw_addr, NULL))) goto bads;
	} else if (__unlikely(gw != NULL)) goto bads;
	if (__unlikely(!_strcasecmp(dest, "DEFAULT"))) {
		if (__unlikely(netmask != NULL)) goto bads;	
		dest_addr = 0;
		netmask_addr = 0;
	} else {
		if (__unlikely(GET_IP(dest, &dest_addr, NULL))) goto bads;
		if (netmask) {
			if (__unlikely(GET_IP(netmask, &netmask_addr, NULL))) goto bads;
		} else netmask_addr = -1;
	}
	if (!IS_NETMASK(netmask_addr)) goto bads;
	dest_addr &= netmask_addr;
	if (add) {
		int ii;
		LIST_FOR_EACH(iff, &interfaces, IF, list) {
			ii = is_in_interface(iff, gw_addr);
			if (ii == 1) goto ok;
			if (__unlikely(ii == 2)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: GATEWAY %s IS LOCAL INTERFACE ADDRESS", net_name, PRINT_IP(a1, gw_addr));
				return -EINVAL;
			}
		}
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: GATEWAY %s IS NOT ON ANY INTERFACE", net_name, PRINT_IP(a1, gw_addr));
		return -EINVAL;
		ok:
		if ((r = allocate_routes(&routes, n_routes + DHCP_RESERVED_ROUTES + 2 + (n_interfaces * 2)))) return r;
		add_route(routes, &n_routes, dest_addr, netmask_addr, gw_addr, iff);
		return 0;
	} else {
		RAISE_SPL(SPL_NET);
		for (i = 0; i < n_routes; i++) {
			if (routes[i].addr == dest_addr && routes[i].netmask == netmask_addr) {
				memmove(&routes[i], &routes[i + 1], (n_routes - i - 1) * sizeof(ROUTE));
				n_routes--;
				LOWER_SPL(SPL_ZERO);
				return 0;
			}
		}
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ROUTE NOT FOUND", net_name);
		return -ENOENT;
	}
}

int TCPIP_GATE(const char * const argv[])
{
	const char * const *arg;
	int state;
	char *onoff = NULL;
	int proxyarp = -1;
	static const struct __param_table params[4] = {
		"", __PARAM_STRING, 1, MAXINT,
		"PROXYARP", __PARAM_BOOL, ~0, 1,
		"NOPROXYARP", __PARAM_BOOL, ~0, 0,
		NULL, 0, 0, 0,
	};
	void *vars[4];
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("TCPIP_GATE AT SPL %08X", KERNEL$SPL);
	vars[0] = &onoff;
	vars[1] = &proxyarp;
	vars[2] = &proxyarp;
	vars[3] = NULL;
	arg = argv;
	state = 0;
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: GATE: SYNTAX ERROR", net_name);
		return -EBADSYN;
	}
	RAISE_SPL(SPL_NET);
	if (!onoff) {
		if (proxyarp >= 0) {
			if (!IFF_GATE) {
				bad_proxyarp:
				LOWER_SPL(SPL_ZERO);
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: GATE: CAN'T SET PROXY ARP WITHOUT GATE", net_name);
				return -EINVAL;
			}
			goto set_proxyarp;
		}
		_printf("IPGATE %s%s\n", IFF_GATE ? "ON" : "OFF", !IFF_GATE ? "" : IFF_PROXYARP ? " /PROXYARP" : " /NOPROXYARP");
	} else if (!_strcasecmp(onoff, "ON")) {
		IFF_GATE = 1;
		set_proxyarp:
		if (proxyarp >= 0) IFF_PROXYARP = proxyarp;
	} else if (!_strcasecmp(onoff, "OFF")) {
		if (proxyarp > 0) goto bad_proxyarp;
		IFF_GATE = 0;
		IFF_PROXYARP = 0;
	} else {
		LOWER_SPL(SPL_ZERO);
		goto bads;
	}
	LOWER_SPL(SPL_ZERO);
	return 0;
}

