#include <SPAD/SYNC.H>
#include <SPAD/PKT.H>
#include <SPAD/SOCKET.H>
#include <SYS/SOCKET.H>
#include <SPAD/SYSLOG.H>
#include <SYS/TIME.H>

#include "TCPIP.H"

#ifdef __IP_LOG

static __u8 logbuf[2*1024*1024];
static int logwriting;
static int logend;
static int logh = -1;
static int logbug = 0;

static void newlog(void);
static void kick_write(void);

struct loghdr {
	__u32 magic;
	unsigned short version_major;
	unsigned short version_minor;
	__s32 thiszone;
	__u32 sigfigs;
	__u32 snaplen;
	__u32 linktype;
};

struct pkthdr {
	unsigned sec;
	unsigned usec;
	__u32 caplen;
	__u32 len;
};

DECL_AST(close_done, SPL_NET, CLOSERQ)
{
	logh = -1;
	RETURN;
}

DECL_AST(open_done_2, SPL_NET, OPENRQ)
{
	if (RQ->status < 0) {
		if (!logbug) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, net_name, "CREATE LOG FAILED: %s", strerror(-RQ->status)), logbug = 1;
		newlog();
		RETURN;
	}
	logh = RQ->status;
	memmove(logbuf + sizeof(struct loghdr), logbuf, logend);
	logwriting += sizeof(struct loghdr);
	logend += sizeof(struct loghdr);
	((struct loghdr *)logbuf)->magic = 0xa1b2c3d4;
	((struct loghdr *)logbuf)->version_major = 2;
	((struct loghdr *)logbuf)->version_minor = 4;
	((struct loghdr *)logbuf)->thiszone = 0;
	((struct loghdr *)logbuf)->sigfigs = 0;
	((struct loghdr *)logbuf)->snaplen = MAX_PACKET_LENGTH_TOP;
	((struct loghdr *)logbuf)->linktype = 1;
	kick_write();
	RETURN;
}

DECL_AST(open_done, SPL_NET, OPENRQ)
{
	static OPENRQ op;
	if (RQ->status < 0) {
		op.fn = open_done_2;
		op.flags = O_CREAT | O_APPEND | O_WRONLY;
		op.path = __IP_LOG;
		op.cwd = NULL;
		RETURN_IORQ(&op, KERNEL$OPEN);
	}
	logh = RQ->status;
	kick_write();
	RETURN;
}

DECL_AST(write_done, SPL_NET, SIORQ)
{
	if (RQ->status < 0) {
		if (!logbug) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, net_name, "WRITE LOG FAILED: %s", strerror(-RQ->status)), logbug = 1;
	} else if (RQ->v.len) {
		if (!logbug) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, net_name, "WRITE LOG FAILED (EOF)"), logbug = 1;
	} else logbug = 0;
	newlog();
	RETURN;
}

static void newlog(void)
{
	memmove(logbuf, logbuf + logwriting, logend - logwriting);
	if ((logwriting = logend = logend - logwriting)) kick_write();
}

static void kick_write(void)
{
	static OPENRQ op;
	static SIORQ wr;
	if (logh == -1) {
		op.fn = open_done;
		op.flags = O_APPEND | O_WRONLY;
		op.path = __IP_LOG;
		op.cwd = NULL;
		CALL_IORQ(&op, KERNEL$OPEN);
		return;
	}
	wr.fn = write_done;
	wr.h = logh;
	wr.v.ptr = (unsigned long)logbuf;
	wr.v.len = logwriting;
	wr.v.vspace = &KERNEL$VIRTUAL;
	wr.progress = 0;
	CALL_IORQ(&wr, KERNEL$WRITE);
}

void iplog(PACKET *p)
{
	struct timeval tv;
	if (!p->data_length) return;
	if (logend + p->data_length + 14 > sizeof logbuf - sizeof(struct loghdr) - sizeof(struct pkthdr)) {
		KERNEL$SYSLOG(__SYSLOG_NET_WARNING, net_name, "WRITE BUFFER OVERFLOW");
		return;
	}
	gettimeofday(&tv, NULL);
	((struct pkthdr *)&logbuf[logend])->sec = tv.tv_sec;
	((struct pkthdr *)&logbuf[logend])->usec = tv.tv_usec;
	((struct pkthdr *)&logbuf[logend])->caplen = p->data_length + 14;
	((struct pkthdr *)&logbuf[logend])->len = p->data_length + 14;
	logend += sizeof(struct pkthdr);
	memcpy(logbuf + logend, &p->data[LL_HEADER - 14], p->data_length + 14);
	logend += p->data_length + 14;
	if (!logwriting) logwriting = logend, kick_write();
}

#endif

static int TCPIP_INIT(const char * const *argv);
static void TCPIP_DONE(void);
static int TCPIP_DCTL(void *ptr, void **unload, const char * const argv[]);
void TCPIP_SOCKET_CTOR(SOCKET *s_);

static const SOCKET_SPACE_OPERATIONS TCPIP_OPS = {
	AF_INET,
	sizeof(TCPIP_SOCKET),
	TCPIP_SOCKET_CTOR,
	TCPIP_INIT,
	TCPIP_DONE,
	TCPIP_DCTL,
	TCPIP_INIT_SOCKET,
	TCPIP_DESTROY_SOCKET,
	TCPIP_CLOSE_SOCKET,
	TCPIP_DUP_SOCKET,
	TCPIP_LINGER,
	TCPIP_IOCTL,
	TCPIP_PARSE_NETACL,
	TCPIP_COPY_NETACL,
	TCPIP_FREE_NETACL,
	TCPIP_DELETE_OFFENSIVE_SOCKETS,
};

static char net_name_[9];
char *const net_name = net_name_;
int errorlevel = 0;
int local_security = 0;
struct utsname utsname;

const char * const ifconfig_arg[] = { "IFCONFIG", "PKT$LO:", "127.0.0.1", "/NETMASK=255.0.0.0", NULL };

static int TCPIP_INIT(const char * const *argv)
{
	int r;
	const char * const *arg;
	int state;
	static const struct __param_table params[5] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_REMOTE_ERRORS", __PARAM_BOOL, ~0, 2,
		"CHECK_INTERFACE", __PARAM_BOOL, ~0, 1,
		"CHECK_ADDRESS", __PARAM_BOOL, ~0, 2,
		NULL, 0, 0, 0,
	};
	static void * const vars[5] = {
		&errorlevel,
		&errorlevel,
		&local_security,
		&local_security,
		NULL,
	};
	memcpy(net_name_, "NET$", 4);
	memcpy(net_name_ + 4, &TCPIP_OPS.af, 4);
	net_name_[8] = 0;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SYNTAX ERROR", net_name);
		return -EBADSYN;
	}
	memset(&utsname, 0, sizeof utsname);
	if (uname(&utsname)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET SYSTEM NAME: %s", net_name, strerror(errno));
		return -errno;
	}
	memset(DHCP_APPARRAY, 0, sizeof DHCP_APPARRAY);
	memset(DHCP_APPOWNER, 0, sizeof DHCP_APPOWNER);
	if ((r = RANDOM_TCP_INIT())) return r;
	NETACL_INIT();
	IP_FRAG_INIT();
	IGMP_INIT();
	if ((r = TCPIP_SOCKETS_INIT())) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE SOCKET HASH: %s", net_name, strerror(-r));
		goto err_sockets;
	}
	if ((r = ARP_INIT())) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INIT ARP: %s", net_name, strerror(-r));
		goto err_arp;
	}
	if ((r = NET$REGISTER_PROTOCOL(ARP_INPUT, NET_PROTOCOL_ARP))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER PROTOCOL ARP (%04X)", net_name, NET_PROTOCOL_ARP);
		goto err_proto_arp;
	}
	if ((r = NET$REGISTER_PROTOCOL(IP_INPUT, NET_PROTOCOL_IP))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER PROTOCOL IP (%04X)", net_name, NET_PROTOCOL_IP);
		goto err_proto_ip;
	}
	LOWER_SPL(SPL_ZERO);
	r = TCPIP_IFCONFIG(ifconfig_arg);
	RAISE_SPL(SPL_NET);
	if (__unlikely(r)) goto err_ifconfig_lo;
	return 0;

	err_ifconfig_lo:
	NET$UNREGISTER_PROTOCOL(NET_PROTOCOL_IP);
	err_proto_ip:
	NET$UNREGISTER_PROTOCOL(NET_PROTOCOL_ARP);
	err_proto_arp:
	ARP_DONE();
	err_arp:
	TCPIP_SOCKETS_DONE();
	err_sockets:
	TCPIP_IF_SHUTDOWN();
	IGMP_DONE();
	IP_FRAG_DONE();
	NETACL_DONE();
	RANDOM_TCP_DONE();
	return r;
}

static void TCPIP_DONE(void)
{
	NET$UNREGISTER_PROTOCOL(NET_PROTOCOL_IP);
	NET$UNREGISTER_PROTOCOL(NET_PROTOCOL_ARP);
	ARP_DONE();
	TCPIP_IF_SHUTDOWN();
	TCPIP_SOCKETS_DONE();
	IGMP_DONE();
	IP_FRAG_DONE();
	NETACL_DONE();
	RANDOM_TCP_DONE();
#ifdef __IP_LOG
	while (logwriting) KERNEL$SLEEP(1);
	if (logh != -1) {
		CLOSERQ c;
		c.fn = close_done;
		c.h = logh;
		CALL_IORQ(&c, KERNEL$CLOSE);
		while (logh != -1) KERNEL$SLEEP(1);
	}
#endif
}

void *tcpip_lnte;

int main(int argc, const char * const argv[])
{
	PKT_CHECK_VERSION;
	return NET$CREATE_SOCKET_SPACE(argc, argv, &TCPIP_OPS, "TCPIP.SYS", &tcpip_lnte);
}

static int TCPIP_DCTL(void *ptr, void **unload, const char * const argv[])
{
	if (__unlikely(!argv[0]) || __unlikely(!argv[1])) {
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SYNTAX ERROR", net_name);
		return -EBADSYN;
	}
	KERNEL$SLEEP(1);	/* give EAPOL some chance to run before configuring DHCP */
	if (!_strcasecmp(argv[1], "IFCONFIG")) {
		int r = TCPIP_IFCONFIG(argv + 1);
		/*TCPIP_SOCKETS_CHECK();*/
		IGMP_CHECK_MULTICASTS();
		return r;
	} else if (!_strcasecmp(argv[1], "ROUTE")) {
		int r = TCPIP_ROUTE(argv + 1);
		IGMP_CHECK_MULTICASTS();
		return r;
	} else if (!_strcasecmp(argv[1], "IPGATE")) {
		return TCPIP_GATE(argv + 1);
	} else goto bads;
}
