#include <STRING.H>

#include "TCPIP.H"

#define ACL_SLOTS	32

typedef union __aclslot ACLSLOT;

union __aclslot {
	struct {
		unsigned char flags;
		unsigned char addresses;
		unsigned char tcp_ports;
		unsigned char udp_ports;
	} h;
	struct {
		in_addr_t addr;
	} a;
	struct {
		in_addr_t mask;
	} m;
	struct {
		in_port_t port_from;
		in_port_t n_ports_minus_1;
	} p;
};

#define NETACL_DENY	0x01
#define NETACL_IN	0x02

struct __netacl {
	int refcount;
	unsigned slots;
	ACLSLOT slot[ACL_SLOTS];
};

struct __slhead netacl_slab;

static void *CREATE_NETACL(SOCKET_NODE *sn, NETACL **netacl)
{
	NETACL *n;
	retry_alloc:
	n = *netacl = __slalloc(&netacl_slab);
	if (__unlikely(!n)) {
		WQ *wq = NET$OOM_NODE_SOCKET(sn, NULL);
		if (__unlikely(!wq)) goto retry_alloc;
		return wq;
	}
	/*__debug_printf("create: %p\n", n);*/
	n->refcount = 1;
	n->slots = 0;
	*netacl = n;
	return NULL;
}

/*
   ^OUT_ACCESS=1.2.3.4-24,1.2.3.5-24.T23-25,T80,U8080
   ^OUT_DENY
   ^IN_ACCESS=.T23-25,T80,U8080
   ^IN_DENY=.T
*/

void *TCPIP_PARSE_NETACL(SOCKET_NODE *sn, char *str, NETACL **netacl)
{
	NETACL *n = *netacl;
	char *e;
	unsigned char flags;
	unsigned s;
	long m;
	if (!n) {
		void *r;
		if (__unlikely((r = CREATE_NETACL(sn, netacl)) != NULL)) return r;
		n = *netacl;
	}
	str++;
	e = strchr(str, '=');
	if (__unlikely(!e)) e = str + strlen(str);
	if (!__strcasexcmp("OUT_ACCESS", str, e)) {
		if (__unlikely(!*e)) return NULL;
		flags = 0;
	} else if (!__strcasexcmp("OUT_DENY", str, e)) {
		flags = NETACL_DENY;
	} else if (!__strcasexcmp("IN_ACCESS", str, e)) {
		if (__unlikely(!*e)) return NULL;
		flags = NETACL_IN;
	} else if (!__strcasexcmp("IN_DENY", str, e)) {
		flags = NETACL_DENY | NETACL_IN;
	} else {
		ebadmod: return __ERR_PTR(-EBADMOD);
	}
	s = n->slots;
	if (__unlikely(s >= ACL_SLOTS)) {
		e2big:
		return __ERR_PTR(-E2BIG);
	}
	n->slot[s].h.flags = flags;
	s++;
	if (__likely(*e)) e++;
	if (__likely(*e) && *e != '.') {
		next_addr:
		if (__unlikely(s >= ACL_SLOTS - 1)) goto e2big;
		if (__unlikely(GET_IP(e, &n->slot[s].a.addr, &e))) goto ebadmod;
		s++;
		n->slot[s].m.mask = htonl(0xffffffff);
		if (*e == '-') {
			char *x;
			e++;
			x = e + strcspn(e, ".,");
			if (__unlikely(*x == '.')) {
				if (__unlikely(GET_IP(e, &n->slot[s].m.mask, &e))) goto bits_mask;
				if (__unlikely(!IS_NETMASK(n->slot[s].m.mask))) goto ebadmod;
			} else {
				bits_mask:
				if (__unlikely(__get_number(e, x, 0, &m)) || __unlikely((unsigned long)m > 32)) goto ebadmod;
				e = x;
				if (__likely(m < 32)) {
					n->slot[s].m.mask = htonl(0xfffffffe << (31 - m));
				}
			}
			if (__unlikely(n->slot[s - 1].a.addr & ~n->slot[s].m.mask)) goto ebadmod;
		}
		s++;
		if (*e == ',') {
			e++;
			goto next_addr;
		}
	}
	if (__unlikely((s - n->slots) >= 512)) goto e2big;
	n->slot[n->slots].h.addresses = (s - n->slots) >> 1;
	n->slot[n->slots].h.tcp_ports = 0;
	n->slot[n->slots].h.udp_ports = 0;
	if (*e == '.') {
		char *b = e;
		unsigned char c;
		unsigned char type;
		unsigned idx[2];
		if (__unlikely(!e[1])) goto ebadmod;
		idx[0] = idx[1] = s;
		c = 0;
		ports_pass_2:
		type = 0;
		e = b;
		do {
			unsigned from, to;
			char *x;
			e++;
			if (__upcasechr(*e) == 'T' || __upcasechr(*e) == 'U') {
				type = __upcasechr(*e) == 'U';
				e++;
				if (!*e || *e == ',') {
					from = 0;
					to = IPPORT_MAX - 1;
					goto set_range;
				}
			}
			x = e + strcspn(e, ",-");
			if (__unlikely(__get_number(e, x, 0, &m)) || __unlikely((unsigned long)m >= IPPORT_MAX)) {
				ebadmod_inloop:
				if (__unlikely(c == 1))
					KERNEL$SUICIDE("TCPIP_PARSE_NETACL: FAILURE IN SECOND PASS, STRING \"%s\"", str);
				goto ebadmod;
			}
			from = to = m;
			e = x;
			if (*e == '-') {
				e++;
				x = e + strcspn(e, ",");
				if (__unlikely(__get_number(e, x, 0, &m)) || __unlikely((unsigned long)m >= IPPORT_MAX)) goto ebadmod_inloop;
				to = m;
				e = x;
			}
			set_range:
			if (__unlikely(idx[(unsigned)type] >= ACL_SLOTS)) goto e2big;
			n->slot[idx[(unsigned)type]].p.port_from = from;
			if (__unlikely((to -= from) < 0)) goto ebadmod_inloop;
			n->slot[idx[(unsigned)type]++].p.n_ports_minus_1 = to;
		} while (*e == ',');
		if (__unlikely(*e)) goto ebadmod_inloop;
		if (!c) {
			n->slot[n->slots].h.udp_ports = idx[1] - s;
			n->slot[n->slots].h.tcp_ports = idx[0] - s;
			idx[1] = idx[0];
			idx[0] = s;
			c = 1;
			goto ports_pass_2;
		}
		if (__unlikely(idx[0] - s != n->slot[n->slots].h.tcp_ports) || __unlikely(idx[1] - idx[0] != n->slot[n->slots].h.udp_ports))
			KERNEL$SUICIDE("TCPIP_PARSE_NETACL: PORT NUMBER MISCOUNTED: S %d, IDX %d %d, COUNTED %d %d, STRING \"%s\"", s, idx[0], idx[1], n->slot[n->slots].h.tcp_ports, n->slot[n->slots].h.udp_ports, str);
		s += n->slot[n->slots].h.tcp_ports + n->slot[n->slots].h.udp_ports;
	} else if (__unlikely(*e)) goto ebadmod;
	/*__debug_printf("parsed: %d %d %d @ %d -> %p\n", n->slot[n->slots].h.addresses, n->slot[n->slots].h.tcp_ports, n->slot[n->slots].h.udp_ports, n->slots, n);*/
	n->slots = s;
	return NULL;
}

static void RESET_NETACL(SOCKET *s_);

void *TCPIP_COPY_NETACL(SOCKET_NODE *sn, NETACL **dest, NETACL *src)
{
	if (__unlikely(*dest != NULL)) TCPIP_FREE_NETACL(*dest);
	if (__likely(src == NULL)) {
		*dest = NULL;
	} else {
		(*dest = src)->refcount++;
	/*__debug_printf("link: %p (%d)\n", src, src->refcount);*/
	}
	NET$FOR_ALL_SOCKETS(sn, RESET_NETACL);
	return NULL;
}

static void RESET_NETACL(SOCKET *s_)
{
	TCPIP_SOCKET *s = (void *)s_;
	s->flags &= ~(SOCK_NETACL_ACCEPT_IN | SOCK_NETACL_ACCEPT_OUT);
}

void TCPIP_FREE_NETACL(NETACL *netacl)
{
	/*__debug_printf("unlink: %p (%d)\n", netacl, netacl->refcount);*/
#if __DEBUG >= 1
	if (__unlikely(netacl->refcount <= 0))
		KERNEL$SUICIDE("TCPIP_FREE_NETACL: INVALID REFCOUNT %d", netacl->refcount);
#endif
	if (!--netacl->refcount) __slfree(netacl);
}

/* addr is in network byte order, port is in host byte order (with +IPPORT_MAX for UDP port, +IPPORT_MAX*2 for IN database) */
int NETACL_SEARCH(SOCKET_NODE *sn, in_addr_t addr, unsigned port)
{
	unsigned char retrn = NETACL_ALL | NETACL_ALLADDR;
	while (1) {
		unsigned idx;
		unsigned char lastflags;
		NETACL *n = NET$GET_NETACL(&sn);
		if (__likely(!n)) {
			retrn |= NETACL_DEFAULT_RESTRICT;
			continue;
		}
		if (__likely(n == NETACL_END)) break;
		if (__unlikely(!n->slots)) {
			continue;
		}
		idx = 0;
		lastflags = NETACL_DENY;
		do {
			unsigned addridx;
			int addrlen;
			if (((port / (IPPORT_MAX * 2 / NETACL_IN)) ^ n->slot[idx].h.flags) & NETACL_IN) goto didnt_pass;
			lastflags = n->slot[idx].h.flags;

			if (__likely(!(n->slot[idx].h.tcp_ports | n->slot[idx].h.udp_ports))) goto port_pass;
			retrn &= ~NETACL_ALL;
			addrlen = n->slot[idx].h.tcp_ports;
			addridx = idx + n->slot[idx].h.addresses * 2 + 1;
			if (__unlikely(port & NETACL_PORT_UDP)) {
				addridx += addrlen;
				addrlen = n->slot[idx].h.udp_ports;
			}
			while (--addrlen >= 0) {
				if ((unsigned)(port & (IPPORT_MAX - 1)) - (unsigned)n->slot[addridx].p.port_from <= (unsigned)n->slot[addridx].p.n_ports_minus_1) goto port_pass;
				addridx++;
			}
			goto didnt_pass;

			port_pass:
			if (__likely(!n->slot[idx].h.addresses)) goto addr_pass;
			retrn &= ~(NETACL_ALL | NETACL_ALLADDR);
			if (addr == htonl(INADDR_ANY)) {
				if (__unlikely(n->slot[idx].h.flags & NETACL_DENY)) goto didnt_pass;
				goto pass;
			}
			addridx = idx + n->slot[idx].h.addresses * 2;
			do {
				if ((addr & n->slot[addridx].m.mask) == n->slot[addridx - 1].a.addr) goto addr_pass;
			} while ((addridx -= 2) > idx);
			goto didnt_pass;
			addr_pass:

			if (__unlikely(n->slot[idx].h.flags & NETACL_DENY)) return -1;
			goto pass;

			didnt_pass:
			idx += 1 + n->slot[idx].h.addresses * 2 + n->slot[idx].h.tcp_ports + n->slot[idx].h.udp_ports;
		} while (__unlikely(idx < n->slots));
#if __DEBUG >= 1
		if (__unlikely(idx > n->slots))
			KERNEL$SUICIDE("NETACL_SEARCH: READ OUT OF NETACL AT POS %u > %u", idx, n->slots);
#endif
		if (__unlikely(!(lastflags & NETACL_DENY))) return -1;
		pass:;
	}
	if (port / (NETACL_PORT_IN / NETACL_DEFAULT_RESTRICT) & retrn & NETACL_DEFAULT_RESTRICT) {
		if (__unlikely((port & (IPPORT_MAX - 1)) < IPPORT_RESERVED))
			return -1;
	}
	return retrn;
}

void NETACL_INIT(void)
{
	KERNEL$SLAB_INIT(&netacl_slab, sizeof(NETACL), 0, VM_TYPE_WIRED_MAPPED, NULL, &NET$MEMORY_AVAIL, "TCPIP$NETACL");
}

void NETACL_DONE(void)
{
	KERNEL$SLAB_DESTROY(&netacl_slab);
}

