#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <SYS/STAT.H>
#include <SYS/POLL.H>
#include <TIME.H>
#include <VALUES.H>

#include "TCPIP.H"

static IO_STUB TCP_IOCTL;
static IO_STUB UDP_IOCTL;

const HANDLE_OPERATIONS TCP_OPS = {
	SPL_X(SPL_NET),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NET$SOCKET_LOOKUP,	/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NET$SOCKET_INSTANTIATE,	/* instantiate */
	NULL,			/* leave */
	NET$SOCKET_DETACH,	/* detach */
	NET$SOCKET_CLOSE,	/* close */
	TCP_READ,		/* read */
	TCP_WRITE,		/* write */
	TCP_RECVMSG,		/* aread */
	TCP_SENDMSG,		/* awrite */
	TCP_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

const HANDLE_OPERATIONS UDP_OPS = {
	SPL_X(SPL_NET),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NET$SOCKET_LOOKUP,	/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NET$SOCKET_INSTANTIATE,	/* instantiate */
	NULL,			/* leave */
	NET$SOCKET_DETACH,	/* detach */
	NET$SOCKET_CLOSE,	/* close */
	UDP_READ,		/* read */
	UDP_WRITE,		/* write */
	UDP_RECVMSG,		/* aread */
	UDP_SENDMSG,		/* awrite */
	UDP_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static XLIST_HEAD local_ports[IPPORT_MAX * 2];
static XLIST_HEAD *socket_hash;
static unsigned socket_hash_mask;

DECL_LIST(LOST_PACKETS);

#ifdef __BIG_ENDIAN
#define SOCKET_HASH(la, lp, da, dp) ((((la) ^ (lp)) ^ ((da) ^ (dp) ^ ((da) << 16))) & socket_hash_mask)
#else
#define SOCKET_HASH(la, lp, da, dp) ((((la) ^ (lp)) ^ ((da) ^ (dp) ^ ((da) >> 8))) & socket_hash_mask)
#endif

#define PORT_USER_START		IPPORT_HIFIRSTAUTO
#define PORT_USER_END		(IPPORT_HILASTAUTO + 1)

#define FLP_UDP		1

TCPIP_SOCKET *LOOKUP_SOCKET(in_addr_t local_addr, unsigned local_port, in_addr_t remote_addr, unsigned remote_port, const HANDLE_OPERATIONS *proto)
{
	TCPIP_SOCKET *s;
	unsigned hash = SOCKET_HASH(local_addr, local_port, remote_addr, remote_port);
	XLIST_FOR_EACH(s, &socket_hash[hash], TCPIP_SOCKET, hash) {
		if (__likely(s->local_addr == local_addr) && __likely(s->local_port == local_port) && __likely(s->remote_addr == remote_addr) && __likely(s->remote_port == remote_port) && __likely(s->op == proto)) return s;
	}
	return NULL;
}

TCPIP_SOCKET *LOOKUP_LISTEN(in_addr_t local_addr, unsigned local_port, unsigned proto)
{
	TCPIP_SOCKET *s;
	XLIST_FOR_EACH(s, &local_ports[local_port + proto], TCPIP_SOCKET, hash) {
		if (__likely(s->local_addr == htonl(INADDR_ANY)) || __likely(s->local_addr == local_addr)) return s;
	}
	return NULL;
}

void INSERT_INTO_HASH(TCPIP_SOCKET *s)
{
	unsigned hash = SOCKET_HASH(s->local_addr, s->local_port, s->remote_addr, s->remote_port);
	DEL_FROM_LIST(&s->hash);
	ADD_TO_XLIST(&socket_hash[hash], &s->hash);
}

static unsigned FIND_LOCAL_PORT_BIND(unsigned flags)
{
	unsigned cnt, port, step, nport;
	XLIST_HEAD *p;
	port = RANDOM_PORT_NUMBER_STEP();
	p = local_ports + ((flags * (IPPORT_MAX / FLP_UDP)) & IPPORT_MAX);
	cnt = IPPORT_MAX + 1;
	step = (port / IPPORT_MAX) | 1;
	while (--cnt) {
		port = (port + step) & (IPPORT_MAX - 1);
		if (__likely(port - PORT_USER_START >= PORT_USER_END - PORT_USER_START)) continue;
		nport = htons(port);
		if (__likely(XLIST_EMPTY(&p[nport]))) return nport;
	}
	return 0;
}

static unsigned FIND_LOCAL_PORT_CONNECT(unsigned flags, in_addr_t local_addr, in_addr_t remote_addr, unsigned remote_port)
{
	unsigned cnt, port, step;
	port = RANDOM_PORT_NUMBER_STEP();
	cnt = IPPORT_MAX + 1;
	step = (port / IPPORT_MAX) | 1;
	while (--cnt) {
		port = (port + step) & (IPPORT_MAX - 1);
		if (__likely(port - PORT_USER_START >= PORT_USER_END - PORT_USER_START)) continue;
		if (__likely(!LOOKUP_SOCKET(local_addr, htons(port), remote_addr, remote_port, flags & FLP_UDP ? LS_UDP : LS_TCP))) return htons(port);
	}
	return 0;
}

void TCPIP_SOCKET_CTOR(SOCKET *s_)
{
#define s ((TCPIP_SOCKET *)s_)
	WQ_INIT(&s->read_wait, "TCPIP$READ_WAIT");
	WQ_INIT(&s->write_wait, "TCPIP$WRITE_WAIT");
	WQ_INIT(&s->linger_wait, "TCPIP$LINGER_WAIT");
	INIT_LIST(&s->sent_queue);
	INIT_LIST(&s->out_queue);
	INIT_LIST(&s->in_queue);
	INIT_LIST(&s->in_ooo_queue);
	INIT_LIST(&s->backlog_preparing);
	INIT_LIST(&s->backlog_connected);
	INIT_TIMER(&s->timer);
	INIT_TIMER(&s->linger_timer);
	s->listener = NULL;
#undef s
}

int TCPIP_INIT_SOCKET(SOCKET *s_, char *opt)
{
	unsigned char type, protocol;
#define s ((TCPIP_SOCKET *)s_)
	if (__unlikely(!(type = opt[0])) || __unlikely(!(protocol = opt[1])) || __unlikely(opt[2])) return -EINVAL;
	type -= 'A';
	protocol -= 'A';
	if (__likely(type == SOCK_STREAM)) {
		if (protocol && __unlikely(protocol != IPPROTO_TCP)) return -EPROTONOSUPPORT;
		s->op = &TCP_OPS;
		s->packet = TCP_NULL_PACKET_1;
		s->ip_tos = IPTOS_THROUGHPUT | IPTOS_PREC_ROUTINE;
		s->mcast_ttl = 1;
		s->sock_type = SOCK_STREAM;
	} else if (__likely(type == SOCK_DGRAM)) {
		if (protocol && __unlikely(protocol != IPPROTO_UDP)) return -EPROTONOSUPPORT;
		s->op = &UDP_OPS;
		s->packet = UDP_PACKET;
		s->ip_tos = IPTOS_LOWDELAY | IPTOS_PREC_ROUTINE;
		s->mcast_ttl = 1;
		s->sock_type = SOCK_DGRAM;
		s->sock_flags |= SOCK_RESETERROR;
	} else return -EPROTONOSUPPORT;
	s->local_addr = htonl(INADDR_ANY);
	s->local_port = htons(IPPORT_ANY);
	s->remote_addr = htonl(INADDR_ANY);
	s->remote_port = htons(IPPORT_ANY);
	s->out_queue_length = 0;
	s->flags = 0;
	s->ack = s->read_seq = 0;
	VOID_LIST_ENTRY(&s->hash);
	s->prepared_packet = NULL;
	s->timer.fn = NULL;
	s->tcp_connect_timeout = TCP_CONNECT_TIMEOUT;
	s->tcp_write_timeout = TCP_WRITE_TIMEOUT;
	s->linger_timer.fn = NULL;
	s->multicast_src = htonl(INADDR_ANY);
	s->n_groups = 0;
	return 0;
#undef s
}

void TCPIP_DUP_SOCKET(SOCKET *s_, SOCKET *os_)
{
#define s ((TCPIP_SOCKET *)s_)
#define os ((TCPIP_SOCKET *)os_)
	s->op = os->op;
	s->ip_tos = os->ip_tos;
	s->mcast_ttl = os->mcast_ttl;
	s->local_addr = os->local_addr;
	s->local_port = os->local_port;
	s->out_queue_length = 0;
	s->flags = 0;
	s->ack = s->read_seq = 0;
	VOID_LIST_ENTRY(&s->hash);
	s->prepared_packet = NULL;
	s->timer.fn = NULL;
	s->tcp_connect_timeout = os->tcp_connect_timeout;
	s->tcp_write_timeout = os->tcp_write_timeout;
	s->linger_timer.fn = NULL;
	s->multicast_src = os->multicast_src;
	s->n_groups = 0;
#undef s
#undef os
}

static void DELPKTQ(TCPIP_SOCKET *s, LIST_HEAD *l, int can_out)
{
	while (!LIST_EMPTY(l)) {
		PACKET *p = LIST_STRUCT(l->next, PACKET, list);
		if (__unlikely(p->flags & PKT_OUTSTANDING) && __unlikely(!can_out))
			KERNEL$SUICIDE("DELPKTQ: OUTSTANDING PACKET ON QUEUE");
		TCPIP_DELETE_PACKET(s, p);
	}
}

void TCPIP_DESTROY_SOCKET(SOCKET *s_)
{
#define s ((TCPIP_SOCKET *)s_)
	unsigned i;
	if (s->packet == TCP_ESTABLISHED && __unlikely((s->flags & (SOCK_FIN_ACKED | SOCK_SHUTDOWN_READ)) != (SOCK_FIN_ACKED | SOCK_SHUTDOWN_READ))) {
		TCP_SEND_SOCKET_RESET(s);
	}
	for (i = 0; i < s->n_groups; i++) IGMP_RELEASE_MCAST_RECORD(&s->groups[i]);
	while (__unlikely(!LIST_EMPTY(&s->backlog_preparing))) {
		TCPIP_SOCKET *ns = LIST_STRUCT(s->backlog_preparing.next, TCPIP_SOCKET, backlog_list);
		TCPIP_DESTROY_SOCKET((SOCKET *)ns);
	}
	while (__unlikely(!LIST_EMPTY(&s->backlog_connected))) {
		TCPIP_SOCKET *ns = LIST_STRUCT(s->backlog_connected.next, TCPIP_SOCKET, backlog_list);
		TCPIP_DESTROY_SOCKET((SOCKET *)ns);
	}
	TCP_RESET_SOCKET(s);
	if (__unlikely(!LIST_EMPTY(&s->in_queue))) DELPKTQ(s, &s->in_queue, 0);
	DEL_FROM_LIST(&s->hash);
	NET$DESTROY_SOCKET((SOCKET *)s);
#undef s
}

void TCP_RESET_SOCKET(TCPIP_SOCKET *s)
{
	s->packet = TCP_NULL_PACKET;
	s->flags |= SOCK_SHUTDOWN_WRITE;
	WQ_WAKE_ALL(&s->read_wait);
	WQ_WAKE_ALL(&s->write_wait);
	WQ_WAKE_ALL(&s->linger_wait);
	if (__unlikely(s->timer.fn != NULL)) {
		KERNEL$DEL_TIMER(&s->timer);
		s->timer.fn = NULL;
	}
	if (__unlikely(s->linger_timer.fn != NULL)) {
		KERNEL$DEL_TIMER(&s->linger_timer);
		s->linger_timer.fn = NULL;
	}
	if (__unlikely(s->listener != NULL)) {
		DEL_FROM_LIST(&s->backlog_list);
		s->listener = NULL;
	}
	if (__unlikely(s->prepared_packet != NULL)) {
		NET$QFREE((SOCKET *)s, pkt_allocated(s->prepared_packet));
		FREE_PACKET(s->prepared_packet, &NET$PKTPOOL, SPL_NET);
		s->prepared_packet = NULL;
	}
	if (__unlikely(!LIST_EMPTY(&s->sent_queue))) DELPKTQ(s, &s->sent_queue, 1);
	if (__unlikely(!LIST_EMPTY(&s->out_queue))) DELPKTQ(s, &s->out_queue, 0);
	if (__unlikely(!LIST_EMPTY(&s->in_ooo_queue))) DELPKTQ(s, &s->in_ooo_queue, 0);
		/* let read() get packets from in_queue */
}

int TCPIP_CLOSE_SOCKET(SOCKET *s_, IORQ *rq)
{
#define s ((TCPIP_SOCKET *)s_)
	if (/*implied ... __unlikely(s->op == &UDP_OPS) ||*/ __unlikely(s->packet != TCP_ESTABLISHED)) {
		TCPIP_DESTROY_SOCKET(s_);
		return 0;
	}
	return TCP_CLOSE(s, SHUT_RDWR, rq);
#undef s
}

void TCP_END_SOCKET(TCPIP_SOCKET *s)
{
	if (s->open_count > 0) {
		TCP_RESET_SOCKET(s);
		return;
	}
	TCPIP_DESTROY_SOCKET((SOCKET *)s);
}

static __finline__ void SYS_BIND(TCPIP_SOCKET *s, struct sockaddr_in *sa, int u, IOCTLRQ *RQ)
{
	int r;
	unsigned p;
	if (__unlikely(sa->sin_family != AF_INET)) {
		RQ->status = -EAFNOSUPPORT;
		return;
	}
	if (__unlikely(s->remote_addr != htonl(INADDR_ANY))) {
		RQ->status = -EISCONN;
		return;
	}
	if (__unlikely(s->local_addr != htonl(INADDR_ANY)) || __unlikely(s->local_port != htons(IPPORT_ANY))) {
		RQ->status = -EINVAL;
		return;
	}
	if (__unlikely(sa->sin_addr.s_addr != htonl(INADDR_ANY))) {
		if (__unlikely(!TCPIP_BINDABLE_ADDRESS(sa->sin_addr.s_addr))) {
			if (u && __likely(TCPIP_MULTICAST_ADDRESS(sa->sin_addr.s_addr))) {
				/* allow binding to a multicast address */
			} else {
				RQ->status = -EADDRNOTAVAIL;
				return;
			}
		}
	}
	if (__likely((p = sa->sin_port) != htons(IPPORT_ANY))) {
		if (__unlikely((r = NETACL_SEARCH(s->node, htonl(INADDR_ANY), ntohs(p) | (u * NETACL_PORT_UDP) | NETACL_PORT_IN)) < 0)) {
			RQ->status = -EACCES;
			return;
		}
		if (__likely(r & NETACL_ALLADDR))
			s->flags |= SOCK_NETACL_ACCEPT_IN;
		if (__likely(sa->sin_addr.s_addr == htonl(INADDR_ANY))) {
			if (__unlikely(!XLIST_EMPTY(&local_ports[u * IPPORT_MAX + p]))) {
				eaddrinuse:
				s->flags &= ~SOCK_NETACL_ACCEPT_IN;
				RQ->status = -EADDRINUSE;
				return;
			}
		} else {
			TCPIP_SOCKET *ss;
			XLIST_FOR_EACH_UNLIKELY(ss, &local_ports[u * IPPORT_MAX + p], TCPIP_SOCKET, hash)
				if (__unlikely(ss->local_addr == sa->sin_addr.s_addr)) goto eaddrinuse;
		}
		set_port:
		s->local_port = p;
		DEL_FROM_LIST(&s->hash);
		ADD_TO_XLIST(&local_ports[u * IPPORT_MAX + p], &s->hash);
	} else {
		p = FIND_LOCAL_PORT_BIND(u * FLP_UDP);
		if (__unlikely(p == htons(IPPORT_ANY))) {
			RQ->status = -EAGAIN;
			return;
		}
		if (__unlikely((r = NETACL_SEARCH(s->node, htonl(INADDR_ANY), ntohs(p) | (u * NETACL_PORT_UDP) | NETACL_PORT_IN)) < 0)) {
			RQ->status = -EACCES;
			return;
		}
		if (__likely(r & NETACL_ALLADDR))
			s->flags |= SOCK_NETACL_ACCEPT_IN;
		goto set_port;
	}
	s->local_addr = sa->sin_addr.s_addr;
	RQ->status = 0;
	return;
}

__finline__ int CONNECT_BIND(TCPIP_SOCKET *s, int u)
{
	unsigned port;
	if (__likely(s->local_port != htons(IPPORT_ANY))) return 0;
	port = FIND_LOCAL_PORT_BIND(u * FLP_UDP);
	if (__unlikely(port == htons(IPPORT_ANY))) {
		return -EAGAIN;
	}
	s->local_port = port;
	DEL_FROM_LIST(&s->hash);
	ADD_TO_XLIST(&local_ports[u * IPPORT_MAX + port], &s->hash);
	return 0;
}

static int IP_SET_OPTIONS(IOCTLRQ *rq)
{
	TCPIP_SOCKET *s = rq->handle->fnode;
	switch (rq->param) {
		case __SO_MAKEPARAM(SOL_IP, IP_TOS):
		case __SO_MAKEPARAM(SOL_IP, IP_MULTICAST_TTL):
		case __SO_MAKEPARAM(SOL_TCP, TCP_NODELAY):
		case __SO_MAKEPARAM(SOL_TCP, TCP_CORK): {
			int r;
			int v;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &v, sizeof v))) {
				if (r == 1) {
					DO_PAGEIN_NORET(rq, &rq->v, PF_READ);
					return 1;
				}
				rq->status = r;
				return 0;
			}
			switch (rq->param) {
				case __SO_MAKEPARAM(SOL_IP, IP_TOS):
					if (__unlikely((unsigned)v > 0xff) ||
					    !TOS_VALID(v)) {
						einval:
						rq->status = -EINVAL;
						return 0;
					}
					s->ip_tos = (s->ip_tos & (IPTOS_CE | IPTOS_ECT)) | v;
					s->flags |= SOCK_TOS_SET;
					break;
				case __SO_MAKEPARAM(SOL_IP, IP_MULTICAST_TTL):
					if (__unlikely(s->op == &TCP_OPS)) goto noprotoopt;
					if (__unlikely((unsigned)v > 0xff)) goto einval;
					s->mcast_ttl = v;
					break;
				case __SO_MAKEPARAM(SOL_TCP, TCP_NODELAY):
					if (v) {
						s->flags = (s->flags & ~SOCK_CORK) | SOCK_NONAGLE;
						if (s->packet == TCP_ESTABLISHED) TCP_SEND_MORE(s);
					}
					else s->flags &= ~SOCK_NONAGLE;
					break;
				case __SO_MAKEPARAM(SOL_TCP, TCP_CORK):
					if (v) s->flags = (s->flags & ~SOCK_NONAGLE) | SOCK_CORK;
					else {
						s->flags &= ~SOCK_CORK;
						if (s->packet == TCP_ESTABLISHED) TCP_SEND_MORE(s);
					}
					break;
			}
			rq->status = 0;
			return 0;
		}
		case __SO_MAKEPARAM(SOL_IP, IP_ADD_MEMBERSHIP):
		case __SO_MAKEPARAM(SOL_IP, IP_DROP_MEMBERSHIP): {
			unsigned i;
			int r;
			struct ip_mreq mreq;
			if (__unlikely(s->op == &TCP_OPS)) goto noprotoopt;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &mreq, sizeof mreq))) {
				if (r == 1) {
					DO_PAGEIN_NORET(rq, &rq->v, PF_READ);
					return 1;
				}
				rq->status = r;
				return 0;
			}
			if (__unlikely(!TCPIP_MULTICAST_ADDRESS(mreq.imr_multiaddr.s_addr))) {
				rq->status = -EINVAL;
				return 0;
			}
			if (__unlikely(mreq.imr_interface.s_addr != htonl(INADDR_ANY)) && __unlikely(!TCPIP_BINDABLE_ADDRESS(mreq.imr_interface.s_addr))) {
				rq->status = -EADDRNOTAVAIL;
				return 0;
			}
			for (i = 0; i < s->n_groups; i++) {
				if (s->groups[i].imr_multiaddr.s_addr == mreq.imr_multiaddr.s_addr && s->groups[i].imr_interface.s_addr == mreq.imr_interface.s_addr) {
					if (__unlikely(rq->param == __SO_MAKEPARAM(SOL_IP, IP_ADD_MEMBERSHIP))) {
						rq->status = -EALREADY;
						return 0;
					}
					IGMP_RELEASE_MCAST_RECORD(&mreq);
					memmove(&s->groups[i], &s->groups[i + 1], (s->n_groups - i - 1) * sizeof(*s->groups));
					s->n_groups--;
					rq->status = 0;
					return 0;
				}
			}
			if (__likely(rq->param != __SO_MAKEPARAM(SOL_IP, IP_ADD_MEMBERSHIP))) {
				rq->status = -EADDRNOTAVAIL;
				return 0;
			}
			if (__unlikely(s->n_groups >= SOCKET_MCAST_GROUPS)) {
				rq->status = -ENOBUFS;
				return 0;
			}
			if (__unlikely(IGMP_GET_MCAST_RECORD(&mreq, (IORQ *)rq))) return 1;
			memcpy(&s->groups[s->n_groups++], &mreq, sizeof mreq);
			rq->status = 0;
			return 0;
		}
		case __SO_MAKEPARAM(SOL_IP, IP_MULTICAST_IF): {
			int r;
			struct in_addr addr;
			if (__unlikely(s->op == &TCP_OPS)) goto noprotoopt;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &addr, sizeof addr))) {
				if (r == 1) {
					DO_PAGEIN_NORET(rq, &rq->v, PF_READ);
					return 1;
				}
				rq->status = r;
				return 0;
			}
			if (__likely(addr.s_addr != htonl(INADDR_ANY)) && __unlikely(!TCPIP_BINDABLE_ADDRESS(addr.s_addr))) {
				rq->status = -EADDRNOTAVAIL;
				return 0;
			}
			s->multicast_src = addr.s_addr;
			rq->status = 0;
			return 0;
		}
		default: {
			noprotoopt:
			rq->status = -ENOPROTOOPT;
			return 0;
		}
	}
}

static int IP_GET_OPTIONS(IOCTLRQ *rq)
{
	TCPIP_SOCKET *s = rq->handle->fnode;
	int r;
	int v;
	void *ret = &v;
	int retl = sizeof v;
	unsigned long iol;
	switch (rq->param) {
		case __SO_MAKEPARAM(SOL_IP, IP_TOS):
			v = s->ip_tos;
			break;
		case __SO_MAKEPARAM(SOL_IP, IP_MULTICAST_TTL):
			if (__unlikely(s->op == &TCP_OPS)) goto noprotoopt;
			v = s->mcast_ttl;
			break;
		case __SO_MAKEPARAM(SOL_TCP, TCP_NODELAY):
			v = !!(s->flags & SOCK_NONAGLE);
			break;
		case __SO_MAKEPARAM(SOL_TCP, TCP_CORK):
			v = !!(s->flags & SOCK_CORK);
			break;
		default:
			noprotoopt:
			rq->status = -ENOPROTOOPT;
			return 0;
	}
	iol = rq->v.len;
	if (__unlikely(rq->v.len > retl)) rq->v.len = retl;
	r = KERNEL$PUT_IOCTL_STRUCT(rq, ret, retl);
	rq->v.len = iol;
	if (__unlikely(r == 1)) {
		DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);
		return 1;
	}
	rq->status = __unlikely(r) ? r : retl;
	return 0;
}

static void SOCKET_STAT(struct stat *stat, unsigned buffer);

static DECL_IOCALL(TCP_IOCTL, SPL_NET, IOCTLRQ)
{
	int r;
	unsigned ioctl;
	HANDLE *h = RQ->handle;
	TCPIP_SOCKET *s;
	struct sockaddr_in sa;
	if (__unlikely(h->op != &TCP_OPS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_NET));
	s = h->fnode;
	ioctl = RQ->ioctl;
	if (__likely(ioctl == IOCTL_SELECT_READ)) {
		if (__likely(!s->sock_error) && __likely(!(s->flags & SOCK_SHUTDOWN_READ)) && (__likely(s->packet != TCP_LISTEN) ? (tcp_seq)(s->ack - s->read_seq) < (__unlikely(s->sock_rcvlowat != 0) ? s->sock_rcvlowat : 1) : LIST_EMPTY(&s->backlog_connected))) {
			if (RQ->param) {
				WQ_WAIT_F(&s->read_wait, RQ);
				RETURN;
			}
			RQ->status = -EWOULDBLOCK;
		} else RQ->status = 0;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_SELECT_WRITE)) {
		if (__likely(!s->sock_error) && __likely(!(s->flags & SOCK_SHUTDOWN_WRITE)) && (__unlikely(s->packet != TCP_ESTABLISHED) || (__likely(s->sock_sndlowat <= 1) ? !s->prepared_packet && TCP_CANT_WRITE(s) : s->sent_queue_length + s->out_queue_length + s->sock_sndlowat > TCP_WRITE_BUFFER(s)))) {
			if (RQ->param) {
				WQ_WAIT_F(&s->write_wait, RQ);
				RETURN;
			}
			RQ->status = -EWOULDBLOCK;
		} else RQ->status = 0;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_POLL)) {
		if (__unlikely(s->sock_error)) r = POLLERR;
		else {
			r = 0;
			if (s->flags & SOCK_SHUTDOWN_READ && s->ack == s->read_seq) r = POLLHUP;
			else if (__likely(s->packet != TCP_LISTEN) ? (tcp_seq)(s->ack - s->read_seq) >= (__unlikely(s->sock_rcvlowat != 0) ? s->sock_rcvlowat : 1) : !LIST_EMPTY(&s->backlog_connected)) r = POLLIN;
			if (__unlikely(s->flags & SOCK_SHUTDOWN_WRITE)) r |= POLLERR;
			else if (s->packet == TCP_ESTABLISHED && (__likely(s->sock_sndlowat <= 1) ? s->prepared_packet || !TCP_CANT_WRITE(s) : s->sent_queue_length + s->out_queue_length + s->sock_sndlowat < TCP_WRITE_BUFFER(s))) r |= POLLOUT;
		}
		RQ->status = r;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_CONNECT)) {
		if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &sa, sizeof sa))) {
			if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		if (__unlikely(sa.sin_family != AF_INET)) {
			RQ->status = -EAFNOSUPPORT;
			RETURN_AST(RQ);
		}
		if (__unlikely(INAPPROPRIATE_TCP_ADDRESS(sa.sin_addr.s_addr)) || __unlikely(sa.sin_port == htons(IPPORT_ANY))) {
			RQ->status = -EINVAL;
			RETURN_AST(RQ);
		}
		if (__unlikely(s->remote_addr != htonl(INADDR_ANY))) {
			if (__unlikely(s->sock_error)) {
				RQ->status = s->sock_error;
			} else if (__unlikely((s->remote_addr ^ sa.sin_addr.s_addr) | (s->remote_port ^ sa.sin_port))) {
				RQ->status = -EISCONN;
			} else {
				if (s->packet == TCP_SYN_SENT) {
					if (h->flags & SOCKET_NONBLOCK) {
						RQ->status = -EINPROGRESS;
						RETURN_AST(RQ);
					} else {
						WQ_WAIT_F(&s->write_wait, RQ);
						RETURN;
					}
				}
				RQ->status = 0;
			}
			RETURN_AST(RQ);
		}
		if (__unlikely((r = NETACL_SEARCH(s->node, sa.sin_addr.s_addr, ntohs(sa.sin_port))) < 0)) {
			RQ->status = -EACCES;
			RETURN_AST(RQ);
		}
		if (__likely(s->local_addr == htonl(INADDR_ANY))) {
			if (__unlikely((s->local_addr = IP_FIND_LOCAL_ADDRESS(sa.sin_addr.s_addr)) == htonl(INADDR_ANY))) {
				RQ->status = -ENETDOWN;
				RETURN_AST(RQ);
			}
		}
		if (__likely(s->local_port == htons(IPPORT_ANY))) {
			if (__unlikely((s->local_port = FIND_LOCAL_PORT_CONNECT(0, s->local_addr, sa.sin_addr.s_addr, sa.sin_port)) == htons(IPPORT_ANY))) {
				RQ->status = -EAGAIN;
				RETURN_AST(RQ);
			}
		}
		if (__unlikely(LOOKUP_SOCKET(s->local_addr, s->local_port, sa.sin_addr.s_addr, sa.sin_port, LS_TCP) != NULL)) {
			RQ->status = -EEXIST;
			RETURN_AST(RQ);
		}
		s->remote_addr = sa.sin_addr.s_addr;
		s->remote_port = sa.sin_port;
		INSERT_INTO_HASH(s);
		INIT_TCP_SOCKET(s);
		s->flags |= SOCK_NETACL_ACCEPT_IN;
		if (h->flags & SOCKET_NONBLOCK) {
			TCP_SEND_SYN(s);
			RQ->status = __unlikely(s->sock_error) ? s->sock_error : -EINPROGRESS;
			RETURN_AST(RQ);
		} else {
			WQ_WAIT_F(&s->write_wait, RQ);
			TCP_SEND_SYN(s);
			RETURN;
		}
	} else if (__likely(ioctl == IOCTL_ACCEPT)) {
		TCPIP_SOCKET *ns;
		if (__unlikely(s->packet != TCP_LISTEN)) {
			RQ->status = -EINVAL;
			RETURN_AST(RQ);
		}
		if (__unlikely(LIST_EMPTY(&s->backlog_connected))) {
			if (h->flags & SOCKET_NONBLOCK) {
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			WQ_WAIT_F(&s->read_wait, RQ);
			RETURN;
		}
		ns = LIST_STRUCT(s->backlog_connected.next, TCPIP_SOCKET, backlog_list);
		if (sizeof sa <= RQ->v.len) {
			unsigned long rl;
			memset(&sa, 0, sizeof sa);
			sa.sin_family = AF_INET;
			sa.sin_len = sizeof sa;
			sa.sin_addr.s_addr = ns->remote_addr;
			sa.sin_port = ns->remote_port;
			rl = RQ->v.len;
			RQ->v.len = sizeof sa;
			r = KERNEL$PUT_IOCTL_STRUCT(RQ, &sa, sizeof sa);
			RQ->v.len = rl;
			if (__unlikely(r)) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
				RQ->status = r;
				RETURN_AST(RQ);
			}
		}
		DEL_FROM_LIST(&ns->backlog_list);
		ns->listener = NULL;
		RQ->status = ns->n;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_BIND)) {
		if (!RQ->v.len) memset(&sa, 0, sizeof sa), sa.sin_family = AF_INET;
		else if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &sa, sizeof sa))) {
			if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		if (__unlikely(s->packet != TCP_NULL_PACKET_1)) {
			RQ->status = -EISCONN;
			RETURN_AST(RQ);
		}
		SYS_BIND(s, &sa, 0, RQ);
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_LISTEN)) {
		if (__unlikely(s->packet != TCP_NULL_PACKET_1)) {
			RQ->status = -EISCONN;
			RETURN_AST(RQ);
		}
		if (__unlikely(r = CONNECT_BIND(s, 0))) {
			RQ->status = r;
			RETURN_AST(RQ);
		}
		s->packet = TCP_LISTEN;
		s->sock_flags |= 1 << SO_ACCEPTCONN;
		s->flags |= SOCK_NETACL_TCP_ACCEPT;
		RQ->status = 0;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_GETPEERNAME) || __likely(ioctl == IOCTL_GETSOCKNAME)) {
		memset(&sa, 0, sizeof sa);
		sa.sin_family = AF_INET;
		sa.sin_len = sizeof sa;
		if (__likely(ioctl == IOCTL_GETPEERNAME)) {
			if (__unlikely(s->packet != TCP_ESTABLISHED) || __unlikely((s->flags & (SOCK_SHUTDOWN_READ | SOCK_SHUTDOWN_WRITE)) == (SOCK_SHUTDOWN_READ | SOCK_SHUTDOWN_WRITE))) {
				RQ->status = -ENOTCONN;
				RETURN_AST(RQ);
			}
			sa.sin_addr.s_addr = s->remote_addr;
			sa.sin_port = s->remote_port;
		} else {
			sa.sin_addr.s_addr = s->local_addr;
			sa.sin_port = s->local_port;
		}
		if (__unlikely(sizeof sa < RQ->v.len)) RQ->v.len = sizeof sa;
		if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(RQ, &sa, RQ->v.len))) {
			if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		RQ->status = sizeof sa;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_SHUTDOWN)) {
		if (__unlikely(RQ->param > SHUT_RDWR)) {
			RQ->status = -EINVAL;
			RETURN_AST(RQ);
		}
		if (__unlikely(s->packet != TCP_ESTABLISHED)) {
			if (__unlikely(s->remote_addr == htonl(INADDR_ANY)) || __unlikely(s->packet != TCP_NULL_PACKET)) {
				RQ->status = -ENOTCONN;
				RETURN_AST(RQ);
			}
		}
		r = TCP_CLOSE(s, RQ->param, (IORQ *)RQ);
		if (__likely(r <= 0)) {
			RQ->status = r;
			RETURN_AST(RQ);
		}
		RETURN;
	} else if (__likely(ioctl == IOCTL_SETSOCKOPT)) {
		if (__likely(__SO_LEVEL(RQ->param) == SOL_SOCKET)) goto generic;
		if (__SO_LEVEL(RQ->param) == SOL_IP || __likely(__SO_LEVEL(RQ->param) == SOL_TCP)) {
			if (__likely(!IP_SET_OPTIONS(RQ))) RETURN_AST(RQ);
			RETURN;
		}
		RQ->status = -EINVAL;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_GETSOCKOPT)) {
		if (__SO_LEVEL(RQ->param) == SOL_SOCKET) goto generic;
		if (__SO_LEVEL(RQ->param) == SOL_IP || __likely(__SO_LEVEL(RQ->param) == SOL_TCP)) {
			if (__likely(!IP_GET_OPTIONS(RQ))) RETURN_AST(RQ);
			RETURN;
		}
		RQ->status = -ENOPROTOOPT;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_NREAD)) {
		RQ->status = (tcp_seq)(s->ack - s->read_seq);
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_STAT)) {
		static struct stat stat;
		SOCKET_STAT(&stat, TCP_WRITE_BUFFER(s));
		if (__likely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &stat, sizeof stat)) <= 0)) {
			RQ->status = r;
			RETURN_AST(RQ);
		}
		DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	} else {
		generic:
		if (__likely(!NET$SOCKET_IOCTL(RQ))) RETURN_AST(RQ);
		RETURN;
	}
}

static DECL_IOCALL(UDP_IOCTL, SPL_NET, IOCTLRQ)
{
	int r;
	unsigned ioctl;
	HANDLE *h = RQ->handle;
	TCPIP_SOCKET *s;
	struct sockaddr_in sa;
	if (__unlikely(h->op != &UDP_OPS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_NET));
	s = h->fnode;
	ioctl = RQ->ioctl;
	if (__likely(ioctl == IOCTL_SELECT_READ)) {
		if (LIST_EMPTY(&s->in_queue) && __likely(!s->sock_error)) {
			if (RQ->param) {
				WQ_WAIT_F(&s->read_wait, RQ);
				RETURN;
			}
			RQ->status = -EWOULDBLOCK;
		} else RQ->status = 0;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_POLL)) {
		r = POLLOUT;
		if (__unlikely(s->sock_error)) r = POLLOUT | POLLERR;
		if (__likely(!LIST_EMPTY(&s->in_queue))) r |= POLLIN;
		RQ->status = r;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_BIND)) {
		if (!RQ->v.len) memset(&sa, 0, sizeof sa), sa.sin_family = AF_INET;
		else if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &sa, sizeof sa))) {
			if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		SYS_BIND(s, &sa, 1, RQ);
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_CONNECT)) {
		if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &sa, sizeof sa))) {
			if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		if (__unlikely(sa.sin_family != AF_INET)) {
			if (__likely(sa.sin_family == AF_UNSPEC)) {
				s->local_addr = htonl(INADDR_ANY);
				s->local_port = htons(IPPORT_ANY);
				s->remote_addr = htonl(INADDR_ANY);
				s->remote_port = htons(IPPORT_ANY);
				DEL_FROM_LIST(&s->hash);
				VOID_LIST_ENTRY(&s->hash);
				s->flags &= ~(SOCK_NETACL_ACCEPT_IN | SOCK_NETACL_ACCEPT_OUT);
				goto ret0;
			}
			RQ->status = -EAFNOSUPPORT;
			RETURN_AST(RQ);
		}
		if (__unlikely(INAPPROPRIATE_UDP_ADDRESS(sa.sin_addr.s_addr)) || __unlikely(sa.sin_port == htons(IPPORT_ANY))) {
			RQ->status = -EINVAL;
			RETURN_AST(RQ);
		}

		if (__unlikely(s->remote_addr != htonl(INADDR_ANY))) {
			if (__likely(!((sa.sin_addr.s_addr ^ s->remote_addr) | (sa.sin_port ^ s->remote_port)))) goto ret0;
			RQ->status = -EISCONN;
			RETURN_AST(RQ);
		}
		if (__unlikely((r = NETACL_SEARCH(s->node, sa.sin_addr.s_addr, ntohs(sa.sin_port) | NETACL_PORT_UDP)) < 0)) {
			RQ->status = -EACCES;
			RETURN_AST(RQ);
		}
		if (__likely(s->local_addr == htonl(INADDR_ANY))) {
			if (__unlikely((s->local_addr = IP_FIND_LOCAL_ADDRESS(sa.sin_addr.s_addr)) == htonl(INADDR_ANY))) {
				RQ->status = -ENETDOWN;
				RETURN_AST(RQ);
			}
		}
		if (__likely(s->local_port == htons(IPPORT_ANY))) {
			if (__unlikely((s->local_port = FIND_LOCAL_PORT_CONNECT(FLP_UDP, s->local_addr, sa.sin_addr.s_addr, sa.sin_port)) == htons(IPPORT_ANY))) {
				RQ->status = -EAGAIN;
				RETURN_AST(RQ);
			}
		}
		if (__unlikely(LOOKUP_SOCKET(s->local_addr, s->local_port, sa.sin_addr.s_addr, sa.sin_port, LS_UDP) != NULL)) {
			RQ->status = -EEXIST;
			RETURN_AST(RQ);
		}
		s->remote_addr = sa.sin_addr.s_addr;
		s->remote_port = sa.sin_port;
		INSERT_INTO_HASH(s);
		s->flags |= SOCK_NETACL_ACCEPT_OUT;
		ret0:
		RQ->status = 0;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_GETPEERNAME) || __likely(ioctl == IOCTL_GETSOCKNAME)) {
		memset(&sa, 0, sizeof sa);
		sa.sin_family = AF_INET;
		sa.sin_len = sizeof sa;
		if (__likely(ioctl == IOCTL_GETPEERNAME)) {
			if (__unlikely(s->remote_addr == htonl(INADDR_ANY))) {
				RQ->status = -ENOTCONN;
				RETURN_AST(RQ);
			}
			sa.sin_addr.s_addr = s->remote_addr;
			sa.sin_port = s->remote_port;
		} else {
			sa.sin_addr.s_addr = s->local_addr;
			sa.sin_port = s->local_port;
		}
		if (__unlikely(sizeof sa < RQ->v.len)) RQ->v.len = sizeof sa;
		if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(RQ, &sa, RQ->v.len))) {
			if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		RQ->status = sizeof sa;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_SELECT_WRITE)) {
		RQ->status = 0;
		RETURN_AST(RQ);
	} else if (__unlikely(ioctl == IOCTL_ACCEPT) || __unlikely(ioctl == IOCTL_SHUTDOWN) || __unlikely(ioctl == IOCTL_LISTEN)) {
		RQ->status = -EOPNOTSUPP;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_SETSOCKOPT)) {
		if (__SO_LEVEL(RQ->param) == SOL_SOCKET) goto generic;
		if (__SO_LEVEL(RQ->param) == SOL_IP) {
			if (__likely(!IP_SET_OPTIONS(RQ))) RETURN_AST(RQ);
			RETURN;
		}
		RQ->status = -ENOPROTOOPT;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_GETSOCKOPT)) {
		if (__likely(__SO_LEVEL(RQ->param) == SOL_SOCKET)) goto generic;
		if (__SO_LEVEL(RQ->param) == SOL_IP) {
			if (__likely(!IP_GET_OPTIONS(RQ))) RETURN_AST(RQ);
			RETURN;
		}
		RQ->status = -ENOPROTOOPT;
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_NREAD)) {
		if (LIST_EMPTY(&s->in_queue)) RQ->status = 0;
		else {
			PACKET *p = LIST_STRUCT(s->in_queue.next, PACKET, list);
			RQ->status = p->data_length - sizeof(struct ip) - sizeof(struct udphdr);
		}
		RETURN_AST(RQ);
	} else if (__likely(ioctl == IOCTL_STAT)) {
		static struct stat stat;
		SOCKET_STAT(&stat, NET$MAX_PACKET_LENGTH - sizeof(struct ip) - sizeof(struct udphdr));
		if (__likely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &stat, sizeof stat)) <= 0)) {
			RQ->status = r;
			RETURN_AST(RQ);
		}
		DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	} else {
		generic:
		if (__likely(!NET$SOCKET_IOCTL(RQ))) RETURN_AST(RQ);
		RETURN;
	}
}

int TCPIP_LINGER(SOCKET *s_, IORQ *rq)
{
	TCPIP_SOCKET *s = (TCPIP_SOCKET *)s_;
	/* implied below ... if (__unlikely(s->op == &UDP_OPS)) return 0; */
	if (__unlikely(s->packet != TCP_ESTABLISHED)) return 0;
	if (__unlikely(s->open_count > 0)) return 0;

	/* Note: if the other party is holding closed window and resonding,
	   TCP timeout never triggers. So we need reduced lingertime as well */
	s->sock_lingertime = TCP_SHUTDOWN_TIMEOUT;

	s->tcp_write_timeout = TCP_SHUTDOWN_TIMEOUT;
	if (__likely(s->timer.fn != NULL)) {
		KERNEL$DEL_TIMER(&s->timer);
		KERNEL$SET_TIMER(JIFFIES_PER_SECOND, &s->timer);
	}
	return TCP_CLOSE(s, SHUT_RDWR, rq);
}

int TCPIP_IOCTL(IOCTLRQ *rq)
{
	unsigned len;
	unsigned long iol;
	int r;
	switch (rq->ioctl) {
		case IOCTL_DHCP_OPTION:
			if (__unlikely(rq->param >= 256)) return -ERANGE;
			if (__unlikely(!(len = DHCP_APPARRAY[rq->param][0]))) return -ENOPROTOOPT;
			iol = rq->v.len;
			if (rq->v.len > len) rq->v.len = len;
			r = KERNEL$PUT_IOCTL_STRUCT(rq, DHCP_APPARRAY[rq->param] + 1, rq->v.len);
			if (__likely(!r)) {
				rq->status = rq->v.len;
				rq->v.len = iol;
				CALL_AST(rq);
				return 1;
			}
			rq->v.len = iol;
			if (__likely(r == 1)) {
				DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);
				return 1;
			}
			return r;
		default:
			return -ENOOP;
	}
}

static void SOCKET_STAT(struct stat *stat, unsigned buffer)
{
	memset(stat, 0, sizeof(struct stat));
	stat->st_mode = 0600 | S_IFSOCK;
	stat->st_nlink = 1;
	stat->st_blksize = 1 << __BSR(buffer);
	stat->st_atime = stat->st_mtime = stat->st_ctime = time(NULL);
}

int TCPIP_SOCKETS_INIT(void)
{
	int i;

	__u64 mem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_MAPPED);
	unsigned hash_size = mem / SOCKET_HASH_DIV;
	if (hash_size < __PAGE_CLUSTER_SIZE) hash_size = __PAGE_CLUSTER_SIZE;
	if (hash_size > PG_SIZE * PG_BANK) hash_size = PG_SIZE * PG_BANK;
	hash_size /= sizeof(XLIST_HEAD *);
	hash_size = 1 << __BSR(hash_size);
	socket_hash_mask = hash_size - 1;

	socket_hash = KERNEL$ALLOC_CONTIG_AREA((socket_hash_mask + 1) * sizeof(XLIST_HEAD *), AREA_DATA);
	if (__IS_ERR(socket_hash)) return __PTR_ERR(socket_hash);

	for (i = 0; i < IPPORT_MAX * 2; i++) {
		INIT_XLIST(&local_ports[i]);
		KERNEL$THREAD_MAY_BLOCK();
	}
	for (i = 0; i <= socket_hash_mask; i++) {
		INIT_XLIST(&socket_hash[i]);
		KERNEL$THREAD_MAY_BLOCK();
	}
	return 0;
}

void TCPIP_SOCKETS_DONE(void)
{
	unsigned i;
	for (i = 0; i < IPPORT_MAX * 2; i++) {
		if (__unlikely(!XLIST_EMPTY(&local_ports[i])))
			KERNEL$SUICIDE("TCPIP_SOCKETS_DONE: %s SOCKET LEAKED AT LOCAL PORT %d", i & IPPORT_MAX ? "UDP" : "TCP", i & (IPPORT_MAX - 1));
		KERNEL$THREAD_MAY_BLOCK();
	}
	for (i = 0; i <= socket_hash_mask; i++) {
		if (__unlikely(!XLIST_EMPTY(&socket_hash[i]))) {
			TCPIP_SOCKET *s = LIST_STRUCT(socket_hash[i].next, TCPIP_SOCKET, hash);
			KERNEL$SUICIDE("TCPIP_SOCKETS_DONE: SOCKET IN HASH LEAKED: LOCAL %08X:%04X, REMOTE %08X:%04X", (unsigned)ntohl(s->local_addr), (unsigned)ntohs(s->local_port), (unsigned)ntohl(s->remote_addr), (unsigned)ntohs(s->remote_port));
		}
		KERNEL$THREAD_MAY_BLOCK();
	}
	KERNEL$FREE_CONTIG_AREA(socket_hash, (socket_hash_mask + 1) * sizeof(XLIST_HEAD *));
}

static void CHECK_SOCKET_LIST(XLIST_HEAD *l)
{
	TCPIP_SOCKET *s, *ss;
	XLIST_FOR_EACH(s, l, TCPIP_SOCKET, hash) {
		if (s->multicast_src != htonl(INADDR_ANY) && !TCPIP_BINDABLE_ADDRESS(s->multicast_src)) s->multicast_src = htonl(INADDR_ANY);
		if (__likely(TCPIP_BINDABLE_ADDRESS(s->local_addr) != NULL)) continue;
		if (__likely(s->local_addr == ntohl(INADDR_ANY)) || (__likely(TCPIP_MULTICAST_ADDRESS(s->local_addr)) && __likely(s->op != &TCP_OPS))) continue;
		ss = LIST_STRUCT(s->hash.prev, TCPIP_SOCKET, hash);
		TCPIP_DESTROY_SOCKET((SOCKET *)s);
		s = ss;
	}
}

void TCPIP_SOCKETS_CHECK(void)
{
	unsigned i;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_NET), spl))) KERNEL$SUICIDE("TCPIP_SOCKETS_CHECK AT SPL %08X", spl);
	RAISE_SPL(SPL_NET);
	for (i = 0; i < IPPORT_MAX * 2; i++) CHECK_SOCKET_LIST(&local_ports[i]);
	for (i = 0; i <= socket_hash_mask; i++) CHECK_SOCKET_LIST(&socket_hash[i]);
	LOWER_SPLX(spl);
}

int TCPIP_SOCKETS_OUSTANDING_PACKETS(int h)
{
	PACKET *p;
	unsigned i;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_NET), spl))) KERNEL$SUICIDE("TCPIP_SOCKETS_OUSTANDING_PACKETS AT SPL %08X", spl);
	RAISE_SPL(SPL_NET);
	for (i = 0; i <= socket_hash_mask; i++) {
		TCPIP_SOCKET *s;
		XLIST_FOR_EACH(s, &socket_hash[i], TCPIP_SOCKET, hash)
			LIST_FOR_EACH(p, &s->sent_queue, PACKET, list)
				if (__unlikely(p->flags & PKT_OUTSTANDING) && p->h == h) {
					ret_1:
					LOWER_SPLX(spl);
					return 1;
				}
	}
	LIST_FOR_EACH_UNLIKELY(p, &LOST_PACKETS, PACKET, list) if (p->h == h) goto ret_1;
	LOWER_SPLX(spl);
	return 0;
}

