#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>

#include <SPAD/PIO.H>
#include "PIO.H"

#define BUFFER_SIZE	3072
#define MAX_POLL_TIME	(JIFFIES_PER_SECOND * 3)
#define ERROR_TIMEOUT	(JIFFIES_PER_SECOND * 10)

#define FLAGS_NONBLOCK	0x01
#define FLAGS_CRLF	0x02

struct lp {
	struct pio_client client;
	void *pio;
	THREAD_RQ thread;
	SIORQ *rq;
	unsigned char h_flags;
	char was_cr;
	WQ wq;
	unsigned char buffer[BUFFER_SIZE];
};

const int LP_SIZE = sizeof(struct lp);

static void *LP_LOOKUP(HANDLE *h, char *str, int open_flags);
static IO_STUB LP_WRITE;
static AST_STUB LP_WROTE;
static long LP_THREAD(void *p);

static const HANDLE_OPERATIONS lp_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	LP_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	LP_WRITE,		/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	KERNEL$NO_OPERATION,	/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static void *LP_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	if (__likely(!_strcasecmp(str, "^NONBLOCK"))) {
		h->flags |= FLAGS_NONBLOCK;
	} else if (__likely(!_strcasecmp(str, "^CRLF"))) {
		h->flags |= FLAGS_CRLF;
	} else {
		return __ERR_PTR(-EBADMOD);
	}
	return NULL;
}

static DECL_IOCALL(LP_WRITE, SPL_DEV, SIORQ)
{
	struct lp *lp;
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &lp_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	lp = PIO_LP(h->fnode);
	if (__unlikely(lp->rq != NULL)) {
		WQ_WAIT_F(&lp->wq, RQ);
		RETURN;
	}
	lp->rq = RQ;
	lp->h_flags = h->flags;	/* handle may become unbound while we are spawning the thread or while the thread blocks, so make copy of flags */
	lp->thread.fn = LP_WROTE;
	lp->thread.thread_main = LP_THREAD;
	lp->thread.p = lp;
	lp->thread.cwd = NULL;
	lp->thread.std_in = -1;
	lp->thread.std_out = -1;
	lp->thread.std_err = -1;
	lp->thread.dlrq = NULL;
	lp->thread.thread = NULL;
	lp->thread.spawned = 0;
	RETURN_IORQ_CANCELABLE(&lp->thread, KERNEL$THREAD, RQ);
}

#include "PIOXPAND.I"

static long LP_THREAD(void *p)
{
	struct lp *lp = p;
	long r;
	VBUF v;
	int s, w;
	u_jiffies_lo_t timeout = 0;
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	RAISE_SPL(SPL_DEV);
	lp->client.size_of_pio_operations = sizeof(struct pio_operations);
	lp->client.size_of_pio_client = sizeof(struct pio_client);
	r = PIO_D_CALL(lp->pio, "PIO", PIO_CMD_ATTACH_SOMETIMES, &lp->client);
	if (__unlikely(r)) goto ret_r;
	if (__unlikely(r = lp->client.op->set_mode(&lp->client, PIO_ECR_MODE_STANDARD))) goto release_ret_r;
	lp->client.op->set_ctl(&lp->client, PIO_CTL_INIT | PIO_CTL_STROBE | PIO_CTL_AUTO_LF);
	next_buf:
	KERNEL$THREAD_MAY_BLOCK();
	v.ptr = lp->buffer;
	v.len = __likely(!(lp->h_flags & FLAGS_CRLF)) ? BUFFER_SIZE : BUFFER_SIZE / 3;
	v.spl = SPL_X(SPL_DEV);
	RAISE_SPL(SPL_VSPACE);
	s = lp->rq->v.vspace->op->vspace_get(&lp->rq->v, &v);
	if (__unlikely(!s)) {
		r = -EVSPACEFAULT;
		goto release_ret_r;
	}
	if (__likely(!(lp->h_flags & FLAGS_CRLF))) {
		w = lp->client.op->write_data_sync(&lp->client, lp->buffer, s);
		lp->was_cr = 0;
	} else {
		int sx = xpand_str(lp->buffer + (BUFFER_SIZE / 3), lp->buffer, s, lp->was_cr, NULL);
		w = lp->client.op->write_data_sync(&lp->client, lp->buffer + (BUFFER_SIZE / 3), sx);
		if (__likely(w > 0)) {
			char new_cr = (lp->buffer + (BUFFER_SIZE / 3))[w - 1] == '\r';
			w = xpand_str(lp->buffer + (BUFFER_SIZE / 3), lp->buffer, s, lp->was_cr, lp->buffer + (BUFFER_SIZE / 3) + w);
			lp->was_cr = new_cr;
		}
	}
	if (__likely(w > 0)) j = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(w < 0)) {
		lp->rq->v.ptr -= s;
		lp->rq->v.len += s;
		r = w;
		if (__likely(!(lp->h_flags & FLAGS_NONBLOCK)) && (r != -EIO || KERNEL$GET_JIFFIES_LO() - j <= TIMEOUT_JIFFIES(ERROR_TIMEOUT))) {
			r = lp->client.op->wait_sync(&lp->client, PIO_STATUS_ERROR | PIO_STATUS_SLCT, PIO_STATUS_ERROR | PIO_STATUS_SLCT | PIO_STATUS_PE | PIO_STATUS_BUSY, timeout);
			if (r >= 0) {
				timeout += timeout / 4 + 1;
				if (timeout > MAX_POLL_TIME) timeout = MAX_POLL_TIME;
				goto next_buf;
			}
		}
		goto release_ret_r;
	}
	timeout = 0;
	lp->rq->progress += w;
	if (__unlikely(w != s)) {
		lp->rq->v.ptr -= s - w;
		lp->rq->v.len += s - w;
	}
	/*__debug_printf("status: %02x\n", lp->client.get_status(&lp->client));*/
	if (lp->rq->v.len != 0) goto next_buf;
	r = lp->rq->progress;
	if (__unlikely(r < 0)) r = -EOVERFLOW;
	release_ret_r:
	PIO_D_CALL(lp->pio, "PIO", PIO_CMD_DETACH, lp);
	ret_r:
	LOWER_SPL(SPL_ZERO);
	return r;
}

static DECL_AST(LP_WROTE, SPL_DEV, THREAD_RQ)
{
	struct lp *lp = GET_STRUCT(RQ, struct lp, thread);
	SIORQ *rq = lp->rq;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), rq);
	SWITCH_PROC_ACCOUNT(rq->handle->name_addrspace, SPL_X(SPL_DEV));
	lp->rq = NULL;
	WQ_WAKE_ALL(&lp->wq);
	if (__unlikely(lp->thread.status == -EVSPACEFAULT)) {
		DO_PAGEIN(rq, &rq->v, PF_READ);
	}
	rq->status = lp->thread.status;
	RETURN_AST(rq);
}

void PIO_INIT_ROOT(HANDLE *h, void *p)
{
	struct lp *lp;
	h->fnode = p;
	h->flags = 0;
	h->flags2 = 0;
	h->op = &lp_operations;
	lp = PIO_LP(p);
	lp->rq = NULL;
	lp->pio = p;
	lp->was_cr = 0;
	WQ_INIT(&lp->wq, "PIO$WQ");
}

void LP_CANCEL(void *p)
{
	struct lp *lp;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("LP_CANCEL AT SPL %08X", KERNEL$SPL);
	lp = PIO_LP(p);
	while (__unlikely(lp->rq != NULL)) {
		KERNEL$CIO((void *)&lp->thread);
		KERNEL$SLEEP(1);
	}
}
