#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <FCNTL.H>
#include <UNISTD.H>
#include <STRING.H>
#include <VALUES.H>
#include <SYS/STAT.H>
#include <TERMIOS.H>
#include <SYS/IOCTL.H>

#define IN_BUFFER_SIZE		256
#define OUT_BUFFER_SIZE		1024
#define DEFAULT_ESCAPE		30

static char *string = NULL;
static char *string_in = NULL;
static char *string_out = NULL;
static char *string_esc = NULL;
static int noescape = 0;
static int esc = DEFAULT_ESCAPE;

static const struct __param_table params[6] = {
	"", __PARAM_STRING, 1, MAXINT,
	"IN", __PARAM_STRING, 1, MAXINT,
	"OUT", __PARAM_STRING, 1, MAXINT,
	"ESCAPE", __PARAM_STRING, 1, 3,
	"NOESCAPE", __PARAM_BOOL, ~0, 1,
	NULL, 0, 0, 0,
};

static void * const vars[6] = {
	&string,
	&string_in,
	&string_out,
	&string_esc,
	&noescape,
	NULL,
};

static char con_in_close = 0, con_out_close = 0;
static int con_in, con_out;
static int h_in, h_out;

static SIORQ in;
static SIORQ out;

static char in_buffer[IN_BUFFER_SIZE + 1]; /* escape char expansion may add up to 1 character */
static char out_buffer[OUT_BUFFER_SIZE];

static char need_cancel = 0;

static WQ_DECL(finish, "SIOCOM$FINISH");

static char error_msg[__MAX_STR_LEN];
static int error_code;

static AST_STUB did_read;
static void process_esc(char *ptr, unsigned long *len);
static void do_cancel(void);

static DECL_AST(did_write, SPL_TTY, SIORQ)
{
	if (__unlikely(RQ->status <= 0)) {
		if (!RQ->status) {
			if (!RQ->v.len) goto no_write;
			RQ->status = -EEOF;
		}
		if (RQ->status != -EINTR) {
			_snprintf(error_msg, __MAX_STR_LEN, "%s WRITE ERROR: %s", RQ == &in ? "PORT" : "CONSOLE", strerror(-RQ->status));
			error_code = RQ->status;
		}
		do_cancel();
	}
	no_write:
	if (__unlikely(need_cancel)) {
		RQ->fn = NULL;
		WQ_WAKE_ALL(&finish);
		RETURN;
	}
	if (__unlikely(RQ->v.len != 0)) {
		RQ->progress = 0;
		RETURN_IORQ(RQ, KERNEL$WRITE);
	}
	RQ->fn = did_read;
	if (__unlikely(RQ == &in)) {
		RQ->h = con_in;
		RQ->v.ptr = (unsigned long)in_buffer;
		RQ->v.len = IN_BUFFER_SIZE;
	} else {
		RQ->h = h_in;
		RQ->v.ptr = (unsigned long)out_buffer;
		RQ->v.len = OUT_BUFFER_SIZE;
	}
	RQ->v.vspace = &KERNEL$VIRTUAL;
	RQ->progress = 0;
	RETURN_IORQ(RQ, KERNEL$READ);
}

static DECL_AST(did_read, SPL_TTY, SIORQ)
{
	if (__unlikely(RQ->status <= 0)) {
		if (RQ->status != -EINTR) {
			if (RQ->status) _snprintf(error_msg, __MAX_STR_LEN, "%s READ ERROR: %s", RQ == &in ? "CONSOLE" : "PORT", strerror(-RQ->status));
			error_code = RQ->status;
		}
		do_cancel();
	}
	if (__unlikely(need_cancel)) {
		RQ->fn = NULL;
		WQ_WAKE_ALL(&finish);
		RETURN;
	}
	RQ->fn = did_write;
	RQ->v.len = RQ->status;
	if (__unlikely(RQ == &in)) {
		RQ->h = h_out;
		RQ->v.ptr = (unsigned long)in_buffer;
		process_esc(in_buffer, &RQ->v.len);
	} else {
		RQ->h = con_out;
		RQ->v.ptr = (unsigned long)out_buffer;
	}
	RQ->v.vspace = &KERNEL$VIRTUAL;
	RQ->progress = 0;
	RETURN_IORQ(RQ, KERNEL$WRITE);
}

static char *print_esc(void)
{
	static char str[7];
	if (esc < 32) _snprintf(str, sizeof str, "CTRL+%c", esc + '@');
	else str[0] = esc, str[1] = 0;
	return str;
}

static void process_esc(char *ptr, unsigned long *len)
{
	static int was_escape = 0;
	static unsigned num;
	char *end = ptr + *len;
	char *dst = ptr;
	while (ptr < end) {
		unsigned char c = *ptr++;
		process_again:
		if (__likely(!was_escape)) {
			if (__unlikely(c == esc)) {
				was_escape = -1;
			} else {
				exact:
				*dst++ = c;
			}
		} else if (was_escape == -1) {
			was_escape = 0;
			if (c == '.') {
				do_cancel();
				break;
			} else if (c == esc) {
				goto exact;
			} else if (c == '?') {
				char *str = print_esc();
				__critical_printf("ESCAPE CHARACTER IS '%s'\n%s . --- EXIT\n%s ? --- HELP\n%s A-Z@[\\]^_ --- SEND CONTROL CHARACTER\n%s 0-9 0-9 0-9 --- SEND CHARACTER WITH SPECIFIED ASCII CODE\n%s %s --- SEND THE CHARACTER '%s' ITSELF\n", str, str, str, str, str, str, str, str);
			} else if (__upcasechr(c) >= '@' && __upcasechr(c) <= '_') {
				*dst++ = __upcasechr(c) - '@';
			} else if (c >= '0' && c <= '9') {
				num = c - '0';
				was_escape = 1;
			} else {
				goto exact;
			}
		} else {
			if (c >= '0' && c <= '9') {
				num = num * 10 + c - '0';
				if (__unlikely(num >= 256)) {
					was_escape = 0;
				} else {
					was_escape++;
					if (was_escape == 3) {
						*dst++ = num;
						was_escape = 0;
					}
				}
			} else {
				*dst++ = num;
				was_escape = 0;
				goto process_again;
			}
		}
	}
	*len -= ptr - dst;
	if (__unlikely((unsigned long)*len > sizeof in_buffer))
		KERNEL$SUICIDE("process_esc: SHOT OUT OF INPUT BUFFER: %ld > %ld", *len, (long)sizeof in_buffer);
}

static void do_cancel(void)
{
	need_cancel = 1;
	KERNEL$CIO((IORQ *)(void *)&in);
	KERNEL$CIO((IORQ *)(void *)&out);
}

int main(int argc, const char * const argv[])
{
	const char * const *arg;
	int state;
	static union {
		struct stat st;
		struct termios t;
	} u;
	arg = argv;
	state = 0;
	h_in = -1;
	h_out = -1;
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		bads:
		_snprintf(error_msg, __MAX_STR_LEN, "SIOCOM: SYNTAX ERROR");
		error_code = -EBADSYN;
		goto ret;
	}
	if (__unlikely(string_esc != NULL)) {
		unsigned long n;
		if (__unlikely(noescape)) goto bads;
		if (__unlikely(!string_esc[0])) goto no_esc;
		if (__likely(string_esc[0] == '^')) {
			esc = (unsigned char)__upcasechr(string_esc[1]) - '@';
			if (__unlikely((unsigned)esc >= 32)) goto bads;
		} else if (!__get_number(string_esc, string_esc + strlen(string_esc), 0, (long *)&n)) {
			if (__unlikely(n >= 256)) goto bads;
			esc = n;
		} else {
			if (__unlikely(string_esc[1])) goto bads;
			esc = (unsigned char)string_esc[0];
		}
	}
	if (__unlikely(noescape)) no_esc: esc = -1;
	if (__likely(string != NULL)) {
		if (__unlikely(string_in != NULL) || __unlikely(string_out != NULL)) goto bads;
		h_in = open(string, O_RDWR);
		if (__unlikely(h_in < 0)) {
			_snprintf(error_msg, __MAX_STR_LEN, "SIOCOM: CAN'T OPEN DEVICE: %s", strerror(errno));
			error_code = -errno;
			goto ret;
		}
		h_out = h_in;
	} else {
		if (__unlikely(!string_in) || __unlikely(!string_out)) goto bads;
		h_in = open(string_in, O_RDONLY);
		if (__unlikely(h_in < 0)) {
			_snprintf(error_msg, __MAX_STR_LEN, "SIOCOM: CAN'T OPEN INPUT DEVICE: %s", strerror(errno));
			error_code = -errno;
			goto ret;
		}
		h_out = open(string_out, O_RDONLY);
		if (__unlikely(h_out < 0)) {
			_snprintf(error_msg, __MAX_STR_LEN, "SIOCOM: CAN'T OPEN OUTPUT DEVICE: %s", strerror(errno));
			error_code = -errno;
			goto ret;
		}
	}
	con_in = KERNEL$STDIN();
	con_out = KERNEL$STDOUT();
	if (__unlikely(fstat(con_in, &u.st)) || __likely(!S_ISFIFO(u.st.st_mode))) {
		char *p = KERNEL$HANDLE_PATH(con_in);
		con_in = open(p, O_RDONLY);
		if (__unlikely(con_in < 0)) {
			_snprintf(error_msg, __MAX_STR_LEN, "SIOCOM: CAN'T REOPEN CONSOLE %s FOR READ: %s", p, strerror(errno));
			error_code = -errno;
			goto ret;
		}
		con_in_close = 1;
		if (!ioctl(con_in, TIOCGETA, &u.t)) {
			u.t.c_lflag &= ~(ISIG | ECHO | ICANON);
			ioctl(con_in, TIOCSETA, &u.t);
		}
	}
	if (__unlikely(fstat(con_out, &u.st)) || __likely(!S_ISFIFO(u.st.st_mode))) {
		char *p = KERNEL$HANDLE_PATH(con_out);
		con_out = open(p, O_WRONLY);
		if (__unlikely(con_out < 0)) {
			_snprintf(error_msg, __MAX_STR_LEN, "SIOCOM: CAN'T REOPEN CONSOLE %s FOR WRITE: %s", p, strerror(errno));
			error_code = -errno;
			goto ret;
		}
		con_out_close = 1;
		if (!ioctl(con_out, TIOCGETA, &u.t)) {
			u.t.c_oflag &= ~(OPOST | ONLCR);
			ioctl(con_out, TIOCSETA, &u.t);
		}
	}
	if (__likely(esc >= 0)) {
		_eprintf("ESCAPE CHARACTER IS '%s'\n", print_esc());
	} else {
		_eprintf("NO ESCAPE CHARACTER\n");
	}
	*error_msg = 0;
	error_code = 0;
	RAISE_SPL(SPL_TTY);
	in.fn = did_write;
	in.status = 1;
	in.v.len = 0;
	out.fn = did_write;
	out.status = 1;
	out.v.len = 0;
	CALL_AST(&in);
	CALL_AST(&out);
	do {
		WQ_WAIT_SYNC(&finish);
	} while (in.fn && out.fn);
	LOWER_SPL(SPL_ZERO);
	ret:
	if (__likely(h_in) >= 0) close(h_in);
	if (__likely(h_out >= 0) && __unlikely(h_in != h_out)) close(h_out);
	if (__likely(con_in >= 0) && __unlikely(con_in_close)) close(con_in);
	if (__likely(con_out >= 0) && __unlikely(con_out_close)) close(con_out);
	if (!*error_msg && !error_code) _eprintf("SIOCOM TERMINATED ON USER'S REQUEST\n");
	strcpy(KERNEL$ERROR_MSG(), error_msg);
	return error_code;
}
