#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/LIST.H>
#include <ERRNO.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/THREAD.H>
#include <SPAD/USB.H>
#include <STDLIB.H>
#include <STDARG.H>

#include "USBPRINT.H"

#define BUFFER_SIZE	2048
#define MAX_ID_SIZE	1024
#define ERROR_TIMEOUT	(JIFFIES_PER_SECOND * 10)
#define BULK_TIMEOUT	-1

typedef struct {
	USB_DEV_INTERFACE *iface;
	USB_ENDPOINT *ctrl;
	USB_ENDPOINT *bulk_out;
	THREAD_RQ thread;
	SIORQ *rq;
	unsigned char h_flags;
	char was_cr;
	WQ wq;
	unsigned char buffer[BUFFER_SIZE];
	__u8 id[MAX_ID_SIZE];
	unsigned idlen;
	void *dlrq;
	void *lnte;
	char dev_name[__MAX_STR_LEN];
} PRINT;

#define FLAGS_NONBLOCK	0x01
#define FLAGS_CRLF	0x02

static void *PRINT_LOOKUP(HANDLE *h, char *str, int open_flags);
static IO_STUB PRINT_WRITE;
static long PRINT_THREAD(void *p);
static AST_STUB PRINT_WROTE;

static const HANDLE_OPERATIONS print_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	PRINT_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	PRINT_WRITE,		/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	KERNEL$NO_OPERATION,	/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static void *PRINT_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	if (__likely(!_strcasecmp(str, "^NONBLOCK"))) {
		h->flags |= FLAGS_NONBLOCK;
	} else if (__likely(!_strcasecmp(str, "^CRLF"))) {
		h->flags |= FLAGS_CRLF;
	} else {
		return __ERR_PTR(-EBADMOD);
	}
	return NULL;
}

static DECL_IOCALL(PRINT_WRITE, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	PRINT *print;
	if (__unlikely(h->op != &print_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	print = h->fnode;
	if (__unlikely(print->rq != NULL)) {
		WQ_WAIT_F(&print->wq, RQ);
		RETURN;
	}
	print->rq = RQ;
	print->h_flags = h->flags; /* handle may become unbound while we are spawning the thread or while the thread blocks, so make copy of flags */
	print->thread.fn = PRINT_WROTE;
	print->thread.thread_main = PRINT_THREAD;
	print->thread.p = print;
	print->thread.cwd = NULL;
	print->thread.std_in = -1;
	print->thread.std_out = -1;
	print->thread.std_err = -1;
	print->thread.dlrq = NULL;
	print->thread.thread = NULL;
	print->thread.spawned = 0;
	RETURN_IORQ_CANCELABLE(&print->thread, KERNEL$THREAD, RQ);
}

static int PRINT_GET_DEVICE_ID(PRINT *print)
{
	int r;
	__u8 *id;
	MALLOC_REQUEST mrq;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("PRINT_GET_DEVICE_ID AT SPL %08X", KERNEL$SPL);
	mrq.size = MAX_ID_SIZE;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) return mrq.status;
	id = mrq.ptr;
	RAISE_SPL(SPL_USB);
	r = USB$SYNC_CTRL(print->ctrl, USBPRINT_REQ_GET_DEVICE_ID, USB$GET_CONFIGURATION_NUMBER(print->iface), (USB$GET_INTERFACE_NUMBER(print->iface) << 8) | USB$GET_ALTSETTING_NUMBER(print->iface), MAX_ID_SIZE - 1, id, USBRQ_ALLOW_SHORT, USB$GET_CTRL_TIMEOUT(print->iface), USB$GET_CTRL_RETRIES(print->iface), print->iface, NULL);
	LOWER_SPL(SPL_DEV);
	if (r >= 0) {
		memcpy(print->id, id, r);
		print->id[r] = 0;
		print->idlen = r;
		/* if (r >= 2) _printf("%s", print->id + 2); */
	}
	free(id);
	return r;
}

static int PRINT_GET_STATUS(PRINT *print)
{
	int r;
	__u8 status;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("PRINT_GET_STATUS AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_USB);
	r = USB$SYNC_CTRL(print->ctrl, USBPRINT_REQ_GET_PORT_STATUS, 0, USB$GET_INTERFACE_NUMBER(print->iface), 1, &status, USB_SYNC_CTRL_NOINTR, USB$GET_CTRL_TIMEOUT(print->iface), USB$GET_CTRL_RETRIES(print->iface), print->iface, NULL);
	LOWER_SPL(SPL_DEV);
	if (__unlikely(r < 0)) return r;
	return status;
}

static int PRINT_DATA(PRINT *print, unsigned char *ptr, unsigned len)
{
	unsigned long wrote;
	int r;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("PRINT_DATA AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_USB);
	r = USB$SYNC_CTRL(print->bulk_out, -1, -1, -1, len, ptr, 0, BULK_TIMEOUT, 0, print->iface, &wrote);
	LOWER_SPL(SPL_DEV);
	/*__debug_printf("len: %d, wrote %ld, ret %d\n", len, wrote, r);*/
	if (__likely(wrote > 0)) return wrote;
	return r;
}

#include "PIOXPAND.I"

static long PRINT_THREAD(void *p)
{
	PRINT *print = p;
	long r;
	int s;
	VBUF v;
	u_jiffies_lo_t j;
	RAISE_SPL(SPL_DEV);
	next_buf:
	j = KERNEL$GET_JIFFIES_LO();
	poll_ready:
	r = PRINT_GET_STATUS(print);
	/*__debug_printf("status: %02x\n", r);*/
	if (__unlikely(r < 0)) goto ret_r;
	if (__unlikely((r & (USBPRINT_STATUS_ERROR | USBPRINT_STATUS_SLCT | USBPRINT_STATUS_PE)) != (USBPRINT_STATUS_ERROR | USBPRINT_STATUS_SLCT))) {
		if (r & USBPRINT_STATUS_PE) r = -ENOSPC;
		else if (!(r & USBPRINT_STATUS_SLCT)) r = -ENOLINK;
		else r = -EIO;
		if (__likely(!(print->h_flags & FLAGS_NONBLOCK)) && (r != -EIO || KERNEL$GET_JIFFIES_LO() - j <= TIMEOUT_JIFFIES(ERROR_TIMEOUT))) {
			KERNEL$SLEEP(JIFFIES_PER_SECOND);
			goto poll_ready;
		}
		goto ret_r;
	}
	KERNEL$THREAD_MAY_BLOCK();
	v.ptr = print->buffer;
	v.len = __likely(!(print->h_flags & FLAGS_CRLF)) ? BUFFER_SIZE : BUFFER_SIZE / 3;
	v.spl = SPL_X(SPL_DEV);
	RAISE_SPL(SPL_VSPACE);
	s = print->rq->v.vspace->op->vspace_get(&print->rq->v, &v);
	if (__unlikely(!s)) {
		r = -EVSPACEFAULT;
		goto ret_r;
	}
	if (__likely(!(print->h_flags & FLAGS_CRLF))) {
		r = PRINT_DATA(print, print->buffer, s);
		print->was_cr = 0;
	} else {
		int sx = xpand_str(print->buffer + BUFFER_SIZE / 3, print->buffer, s, print->was_cr, NULL);
		r = PRINT_DATA(print, print->buffer + BUFFER_SIZE / 3, sx);
		if (r > 0) {
			char new_cr = (print->buffer + (BUFFER_SIZE / 3))[r - 1] == '\r';
			r = xpand_str(print->buffer + (BUFFER_SIZE / 3), print->buffer, s, print->was_cr, print->buffer + (BUFFER_SIZE / 3) + r);
			print->was_cr = new_cr;
		}
	}
	if (__unlikely(r < 0)) {
		print->rq->v.ptr -= s;
		print->rq->v.len += s;
		goto ret_r;
	}
	print->rq->progress += r;
	if (__unlikely(r != s)) {
		print->rq->v.ptr -= s - r;
		print->rq->v.len += s - r;
	}
	if (print->rq->v.len != 0) goto next_buf;
	r = print->rq->progress;
	if (__unlikely(r < 0)) r = -EOVERFLOW;
	ret_r:
	LOWER_SPL(SPL_ZERO);
	return r;
}

static DECL_AST(PRINT_WROTE, SPL_DEV, THREAD_RQ)
{
	PRINT *print = GET_STRUCT(RQ, PRINT, thread);
	SIORQ *rq = print->rq;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), rq);
	SWITCH_PROC_ACCOUNT(rq->handle->name_addrspace, SPL_X(SPL_DEV));
	print->rq = NULL;
	WQ_WAKE_ALL(&print->wq);
	if (__unlikely(print->thread.status == -EVSPACEFAULT)) {
		DO_PAGEIN(rq, &rq->v, PF_READ);
	}
	rq->status = print->thread.status;
	RETURN_AST(rq);
}

static int PRINT_TEST(USB_DEV_INTERFACE *iface, long proto)
{
	USB_DESCRIPTOR_INTERFACE *desc = USB$GET_INTERFACE_DESCRIPTOR(iface);
	if (__unlikely(desc->bInterfaceClass != USB_CLASS_PRINTER)) return USB_SKIP;
	if (__unlikely(desc->bInterfaceProtocol != proto)) return USB_SKIP;
	return USB_ATTACH_INTERFACE;
}

static void print_init_root(HANDLE *h, void *p)
{
	PRINT *print = p;
	h->fnode = print;
	h->flags = 0;
	h->flags2 = 0;
	h->op = &print_operations;
}

static int PRINT_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	int r;
	MALLOC_REQUEST mrq;
	USB_DEV_INTERFACE *iface;
	PRINT *print;
	USB_ARGS *args = NULL;
	const char *opt, *optend, *str;
	const char * const *arg = argv;
	int state = 0;
	static const struct __param_table params[2] = {
		NULL, 0, 0, 0,
	};
	void *vars[1];
	vars[0] = NULL;
	if (__unlikely(__parse_params(&arg, &state, params, vars, &opt, &optend, &str))) {
		if (__unlikely(USB$PARSE_PARAMS(opt, optend, str, &args))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBPRINT: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}

	if (__IS_ERR(iface = USB$ATTACH_DRIVER(args, PRINT_TEST, USBPRINT_PROTOCOL_BIDIRECTIONAL))) {
		if (__IS_ERR(iface = USB$ATTACH_DRIVER(args, PRINT_TEST, USBPRINT_PROTOCOL_UNIDIRECTIONAL))) {
			r = __PTR_ERR(iface);
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBPRINT: %s", strerror(-r));
			goto ret0;
		}
	}

	mrq.size = sizeof(PRINT);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBPRINT: %s", strerror(-r));
		goto ret1;
	}
	print = mrq.ptr;
	print->iface = iface;
	WQ_INIT(&print->wq, "USBPRINT$WQ");
	print->was_cr = 0;
	print->rq = NULL;

	r = USB$GET_DEVICE_NAME(print->dev_name, __MAX_STR_LEN, print->iface, args, "PIO$USBPRINT");
	if (__unlikely(r)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBPRINT: CAN'T GET DEVICE NAME: %s", strerror(-r));
		goto ret2;
	}

	print->ctrl = USB$GET_DEFAULT_ENDPOINT(print->iface);
	if (__unlikely(__IS_ERR(print->bulk_out = USB$FIND_ENDPOINT(print->iface, USB_EP_BULK_OUT, 1, BUFFER_SIZE)))) {
		r = __PTR_ERR(print->bulk_out);
		print->bulk_out = NULL;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET BULK OUT ENDPOINT: %s", print->dev_name, strerror(-r));
		goto ret2;
	}

	RAISE_SPL(SPL_DEV);
	r = PRINT_GET_DEVICE_ID(print);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET DEVICE ID: %s", print->dev_name, strerror(-r));
		goto ret2;
	}

	/*_printf("usbprint: %s\n", print->dev_name);*/

	r = KERNEL$REGISTER_DEVICE(print->dev_name, "USBPRINT.SYS", 0, print, print_init_root, NULL, NULL, NULL, PRINT_UNLOAD, &print->lnte, USB$GET_CONTROLLER_NAME(iface), NULL);
	if (__unlikely(r < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", print->dev_name, strerror(-r));
		goto ret2;
	}
	print->dlrq = KERNEL$TSR_IMAGE();
	strlcpy(KERNEL$ERROR_MSG(), print->dev_name, __MAX_STR_LEN);
	USB$FREE_PARAMS(args);
	return 0;

	ret2:
	free(print);
	ret1:
	USB$DETACH_DRIVER(iface);
	ret0:
	USB$FREE_PARAMS(args);
	return r;
}

static int PRINT_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	PRINT *print = p;
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(print->lnte, argv))) return r;
	RAISE_SPL(SPL_DEV);
	while (__unlikely(print->rq != NULL)) {
		KERNEL$CIO((void *)&print->thread);
		KERNEL$SLEEP(1);
	}
	LOWER_SPL(SPL_ZERO);
	USB$DETACH_DRIVER(print->iface);
	*release = print->dlrq;
	free(print);
	return 0;
}
