#include <SPAD/LIBC.H>
#include <ARCH/BSF.H>

#include "AC97.H"
#include "AC97REG.H"

unsigned ac97_id;
unsigned ac97_xtd_id;

int ac97_vendor_id1;
int ac97_vendor_id2;

unsigned OSS_MIXER_DEVMASK = 0;
unsigned OSS_MIXER_STEREODEVS = 0;
unsigned OSS_MIXER_RECMASK = 0;
unsigned OSS_MIXER_OUTMASK = 0;

static struct {
	char bits;
	int stereo:1;
	int invert:1;
	int double_mute:1;
	int rd:1;
	int shift;
	unsigned mask;
	unsigned mixer;
} mixers[0x80];

unsigned long anyrate[48000 / 8 / sizeof(unsigned long) + 1];
unsigned long rates[5][48000 / 8 / sizeof(unsigned long) + 1];

static void check_volume(int i, unsigned mixer, int rd)
{
	int r;
	int j;
	r = read_ac97(i);
	if (r <= 0) return;
	/*__debug_printf("vol: %d\n", i);*/
	if (i == AC_PC_BEEP_VOLUME) {
		write_ac97(i, 0xa << 1);
		if ((read_ac97(i) & (0xf << 1)) != (0xa << 1)) return;
		write_ac97(i, 0);
		mixers[i].shift = 1;
		mixers[i].bits = 4;
		goto x;
	}
	for (j = 1; j < 0x40 && (write_ac97(i, j), read_ac97(i) == j); j <<= 1);
	if (j <= 1) {
		if (i == AC_CENTER_LFE_VOLUME || i == AC_SURR_VOLUME) write_ac97(i, 0x8080);
		else write_ac97(i, 0x8000);
		return;
	}
	mixers[i].bits = __BSR(j);
	j >>= 1;
	write_ac97(i, (j << 8) | j);
	if (i != AC_MONO_VOLUME && i != AC_PHONE_VOLUME && i != AC_MIC_VOLUME && i != AC_RECORD_GAIN_MIC) {
		if (read_ac97(i) == ((j << 8) | j)) mixers[i].stereo = 1;
	}
	x:
	if (rd) mixers[i].rd = 1;
	if (i == AC_RECORD_GAIN || i == AC_RECORD_GAIN_MIC) mixers[i].invert = 1;
	if (i == AC_CENTER_LFE_VOLUME || i == AC_SURR_VOLUME) mixers[i].double_mute = 1;
	if (i == AC_MIC_VOLUME) mixers[i].mask = 0x40;
	mixers[i].mixer = mixer;
	OSS_MIXER_DEVMASK |= 1 << mixer;
	if (mixers[i].stereo) OSS_MIXER_STEREODEVS |= 1 << mixer;
	if (i == AC_PHONE_VOLUME || i == AC_MIC_VOLUME || i == AC_LINE_IN_VOLUME || i == AC_CD_VOLUME || i == AC_VIDEO_VOLUME || i == AC_AUX_IN_VOLUME || i == SOUND_MIXER_AUX_OUT || i == SOUND_MIXER_MONO_OUT) OSS_MIXER_RECMASK |= 1 << mixer;
	if (i == AC_MASTER_VOLUME || i == AC_AUX_OUT_VOLUME || i == AC_MONO_VOLUME || i == AC_PC_BEEP_VOLUME || i == AC_CD_VOLUME || i == AC_VIDEO_VOLUME || i == AC_PCM_OUT_VOLUME || i == AC_CENTER_LFE_VOLUME || i == AC_SURR_VOLUME) {
		write_ac97(i, mixers[i].mask);
	}
}

int AC97_INIT(void)
{
	unsigned i, j;
	int r;
	memset(mixers, 0, sizeof mixers);
	write_ac97(AC_RESET_ID, 0);
	ac97_id = read_ac97(AC_RESET_ID);
	if ((int)ac97_id < 0) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CAN'T READ CODEC ID");
		return -ENXIO;
	}
	ac97_vendor_id1 = read_ac97(AC_VENDOR_ID1);
	ac97_vendor_id2 = read_ac97(AC_VENDOR_ID2);
	r = read_ac97(AC_MASTER_VOLUME);
	if (r <= 0 || r == 0xffff) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC NOT PRESENT");
		return -ENXIO;
	}
	ac97_xtd_id = read_ac97(AC_XTD_AUDIO_ID);
	if (ac97_xtd_id & AC_XTD_ID_VRA) {
		set_vra:
		write_ac97(AC_XTD_AUDIO_STAT_CTRL, read_ac97(AC_XTD_AUDIO_STAT_CTRL) | AC_XTD_CTRL_VRA);
	} else {
		/* via82cxxx_audio.c, end of via_ac97_reset */
		write_ac97(AC_XTD_AUDIO_ID, ac97_xtd_id | AC_XTD_ID_VRA);
		ac97_xtd_id = read_ac97(AC_XTD_AUDIO_ID);
		if (ac97_xtd_id & AC_XTD_ID_VRA) goto set_vra;
	}
	/*{
		for (i = 0; i < 0x80; i += 2) {
			if (!(i & 0xe)) __debug_printf("%02x : ", i);
			__debug_printf(" %04X", read_ac97(i));
			if ((i & 0xe) == 0xe) __debug_printf("\n");
		}
	}*/
	memset(rates, 0, sizeof rates);
	memset(anyrate, 0, sizeof anyrate);
	for (i = 0; i < 5; i++) {
			/* enable this when 6-ch will be implemented */
		if (i != 0 && i != 3) continue;
		if (!(ac97_xtd_id & AC_XTD_ID_VRA)) {
			__BS(&rates[i], 48000);
			__BS(anyrate, 48000);
			KERNEL$THREAD_MAY_BLOCK();
			continue;
		}
		for (j = 1; j <= 48000; j++) {
			write_ac97(AC_PCM_FRONT_DAC_RATE + 2 * i, j);
			if (read_ac97(AC_PCM_FRONT_DAC_RATE + 2 * i) == j) {
				__BS(&rates[i], j);
				__BS(anyrate, j);
			}
			KERNEL$THREAD_MAY_BLOCK();
		}
	}
	check_volume(AC_MASTER_VOLUME, SOUND_MIXER_VOLUME, 1);
	check_volume(AC_AUX_OUT_VOLUME, SOUND_MIXER_AUX_OUT, 1);
	check_volume(AC_MONO_VOLUME, SOUND_MIXER_MONO_OUT, 1);
	check_volume(AC_PC_BEEP_VOLUME, SOUND_MIXER_SPEAKER, 1);
	check_volume(AC_PHONE_VOLUME, SOUND_MIXER_LINE3, 1);
	check_volume(AC_MIC_VOLUME, SOUND_MIXER_MIC, 1);
	check_volume(AC_LINE_IN_VOLUME, SOUND_MIXER_LINE, 1);
	check_volume(AC_CD_VOLUME, SOUND_MIXER_CD, 1);
	check_volume(AC_VIDEO_VOLUME, SOUND_MIXER_LINE2, 1);
	check_volume(AC_AUX_IN_VOLUME, SOUND_MIXER_LINE1, 1);
	check_volume(AC_PCM_OUT_VOLUME, SOUND_MIXER_PCM, 1);
	check_volume(AC_RECORD_GAIN, SOUND_MIXER_RECLEV, 1);
	check_volume(AC_RECORD_GAIN_MIC, SOUND_MIXER_RECLEV, 0);
	/*
	check_volume(AC_CENTER_LFE_VOLUME, SOUND_MIXER_VOLUME, 0);
	check_volume(AC_SURR_VOLUME, SOUND_MIXER_VOLUME, 0);
	*/
	if (ac97_id & AC_ID_BASS_TREBLE) {
		OSS_MIXER_DEVMASK |= (1 << SOUND_MIXER_BASS) | (1 << SOUND_MIXER_TREBLE);
		write_ac97(AC_MASTER_TONE, 0x0F0F);
	}
	if (ac97_vendor_id1 == 0x414c && ac97_vendor_id2 <= 0x4710) {
	/* this codec is shitty, when volume set to max, something overflows
	   in it and it produces scratching sounds */
		SND_MIXER_WRITE(SOUND_MIXER_PCM, (50 << 8) | 50);
	}
	/*{
		int i;
		for (i = 0; i < 0x40; i += 2) __debug_printf("%02x: %04x  ", i, read_ac97(i));
	}*/
	/*__debug_printf("codec: %x, %x\n", read_ac97(AC_VENDOR_ID1), read_ac97(AC_VENDOR_ID2));*/
	
	return 0;
}

static int test_rate(HANDLE *h, unsigned rate, int open_flags)
{
	switch (open_flags) {
		case 0:
			return __BT(&rates[0], rate) | __BT(&rates[3], rate);
		case _O_KRNL_READ:
			return __BT(&rates[3], rate);
		case _O_KRNL_WRITE:
			return __BT(&rates[0], rate);
		case _O_KRNL_READ | _O_KRNL_WRITE:
			return __BT(&rates[0], rate) & __BT(&rates[3], rate);
		default:
			KERNEL$SUICIDE("test_rate: INVALID OPEN FLAGS 0%o", open_flags);
	}
}

void AC97_FIXUP_RATE(HANDLE *h, int open_flags)
{
	int diff;
	int rate;
	open_flags &= _O_KRNL_READ | _O_KRNL_WRITE;
	rate = FORMAT_RATE(h);
	for (diff = 0; diff < 48000; diff++) {
		if (rate + diff > 48000 || __likely(!__BT(anyrate, rate + diff))) goto no_plus;
		if (test_rate(h, rate + diff, open_flags)) {
			FORMAT_SETRATE(h, rate + diff);
			return;
		}
		no_plus:;
		if (rate - diff < 1 || __likely(!__BT(anyrate, rate - diff))) goto no_minus;
		if (test_rate(h, rate - diff, open_flags)) {
			FORMAT_SETRATE(h, rate + diff);
			return;
		}
		no_minus:;
	}
	FORMAT_SETRATE(h, 48000);
	return;
}

void AC97_SET_PLAYBACK_RATE(unsigned rate)
{
	if (__unlikely(!(ac97_xtd_id & AC_XTD_ID_VRA))) return;
	write_ac97(AC_PCM_FRONT_DAC_RATE, rate);
}

void AC97_SET_RECORD_RATE(unsigned rate)
{
	if (__unlikely(!(ac97_xtd_id & AC_XTD_ID_VRA))) return;
	write_ac97(AC_LR_ADC_RATE, rate);
}

static unsigned SCALE(unsigned val, unsigned s_from, unsigned s_to)
{
	return (val * s_to + (s_from >> 1)) / s_from;
}

long SND_MIXER_READ(unsigned mix)
{
	int i;
	for (i = 0; i < 0x80; i += 2) if (mixers[i].bits && mixers[i].rd && mixers[i].mixer == mix) {
		unsigned val = read_ac97(i);
		unsigned l = (val >> 8) & ((1 << mixers[i].bits) - 1);
		unsigned r = (val >> mixers[i].shift) & ((1 << mixers[i].bits) - 1);
		if (!mixers[i].invert) {
			l = (1 << mixers[i].bits) - 1 - l;
			r = (1 << mixers[i].bits) - 1 - r;
		}
		l = SCALE(l, (1 << mixers[i].bits) - 1, 99) + 1;
		r = SCALE(r, (1 << mixers[i].bits) - 1, 99) + 1;
		if (!mixers[i].double_mute) {
			if (val & 0x8000) l = r = 0;
		} else {
			if (val & 0x8000) l = 0;
			if (val & 0x0080) r = 0;
		}
		if (!mixers[i].stereo) return r;
		return l + (r << 8);
	}
	if (ac97_id & AC_ID_BASS_TREBLE && (mix == SOUND_MIXER_BASS || mix == SOUND_MIXER_TREBLE)) {
		unsigned val = read_ac97(AC_MASTER_TONE);
		if (mix == SOUND_MIXER_BASS) val >>= 8;
		val &= 0xf;
		if (val == 0xf) return 0;
		val = 0xe - val;
		return SCALE(val, 0xe, 99) + 1;
	}
	return -EINVAL;
}

void SND_MIXER_WRITE(unsigned mix, unsigned long val)
{
	int i;
	if (ac97_id & AC_ID_BASS_TREBLE && (mix == SOUND_MIXER_BASS || mix == SOUND_MIXER_TREBLE)) {
		unsigned r;
		if (!val) val = 0xf;
		else val = 0xe - SCALE(val - 1, 99, 0xe);
		r = read_ac97(AC_MASTER_TONE);
		if (mix == SOUND_MIXER_BASS) r = (r & 0xf0ff) | (val << 8);
		else r = (r & 0xfff0) | val;
		write_ac97(AC_MASTER_TONE, r);
		return;
	}
	for (i = 0; i < 0x80; i += 2) if (mixers[i].bits && mixers[i].rd && mixers[i].mixer == mix) {
		unsigned l = val & 0xff;
		unsigned r = val >> 8;
		if (l) l--;
		if (r) r--;
		l = SCALE(l, 99, (1 << mixers[i].bits) - 1);
		r = SCALE(r, 99, (1 << mixers[i].bits) - 1);
		if (!mixers[i].invert) {
			l = (1 << mixers[i].bits) - 1 - l;
			r = (1 << mixers[i].bits) - 1 - r;
		}
		r <<= mixers[i].shift;
		if (!mixers[i].double_mute) {
			if (!val) l = 0x80, r = 0x00;
		} else {
			if (!(val & 0x00ff)) l = 0x80;
			if (!(val & 0xff00)) r = 0x80;
		}
		if (!mixers[i].stereo && l != 0x80) r = l << mixers[i].shift, l = 0;
		write_ac97(i, (l << 8) | r | mixers[i].mask);
	}
}

long SND_MIXER_READ_RECSRC(void)
{
	unsigned v = read_ac97(AC_RECORD_SELECT) & 7;
	switch (v) {
		case AC_RECORD_MIC: return SOUND_MASK_MIC;
		case AC_RECORD_CD: return SOUND_MASK_CD;
		case AC_RECORD_VIDEO: return SOUND_MASK_LINE2;
		case AC_RECORD_AUX: return SOUND_MASK_LINE1;
		case AC_RECORD_LINE: return SOUND_MASK_LINE;
		case AC_RECORD_STEREO: return SOUND_MASK_AUX_OUT;
		case AC_RECORD_MONO: return SOUND_MASK_MONO_OUT;
		case AC_RECORD_PHONE: return SOUND_MASK_LINE3;
	}
	return -EINVAL;
}

void SND_MIXER_WRITE_RECSRC(unsigned long mask)
{
	unsigned v;
	switch (mask) {
		case SOUND_MASK_MIC: v = AC_RECORD_MIC; break;
		case SOUND_MASK_CD: v = AC_RECORD_CD; break;
		case SOUND_MASK_LINE2: v = AC_RECORD_VIDEO; break;
		case SOUND_MASK_LINE1: v = AC_RECORD_AUX; break;
		case SOUND_MASK_LINE: v = AC_RECORD_LINE; break;
		case SOUND_MASK_AUX_OUT: v = AC_RECORD_STEREO; break;
		case SOUND_MASK_MONO_OUT: v = AC_RECORD_MONO; break;
		case SOUND_MASK_LINE3: v = AC_RECORD_PHONE; break;
		default: return;
	}
	write_ac97(AC_RECORD_SELECT, (v << 8) | v);
}

void SND_MIXER_WRITE_OUTSRC(unsigned long mask)
{
}

long SND_MIXER_READ_OUTSRC(void)
{
	return 0;
}

