#include <SPAD/PCI.H>
#include <SPAD/PCI_IDS.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/IO.H>
#include <SPAD/TIMER.H>
#include <SYS/SOUNDCARD.H>
#include <SPAD/IRQ.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>
#include <VALUES.H>

#include "HDA.H"
#include "HDACODEC.H"
#include "HDAREG.H"

static const struct pci_id_s devices[] = {
	{0x8086, 0x2668, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH6", 0 },
	{0x8086, 0x27d8, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH7", 0 },
	{0x10b9, 0x5461, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ALI", 0 },
	{0},
};

static pci_id_t pci_id;
static __u8 *mem;
char dev_name[__MAX_STR_LEN];
static char *chip_name;

static AST AZ_IRQ_AST;
static IRQ_HANDLE *AZ_IRQ_CTRL;

#define AZ_READ_8(reg)		mmio_inb(mem + (reg))
#define AZ_READ_16(reg)		mmio_inw(mem + (reg))
#define AZ_READ_24(reg)		(AZ_READ_16(reg) + (AZ_READ_8((reg) + 2) << 16))
#define AZ_READ_32(reg)		mmio_inl(mem + (reg))
#define AZ_WRITE_8(reg, val)	mmio_outb(mem + (reg), val)
#define AZ_WRITE_16(reg, val)	mmio_outw(mem + (reg), val)
#define AZ_WRITE_24(reg, val)	(AZ_WRITE_8((reg) + 2, (val) >> 16), AZ_WRITE_16(reg, val))
#define AZ_WRITE_32(reg, val)	mmio_outl(mem + (reg), val)

struct az_stream {
	struct bm_entry frag[MAX_DESCS];
};

struct az_desc {
	struct az_stream stream[2];
	__u32 corb[256];
	struct rirb_entry rirb[256];
};

#define DESC_SIZE ((sizeof(struct az_desc) + __PAGE_CLUSTER_SIZE_MINUS_1) & __NOT_PAGE_CLUSTER_SIZE_MINUS_1)

char OSS_DMA_USE_64;
static __u32 az_gcap;
static struct az_desc *desc;
static __u64 desc_dmaaddr;
static vspace_dmaunlock_t *desc_unlock_32;
static vspace_dma64unlock_t *desc_unlock_64;
static unsigned corb_size;
static unsigned rirb_size;
static unsigned corb_write;
static unsigned rirb_read;
static __u32 intctl;
static unsigned codec;

static jiffies_t HDA_JIFFIES[OSS_N_STREAMS];
static TIMER HDA_TIMEOUT[OSS_N_STREAMS];

const int RATELIST[129] = {
	48000, 24000, 16000, -1, 9600, 8000, -1, 6000,
	96000, -1, 32000, -1, -1, -1, -1, -1,
	144000, -1, -1, -1, -1, -1, -1, -1,
	192000, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	44100, 22050, -1, 11025, -1, -1, -1, -1,
	88200, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	176400, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	0,
};

unsigned long PLAYBACK_RATES[(128 + 8*sizeof(unsigned long)-1) / 8 / sizeof(unsigned long)];
unsigned long RECORD_RATES[(128 + 8*sizeof(unsigned long)-1) / 8 / sizeof(unsigned long)];

const HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	OSS_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	OSS_INSTANTIATE,	/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};


static void HDA_STREAM_INTERRUPT(int record);

static int AZ_RESET(void)
{
	int i;
	AZ_WRITE_16(AZ_STATESTS, 0xffff);
	AZ_WRITE_32(AZ_GCTL, 0);
	i = 0;
	while (KERNEL$UDELAY(10), AZ_READ_32(AZ_GCTL) & AZ_GCTL_RESET) {
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO SET RESET STATE");
			return -1;
		}
	}
	KERNEL$UDELAY(300);
	AZ_WRITE_32(AZ_GCTL, AZ_GCTL_RESET);
	KERNEL$UDELAY(10000);
	i = 0;
	while (KERNEL$UDELAY(10), !(AZ_READ_32(AZ_GCTL) & AZ_GCTL_RESET)) {
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO CLEAR RESET STATE");
			return -1;
		}
	}
	KERNEL$UDELAY(1000);
	return 0;
}

static void AZ_SETUP_BUFFERS(void)
{
	int i;
	AZ_WRITE_8(AZ_CORBCTL, 0);
	AZ_WRITE_16(AZ_CORBRP, AZ_CORBRP_RESET);
	i = 0;
	while (!(AZ_READ_16(AZ_CORBRP) & AZ_CORBRP_RESET)) {
		KERNEL$UDELAY(10);
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO SET CORB RESET STATE");
			break;
		}
	}
	AZ_WRITE_16(AZ_CORBRP, 0);
	i = 0;
	while (AZ_READ_16(AZ_CORBRP) & AZ_CORBRP_RESET) {
		KERNEL$UDELAY(10);
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO CLEAR CORB RESET STATE");
			break;
		}
	}
	KERNEL$UDELAY(10);
	AZ_WRITE_32(AZ_CORBLBASE, desc_dmaaddr + (int)((struct az_desc *)NULL)->corb);
	AZ_WRITE_32(AZ_CORBUBASE, desc_dmaaddr >> 32);
	AZ_WRITE_16(AZ_CORBWP, 0);
	AZ_WRITE_8(AZ_CORBCTL, AZ_CORBCTL_ENABLE | AZ_CORBCTL_CMEI);
	corb_write = 0;
	AZ_WRITE_8(AZ_RIRBCTL, 0);
	AZ_WRITE_16(AZ_RIRBWP, AZ_RIRBWP_RESET);
	AZ_WRITE_32(AZ_RIRBLBASE, desc_dmaaddr + (int)((struct az_desc *)NULL)->rirb);
	AZ_WRITE_32(AZ_RIRBUBASE, desc_dmaaddr >> 32);
	AZ_WRITE_16(AZ_RINTCNT, rirb_size / 4);
	AZ_WRITE_8(AZ_RIRBCTL, AZ_RIRBCTL_OVER_INT | AZ_RIRBCTL_ENABLE | AZ_RIRBCTL_ENABLE_INT);
	rirb_read = 0;
}

static DECL_IRQ_AST(AZ_IRQ, SPL_SND, AST)
{
	__u32 intstat = AZ_READ_32(AZ_INTSTS) & intctl;
	/*if (intstat) __debug_printf("irq...: %x\n", intstat);*/
	if (__unlikely(intstat & AZ_INTSTS_CIS)) {
		unsigned corbst, rirbsts;
		unsigned statests = AZ_READ_16(AZ_STATESTS);
		AZ_WRITE_16(AZ_STATESTS, statests);
	
		corbst = AZ_READ_8(AZ_CORBST);
		if (__unlikely(corbst & AZ_CORBST_CMEI)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CORB MEMORY ERROR");
			AZ_WRITE_8(AZ_CORBST, AZ_CORBST_CMEI);
			AZ_SETUP_BUFFERS();
			goto skip_buffer_int;
		}
		rirbsts = AZ_READ_8(AZ_RIRBSTS);
		if (__unlikely(rirbsts & AZ_RIRBSTS_OVER_INT))
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, dev_name, "RESPONSE INPUT BUFFER OVERFLOW");
		if (__likely(rirbsts & (AZ_RIRBSTS_OVER_INT | AZ_RIRBSTS_INT))) {
			unsigned wp = AZ_READ_16(AZ_RIRBWP);
			unsigned i = rirb_read;
			while (1) {
				unsigned ex = __32LE2CPU(desc->rirb[i].extd);
				if (__unlikely((ex & RIRB_CODEC_MASK) != codec)) goto skip_resp;
				HDA_CODEC_RESPONSE(ex & RIRB_CODEC_MASK, __32LE2CPU(desc->rirb[i].resp), ex & RIRB_UNSOLICITED);
				skip_resp:
				if (__likely(i == wp)) break;
				i = (i + 1) & (rirb_size - 1);
			}
			rirb_read = (i + 1) & (rirb_size - 1);
			AZ_WRITE_8(AZ_RIRBSTS, rirbsts & (AZ_RIRBSTS_OVER_INT | AZ_RIRBSTS_INT));
		}
		skip_buffer_int:;
	}
	if (__likely(intstat & AZ_INTSTS_OUTPUT0)) {
		HDA_STREAM_INTERRUPT(0);
	}
	if (__unlikely(intstat & AZ_INTSTS_INPUT0)) {
		HDA_STREAM_INTERRUPT(1);
	}
	KERNEL$UNMASK_IRQ(AZ_IRQ_CTRL);
	RETURN;
}

int HDA_CODEC_REQUEST(unsigned codec_id, unsigned node, unsigned verb)
{
	unsigned rp;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("HDA_CODEC_REQUEST AT SPL %08X", KERNEL$SPL);
	rp = (AZ_READ_16(AZ_CORBRP) - 1) & (corb_size - 1);
	if (__unlikely(corb_write == rp)) return -1;
	corb_write = (corb_write + 1) & (corb_size - 1);
	desc->corb[corb_write] = __32CPU2LE((codec_id << CORB_CODEC_SHIFT) + (node << CORB_NODE_SHIFT) + (verb << CORB_VERB_SHIFT));
	AZ_WRITE_16(AZ_CORBWP, corb_write);
	return 0;
}

static unsigned stream_frag[OSS_N_STREAMS];

void SND_START(int record)
{
	unsigned base, stream_id, int_id;
	unsigned i, cnt, c_fm;
	struct az_stream *stream;
	struct page_desc *pages;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
		int_id = AZ_INTCTL_OUTPUT0;
		stream_id = PLAYBACK_STREAM_ID;
	} else {
		base = AZ_INPUT0;
		int_id = AZ_INTCTL_INPUT0;
		stream_id = RECORD_STREAM_ID;
	}
	stream = &desc->stream[record];
	pages = DATA_PAGES[record];
	c_fm = (oss_rate[record] << AZ_SDFMT_RATE_SHIFT) | AZ_SDFMT_BITS_16 | (1 << AZ_SDFMT_CHAN_SHIFT);
	HDA_CODEC_SET_FORMAT(record, c_fm);
	memset(stream, 0, MAX_DESCS * sizeof(struct bm_entry));
	if (__unlikely(oss_frags[record] << oss_fragsize_bits[record] >> PAGESIZE_BITS > N_PAGES))
		KERNEL$SUICIDE("SND_START: %s: OUT OF PAGES (%u >= %u), STREAM %d", dev_name, oss_frags[record] << oss_fragsize_bits[record] >> PAGESIZE_BITS, N_PAGES, record);
	for (i = 0; i < oss_frags[record]; i++) {
		stream->frag[i].addr = __64CPU2LE(pages[i << oss_fragsize_bits[record] >> PAGESIZE_BITS].device + ((i << oss_fragsize_bits[record]) & ((1 << PAGESIZE_BITS) - 1)));
		stream->frag[i].len = 1 << oss_fragsize_bits[record];
		stream->frag[i].flags = BM_IOC;
	}
	AZ_WRITE_24(base + AZ_SDCTL, 0);
	KERNEL$UDELAY(10);
	AZ_WRITE_24(base + AZ_SDCTL, AZ_SDCTL_SRST);
	cnt = 0;
	while (KERNEL$UDELAY(3), !(AZ_READ_24(base + AZ_SDCTL) & AZ_SDCTL_SRST)) {
		if (__unlikely(++cnt >= 1000)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "STREAM %d FAILED TO SET RESET STATE", record);
			return;
		}
	}
	AZ_WRITE_24(base + AZ_SDCTL, 0);
	cnt = 0;
	while (KERNEL$UDELAY(3), AZ_READ_24(base + AZ_SDCTL) & AZ_SDCTL_SRST) {
		if (__unlikely(++cnt >= 1000)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "STREAM %d FAILED TO CLEAR RESET STATE", record);
			return;
		}
	}
	KERNEL$UDELAY(10);
	AZ_WRITE_8(base + AZ_SDSTS, AZ_READ_8(base + AZ_SDSTS));
	AZ_WRITE_24(base + AZ_SDCTL, (stream_id << AZ_SDCTL_NUMBER_SHIFT) | AZ_SDCTL_PRIO | AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL);
	AZ_WRITE_32(base + AZ_SDCBL, oss_frags[record] << oss_fragsize_bits[record]);
	AZ_WRITE_16(base + AZ_SDFMT, c_fm);
	AZ_WRITE_16(base + AZ_SDLVI, oss_frags[record] - 1);
	AZ_WRITE_32(base + AZ_SDBPL, desc_dmaaddr + ((__u8 *)stream - (__u8 *)desc));
	AZ_WRITE_32(base + AZ_SDBPU, desc_dmaaddr >> 32);
	/*__debug_printf("%016Lx\n", desc_dmaaddr);*/
	KERNEL$UDELAY(10);
	intctl |= int_id;
	AZ_WRITE_32(AZ_INTCTL, intctl);
	AZ_WRITE_16(base + AZ_SDCTL, AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL | AZ_SDCTL_RUN);
	stream_frag[record] = 0;
	OSS_STREAM_FLAGS[record] |= OSS_RUNNING;
	KERNEL$DEL_TIMER(&HDA_TIMEOUT[record]);
	HDA_JIFFIES[record] = OSS_STREAM_TIMEOUT(record);
	KERNEL$SET_TIMER(HDA_JIFFIES[record], &HDA_TIMEOUT[record]);
}

void SND_STOP(int record)
{
	unsigned base, int_id, cnt;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
		int_id = AZ_INTCTL_OUTPUT0;
	} else {
		base = AZ_INPUT0;
		int_id = AZ_INTCTL_INPUT0;
	}
	KERNEL$DEL_TIMER(&HDA_TIMEOUT[record]);
	SET_TIMER_NEVER(&HDA_TIMEOUT[record]);
	AZ_WRITE_16(base + AZ_SDCTL, 0);
	cnt = 0;
	while (AZ_READ_16(base + AZ_SDCTL) & AZ_SDCTL_RUN) {
		KERNEL$UDELAY(1);
		if (++cnt >= 1000) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "STREAM %d FAILED TO STOP", record);
			AZ_WRITE_24(base + AZ_SDCTL, AZ_SDCTL_SRST);
			break;
		}
	}
	intctl &= ~int_id;
	AZ_WRITE_32(AZ_INTCTL, intctl);
}

void SND_PAUSE(int record)
{
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
	} else {
		base = AZ_INPUT0;
	}
	AZ_WRITE_16(base + AZ_SDCTL, AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL);
	KERNEL$DEL_TIMER(&HDA_TIMEOUT[record]);
	SET_TIMER_NEVER(&HDA_TIMEOUT[record]);
}

void SND_RESUME(int record)
{
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
	} else {
		base = AZ_INPUT0;
	}
	AZ_WRITE_16(base + AZ_SDCTL, AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL | AZ_SDCTL_RUN);
	KERNEL$DEL_TIMER(&HDA_TIMEOUT[record]);
	KERNEL$SET_TIMER(HDA_JIFFIES[record], &HDA_TIMEOUT[record]);
}

unsigned SND_GET_PTR(int record)
{
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
	} else {
		base = AZ_INPUT0;
	}
	return AZ_READ_32(base + AZ_SDLPIB);
}

static void HDA_STREAM_INTERRUPT(int record)
{
	unsigned stat;
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
	} else {
		base = AZ_INPUT0;
	}
	stat = AZ_READ_8(base + AZ_SDSTS);
	if (__unlikely(stat & AZ_SDSTS_ERR)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "MEMORY ERROR ON STREAM %d", record);
		AZ_WRITE_8(base + AZ_SDSTS, AZ_SDSTS_ERR);
		OSS_STOP(record);
		return;
	}
	if (__unlikely(stat & AZ_SDSTS_FIFO_ERR)) {
		KERNEL$SYSLOG(__SYSLOG_HW_WARNING, dev_name, "FIFO UNDER/OVERFLOW ON STREAM %d", record);
		AZ_WRITE_8(base + AZ_SDSTS, AZ_SDSTS_FIFO_ERR);
	}
	if (__likely(stat & AZ_SDSTS_COMPL)) {
		unsigned pos;
		unsigned cnt;
		AZ_WRITE_8(base + AZ_SDSTS, AZ_SDSTS_COMPL);
		pos = SND_GET_PTR(record);
		pos >>= oss_fragsize_bits[record];
		cnt = oss_frags[record];
		while (stream_frag[record] != pos) {
			if (__unlikely(OSS_STREAM_INTR(record))) return;
			if (__unlikely(++stream_frag[record] == oss_frags[record])) stream_frag[record] = 0;
			if (__unlikely(!cnt--)) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "BAD POINTER RETURNED (%u); FRAGS %u, FRAGSIZE_BITS %u, STREAM %d", pos, oss_frags[record], oss_fragsize_bits[record], record);
				break;
			}
			KERNEL$DEL_TIMER(&HDA_TIMEOUT[record]);
			KERNEL$SET_TIMER(HDA_JIFFIES[record], &HDA_TIMEOUT[record]);
		}
	}
}

static void HDA_TIMEOUT_FN(TIMER *t)
{
	int record;
	LOWER_SPL(SPL_SND);
	SET_TIMER_NEVER(t);
	record = HDA_TIMEOUT - t;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INTERRUPT TIMEOUT ON STREAM %d", record);
	OSS_STOP(record);
}

static void *lnte, *dlrq;
static int HDA_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	unsigned memory = DEFAULT_MEMORY;
	const char * const *arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	int irq;
	const char *opt, *optend, *str;
	int r;
	union {
		char gstr[__MAX_STR_LEN];
		VDESC v;
	} u;
	unsigned statests;
	int i;
	unsigned pages, page_size;
	static const struct __param_table params[2] = {
		"MEMORY", __PARAM_UNSIGNED_INT, 1, MAXINT,
		NULL, 0, 0, 0,
	};
	void *vars[2];
	vars[0] = &memory;
	vars[1] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "HDA: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (memory & (memory - 1) || memory > MAX_MEMORY_SIZE) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "HDA: BAD MEMORY SIZE (MUST BE POWER OF 2 AND BETWEEN %u AND %u)", (unsigned)__PAGE_CLUSTER_SIZE, (unsigned)MAX_MEMORY_SIZE);
		r = -EBADSYN;
		goto ret0;
	}
	if (memory < __PAGE_CLUSTER_SIZE) memory = __PAGE_CLUSTER_SIZE;
	if (PCI$FIND_DEVICE(devices, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, NULL, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "HDA: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	pci_id = id;
	_snprintf(dev_name, sizeof(dev_name), "SND$HDA@" PCI_ID_FORMAT, pci_id);
	/*__debug_printf("command: %04x\n", PCI$READ_CONFIG_WORD(pci_id, PCI_COMMAND));*/
	PCI$ENABLE_DEVICE(pci_id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER | PCI_COMMAND_INT_DIS);
	/*__debug_printf("command: %04x\n", PCI$READ_CONFIG_WORD(pci_id, PCI_COMMAND));
	__debug_printf("status: %04x\n", PCI$READ_CONFIG_WORD(pci_id, PCI_STATUS));
	__debug_printf("bist: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_BIST));
	__debug_printf("header type: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_HEADER_TYPE));
	__debug_printf("latency timer: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_LATENCY_TIMER));
	__debug_printf("cache size: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_CACHE_LINE_SIZE));*/
	irq = PCI$READ_INTERRUPT_LINE(pci_id);
	mem = PCI$MAP_MEM_RESOURCE(pci_id, 0, AZ_REGSPACE);
	if (!mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURC", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
		r = __PTR_ERR(mem);
		goto ret1;
	}
	if (AZ_RESET()) {
		cant_init:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CONTROLLER FAILED TO INITIALIZE", dev_name);
		r = -ENXIO;
		goto ret2;
	}
	az_gcap = AZ_READ_32(AZ_GCAP);
	OSS_DMA_USE_64 = !!(AZ_GCAP_64 & AZ_GCAP_64);
	statests = AZ_READ_16(AZ_STATESTS);
	if (!statests) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "NO CODEC ON HDA LINK");
		r = -ENXIO;
		goto ret2;
	}

	for (i = 0; i < 2; i++) {
		unsigned reg = !i ? AZ_CORBSIZE : AZ_RIRBSIZE;
		unsigned sizereg = AZ_READ_8(reg);
		unsigned size = 0;	/* warning, go away */
		if ((sizereg & AZ_CORBSIZE_MASK) != AZ_CORBSIZE_256) {
			if (size & AZ_CORBSIZE_CAP_256)
				AZ_WRITE_8(reg, (sizereg & ~AZ_CORBSIZE_MASK) | AZ_CORBSIZE_256);
			else if (size & AZ_CORBSIZE_CAP_16)
				AZ_WRITE_8(reg, (sizereg & ~AZ_CORBSIZE_MASK) | AZ_CORBSIZE_16);
			else if (size & AZ_CORBSIZE_CAP_2)
				AZ_WRITE_8(reg, (sizereg & ~AZ_CORBSIZE_MASK) | AZ_CORBSIZE_2);
			sizereg = AZ_READ_8(reg);
		}
		if ((sizereg & AZ_CORBSIZE_MASK) == AZ_CORBSIZE_256)
			size = 256;
		else if ((sizereg & AZ_CORBSIZE_MASK) == AZ_CORBSIZE_16)
			size = 16;
		else if ((sizereg & AZ_CORBSIZE_MASK) == AZ_CORBSIZE_2)
			size = 2;
		else {
			KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "UNSUPPORTED %s SIZE (REG=%02X)", !i ? "CORB" : "RIRB", sizereg);
			goto cant_init;
		}
		if (!i) corb_size = size;
		else rirb_size = size;
	}

	desc = KERNEL$ALLOC_CONTIG_AREA(DESC_SIZE, AREA_PHYSCONTIG | AREA_DATA | (OSS_DMA_USE_64 ? AREA_PCIDMA64 : AREA_PCIDMA) | AREA_ALIGN, (unsigned long)__PAGE_CLUSTER_SIZE);
	if (__IS_ERR(desc)) {
		r = __PTR_ERR(desc);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE COMMAND BUFFER", dev_name);
		goto ret2;
	}

	if ((r = KERNEL$SET_MEMORY_PAT(KERNEL$VIRT_2_PHYS(desc), DESC_SIZE, PAT_UC))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T SET UNCACHED MEMORY MODE", dev_name);
		goto ret25;
	}

	u.v.ptr = (unsigned long)desc;
	u.v.len = sizeof(struct az_desc);
	u.v.vspace = &KERNEL$VIRTUAL;
	if (!OSS_DMA_USE_64) {
		VDMA dma;
		dma.spl = SPL_X(SPL_ZERO);
		RAISE_SPL(SPL_VSPACE);
		desc_unlock_32 = KERNEL$VIRTUAL.op->vspace_dmalock(&u.v, PF_READ | PF_WRITE, &dma);
		desc_dmaaddr = dma.ptr;
	} else {
		VDMA64 dma;
		dma.spl = SPL_X(SPL_ZERO);
		RAISE_SPL(SPL_VSPACE);
		desc_unlock_64 = KERNEL$VIRTUAL.op->vspace_dma64lock(&u.v, PF_READ | PF_WRITE, &dma);
		desc_dmaaddr = dma.ptr;
	}
	
	AZ_SETUP_BUFFERS();

	intctl = 0;
	AZ_IRQ_AST.fn = AZ_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(irq, &AZ_IRQ_CTRL, IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &AZ_IRQ_AST, dev_name)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret3;
	}
	RAISE_SPL(SPL_SND);
	intctl = AZ_INTCTL_GIE | AZ_INTCTL_CIE;
	AZ_WRITE_32(AZ_INTCTL, intctl);
	LOWER_SPL(SPL_ZERO);

	KERNEL$SLEEP(JIFFIES_PER_SECOND / 50 + 1);	/* ICH 7 needs this */

	r = 0;
	/* it is already known that statests != 0 */
	do {
		int rr;
		codec = __BSF(statests);
		RAISE_SPL(SPL_SND);
		rr = HDA_CODEC_INIT(codec);
		LOWER_SPL(SPL_ZERO);
		if (!rr) goto codec_ok;
		if (!r) r = rr;
	} while (statests &= ~(1 << codec));
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INIT HDA CODEC", dev_name);
	goto ret4;

	codec_ok:

	OSS_INIT();
	pages = MAX_DESCS;
	page_size = memory / MAX_DESCS;
	while (page_size < __PAGE_CLUSTER_SIZE) {
		page_size <<= 1;
		pages >>= 1;
	}
	if ((r = OSS_ALLOC_DMAPAGES(pages, page_size))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC DMA PAGES: %s", dev_name, strerror(-r));
		goto ret5;
	}

	for (i = 0; i < OSS_N_STREAMS; i++) {
		INIT_TIMER(&HDA_TIMEOUT[i]);
		HDA_TIMEOUT[i].fn = HDA_TIMEOUT_FN;
		SET_TIMER_NEVER(&HDA_TIMEOUT[i]);
	}

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(u.gstr, pci_id));
	_printf("%s: MEM %"__64_format"X, IRQ %d\n", dev_name, PCI$READ_MEM_RESOURCE(pci_id, 0, NULL), irq);

	r = KERNEL$REGISTER_DEVICE(dev_name, "HDA.SYS", 0, NULL, OSS_SET_ROOT, NULL, NULL, NULL, HDA_UNLOAD, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret6;
	}

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, dev_name);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	ret6:
	OSS_FREE_DMAPAGES();
	ret5:
	HDA_CODEC_DESTROY();
	ret4:
	KERNEL$MASK_IRQ(AZ_IRQ_CTRL);
	AZ_WRITE_32(AZ_INTCTL, 0);
	KERNEL$RELEASE_IRQ(AZ_IRQ_CTRL, IRQ_REQUEST_MASKED | IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &AZ_IRQ_AST);
	ret3:
	if (!OSS_DMA_USE_64) desc_unlock_32(desc_dmaaddr);
	else desc_unlock_64(desc_dmaaddr);
	ret25:
	KERNEL$SET_MEMORY_PAT(KERNEL$VIRT_2_PHYS(desc), DESC_SIZE, PAT_WB);
	KERNEL$FREE_CONTIG_AREA(desc, DESC_SIZE);
	ret2:
	PCI$UNMAP_MEM_RESOURCE(pci_id, mem, AZ_REGSPACE);
	ret1:
	PCI$FREE_DEVICE(pci_id);
	ret0:
	return r;
}

static int HDA_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	OSS_STOP_ALL(-1);
	OSS_FREE_DMAPAGES();
	HDA_CODEC_DESTROY();
	KERNEL$MASK_IRQ(AZ_IRQ_CTRL);
	AZ_WRITE_32(AZ_INTCTL, 0);
	KERNEL$RELEASE_IRQ(AZ_IRQ_CTRL, IRQ_REQUEST_MASKED | IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &AZ_IRQ_AST);
	if (!OSS_DMA_USE_64) desc_unlock_32(desc_dmaaddr);
	else desc_unlock_64(desc_dmaaddr);
	KERNEL$SET_MEMORY_PAT(KERNEL$VIRT_2_PHYS(desc), DESC_SIZE, PAT_WB);
	KERNEL$FREE_CONTIG_AREA(desc, DESC_SIZE);
	PCI$UNMAP_MEM_RESOURCE(pci_id, mem, AZ_REGSPACE);
	PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
