#include <SPAD/PCI.H>
#include <SPAD/PCI_IDS.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/IO.H>
#include <SPAD/TIMER.H>
#include <SYS/SOUNDCARD.H>
#include <SPAD/IRQ.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>

#include "AC97.H"

#include "I810REG.H"

#define N_DMADESCS_BITS		5
#define N_DMADESCS		(1 << N_DMADESCS_BITS)
#define RUN_SIZE_BITS		(__unlikely(i810_flags & CAP_SIS_BYTE_DESCS) ? 15 : 16)
#define RUN_SIZE		(1 << RUN_SIZE_BITS)
#define MAX_RUN_SIZE		(1 << 16)

#define OSS_N_STREAMS		2
#define OSS_DEFINE_TIMEOUT
#define OSS_DEFINE_FRAGINTR
#define OSS_IS_DUPLEX		1
#define OSS_PCI_DMA
#define OSS_PRECISE_TIMING
#define MIN_CHANNELS		1
#define MAX_CHANNELS		1
#define MIN_FRAGS		(__likely(has_intr) ? 2 : 4)
#define MAX_FRAGS		N_DMADESCS

static const char FORMATS[_FMT_MAX + 1] = { 0, 0, 0, 0, 1, 0, 0, 0,
					    0, 0, 0, 0, 0, 0, 0, 0 };

int min_fragsize;

#define DEFAULT_FORMAT		_FMT_S16_LE
#define DEFAULT_MIN_FRAGSIZE	32
#define MIN_FRAGSIZE		(__likely(has_intr) ? DEFAULT_MIN_FRAGSIZE : min_fragsize)
#define SPL_SND			SPL_AC97
#define MIN_RATE		1
#define MAX_RATE		48000

#define NEED_VALIDATE_FRAGS

static __finline__ void VALIDATE_FRAGS(unsigned *p)
{
	*p = 1 << __BSR(*p | 1);
}

#include "OSS.H"

/* based on and partly copied from linux-2.4.26/drivers/sound/i810_audio.c
	Alan Cox <alan@redhat.com>
	Zach Brown <zab@redhat.com>
	Jaroslav Kysela <perex@suse.cz>
	Thomas Sailer <sailer@ife.ee.ethz.ch>
*/

#define CAP_MMIO			0x10
#define CAP_20BIT_AUDIO_SUPPORT		0x20
#define CAP_SIS_IGNORE_CODEC_ERRORS	0x40
#define CAP_SIS_BYTE_DESCS		0x80
#define CAP_N_CODECS			0x03

#ifndef PCI_DEVICE_ID_INTEL_82801
#define PCI_DEVICE_ID_INTEL_82801	0x2415
#endif
#ifndef PCI_DEVICE_ID_INTEL_82901
#define PCI_DEVICE_ID_INTEL_82901	0x2425
#endif
#ifndef PCI_DEVICE_ID_INTEL_ICH2
#define PCI_DEVICE_ID_INTEL_ICH2	0x2445
#endif
#ifndef PCI_DEVICE_ID_INTEL_ICH3
#define PCI_DEVICE_ID_INTEL_ICH3	0x2485
#endif
#ifndef PCI_DEVICE_ID_INTEL_ICH4
#define PCI_DEVICE_ID_INTEL_ICH4	0x24c5
#endif
#ifndef PCI_DEVICE_ID_INTEL_ICH5
#define PCI_DEVICE_ID_INTEL_ICH5	0x24d5
#endif
#ifndef PCI_DEVICE_ID_INTEL_ICH6_3
#define PCI_DEVICE_ID_INTEL_ICH6_3	0x266e
#endif
#ifndef PCI_DEVICE_ID_INTEL_ICH7
#define PCI_DEVICE_ID_INTEL_ICH7	0x27de
#endif
#ifndef PCI_DEVICE_ID_INTEL_440MX
#define PCI_DEVICE_ID_INTEL_440MX	0x7195
#endif
#ifndef PCI_DEVICE_ID_INTEL_ESB_5
#define PCI_DEVICE_ID_INTEL_ESB_5	0x25a6
#endif
#ifndef PCI_DEVICE_ID_SI_7012
#define PCI_DEVICE_ID_SI_7012		0x7012
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_MCP1_AUDIO
#define PCI_DEVICE_ID_NVIDIA_MCP1_AUDIO 0x01b1
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_MCP2_AUDIO
#define PCI_DEVICE_ID_NVIDIA_MCP2_AUDIO 0x006a
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_MCP3_AUDIO
#define PCI_DEVICE_ID_NVIDIA_MCP3_AUDIO 0x00da
#endif
#ifndef PCI_DEVICE_ID_NVIDIA_CK8S_AUDIO
#define PCI_DEVICE_ID_NVIDIA_CK8S_AUDIO 0x00ea
#endif
#ifndef PCI_DEVICE_ID_AMD_768_AUDIO
#define PCI_DEVICE_ID_AMD_768_AUDIO	0x7445
#endif
#ifndef PCI_DEVICE_ID_AMD_8111_AC97
#define PCI_DEVICE_ID_AMD_8111_AC97	0x746d
#endif

static const struct pci_id_s devices[] = {
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH 82801AA", 1 },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82901, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH 82901AB", 1 },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_440MX, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL 440MX", 1 },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH2, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH2", 1 },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH3, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH3", 2 },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH4, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH4", 3 | CAP_MMIO | CAP_20BIT_AUDIO_SUPPORT },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH5", 3 | CAP_MMIO | CAP_20BIT_AUDIO_SUPPORT },
	{PCI_VENDOR_ID_SI, PCI_DEVICE_ID_SI_7012, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SIS 7012", 2 | CAP_SIS_IGNORE_CODEC_ERRORS | CAP_SIS_BYTE_DESCS },
	{PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_MCP1_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "NVIDIA NFORCE", 2 },
	{PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_MCP2_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "NVIDIA NFORCE2", 2 },
	{PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_MCP3_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "NVIDIA NFORCE3", 2 },
	{PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_CK8S_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "NVIDIA CK8S", 2 },
	{PCI_VENDOR_ID_AMD, PCI_DEVICE_ID_AMD_768_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "AMD 768", 2 },
	{PCI_VENDOR_ID_AMD, PCI_DEVICE_ID_AMD_8111_AC97, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "AMD 8111", 3 | CAP_MMIO },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ESB_5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL 6300ESB", 3 | CAP_MMIO | CAP_20BIT_AUDIO_SUPPORT },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH6_3, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH6", 3 | CAP_MMIO | CAP_20BIT_AUDIO_SUPPORT },
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH7, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH7", 3 | CAP_MMIO | CAP_20BIT_AUDIO_SUPPORT },
	{0},
};

static char dev_name[__MAX_STR_LEN];
static char *chip_name;
static unsigned long i810_flags;

static io_t ac97io, io;

static IO_RANGE ac97_range, range;

static pci_id_t pci_id;

static unsigned has_intr;

#define IOCE		(CR_IOCE * has_intr)

static __finline__ unsigned read_8(unsigned reg)
{
	return io_inb(io + reg);
}

static __finline__ unsigned read_16(unsigned reg)
{
	return io_inw(io + reg);
}

static __finline__ unsigned read_32(unsigned reg)
{
	return io_inl(io + reg);
}

static __finline__ void write_8(unsigned reg, unsigned val)
{
	io_outb(io + reg, val);
}

static __finline__ void write_16(unsigned reg, unsigned val)
{
	io_outw(io + reg, val);
}

static __finline__ void write_32(unsigned reg, unsigned val)
{
	io_outl(io + reg, val);
}

static int read_ac97(unsigned reg)
{
	int cnt = 0;
	unsigned v, sta;
	while (__unlikely(read_8(ACC_SEMA) & ACC_SEMA_CAS)) {
		KERNEL$UDELAY(10);
		if (++cnt >= 100) {
			write_8(ACC_SEMA, 0);
			break;
		}
	}
	v = io_inw(ac97io + reg);
	sta = read_32(GLOB_STA);
	/* on SiS I sometimes get this error, but otherwise if I comment it out,
	   it works fine ... */
	if (__unlikely(sta & GLOB_STA_RCS) && __unlikely(!(i810_flags & CAP_SIS_IGNORE_CODEC_ERRORS))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC ACCESS TIMED OUT");
		write_32(GLOB_STA, sta & ~GLOB_STA_RCS);
		return -1;
	}
	return v;
}

static void write_ac97(unsigned reg, int val)
{
	int cnt = 0;
	while (__unlikely(read_8(ACC_SEMA) & ACC_SEMA_CAS)) {
		KERNEL$UDELAY(10);
		if (++cnt >= 100) {
			write_8(ACC_SEMA, 0);
			break;
		}
	}
	io_outw(ac97io + reg, val);
}

__finline__ void SND_FIXUP_PARAMS(HANDLE *h, int open_flags)
{
	AC97_FIXUP_RATE(h, open_flags);
}

#include "AC97.I"

static const HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	OSS_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	OSS_INSTANTIATE,	/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

#include "OSS.I"

typedef struct {
	__u32 pointer;
	__u16 length;
	__u16 flags;
} DMADESC;

#define DMA_IOC		0x8000
#define DMA_BUP		0x4000

static char play_desc_[sizeof(DMADESC) * (N_DMADESCS + 1) - 1];
static char rec_desc_[sizeof(DMADESC) * (N_DMADESCS + 1) - 1];
static DMADESC *PLAY_DESC;
static DMADESC *REC_DESC;
static __u32 PLAY_DEVADDR;
static __u32 REC_DEVADDR;
static vspace_dmaunlock_t *PLAY_UNMAP = NULL;
static vspace_dmaunlock_t *REC_UNMAP = NULL;
static unsigned PLAY_LAST_CIV;
static unsigned REC_LAST_CIV;
static int pa;
static int ra;
static jiffies_t I810_PLAYBACK_JIFFIES;
static DECL_TIMER(I810_PLAYBACK_TIMEOUT);
static jiffies_t I810_RECORD_JIFFIES;
static DECL_TIMER(I810_RECORD_TIMEOUT);

static void SET_LOOP_LVI(void);

void SND_START(int record)
{
	VDESC desc;
	VDMA dma;
	int frags, fragsize_bits;
	int desc_len_bits;
	int i;
	u_jiffies_lo_t j, jj;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("SND_START AT SPL %08X", KERNEL$SPL);
	SND_STOP(record);
	if (__likely(!record)) {
		AC97_SET_PLAYBACK_RATE(oss_rate[0]);
		frags = oss_frags[0];
		fragsize_bits = oss_fragsize_bits[0];
	} else {
		AC97_SET_RECORD_RATE(oss_rate[1]);
		frags = oss_frags[1];
		fragsize_bits = oss_fragsize_bits[1];
	}
	desc_len_bits = __BSR((frags << fragsize_bits) >> N_DMADESCS_BITS);
	for (i = 0; i < N_DMADESCS; i++) {
		DMADESC *d = (__likely(!record) ? PLAY_DESC : REC_DESC) + i;
		d->length = __16CPU2LE(1 << (desc_len_bits - !(i810_flags & CAP_SIS_BYTE_DESCS)));
		d->pointer = __32CPU2LE(DATA_PAGES[record][i << desc_len_bits >> RUN_SIZE_BITS].device + ((i << desc_len_bits) & (RUN_SIZE - 1)));
		d->flags = __16CPU2LE(0);
		if (__likely(has_intr) && __likely((i & ((1 << (N_DMADESCS_BITS - __BSR(frags))) - 1)) == ((1 << (N_DMADESCS_BITS - __BSR(frags))) - 1))) d->flags |= __16CPU2LE(DMA_IOC);
	}
	if (__likely(!record)) {
		desc.ptr = (unsigned long)PLAY_DESC;
		desc.len = sizeof(DMADESC) * N_DMADESCS;
		desc.vspace = &KERNEL$VIRTUAL;
		dma.spl = SPL_X(SPL_SND);
		RAISE_SPL(SPL_VSPACE);
		PLAY_UNMAP = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ, &dma);
		if (__unlikely(dma.len != desc.len))
			KERNEL$SUICIDE("SND_START: CAN'T DMALOCK BUFFER LIST");
		PLAY_DEVADDR = dma.ptr;
		write_8(PO_CR, CR_RR);
		j = KERNEL$GET_JIFFIES_LO();
		while (jj = KERNEL$GET_JIFFIES_LO(), read_8(PO_CR) & CR_RR) if (jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 1000)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "PLAYBACK RESET TIMEOUT");
			break;
		}
		write_16(PO_SR, SR_LVBCI | SR_BCIS | SR_FIFOE);
		write_32(PO_BDBAR, PLAY_DEVADDR);
		if (__unlikely(OSS_STREAM_FLAGS[0] & OSS_LOOP)) SET_LOOP_LVI();
		else SND_SUBMIT_FRAGMENT(0, oss_buf_fragments[0]);
		write_8(PO_CR, IOCE | CR_RPBM);
		PLAY_LAST_CIV = 0;
		pa = 0;
		OSS_STREAM_FLAGS[0] |= OSS_RUNNING;
		KERNEL$DEL_TIMER(&I810_PLAYBACK_TIMEOUT);
		I810_PLAYBACK_JIFFIES = __likely(has_intr) ? OSS_STREAM_TIMEOUT(0) : OSS_STREAM_FRAGINTR(0);
		KERNEL$SET_TIMER(I810_PLAYBACK_JIFFIES, &I810_PLAYBACK_TIMEOUT);
	} else {
		desc.ptr = (unsigned long)REC_DESC;
		desc.len = sizeof(DMADESC) * N_DMADESCS;
		desc.vspace = &KERNEL$VIRTUAL;
		dma.spl = SPL_X(SPL_SND);
		RAISE_SPL(SPL_VSPACE);
		REC_UNMAP = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ, &dma);
		if (__unlikely(dma.len != desc.len))
			KERNEL$SUICIDE("SND_START: CAN'T DMALOCK BUFFER LIST");
		REC_DEVADDR = dma.ptr;
		write_8(PI_CR, CR_RR);
		j = KERNEL$GET_JIFFIES_LO();
		while (jj = KERNEL$GET_JIFFIES_LO(), read_8(PI_CR) & CR_RR) if (jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 1000)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "RECORD RESET TIMEOUT");
			break;
		}
		write_16(PI_SR, SR_LVBCI | SR_BCIS | SR_FIFOE);
		write_32(PI_BDBAR, REC_DEVADDR);
		SND_SUBMIT_FRAGMENT(1, oss_frags[1] - 1);
		/*write_8(PI_LVI, N_DMADESCS - 1);*/
		write_8(PI_CR, IOCE | CR_RPBM);
		REC_LAST_CIV = 0;
		ra = 0;
		OSS_STREAM_FLAGS[1] |= OSS_RUNNING;
		KERNEL$DEL_TIMER(&I810_RECORD_TIMEOUT);
		I810_RECORD_JIFFIES = __likely(has_intr) ? OSS_STREAM_TIMEOUT(1) : OSS_STREAM_FRAGINTR(1);
		KERNEL$SET_TIMER(I810_RECORD_JIFFIES, &I810_RECORD_TIMEOUT);
	}
}

void SND_STOP(int record)
{
	if (__likely(!record)) {
		write_8(PO_CR, 0);
		write_8(PO_CR, CR_RR);
		write_16(PO_SR, SR_LVBCI | SR_BCIS | SR_FIFOE);
		if (__likely(PLAY_UNMAP != NULL)) PLAY_UNMAP(PLAY_DEVADDR), PLAY_UNMAP = NULL;
		KERNEL$DEL_TIMER(&I810_PLAYBACK_TIMEOUT);
		SET_TIMER_NEVER(&I810_PLAYBACK_TIMEOUT);
	} else {
		write_8(PI_CR, 0);
		write_8(PI_CR, CR_RR);
		write_16(PI_SR, SR_LVBCI | SR_BCIS | SR_FIFOE);
		if (__likely(REC_UNMAP != NULL)) REC_UNMAP(REC_DEVADDR), REC_UNMAP = NULL;
		KERNEL$DEL_TIMER(&I810_RECORD_TIMEOUT);
		SET_TIMER_NEVER(&I810_RECORD_TIMEOUT);
	}
}

void SND_PAUSE(int record)
{
	if (__likely(!record)) {
		write_8(PO_CR, IOCE);
		KERNEL$DEL_TIMER(&I810_PLAYBACK_TIMEOUT);
		SET_TIMER_NEVER(&I810_PLAYBACK_TIMEOUT);
	} else {
		write_8(PO_CR, IOCE);
		KERNEL$DEL_TIMER(&I810_RECORD_TIMEOUT);
		SET_TIMER_NEVER(&I810_RECORD_TIMEOUT);
	}
}

void SND_RESUME(int record)
{
	if (__likely(!record)) {
		write_8(PO_CR, IOCE | CR_RPBM);
		KERNEL$DEL_TIMER(&I810_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(I810_PLAYBACK_JIFFIES, &I810_PLAYBACK_TIMEOUT);
	} else {
		write_8(PO_CR, IOCE | CR_RPBM);
		KERNEL$DEL_TIMER(&I810_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(I810_RECORD_JIFFIES, &I810_RECORD_TIMEOUT);
	}
}

void SND_SUBMIT_FRAGMENT(int record, int frag)
{
	if (!record) {
		write_8(PO_LVI, ((frag << (N_DMADESCS_BITS - __BSR(oss_frags[0]))) + ((1 << (N_DMADESCS_BITS - __BSR(oss_frags[0]))) - 1)) & 0x1f);
	} else {
		write_8(PI_LVI, ((frag << (N_DMADESCS_BITS - __BSR(oss_frags[1]))) + ((1 << (N_DMADESCS_BITS - __BSR(oss_frags[1]))) - 1)) & 0x1f);
	}
}

static void SET_LOOP_LVI(void)
{
	/*__debug_printf("civ %d, lvi %d, new lvi %d\n", read_8(PO_CIV), read_8(PO_LVI), (read_8(PO_CIV) - (1 << (N_DMADESCS_BITS - __BSR(oss_playback_frags)))));*/
	write_8(PO_LVI, (read_8(PO_CIV) - (1 << (N_DMADESCS_BITS - __BSR(oss_frags[0])))));
}

static void I810_PLAYBACK_INTR(void)
{
	unsigned civ = read_8(PO_CIV);
	unsigned sr = read_16(PO_SR);
	unsigned frags;
	if (__unlikely(pa) && __unlikely(((PLAY_LAST_CIV + 1) & 0x1f) == civ)) {
		civ = PLAY_LAST_CIV;
		goto sc;
	}
	pa = 0;
	if (__likely(has_intr)) {
		if (__unlikely(sr & SR_LVBCI)) civ = (civ + 1) & 0x1f, pa = 1;
	} else {
		if (civ == (read_8(PO_LVI) & 0x1f)) civ = (civ + 1) & 0x1f, pa = 1;
	}
	sc:
	frags = ((civ >> (N_DMADESCS_BITS - __BSR(oss_frags[0]))) - (PLAY_LAST_CIV >> (N_DMADESCS_BITS - __BSR(oss_frags[0])))) & (oss_frags[0] - 1);
	PLAY_LAST_CIV = civ;
	/*__debug_printf("intr: civ %d, lvi %d, lvb %d, oss_playback_frags %d, oss_playback_buf_fragments %d, frags %d, intrtime %u\n", civ, read_8(PO_LVI) & 0x1f, !!(sr & SR_LVBCI), oss_playback_frags, oss_playback_buf_fragments, frags, OSS_PLAYBACK_INTRTIME);*/
	write_16(PO_SR, SR_LVBCI | SR_BCIS | SR_FIFOE);
	while (frags--) {
		if (__unlikely(OSS_STREAM_INTR(0))) return;
	}
	if (__unlikely(OSS_STREAM_FLAGS[0] & OSS_LOOP)) {
		SET_LOOP_LVI();
	}
	KERNEL$DEL_TIMER(&I810_PLAYBACK_TIMEOUT);
	KERNEL$SET_TIMER(I810_PLAYBACK_JIFFIES, &I810_PLAYBACK_TIMEOUT);
}

static void I810_RECORD_INTR(void)
{
	unsigned civ = read_8(PI_CIV);
	unsigned sr = read_16(PI_SR);
	unsigned frags;
	if (__unlikely(ra) && __unlikely(((REC_LAST_CIV + 1) & 0x1f) == civ)) {
		civ = REC_LAST_CIV;
		goto sc;
	}
	ra = 0;
	if (__likely(has_intr)) {
		if (__unlikely(sr & SR_LVBCI)) civ = (civ + 1) & 0x1f, ra = 1;
	} else {
		if (civ == (read_8(PI_LVI) & 0x1f)) civ = (civ + 1) & 0x1f, ra = 1;
	}
	sc:
	frags = ((civ >> (N_DMADESCS_BITS - __BSR(oss_frags[1]))) - (REC_LAST_CIV >> (N_DMADESCS_BITS - __BSR(oss_frags[1])))) & (oss_frags[1] - 1);
	REC_LAST_CIV = civ;
	/*__debug_printf("intr: civ %d, rec_frags %d, rec_buf_fragments %d, frags %d, lvbci %d, lvi %d\n", civ, oss_record_frags, oss_record_buf_fragments, frags, !!(sr & SR_LVBCI), read_8(PI_LVI));*/
	write_16(PI_SR, SR_LVBCI | SR_BCIS | SR_FIFOE);
	while (frags--) {
		if (__unlikely(OSS_STREAM_INTR(1))) return;
	}
	KERNEL$DEL_TIMER(&I810_RECORD_TIMEOUT);
	KERNEL$SET_TIMER(I810_RECORD_JIFFIES, &I810_RECORD_TIMEOUT);
}

static void I810_PLAYBACK_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	SET_TIMER_NEVER(t);
	if (__likely(!has_intr)) {
		I810_PLAYBACK_INTR();
		return;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "PLAYBACK INTERRUPT TIMEOUT");
	OSS_STOP(0);
}

static void I810_RECORD_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	SET_TIMER_NEVER(t);
	if (__likely(!has_intr)) {
		I810_RECORD_INTR();
		return;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "RECORD INTERRUPT TIMEOUT");
	OSS_STOP(1);
}

unsigned SND_GET_PTR(int record)
{
	if (!record) {
		int shift;
		unsigned civ, left;
		unsigned cnt = 0;
		again_p:
		civ = read_8(PO_CIV);
		left = read_16(PO_PICB);
		if (__unlikely(civ != read_8(PO_CIV))) if (cnt++ < 5) goto again_p;
		civ &= (N_DMADESCS - 1);
		shift = (i810_flags & CAP_SIS_BYTE_DESCS) ? 0 : 1;
		return ((civ << __BSR(__16LE2CPU(PLAY_DESC[0].length))) + (__16LE2CPU(PLAY_DESC[0].length) - left)) << shift;
	} else {
		int shift;
		unsigned civ, left;
		unsigned cnt = 0;
		again_r:
		civ = read_8(PI_CIV);
		left = read_16(PI_PICB);
		if (__unlikely(civ != read_8(PI_CIV))) if (cnt++ < 5) goto again_r;
		civ &= (N_DMADESCS - 1);
		shift = (i810_flags & CAP_SIS_BYTE_DESCS) ? 0 : 1;
		return ((civ << __BSR(__16LE2CPU(REC_DESC[0].length))) + (__16LE2CPU(REC_DESC[0].length) - left)) << shift;
	}
}

static AST I810_IRQ_AST;
static IRQ_HANDLE *I810_IRQ_CTRL;

static DECL_IRQ_AST(I810_IRQ, SPL_SND, AST)
{
	unsigned stat = read_32(GLOB_STA);
	if (__likely(stat & GLOB_STA_POINT)) I810_PLAYBACK_INTR();
	if (__unlikely(stat & GLOB_STA_PIINT)) I810_RECORD_INTR();
	if (__unlikely(stat & (GLOB_STA_MINT | GLOB_STA_GSCI | GLOB_STA_PRI | GLOB_STA_SRI))) {
		write_8(MC_CR, CR_RR);
		write_32(GLOB_STA, stat & (GLOB_STA_GSCI | GLOB_STA_PRI | GLOB_STA_SRI));
	}
	KERNEL$UNMASK_IRQ(I810_IRQ_CTRL);
	RETURN;
}

static void *lnte, *dlrq;
static int I810_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	int pages = -1;
	int irq;
	int nointr = 0;
	const char * const *arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	const char *opt, *optend, *str;
	int r;
	unsigned v;
	char gstr[__MAX_STR_LEN];
	has_intr = 0;
	static const struct __param_table params[3] = {
		"MEMORY", __PARAM_INT, MAX_RUN_SIZE, N_DMADESCS * MAX_RUN_SIZE + 1,
		"NOIRQ", __PARAM_BOOL, ~0, 1,
		NULL, 0, 0, 0,
	};
	void *vars[3];
	vars[0] = &pages;
	vars[1] = &nointr;
	vars[2] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "I810: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (PCI$FIND_DEVICE(devices, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &i810_flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "I810: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	pci_id = id;
	_snprintf(dev_name, sizeof(dev_name), "SND$I810@" PCI_ID_FORMAT, id);
	if (pages == -1) pages = RUN_SIZE * N_DMADESCS;
	if (pages & (pages - 1) || pages > RUN_SIZE * N_DMADESCS) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "I810: BAD MEMORY SIZE (MUST BE POWER OF 2 AND LESS OR EQUAL THAN %u)", RUN_SIZE * N_DMADESCS);
		r = -EBADSYN;
		goto ret1;
	}
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	ac97io = PCI$READ_IO_RESOURCE(id, 0);
	io = PCI$READ_IO_RESOURCE(id, 1);
	irq = PCI$READ_INTERRUPT_LINE(id);
	if (!io || !ac97io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	range.start = io;
	range.len = I810_REGSPACE;
	range.name = dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, range.start, range.start + range.len - 1, strerror(-r));
		goto ret1;
	}
	ac97_range.start = ac97io;
	ac97_range.len = AC97_REGSPACE;
	ac97_range.name = dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&ac97_range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, ac97_range.start, ac97_range.start + ac97_range.len - 1, strerror(-r));
		goto ret_11;
	}
	write_32(GLOB_CNT, 0);
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 50 + 1);
	write_32(GLOB_CNT, GLOB_CNT_COLD_RESET);
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 50 + 1);
	v = read_32(GLOB_CNT);
	if ((v & 0x1f) != GLOB_CNT_COLD_RESET) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "CAN'T RESET");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T RESET", dev_name);
		r = -ENXIO;
		goto ret_12;
	}
	r = AC97_INIT();
	if (r) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE AC97 CODEC", dev_name);
		goto ret_12;
	}
	v = read_32(GLOB_STA);
	if (!(v & GLOB_STA_PCR)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "PRIMARY CODEC NOT READY");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE AC97 CODEC", dev_name);
		r = -ENXIO;
		goto ret_12;
	}
	pages >>= RUN_SIZE_BITS;
	OSS_INIT();
	if ((r = OSS_ALLOC_DMAPAGES(pages, RUN_SIZE))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC DMA PAGES: %s", dev_name, strerror(-r));
		goto ret_12;
	}

	PLAY_DESC = (DMADESC *)(((unsigned long)play_desc_ + 7) & ~7);
	REC_DESC = (DMADESC *)(((unsigned long)rec_desc_ + 7) & ~7);

	I810_PLAYBACK_TIMEOUT.fn = I810_PLAYBACK_TIMEOUT_FN;
	SET_TIMER_NEVER(&I810_PLAYBACK_TIMEOUT);
	I810_RECORD_TIMEOUT.fn = I810_RECORD_TIMEOUT_FN;
	SET_TIMER_NEVER(&I810_RECORD_TIMEOUT);

	I810_IRQ_AST.fn = I810_IRQ;
	if (nointr || ((r = KERNEL$REQUEST_IRQ(irq, &I810_IRQ_CTRL, IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &I810_IRQ_AST, dev_name)) < 0)) {
#if 0
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret2;
#endif
		int i;
		unsigned maxrate = MAX_RATE;
		has_intr = 0;
		i = PAGESIZE_BITS - 1;
		while (1) {
			if (i < 0) break;
			if ((1 << i) < DEFAULT_MIN_FRAGSIZE) break;
			if (RAW_FRAGINTR(i, _FMT_S16_LE, 1, maxrate) < 0) break;
			i--;
		}
		i++;
		min_fragsize = 1 << i;
		/*__debug_printf("max rate %u, min fragsize %u\n", maxrate, min_fragsize);*/
	} else {
		has_intr = 1;
	}

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, pci_id));
	_printf("%s: IO "IO_FORMAT", AC97IO "IO_FORMAT, dev_name, io, ac97io);
	if (has_intr) _printf(", IRQ %d", irq);
	_printf("\n");

	r = KERNEL$REGISTER_DEVICE(dev_name, "I810.SYS", 0, NULL, OSS_SET_ROOT, NULL, NULL, NULL, I810_UNLOAD, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret3;
	}

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, dev_name);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	ret3:
	if (has_intr) KERNEL$RELEASE_IRQ(I810_IRQ_CTRL, IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &I810_IRQ_AST);
#if 0
	ret2:
#endif
	OSS_FREE_DMAPAGES();
	ret_12:
	KERNEL$UNREGISTER_IO_RANGE(&ac97_range);
	ret_11:
	KERNEL$UNREGISTER_IO_RANGE(&range);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int I810_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	if (has_intr) KERNEL$MASK_IRQ(I810_IRQ_CTRL);
	OSS_STOP_ALL(-1);
	if (has_intr) KERNEL$RELEASE_IRQ(I810_IRQ_CTRL, IRQ_REQUEST_MASKED | IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &I810_IRQ_AST);
	OSS_FREE_DMAPAGES();
	KERNEL$UNREGISTER_IO_RANGE(&ac97_range);
	KERNEL$UNREGISTER_IO_RANGE(&range);
	PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
