#include "OSS.H"

#include <ERRNO.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>
#include <SYS/TYPES.H>
#include <SYS/PARAM.H>
#include <ENDIAN.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <VALUES.H>

#include <SYS/SOUNDCARD.H>

#define DEFAULT_SUBDIVIDE	3000	/* 3 sec */
#define MAX_SUBDIVIDE_FRAGS	8

#define O_LOOP_FLAG	_O_AUX1

#define SELECT_PLAYBACK_STREAM(file, stream, fail)		\
{								\
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))		\
		KERNEL$SUICIDE("SELECT_PLAYBACK_STREAM AT SPL %08X", KERNEL$SPL);								\
	if (__unlikely(!((file)->flags & _O_KRNL_WRITE))) {	\
		fail;						\
		__unreachable_code();				\
	}							\
	(stream) = 0;						\
}

#define SELECT_RECORD_STREAM(file, stream, fail)		\
{								\
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))		\
		KERNEL$SUICIDE("SELECT_RECORD_STREAM AT SPL %08X", KERNEL$SPL);									\
	if (__unlikely(!((file)->flags & _O_KRNL_READ))) {	\
		fail;						\
		__unreachable_code();				\
	}							\
	(stream) = 1;						\
}

/*
 * ^RATE=12345
 * ^CHANNELS=1, 2
 * ^FORMAT=0 - 14
 * ^FRAGSIZE=4 - 16
 * ^FRAGS=2 - 4096
 */

static void *SET_RATE(HANDLE *h, unsigned rate, int file_flags)
{
#ifdef MIN_RATE
	if (__unlikely(rate < MIN_RATE)) rate = MIN_RATE;
#endif
#ifdef MAX_RATE
	if (__unlikely(rate > MAX_RATE)) rate = MAX_RATE;
#endif
	if (__unlikely(rate > FORMAT_MAXRATE)) rate = FORMAT_MAXRATE;
#ifdef OSS_FIXED_RATES
	if (OSS_FIXED_RATES) {
		int i, j = -1, mindiff = MAXINT;
		for (i = 0; RATELIST[i]; i++) if (__likely((unsigned)RATELIST[i] <= FORMAT_MAXRATE)
#ifdef RATELIST_RESTRICT
			&& __likely(!RATELIST_RESTRICT(h, i))
#endif
			) {
			int diff = abs(rate - RATELIST[i]);
			if (__likely(diff < mindiff)) j = i, mindiff = diff;
		}
		if (__unlikely(j == -1)) return __ERR_PTR(-EINVAL);
		FORMAT_SETRATE(h, j);
	} else
#endif
	{
		FORMAT_SETRATE(h, rate);
	}
	SND_FIXUP_PARAMS(h, file_flags);
	return NULL;
}

static void *SET_CHANNELS(HANDLE *h, unsigned ch, int file_flags)
{
	if (__unlikely(ch < MIN_CHANNELS)) ch = MIN_CHANNELS;
	if (__unlikely(ch > MIN(MAX_CHANNELS,FORMAT_MAXCHANNELS))) ch = MIN(MAX_CHANNELS,FORMAT_MAXCHANNELS);
	FORMAT_SETCHANNELS(h, ch);
	SND_FIXUP_PARAMS(h, file_flags);
	return NULL;
}

static void *SET_FORMAT(HANDLE *h, unsigned f, int file_flags)
{
	if (__unlikely(f > MIN(_FMT_MAX,FORMAT_MAXFORMAT)) || __unlikely(!FORMATS[f])) {
		if (f == _FMT_MU_LAW || f == _FMT_A_LAW || f == _FMT_S8 || f == _FMT_IMA_ADPCM) f = _FMT_U8;
		else f = _FMT_S16_LE;
		if (__unlikely(!FORMATS[f])) f = DEFAULT_FORMAT;
	}
	FORMAT_SETFORMAT(h, f);
	SND_FIXUP_PARAMS(h, file_flags);
	return NULL;
}

static void *SET_FRAGSIZE(HANDLE *h, unsigned fragsize_bits)
{
	if (__unlikely(fragsize_bits >= sizeof(int) * 8 - 1)) fragsize_bits = sizeof(int) * 8 - 2;
	if (__unlikely((1 << fragsize_bits) < MIN_FRAGSIZE)) fragsize_bits = __BSR(MIN_FRAGSIZE);
	if (__unlikely(fragsize_bits > PAGESIZE_BITS)) fragsize_bits = PAGESIZE_BITS;
	if (N_PAGES == 1 && __unlikely(fragsize_bits > PAGESIZE_BITS - 1)) fragsize_bits = PAGESIZE_BITS - 1;
	if (__unlikely(fragsize_bits > FORMAT_MAXFRAGSIZE)) fragsize_bits = FORMAT_MAXFRAGSIZE;
	FORMAT_SETFRAGSIZE(h, fragsize_bits);
	return NULL;
}

static void *SET_FRAGS(HANDLE *h, unsigned frags)
{
	unsigned fragsize_bits, frags1;
	if (__unlikely(!FORMAT_FRAGSIZE(h))) SET_FRAGSIZE(h, PAGESIZE_BITS);
	fragsize_bits = FORMAT_FRAGSIZE(h);
	if (__unlikely(frags < MIN_FRAGS)) frags = MIN_FRAGS;
	if (__unlikely(frags > MIN(MAX_FRAGS,FORMAT_MAXFRAGS))) frags = MIN(MAX_FRAGS,FORMAT_MAXFRAGS);
	if (__unlikely(frags != frags << fragsize_bits >> fragsize_bits)) frags = MAXINT >> fragsize_bits;
	frags1 = frags;
	set_frags_again:
	if (__unlikely(frags << fragsize_bits > N_PAGES << PAGESIZE_BITS))
		frags = N_PAGES << (PAGESIZE_BITS - fragsize_bits);
	if (__unlikely(frags < MIN_FRAGS) && (1 << fragsize_bits) > MIN_FRAGSIZE) {
		frags = frags1;
		fragsize_bits--;
		FORMAT_SETFRAGSIZE(h, fragsize_bits);
		goto set_frags_again;
	}
#ifdef NEED_VALIDATE_FRAGS
	VALIDATE_FRAGS(&frags);
#endif
	FORMAT_SETFRAGS(h, frags);
	return NULL;
}

static void *SET_SUBDIVIDE(HANDLE *h, unsigned s)
{
	if (__unlikely(s > FORMAT_MAXSUBDIVIDE)) s = FORMAT_MAXSUBDIVIDE;
	FORMAT_SETSUBDIVIDE(h, s);
	return NULL;
}

void *OSS_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	char *s;
	unsigned long n;
	vspace_unmap_t *funmap;
	void *r;
	FFILE *f;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_LOOKUP AT SPL %08X", KERNEL$SPL);
	f = KERNEL$MAP_FILE_STRUCT(h, NULL, &funmap);
	if (__unlikely(!f))
		return (void *)1;
	s = strchr(str, '=');
	if (__unlikely(!s)) {
		if (__likely(!_strcasecmp(str, "^NONBLOCK"))) {
			FORMAT_SETNONBLOCK(h, 1);
			r = NULL;
			goto ret;
		}
		r = __ERR_PTR(-EBADMOD);
		goto ret;
	}
	if (__unlikely(__get_number(s + 1, s + strlen(s), 0, (long *)&n)) || __unlikely(n > MAXINT)) {
		if (!__strcasexcmp("^FORMAT", str, s)) {
			if (!_strcasecmp(s + 1, "MU-LAW")) n = _FMT_MU_LAW;
			else if (!_strcasecmp(s + 1, "A-LAW")) n = _FMT_A_LAW;
			else if (!_strcasecmp(s + 1, "S8")) n = _FMT_S8;
			else if (!_strcasecmp(s + 1, "U8")) n = _FMT_U8;
			else if (!_strcasecmp(s + 1, "S16")) n = _FMT_S16_LE;
			else if (!_strcasecmp(s + 1, "S16LE")) n = _FMT_S16_LE;
			else if (!_strcasecmp(s + 1, "S16BE")) n = _FMT_S16_BE;
			else if (!_strcasecmp(s + 1, "U16LE")) n = _FMT_U16_LE;
			else if (!_strcasecmp(s + 1, "U16BE")) n = _FMT_U16_BE;
			else if (!_strcasecmp(s + 1, "S32LE")) n = _FMT_S32_LE;
			else if (!_strcasecmp(s + 1, "S32BE")) n = _FMT_S32_BE;
			else if (!_strcasecmp(s + 1, "U32LE")) n = _FMT_U32_LE;
			else if (!_strcasecmp(s + 1, "U32BE")) n = _FMT_U32_BE;
			else if (!_strcasecmp(s + 1, "ADPCM")) n = _FMT_IMA_ADPCM;
			else if (!_strcasecmp(s + 1, "MPEG")) n = _FMT_MPEG;
			else if (!_strcasecmp(s + 1, "AC3")) n = _FMT_AC3;
			else {
				r = __ERR_PTR(-EINVAL);
				goto ret;
			}
			SET_FORMAT(h, n, f->flags);
			if (__unlikely(FORMAT_FORMAT(h) != n)) {
				r = __ERR_PTR(-EPROTO);
				goto ret;
			}
			r = NULL;
			goto ret;
		}
		r = __ERR_PTR(-EINVAL);
		goto ret;
	}
	if (!__strcasexcmp("^RATE", str, s)) {
		r = SET_RATE(h, n, f->flags);
		goto ret;
	}
	if (!__strcasexcmp("^CHANNELS", str, s)) {
		r = SET_CHANNELS(h, __likely(n != 0) ? n - 1 : 0, f->flags);
		goto ret;
	}
	if (!__strcasexcmp("^FORMAT", str, s)) {
		r = SET_FORMAT(h, n, f->flags);
		goto ret;
	}
	if (!__strcasexcmp("^FRAGSIZE", str, s)) {
		r = SET_FRAGSIZE(h, n);
		goto ret;
	}
	if (!__strcasexcmp("^FRAGS", str, s)) {
		r = SET_FRAGS(h, n);
		goto ret;
	}
	if (!__strcasexcmp("^SUBDIVIDE", str, s)) {
		r = SET_SUBDIVIDE(h, n);
		goto ret;
	}
	r = __ERR_PTR(-EBADMOD);
	goto ret;

	ret:
	funmap(f);
	return r;
}

void OSS_LOOKUP_IO(HANDLE *h, char *str, IORQ *rq, int open_flags)
{
	vspace_unmap_t *funmap;
	FFILE *f;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_LOOKUP_IO AT SPL %08X", KERNEL$SPL);
	f = KERNEL$MAP_FILE_STRUCT(h, rq, &funmap);
	if (__likely(!f)) return;
	funmap(f);
	CALL_IORQ_LSTAT_EXPR(rq, (IO_STUB *)rq->tmp1);
}

void *OSS_INSTANTIATE(HANDLE *h, IORQ *rq, int open_flags)
{
	int stream;
	FFILE *file;
	vspace_unmap_t *funmap;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_INSTANTIATE AT SPL %08X", KERNEL$SPL);
	if (__likely(!open_flags)) return NULL;
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)rq, &funmap);
	if (__unlikely(!file)) return (void *)3;
	SELECT_PLAYBACK_STREAM(file, stream, goto skip_play;);
	RAISE_SPL(SPL_SND);
	oss_stat_underruns[stream] = 0;
	if (__unlikely(OSS_STREAM_FLAGS[stream] & (OSS_BLOCK | OSS_LOOP))) {
		OSS_STOP(stream);
	}
	LOWER_SPL(SPL_OSS);
	skip_play:
	SELECT_RECORD_STREAM(file, stream, goto skip_rec;);
	RAISE_SPL(SPL_SND);
	if (__unlikely(stream >= 0)) {
		oss_stat_underruns[stream] = 0;
		if (__unlikely(OSS_STREAM_FLAGS[stream] & (OSS_BLOCK | OSS_LOOP))) {
			OSS_STOP(stream);
		}
	}
	LOWER_SPL(SPL_OSS);
	skip_rec:
	funmap(file);
	return NULL;
}

static void TRIGGER_PLAYBACK_STREAM(int stream);

int OSS_CLOSE(HANDLE *h, IORQ *RQ)
{
	int stream;
	FFILE *file;
	int r;
	vspace_unmap_t *funmap;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_CLOSE AT SPL %08X", KERNEL$SPL);
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	if (__unlikely(!file)) return 1;
	r = 0;
	if (__unlikely(!(file->flags & _O_KRNL_WRITE))) goto retz;
	if (__unlikely(file->l_refcount > 1) && __likely(RQ->tmp1 != (unsigned long)KERNEL$WAKE_IOCTL)) goto retz;
	SELECT_PLAYBACK_STREAM(file, stream, goto retz;);
	RAISE_SPL(SPL_SND);
	if (__unlikely(OSS_STREAM_FLAGS[stream] & (OSS_LOOP | OSS_BLOCK))) {
		OSS_STOP(0);
		retzs:
		LOWER_SPL(SPL_OSS);
		goto retz;
	}
	if (__unlikely((off_t)(file->pos - (oss_file_pos[stream] - (1 << oss_fragsize_bits[stream]))) < 0)) {
		goto retzs;
	}
	if (__likely((off_t)(file->pos - (oss_file_pos[stream] + (oss_buf_fragments[stream] << oss_fragsize_bits[stream]))) > 0)) {
		goto retzs;
	}
	if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) {
		TRIGGER_PLAYBACK_STREAM(stream);
		if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) goto retzs;
	}
	WQ_WAIT_F(&OSS_STREAM_WAIT[stream], RQ);
	LOWER_SPL(SPL_OSS);
	r = 1;
	retz:
	funmap(file);
	return r;
}

void OSS_SET_ROOT(HANDLE *h, void *f)
{
	h->op = &SOUND_OPERATIONS;
	h->flags = 0;
	h->flags2 = 0;
	SET_RATE(h, 8000, _O_KRNL_READ);
	SET_CHANNELS(h, 0, _O_KRNL_READ);
	SET_FORMAT(h, _FMT_U8, _O_KRNL_READ);
	SET_SUBDIVIDE(h, DEFAULT_SUBDIVIDE);
}

static void CLEAR(__u8 *ptr, unsigned size, int format)
{
	switch (format) {
		default:
		case _FMT_S8:
		case _FMT_S16_LE:
		case _FMT_S16_BE:
		case _FMT_S32_LE:
		case _FMT_S32_BE:
			memset(ptr, 0, size);
			break;
		case _FMT_U8:
			memset(ptr, 0x80, size);
			break;
		case _FMT_U16_LE:
			while (size--) *ptr = !((unsigned long)ptr & 1) ? 0x00 : 0x80, ptr++;
			break;
		case _FMT_U16_BE:
			while (size--) *ptr = !((unsigned long)ptr & 1) ? 0x80 : 0x00, ptr++;
			break;
		case _FMT_U32_LE:
			while (size--) *ptr = ((unsigned long)ptr & 3) != 3 ? 0x00 : 0x80, ptr++;
			break;
		case _FMT_U32_BE:
			while (size--) *ptr = !((unsigned long)ptr & 3) ? 0x80 : 0x00, ptr++;
			break;
	}
}

static void MIXIN(__u8 *dst, __u8 *src, unsigned size, int format)
{
	switch (format) {
		default:
			memcpy(dst, src, size);
			break;
		case _FMT_U8:
			while (size) {
				int val = *dst + *src - 0x80;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val >= 0x100)) val = 0xff;
				*dst = val;
				src++, dst++, size--;
			}
			break;
		case _FMT_S8:
			while (size) {
				int val = (__s8)*dst + (__s8)*src;
				if (__unlikely(val < -0x80)) val = -0x80;
				if (__unlikely(val >= 0x80)) val = 0x7f;
				*dst = val;
				src++, dst++, size--;
			}
			break;
		case _FMT_U16_LE:
			while (size > 1) {
				int val = __16LE2CPU(*(__u16 *)dst) + __16LE2CPU(*(__u16 *)src) - 0x8000;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val >= 0x10000)) val = 0xffff;
				*(__u16 *)dst = __16CPU2LE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_U16_BE:
			while (size > 1) {
				int val = __16BE2CPU(*(__u16 *)dst) + __16BE2CPU(*(__u16 *)src) - 0x8000;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val >= 0x10000)) val = 0xffff;
				*(__u16 *)dst = __16CPU2BE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_S16_LE:
			while (size > 1) {
				int val = __16LE2CPU(*(__s16 *)dst) + __16LE2CPU(*(__s16 *)src);
				if (__unlikely(val < -0x8000)) val = -0x8000;
				if (__unlikely(val >= 0x8000)) val = 0x7fff;
				*(__s16 *)dst = __16CPU2LE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_S16_BE:
			while (size > 1) {
				int val = __16BE2CPU(*(__s16 *)dst) + __16BE2CPU(*(__s16 *)src);
				if (__unlikely(val < -0x8000)) val = -0x8000;
				if (__unlikely(val >= 0x8000)) val = 0x7fff;
				*(__s16 *)dst = __16CPU2BE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_U32_LE:
			while (size > 3) {
				__s64 val = (__s64)__32LE2CPU(*(__u32 *)dst) + (__s64)__32LE2CPU(*(__u32 *)src) - 0x80000000U;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val > 0xffffffffU)) val = 0xffffffffU;
				*(__u32 *)dst = __32CPU2LE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
		case _FMT_U32_BE:
			while (size > 3) {
				__s64 val = (__s64)__32BE2CPU(*(__u32 *)dst) + (__s64)__32BE2CPU(*(__u32 *)src) - 0x80000000U;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val > 0xffffffffU)) val = 0xffffffffU;
				*(__u32 *)dst = __32CPU2BE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
		case _FMT_S32_LE:
			while (size > 3) {
				__s64 val = (__s64)__32LE2CPU(*(__s32 *)dst) + (__s64)__32LE2CPU(*(__s32 *)src);
				if (__unlikely(val < -(__s64)0x80000000U)) val = -(__s64)0x80000000U;
				if (__unlikely(val >= (__s64)0x80000000U)) val = (__s64)0x80000000U;
				*(__s32 *)dst = __32CPU2LE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
		case _FMT_S32_BE:
			while (size > 3) {
				__s64 val = (__s64)__32BE2CPU(*(__s32 *)dst) + (__s64)__32BE2CPU(*(__s32 *)src);
				if (__unlikely(val < -(__s64)0x80000000U)) val = -(__s64)0x80000000U;
				if (__unlikely(val >= (__s64)0x80000000U)) val = (__s64)0x80000000U;
				*(__s32 *)dst = __32CPU2BE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
	}
}

static __u8 *frag_data(int stream, unsigned frag)
{
	unsigned page;
	while (__unlikely(frag >= oss_frags[stream])) frag -= oss_frags[stream];
	page = frag >> (PAGESIZE_BITS - oss_fragsize_bits[stream]);
	if (__unlikely(page >= N_PAGES)) KERNEL$SUICIDE("frag_data: OUT OF PAGES, STREAM %d, REQUESTED FRAG %d, FRAGS %d, FRAGSIZE_BITS %d, PAGES %d, PAGESIZE_BITS %d", stream, frag, oss_frags[stream], oss_fragsize_bits[stream], N_PAGES, PAGESIZE_BITS);
	return (__u8 *)DATA_PAGES[stream][page].virtual + ((frag << oss_fragsize_bits[stream]) & ((1 << PAGESIZE_BITS) - 1));
}

static void default_frags(HANDLE *h, int open_flags)
{
	unsigned fragsize = FORMAT_FRAGSIZE(h), frags = FORMAT_FRAGS(h);
	unsigned old_fragsize, old_frags;
	unsigned div;
	if (__unlikely(!frags) && __unlikely(!fragsize) && __likely(div = FORMAT_SUBDIVIDE(h))) {
		int bit;
		unsigned long s, s2 = MAXLONG;
		unsigned fmt = FORMAT_FORMAT(h), chan = FORMAT_CHANNELS(h), rate = OSS_GET_RATE(FORMAT_RATE(h));
		rate = (__u64)rate * (__u64)div / 1000;
		fragsize = PAGESIZE_BITS;
		frags = FORMAT_MAXFRAGS;
		again:
		SET_FRAGSIZE(h, fragsize);
		SET_FRAGS(h, frags);
		fragsize = FORMAT_FRAGSIZE(h);
		frags = FORMAT_FRAGS(h);
		s = _FMT_DIV(frags << fragsize, fmt, chan);
		if (__likely(s > rate) && __likely(s < s2)) {
			s2 = s;
			if (frags >> 1 >= MIN_FRAGS) frags >>= 1;
			else if (1 << fragsize > MIN_FRAGSIZE) fragsize--;
			else goto ss;
			goto again;
		}

		old_fragsize = fragsize;
		old_frags = frags;
		s2 = s;

		for (bit = __BSR(frags) - 1; frags < FORMAT_MAXFRAGS && bit > -32; bit--) {
			unsigned xfrags = frags;
			if (bit >= 0) {
				if (__unlikely(frags & (1 << bit))) continue;
				frags |= 1 << bit;
			} else {
				if (1 << fragsize <= MIN_FRAGSIZE) break;
				if (frags << 1 > MAX_SUBDIVIDE_FRAGS) break;
				frags = (frags << 1) | 1;
				fragsize--;
			}
			SET_FRAGSIZE(h, fragsize);
			SET_FRAGS(h, frags);
			fragsize = FORMAT_FRAGSIZE(h);
			frags = FORMAT_FRAGS(h);
			s = _FMT_DIV(frags << fragsize, fmt, chan);
			if (s <= s2) {
				break;
			} else if (__unlikely(s > rate)) {
				frags = xfrags;
			} else {
				old_fragsize = fragsize;
				old_frags = frags;
				s2 = s;
			}
		}
		SET_FRAGSIZE(h, old_fragsize);
		SET_FRAGS(h, old_frags);

		ss:;
		/*__debug_printf("frags %d, fragsize %d\n", FORMAT_FRAGS(h), FORMAT_FRAGSIZE(h));*/
	} else {
		if (__unlikely(!fragsize)) SET_FRAGSIZE(h, PAGESIZE_BITS);
		if (__unlikely(!frags)) SET_FRAGS(h, FORMAT_MAXFRAGS);
	}
	SND_FIXUP_PARAMS(h, open_flags);
}

int OSS_STREAM_FLAGS[OSS_N_STREAMS];

int oss_frags[OSS_N_STREAMS], oss_fragsize_bits[OSS_N_STREAMS];
int oss_fragment[OSS_N_STREAMS], oss_buf_fragments[OSS_N_STREAMS];
int oss_rate[OSS_N_STREAMS], oss_channels[OSS_N_STREAMS], oss_format[OSS_N_STREAMS];
off_t oss_file_pos[OSS_N_STREAMS];

int oss_stat_bytes[OSS_N_STREAMS], oss_stat_frags[OSS_N_STREAMS];
unsigned long oss_stat_bytes_drop[OSS_N_STREAMS];
int oss_stat_underruns[OSS_N_STREAMS];

WQ OSS_STREAM_WAIT[OSS_N_STREAMS];

static void TRIGGER_RECORD_STREAM(HANDLE *h, int stream)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("TRIGGER_RECORD_STREAM AT SPL %08X", KERNEL$SPL);
	ASSERT_RECORD_STREAM(stream);	
	if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) {
		if (!OSS_IS_DUPLEX) OSS_STOP_ALL(stream);
		SND_FIXUP_PARAMS(h, _O_KRNL_READ);
		oss_frags[stream] = FORMAT_FRAGS(h);
		oss_fragsize_bits[stream] = FORMAT_FRAGSIZE(h);
		oss_rate[stream] = FORMAT_RATE(h);
		oss_channels[stream] = FORMAT_CHANNELS(h);
		oss_format[stream] = FORMAT_FORMAT(h);
		oss_file_pos[stream] = 0;
		oss_fragment[stream] = 0;
		oss_buf_fragments[stream] = 0;
		SND_START(stream);
	}
}

DECL_IOCALL(OSS_READ, SPL_OSS, SIORQ)
{
	HANDLE *h = RQ->handle;
	VBUF vbuf;
	FFILE *file;
	vspace_unmap_t *funmap;
	int stream;
	int frag;
	long s;
	if (__unlikely(h->op != &SOUND_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));

	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	if (__unlikely(!file)) RETURN;
	SELECT_RECORD_STREAM(file, stream, {
		funmap(file);
		RQ->status = -EPERM;
		RETURN_AST(RQ);
	});
	default_frags(h, file->flags);
	funmap(file);

	if (!RQ->v.len) goto zlen;
	again:
	if (__unlikely(OSS_STREAM_FLAGS[stream] & OSS_BLOCK)) {
		RQ->status = -EAGAIN;
		RETURN_AST(RQ);
	}
	RAISE_SPL(SPL_SND);
	TRIGGER_RECORD_STREAM(h, stream);
	if (__unlikely(!oss_buf_fragments[stream])) {
		if (FORMAT_NONBLOCK(h)) {
			LOWER_SPL(SPL_OSS);
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		WQ_WAIT_F(&OSS_STREAM_WAIT[stream], RQ);
		LOWER_SPL(SPL_OSS);
		RETURN;
	}
	LOWER_SPL(SPL_OSS);
	frag = oss_fragment[stream] - oss_buf_fragments[stream];
	if (frag < 0) frag += oss_frags[stream];
	vbuf.ptr = frag_data(stream, frag) + (unsigned)oss_file_pos[stream];
	vbuf.len = (1 << oss_fragsize_bits[stream]) - (unsigned)oss_file_pos[stream];
	vbuf.spl = SPL_X(SPL_OSS);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!(s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf)))) {
		DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	}
	RQ->progress += s;
	if ((unsigned)oss_file_pos[stream] + s == (1 << oss_fragsize_bits[stream])) {
		RAISE_SPL(SPL_SND);
		if (__likely(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))
			SND_SUBMIT_FRAGMENT(stream, frag);
		oss_file_pos[stream] = 0;
		oss_buf_fragments[stream]--;
		LOWER_SPL(SPL_OSS);
	} else {
		oss_file_pos[stream] += s;
	}
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (RQ->v.len) goto again;
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static __finline__ void OSS_CLR_STREAM(int stream)
{
	oss_file_pos[stream] = 0;
	oss_fragment[stream] = STREAM_IS_RECORD(stream) ? 0 : -1;
	oss_buf_fragments[stream] = 0;
	oss_stat_bytes[stream] = 0;
	oss_stat_frags[stream] = 0;
	oss_stat_bytes_drop[stream] = 0;
}

void OSS_STOP(int stream)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("OSS AT SPL %08X", KERNEL$SPL);
	OSS_CLR_STREAM(stream);
	if (__likely(OSS_STREAM_FLAGS[stream] & OSS_RUNNING)) {
		oss_stat_underruns[stream]++;
		SND_STOP(stream);
	}
	OSS_STREAM_FLAGS[stream] &= ~(OSS_RUNNING | OSS_CONTENTION_STOP | OSS_BLOCK | OSS_LOOP);
	WQ_WAKE_ALL(&OSS_STREAM_WAIT[stream]);
}

static void TRIGGER_PLAYBACK_STREAM(int stream)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("TRIGGER_PLAYBACK_STREAM AT SPL %08X", KERNEL$SPL);
	ASSERT_PLAYBACK_STREAM(stream);
	if (__likely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) {
		if (!OSS_IS_DUPLEX) OSS_STOP_ALL(stream);
		if (__unlikely(!oss_buf_fragments[stream]) && !(OSS_STREAM_FLAGS[stream] & OSS_LOOP)) return;
		if (__likely(oss_buf_fragments[stream]))
			oss_buf_fragments[stream]--;
		oss_fragment[stream]++;
		if (__unlikely(oss_fragment[stream] >= oss_frags[stream]))
			oss_fragment[stream] = 0;
		oss_file_pos[stream] += 1 << oss_fragsize_bits[stream];
		SND_START(stream);
	}
}

#define READ_BUFFER_SIZE	4096

static __u8 read_buffer[READ_BUFFER_SIZE];

DECL_IOCALL(OSS_WRITE, SPL_OSS, SIORQ)
{
	HANDLE *h = RQ->handle;
	VBUF vbuf;
	FFILE *file;
	vspace_unmap_t *funmap;
	int stream;
	int x;
	off_t fpos;
	long dpos;
	long s;
	int pf;
	if (__unlikely(h->op != &SOUND_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	if (__unlikely(!file)) RETURN;
	SELECT_PLAYBACK_STREAM(file, stream, {
		RQ->status = -EPERM;
		goto unmap_ast;
	});
	default_frags(h, file->flags);
	if (__unlikely(!RQ->v.len)) goto zlen;
	if (__unlikely(FORMAT_FRAGS(h) != oss_frags[stream]) ||
	    __unlikely(FORMAT_FRAGSIZE(h) != oss_fragsize_bits[stream])) {
		if (__likely(oss_fragment[stream] == -1) && __likely(!oss_buf_fragments[stream])) {
			oss_frags[stream] = FORMAT_FRAGS(h);
			oss_fragsize_bits[stream] = FORMAT_FRAGSIZE(h);
		}
	}
	RAISE_SPL(SPL_SND);
	if (__unlikely(FORMAT_RATE(h) != oss_rate[stream]) ||
	    __unlikely(FORMAT_CHANNELS(h) != oss_channels[stream]) ||
	    __unlikely(FORMAT_FORMAT(h) != oss_format[stream])) {
		if (__likely(oss_fragment[stream] == -1) && __likely(!oss_buf_fragments[stream])) {
			SND_FIXUP_PARAMS(h, _O_KRNL_WRITE);
			oss_rate[stream] = FORMAT_RATE(h);
			oss_channels[stream] = FORMAT_CHANNELS(h);
			oss_format[stream] = FORMAT_FORMAT(h);
		} else {
			OSS_STREAM_FLAGS[stream] |= OSS_CONTENTION_STOP;
			wfs:
			if (__unlikely(OSS_STREAM_FLAGS[stream] & OSS_BLOCK)) {
				LOWER_SPL(SPL_OSS);
				RQ->status = -EAGAIN;
				goto unmap_ast;
			}
			if (FORMAT_NONBLOCK(h)) {
				TRIGGER_PLAYBACK_STREAM(stream);
				LOWER_SPL(SPL_OSS);
				RQ->status = -EWOULDBLOCK;
				goto unmap_ast;
			}
			WQ_WAIT_F(&OSS_STREAM_WAIT[stream], RQ);
			TRIGGER_PLAYBACK_STREAM(stream);
			LOWER_SPL(SPL_OSS);
			goto unmap_return;
		}
	}
	LOWER_SPL(SPL_OSS);
	if (__unlikely(!(file->flags & _O_KRNL_WRITE))) {
		RQ->status = -EPERM;
		goto unmap_ast;
	}
	again:
	if (__unlikely(x = (int)(fpos = file->pos) & (_FMT_SIZE(oss_format[stream]) - 1))) {
		*(__u32 *)(void *)read_buffer = *(__u32 *)(void *)&file->aux;
	}
	fpos -= x;
	vbuf.ptr = read_buffer + x;
	vbuf.len = (1 << oss_fragsize_bits[stream]) - (fpos & ((1 << oss_fragsize_bits[stream]) - 1)) - x;
	if (vbuf.len + x > READ_BUFFER_SIZE) vbuf.len = READ_BUFFER_SIZE - x;
	vbuf.spl = SPL_X(SPL_OSS);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!(s = RQ->v.vspace->op->vspace_get(&RQ->v, &vbuf)))) {
		funmap(file);
		DO_PAGEIN(RQ, &RQ->v, PF_READ);
	}
	RQ->progress += s;
	s += x;
	RAISE_SPL(SPL_SND);
	repeat:
	dpos = fpos - oss_file_pos[stream];
	if ((off_t)dpos < 0) fpos = oss_file_pos[stream], dpos = 0;
	if ((off_t)dpos > (oss_buf_fragments[stream] << oss_fragsize_bits[stream])) fpos = oss_file_pos[stream] + (dpos = oss_buf_fragments[stream] << oss_fragsize_bits[stream]);
	if (dpos >= oss_buf_fragments[stream] << oss_fragsize_bits[stream]) {
		if (__unlikely(OSS_STREAM_FLAGS[stream] & OSS_CONTENTION_STOP)) goto wfs;
		if (__unlikely(oss_buf_fragments[stream] >= oss_frags[stream] - (oss_fragment[stream] != -1))) {
			s -= x;
			RQ->v.ptr -= s;
			RQ->v.len += s;
			RQ->progress -= s;
			WQ_WAIT_F(&OSS_STREAM_WAIT[stream], RQ);
			LOWER_SPL(SPL_OSS);
			unmap_return:
			funmap(file);
			RETURN;
		}
		pf = oss_fragment[stream] + 1 + oss_buf_fragments[stream];
		LOWER_SPL(SPL_OSS);
/*__debug_printf("clear(%d)", pf);*/
		CLEAR(frag_data(stream, pf), 1 << oss_fragsize_bits[stream], oss_format[stream]);
#ifdef OSS_CANT_STOP
		if (__likely(!(OSS_STREAM_FLAGS[stream] & OSS_LOOP))) if (oss_buf_fragments[stream] < oss_frags[stream] - (oss_fragment[stream] != -1) - 1) CLEAR(frag_data(stream, pf + 1), 1 << oss_fragsize_bits[stream], oss_format[stream]);
#endif
		RAISE_SPL(SPL_SND);
		if (__unlikely(pf != oss_fragment[stream] + 1 + oss_buf_fragments[stream])) goto repeat;
		oss_buf_fragments[stream]++;
		goto repeat;
	}
	pf = oss_fragment[stream] + 1 + (dpos >> oss_fragsize_bits[stream]);
	LOWER_SPL(SPL_OSS);
	if (__unlikely((dpos & ((1 << oss_fragsize_bits[stream]) - 1)) + s > 1 << oss_fragsize_bits[stream]))
		KERNEL$SUICIDE("OSS_WRITE: MIXING OUT OF FRAGMENT: DPOS %ld, FRAGSIZE_BITS %d, S %ld", dpos, oss_fragsize_bits[stream], s);
/*__debug_printf("mixin(%d,%d)", pf, (dpos & ((1 << oss_fragsize_bits[stream]) - 1)));*/
	MIXIN(frag_data(stream, pf) + (dpos & ((1 << oss_fragsize_bits[stream]) - 1)), read_buffer, s & ~(_FMT_SIZE(oss_format[stream]) - 1), oss_format[stream]);
	if (__unlikely(x = s & (_FMT_SIZE(oss_format[stream]) - 1))) {
		memcpy(&file->aux, read_buffer + (s & ~(_FMT_SIZE(oss_format[stream]) - 1)), x);
	}
	file->pos = fpos + s;
	if (__unlikely(oss_fragment[stream] == -1) && __likely(oss_buf_fragments[stream] >= 2) && __likely(!(OSS_STREAM_FLAGS[stream] & OSS_BLOCK))) {
		RAISE_SPL(SPL_SND);
		TRIGGER_PLAYBACK_STREAM(stream);
		LOWER_SPL(SPL_OSS);
	}
	if (fpos + s == oss_file_pos[stream] + (oss_buf_fragments[stream] << oss_fragsize_bits[stream])) {
		if (__likely((OSS_STREAM_FLAGS[stream] & (OSS_RUNNING | OSS_LOOP)) == OSS_RUNNING)) SND_SUBMIT_FRAGMENT(0, oss_fragment[stream] + oss_buf_fragments[stream]);
	}
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_OSS));
	TEST_LOCKUP_LOOP(RQ, goto unmap_return;);
	if (RQ->v.len) goto again;
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	unmap_ast:
	funmap(file);
	RETURN_AST(RQ);
}

static void TEST_LOOP_OUTPUT(HANDLE *h, FFILE *file, int stream)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("TEST_LOOP_OUTPUT AT SPL %08X", KERNEL$SPL);
	ASSERT_PLAYBACK_STREAM(stream);
	if (file->flags & O_LOOP_FLAG) {
		default_frags(h, file->flags);
		if (!(OSS_STREAM_FLAGS[stream] & OSS_LOOP) ||
		    __unlikely(FORMAT_RATE(h) != oss_rate[stream]) ||
		    __unlikely(FORMAT_CHANNELS(h) != oss_channels[stream]) ||
		    __unlikely(FORMAT_FORMAT(h) != oss_format[stream])) {
			OSS_STOP(stream);
			SND_FIXUP_PARAMS(h, _O_KRNL_WRITE);
			oss_frags[stream] = FORMAT_FRAGS(h);
			oss_fragsize_bits[stream] = FORMAT_FRAGSIZE(h);
			oss_rate[stream] = FORMAT_RATE(h);
			oss_channels[stream] = FORMAT_CHANNELS(h);
			oss_format[stream] = FORMAT_FORMAT(h);
			OSS_STREAM_FLAGS[stream] |= OSS_LOOP;
			TRIGGER_PLAYBACK_STREAM(stream);
		}
	}
}

int OSS_STREAM_INTR(int stream)
{
	if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) return 1;
	if (__likely(!STREAM_IS_RECORD(stream))) {
		if (!oss_buf_fragments[stream] && __likely((OSS_STREAM_FLAGS[stream] & (OSS_LOOP | OSS_CONTENTION_STOP)) != OSS_LOOP)) {
			OSS_STOP(stream);
			return 1;
		}
#ifdef OSS_CANT_STOP
		if (__likely(!(OSS_STREAM_FLAGS[stream] & OSS_LOOP))) CLEAR(frag_data(stream, oss_fragment[stream] + 1 + oss_buf_fragments[stream]), 1 << oss_fragsize_bits[stream], oss_format[stream]);
#endif
		oss_fragment[stream]++;
		if (__unlikely(oss_fragment[stream] >= oss_frags[stream])) oss_fragment[stream] = 0;
		if (__unlikely(oss_buf_fragments[stream] == 1) && __likely(!(OSS_STREAM_FLAGS[stream] & OSS_LOOP))) SND_SUBMIT_FRAGMENT(0, oss_fragment[stream]);
		if (__unlikely(oss_fragment[stream] >> (PAGESIZE_BITS - oss_fragsize_bits[stream]) >= N_PAGES)) KERNEL$SUICIDE("OSS_STREAM_INTR: OUT OF PAGE: FRAGMENT %d, PAGESIZE_BITS %d, FRAGSIZE_BITS %d, N_PAGES %d", oss_fragment[stream], PAGESIZE_BITS, oss_fragsize_bits[stream], N_PAGES);
		if (__likely(oss_buf_fragments[stream])) oss_buf_fragments[stream]--;
		oss_file_pos[stream] += 1 << oss_fragsize_bits[stream];
	} else {
		oss_fragment[stream]++;
		if (__unlikely(oss_fragment[stream] >= oss_frags[stream])) oss_fragment[stream] = 0;
		oss_buf_fragments[stream]++;
		if (__unlikely(oss_buf_fragments[stream] >= oss_frags[stream])) {
			OSS_STOP(stream);
			return 1;
		}
	}

	oss_stat_bytes[stream] += 1 << oss_fragsize_bits[stream];
	oss_stat_frags[stream]++;
	if (__unlikely(oss_stat_bytes[stream] >= MAXINT >> 1)) {
		oss_stat_bytes[stream] -= MAXINT >> 1;
		oss_stat_bytes_drop[stream] += MAXINT >> 1;
	}
	WQ_WAKE_ALL_PL(&OSS_STREAM_WAIT[stream]);
	return 0;
}

#ifdef OSS_DEFINE_TIMEOUT

jiffies_t RAW_TIMEOUT(int fragsize_bits, int format, int channels, int rate)
{
	return (__s64)((__u64)_FMT_DIV(1 << (fragsize_bits), format, channels) * (JIFFIES_PER_SECOND * 3) / (rate * 2) + 2);
}

jiffies_t OSS_STREAM_TIMEOUT(int stream)
{
	return RAW_TIMEOUT(oss_fragsize_bits[stream], oss_format[stream], oss_channels[stream], OSS_GET_RATE(oss_rate[stream]));
}

#endif

#ifdef OSS_DEFINE_FRAGINTR

jiffies_t RAW_FRAGINTR(int fragsize_bits, int format, int channels, int rate)
{
	return (__s64)((__u64)_FMT_DIV(1 << (fragsize_bits), format, channels) * JIFFIES_PER_SECOND / rate - 3 * KERNEL$JIFFIES_STEP);
}

jiffies_t OSS_STREAM_FRAGINTR(int stream)
{
	return RAW_FRAGINTR(oss_fragsize_bits[stream], oss_format[stream], oss_channels[stream], OSS_GET_RATE(oss_rate[stream]));
}

#endif

static int bytes_to_read(FFILE *f, int stream)
{
	ASSERT_RECORD_STREAM(stream);
	if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) return 0;
	return (oss_frags[stream] << oss_fragsize_bits[stream]) - (unsigned)oss_file_pos[stream];
}

static int bytes_to_write(HANDLE *h, FFILE *f, int stream)
{
	int a;
	off_t fpos;
	int x;
	ASSERT_PLAYBACK_STREAM(stream);
	fpos = f->pos;
	x = (int)fpos & (_FMT_SIZE(oss_format[stream]) - 1);
	if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) return (FORMAT_FRAGS(h) << FORMAT_FRAGSIZE(h)) - x;
	if (__unlikely(fpos < oss_file_pos[stream])) fpos = oss_file_pos[stream] | x;
	if (__unlikely(fpos > oss_file_pos[stream] + (oss_buf_fragments[stream] << oss_fragsize_bits[stream]))) fpos = (oss_file_pos[stream] + (oss_buf_fragments[stream] << oss_fragsize_bits[stream])) | x;
	a = oss_file_pos[stream] + ((oss_frags[stream] - 1) << oss_fragsize_bits[stream]) - fpos;
	if (__unlikely(a < 0)) a = 0;
	return a;
}

static __finline__ int bytes_written(FFILE *f, int stream)
{
	unsigned add;
	off_t fpos;
	fpos = f->pos;
	ASSERT_PLAYBACK_STREAM(stream);
	if (__unlikely(fpos < oss_file_pos[stream])) fpos = oss_file_pos[stream];
	if (__unlikely(fpos > oss_file_pos[stream] + (oss_buf_fragments[stream] << oss_fragsize_bits[stream]))) fpos = (oss_file_pos[stream] + (oss_buf_fragments[stream] << oss_fragsize_bits[stream]));
	add = 0;
#ifdef OSS_PRECISE_TIMING
	if (__likely(OSS_STREAM_FLAGS[stream] & OSS_RUNNING)) {
		int p = SND_GET_PTR(stream) - (oss_fragment[stream] << oss_fragsize_bits[stream]);
		if (__unlikely(p < 0)) p += oss_frags[stream] << oss_fragsize_bits[stream];
		add = (1 << oss_fragsize_bits[stream]) - p;
	}
#endif
	return fpos - oss_file_pos[stream] + add;
}

DECL_IOCALL(OSS_IOCTL, SPL_OSS, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	vspace_unmap_t *funmap;
	FFILE *file;
	int stream;
	if (__unlikely(h->op != &SOUND_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	if (__unlikely(!file)) RETURN;
	switch (RQ->ioctl) {
		case IOCTL_SELECT_READ: {
			SELECT_RECORD_STREAM(file, stream, goto eperm;);
			RAISE_SPL(SPL_SND);
			if (bytes_to_read(file, stream)) {
				LOWER_SPL(SPL_OSS);
				RQ->status = 0;
				break;
			}
			if (RQ->param) {
				WQ_WAIT_F(&OSS_STREAM_WAIT[stream], RQ);
				LOWER_SPL(SPL_OSS);
				funmap(file);
				RETURN;
			}
			LOWER_SPL(SPL_OSS);
			RQ->status = -EWOULDBLOCK;
			break;
		}
		case IOCTL_SELECT_WRITE: {
			SELECT_PLAYBACK_STREAM(file, stream, goto eperm;);
			default_frags(h, file->flags);
			RAISE_SPL(SPL_SND);
			if (bytes_to_write(h, file, stream)) {
				LOWER_SPL(SPL_OSS);
				RQ->status = 0;
				break;
			}
			if (RQ->param) {
				WQ_WAIT_F(&OSS_STREAM_WAIT[stream], RQ);
				LOWER_SPL(SPL_OSS);
				funmap(file);
				RETURN;
			}
			LOWER_SPL(SPL_OSS);
			RQ->status = -EWOULDBLOCK;
			break;
		}
		case IOCTL_NREAD: {
			SELECT_RECORD_STREAM(file, stream, goto eperm;);
			RAISE_SPL(SPL_SND);
			RQ->status = bytes_to_read(file, stream);
			LOWER_SPL(SPL_OSS);
			break;
		}
		case IOCTL_SND_SYNC: {
			switch (RQ->param) {
				case PARAM_SND_SYNC_RESET: {
					SELECT_PLAYBACK_STREAM(file, stream, goto snd_sync_skip_play;);
					RAISE_SPL(SPL_SND);
					OSS_STOP(stream);
					LOWER_SPL(SPL_OSS);
					snd_sync_skip_play:
					SELECT_RECORD_STREAM(file, stream, goto snd_sync_skip_rec;);
					RAISE_SPL(SPL_SND);
					OSS_STOP(stream);
					LOWER_SPL(SPL_OSS);
					snd_sync_skip_rec:
					RQ->status = 0;
					break;
				}
				case PARAM_SND_SYNC_SYNC: {
					funmap(file);
					if (OSS_CLOSE(h, (IORQ *)RQ)) RETURN;
					
					RQ->status = 0;
					RETURN_AST(RQ);
				}
				case PARAM_SND_SYNC_POST: {
					SELECT_PLAYBACK_STREAM(file, stream, goto snd_post_skip;);
					RAISE_SPL(SPL_SND);
					TRIGGER_PLAYBACK_STREAM(stream);
					LOWER_SPL(SPL_OSS);
					snd_post_skip:
					RQ->status = 0;
					break;
				}
				default: {
					RQ->status = -EINVAL;
					break;
				}
			}
			break;
		}
		case IOCTL_SND_GETRATE: {
			RQ->status = OSS_GET_RATE(FORMAT_RATE(h));
			break;
		}
		case IOCTL_SND_GETCHANNELS: {
			RQ->status = FORMAT_CHANNELS(h) + 1;
			break;
		}
		case IOCTL_SND_GETFORMAT: {
			RQ->status = FORMAT_FORMAT(h);
			break;
		}
		case IOCTL_SND_GETFORMATMASK: {
			int i;
			long ret = 0;
			for (i = 0; i <= _FMT_MAX; i++) if (__unlikely(FORMATS[i])) ret |= 1 << i;
			if (__unlikely(ret < 0)) ret &= ~__LONG_SGN_BIT;
			RQ->status = ret;
			break;
		}
		case IOCTL_SND_GETSPACE: {
			int frags, fragsize_bits;
			int r;
			audio_buf_info info;
			memset(&info, 0, sizeof info);
			default_frags(h, file->flags);
			if (__likely(RQ->param == PARAM_SND_GETSPACE_PLAYBACK)) {
				gets_playback:
				SELECT_PLAYBACK_STREAM(file, stream, goto eperm;);
				RAISE_SPL(SPL_SND);
				TEST_LOOP_OUTPUT(h, file, stream);
				info.bytes = bytes_to_write(h, file, stream);
				if (__likely(OSS_STREAM_FLAGS[stream] & OSS_RUNNING)) {
					frags = oss_frags[stream];
					fragsize_bits = oss_fragsize_bits[stream];
					LOWER_SPL(SPL_OSS);
				} else {
					LOWER_SPL(SPL_OSS);
					frags = FORMAT_FRAGS(h);
					fragsize_bits = FORMAT_FRAGSIZE(h);
				}
			} else if (__likely(RQ->param == PARAM_SND_GETSPACE_RECORD)) {
				gets_record:
				SELECT_RECORD_STREAM(file, stream, goto eperm;);
				RAISE_SPL(SPL_SND);
				info.bytes = bytes_to_read(file, stream);
				if (__likely(OSS_STREAM_FLAGS[stream] & OSS_RUNNING)) {
					frags = oss_frags[stream];
					fragsize_bits = oss_fragsize_bits[stream];
					LOWER_SPL(SPL_OSS);
				} else {
					LOWER_SPL(SPL_OSS);
					frags = FORMAT_FRAGS(h);
					fragsize_bits = FORMAT_FRAGSIZE(h);
				}
			} else if (__likely(RQ->param == PARAM_SND_GETSPACE_CURRENT)) {
				if (__likely(file->flags & _O_KRNL_WRITE)) goto gets_playback;
				else if (__likely(file->flags & _O_KRNL_READ)) goto gets_record;
				else goto eperm;
			} else {
				RQ->status = -EINVAL;
				break;
			}
			info.fragstotal = frags;
			info.fragsize = 1 << fragsize_bits;
			info.fragments = info.bytes >> fragsize_bits;
			/*__debug_printf("buf_fragments: %d\n", oss_playback_buf_fragments);
			__debug_printf("ospace: bytes %d, fragstotal %d, fragsize %d, fragments %d", info.bytes, info.fragstotal, info.fragsize, info.fragments);*/
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &info, sizeof info)) == 1)) {
				funmap(file);
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			RQ->status = r;
			break;
		}
		case IOCTL_SND_GETCAPS: {
			long caps = 1 | DSP_CAP_TRIGGER | DSP_CAP_MMAP;
#ifdef OSS_PRECISE_TIMING
			caps |= DSP_CAP_REALTIME;
#endif
			if (OSS_IS_DUPLEX) caps |= DSP_CAP_DUPLEX;
			RQ->status = caps;
			break;
		}
		case IOCTL_SND_GETTRIGGER: {
			long trig = 0;
			SELECT_PLAYBACK_STREAM(file, stream, goto snd_gettrigger_skip_play;);
			if (!(OSS_STREAM_FLAGS[stream] & OSS_BLOCK)) trig |= PCM_ENABLE_OUTPUT;
			snd_gettrigger_skip_play:
			SELECT_RECORD_STREAM(file, stream, goto snd_gettrigger_skip_rec;);
			if (!(OSS_STREAM_FLAGS[stream] & OSS_BLOCK)) trig |= PCM_ENABLE_INPUT;
			snd_gettrigger_skip_rec:
			RQ->status = trig;
			break;
		}
		case IOCTL_SND_SETTRIGGER: {
			if (__unlikely(RQ->param & ~(PCM_ENABLE_INPUT | PCM_ENABLE_OUTPUT))) {
				RQ->status = -EINVAL;
				break;
			}
			default_frags(h, file->flags);
			SELECT_PLAYBACK_STREAM(file, stream, goto snd_settrigger_skip_play;);
			RAISE_SPL(SPL_SND);
			if (RQ->param & PCM_ENABLE_OUTPUT) {
				OSS_STREAM_FLAGS[stream] &= ~OSS_BLOCK;
				TEST_LOOP_OUTPUT(h, file, stream);
				if (OSS_STREAM_FLAGS[stream] & OSS_RUNNING) SND_RESUME(stream);
				else TRIGGER_PLAYBACK_STREAM(stream);
			} else {
				OSS_STREAM_FLAGS[stream] |= OSS_BLOCK;
				if (OSS_STREAM_FLAGS[stream] & OSS_RUNNING) SND_PAUSE(stream);
			}
			LOWER_SPL(SPL_OSS);
			snd_settrigger_skip_play:
			SELECT_RECORD_STREAM(file, stream, goto snd_settrigger_skip_rec;);
			RAISE_SPL(SPL_SND);
			if (RQ->param & PCM_ENABLE_INPUT) {
				OSS_STREAM_FLAGS[stream] &= ~OSS_BLOCK;
				if (OSS_STREAM_FLAGS[stream] & OSS_RUNNING) SND_RESUME(stream);
				else TRIGGER_RECORD_STREAM(h, stream);
			} else {
				OSS_STREAM_FLAGS[stream] |= OSS_BLOCK;
				if (OSS_STREAM_FLAGS[stream] & OSS_RUNNING) SND_PAUSE(stream);
			}
			LOWER_SPL(SPL_OSS);
			snd_settrigger_skip_rec:
			RQ->status = 0;
			break;
		}
		case IOCTL_SND_GETPTR: {
			int r;
			count_info info;
			memset(&info, 0, sizeof info);
			if (__likely(RQ->param == PARAM_SND_GETPTR_PLAYBACK)) {
				SELECT_PLAYBACK_STREAM(file, stream, goto eperm;);
			} else if (__likely(RQ->param == PARAM_SND_GETPTR_RECORD)) {
				SELECT_RECORD_STREAM(file, stream, goto eperm;);
			} else {
				RQ->status = -EINVAL;
				break;
			}
			RAISE_SPL(SPL_SND);
			if (__likely(RQ->param == PARAM_SND_GETPTR_PLAYBACK))
				TEST_LOOP_OUTPUT(h, file, stream);
			if (__likely(OSS_STREAM_FLAGS[stream] & OSS_RUNNING)) {
#ifdef OSS_PRECISE_TIMING
				int p;
				info.ptr = SND_GET_PTR(stream);
				p = info.ptr - (oss_fragment[stream] << oss_fragsize_bits[stream]);
				if (__unlikely(p < 0)) p += oss_frags[stream] << oss_fragsize_bits[stream];
				info.bytes = oss_stat_bytes[stream] + p;
#else
				info.ptr = oss_fragment[stream] << oss_fragsize_bits[stream];
				info.bytes = oss_stat_bytes[stream];
#endif
				info.blocks = oss_stat_frags[stream];
			}
			LOWER_SPL(SPL_OSS);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &info, sizeof info)) == 1)) {
				funmap(file);
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			if (__likely(!r)) {
				RAISE_SPL(SPL_SND);
				oss_stat_frags[stream] -= info.blocks;
				LOWER_SPL(SPL_OSS);
			}
			RQ->status = r;
			break;
		}
		case IOCTL_SND_GETODELAY: {
			SELECT_PLAYBACK_STREAM(file, stream, goto eperm;);
			RAISE_SPL(SPL_SND);
			if (__likely(RQ->param == PARAM_SND_GETPTR_PLAYBACK))
				TEST_LOOP_OUTPUT(h, file, stream);
			RQ->status = bytes_written(file, stream);
			LOWER_SPL(SPL_OSS);
			break;
		}
		case IOCTL_SND_GETERROR: {
			int playback_stream, record_stream;
			int r;
			audio_errinfo ae;
			SELECT_PLAYBACK_STREAM(file, playback_stream, {
				playback_stream = -1;
				goto snd_geterror_play_skip;
			});
			snd_geterror_play_skip:
			SELECT_RECORD_STREAM(file, record_stream, {
				record_stream = -1;
				goto snd_geterror_rec_skip;
			});
			snd_geterror_rec_skip:
			memset(&ae, 0, sizeof ae);
			RAISE_SPL(SPL_SND);
			if (__likely(playback_stream >= 0)) {
				ae.play_ptradjust = oss_stat_bytes_drop[playback_stream];
				ae.play_underruns = oss_stat_underruns[playback_stream];
			}
			if (__unlikely(record_stream >= 0)) {
				ae.rec_ptradjust = oss_stat_bytes_drop[record_stream];
				ae.rec_overruns = oss_stat_underruns[record_stream];
			}
			LOWER_SPL(SPL_OSS);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &ae, sizeof ae)) == 1)) {
				funmap(file);
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			if (__likely(!r)) {
				RAISE_SPL(SPL_SND);
				
				if (__likely(playback_stream >= 0)) {
					oss_stat_bytes_drop[playback_stream] -= ae.play_ptradjust;
					oss_stat_bytes_drop[playback_stream] -= ae.rec_ptradjust;
				}
				if (__unlikely(record_stream >= 0)) {
					oss_stat_underruns[record_stream] -= ae.play_underruns;
					oss_stat_underruns[record_stream] -= ae.rec_overruns;
				}
				LOWER_SPL(SPL_OSS);
			}
			RQ->status = r;
			break;
		}
		case IOCTL_CAN_MMAP: {
			file->flags |= O_LOOP_FLAG;
			RQ->status = 0;
			break;
		}
#ifdef OSS_MIXER
		case IOCTL_SND_MIXWRITE: {
			unsigned mix = RQ->param & PARAM_MIX_IDX_MASK;
			unsigned long mix_data = RQ->param >> PARAM_MIX_DATA_SHIFT;
			RAISE_SPL(SPL_SND);
			if (mix == SOUND_MIXER_RECSRC) SND_MIXER_WRITE_RECSRC(mix_data & OSS_MIXER_RECMASK);
			else if (mix == SOUND_MIXER_OUTSRC) SND_MIXER_WRITE_OUTSRC(mix_data & OSS_MIXER_OUTMASK);
			else if (__likely(mix < SOUND_MIXER_NRDEVICES) && __likely(OSS_MIXER_DEVMASK & (1 << mix))) SND_MIXER_WRITE(mix, mix_data);
		}
			/* fall through */
		case IOCTL_SND_MIXREAD: {
			unsigned mix = RQ->param & PARAM_MIX_IDX_MASK;
			RAISE_SPL(SPL_SND);
			if (mix == SOUND_MIXER_CAPS) RQ->status = OSS_MIXER_CAPS;
			else if (mix == SOUND_MIXER_DEVMASK) RQ->status = OSS_MIXER_DEVMASK;
			else if (mix == SOUND_MIXER_STEREODEVS) RQ->status = OSS_MIXER_STEREODEVS;
			else if (mix == SOUND_MIXER_RECSRC) RQ->status = SND_MIXER_READ_RECSRC();
			else if (mix == SOUND_MIXER_RECMASK) RQ->status = OSS_MIXER_RECMASK;
			else if (mix == SOUND_MIXER_OUTSRC) RQ->status = SND_MIXER_READ_OUTSRC();
			else if (mix == SOUND_MIXER_OUTMASK) RQ->status = OSS_MIXER_OUTMASK;
			else if (__likely(mix < SOUND_MIXER_NRDEVICES) && __likely(OSS_MIXER_DEVMASK & (1 << mix))) RQ->status = SND_MIXER_READ(mix);
			else RQ->status = -EINVAL;
			LOWER_SPL(SPL_OSS);
			break;
		}
#endif
		default: {
			RQ->status = -ENOOP;
			break;
		}
	}
	if (0) {
		eperm:
		RQ->status = -EPERM;
	}
	funmap(file);
	RETURN_AST(RQ);
}

PAGE *OSS_GET_PAGE(HANDLE *h, __v_off idx, int wr)
{
	FFILE *f;
	vspace_unmap_t *funmap;
	int stream;

	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_GET_PAGE AT SPL %08X", KERNEL$SPL);
	if (__unlikely(wr & ~PF_RW)) return __ERR_PTR(-EINVAL);
	if (__unlikely(idx >= N_PAGES << PAGESIZE_BITS)) return __ERR_PTR(-ERANGE);
	/*if (random() & 7) return NULL;*/
	f = KERNEL$MAP_FILE_STRUCT(h, NULL, &funmap);
	if (__unlikely(!f)) return NULL;
	SELECT_PLAYBACK_STREAM(f, stream, {
		funmap(f);
		return __ERR_PTR(-EPERM);
	});
	funmap(f);
	return KERNEL$PHYS_2_PAGE(DATA_PAGES[stream][(unsigned)idx >> PAGESIZE_BITS].physical + (idx & ((1 << PAGESIZE_BITS) - 1)));
}

IORQ *OSS_GET_PAGEIN_RQ(VDESC *desc, IORQ *rq, int wr)
{
	FFILE *f;
	vspace_unmap_t *funmap;
	f = KERNEL$MAP_FILE_STRUCT(desc->vspace, rq, &funmap);
	if (!f) return NULL;
	funmap(f);
	return rq;
}

#if defined(OSS_ISA_DMA) || defined(OSS_PCI_DMA)

unsigned N_PAGES, PAGESIZE_BITS;

struct page_desc *DATA_PAGES[OSS_N_STREAMS];

static unsigned PAGE_SIZE;

int OSS_ALLOC_DMAPAGES(unsigned n_pages, unsigned page_size)
{
	union {
		MALLOC_REQUEST mrq;
#ifdef OSS_PCI_DMA
		VDMA dma;
#ifdef OSS_DMA_CAPABLE_64
		VDMA64 dma64;
#endif
#endif
	} u;
#ifdef OSS_PCI_DMA
	VDESC desc;
	int spl = KERNEL$SPL;
#endif
	int p, i, n;
	PAGESIZE_BITS = __BSR(page_size);
	if (1 << PAGESIZE_BITS != page_size || !page_size) KERNEL$SUICIDE("OSS_ALLOC_ISA_DMAPAGES: BOGUS PAGE SIZE %X", page_size);
	if (page_size < __PAGE_CLUSTER_SIZE) page_size = __PAGE_CLUSTER_SIZE;
	PAGE_SIZE = page_size;
	N_PAGES = n_pages;

	memset(DATA_PAGES, 0, sizeof DATA_PAGES);
	for (p = 0; p < OSS_N_STREAMS; p++) {
		if (!OSS_IS_DUPLEX && p) {
			DATA_PAGES[p] = DATA_PAGES[0];
			continue;
		}

		u.mrq.size = n_pages * sizeof(struct page_desc);
		SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
		if (u.mrq.status < 0) {
			OSS_FREE_DMAPAGES();
			return u.mrq.status;
		}
		memset(u.mrq.ptr, 0, u.mrq.size);
		DATA_PAGES[p] = u.mrq.ptr;

		for (i = 0; i < n_pages; i++) {
			void *ptr;
			int alloc_flags;
#ifdef OSS_ISA_DMA
			alloc_flags = AREA_PHYSCONTIG | AREA_ISADMA;
#else
			alloc_flags = AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA;
#ifdef OSS_DMA_CAPABLE_64
			if (OSS_DMA_CAPABLE_64) alloc_flags = AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA64;
#endif
#endif
			ptr = KERNEL$ALLOC_CONTIG_AREA(PAGE_SIZE, alloc_flags | AREA_ALIGN, (unsigned long)PAGE_SIZE);
			if (__IS_ERR(ptr)) {
				OSS_FREE_DMAPAGES();
				return __PTR_ERR(ptr);
			}
			memset(ptr, 0, PAGE_SIZE);
			DATA_PAGES[p][i].virtual = ptr;
			DATA_PAGES[p][i].physical = KERNEL$VIRT_2_PHYS(ptr);
#ifdef OSS_CACHEMODE
			{
				int r;
				if (__unlikely(r = KERNEL$SET_MEMORY_PAT(DATA_PAGES[p][i].physical, PAGE_SIZE, OSS_CACHEMODE))) {
					OSS_FREE_DMAPAGES();
					return r;
				}
			}
#endif
#ifdef OSS_PCI_DMA
			desc.ptr = DATA_PAGES[p][i].physical;
			desc.len = PAGE_SIZE;
			desc.vspace = &KERNEL$PHYSICAL;
#ifdef OSS_DMA_CAPABLE_64
			if (!OSS_DMA_CAPABLE_64) {
#endif
				u.dma.spl = spl;
				RAISE_SPL(SPL_VSPACE);
				DATA_PAGES[p][i].u.unlock32 = KERNEL$PHYSICAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &u.dma);
				if (__unlikely(u.dma.len != PAGE_SIZE)) {
					KERNEL$SUICIDE("OSS_ALLOC_DMAPAGES: %s: CAN'T LOCK DMA", dev_name);
				}
				DATA_PAGES[p][i].device = u.dma.ptr;
#ifdef OSS_DMA_CAPABLE_64
			} else {
				u.dma64.spl = spl;
				RAISE_SPL(SPL_VSPACE);
				DATA_PAGES[p][i].u.unlock64 = KERNEL$PHYSICAL.op->vspace_dma64lock(&desc, PF_READ | PF_WRITE, &u.dma64);
				if (__unlikely(u.dma64.len != PAGE_SIZE)) {
					KERNEL$SUICIDE("OSS_ALLOC_DMAPAGES: %s: CAN'T LOCK DMA", dev_name);
				}
				DATA_PAGES[p][i].device = u.dma64.ptr;
			}
#endif
#endif
			for (n = 0; n < PAGE_SIZE; n += __PAGE_CLUSTER_SIZE) {
				KERNEL$VM_PREPARE_PAGE_FOR_MMAP(KERNEL$PHYS_2_PAGE(DATA_PAGES[p][i].physical + n));
			}
		}
	}
	return 0;
}

void OSS_FREE_DMAPAGES(void)
{
	int p, i, n;
	for (p = 0; p < OSS_N_STREAMS; p++) {
		if (!DATA_PAGES[p]) continue;
		if (!OSS_IS_DUPLEX && p)
			goto set_z;
		for (i = 0; i < N_PAGES; i++) {
			if (!DATA_PAGES[p][i].virtual) continue;
			for (n = 0; n < PAGE_SIZE; n += __PAGE_CLUSTER_BITS) {
				WQ *wq;
				PAGE *pg = KERNEL$PHYS_2_PAGE(DATA_PAGES[p][i].physical + n);
				while (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(pg)) != NULL)) {
					WQ_WAIT_SYNC(wq);
					LOWER_SPL(SPL_ZERO);
				}
				LOWER_SPL(SPL_ZERO);
			}
#ifdef OSS_PCI_DMA
#ifdef OSS_DMA_CAPABLE_64
			if (OSS_DMA_CAPABLE_64) DATA_PAGES[p][i].u.unlock64(DATA_PAGES[p][i].device);
			else
#endif
			DATA_PAGES[p][i].u.unlock32(DATA_PAGES[p][i].device);
#endif
#ifdef OSS_CACHEMODE
			KERNEL$SET_MEMORY_PAT(DATA_PAGES[p][i].physical, PAGE_SIZE, PAT_WB);
#endif
			KERNEL$FREE_CONTIG_AREA(DATA_PAGES[p][i].virtual, PAGE_SIZE);
		}
		free(DATA_PAGES[p]);
		set_z:
		DATA_PAGES[p] = NULL;
	}
}

#endif

void OSS_INIT(void)
{
	int i;
	for (i = 0; i < OSS_N_STREAMS; i++) {
		WQ_INIT(&OSS_STREAM_WAIT[i], "OSS$STREAM_WAIT");
		oss_stat_underruns[i] = 0;
		OSS_CLR_STREAM(i);
	}
}

void OSS_STOP_ALL(int except)
{
	int spl = KERNEL$SPL;
	int i;
	for (i = 0; i < OSS_N_STREAMS; i++) {
		if (i != except) {
			RAISE_SPL(SPL_SND);
			OSS_STOP(i);
			LOWER_SPLX(spl);
		}
	}
}
