#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV_KRNL.H>
#include <SYS/SOUNDCARD.H>
#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <SPAD/ISADMA.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>
#include <VALUES.H>
#include <ARCH/BSF.H>

#include "SBREG.H"

#define DSP_TIMEOUT	300000

#define SB_V1		0
#define SB_V20		1
#define SB_V21		2
#define SB_V3		3
#define SB_V4		4

static char * const vers_string[5] = {
	"SB 1.0", "SB 2.0", "SB 2.1", "SB PRO", "SB 16"
};

static int IRQ;
static int DMA1;
static int DMA2;
static char DEV_NAME[__MAX_STR_LEN];
static void *LNTE, *DLRQ;
static IO_RANGE RANGE;
static io_t IO;
static unsigned char play_rec = 0;
static unsigned char version;
static AST SB_IRQ_AST;
static IRQ_HANDLE *SB_IRQ_CTRL;

static jiffies_t SB_TIME;
static DECL_TIMER(SB_TIMEOUT);
static int xfer_dma;
static char xfer_16bit;
static char highspeed;

#define OSS_N_STREAMS		2
#define OSS_IS_DUPLEX		0
#define OSS_DEFINE_TIMEOUT

#define OSS_MIXER
#define OSS_MIXER_CAPS		0
#define OSS_ISA_DMA
#define OSS_PRECISE_TIMING
#define OSS_FIXED_RATES		(version < SB_V4)
#define SPL_SND			SPL_ISA_SND

static unsigned OSS_MIXER_DEVMASK = 0;
static unsigned OSS_MIXER_STEREODEVS = 0;
static unsigned OSS_MIXER_RECMASK = 0;
static unsigned OSS_MIXER_OUTMASK = 0;

#include "OSS.H"

static int DSP_READ(void)
{
	int i;
	for (i = 0; i < DSP_TIMEOUT; i++) {
		KERNEL$DI();
		if (__likely(io_inb(IO + SB_READ_STATUS) & SB_READ_READY)) {
			int r = io_inb(IO + SB_READ);
			KERNEL$EI();
			return r;
		}
		KERNEL$EI();
		KERNEL$UDELAY(1);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "TIMEOUT ON DSP READ");
	return -1;
}

static int DSP_WRITE(__u8 val)
{
	int i;
	for (i = 0; i < DSP_TIMEOUT; i++) {
		KERNEL$DI();
		if (__likely(!(io_inb(IO + SB_WRITE_STATUS) & SB_WRITE_BUSY))) {
			io_outb(IO + SB_WRITE, val);
			KERNEL$EI();
			return 0;
		}
		KERNEL$EI();
		KERNEL$UDELAY(1);
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "TIMEOUT ON DSP WRITE %02X", val);
	return -1;
}

static int DSP_RESET(void)
{
	__u8 v;
	io_outb(IO + SB_RESET, 1);
	KERNEL$UDELAY(10);
	io_outb(IO + SB_RESET, 0);
	KERNEL$UDELAY(30);
	v = DSP_READ();
	if (__likely(v == 0xaa)) return 0;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "DSP RESET RETURNS INCORRECT VALUE %02X", v);
	return -1;
}

static __u8 READ_MIXER_REG(__u8 reg)
{
	__u8 val;
	io_outb(IO + SB_MIXER_ADDR, reg);
	KERNEL$UDELAY(20);
	val = io_inb(IO + SB_MIXER_DATA);
	KERNEL$UDELAY(20);
	return val;
}

static void WRITE_MIXER_REG(__u8 reg, __u8 val)
{
	io_outb(IO + SB_MIXER_ADDR, reg);
	KERNEL$UDELAY(20);
	io_outb(IO + SB_MIXER_DATA, val);
	KERNEL$UDELAY(20);
}

static void CHANGE_MIXER_REG(__u8 reg, __u8 mask, __u8 val)
{
	__u8 x;
#if __DEBUG >= 1
	if (__unlikely(val & ~mask))
		KERNEL$SUICIDE("CHANGE_MIXER_REG: BAD VALUE(%02X)/MASK(%02X)", val, mask);
#endif
	x = READ_MIXER_REG(reg);
	if ((x & mask) == val) return;
	x = (x & ~mask) | val;
	io_outb(IO + SB_MIXER_DATA, x);
	KERNEL$UDELAY(20);
}

static const HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	OSS_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	OSS_INSTANTIATE,	/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

#define MIN_CHANNELS	0
#define MAX_CHANNELS	(version >= SB_V3)
#define MIN_FRAGS	2
#define MAX_FRAGS	MAXUINT
#define MIN_FRAGSIZE	4096

static char FORMATS[_FMT_MAX + 1];

#define DEFAULT_FORMAT	_FMT_U8

/*
#include <stdio.h>

int main(void)
{
	int i;
	for (i = 0; i < 65536; i += 256) {
		printf("%d, ", 256000000 / (65536 - i));
	}
	return 0;
}
*/

static const int RATELIST[] = {
	-1, -1, -1, -1, -1, -1, 4000, 4016,
	4032, 4048, 4065, 4081, 4098, 4115, 4132, 4149,
	4166, 4184, 4201, 4219, 4237, 4255, 4273, 4291,
	4310, 4329, 4347, 4366, 4385, 4405, 4424, 4444,
	4464, 4484, 4504, 4524, 4545, 4566, 4587, 4608,
	4629, 4651, 4672, 4694, 4716, 4739, 4761, 4784,
	4807, 4830, 4854, 4878, 4901, 4926, 4950, 4975,
	5000, 5025, 5050, 5076, 5102, 5128, 5154, 5181,
	5208, 5235, 5263, 5291, 5319, 5347, 5376, 5405,
	5434, 5464, 5494, 5524, 5555, 5586, 5617, 5649,
	5681, 5714, 5747, 5780, 5813, 5847, 5882, 5917,
	5952, 5988, 6024, 6060, 6097, 6134, 6172, 6211,
	6250, 6289, 6329, 6369, 6410, 6451, 6493, 6535,
	6578, 6622, 6666, 6711, 6756, 6802, 6849, 6896,
	6944, 6993, 7042, 7092, 7142, 7194, 7246, 7299,
	7352, 7407, 7462, 7518, 7575, 7633, 7692, 7751,
	7812, 7874, 7936, 8000, 8064, 8130, 8196, 8264,
	8333, 8403, 8474, 8547, 8620, 8695, 8771, 8849,
	8928, 9009, 9090, 9174, 9259, 9345, 9433, 9523,
	/*9615*/9600, 9708, 9803, 9900, 10000, 10101, 10204, 10309,
	10416, 10526, 10638, 10752, 10869, 10989, /*11111*/11025, 11235,
	11363, 11494, 11627, 11764, 11904, 12048, 12195, 12345,
	12500, 12658, 12820, 12987, 13157, 13333, 13513, 13698,
	13888, 14084, 14285, 14492, 14705, 14925, 15151, 15384,
	15625, 15873, /*16129*/16000, 16393, 16666, 16949, 17241, 17543,
	17857, 18181, 18518, 18867, 19230, 19607, 20000, 20408,
	20833, 21276, /*21739*/22050, 22222, 22727, 23255, /*23809*/24000, 24390,
	25000, 25641, 26315, 27027, 27777, 28571, 29411, 30303,
	/*31250*/32000, 32258, 33333, 34482, 35714, 37037, 38461, 40000,
	41666, /*43478*/44100,
	/*45454, 47619, 50000, 52631, 55555, 58823,
	62500, 66666, 71428, 76923, 83333, 90909, 100000, 111111,
	125000, 142857, 166666, 200000, 250000, 333333, 500000, 1000000*/
	0
};

#define RATE_LIMIT_RECORD_LOSPEED	179
#define RATE_LIMIT_RECORD_V21_HISPEED	189
#define RATE_LIMIT_PLAYBACK_HISPEED	212
#define RATE_LIMIT_V3_STEREO		212

#define MIN_RATE	(version == SB_V4 ? 5000 : 4000)
#define MAX_RATE	(version < SB_V21 ? 22727 : 44100)

#define RATELIST_RESTRICT(h, i)		((i) & FORMAT_CHANNELS(h))

void SND_FIXUP_PARAMS(HANDLE *h, int open_flags)
{
	unsigned char rate_limit;
	if (version >= SB_V4) {
		if (__unlikely(_FMT_SIZE(FORMAT_FORMAT(h)) == 1) && (FORMAT_FRAGS(h) << FORMAT_FRAGSIZE(h)) > 65536) {
			if (FORMAT_FRAGSIZE(h) > __BSR_CONST(MIN_FRAGSIZE))
				FORMAT_SETFRAGSIZE(h, FORMAT_FRAGSIZE(h) - 1);
			else
				FORMAT_SETFRAGS(h, FORMAT_FRAGS(h) >> 1);
		}
		return;
	}
	rate_limit = 255;
	/* already set in MAX_RATE
	if (__unlikely(version < SB_V21)) rate_limit = RATE_LIMIT_PLAYBACK_HISPEED;
	*/
	if (__unlikely(open_flags & _O_KRNL_READ)) {
		if (__unlikely(version < SB_V21)) rate_limit = RATE_LIMIT_RECORD_LOSPEED;
		else if (__unlikely(version == SB_V21)) rate_limit = RATE_LIMIT_RECORD_V21_HISPEED;
	}
	if (FORMAT_CHANNELS(h)) rate_limit = RATE_LIMIT_V3_STEREO;
	if (__unlikely(FORMAT_RATE(h) > rate_limit))
		FORMAT_SETRATE(h, rate_limit);
	if (__unlikely(FORMAT_RATE(h) & FORMAT_CHANNELS(h)))
		FORMAT_SETRATE(h, FORMAT_RATE(h) & ~1);

}

__finline__ void SND_SUBMIT_FRAGMENT(int record, int frag)
{
}

#include "OSS.I"

#define AUTO_DMA	(__likely(version >= SB_V20))

static void SETUP_NEXT_XFER(int record)
{
	unsigned rate, frag, frags, format;
	rate = oss_rate[record];
	if (__likely(oss_channels[record])) rate += (256 - rate) / 2;
	frag = oss_fragment[record];
	frags = oss_frags[record];
	if (AUTO_DMA) frag = 0;
	else frags = 1;
	format = oss_format[record];
	ISADMA$SETUP_TRANSFER(xfer_dma, record ^ 1, AUTO_DMA, DATA_PAGES[record][0].physical + (frag << oss_fragsize_bits[record]), frags << oss_fragsize_bits[record]);
	if (version < SB_V4) {
		if (__unlikely(DSP_WRITE(AUTO_DMA ? SBCMD_SET_TRANSFER_SIZE : SBCMD_8BIT_SINGLE_DMA_OUTPUT + (record * (SBCMD_8BIT_SINGLE_DMA_INPUT - SBCMD_8BIT_SINGLE_DMA_OUTPUT))))) goto dsp_error;
	} else {
		if (__unlikely(DSP_WRITE(SBCMD_8BIT_IO + SBCMD_IO_FIFO + SBCMD_IO_AUTO_DMA - xfer_16bit * (SBCMD_8BIT_IO - SBCMD_16BIT_IO) + (record * SBCMD_IO_INPUT)))) goto dsp_error;
		if (__unlikely(DSP_WRITE((format == _FMT_S8 || format == _FMT_S16_LE) * SBCMD_IOMODE_SIGNED + oss_channels[record] * SBCMD_IOMODE_STEREO))) goto dsp_error;
	}
	if (__unlikely(DSP_WRITE((1 << (oss_fragsize_bits[record] - xfer_16bit)) - 1))) goto dsp_error;
	if (__unlikely(DSP_WRITE(((1 << (oss_fragsize_bits[record] - xfer_16bit)) - 1) >> 8))) goto dsp_error;
	highspeed = 0;
	if (__likely(AUTO_DMA) && version < SB_V4) {
		if (rate > (!record || version >= SB_V3 ? RATE_LIMIT_PLAYBACK_HISPEED : RATE_LIMIT_RECORD_LOSPEED)) {
			if (__unlikely(DSP_WRITE(SBCMD_8BIT_HIGHSPEED_AUTO_DMA_OUTPUT + (record * (SBCMD_8BIT_HIGHSPEED_AUTO_DMA_INPUT - SBCMD_8BIT_HIGHSPEED_AUTO_DMA_OUTPUT))))) goto dsp_error;
			highspeed = 1;
		} else {
			if (__unlikely(DSP_WRITE(SBCMD_8BIT_AUTO_DMA_OUTPUT + (record * (SBCMD_8BIT_AUTO_DMA_INPUT - SBCMD_8BIT_AUTO_DMA_OUTPUT))))) goto dsp_error;
		}
	}
	KERNEL$DEL_TIMER(&SB_TIMEOUT);
	KERNEL$SET_TIMER(SB_TIME, &SB_TIMEOUT);
	return;

	dsp_error:
	SND_STOP(record);
}

void SND_START(int record)
{
	if (__likely(!record)) {
		if (__unlikely(DSP_WRITE(SBCMD_SPEAKER_ON))) goto dsp_error;
	}
	if (version < SB_V4) {
		if (__unlikely(DSP_WRITE(SBCMD_SET_TIME_CONSTANT))) goto dsp_error;
		if (__unlikely(DSP_WRITE(oss_rate[record] + (oss_channels[record] ? (256 - oss_rate[record]) / 2 : 0)))) goto dsp_error;
	} else {
		if (__unlikely(DSP_WRITE(SBCMD_SET_OUTPUT_SAMPLING_RATE + record * (SBCMD_SET_INPUT_SAMPLING_RATE - SBCMD_SET_OUTPUT_SAMPLING_RATE)))) goto dsp_error;
		if (__unlikely(DSP_WRITE(oss_rate[record] >> 8))) goto dsp_error;
		if (__unlikely(DSP_WRITE(oss_rate[record]))) goto dsp_error;
	}
	if (__likely(!record)) {
		if (version == SB_V3) {
			__u8 s = 0;
			if (oss_channels[record]) s = SBMIXER3_OUTPUT_STEREO | SBMIXER3_OUTPUT_LOW_PASS_DIS;
			else if (OSS_GET_RATE(oss_rate[record]) >= 36000) s = SBMIXER3_OUTPUT_LOW_PASS_DIS;
			CHANGE_MIXER_REG(SBMIXER3_OUTPUT, SBMIXER3_OUTPUT_STEREO | SBMIXER3_OUTPUT_LOW_PASS_DIS, s);
		}
	} else {
		if (version == SB_V3) {
			__u8 s;
			if (oss_channels[record]) goto output_filt_dis;
			if (OSS_GET_RATE(oss_rate[record]) < 18000) s = 0;
			else if (OSS_GET_RATE(oss_rate[record]) < 36000) s = SBMIXER3_INPUT_LOW_PASS_8;
			else output_filt_dis: s = SBMIXER3_INPUT_LOW_PASS_8 | SBMIXER3_INPUT_LOW_PASS_DIS;
			CHANGE_MIXER_REG(SBMIXER3_INPUT, SBMIXER3_INPUT_LOW_PASS_8 | SBMIXER3_INPUT_LOW_PASS_DIS, s);
			if (__unlikely(DSP_WRITE(SBCMD_INPUT_MONO + (oss_channels[record] * (SBCMD_INPUT_STEREO - SBCMD_INPUT_MONO))))) goto dsp_error;
		}
	}
	OSS_STREAM_FLAGS[record] |= OSS_RUNNING;
	play_rec = record;
	xfer_16bit = !(oss_format[record] == _FMT_S8 || oss_format[record] == _FMT_U8);
	xfer_dma = __unlikely(!xfer_16bit) ? DMA1 : DMA2;
	SB_TIME = OSS_STREAM_TIMEOUT(record);
	SETUP_NEXT_XFER(record);
	return;

	dsp_error:
	return;
}

void SND_STOP(int record)
{
	/* Always reset. ESS 1688 chokes on exit auto dma command */
	if (1 || highspeed || !AUTO_DMA) {
		DSP_RESET();
	} else {
		DSP_WRITE(SBCMD_8BIT_EXIT_AUTO_DMA - xfer_16bit * (SBCMD_8BIT_EXIT_AUTO_DMA - SBCMD_16BIT_EXIT_AUTO_DMA));
	}
	if (!record) DSP_WRITE(SBCMD_SPEAKER_OFF);
	ISADMA$END_TRANSFER(xfer_dma);
	KERNEL$DEL_TIMER(&SB_TIMEOUT);
	SET_TIMER_NEVER(&SB_TIMEOUT);
}

void SND_PAUSE(int record)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("SND_PAUSE AT SPL %08X", KERNEL$SPL);
	if (highspeed) ISADMA$PAUSE_TRANSFER(xfer_dma);
	else DSP_WRITE(SBCMD_8BIT_PAUSE_DMA + xfer_16bit * (SBCMD_16BIT_PAUSE_DMA - SBCMD_8BIT_PAUSE_DMA));
	KERNEL$DEL_TIMER(&SB_TIMEOUT);
	SET_TIMER_NEVER(&SB_TIMEOUT);
}

void SND_RESUME(int record)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("SND_RESUME AT SPL %08X", KERNEL$SPL);
	if (highspeed) ISADMA$RESUME_TRANSFER(xfer_dma);
	else DSP_WRITE(SBCMD_8BIT_CONTINUE_DMA + xfer_16bit * (SBCMD_16BIT_CONTINUE_DMA - SBCMD_8BIT_CONTINUE_DMA));
	KERNEL$DEL_TIMER(&SB_TIMEOUT);
	KERNEL$SET_TIMER(SB_TIME, &SB_TIMEOUT);
}

unsigned SND_GET_PTR(int record)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("SND_GET_PTR AT SPL %08X", KERNEL$SPL);
#endif
	return ISADMA$GET_ADDRESS(xfer_dma) - DATA_PAGES[play_rec][0].physical;
}

static void SB_AUTODMA_INTR(int record)
{
	unsigned frag = (SND_GET_PTR(record) >> oss_fragsize_bits[record]);
	if (__unlikely(frag >= oss_frags[record])) frag = 0;
	while (oss_fragment[record] != frag) {
		if (__unlikely(OSS_STREAM_INTR(record))) return;
		KERNEL$DEL_TIMER(&SB_TIMEOUT);
		KERNEL$SET_TIMER(SB_TIME, &SB_TIMEOUT);
	}
}

static __finline__ void SB_INTR(void)
{
	unsigned char src = 0xff;
	if (version == SB_V4) {
		src = READ_MIXER_REG(SBMIXER_IRQ_STAT);
		if (__unlikely(!(src & 3))) return;
		if (__likely(src & 2))
			io_inb(IO + SB_ACK_INTR16);
	}
	if (__likely(src & 1))
		io_inb(IO + SB_READ_STATUS);
	if (__likely(OSS_STREAM_FLAGS[play_rec] & OSS_RUNNING)) {
		if (AUTO_DMA) {
			SB_AUTODMA_INTR(play_rec);
		} else {
			if (__unlikely(OSS_STREAM_INTR(play_rec))) return;
			ISADMA$END_TRANSFER(xfer_dma);
			SETUP_NEXT_XFER(play_rec);
		}
	}
}

static DECL_IRQ_AST(SB_IRQ, SPL_SND, AST)
{
	SB_INTR();
	KERNEL$UNMASK_IRQ(SB_IRQ_CTRL);
	RETURN;
}

static void SB_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	SET_TIMER_NEVER(t);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "INTERRUPT TIMEOUT");
	OSS_STOP(play_rec);
}

long SND_MIXER_READ_RECSRC(void)
{
	long mask = 0;
	if (version == SB_V4) {
		__u8 val = READ_MIXER_REG(SBMIXER4_INPUT_LEFT_SWITCH) | READ_MIXER_REG(SBMIXER4_INPUT_RIGHT_SWITCH);
		if (val & (SBMIXER4_INPUT_SWITCH_MIDI_L | SBMIXER4_INPUT_SWITCH_MIDI_R)) mask |= SOUND_MASK_SYNTH;
		if (val & (SBMIXER4_INPUT_SWITCH_LINE_L | SBMIXER4_INPUT_SWITCH_LINE_R)) mask |= SOUND_MASK_LINE;
		if (val & (SBMIXER4_INPUT_SWITCH_MIC)) mask |= SOUND_MASK_MIC;
		if (val & (SBMIXER4_INPUT_SWITCH_CD_L | SBMIXER4_INPUT_SWITCH_CD_R)) mask |= SOUND_MASK_CD;
	} else if (version == SB_V3) {
		__u8 val = READ_MIXER_REG(SBMIXER3_INPUT) & SBMIXER3_INPUT_SOURCE_MASK;
		if (val == SBMIXER3_INPUT_SOURCE_CD) mask = SOUND_MASK_CD;
		else if (val == SBMIXER3_INPUT_SOURCE_LINE) mask = SOUND_MASK_LINE;
		else mask = SOUND_MASK_MIC;
	}
	return mask;
}

void SND_MIXER_WRITE_RECSRC(unsigned long mask)
{
	if (version == SB_V4) {
		__u8 val_l = 0;
		__u8 val_r = 0;
		if (mask & SOUND_MASK_SYNTH) val_l |= SBMIXER4_INPUT_SWITCH_MIDI_L, val_r |= SBMIXER4_INPUT_SWITCH_MIDI_R;
		if (mask & SOUND_MASK_LINE) val_l |= SBMIXER4_INPUT_SWITCH_LINE_L, val_r |= SBMIXER4_INPUT_SWITCH_LINE_R;
		if (mask & SOUND_MASK_MIC) val_l |= SBMIXER4_INPUT_SWITCH_MIC, val_r |= SBMIXER4_INPUT_SWITCH_MIC;
		if (mask & SOUND_MASK_CD) val_l |= SBMIXER4_INPUT_SWITCH_CD_L, val_r |= SBMIXER4_INPUT_SWITCH_CD_R;
		CHANGE_MIXER_REG(SBMIXER4_INPUT_LEFT_SWITCH, SBMIXER4_INPUT_SWITCH_MIDI_L | SBMIXER4_INPUT_SWITCH_MIDI_R | SBMIXER4_INPUT_SWITCH_LINE_L | SBMIXER4_INPUT_SWITCH_LINE_R | SBMIXER4_INPUT_SWITCH_MIC | SBMIXER4_INPUT_SWITCH_CD_L | SBMIXER4_INPUT_SWITCH_CD_R, val_l);
		CHANGE_MIXER_REG(SBMIXER4_INPUT_RIGHT_SWITCH, SBMIXER4_INPUT_SWITCH_MIDI_L | SBMIXER4_INPUT_SWITCH_MIDI_R | SBMIXER4_INPUT_SWITCH_LINE_L | SBMIXER4_INPUT_SWITCH_LINE_R | SBMIXER4_INPUT_SWITCH_MIC | SBMIXER4_INPUT_SWITCH_CD_L | SBMIXER4_INPUT_SWITCH_CD_R, val_r);
	} else if (version == SB_V3) {
		__u8 val;
		if (mask == SOUND_MASK_LINE) val = SBMIXER3_INPUT_SOURCE_LINE;
		else if (mask == SOUND_MASK_MIC) val = SBMIXER3_INPUT_SOURCE_MIC;
		else if (mask == SOUND_MASK_CD) val = SBMIXER3_INPUT_SOURCE_CD;
		else return;
		CHANGE_MIXER_REG(SBMIXER3_INPUT, SBMIXER3_INPUT_SOURCE_MASK, val);
	}
}

long SND_MIXER_READ_OUTSRC(void)
{
	if (version == SB_V4) {
		__u8 val = READ_MIXER_REG(SBMIXER4_OUTPUT_SWITCH);
		long mask = 0;
		if (val & (SBMIXER4_OUTPUT_SWITCH_LINE_L | SBMIXER4_OUTPUT_SWITCH_LINE_R)) mask |= SOUND_MASK_LINE;
		if (val & SBMIXER4_OUTPUT_SWITCH_MIC) mask |= SOUND_MASK_MIC;
		if (val & (SBMIXER4_OUTPUT_SWITCH_CD_L | SBMIXER4_OUTPUT_SWITCH_CD_R)) mask |= SOUND_MASK_CD;
		return mask;
	}
	return 0;
}

void SND_MIXER_WRITE_OUTSRC(unsigned long mask)
{
	if (version == SB_V4) {
		__u8 val = 0;
		if (mask & SOUND_MASK_LINE) val |= SBMIXER4_OUTPUT_SWITCH_LINE_L | SBMIXER4_OUTPUT_SWITCH_LINE_R;
		if (mask & SOUND_MASK_MIC) val |= SBMIXER4_OUTPUT_SWITCH_MIC;
		if (mask & SOUND_MASK_CD) val |= SBMIXER4_OUTPUT_SWITCH_CD_L | SBMIXER4_OUTPUT_SWITCH_CD_R;
		CHANGE_MIXER_REG(SBMIXER4_OUTPUT_SWITCH, SBMIXER4_OUTPUT_SWITCH_LINE_L | SBMIXER4_OUTPUT_SWITCH_LINE_R | SBMIXER4_OUTPUT_SWITCH_MIC | SBMIXER4_OUTPUT_SWITCH_CD_L | SBMIXER4_OUTPUT_SWITCH_CD_R, val);
	}
}

typedef struct {
	__u8 nr, left, left_mask, right, right_mask;
} MIXER;

static const MIXER MIX_DUMMY[] = {
	{ 0, 0, 0, 0, 0 }
};

static const MIXER MIX_V2[] = {
	{ SOUND_MIXER_VOLUME, SBMIXER2_MASTER, SBMIXER2_MASTER_MASK, 0, 0 },
	{ SOUND_MIXER_SYNTH, SBMIXER2_MIDI, SBMIXER2_MIDI_MASK, 0, 0 },
	{ SOUND_MIXER_PCM, SBMIXER2_VOICE, SBMIXER2_VOICE_MASK, 0, 0 },
	{ SOUND_MIXER_CD, SBMIXER2_CD, SBMIXER2_CD_MASK, 0, 0 },
	{ 0, 0, 0, 0, 0 }
};

static const MIXER MIX_V3[] = {
	{ SOUND_MIXER_VOLUME, SBMIXER34_MASTER, SBMIXER34_MASTER_LEFT_MASK, SBMIXER34_MASTER, SBMIXER34_MASTER_RIGHT_MASK },
	{ SOUND_MIXER_SYNTH, SBMIXER34_MIDI, SBMIXER34_MIDI_LEFT_MASK, SBMIXER34_MIDI, SBMIXER34_MIDI_RIGHT_MASK },
	{ SOUND_MIXER_PCM, SBMIXER34_VOICE, SBMIXER34_VOICE_LEFT_MASK, SBMIXER34_VOICE, SBMIXER34_VOICE_RIGHT_MASK },
	{ SOUND_MIXER_LINE, SBMIXER34_LINE, SBMIXER34_LINE_LEFT_MASK, SBMIXER34_LINE, SBMIXER34_LINE_RIGHT_MASK },
	{ SOUND_MIXER_MIC, SBMIXER34_MIC, SBMIXER34_MIC_MASK, 0, 0 },
	{ SOUND_MIXER_CD, SBMIXER34_CD, SBMIXER34_CD_LEFT_MASK, SBMIXER34_CD, SBMIXER34_CD_RIGHT_MASK },
	{ 0, 0, 0, 0, 0 }
};

static const MIXER MIX_V4[] = {
	{ SOUND_MIXER_VOLUME, SBMIXER4_MASTER_LEFT, SBMIXER4_MASTER_LEFT_MASK, SBMIXER4_MASTER_RIGHT, SBMIXER4_MASTER_RIGHT_MASK },
	{ SOUND_MIXER_BASS, SBMIXER4_BASS_LEFT, SBMIXER4_BASS_LEFT_MASK, SBMIXER4_BASS_RIGHT, SBMIXER4_BASS_RIGHT_MASK },
	{ SOUND_MIXER_TREBLE, SBMIXER4_TREBLE_LEFT, SBMIXER4_TREBLE_LEFT_MASK, SBMIXER4_TREBLE_RIGHT, SBMIXER4_TREBLE_RIGHT_MASK },
	{ SOUND_MIXER_SYNTH, SBMIXER4_MIDI_LEFT, SBMIXER4_MIDI_LEFT_MASK, SBMIXER4_MIDI_RIGHT, SBMIXER4_MIDI_RIGHT_MASK },
	{ SOUND_MIXER_PCM, SBMIXER4_VOICE_LEFT, SBMIXER4_VOICE_LEFT_MASK, SBMIXER4_VOICE_RIGHT, SBMIXER4_VOICE_RIGHT_MASK },
	{ SOUND_MIXER_SPEAKER, SBMIXER4_SPEAKER, SBMIXER4_SPEAKER_MASK, 0, 0 },
	{ SOUND_MIXER_LINE, SBMIXER4_LINE_LEFT, SBMIXER4_LINE_LEFT_MASK, SBMIXER4_LINE_RIGHT, SBMIXER4_LINE_RIGHT_MASK },
	{ SOUND_MIXER_MIC, SBMIXER4_MIC, SBMIXER4_MIC_MASK, 0, 0 },
	{ SOUND_MIXER_CD, SBMIXER4_CD_LEFT, SBMIXER4_CD_LEFT_MASK, SBMIXER4_CD_RIGHT, SBMIXER4_CD_RIGHT_MASK },
	{ 0, 0, 0, 0, 0 }
};

#define MIX_NOT_EOF(m)	((m)->left_mask)

static const MIXER *MIX;

static const MIXER *FIND_MIXER(unsigned mix)
{
	const MIXER *m;
	for (m = MIX; MIX_NOT_EOF(m); m++) if (m->nr == mix) return m;
	KERNEL$SUICIDE("FIND_MIXER: INVALID MIXER %u", mix);
}

static unsigned SCALE(unsigned val, unsigned s_from, unsigned s_to)
{
	return (val * s_to + (s_from >> 1)) / s_from;
}

long SND_MIXER_READ(unsigned mix)
{
	const MIXER *m = FIND_MIXER(mix);
	__u8 val = READ_MIXER_REG(m->left);
	unsigned left = (val & m->left_mask) >> __BSF(m->left_mask);
	unsigned right = 0;
	left = SCALE(left, m->left_mask >> __BSF(m->left_mask), 100);
	if (m->right_mask) {
		if (m->left != m->right) val = READ_MIXER_REG(m->right);
		right = (val & m->right_mask) >> __BSF(m->right_mask);
		right = SCALE(right, m->right_mask >> __BSF(m->right_mask), 100);
	}
	return left + (right << 8);
}

void SND_MIXER_WRITE(unsigned mix, unsigned long val)
{
	const MIXER *m = FIND_MIXER(mix);
	unsigned left = val & 0xff;
	unsigned right = (val >> 8) & 0xff;
	if (__unlikely(left > 100)) left = 100;
	if (__unlikely(right > 100)) right = 100;
	left = SCALE(left, 100, m->left_mask >> __BSF(m->left_mask));
	right = SCALE(right, 100, m->right_mask >> __BSF(m->right_mask));
	if (!m->right_mask) {
		goto chg_left;
	} if (m->left == m->right) {
		CHANGE_MIXER_REG(m->left, m->left_mask | m->right_mask, (left << __BSF(m->left_mask)) | (right << __BSF(m->right_mask)));
	} else {
		CHANGE_MIXER_REG(m->right, m->right_mask, right << __BSF(m->right_mask));
		chg_left:
		CHANGE_MIXER_REG(m->left, m->left_mask, left << __BSF(m->left_mask));
	}
}

static int SB_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	const char * const *arg;
	int state;
	int r;
	int i;
	unsigned long io;
	short vers_major, vers_minor;
	char forced_downgrade;
	int force_version;
	static const struct __param_table mode_params[6] = {
		"1", __PARAM_BOOL, ~0, SB_V1,
		"2", __PARAM_BOOL, ~0, SB_V20,
		"2.1", __PARAM_BOOL, ~0, SB_V21,
		"3", __PARAM_BOOL, ~0, SB_V3,
		"4", __PARAM_BOOL, ~0, SB_V4,
		NULL, 0, 0, 0,
	};
	void *mode_vars[6];
	static const struct __param_table params[6] = {
		"IO", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - SB_REGSPACE + 1,
		"IRQ", __PARAM_INT, 0, IRQ_LIMIT + 1,
		"DMA", __PARAM_INT, DMA_8BIT_MIN, DMA_8BIT_MAX,
		"DMA2", __PARAM_INT, DMA_16BIT_MIN, DMA_16BIT_MAX,
		"MODE", __PARAM_EXTD_ONE, (long)&mode_params, 0,
		NULL, 0, 0, 0,
	};
	void *vars[6];
	vars[0] = &io;
	vars[1] = &IRQ;
	vars[2] = &DMA1;
	vars[3] = &DMA2;
	vars[4] = &mode_vars;
	vars[5] = NULL;
	mode_vars[0] = &force_version;
	mode_vars[1] = &force_version;
	mode_vars[2] = &force_version;
	mode_vars[3] = &force_version;
	mode_vars[4] = &force_version;
	mode_vars[5] = NULL;
	io = 0x220;
	IRQ = -1;
	DMA1 = -1;
	DMA2 = -1;
	force_version = -1;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SB: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}
	IO = io;
	_snprintf(DEV_NAME, __MAX_STR_LEN, "SND$SB@" IO_ID, IO);
	if (__unlikely(IRQ == -1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SB: IRQ NOT SPECIFIED");
		r = -EBADSYN;
		goto ret0;
	}
	RANGE.start = IO;
	RANGE.len = SB_REGSPACE;
	RANGE.name = DEV_NAME;
	if ((r = KERNEL$REGISTER_IO_RANGE(&RANGE))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", DEV_NAME, RANGE.start, RANGE.start + RANGE.len - 1, strerror(-r));
		goto ret0;
	}

	if (DSP_RESET()) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO RESET DSP", DEV_NAME);
		r = -ENODEV;
		goto ret1;
	}

	if (DSP_WRITE(SBCMD_GET_DSP_VERSION)) {
		cant_init:
		r = -ENXIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE DSP", DEV_NAME);
		goto ret1;
	}
	if ((vers_major = DSP_READ()) < 0) goto cant_init;
	if ((vers_minor = DSP_READ()) < 0) goto cant_init;
	/* __debug_printf("%d.%d\n", vers_major, vers_minor); */
	if (vers_major == 1) version = SB_V1;
	else if (vers_major == 2 && !vers_minor) version = SB_V20;
	else if (vers_major == 2) version = SB_V21;
	else if (vers_major == 3) version = SB_V3;
	else if (vers_major == 4) version = SB_V4;
	else if (force_version >= 0) version = force_version;
	else {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNKNOWN VERSION %d.%d", DEV_NAME, vers_major, vers_minor);
		r = -ENODEV;
		goto ret1;
	}
	forced_downgrade = 0;
	if (force_version >= 0 && force_version < version) {
		version = force_version;
		forced_downgrade = 1;
	}

	if (DMA1 == -1) {
		DMA1 = 1;
	}
	if (DMA2 == -1 && version == SB_V4) {
		DMA2 = 5;
	}

	memset(FORMATS, 0, sizeof FORMATS);
	FORMATS[_FMT_U8] = 1;
	if (version == SB_V4) {
		FORMATS[_FMT_S8] = 1;
		FORMATS[_FMT_S16_LE] = 1;
		FORMATS[_FMT_U16_LE] = 1;
	}

	if (DSP_WRITE(SBCMD_SPEAKER_OFF)) goto cant_init;

	if (version == SB_V4) {
		OSS_MIXER_RECMASK = SOUND_MASK_SYNTH | SOUND_MASK_LINE | SOUND_MASK_MIC | SOUND_MASK_CD;
		OSS_MIXER_OUTMASK = SOUND_MASK_LINE | SOUND_MASK_MIC | SOUND_MASK_CD;
		MIX = MIX_V4;
	} else if (version == SB_V3) {
		OSS_MIXER_RECMASK = SOUND_MASK_LINE | SOUND_MASK_MIC | SOUND_MASK_CD;
		MIX = MIX_V3;
	} else if (version >= SB_V20) {
		MIX = MIX_V2;
	} else {
		MIX = MIX_DUMMY;
	}
	for (i = 0; MIX_NOT_EOF(&MIX[i]); i++) {
		OSS_MIXER_DEVMASK |= 1 << MIX[i].nr;
		if (MIX[i].right_mask) OSS_MIXER_STEREODEVS |= 1 << MIX[i].nr;
	}

	if (OSS_MIXER_DEVMASK)
		WRITE_MIXER_REG(SBMIXER_RESET, 0);

	KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);

	if (OSS_MIXER_DEVMASK & SOUND_MASK_SYNTH)
		SND_MIXER_WRITE(SOUND_MIXER_SYNTH, 0);
	
	if (OSS_MIXER_DEVMASK & SOUND_MASK_VOLUME)
		SND_MIXER_WRITE(SOUND_MIXER_VOLUME, (100 << 8) | 100);

	if (OSS_MIXER_DEVMASK & SOUND_MASK_PCM)
		SND_MIXER_WRITE(SOUND_MIXER_PCM, (100 << 8) | 100);
	
	if (OSS_MIXER_OUTMASK)
		SND_MIXER_WRITE_OUTSRC(SOUND_MASK_CD);

	OSS_INIT();

	if ((r = OSS_ALLOC_DMAPAGES(1, 65536 * (1 + (DMA2 >= 0))))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC DMA PAGES: %s", DEV_NAME, strerror(-r));
		goto ret1;
	}

	if ((r = ISADMA$REQUEST_CHANNEL(DMA1))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, DEV_NAME, "COULD NOT GET DMA %d: %s", DMA1, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET DMA", DEV_NAME);
		goto ret2;
	}
	if (DMA2 >= 0 && (r = ISADMA$REQUEST_CHANNEL(DMA2))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, DEV_NAME, "COULD NOT GET DMA %d: %s", DMA2, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET DMA", DEV_NAME);
		goto ret25;
	}

	SB_TIMEOUT.fn = SB_TIMEOUT_FN;
	SET_TIMER_NEVER(&SB_TIMEOUT);

	SB_IRQ_AST.fn = SB_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(IRQ, &SB_IRQ_CTRL, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_EXCLUSIVE, NULL, &SB_IRQ_AST, DEV_NAME)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, DEV_NAME, "COULD NOT GET IRQ %d: %s", IRQ, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", DEV_NAME);
		goto ret3;
	}
	_printf("%s: SOUND BLASTER %d.%d", DEV_NAME, vers_major, vers_minor);
	if (forced_downgrade) {
		_printf(" (DOWNGRADED TO %s)", vers_string[version]);
	}
	_printf("\n");
	_printf("%s: IO "IO_FORMAT", IRQ %d, DMA %d", DEV_NAME, IO, IRQ, DMA1);
	if (DMA2 >= 0) _printf(", DMA2 %d", DMA2);
	_printf("\n");

	r = KERNEL$REGISTER_DEVICE(DEV_NAME, "SB.SYS", 0, NULL, OSS_SET_ROOT, NULL, NULL, NULL, SB_UNLOAD, &LNTE, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", DEV_NAME, strerror(-r));
		goto ret4;
	}

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, DEV_NAME);
	DLRQ = KERNEL$TSR_IMAGE();
	return 0;

	ret4:
	KERNEL$RELEASE_IRQ(SB_IRQ_CTRL, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_EXCLUSIVE, NULL, &SB_IRQ_AST);
	ret3:
	if (DMA2 >= 0) ISADMA$RELEASE_CHANNEL(DMA2);
	ret25:
	ISADMA$RELEASE_CHANNEL(DMA1);
	ret2:
	OSS_FREE_DMAPAGES();
	ret1:
	KERNEL$UNREGISTER_IO_RANGE(&RANGE);
	ret0:
	return r;
}

static int SB_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(LNTE, argv))) return r;
	OSS_STOP_ALL(-1);
	KERNEL$RELEASE_IRQ(SB_IRQ_CTRL, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_EXCLUSIVE, NULL, &SB_IRQ_AST);
	if (DMA2 >= 0) ISADMA$RELEASE_CHANNEL(DMA2);
	ISADMA$RELEASE_CHANNEL(DMA1);
	OSS_FREE_DMAPAGES();
	KERNEL$UNREGISTER_IO_RANGE(&RANGE);
	*release = DLRQ;
	return 0;
}
