#include <SPAD/PCI.H>
#include <SPAD/PCI_IDS.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/IO.H>
#include <SPAD/TIMER.H>
#include <SYS/SOUNDCARD.H>
#include <SPAD/IRQ.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>
#include <SYS/PARAM.H>

#include "AC97.H"
#include "AC97REG.H"

#include "VIA82REG.H"

#define MAX_DMADESCS	128
#define MAX_RUN		MIN(PG_SIZE * PG_BANK, 0x00400000)
#define DEFAULT_MEMORY	(1024 * 1024)
#define MAX_MEMORY	MIN((__u64)PG_SIZE * PG_BANK * MAX_DMADESCS, (__u64)0x80000000)

#define AC97_TRIES	3
#define AC97_TIMEOUT	1024
#define AC97_CODEC_READY_TIMEOUT TIMEOUT_JIFFIES(JIFFIES_PER_SECOND * 3 / 4)
#define STOP_CHANNEL_TIMEOUT	TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 100)

#define CODEC	0

#define SPL_SND		SPL_AC97
#define OSS_N_STREAMS	2
#define OSS_IS_DUPLEX	1
#define OSS_DEFINE_TIMEOUT
#define OSS_PCI_DMA
#define OSS_PRECISE_TIMING
#define MIN_CHANNELS	0
#define MAX_CHANNELS	1
#define MIN_RATE	1
#define MAX_RATE	48000
#define MIN_FRAGS	2
#define MAX_FRAGS	MAX_DMADESCS
static const char FORMATS[_FMT_MAX + 1] = { 0, 0, 0, 1, 1, 0, 0, 0,
					    0, 0, 0, 0, 0, 0, 0, 0 };
#define DEFAULT_FORMAT	_FMT_S16_LE
#define MIN_FRAGSIZE	128

#include "OSS.H"

#define FLAG_8233		0x80
#define FLAG_DXS_MASK		0x0f
#define FLAG_DXS_NO		0x01
#define FLAG_DXS_PREFER_NO	0x02
#define FLAG_DXS_PREFER_YES	0x04
#define FLAG_DXS_FORCE		0x08

static const struct pci_id_s devices[] = {
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0x10, 0xff, "VIA 82C686A", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0x11, 0xff, "VIA 82C686B", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0x12, 0xff, "VIA 82C686C", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0x13, 0xff, "VIA 82C686D", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0x14, 0xff, "VIA 82C686E", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0x20, 0xff, "VIA 82C686H", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA 82C686 - unknown", FLAG_DXS_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x10, 0xff, "VIA 8233-Pre", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x20, 0xff, "VIA 8233C", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x30, 0xff, "VIA 8233", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x40, 0xff, "VIA 8233A", FLAG_8233 | FLAG_DXS_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x50, 0xff, "VIA 8235", FLAG_8233 | FLAG_DXS_PREFER_YES },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x60, 0xff, "VIA 8237", FLAG_8233 | FLAG_DXS_PREFER_YES },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0x70, 0xff, "VIA 8251", FLAG_8233 | FLAG_DXS_PREFER_YES },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA 8233 - unknown", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0},
};

static char dev_name[__MAX_STR_LEN];
static char *chip_name;
static unsigned long via_flags;

static pci_id_t pci_id;
static IO_RANGE range;
static io_t io;

static char desc_[sizeof(VIA_DMADESC) * (OSS_N_STREAMS * MAX_DMADESCS + 1) - 1];

static VIA_DMADESC *STREAM_DESC[OSS_N_STREAMS];
static __u32 STREAM_DEVADDR[OSS_N_STREAMS];
static vspace_dmaunlock_t *STREAM_UNMAP[OSS_N_STREAMS] = { NULL };
static unsigned STREAM_FRAG[OSS_N_STREAMS];
static jiffies_lo_t STREAM_JIFFIES[OSS_N_STREAMS];
static TIMER STREAM_TIMEOUT[OSS_N_STREAMS];

static AST VIA_IRQ_AST;
static IRQ_HANDLE *VIA_IRQ_CTRL;

static __finline__ __u8 READ_8(unsigned reg)
{
	return io_inb(io + reg);
}

static __finline__ __u16 READ_16(unsigned reg)
{
	return io_inw(io + reg);
}

static __finline__ __u32 READ_32(unsigned reg)
{
	return io_inl(io + reg);
}

static __finline__ void WRITE_8(unsigned reg, __u8 val)
{
	io_outb(io + reg, val);
}

static __finline__ void WRITE_16(unsigned reg, __u16 val)
{
	io_outw(io + reg, val);
}

static __finline__ void WRITE_32(unsigned reg, __u32 val)
{
	io_outl(io + reg, val);
}

static __u8 playback_channel = VIA_OUT_CHANNEL_BASE;

#define VIA_CHANNEL(rec)	(__likely(!(rec)) ? playback_channel : __unlikely(!(via_flags & FLAG_8233)) ? VIA_IN_CHANNEL_BASE : VIA_NEW_IN0_CHANNEL_BASE)
#define VIA_CHANNEL_MODE(ch)	((__unlikely(!(via_flags & FLAG_8233))) ? 0 : (ch) == VIA_NEW_3D_CHANNEL_BASE ? 2 : 1)

static int USE_DXS(int channels, int rate)
{
	if (via_flags & FLAG_DXS_NO) return 0;
	if (via_flags & FLAG_DXS_FORCE) return 1;
	if (via_flags & FLAG_DXS_PREFER_YES) return channels <= 1;
	return channels <= 1 && !(ac97_xtd_id & AC_XTD_ID_VRA) && rate != 48000;
}

void SND_FIXUP_PARAMS(HANDLE *h, int open_flags)
{
	int dxs;
	open_flags &= _O_KRNL_READ | _O_KRNL_WRITE;
	dxs = USE_DXS(FORMAT_CHANNELS(h), FORMAT_RATE(h));
	if (__likely(dxs)) {
		if (__unlikely(FORMAT_CHANNELS(h) > 1)) {
			FORMAT_SETCHANNELS(h, 1);
		}
		if (__unlikely(!open_flags) || __likely(open_flags == _O_KRNL_WRITE)) {
			return;
		}
		open_flags &= ~_O_KRNL_WRITE;
	}
	AC97_FIXUP_RATE(h, open_flags);
}

static void STOP_CHANNEL(unsigned ch)
{
	u_jiffies_lo_t j, jj;
	__u8 flag;
	if (READ_8(ch + VIA_STATUS) & VIA_STATUS_ACTIVE) {
		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_SGD_TERMINATE);
		j = KERNEL$GET_JIFFIES_LO();
		while (jj = KERNEL$GET_JIFFIES_LO(), READ_8(ch + VIA_STATUS) & VIA_STATUS_ACTIVE) {
			if (jj - j > STOP_CHANNEL_TIMEOUT) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNABLE TO STOP CHANNEL %02X, STATUS %02X", ch, READ_8(ch + VIA_STATUS));
				break;
			}
		}
	} else {
		WRITE_8(ch + VIA_CONTROL, 0);
	}
	if (ch < VIA_NEW_3D_CHANNEL_BASE && via_flags & FLAG_8233) {
		WRITE_8(ch + VIA_LEFTVOL, 0x3f);
		WRITE_8(ch + VIA_RIGHTVOL, 0x3f);
	} else {
		WRITE_8(ch + VIA_TYPE, 0x00);
	}
	flag = READ_8(ch + VIA_STATUS);
	if (flag) WRITE_8(ch + VIA_STATUS, flag);
	KERNEL$UDELAY(20);
}

void SND_START(int stream)
{
	unsigned i;
	VIA_DMADESC *d;
	VDESC v;
	VDMA dma;
	unsigned ch;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("SND_START AT SPL %08X", KERNEL$SPL);
	if (__likely(!STREAM_IS_RECORD(stream))) {
		if (!(via_flags & FLAG_8233)) {
			playback_channel = VIA_OUT_CHANNEL_BASE;
			AC97_SET_PLAYBACK_RATE(oss_rate[stream]);
		} else if (!USE_DXS(oss_channels[stream], oss_rate[stream])) {
			playback_channel = VIA_NEW_3D_CHANNEL_BASE;
			AC97_SET_PLAYBACK_RATE(oss_rate[stream]);
		} else {
			playback_channel = VIA_NEW_DXS0_CHANNEL_BASE;
			AC97_SET_PLAYBACK_RATE(48000);
		}
	} else {
		AC97_SET_RECORD_RATE(oss_rate[stream]);
	}
	/*__debug_printf("pages: %p\n", pages);*/
	for (i = 0, d = STREAM_DESC[stream]; i < oss_frags[stream]; i++, d++) {
		/*__debug_printf("%x : %08x\n", i, pages[i << fragsize_bits >> PAGESIZE_BITS].device);*/
		d->base = __32CPU2LE(DATA_PAGES[stream][i << oss_fragsize_bits[stream] >> PAGESIZE_BITS].device + ((i << oss_fragsize_bits[stream]) & ((1 << PAGESIZE_BITS) - 1)));
		d->len_flags = __32CPU2LE((1 << oss_fragsize_bits[stream]) | (__unlikely(i == oss_frags[stream] - 1) ? DESC_LEN_EOL : DESC_LEN_FLAG));
	}
	v.ptr = (unsigned long)STREAM_DESC[stream];
	v.len = sizeof(VIA_DMADESC) * oss_frags[stream];
	v.vspace = &KERNEL$VIRTUAL;
	dma.spl = SPL_X(SPL_SND);
	RAISE_SPL(SPL_VSPACE);
	STREAM_UNMAP[stream] = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ, &dma);
	if (__unlikely(dma.len != v.len))
		KERNEL$SUICIDE("SND_START: CAN'T DMALOCK BUFFER LIST");
	STREAM_DEVADDR[stream] = dma.ptr;

	ch = VIA_CHANNEL(stream);
	WRITE_32(ch + VIA_ADDR, dma.ptr);

	if (__unlikely(!VIA_CHANNEL_MODE(ch))) {
		__u8 type = VIA_TYPE_INT_FLAG | VIA_TYPE_INT_EOL;
		if (__likely(!STREAM_IS_RECORD(stream))) type |= VIA_TYPE_INT_LSAMPLE;
		else type |= VIA_TYPE_INT_LLINE | VIA_TYPE_REC_FIFO;
		if (__likely(oss_format[stream] == _FMT_S16_LE)) type |= VIA_TYPE_16_BIT;
		type |= oss_channels[stream] << __BSF_CONST(VIA_TYPE_STEREO);
		WRITE_8(ch + VIA_TYPE, type);
		KERNEL$UDELAY(20);

		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_SGD_TRIGGER);
	} else if (__likely(VIA_CHANNEL_MODE(ch) == 1)) {
		__u32 stoprate = 0;
		if (__likely(!STREAM_IS_RECORD(stream))) {
			stoprate = (__u64)oss_rate[stream] * 0x100000 / 48000;
			if (stoprate > VIA_STOPRATE_SAMPLE_RATE) stoprate = VIA_STOPRATE_SAMPLE_RATE;
		}
		if (__likely(oss_format[stream] == _FMT_S16_LE)) stoprate |= VIA_STOPRATE_16_BIT;
		stoprate |= oss_channels[stream] << __BSF_CONST(VIA_STOPRATE_STEREO);
		stoprate |= VIA_STOPRATE_STOP_IDX;
		WRITE_32(ch + VIA_STOPRATE, stoprate);

		if (STREAM_IS_RECORD(stream)) {
			WRITE_8(ch + VIA_TYPE, VIA_TYPE_REC_FIFO);
		}
		KERNEL$UDELAY(20);

		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_INT_ON_FLAG | VIA_CONTROL_INT_ON_EOL | VIA_CONTROL_AUTO_RESTART | VIA_CONTROL_SGD_TRIGGER);

		if (!STREAM_IS_RECORD(stream)) {
			WRITE_8(ch + VIA_LEFTVOL, 0x00);
			WRITE_8(ch + VIA_RIGHTVOL, 0x00);
		}
	} else {
		__u8 type;
		__u32 stoprate;
		stoprate = VIA_STOPRATE_STOP_IDX;
		switch (oss_channels[stream] + 1) {
			case 1: stoprate |= (1<<0) | (1<<4); break;
			case 2: stoprate |= (1<<0) | (2<<4); break;
			case 3: stoprate |= (1<<0) | (2<<4) | (5<<8); break;
			case 4: stoprate |= (1<<0) | (2<<4) | (3<<8) | (4<<12); break;
			case 5: stoprate |= (1<<0) | (2<<4) | (3<<8) | (4<<12) | (5<<16); break;
			case 6: stoprate |= (1<<0) | (2<<4) | (3<<8) | (4<<12) | (5<<16) | (6<<20); break;
			default: KERNEL$SUICIDE("SND_START: BAD CHANNEL NUMBER: %d, STREAM %d", oss_channels[stream], stream);
		}
		WRITE_32(ch + VIA_STOPRATE, stoprate);
		type = (oss_channels[stream] + 1) << __BSF_CONST(VIA_CH4_TYPE_CHANNELS);
		if (__likely(oss_format[stream] == _FMT_S16_LE)) type |= VIA_CH4_TYPE_16BIT;
		WRITE_8(ch + VIA_TYPE, type);
		KERNEL$UDELAY(20);

		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_INT_ON_FLAG | VIA_CONTROL_INT_ON_EOL | VIA_CONTROL_AUTO_RESTART | VIA_CONTROL_SGD_TRIGGER);
	}
	STREAM_FRAG[stream] = 0;
	OSS_STREAM_FLAGS[stream] |= OSS_RUNNING;
	KERNEL$DEL_TIMER(&STREAM_TIMEOUT[stream]);
	STREAM_JIFFIES[stream] = OSS_STREAM_TIMEOUT(stream);
	KERNEL$SET_TIMER(STREAM_JIFFIES[stream], &STREAM_TIMEOUT[stream]);
}

void SND_STOP(int stream)
{
	STOP_CHANNEL(VIA_CHANNEL(stream));
	if (__likely(STREAM_UNMAP[stream] != NULL)) STREAM_UNMAP[stream](STREAM_DEVADDR[stream]), STREAM_UNMAP[stream] = NULL;
	KERNEL$DEL_TIMER(&STREAM_TIMEOUT[stream]);
	SET_TIMER_NEVER(&STREAM_TIMEOUT[stream]);
}

void SND_PAUSE(int stream)
{
	unsigned ch;
	__u8 ctrl;
	KERNEL$DEL_TIMER(&STREAM_TIMEOUT[stream]);
	SET_TIMER_NEVER(&STREAM_TIMEOUT[stream]);
	ch = VIA_CHANNEL(stream);
	ctrl = READ_8(ch + VIA_CONTROL);
	WRITE_8(ch + VIA_CONTROL, (ctrl & ~(VIA_CONTROL_SGD_TERMINATE | VIA_CONTROL_SGD_TRIGGER)) | VIA_CONTROL_SGD_PAUSE);
}

void SND_RESUME(int stream)
{
	unsigned ch;
	__u8 ctrl;
	KERNEL$DEL_TIMER(&STREAM_TIMEOUT[stream]);
	KERNEL$SET_TIMER(STREAM_JIFFIES[stream], &STREAM_TIMEOUT[stream]);
	ch = VIA_CHANNEL(stream);
	ctrl = READ_8(ch + VIA_CONTROL);
	WRITE_8(ch + VIA_CONTROL, ctrl & ~(VIA_CONTROL_SGD_TERMINATE | VIA_CONTROL_SGD_TRIGGER | VIA_CONTROL_SGD_PAUSE));
}

unsigned SND_GET_PTR(int stream)
{
	unsigned val;
	unsigned ch;
	__u32 ptr;
	ch = VIA_CHANNEL(stream);
	ptr = READ_32(ch + VIA_BLOCK_COUNT);
	val = (ptr >> (__BSF_CONST(VIA_BLOCK_COUNT_INDEX) - oss_fragsize_bits[stream])) + (1 << oss_fragsize_bits[stream]) - (ptr & VIA_BLOCK_COUNT_OFFSET);
	if (__unlikely(val >= oss_frags[stream] << oss_fragsize_bits[stream])) val = 0;
	return val;
}

__finline__ void SND_SUBMIT_FRAGMENT(int stream, int frag)
{
}

static void INTERRUPT(int stream);

static DECL_IRQ_AST(VIA_IRQ, SPL_SND, AST)
{
	int i;
	for (i = 0; i < OSS_N_STREAMS; i++) INTERRUPT(i);
	KERNEL$UNMASK_IRQ(VIA_IRQ_CTRL);
	RETURN;
}

static void INTERRUPT(int stream)
{
	__u32 pos;
	__u8 frag;
	unsigned frags;
	unsigned ch = VIA_CHANNEL(stream);
	__u8 status = READ_8(ch + VIA_STATUS);
	if (__likely(!(status & (VIA_STATUS_EOL | VIA_STATUS_FLAG | VIA_STATUS_STOPPED)))) return;
	WRITE_8(ch + VIA_STATUS, status & (VIA_STATUS_EOL | VIA_STATUS_FLAG | VIA_STATUS_STOPPED));
	pos = READ_32(ch + VIA_BLOCK_COUNT);
	frag = (pos >> __BSF_CONST(VIA_BLOCK_COUNT_INDEX)) + !(pos & VIA_BLOCK_COUNT_OFFSET);

	if (__unlikely(!(OSS_STREAM_FLAGS[stream] & OSS_RUNNING))) return;
	frags = (oss_frags[stream] + frag - STREAM_FRAG[stream]) % oss_frags[stream];
	STREAM_FRAG[stream] = frag;
	if (__likely(frags)) {
		do {
			if (__unlikely(OSS_STREAM_INTR(stream))) return;
		} while (__unlikely(--frags));
		KERNEL$DEL_TIMER(&STREAM_TIMEOUT[stream]);
		KERNEL$SET_TIMER(STREAM_JIFFIES[stream], &STREAM_TIMEOUT[stream]);
	}
}

static void STREAM_TIMEOUT_FN(TIMER *t)
{
	int stream;
	LOWER_SPL(SPL_SND);
	SET_TIMER_NEVER(t);
	stream = t - STREAM_TIMEOUT;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INTERRUPT TIMEOUT ON %s STREAM %d", STREAM_IS_RECORD(stream) ? "RECORD" : "PLAYBACK", stream);
	OSS_STOP(stream);
}

static int read_ac97(unsigned reg)
{
	unsigned i;
	__u32 x;
	for (i = 0; i < AC97_TRIES * AC97_TIMEOUT; i++) {
		if (__likely(!(i % AC97_TIMEOUT))) {
			x = VIA_AC97_CTRL_READ | (reg << __BSF_CONST(VIA_AC97_CTRL_ADDRESS));
			if (!CODEC) x |= VIA_AC97_CTRL_1ST_CODEC_VALID;
			else x |= VIA_AC97_CTRL_2ND_CODEC | VIA_AC97_CTRL_2ND_CODEC_VALID;
			WRITE_32(VIA_AC97_CTRL, x);
			KERNEL$UDELAY(20);
		}
		KERNEL$UDELAY(1);
		x = READ_32(VIA_AC97_CTRL);
		if (x & VIA_AC97_CTRL_BUSY) continue;
		if (!(x & (!CODEC ? VIA_AC97_CTRL_1ST_CODEC_VALID : VIA_AC97_CTRL_2ND_CODEC_VALID))) continue;
		goto got_it;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC %d READ TIMED OUT, REGISTER %02X, STATUS %08X", CODEC, reg, x);
	return -1;

	got_it:
	KERNEL$UDELAY(25);
	x = READ_32(VIA_AC97_CTRL);
	if (x & VIA_AC97_CTRL_BUSY ||
	    !(x & (!CODEC ? VIA_AC97_CTRL_1ST_CODEC_VALID : VIA_AC97_CTRL_2ND_CODEC_VALID)) ||
	    (x & VIA_AC97_CTRL_ADDRESS) >> __BSF_CONST(VIA_AC97_CTRL_ADDRESS) != reg) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC %d HAS INVALID STATUS WHEN READING REGISTER %02X: %08X", CODEC, reg, x);
		return -1;
	}
	return x & VIA_AC97_CTRL_VALUE;
}

static void write_ac97(unsigned reg, int val)
{
	unsigned i;
	__u32 x = (reg << __BSF_CONST(VIA_AC97_CTRL_ADDRESS)) | val;
	if (CODEC) x |= VIA_AC97_CTRL_2ND_CODEC;
	WRITE_32(VIA_AC97_CTRL, x);
	KERNEL$UDELAY(10);
	for (i = 0; i < AC97_TIMEOUT; i++) {
		KERNEL$UDELAY(1);
		if (!(READ_32(VIA_AC97_CTRL) & VIA_AC97_CTRL_BUSY)) return;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC %d WRITE TIMED OUT, REGISTER %02X, VALUE %04X", CODEC, reg, val);
}

static const HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	OSS_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	OSS_INSTANTIATE,	/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

#include "OSS.I"

#include "AC97.I"

static int RESET_AC97(void)
{
	u_jiffies_lo_t j, jj;
	__u32 ac97_ctrl;
	__u8 ac_stat = PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_STATUS);
	if (!(ac_stat & (!CODEC ? VIA_PCI_AC_STATUS_C00_READY : VIA_PCI_AC_STATUS_C01_READY))) {
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL, VIA_PCI_AC_CTRL_ENABLE | VIA_PCI_AC_CTRL_RESET | VIA_PCI_AC_CTRL_SYNC);
		KERNEL$UDELAY(100);
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL, 0);
		KERNEL$UDELAY(100);
		enable_it:
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL, VIA_PCI_AC_CTRL_ENABLE | VIA_PCI_AC_CTRL_RESET | VIA_PCI_AC_CTRL_VRA | VIA_PCI_AC_CTRL_PCM);
		KERNEL$UDELAY(100);
	} else {
		__u8 ac_ctrl = PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL);
		if (ac_ctrl != (VIA_PCI_AC_CTRL_ENABLE | VIA_PCI_AC_CTRL_RESET | VIA_PCI_AC_CTRL_VRA | VIA_PCI_AC_CTRL_PCM)) goto enable_it;
	}
	j = KERNEL$GET_JIFFIES_LO();
	while (jj = KERNEL$GET_JIFFIES_LO(), !((ac_stat = PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_STATUS)) & (!CODEC ? VIA_PCI_AC_STATUS_C00_READY : VIA_PCI_AC_STATUS_C01_READY))) {
		if (jj - j > AC97_CODEC_READY_TIMEOUT) {
			break;
		}
		KERNEL$SLEEP(1);
	}
	ac97_ctrl = READ_32(VIA_AC97_CTRL);
	if (ac97_ctrl & VIA_AC97_CTRL_BUSY) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "CODEC NOT READY AFTER RESET, STATUS %02X, CTRL %02X, AC97_CTRL %08X", ac_stat, PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL), ac97_ctrl);
		return -ENXIO;
	}
	if (!(via_flags & FLAG_8233)) {
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_FN_NMI_CTRL, VIA_PCI_FN_NMI_CTRL_IRQ_SMI);
		WRITE_32(VIA_GPI_INT_ENABLE, 0);
	}
	return 0;
}

static void PCM_RESET(void)
{
	unsigned ch;
	for (ch = 0; ch < (via_flags & FLAG_8233 ? 0x80 : 0x30); ch++) {
		if (ch != VIA_NEW_FM_CHANNEL_BASE) STOP_CHANNEL(ch);
	}
}

static void *lnte, *dlrq;
static int VIA_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	int memory = -1;
	int dxs = -1;
	unsigned pages, page_size;
	int irq;
	const char * const *arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	const char *opt, *optend, *str;
	int r;
	int i;
	VIA_DMADESC *d;
	char gstr[__MAX_STR_LEN];
	static const struct __param_table dxs_params[5] = {
		"NO", __PARAM_BOOL, ~0, FLAG_DXS_NO,
		"PREFER_NO", __PARAM_BOOL, ~0, FLAG_DXS_PREFER_NO,
		"PREFER_YES", __PARAM_BOOL, ~0, FLAG_DXS_PREFER_YES,
		"FORCE", __PARAM_BOOL, ~0, FLAG_DXS_FORCE,
		NULL, 0, 0, 0,
	};
	void *dxs_vars[5];
	static const struct __param_table params[3] = {
		"MEMORY", __PARAM_INT, PG_SIZE, MAX_DMADESCS * MAX_RUN + 1,
		"DXS", __PARAM_EXTD, (long)&dxs_params, 0,
		NULL, 0, 0, 0,
	};
	void *vars[3];
	vars[0] = &memory;
	vars[1] = dxs_vars;
	vars[2] = NULL;
	dxs_vars[0] = &dxs;
	dxs_vars[1] = &dxs;
	dxs_vars[2] = &dxs;
	dxs_vars[3] = &dxs;
	dxs_vars[4] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "VIA82: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (memory < 0) {
		memory = MAX_DMADESCS * __PAGE_CLUSTER_SIZE;
		if (memory > DEFAULT_MEMORY) memory = DEFAULT_MEMORY;
	} else {
		if ((memory - 1) || memory > MAX_MEMORY) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "VIA82: BAD MEMORY SIZE (MUST BE POWER OF 2 AND BETWEEN %u AND %u)", (unsigned)__PAGE_CLUSTER_SIZE, (unsigned)MAX_MEMORY);
			r = -EBADSYN;
			goto ret0;
		}
		if (memory < __PAGE_CLUSTER_SIZE)
			memory = __PAGE_CLUSTER_SIZE;
	}
	if (PCI$FIND_DEVICE(devices, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &via_flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "VIA82: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	pci_id = id;
	_snprintf(dev_name, sizeof(dev_name), "SND$VIA82@" PCI_ID_FORMAT, id);
	if (dxs >= 0) via_flags = (via_flags & ~FLAG_DXS_MASK) | dxs;
	if (!(via_flags & FLAG_8233) && via_flags & FLAG_DXS_MASK & ~FLAG_DXS_NO) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCAPABILITY, dev_name, "VIA82C686 DOESN'T HAVE DXS");
		via_flags = (via_flags & ~FLAG_DXS_MASK) | FLAG_DXS_NO;
	}
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	io = PCI$READ_IO_RESOURCE(id, 0);
	irq = PCI$READ_INTERRUPT_LINE(id);
	if (!io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	range.start = io;
	range.len = VIA82_REGSPACE;
	range.name = dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, range.start, range.start + range.len - 1, strerror(-r));
		goto ret1;
	}
	r = RESET_AC97();
	if (r) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T RESET AC97 LINK", dev_name);
		goto ret15;
	}
	r = AC97_INIT();
	if (r) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE AC97 CODEC", dev_name);
		goto ret15;
	}
	PCM_RESET();
	OSS_INIT();
	pages = MAX_DMADESCS;
	page_size = memory / MAX_DMADESCS;
	while (page_size < __PAGE_CLUSTER_SIZE) {
		page_size <<= 1;
		pages >>= 1;
	}
	if ((r = OSS_ALLOC_DMAPAGES(pages, page_size))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC DMA PAGES: %s", dev_name, strerror(-r));
		goto ret15;
	}

	d = (VIA_DMADESC *)(((unsigned long)desc_ + 7) & ~7);
	for (i = 0; i < OSS_N_STREAMS; i++) {
		STREAM_DESC[i] = d;
		d += MAX_DMADESCS;
		INIT_TIMER(&STREAM_TIMEOUT[i]);
		STREAM_TIMEOUT[i].fn = STREAM_TIMEOUT_FN;
		SET_TIMER_NEVER(&STREAM_TIMEOUT[i]);
	}

	VIA_IRQ_AST.fn = VIA_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(irq, &VIA_IRQ_CTRL, IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &VIA_IRQ_AST, dev_name)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret2;
	}

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, pci_id));
	_printf("%s: IO "IO_FORMAT", IRQ %d\n", dev_name, io, irq);

	r = KERNEL$REGISTER_DEVICE(dev_name, "VIA82.SYS", 0, NULL, OSS_SET_ROOT, NULL, NULL, NULL, VIA_UNLOAD, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret3;
	}

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, dev_name);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	ret3:
	KERNEL$RELEASE_IRQ(VIA_IRQ_CTRL, IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &VIA_IRQ_AST);
	ret2:
	OSS_FREE_DMAPAGES();
	ret15:
	KERNEL$UNREGISTER_IO_RANGE(&range);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int VIA_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	KERNEL$MASK_IRQ(VIA_IRQ_CTRL);
	OSS_STOP_ALL(-1);
	KERNEL$RELEASE_IRQ(VIA_IRQ_CTRL, IRQ_REQUEST_MASKED | IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &VIA_IRQ_AST);
	OSS_FREE_DMAPAGES();
	KERNEL$UNREGISTER_IO_RANGE(&range);
	PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
