#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <UNISTD.H>
#include <FCNTL.H>
#include <ERRNO.H>
#include <STRING.H>
#include <SYS/STAT.H>
#include <ARCH/PAGE.H>

#include "STRUCT.H"

#include <KERNEL/GETHSIZE.I>

#define MIN_SIZE	((__u64)(SWAP_MIN_SIZE > 2 * __PAGE_CLUSTER_SIZE ? SWAP_MIN_SIZE : 2 * __PAGE_CLUSTER_SIZE))
#define MAX_SPAD_PAGES	((__u64)PG_SIZE * PG_BANK * 8)

int main(int argc, const char * const argv[])
{
	int linux20 = 0;
	int expand = 0;
	int trunc = 0;
	int created = 0;
	char *filename = NULL;
	__u64 ssize = 0;
	__u64 ssize_pg;
	const char * const *arg = argv;
	int state = 0;
	int h;
	int r;
	int w;
	union swap_header swp;
	static const struct __param_table params[6] = {
		"", __PARAM_STRING, 1, __MAX_STR_LEN,
		"", __PARAM_UNSIGNED_INT64, SWAP_MIN_SIZE, 1,
		"LINUX20", __PARAM_BOOL, ~0, 1,
		"EXPAND", __PARAM_BOOL, ~0, 1,
		"TRUNCATE", __PARAM_BOOL, ~0, 1,
		NULL, 0, 0, 0,
	};
	void *vars[6];
	vars[0] = &filename;
	vars[1] = &ssize;
	vars[2] = &linux20;
	vars[3] = &expand;
	vars[4] = &trunc;
	vars[5] = NULL;
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		badsyn:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: SYNTAX ERROR");
		return -EBADSYN;
	}
	if (__unlikely(!filename)) goto badsyn;
	if (__unlikely(expand) && __unlikely(!ssize)) goto badsyn;
	if (__unlikely(trunc) && __unlikely(!ssize)) goto badsyn;
	if (__unlikely(expand) && __unlikely(trunc)) goto badsyn;
	h = open(filename, O_RDWR);
	if (__unlikely(h == -1)) {
		if (__unlikely(!ssize)) {
			if (errno == ENOENT) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: CAN'T CREATE FILE BECAUSE SIZE NOT SPECIFIED");
			else goto open_err;
			return -errno;
		}
		h = open(filename, O_RDWR | O_CREAT, 0600);
		if (__unlikely(h == -1)) {
			open_err:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: CAN'T OPEN OR CREATE %s: %s", filename, strerror(errno));
			return -errno;
		}
		created = 1;
	}
	if (__unlikely(expand)) {
		if (__unlikely(ftruncate(h, ssize))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: CAN'T EXPAND FILE: %s", strerror(errno));
			r = -errno;
			goto ret;
		}
	}
	if (__unlikely(trunc)) {
		if (__unlikely(ftruncate(h, 0))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: CAN'T TRUNCATE FILE: %s", strerror(errno));
			r = -errno;
			goto ret;
		}
	}
	if (__likely(!ssize)) {
		ssize = gethsize(h);
		if (__unlikely(ssize < SWAP_MIN_SIZE)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: TOO SMALL FILE SIZE: %"__64_format"u", ssize);
			r = -EFTYPE;
			goto ret;
		}
	}
	ssize_pg = ssize >> LINUX_PAGE_SIZE_BITS;

	if (__unlikely(linux20) && __unlikely(ssize_pg > sizeof(swp.magic.reserved) * 8)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: TOO LONG FILE FOR LINUX20 FORMAT (SIZE %"__64_format"u, MAX %"__64_format"u)", ssize, (__u64)(sizeof(swp.magic.reserved) * 8) << LINUX_PAGE_SIZE_BITS);
		r = -EFBIG;
		goto ret;
	}
	if (__unlikely(ssize < MIN_SIZE)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: TOO SMALL FILE (SIZE %"__64_format"u, MAX %"__64_format"u)", ssize, MIN_SIZE);
		r = -EFTYPE;
		goto ret;
	}
	if (__unlikely(ssize >> __PAGE_CLUSTER_BITS > MAX_SPAD_PAGES)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: TOO LONG FILE (SIZE %"__64_format"u, MAX %"__64_format"u)", ssize, MAX_SPAD_PAGES << __PAGE_CLUSTER_BITS);
		r = -EFBIG;
		goto ret;
	}

	memset(&swp, 0, sizeof swp);
	if (__likely(!linux20)) {
		swp.info.version = 1;
		swp.info.last_page = ssize_pg - 1;
		memcpy(swp.magic.magic, SWAP_LINUX_V2_MAGIC, strlen(SWAP_LINUX_V2_MAGIC));
	} else {
		unsigned i;
		for (i = 1; i < ssize_pg; i++)
			swp.magic.reserved[i >> 3] |= 1 << (i & 7);
		memcpy(swp.magic.magic, SWAP_LINUX_V1_MAGIC, strlen(SWAP_LINUX_V1_MAGIC));
	}
	if (__unlikely((w = pwrite(h, &swp, sizeof swp, 0)) != sizeof swp)) {
		if (w < 0) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: CAN'T WRITE: %s", strerror(errno));
			r = -errno;
			goto ret;
		} else {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: SHORT WRITE: %u OF %u BYTES", w, (unsigned)sizeof swp);
			r = -EEOF;
			goto ret;
		}
	}
	if (__unlikely(fsync(h))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MKSWAP: CAN'T FSYNC: %s", strerror(errno));
		r = -errno;
		goto ret;
	}
	close(h);
	return 0;

	ret:
	close(h);
	if (created) unlink(filename);
	return r;
}
