#include <SPAD/LIST.H>
#include <SPAD/VM.H>

#include "SWAPPER.H"

static const VMENTITY_T swap_calls = { SWAP_CHECKMAP, SWAP_WRITE, SWAP_SWAPOUT, "SWAP" };

int swap_vm_entity;

int wantfree_active;
IORQ wantfree;
XLIST_HEAD wantfree_pages;
WQ_DECL(wantfree_done, "SWAPPER$WANTFREE_DONE");

void *COPY_PAGEDIR(PAGEDIR **d, SWAPNODE *s)
{
	int i;
	QUOTA *zap;
	PAGEDIR *p, *np, *r;
	ASSERT_LOCKED();
	check_quota(ACCT_FROM_SWAPNODE(s), "COPY_PAGEDIR 1");
	if (__unlikely(!(np = __slalloc(&pagedirs)))) {
		return (void *)(UNPAGED_ALLOC_FAILED() + 1);
	}
	QALLOC(&s->unpageq, sizeof(PAGEDIR), unpageq_isroot, unpageq_parent, Q_NULL_CALL, zap, {
		__slow_slfree(np);
	check_quota(ACCT_FROM_SWAPNODE(s), "COPY_PAGEDIR 2");
		OUT_OF_UNPAGED(zap);
		return (void *)1;
	});
	np->refcount = 1;
	np->account = ACCT_FROM_SWAPNODE(s);
	p = *d;
	*d = np;
	for (i = 0; i < PGDIR_SIZE; i++) {
		if ((r = np->pagedir[i] = p->pagedir[i])) {
			r->refcount++;
			if (__unlikely(ACCT_IS_LDCACHE(r->account))) REFERENCE_LDCACHE(s, r->account, 1, &KERNEL$PROC_KERNEL);
		}
	}
	p->refcount--;
	if (__unlikely(ACCT_IS_LDCACHE(p->account))) REFERENCE_LDCACHE(s, p->account, -1, &KERNEL$PROC_KERNEL);
	check_quota(ACCT_FROM_SWAPNODE(s), "COPY_PAGEDIR 3");
	return NULL;
}

void *SWAP_ZAP(PAGEDIR **pp, unsigned depth, unsigned long from, unsigned long to, unsigned long node, PROC *owner)
{
	void *v;
	PAGEDIR *p = *pp;
	PAGE *page;
	ASSERT_LOCKED();
	check_quota(node, "SWAP_ZAP 1");
	if (__unlikely(!p->refcount)) KERNEL$SUICIDE("SWAP_ZAP: CLEARING UNREFERENCED NODE");
	if (__likely(depth)) {
		unsigned i, j;
		unsigned long l1, l2, lm;
		int dp;
	check_quota(node, "SWAP_ZAP 2");
		depth--;
		dp = depth * PGDIR_SIZE_BITS;
		i = from >> dp;
		j = to >> dp;
		if (__unlikely(i >= PGDIR_SIZE) || __unlikely(j >= PGDIR_SIZE) || __unlikely(i > j))
			KERNEL$SUICIDE("SWAP_ZAP: BAD LIMITS, DEPTH %d, FROM %lX, TO %lX, I %u, J %u", depth, from, to, i, j);
		lm = (1 << dp) - 1;
		l1 = from & lm;
		l2 = to & lm;
		if (__unlikely(p->refcount > 1)) {
			if (__likely(!from) && __likely(j == PGDIR_SIZE - 1) && __likely(l2 == lm)) {
				p->refcount--;
				if (__unlikely(ACCT_IS_LDCACHE(p->account)) && __likely(!ACCT_IS_LDCACHE(node))) REFERENCE_LDCACHE(ACCT_SWAPNODE(node), p->account, -1, owner);
				*pp = NULL;
	check_quota(node, "SWAP_ZAP 3");
				return NULL;
			}
			if (__unlikely(ACCT_IS_LDCACHE(node)))
				KERNEL$SUICIDE("SWAP_ZAP: COPYING PAGEDIR ON LDCACHE");
			if (__unlikely((v = COPY_PAGEDIR(pp, ACCT_SWAPNODE(node))) != NULL)) return v;
			p = *pp;
		}
		if (i == j) {
			if (p->pagedir[i]) return SWAP_ZAP(&p->pagedir[i], depth, l1, l2, node, owner);
	check_quota(node, "SWAP_ZAP 4");
			return NULL;
		}
		if (p->pagedir[i]) if (__unlikely((v = SWAP_ZAP(&p->pagedir[i], depth, l1, lm, node, owner)) != NULL)) return v;
		if (p->pagedir[j]) if (__unlikely((v = SWAP_ZAP(&p->pagedir[j], depth, 0, l2, node, owner)) != NULL)) return v;
		for (i++; i < j; i++)
			if (p->pagedir[i]) if (__unlikely((v = SWAP_ZAP(&p->pagedir[i], depth, 0, lm, node, owner)) != NULL)) return v;
		if (__unlikely(!from) && __unlikely(j == PGDIR_SIZE - 1) && __unlikely(l2 == lm)) {
#if __DEBUG >= 1
			for (i = 0; i < PGDIR_SIZE; i++) if (__unlikely(p->pagedir[i] != NULL)) KERNEL$SUICIDE("SWAP_ZAP: PAGEDIR NOT ERASED, USED INDEX %d, DEPTH %d, FROM %lX, TO %lX", i, depth, from, to);
#endif
			QFREE(&ACCT_SWAPNODE(p->account)->unpageq, sizeof(PAGEDIR), unpageq_isroot, unpageq_parent, Q_NULL_CALL);
			__slfree(p);
			*pp = NULL;
	check_quota(node, "SWAP_ZAP 5");
			return NULL;
		}
	check_quota(node, "SWAP_ZAP 6");
		return NULL;
	}
	check_quota(node, "SWAP_ZAP 7");
	if (__unlikely(p->refcount > 1)) {
		p->refcount--;
		if (__unlikely(ACCT_IS_LDCACHE(p->account)) && __likely(!ACCT_IS_LDCACHE(node))) REFERENCE_LDCACHE(ACCT_SWAPNODE(node), p->account, -1, owner);
		/* otherwise refcount is handled in SWAP_FREE_PAGENODE */
		*pp = NULL;
	check_quota(node, "SWAP_ZAP 8");
		return NULL;
	}
#define pg ((PAGENODE *)p)
	page = pg->page;
	if (__likely(page != NULL)) {
		WQ *wq;
	check_quota(node, "SWAP_ZAP 9");
		wq = KERNEL$VM_UNMAP_PAGE(page);
		if (__unlikely(wq != NULL)) {
	check_quota(node, "SWAP_ZAP 10");
			page->fnode = (void *)((pg->swap_pos << 1) | PAGE_FNODE_FREED);
			SWAPPAGE_ADD_TO_WANTFREE(page, wq);
			LOWER_SPL(SPL_FS);
			goto xx;
		}
	check_quota(node, "SWAP_ZAP 11");
		LOWER_SPL(SPL_FS);
		free_page(page, owner);
	}
	check_quota(node, "SWAP_ZAP 12");
	if (__likely(pg->swap_pos)) CLEAR_SWPALLOC_BIT(pg->swap_pos);
	check_quota(node, "SWAP_ZAP 13");
	xx:
	check_quota(node, "SWAP_ZAP 14");
	SWAP_FREE_PAGENODE(pg);
	*pp = NULL;
	check_quota(node, "SWAP_ZAP 15");
	return NULL;
#undef pg
}

static void SWAP_DESTROY_ALL_DATA(SWAPNODE *s)
{
	int i;
	for (i = 0; i < TOP_PGDIR_SIZE; i++) if (s->pagedir[i]) {
		if (__unlikely(SWAP_ZAP(&s->pagedir[i], PGDIR_DEPTH, 0, (1 << (PGDIR_DEPTH * PGDIR_SIZE_BITS)) - 1, ACCT_FROM_SWAPNODE(s), &KERNEL$PROC_KERNEL) != NULL))
			KERNEL$SUICIDE("SWAP_DESTROY_SWAPNODE_AND_DATA: SWAP_ZAP RETURNED NONZERO");
		if (__unlikely(s->pagedir[i] != NULL))
			KERNEL$SUICIDE("SWAP_DESTROY_SWAPNODE_AND_DATA: CAN'T ZAP PAGEDIR");
	}
}

void SWAP_DESTROY_SWAPNODE_AND_DATA(SWAPNODE *s)
{
	int i;
	SWAPNODE *oldswapnode = NULL;
	ASSERT_LOCKED();
	recurse:
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_SWAPNODE_AND_DATA 1");
	for (i = 0; i < CHILDHASH_SIZE; i++) if (__unlikely(!XLIST_EMPTY(&s->childhash[i]))) {
		if (!oldswapnode) oldswapnode = s;
		s = LIST_STRUCT(s->childhash[i].next, SWAPNODE, hash);
		goto recurse;
	}
	while (__unlikely(!XLIST_EMPTY(&s->handles))) {
		HANDLE *h = LIST_STRUCT(s->handles.next, HANDLE, fnode_entry);
		KERNEL$DETACH_HANDLE(h);
	}
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_SWAPNODE_AND_DATA 2");
	SWAP_DESTROY_ALL_DATA(s);
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_SWAPNODE_AND_DATA 3");
#if __DEBUG >= 1
	if (__unlikely(!XLIST_EMPTY(&s->ldrefs)))
		KERNEL$SUICIDE("SWAP_DESTROY_SWAPNODE_AND_DATA: LDREFS LEAKED");
#endif
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_SWAPNODE_AND_DATA 4");
#ifdef SWAP_TTY
	if (__unlikely(s->tty != NULL))
		SWAPTTY_FREE(s->tty);
#endif
	QDONE2(&s->pageq, "PAGEQ", &s->unpageq, "UNPAGEQ");
	if (__likely(s->parent != NULL)) {
		QFREE(&s->parent->unpageq, sizeof(SWAPNODE), unpageq_isroot, unpageq_parent, Q_NULL_CALL);
	}
	DEL_FROM_LIST(&s->hash);
	__slfree(s);
	if (__unlikely(oldswapnode != NULL)) {
		s = oldswapnode;
		oldswapnode = 0;
		goto recurse;
	}
}

void *SWAP_DESTROY_DATA(SWAPNODE *s, unsigned long start, unsigned long end, PROC *owner)
{
	int i, j;
	unsigned long l1, l2;
	void *v;
	ASSERT_LOCKED();
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_DATA 1");
	start >>= PG_SIZE_BITS + PG_CLUSTER_BITS;
	end >>= PG_SIZE_BITS + PG_CLUSTER_BITS;
	end--;
	if (__unlikely(start >= PGDIR_DIV * TOP_PGDIR_SIZE)) goto ret0;
	if (__unlikely(end >= PGDIR_DIV * TOP_PGDIR_SIZE)) end = PGDIR_DIV * TOP_PGDIR_SIZE - 1;
	if (__unlikely(start > end)) goto ret0;
	i = start >> (PGDIR_SIZE_BITS * PGDIR_DEPTH);
	j = end >> (PGDIR_SIZE_BITS * PGDIR_DEPTH);
	l1 = start & (PGDIR_DIV - 1);
	l2 = end & (PGDIR_DIV - 1);
	if (i == j) {
		if (s->pagedir[i]) if (__unlikely((v = SWAP_ZAP(&s->pagedir[i], PGDIR_DEPTH, l1, l2, ACCT_FROM_SWAPNODE(s), owner)) != NULL)) goto ret1;
	} else {
		if (s->pagedir[i]) if (__unlikely((v = SWAP_ZAP(&s->pagedir[i], PGDIR_DEPTH, l1, PGDIR_DIV - 1, ACCT_FROM_SWAPNODE(s), owner)) != NULL)) goto ret1;
		if (s->pagedir[j]) if (__unlikely((v = SWAP_ZAP(&s->pagedir[j], PGDIR_DEPTH, 0, l2, ACCT_FROM_SWAPNODE(s), owner)) != NULL)) goto ret1;
		for (i++; i < j; i++) if (s->pagedir[i]) if (__unlikely((v = SWAP_ZAP(&s->pagedir[i], PGDIR_DEPTH, 0, PGDIR_DIV - 1, ACCT_FROM_SWAPNODE(s), owner)) != NULL)) goto ret1;
	}
	ret0:
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_DATA 2");
	return 0;
	ret1:
	check_quota(ACCT_FROM_SWAPNODE(s), "SWAP_DESTROY_DATA 3");
	return v;
}

void SWAP_FORK(SWAPNODE *sf, SWAPNODE *st)
{
	int i;
	PAGEDIR *r;
	check_quota(ACCT_FROM_SWAPNODE(sf), "SWAP_FORK 1");
	SWAP_LOCK(SW_LOCK);
	if (__unlikely(st->ldrefsptr != st)) {
		goto unlock_ret;
	}
	if (__unlikely(!XLIST_EMPTY(&st->ldrefs))) {
		goto unlock_ret;
	}
	for (i = 0; i < TOP_PGDIR_SIZE; i++) {
		if (__unlikely(st->pagedir[i] != NULL)) {
			SWAP_DESTROY_ALL_DATA(st);
			break;
		}
	}
	st->ldrefsptr = sf->ldrefsptr;
	for (i = 0; i < TOP_PGDIR_SIZE; i++)
		if ((r = sf->pagedir[i])) {
			(st->pagedir[i] = r)->refcount++;
			if (__unlikely(ACCT_IS_LDCACHE(r->account))) REFERENCE_LDCACHE(st, r->account, 1, &KERNEL$PROC_KERNEL);
		}
	unlock_ret:
	SWAP_LOCK(SW_UNLOCK);
	check_quota(ACCT_FROM_SWAPNODE(sf), "SWAP_FORK 3");
}

int UNPAGED_ALLOC_FAILED(void)
{
	if (__unlikely(KERNEL$OOM(VM_TYPE_USER_MAPPED))) {
		if (!PRUNE_LDCACHE()) OUT_OF_UNPAGED(&root->unpageq);
		return 0;
	}
	return 1;
}

void OUT_OF_UNPAGED(QUOTA *q)
{
	QUOTA *zap;
	SWAPNODE *s;
	int locked;
	QZAP(q, unpageq_for_all_subnodes, unpageq_for_all_subnodes_tail, zap, 0);
	s = LIST_STRUCT(zap, SWAPNODE, unpageq);
	if (__unlikely(s == root)) return;
	locked = LOCK_STATE;
	if (!locked) SWAP_LOCK(SW_LOCK);
	SWAP_DESTROY_SWAPNODE_AND_DATA(s);
	if (!locked) SWAP_LOCK(SW_UNLOCK);
}

void OUT_OF_PAGED(QUOTA *q)
{
	QUOTA *zap;
	SWAPNODE *s;
	int locked;
	QZAP(q, pageq_for_all_subnodes, pageq_for_all_subnodes_tail, zap, 0);
	s = LIST_STRUCT(zap, SWAPNODE, pageq);
	if (__unlikely(s == root)) return;
	locked = LOCK_STATE;
	if (!locked) SWAP_LOCK(SW_LOCK);
	SWAP_DESTROY_SWAPNODE_AND_DATA(s);
	if (!locked) SWAP_LOCK(SW_UNLOCK);
}

#ifdef SWAP_LOCKING
volatile int swap_lock = 0;
WQ_DECL(swap_lock_wq, "SWAPPER$SWAP_LOCK_WQ");
#endif

DECL_IOCALL(SWP_PROCESS_WANTFREE, SPL_FS, IORQ)
{
	int f = 0;
	PAGE *p, *pp;
	WQ *wq;
	wantfree_active = 0;
	if (__unlikely((wantfree.status & RQS_ACTION_MASK) != RQS_PROCESSING))
		KERNEL$SUICIDE("SWP_PROCESS_WANTFREE: STATUS: %lX", wantfree.status);
	/*wantfree.tmp1 = (unsigned long)SWP_PROCESS_WANTFREE;*/
	XLIST_FOR_EACH(p, &wantfree_pages, PAGE, free_entry) {
		unsigned swp;
		if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p)) != NULL)) {
			if (!wantfree_active) {
				WQ_WAIT(wq, &wantfree, SWP_PROCESS_WANTFREE);
				wantfree_active = 1;
			}
			LOWER_SPL(SPL_FS);
			continue;
		}
		LOWER_SPL(SPL_FS);
		if (__unlikely(!((unsigned)(unsigned long)p->fnode & PAGE_FNODE_FREED))) {
			int r;
			pp = LIST_STRUCT(p->free_entry.prev, PAGE, free_entry);
			if (!(r = SWAPOUT_PAGE(p, NULL, 1, &KERNEL$PROC_KERNEL))) {
				f = 1;
				p = pp;
				continue;
			}
			if (__unlikely(r == 2)) {
				SWAPPAGE_OUT_OF_WANTFREE(p);
				p = pp;
				f = 1;
			}
			continue;
		}
		swp = (unsigned)(unsigned long)p->fnode >> 1;
		if (swp) CLEAR_SWPALLOC_BIT(swp);
		f = 1;
		pp = LIST_STRUCT(p->free_entry.prev, PAGE, free_entry);
		free_page(p, &KERNEL$PROC_KERNEL);
		p = pp;
	}
	if (f) WQ_WAKE_ALL(&wantfree_done);
	RETURN;
}

void SWAPPAGE_ADD_TO_WANTFREE(PAGE *p, WQ *wq)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_VSPACE)))
		KERNEL$SUICIDE("SWAPPAGE_ADD_TO_WANTFREE AT SPL %08X", KERNEL$SPL);
	if (__likely(!(p->flags & PAGE_WANTFREE))) {
		p->flags |= PAGE_WANTFREE;
		ADD_TO_XLIST(&wantfree_pages, &p->free_entry);
	}
	if (!wantfree_active) {
		WQ_WAIT(wq, &wantfree, SWP_PROCESS_WANTFREE);
		wantfree_active = 1;
	}
}

void SWAPPAGE_OUT_OF_WANTFREE(PAGE *p)
{
	RAISE_SPL(SPL_VSPACE);
	if (p->flags & PAGE_WANTFREE) {
		p->flags &= ~PAGE_WANTFREE;
		DEL_FROM_LIST(&p->free_entry);
	}
	LOWER_SPL(SPL_FS);
	WQ_WAKE_ALL(&p->wait);
/* do not count it as IO-in-progress in swapper */
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_REMOVE_VM_ENTITY(&p->e);
	TEST_SPL(SPL_FS, SPL_CACHE);
	KERNEL$CACHE_INSERT_VM_ENTITY(&p->e, &KERNEL$PROC_KERNEL, 0);
	LOWER_SPL(SPL_FS);
}

int SWAPDATA_INIT(void)
{
	wantfree_active = 0;
	wantfree.status = RQS_PROCESSING;
	INIT_XLIST(&wantfree_pages);
	return KERNEL$CACHE_REGISTER_VM_TYPE(&swap_vm_entity, &swap_calls);
}

void SWAPDATA_DONE(void)
{
	KERNEL$CACHE_UNREGISTER_VM_TYPE(swap_vm_entity);
}


