#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/BIO.H>
#include <SPAD/IOCTL.H>
#include <ARCH/LINUXPG.H>
#include <SPAD/VM.H>
#include <SPAD/TIMER.H>

#include "SWAPMAP.H"
#include "SWAPPER.H"
#include "STRUCT.H"

#if PAGE_CLUSTER_SIZE <= LINUX_PAGE_SIZE
too small page size
#endif

unsigned long total_pages;

PAGEZONE pagezone;
WQ freemem;
LIST_HEAD all_pages;

swp_map_t *swpalloc;
static unsigned swpalloc_allocated;
unsigned swpalloc_min;
unsigned swpalloc_min_bit;

int swphndl;
unsigned swppages;
unsigned swppages_free;
unsigned optimal_sectors;

struct __slhead pagenodes;
struct __slhead pagedirs;
struct __slhead swapnodes;
#ifdef SWAP_TTY
struct __slhead swapttys;
#endif
struct __slhead ldcaches;
struct __slhead ldrefs;
struct __slhead swaprqs;
WQ swaprqs_free;

SWAPNODE *root;

static SWAPRQ *dummy_swaprq = NULL;

char swapper_device[__MAX_STR_LEN];

static void *dlrq;
static void *lnte;

static char read_buffer_[LINUX_PAGE_SIZE + 2 * BIO_SECTOR_SIZE];
static union swap_header* read_buffer;

static void PAGEDIR_CTOR(struct __slhead *g, void *o);
static void SWAPNODE_CTOR(struct __slhead *g, void *o);
static void LDCACHE_CTOR(struct __slhead *g, void *o);
static void SWAP_INIT_ROOT(HANDLE *h, void *null);
static int SWAPPER_UNLOAD(void *p, void **release, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	char *swpdev;
	int r, i;
	const char * const *arg;
	int state;
	PAGE *p;
	union {
		OPENRQ openrq;
		CHHRQ chhrq;
		struct {
			BIORQ biorq;
			BIODESC desc;
		} b;
		IOCTLRQ ioctlrq;
	} u;
	static const struct __param_table params[2] = {
		"", __PARAM_STRING, 1, __MAX_STR_LEN,
		NULL, 0, 0, 0,
	};
	void *vars[2];
	vars[0] = &swpdev;
	vars[1] = NULL;
	swpdev = NULL;
	if (__unlikely(PAGE_CLUSTER_SIZE != __PAGE_CLUSTER_SIZE)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER COMPILED WITH PAGE SIZE %u, KERNEL COMPILED WITH PAGE SIZE %u", (unsigned)PAGE_CLUSTER_SIZE, (unsigned)__PAGE_CLUSTER_SIZE);
		return -EINVAL;
	}
	total_pages = KERNEL$GET_MEMORY_SIZE(VM_TYPE_USER_UNMAPPED) >> (PG_SIZE_BITS + PG_CLUSTER_BITS);
	swphndl = -1;
	read_buffer = (union swap_header *)((unsigned long)(read_buffer_ + BIO_SECTOR_SIZE - 1) & ~(unsigned long)(BIO_SECTOR_SIZE - 1));
	WQ_INIT(&freemem, "SWAPPER$FREEMEM");
	WQ_INIT(&swaprqs_free, "SWAPPER$SWAPRQS_FREE");
	INIT_LIST(&all_pages);
	VFS$ZINIT(&pagezone, VM_TYPE_USER_UNMAPPED, "SWAPPER PAGES");
	if (SWAPDATA_INIT()) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: CAN'T REFISTER SWAP VM ENTITY");
		r = -ENFILE;
		goto err_2;
	}
	if (__unlikely(LDCACHE_INIT())) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: CAN'T REFISTER LDCACHE VM ENTITY");
		r = -ENFILE;
		goto err_1;
	}
	KERNEL$SLAB_INIT(&pagenodes, sizeof(PAGENODE), 0, VM_TYPE_USER_MAPPED, NULL, NULL, "SWAPPER$PAGENODE");
	KERNEL$SLAB_INIT(&pagedirs, sizeof(PAGEDIR), 0, VM_TYPE_USER_MAPPED, PAGEDIR_CTOR, NULL, "SWAPPER$PAGEDIR");
	KERNEL$SLAB_INIT(&swapnodes, sizeof(SWAPNODE), 0, VM_TYPE_USER_MAPPED, SWAPNODE_CTOR, NULL, "SWAPPER$SWAPNODE");
#ifdef SWAP_TTY
	KERNEL$SLAB_INIT(&swapttys, SIZEOF_SWAPTTY, 0, VM_TYPE_USER_MAPPED, NULL, NULL, "SWAPPER$SWAPTTY");
#endif
	KERNEL$SLAB_INIT(&ldcaches, sizeof(LDCACHE), 0, VM_TYPE_USER_MAPPED, LDCACHE_CTOR, NULL, "SWAPPER$LDCACHE");
	KERNEL$SLAB_INIT(&ldrefs, sizeof(LDREF), 0, VM_TYPE_USER_MAPPED, NULL, NULL, "SWAPPER$LDREF");
	KERNEL$SLAB_INIT(&swaprqs, sizeof(SWAPRQ), 0, VM_TYPE_USER_MAPPED, SWAPRQ_CTOR, NULL, "SWAPPER$SWAPRQ");

	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: SYNTAX ERROR");
		r = -EBADSYN;
		goto err0;
	}
	swppages_free = 0;
	if (!swpdev) {
		swapper_device[0] = 0;
		swppages = 0;
		goto skip_swap;
	}
	strcpy(swapper_device, swpdev);
	u.openrq.flags = O_RDWR | O_DIRECT;
	u.openrq.path = swpdev;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: ERROR OPENING DEVICE %s: %s", swpdev, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto err0;
	}
	swphndl = u.openrq.status;
	u.chhrq.h = swphndl;
	u.chhrq.option = "DIRECT";
	u.chhrq.value = "";
	SYNC_IO(&u.chhrq, KERNEL$CHANGE_HANDLE);
	u.ioctlrq.h = swphndl;
	u.ioctlrq.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
	u.ioctlrq.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_WRITE;
	u.ioctlrq.v.ptr = 0;
	u.ioctlrq.v.len = 0;
	u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&u.ioctlrq, KERNEL$IOCTL);
	optimal_sectors = u.ioctlrq.status >= 0 ? u.ioctlrq.status >> BIO_SECTOR_SIZE_BITS : 0;
	u.b.biorq.h = swphndl;
	u.b.biorq.sec = 0;
	u.b.biorq.nsec = (LINUX_PAGE_SIZE + BIO_SECTOR_SIZE - 1) >> BIO_SECTOR_SIZE_BITS;
	u.b.biorq.flags = BIO_READ;
	u.b.biorq.desc = &u.b.desc;
	u.b.biorq.proc = &KERNEL$PROC_KERNEL;
	u.b.biorq.fault_sec = -1;
	u.b.desc.v.ptr = (unsigned long)read_buffer;
	u.b.desc.v.len = u.b.biorq.nsec << BIO_SECTOR_SIZE_BITS;
	u.b.desc.v.vspace = &KERNEL$VIRTUAL;
	u.b.desc.next = NULL;
	SYNC_IO_CANCELABLE(&u.b.biorq, KERNEL$BIO);
	if (u.b.biorq.status < 0) {
		r = u.b.biorq.status;
		goto err1;
	}
	if (!memcmp(read_buffer->magic.magic, SWAP_LINUX_V1_MAGIC, 10)) {
		int i;
		for (i = 1; i < (LINUX_PAGE_SIZE - 10) * 8; i++) {
			if (!(read_buffer->magic.reserved[i >> 3] & (1 << (i & 7)))) break;
		}
		swppages = i / (PAGE_CLUSTER_SIZE / LINUX_PAGE_SIZE);
		for (; i < (LINUX_PAGE_SIZE - 10) * 8; i++) {
			if (read_buffer->magic.reserved[i >> 3] & (1 << (i & 7))) {
				badpg:
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: SWAP FILE HAS BAD PAGES; NOT SUPPORTED", swpdev);
				r = -EINVAL;
				goto err1;
			}
		}
	} else if (!memcmp(read_buffer->magic.magic, SWAP_LINUX_V2_MAGIC, 10)) {
		if (read_buffer->info.version != 1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: SWAP FILE HAS VERSION %d; NOT SUPPORTED", swpdev, read_buffer->info.version);
			r = -EINVAL;
			goto err1;
		}
		if (read_buffer->info.nr_badpages) goto badpg;
		swppages = ((__u64)read_buffer->info.last_page + 1) / (PAGE_CLUSTER_SIZE / LINUX_PAGE_SIZE);
	} else {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: SWAP FILE SIGNATURE NOT FOUND", swpdev);
		r = -EINVAL;
		goto err1;
	}
	if (swppages <= 1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: NO PAGES IN SWAP FILE", swpdev);
		r = -EINVAL;
		goto err1;
	}
	if (swppages > MAXINT) swppages = MAXINT;

	skip_swap:

	for (i = 0; i < RESERVED_PAGES; i++) {
		while (__unlikely(!(p = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED)))) {
			if (KERNEL$OOM(VM_TYPE_WIRED_UNMAPPED)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: OUT OF MEMORY FOR RESERVED PAGES");
				r = -ENOMEM;
				goto err1;
			}
			if ((r = WQ_WAIT_SYNC_CANCELABLE(&KERNEL$FREEMEM_WAIT))) goto err1;
		}
		VFS$ZRESERVE(&pagezone, p);
	}

	for (i = 0; i < 6; i++) {
		if (__unlikely(r = KERNEL$SLAB_RESERVE(i == 0 ? &pagenodes : i == 1 ? &pagedirs : i == 2 ? &swapnodes : i == 3 ? &ldcaches : i == 4 ? &ldrefs : i == 5 ? &swaprqs : NULL, i == 5 ? 2 : 1))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: OUT OF MEMORY FOR SLABS");
			goto err1;
		}
	}

	dummy_swaprq = __slalloc(&swaprqs);
	if (__unlikely(!dummy_swaprq))
		KERNEL$SUICIDE("SWAPPER: CAN'T ALLOCATE DUMMY SWAPRQ");

	if (swppages) {
		swpalloc_allocated = (swppages + 8 * sizeof(swp_map_t) - 1) / (8 * sizeof(swp_map_t)) * sizeof(swp_map_t);
		swpalloc = KERNEL$ALLOC_CONTIG_AREA(swpalloc_allocated, AREA_DATA);
		if (__IS_ERR(swpalloc)) {
			r = __PTR_ERR(swpalloc);
			swpalloc = NULL;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: CAN'T ALLOCATE SWAP ALLOC MAP: %s", strerror(-r));
			goto err1;
		}
		memset(swpalloc, 0xff, swpalloc_allocated);
		for (i = 1; i < swppages; i++) CLEAR_SWPALLOC_BIT(i);
		swpalloc_min = 0;
		swpalloc_min_bit = 0;
	} else swpalloc = NULL;

	if (!(root = __slalloc(&swapnodes)))
		KERNEL$SUICIDE("SWAPPER: CAN'T ALLOCATE ROOT SWAPNODE");
	init_swapnode(root);
	
	VOID_LIST_ENTRY(&root->hash);
	root->jobname = 0;
	root->parent = NULL;
	QINIT(&root->pageq);
	root->pageq.q_limit = MAXINT;
	QINIT2(&root->pageq);
	QINIT(&root->unpageq);
	{
		__u64 mem;
		mem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_USER_MAPPED);
		mem >>= 3;
		if (mem > MAXLONG) mem = MAXINT;
		root->unpageq.q_limit = mem;
	}
	QINIT2(&root->unpageq);
	root->depth = 0;

	r = KERNEL$REGISTER_DEVICE(SWAPPER_DEVICE_NAME, "SWAPPER.SYS", LNTE_PUBLIC | LNTE_VM_ESSENTIAL, NULL, SWAP_INIT_ROOT, NULL, NULL, NULL, SWAPPER_UNLOAD, &lnte, swphndl >= 0 ? KERNEL$HANDLE_PATH(swphndl) : NULL, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, SWAPPER_DEVICE_NAME ": ERROR REGISTERING DEVICE: %s", strerror(-r));
		goto err3;
	}
	strcpy(KERNEL$ERROR_MSG(), SWAPPER_DEVICE_NAME);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	err3:
	__slow_slfree(root);
	if (swpalloc)
		KERNEL$FREE_CONTIG_AREA(swpalloc, swpalloc_allocated);
	err1:
	if (swphndl >= 0) KERNEL$FAST_CLOSE(swphndl);
	err0:
	RAISE_SPL(SPL_FS);
	LDCACHE_DONE();
	LOWER_SPL(SPL_ZERO);
	KERNEL$SLAB_DESTROY(&pagenodes);
	KERNEL$SLAB_DESTROY(&pagedirs);
	KERNEL$SLAB_DESTROY(&ldcaches);
	KERNEL$SLAB_DESTROY(&ldrefs);
	KERNEL$SLAB_DESTROY(&swapnodes);
#ifdef SWAP_TTY
	KERNEL$SLAB_DESTROY(&swapttys);
#endif
	if (dummy_swaprq != NULL) __slow_slfree(dummy_swaprq);
	KERNEL$SLAB_DESTROY(&swaprqs);
	err_1:
	SWAPDATA_DONE();
	err_2:
	VFS$ZDONE(&pagezone);
	return r;
}

static void PAGEDIR_CTOR(struct __slhead *g, void *o)
{
	memset(o, 0, sizeof(PAGEDIR));
}

static void SWAPNODE_CTOR(struct __slhead *g, void *o)
{
	SWAPNODE *s = o;
	int i;
	INIT_XLIST(&s->handles);
	INIT_XLIST(&s->ldrefs);
	for (i = 0; i < CHILDHASH_SIZE; i++) INIT_XLIST(&s->childhash[i]);
	memset(s->pagedir, 0, sizeof s->pagedir);
}

static void LDCACHE_CTOR(struct __slhead *g, void *o)
{
	LDCACHE *s = o;
	memset(&s->pagedir, 0, sizeof s->pagedir);
	s->parent = NULL;
	CACHE_CONSTRUCT_VM_ENTITY(&s->vme);
	s->vme.type = ldcache_vm_entity;
}

static void SWAP_INIT_ROOT(HANDLE *h, void *null)
{
	h->fnode = root;
	h->op = &SWAP_OPERATIONS;
	/* if flags will be used, they must be also added to SWAP_CLONE */
}

static int SWAPPER_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	RAISE_SPL(SPL_FS);
	SWAP_LOCK(SW_LOCK);
	SWAP_DESTROY_SWAPNODE_AND_DATA(root);
	SWAP_LOCK(SW_UNLOCK);
	LDCACHE_DONE();
	while (__unlikely(!LIST_EMPTY(&all_pages))) KERNEL$SLEEP(1);
	LOWER_SPL(SPL_ZERO);
	if (swppages && __unlikely(swppages_free != swppages - 1))
		KERNEL$SUICIDE("SWAPPER_UNLOAD: SWPPAGES_FREE LEAKED: %u, SWPPAGES %u", swppages_free, swppages);
	if (swpalloc)
		KERNEL$FREE_CONTIG_AREA(swpalloc, swpalloc_allocated);
	if (swphndl >= 0) KERNEL$FAST_CLOSE(swphndl);
	SWAPDATA_DONE();
	VFS$ZDONE(&pagezone);
	KERNEL$SLAB_DESTROY(&pagenodes);
	KERNEL$SLAB_DESTROY(&pagedirs);
	KERNEL$SLAB_DESTROY(&ldcaches);
	KERNEL$SLAB_DESTROY(&ldrefs);
	KERNEL$SLAB_DESTROY(&swapnodes);
#ifdef SWAP_TTY
	KERNEL$SLAB_DESTROY(&swapttys);
#endif
	if (dummy_swaprq != NULL) __slow_slfree(dummy_swaprq);
	KERNEL$SLAB_DESTROY(&swaprqs);
	*release = dlrq;
	return 0;
}



