#ifndef _SWAPPER_SWAPPER_H
#define _SWAPPER_SWAPPER_H

#include <SPAD/LIST.H>
#include <SPAD/WQ.H>
#include <SPAD/PAGEZONE.H>
#include <SPAD/SLAB.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/BIO.H>
#include <SPAD/PROC.H>
#include <KERNEL/VMDEF.H>

#include <SPAD/SWAPPER.H>

#include "SWAPMAP.H"

#define RESERVED_PAGES		16
#define EMERGENCY_RESERVE_PAGES	1

#define SWAPPER_DEVICE_NAME	"SYS$SWAPPER"

/*#define DEBUG_QUOTA*/

/* invariant: there must be at least one page in zone, in swapfile or
   or one discardable swap-cached page --- thus we allocate the last
   page from zone only for swapins */

#define SECTORS_PER_PAGE	(PAGE_CLUSTER_SIZE / BIO_SECTOR_SIZE)

/* SWAPPER.C */

#ifdef DEBUG_QUOTA
void check_quota(unsigned long s, char *msg);
#else
#define check_quota(a, b) do { } while (0)
#endif

extern unsigned long total_pages;

extern PAGEZONE pagezone;
extern WQ freemem;
extern LIST_HEAD all_pages;

/* This may be typedefed to __u8, __u16, __u32, __u64 --- swapper will search
   for free block accordingly */
typedef __u8 swp_map_t;
/* Retry search after 256 pages */
#define SWP_RETRY_MASK		255

#define SWP_MAP_BITS		(sizeof(swp_map_t) * 8)
#define SWP_MAP_BITS_MASK	(SWP_MAP_BITS - 1)
#define SWP_MAP_BITS_L2		(__BSR_CONST(SWP_MAP_BITS))

extern swp_map_t *swpalloc;
extern unsigned swpalloc_min;
extern unsigned swpalloc_min_bit;

extern int swphndl;
extern unsigned swppages;
extern unsigned swppages_free;
extern unsigned optimal_sectors;

extern struct __slhead pagenodes;
extern struct __slhead pagedirs;
extern struct __slhead swapnodes;
#ifdef SWAP_TTY
extern struct __slhead swapttys;
#endif
extern struct __slhead ldcaches;
extern struct __slhead ldrefs;
extern struct __slhead swaprqs;
extern WQ swaprqs_free;

extern SWAPNODE *root;

extern char swapper_device[__MAX_STR_LEN];

static __finline__ void init_swapnode(SWAPNODE *s)
{
	s->pagedir[0] = NULL;
	s->ldrefsptr = s;
	QINIT(&s->pageq);
	QINIT2(&s->pageq);
	QINIT(&s->unpageq);
	QINIT2(&s->unpageq);
#ifdef SWAP_TTY
	s->tty = NULL;
#endif
}

static __finline__ void init_ldcache(LDCACHE *s)
{
	s->pagedir[0] = NULL;
	QINIT(&s->pageq);
	QINIT2(&s->pageq);
	QINIT(&s->unpageq);
	QINIT2(&s->unpageq);
}

/* SWAPOP.C */

extern const HANDLE_OPERATIONS SWAP_OPERATIONS;

SWAPNODE *SWAP_FIND_NODE(SWAPNODE *s, char *str);
void *SWAP_INSTANTIATE(HANDLE *h, IORQ *rq, int open_flags);
void SWAP_DETACH(HANDLE *h);

/* SWAPDATA.C */

extern int swap_vm_entity;

extern int wantfree_active;
extern IORQ wantfree;
extern XLIST_HEAD wantfree_pages;
extern WQ wantfree_done;
extern IO_STUB SWP_PROCESS_WANTFREE;

void SWAPPAGE_ADD_TO_WANTFREE(PAGE *p, WQ *wq);
void SWAPPAGE_OUT_OF_WANTFREE(PAGE *p);

void *COPY_PAGEDIR(PAGEDIR **d, SWAPNODE *s);
void *SWAP_ZAP(PAGEDIR **pp, unsigned depth, unsigned long from, unsigned long to, unsigned long node, PROC *owner);
void SWAP_DESTROY_SWAPNODE_AND_DATA(SWAPNODE *s);
void *SWAP_DESTROY_DATA(SWAPNODE *s, unsigned long start, unsigned long end, PROC *owner);
void SWAP_FORK(SWAPNODE *sf, SWAPNODE *st);
/* returns: 0 --- retry, 1 --- wait on FREEMEM_WAIT */
int UNPAGED_ALLOC_FAILED(void);
void OUT_OF_UNPAGED(QUOTA *q);
void OUT_OF_PAGED(QUOTA *q);

static __finline__ int TEST_SWPALLOC_BIT(unsigned i)
{
#if __DEBUG >= 1
	if (__unlikely(i >= swppages))
		KERNEL$SUICIDE("TEST_SWPALLOC_BIT: OUT OF SWAP MAP: %u >= %u (swpalloc_min %u, swpalloc_min_bit %u)", i, swppages, swpalloc_min, swpalloc_min_bit);
#endif
	return (swpalloc[i >> SWP_MAP_BITS_L2] >> (i & SWP_MAP_BITS_MASK)) & 1;
}

static __finline__ void CLEAR_SWPALLOC_BIT(unsigned i)
{
	unsigned idx;
#if __DEBUG >= 1
	if (__unlikely(!TEST_SWPALLOC_BIT(i)))
		KERNEL$SUICIDE("CLEAR_SWPALLOC_BIT: BIT %u ALREADY CLEAR", i);
#endif
	idx = i >> SWP_MAP_BITS_L2;
	if (__unlikely(!(swpalloc[idx] &= ~(1 << (i & SWP_MAP_BITS_MASK)))) && __unlikely(idx < swpalloc_min))
		swpalloc_min = idx;
	if (__unlikely(idx < swpalloc_min_bit)) swpalloc_min_bit = idx;
	swppages_free++;
	WQ_WAKE_ALL(&freemem);
}

static __finline__ void SET_SWPALLOC_BIT(unsigned i)
{
#if __DEBUG >= 1
	if (__unlikely(TEST_SWPALLOC_BIT(i)))
		KERNEL$SUICIDE("SET_SWPALLOC_BIT: BIT %u ALREADY SET", i);
#endif
	swpalloc[i >> SWP_MAP_BITS_L2] |= 1 << (i & SWP_MAP_BITS_MASK);
	swppages_free--;
}

#ifdef SWAP_LOCKING

/*
 * Lock was originaly thought to allow routines at SPL_VSPACE to access
 * swapnode tree. But there are no such routines now, so I disabled it.
 * I left the code there for the case it would be needed sometimes.
 */

extern volatile int swap_lock;
extern WQ swap_lock_wq;
#define SW_LOCK		0
#define SW_UNLOCK	1

static __finline__ void SWAP_LOCK(int op)
{
#if __DEBUG >= 1
	if (__unlikely(swap_lock != op))
		KERNEL$SUICIDE("SWAP_LOCK: SWAP ALREADY LOCKED: %d", swap_lock);
#endif
	__barrier();
	swap_lock = op ^ 1;
	__barrier();
	if (op == SW_UNLOCK) WQ_WAKE_ALL(&swap_lock_wq);
}

static __finline__ void ASSERT_LOCKED(void)
{
#if __DEBUG >= 1
	if (__unlikely(!swap_lock))
		KERNEL$SUICIDE("ASSERT_LOCKED: NOT LOCKED");
#endif
}

#define LOCK_STATE	swap_lock

#else

#define SW_LOCK		0
#define SW_UNLOCK	0

static __finline__ void SWAP_LOCK(int op)
{
}

static __finline__ void ASSERT_LOCKED(void)
{
}

#define LOCK_STATE	0

#endif

int SWAPDATA_INIT(void);
void SWAPDATA_DONE(void);

/* SWAPPND.I */

PAGEDIR **SWAP_RETURN_PAGENODE(SWAPNODE *s, unsigned long idx, int *wr);
PAGENODE *SWAP_SIMPLE_RETURN_PAGENODE(SWAPNODE *s, unsigned long idx);

/* SWAPPAGE.C */

void SWAP_FREE_PAGENODE(PAGENODE *pg);
PAGE *SWAP_CREATE_PAGE(unsigned long s, unsigned long idx, int wr, IORQ *rq, PROC *owner);
extern IO_STUB SWAP_PAGE_RELEASE;
PAGE *SWAP_GET_PAGE(HANDLE *h, __v_off idx, int wr);
IORQ *SWAP_GET_PAGEIN_RQ(VDESC *dec, IORQ *rq, int wr);
int SWAP_SWAP_OP(HANDLE *h, int op, ...);
void SWAP_WRITE(VMENTITY *e, PROC *proc, int trashing);
int SWAP_CHECKMAP(VMENTITY *e);
int SWAP_SWAPOUT(VMENTITY *e);
WQ *SWAP_PAGE_LOCKDOWN(PAGE *p, int lock);

static __finline__ void setup_node_page(PAGE *p, PAGENODE *n, PROC *proc)
{
	ASSERT_LOCKED();
	n->page = p;
	WQ_INIT(&p->wait, "SWAPPER$PAGE_WAIT");
	p->release = SWAP_PAGE_RELEASE;
	p->lockdown = SWAP_PAGE_LOCKDOWN;
	p->fnode = n;
	p->flags = PAGE_WRITEABLE;
	INIT_XLIST(&p->mapping);
	CACHE_CONSTRUCT_VM_ENTITY(&p->e);
	RAISE_SPL(SPL_CACHE);
	p->e.type = swap_vm_entity;
	KERNEL$CACHE_INSERT_VM_ENTITY(&p->e, proc, VM_ENTITY_NOSTREAM);
	LOWER_SPL(SPL_FS);
	ADD_TO_LIST(&all_pages, &p->node_entry);
}

static __finline__ void free_page(PAGE *page, PROC *owner)
{
	KERNEL$NOTIFY_FREE(owner, PAGE_CLUSTER_SIZE);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(page->flags & PAGE_WANTFREE)) DEL_FROM_LIST(&page->free_entry);
	LOWER_SPL(SPL_FS);
	WQ_WAKE_ALL(&page->wait);
	DEL_FROM_LIST(&page->node_entry);
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_REMOVE_VM_ENTITY(&page->e);
	LOWER_SPL(SPL_FS);
	PAGEZONE_FREE(&pagezone, page, EMERGENCY_RESERVE_PAGES);
	WQ_WAKE_ALL(&freemem);
}

/* SWAPIO.C */

typedef struct __swaprq SWAPRQ;

struct __swaprq {
	SWAPRQ *next;
	SWAPRQ *last;
	PAGE *page;
	PROC_TAG tag;
	BIORQ rq;
	BIODESC desc;
	IORQ *iorq;
	unsigned char allocated_from_beginning;
	__u64 error_kill_seq;
};

void SWAPRQ_CTOR(struct __slhead *g, void *o);
void SWAPIN(PAGENODE *pn, IORQ *rq, PROC *owner, int wr);
int SWAPOUT_PAGE(PAGE *p, IORQ *rq, int need_free, PROC *owner);
void SWAPOUT(IORQ *rq);

/* SWAPLD.C */

extern int ldcache_vm_entity;

void REFERENCE_LDCACHE(SWAPNODE *sn, unsigned long account, long diff, PROC *owner);
void SWAPLD_SET_ESTABLISHED(LDCACHE *ld);
int SWAPLD_QUERY(LD_FINAL_HASH *h, unsigned long *addr);
int PRUNE_LDCACHE(void);
int TEST_PRUNE_LDCACHE(void);
int LDCACHE_INIT(void);
void LDCACHE_DONE(void);

/* SWAPLINK.C */

int LDCACHE_IOCTL(SWAPNODE *s, IOCTLRQ *rq);
int SWAPLD_PREPARE(SWAPNODE *sn, LD_PREPARE *ldp, IOCTLRQ *rq);
int SWAPLD_ATTACH(SWAPNODE *s, LD_FINAL_HASH *h, unsigned long addr, IOCTLRQ *rq);

#ifdef SWAP_TTY

/* SWAPTTY.C */

void *SWAPTTYP_LOOKUP(HANDLE *h);
void *SWAPPTY_LOOKUP(HANDLE *h);
void SWAPTTY_FREE(SWAPTTY *t);

#endif

#endif
