#include <SYS/TYPES.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/TTY.H>

#include "SWAPPER.H"
#include "SWAPMAP.H"

#ifdef SWAP_TTY

#define SWAP_TTY_BUFFER		2048
#define SWAP_TTY_KBD_BUFFER	256

/* handle->flags */
#define PTY_NONBLOCK		1

struct __swaptty {
	TTY tty;
	int writeout_pos;
	WQ read_wait;
	char buf[SWAP_TTY_BUFFER];
};

const unsigned SIZEOF_SWAPTTY = sizeof(SWAPTTY);

static void SWAPTTY_WRITEOUT(TTY *tty);

static void *PTY_LOOKUP(HANDLE *h, char *str, int open_flags);
static IO_STUB PTY_READ;
static IO_STUB PTY_WRITE;
static IO_STUB PTY_IOCTL;

static const HANDLE_OPERATIONS PTY_OPERATIONS = {
	SPL_X(SPL_TTY),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	PTY_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	SWAP_INSTANTIATE,
	NULL,			/* leave */
	SWAP_DETACH,
	NULL,			/* close */
	PTY_READ,
	PTY_WRITE,
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	PTY_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static void *SWAPTTY_CREATE(SWAPNODE *s)
{
	int r;
	SWAPTTY *t = __slalloc(&swapttys);
	if (__unlikely(!t)) {
		if (!UNPAGED_ALLOC_FAILED()) return (void *)2;
		return &KERNEL$FREEMEM_WAIT;
	}
	t->writeout_pos = -1;
	WQ_INIT(&t->read_wait, "SWAPPER$TTY_READ_WAIT");
	t->tty.buf = t->buf;
	t->tty.buf_size = SWAP_TTY_BUFFER;
	t->tty.writeout = SWAPTTY_WRITEOUT;
	t->tty.dtty = NULL;
	r = TTY$CREATE(&t->tty);
	if (__unlikely(r)) {
		__slow_slfree(t);
		return __ERR_PTR(r);
	}
	s->tty = t;
	return NULL;
}

void *SWAPTTYP_LOOKUP(HANDLE *h)
{
	SWAPNODE *s = h->fnode;
	if (__unlikely(!s->tty)) {
		void *p = SWAPTTY_CREATE(s);
		if (__unlikely(p != NULL)) return p;
	}
	TTY$INIT_ROOT(h, &s->tty->tty);
	return NULL;
}

void *SWAPPTY_LOOKUP(HANDLE *h)
{
	SWAPNODE *s = h->fnode;
	if (__unlikely(!s->tty)) {
		void *p = SWAPTTY_CREATE(s);
		if (__unlikely(p != NULL)) return p;
	}
	h->op = &PTY_OPERATIONS;
	h->flags = 0;
	h->flags2 = 0;
	return NULL;
}

void SWAPTTY_FREE(SWAPTTY *t)
{
	TTY$DESTROY(&t->tty);
	WQ_WAKE_ALL(&t->read_wait);
	__slfree(t);
}

static void *PTY_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	SWAPTTY *t;
	char *e = strchr(str, '=');
	if (!e) {
		if (__likely(!_strcasecmp(str, "^NONBLOCK"))) {
			h->flags |= PTY_NONBLOCK;
		} else return __ERR_PTR(-EBADMOD);
		return NULL;
	} else if (!__strcasexcmp("^WINSIZE", str, e)) {
		/* copied in TTY.C */
		long x, y, cx, cy;
		char *d1, *d2, *d3;
		d1 = strchr(e + 1, ',');
		if (__unlikely(!d1)) return __ERR_PTR(-EBADMOD);
		d2 = strchr(d1 + 1, ',');
		if (__unlikely(!d2)) return __ERR_PTR(-EBADMOD);
		d3 = strchr(d2 + 1, ',');
		if (__unlikely(!d3)) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(e + 1, d1, 1, &x))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(d1 + 1, d2, 1, &y))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(d2 + 1, d3, 1, &cx))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(d3 + 1, d3 + strlen(d3), 1, &cy))) return __ERR_PTR(-EBADMOD);
		t = ((SWAPNODE *)h->fnode)->tty;
		t->tty.xsize = x;
		t->tty.ysize = y;
		t->tty.xpixels = cx;
		t->tty.ypixels = cy;
		t->tty.bits |= TTY_LOCK_WINDOW_SIZE;
		WQ_WAKE_ALL(&t->tty.state_wait);
	} else {
		return __ERR_PTR(-EBADMOD);
	}
	return NULL;
}

static void SWAPTTY_WRITEOUT(TTY *tty)
{
	SWAPTTY *t = GET_STRUCT(tty, SWAPTTY, tty);
	if (__unlikely(t->writeout_pos >= 0))
		KERNEL$SUICIDE("SWAPTTY_WRITEOUT: WRITEOUT ALREADY READY (POS %d)", t->writeout_pos);
	t->writeout_pos = 0;
	WQ_WAKE_ALL_PL(&t->read_wait);
}

static DECL_IOCALL(PTY_READ, SPL_TTY, SIORQ)
{
	HANDLE *h = RQ->handle;
	SWAPTTY *t;
	VBUF vbuf;
	int s;
	if (__unlikely(h->op != &PTY_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_TTY));
	t = ((SWAPNODE *)h->fnode)->tty;
	if (t->writeout_pos < 0) {
		if (h->flags & PTY_NONBLOCK) {
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		WQ_WAIT_F(&t->read_wait, RQ);
		RETURN;
	}
	again:
	vbuf.ptr = t->tty.buffer_to_write + t->writeout_pos;
	vbuf.len = t->tty.buffer_to_write_len - t->writeout_pos;
	vbuf.spl = SPL_X(SPL_TTY);
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf);
	if (__unlikely(!s)) {
		DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	}
	RQ->progress += s;
	t->writeout_pos += s;
	if (t->writeout_pos == t->tty.buffer_to_write_len) {
		t->writeout_pos = -1;
		TTY$WRITE_DONE(&t->tty);
	}
	if (t->writeout_pos >= 0 && RQ->v.len) {
		TEST_LOCKUP_LOOP(RQ, RETURN);
		goto again;
	}
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static char write_buf[SWAP_TTY_KBD_BUFFER];

static const VBUF write_v = { SPL_X(SPL_TTY), SWAP_TTY_KBD_BUFFER, write_buf };

static DECL_IOCALL(PTY_WRITE, SPL_TTY, SIORQ)
{
	HANDLE *h = RQ->handle;
	SWAPTTY *t;
	int s, q;
	if (__unlikely(h->op != &PTY_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp2 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_TTY));
	t = ((SWAPNODE *)h->fnode)->tty;
	again:
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_get(&RQ->v, &write_v);
	if (__unlikely(!s)) {
		DO_PAGEIN(RQ, &RQ->v, PF_READ);
	}
	q = TTY$IN(&t->tty, write_buf, s);
	RQ->v.ptr -= s - q;
	RQ->v.len += s - q;
	RQ->progress += q;
	if (__unlikely(!q)) {
		if (h->flags & PTY_NONBLOCK) {
			if (RQ->progress) goto end;
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		WQ_WAIT_F(&t->tty.in_wait, RQ);
		RETURN;
	}
	if (__unlikely(RQ->v.len != 0)) goto again;
	end:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static DECL_IOCALL(PTY_IOCTL, SPL_TTY, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	SWAPTTY *t;
	if (__unlikely(h->op != &PTY_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp2 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_TTY));
	t = ((SWAPNODE *)h->fnode)->tty;
	switch (RQ->ioctl) {
		case IOCTL_SELECT_READ: {
			if (t->writeout_pos < 0) {
				if (RQ->param) {
					WQ_WAIT_F(&t->read_wait, RQ);
					RETURN;
				}
				RQ->status = -EWOULDBLOCK;
			} else {
				RQ->status = 0;
			}
			RETURN_AST(RQ);
		}
/* no IOCTL_SELECT_WRITE --- we can't tell if write will or won't block,
   it depends on data written :-/ */
   		case IOCTL_NREAD: {
			if (t->writeout_pos < 0) RQ->status = 0;
			else RQ->status = t->tty.buffer_to_write_len - t->writeout_pos;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

#endif
